;;; -*- Mode: LISP; Syntax: Common-lisp; Package: YY; Base: 10 -*-
;;; bitmap.lisp
;;;
;;;  Copyright (C) 1989,1990,1991 Aoyama Gakuin University
;;;
;;;		All Rights Reserved
;;;
;;; This software is developed for the YY project of Aoyama Gakuin University.
;;; Permission to use, copy, modify, and distribute this software
;;; and its documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notices appear in all copies and that
;;; both that copyright notice and this permission notice appear in 
;;; supporting documentation, and that the name of Aoyama Gakuin
;;; not be used in advertising or publicity pertaining to distribution of
;;; the software without specific, written prior permission.
;;;
;;; This software is made available AS IS, and Aoyama Gakuin makes no
;;; warranty about the software, its performance or its conformity to
;;; any specification. 
;;;
;;; To make a contact: Send E-mail to ida@csrl.aoyama.ac.jp for overall
;;; issues. To ask specific questions, send to the individual authors at
;;; csrl.aoyama.ac.jp. To request a mailing list, send E-mail to 
;;; yyonx-request@csrl.aoyama.ac.jp.
;;;
;;; Authors:
;;;   version 1.3 91/01/23 by t.kosaka

;;; $@%S%C%H%^%C%W%/%i%9(J

(in-package :yy)

;;; print-object
(defmethod print-object ((bitmap bitmap) stream)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (format stream "\#<Bitmap size ~ax~a>" (bitmap-width bitmap)
	  (bitmap-height bitmap)))

;;; $@%S%C%H%^%C%W$+(J
;;; bitmapp object
;;; ARG.
;;;         object  = $@%S%C%H%^%C%W%$%s%9%?%s%9(J
;;; RET.
;;;      T or NIL
(defun bitmapp (object)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (typep object 'bitmap))


;;; $@%S%C%H%^%C%W$N@8@.(J
;;; make-bitmap width height &key image
;;; ARG.
;;;         width   =  $@I}(J
;;;         height  =  $@9b$5(J
;;;         image   =  $@%$%a!<%8%$%s%9%?%s%9(J
(defun make-bitmap (width height &key (image nil))
  (declare (function image-data (t) vector)
	   (function put-image-xy-bitmap (t t integer integer 
					    integer integer integer 
					    integer) T)
	   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let* ((instance (make-instance 'bitmap
				  :bitmap-width width
				  :bitmap-height height))
#+ :YY2.0
	 (tno (with-object-make-territory instance
		       :width width :height height
		       :visible nil
		       :transparent T))
#- :YY2.0
         (tno (with-object-make-territory instance
                       :width width :height height
                       :visible nil))
	 )
    (setf (bitmap-territory-no instance) tno)
    ;;; $@%$%a!<%8$rE>Aw(J
    (if image
	(put-image-xy-bitmap instance image 0 0 width height 0 0))
    instance))
      
;;; $@%S%C%H%^%C%W$N>C5n(J
;;; flush-bitmap bitmap
;;; ARG.
;;;       bitmap  = $@%S%C%H%^%C%W%$%s%9%?%s%9(J
(defmethod flush-bitmap ((bitmap bitmap))
(declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((val (yy-protocol-5 (bitmap-territory-no bitmap))))
    (if (zerop val)
	(error "Sorry ,can not remove ~a" bitmap)
      (progn 
	(delete-lisp-object (bitmap-territory-no bitmap))
      nil)))
  )
  
;;; $@%S%C%H%^%C%W$N3HBg!?=L>.(J
;;; expange-bitmap bitmap x-ratio y-ratio &key create
;;; ARG.
;;;         bitmap      =  $@BP>]$J$k%S%C%H%^%C%W%$%s%9%?%s%9(J
;;;         x-ratio     =  $@?eJ?J}8~$N3HBg(J/$@=L>.G\N((J 
;;;         y-ratio     =  $@?bD>J}8~$N3HBg(J/$@=L>.G\N((J
;;;                        $@<B:]$K3HBg!?=L>.$5$l$kC10L$O(J1/100$@$G$"$k!#(J
;;;                        $@$=$l$h$j:Y$+$$$b$N$O4]$a$i$l$k(J
;;;         create      =  $@%S%C%H%^%C%W$r?7$?$K@8@.$9$k$+(J
;;;                         T -> $@@8@.(J
;;; RET.
;;;         $@%S%C%H%^%C%W%$%s%9%?%s%9(J
(defmethod expange-bitmap ((bitmap bitmap) x-ratio y-ratio 
			   &key (create T))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let* ((xx (round (* x-ratio 100)))
	 (yy (round (* y-ratio 100)))
	 (width (round (* x-ratio (bitmap-width bitmap))))
	 (height (round (* y-ratio (bitmap-height bitmap))))
	 (temp-tno 
#+:YY2.0
	  (make-territory
		    :width width :height height
		    :visible nil
		    :window-mode t
		    :transparent T)
          (make-territory
                    :width width :height height
                    :visible nil
                    :window-mode t)
	  ))
    ;;; $@3HBg!?=L>.(J
    (yy-protocol-38 (bitmap-territory-no bitmap)
		    temp-tno xx yy)

    (if create
	;;;$@@8@.(J
	(let ((ins
	       (make-instance 'bitmap :bitmap-territory-no temp-tno
			      :bitmap-width width :bitmap-height height)))
	  (set-territory-object temp-tno ins)
	  ins)
      (progn 
	(yy-protocol-5 (bitmap-territory-no bitmap))
	(delete-lisp-object (bitmap-territory-no bitmap))
	(set-territory-object temp-tno bitmap)
	(setf (slot-value bitmap 'bitmap-width) width
	      (slot-value bitmap 'bitmap-height) height
	      (slot-value bitmap 'bitmap-territory-no) temp-tno)
	bitmap))
    ))

;;; $@%S%C%H%^%C%W$N2sE>(J
;;; rotate-bitmap bitmap dgree &key create
;;; ARG.
;;;           bitmap   =  $@%S%C%H%^%C%W%$%s%9%?%s%9(J
;;;           dgree    =  $@2sE>3QEY(J ($@%G%#%0%j!<(J)
;;;           create   = $@%S%C%H%^%C%W$r?7$?$K@8@.$9$k$+(J
;;;                      T -> $@@8@.(J
;;; RET.
;;;           $@%S%C%H%^%C%W%$%s%9%?%s%9(J
(defmethod rotate-bitmap ((bitmap bitmap) dgree &key (create T))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let* ((xros (sqrt (+ (* (bitmap-width bitmap)
			   (bitmap-width bitmap))
			(* (bitmap-height bitmap)
			   (bitmap-height bitmap)))))
	 (height (floor (* xros (sin (* pi dgree 180.0)))))
	 (width (floor (* xros (cos (* pi dgree 180.0)))))

	 (temp-tno 
#+ :YY2.0
	  (make-territory
		    :width width :height height
		    :visible nil
		    :window-mode t
		    :transparent T)
	  (make-territory
                    :width width :height height
                    :visible nil
                    :window-mode t)))
    ;;; $@2sE>(J
    (yy-protocol-39 (bitmap-territory-no bitmap)
		    temp-tno (floor (* dgree 64.0)))
    (if create
	;;;$@@8@.(J
	(let ((ins
	       (make-instance 'bitmap :bitmap-territory-no temp-tno
			      :bitmap-width width :bit-map-height height)))
	  (set-territory-object temp-tno ins)
	  ins)
      (progn 
	(yy-protocol-5 (bitmap-territory-no bitmap))
	(delete-lisp-object (bitmap-territory-no bitmap))
	(set-territory-object temp-tno bitmap)
	(setf (slot-value bitmap 'bitmap-width) width
	      (slot-value bitmap 'bitmap-height) height
	      (slot-value bitmap 'bitmap-territory-no) temp-tno)
	bitmap))
    ))


  
;;; $@%$%a!<%8$N<h$j9~$_(J
;;; get-image bitmap position width height
;;; ARG.
;;;          bitmap           =  $@%S%C%H%^%C%W%$%s%9%?%s%9(J
;;;          position         =  $@%S%C%H%^%C%W$G$N0LCV%$%s%9%?%s%9(J
;;;          width            =  $@%$%a!<%8$NI}(J
;;;          height           =  $@%$%a!<%8$N9b$5(J
;;; RET.
;;;          image instance
(defmethod get-image ((bitmap bitmap) (pos position) (width integer)
                      (height integer))
  (declare (inline - + )
	   (function get-image-xy-internal (integer integer integer 
					    integer integer integer 
					    integer) T)
	   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (get-image-xy-internal (bitmap-territory-no bitmap)
                         (position-x pos) (position-y pos)
                         (bitmap-width bitmap)
                         (bitmap-height bitmap) width height)
    )

;;; $@%$%a!<%8$N<h$j9~$_(J XY
;;; get-image bitmap x y width height
;;; ARG.
;;;          bitmap           =  $@%S%C%H%^%C%W%$%s%9%?%s%9(J
;;;          xy               =  $@%S%C%H%^%C%W$G$N0LCV(J
;;;          width            =  $@%$%a!<%8$NI}(J
;;;          height           =  $@%$%a!<%8$N9b$5(J
;;; RET.
;;;          image instance
(defmethod get-image-xy ((bitmap bitmap) (x integer) (y integer) (width integer)
                      (height integer))
  (declare (inline - + )
	   (function get-image-xy-internal (integer integer integer
                                            integer integer integer
                                            integer) T)
	   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (get-image-xy-internal (bitmap-territory-no bitmap)
                         x y
                         (bitmap-width bitmap)
                         (bitmap-height bitmap) width height)
    )

;;; $@%$%a!<%8%G!<%?$r<h$j9~$`(J
(defun get-image-data-bitmap (bitmap x y width height)
  (declare (function get-image-data-xy-internal (integer integer integer
                                            integer integer integer
                                            integer) T)
	   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (get-image-data-xy-internal (bitmap-territory-no bitmap)
                         x y
                         (bitmap-width bitmap)
                         (bitmap-height bitmap) width height)
    )


;;; End of file

