;;; -*- Mode: LISP; Package: YY-GEO; Syntax: Common-Lisp; Base: 10 -*-
(in-package :yy-geo)


;----------------------------------------------------------------------------
;        1  -----  2
;      / |       / |
;    4  -----  3   |
;    |   |     |   |
;    |   8  ---|-  7
;    | /       | /
;    5  -----  6
(defun make-cube (xmin ymin zmin xmax ymax zmax ID)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((body   (make-instance 'polyhedron :ID# ID)))
    (let ((v1    (make-vertex body xmin ymin zmax))
	  (v2    (make-vertex body xmin ymax zmax))
	  (v3    (make-vertex body xmax ymax zmax))
	  (v4    (make-vertex body xmax ymin zmax))
	  (v5    (make-vertex body xmax ymin zmin))
	  (v6    (make-vertex body xmax ymax zmin))
	  (v7    (make-vertex body xmin ymax zmin))
	  (v8    (make-vertex body xmin ymin zmin))
	  )
      (make-square-from-vertices body v1 v2 v3 v4)
      (make-square-from-vertices body v5 v6 v7 v8)
      (make-square-from-vertices body v2 v7 v6 v3)
      (make-square-from-vertices body v8 v1 v4 v5)
      (make-square-from-vertices body v4 v3 v6 v5)
      (make-square-from-vertices body v8 v7 v2 v1)
      )
    body))


;----------------------------------------------------------------------------
(defun make-pseudo-cylinder (x-center y-center radius zmin zmax ID &optional (n 18))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((polyhedron (make-instance 'polyhedron :ID# ID))
	(vector (list 0 0 (- zmin zmax)))
	(ang-unit (/ (* 2 *pi*) n))
	(top-vertices nil)
	(bottom-vertices))
    (dotimes (i n)
      (let ((ang (* ang-unit i)))
	(let ((x (+ x-center (* radius (sin ang))))
	      (y (+ y-center (* radius (cos ang)))))
	  (push (make-vertex polyhedron x y zmax) top-vertices))))
    (setf bottom-vertices
	  (mapcar #'(lambda (x)
		      (make-vertex-from-previous-vector x vector))
		  top-vertices))
    (setf top-vertices (reverse top-vertices))
    (let ((length (length top-vertices)))
      ;; top face
      (make-polygon-from-vertices polyhedron top-vertices)
      ;; bottom face
      (make-polygon-from-vertices polyhedron bottom-vertices)
      (setf bottom-vertices (reverse bottom-vertices))
      (DOTIMES (pos length)
	(let ((pos+1 (+ 1 pos)))
	  (let ((v1 (nth-with-ring pos top-vertices length))
		(v2 (nth-with-ring pos+1 top-vertices length))
		(p1 (nth-with-ring pos bottom-vertices length))
		(p2 (nth-with-ring pos+1 bottom-vertices length)))
	    (make-square-from-vertices polyhedron p1 p2 v2 v1)
	    ))))
    polyhedron))

;----------------------------------------------------------------------------
(defun make-pseudo-sphere (x-center y-center z-center radius ID &optional (n 18))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((polyhedron (make-instance 'polyhedron :ID# ID))
	(vertices-list nil)
	(vertices))
    (dotimes (zi (- (floor n 2) 1))
      (let* ((ang-unit (/ (* 2 *pi*) n))
	     (zang (* ang-unit (+ 1 zi)))
	     (z (+ z-center (* radius (cos zang))))
	     (r (* radius (sin zang))))
	(setf vertices nil)
	(dotimes (i n)
	  (let ((ang (* ang-unit i)))
	    (let ((x (+ x-center (* r (sin ang))))
		  (y (+ y-center (* r (cos ang)))))
	      (push (make-vertex polyhedron x y z) vertices))))
	(push vertices vertices-list)))
    (let* ((bottom-vertices (car vertices-list))
	   (length (length bottom-vertices)))
      ;; bottom face
      (make-polygon-from-vertices polyhedron bottom-vertices)
      (dolist (top-vertices (cdr vertices-list))
	(DOTIMES (pos length)
	  (let ((pos+1 (+ 1 pos)))
	    (let ((v1 (nth-with-ring pos top-vertices length))
		  (v2 (nth-with-ring pos+1 top-vertices length))
		  (p1 (nth-with-ring pos bottom-vertices length))
		  (p2 (nth-with-ring pos+1 bottom-vertices length)))
	      (make-square-from-vertices polyhedron v1 v2 p2 p1))))
	(setf bottom-vertices top-vertices)))
    (let ((top-vertices (car (last vertices-list))))
      ;; top face
      (make-polygon-from-vertices polyhedron (reverse top-vertices)))
    polyhedron))
	

;----------------------------------------------------------------------------
;        e1
;     p1-----p2
;     |      |
;   e4| face |e2
;     |      |
;     p4-----p3
;        e3
(defmethod make-square-from-vertices ((polyhedron polyhedron)
				      (p1 vertex)(p2 vertex)
				      (p3 vertex)(p4 vertex))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((new-face (make-instance 'face
				 :polyhedron polyhedron
				 :vertices (list p1 p2 p3 p4)
				 ))
	(e1       (or (segment-between polyhedron p1 p2) (make-edge polyhedron p1 p2)))
	(e2       (or (segment-between polyhedron p2 p3) (make-edge polyhedron p2 p3)))
	(e3       (or (segment-between polyhedron p3 p4) (make-edge polyhedron p3 p4)))
	(e4       (or (segment-between polyhedron p4 p1) (make-edge polyhedron p4 p1)))
	)
    (set-right-parts-from-vertex e1 p1 e2 new-face e4)
    (set-right-parts-from-vertex e2 p2 e3 new-face e1)
    (set-right-parts-from-vertex e3 p3 e4 new-face e2)
    (set-right-parts-from-vertex e4 p4 e1 new-face e3)
    (with-slots (edges) new-face
      (setf edges (list e1 e2 e3 e4)))
    (values new-face e1 e2 e3 e4)))
;----------------------------------------------------------------------------
; 
;     p1 
;     | \
;   e3|   \e1
;     |face \
;     p3-----p2
;        e2

(defmethod make-triangle-from-vertices ((polyhedron polyhedron)
					(p1 vertex)(p2 vertex)
					(p3 vertex))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((new-face (make-instance 'face
				 :polyhedron polyhedron
				 :vertices (list p1 p2 p3)
				 ))
	(e1       (or (segment-between polyhedron p1 p2) (make-edge polyhedron p1 p2)))
	(e2       (or (segment-between polyhedron p2 p3) (make-edge polyhedron p2 p3)))
	(e3       (or (segment-between polyhedron p3 p1) (make-edge polyhedron p3 p1)))
	)
    (set-right-parts-from-vertex e1 p1 e2 new-face e3)
    (set-right-parts-from-vertex e2 p2 e3 new-face e1)
    (set-right-parts-from-vertex e3 p3 e1 new-face e2)
    (with-slots (edges) new-face
      (setf edges (list e1 e2 e3)))
    (values new-face e1 e2 e3)))

;----------------------------------------------------------------------------
(defmethod make-polygon-from-vertices ((polyhedron polyhedron)
				       (vertices list))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((new-face (make-instance 'face
				 :polyhedron polyhedron
				 :vertices vertices))
	(new-edges nil))
    (let ((length (length vertices)))
      (dotimes (pos length)
	(let ((pos+1 (+ 1 pos)))
	  (let ((p1 (nth-with-ring pos vertices length))
		(p2 (nth-with-ring pos+1 vertices length)))
	    (push (or (segment-between polyhedron p1 p2)
		      (make-edge polyhedron p1 p2))
		  new-edges))))
      (setf new-edges (reverse new-edges))
      (dotimes (pos length)
	(let ((pos+1 (+ pos 1))
	      (pos-1 (- pos 1)))
	  (set-right-parts-from-vertex
	    (nth-with-ring pos new-edges length)
	    (nth-with-ring pos vertices length)
	    (nth-with-ring pos+1 new-edges length)
	    new-face
	    (nth-with-ring pos-1 new-edges length))))
      (with-slots (edges) new-face
	(setf edges new-edges))
      new-face)))

;----------------------------------------------------------------------------
;      \         /		;      \         /		
;ccw-back\     /   back  	; ccw-for\     / for  	
;          \ /			;          \ /			
;            tail		;            head			
;           |			;           |			
;  right    |   left		;   left    |     right
;           |			;           |			
;            head		;            tail			
;          / \			;          / \			
;    for /     \ ccw-for	;  back  /     \ ccw-back
;      /         \		;      /         \			
;
(defmethod set-right-parts-from-vertex ((edge edge) (vertex vertex) (wing edge)
					(face face) (ccw-wing edge))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (head tail right-face left-face
		    for-wing back-wing ccw-for-wing ccw-back-wing) edge
    (cond ((eql vertex tail)
	   (setq for-wing      (or wing     for-wing)
		 right-face    (or face     right-face)
		 ccw-back-wing (or ccw-wing ccw-back-wing))
	   )
	  ((eql vertex head)
	   (setq back-wing     (or wing     back-wing)
		 left-face     (or face     left-face)
		 ccw-for-wing  (or ccw-wing ccw-for-wing))
	   )
	  )))


