;;; -*- Mode: LISP; Syntax: COMMON-LISP; Base: 10.; Package: XIT; Patch-File: T; -*-
;;;_____________________________________________________________________________
;;;
;;;                       System: XIT
;;;                       Module: Window
;;;                       (Version 1.0)
;;;
;;; Copyright (c): Forschungsgruppe DRUID, Juergen Herczeg
;;;                Universitaet Stuttgart
;;;
;;; File: /usr/local/lisp/xit/utilities/mouse-documentation.lisp
;;; File Creation Date: 5/19/90 09:48:11
;;; Last Modification Time: 01/28/93 14:49:02
;;;
;;;
;;; Changes (worth to be mentioned):
;;; ================================
;;; 01/31/1991 (Matthias) added: show-mouse-documentation (string) method
;;;                              with-mouse-documentation 
;;;                                  (docu-string <body>) macro
;;;                              for displaying explicit docu strings
;;;                       changed: remove-mouse-documentation-window
;;; 
;;; 9/07/90  Hubertus  - added test to prevent display operations on a
;;;                      destroyed mouse-documentation-window.
;;; 12/16/1991 (Matthias) - added test in update-mouse-documentation-window
;;;                         and trigger for resize of toplevel
;;;_____________________________________________________________________________

(in-package :xit)

;;;____________________________
;;;
;;; Process Locking Primitives
;;;____________________________

;;; From mpc/process-macros.lisp:

#-excl
(defun make-lock (name)
  (declare (ignore name))
  nil)

#+excl
(defun make-lock (name)
  (mp:make-process-lock :name name))

;;; HOLDING-LOCK: Execute a body of code with a lock held.

#-excl
(defmacro holding-lock ((lock &key timeout) &body body)
  (declare (ignore lock timeout))
  `(progn ,@body))

#+excl 
(defmacro holding-lock ((lock &key timeout) &body body)
  ;; sorry, mp:with-process-lock does not return values of BODY!
  (let ((values (gensym)))
    `(let ((,values nil))
       (mp:with-process-lock (,lock :timeout ,timeout)
	 (setq ,values (multiple-value-list (progn ,@body))))
       (values-list ,values))))

;;; WITHOUT-ABORTS

;;; Inhibits asynchronous keyboard aborts inside the body of the
;;; macro. This macro is wrapped around request writing and reply 
;;; reading to ensure that requests are atomically written and
;;; replies are atomically read from the buffer. 

#-excl
(defmacro without-aborts (&body body)
  `(progn ,@body))

#+excl
(defmacro without-aborts (&body body)
  `(excl:without-interrupts
     ,@body))

 

;_____________________________________________________________________________
;
;                   Mouse Documentation Window
;_____________________________________________________________________________

(defcontact mouse-documentation-window (text-dispel)
   ((name :initform :mouse-documentation)
    (inside-border :initform 5)
    (display-position :initform :upper-left)
    (adjust-size? :initform nil)
    (process :initform nil)
    (changed? :initform t)
    (lock :initform (make-lock "Mouse Documentation Lock")
	  :initarg :lock
	  :reader mouse-documentation-lock)
    (sleep :initform .1 :initarg :sleep :accessor mouse-documentation-sleep)))

(define-resources
  (* mouse-documentation-window background) "black"
  (* mouse-documentation-window foreground) "white"
  (* mouse-documentation-window border-width) 0
  (* mouse-documentation-window width) 1000
  (* mouse-documentation-window font) '(:family :helvetica :face :bold :size 14))

#||
(defmacro protecting-mouse-documentation ((mouse-documentation-window &key timeout)
				 &body body)
  `(holding-lock ((mouse-documentation-lock mouse-documentation-window)
		  :timeout timeout)
		 ,.body))
||#

(defmacro protecting-mouse-documentation ((mouse-documentation-window &key timeout)
				 &body body)
  `(without-aborts
     ,.body))
				    
#+excl
(defmethod initialize-mouse-documentation-process ((self mouse-documentation-window))
  (with-slots (process) self
	      (setf process
		(mp::process-run-function "Mouse Documentation"
					  #'mouse-documentation-loop self))
	      (mp::process-enable process)))

#+excl
(defmethod kill-mouse-documentation-process ((self mouse-documentation-window))
  (with-slots (process view) self
    (when process (mp::process-kill process))))

#||
#+excl
(defmethod mouse-documentation-loop ((self mouse-documentation-window))
  (with-slots (text sleep changed?) self
    (loop
      (when changed?
	 (setf changed? nil)
	 (update self)
	 ;(display-finish-output *display*)
	 ;(display-force-output *display*)
	 )	
      (mp::process-sleep sleep))))
||#

#+excl
(defmethod mouse-documentation-loop ((self mouse-documentation-window))
  (with-slots (text sleep changed?) self
    (loop
      (mp::process-wait "waiting for changes"
			#'(lambda (self)
			    (with-slots (changed?) self
			      changed?))
			self)      
      (setf changed? nil)
      (update self)
	 ;(display-finish-output *display*)
	 ;(display-force-output *display*)
	 )	 
      ))

(defun make-mouse-documentation-window (&optional (parent nil parent-p))
  (declare (special *mouse-documentation-window*))
  (setq *mouse-documentation-window*
      (apply #'make-window `mouse-documentation-window
	     :state :managed
	     (when parent-p `(:parent ,parent))))
  (update-mouse-documentation-window)
  #+excl
  (initialize-mouse-documentation-process *mouse-documentation-window*)
  (totop-window *mouse-documentation-window*))

#+excl
(defmethod destroy :before ((self mouse-documentation-window))
  (kill-mouse-documentation-process self))

;;; Overwrite dummy methods (02/01/1991 (Matthias))

(defmethod show-documentation ((self mouse-documentation-window) documentation)
  (unless (destroyed-p self)
    (with-slots (text changed?) self
      (protecting-mouse-documentation ((mouse-documentation-lock self))
		    #+excl
		    (setf text documentation)
		    #-excl
		    (setf (text self) documentation)
		    (setf changed? t)))))

(defmethod hide-documentation ((self mouse-documentation-window))
  (unless (destroyed-p self)
    (with-slots (text changed?) self
      (protecting-mouse-documentation ((mouse-documentation-lock self))
		    #+excl
		    (setf text "")
		    #-excl
		    (setf (text self) "")
		    (setf changed? t)))))

(defmethod changing-documentation ((self mouse-documentation-window)
				   documentation continuation)
  (with-slots (display process) self
    (let ((saved-mouse-documentation (text self)))
      (unwind-protect
	  (progn
	    (show-documentation self documentation)
	    #+excl
	    (when process (mp::process-reset process))
	    (funcall continuation))
	(show-documentation self saved-mouse-documentation)
	#+excl
	(when process (mp::process-reset process))))))

;;; Update mouse documentation window when toplevel is resized
;;
(defmethod resize :after ((self toplevel-window) width height border-width)
  (declare (ignore width height border-width))
  (update-mouse-documentation-window))

(defun update-mouse-documentation-window ()
  (declare (special *mouse-documentation-window*))
  (when (and *mouse-documentation-window*
	     (not (toplevel-p *mouse-documentation-window*)))
    (with-slots (parent height) *mouse-documentation-window*
     (with-slots ((parent-width width) (parent-height height)) parent
	(newsize-window *mouse-documentation-window*
			0 (- parent-height height) parent-width height)))))


(defun remove-mouse-documentation-window ()
  (declare (special *mouse-documentation-window*))
  (when *mouse-documentation-window*
    (destroy *mouse-documentation-window*)
    (setq *mouse-documentation-window* nil)))
  	       
;_____________________________________________________________________________
;
;              mouse documentation for interaction windows
;_____________________________________________________________________________

(defun remove-mouse-documentation-for (window)
  (when (typep window 'interaction-window)
    (setf (mouse-documentation window) nil))
  (when (typep window 'composite)
    (dolist (child (composite-children window))
      (remove-mouse-documentation-for child))))

(defun mouse-documentation-if-needed-for (window)
  (when (typep window 'interaction-window)
    (setf (compute-mouse-documentation? window) :if-needed))
  (when (typep window 'composite)
    (dolist (child (composite-children window))
      (mouse-documentation-if-needed-for child))))

(defun make-default-mouse-documentation-window (&optional (parent *toplevel*))
  (declare (special *toplevel*))
  (make-mouse-documentation-window parent)
  (mouse-documentation-if-needed-for parent))


;_____________________________________________________________________________
;
;                         dependencies
;_____________________________________________________________________________

(defun remove-mouse-documentation-window-close-hook (display)
  (remove-mouse-documentation-window))

(add-close-display-hook 'remove-mouse-documentation-window-close-hook)

;_____________________________________________________________________________
;
;                      instance creation
;_____________________________________________________________________________


(if (or *multiple-shells-p* (toplevel-initialized-p))
    (make-mouse-documentation-window)
  (add-open-toplevel-hook 'make-mouse-documentation-window))



