;;; -*- Mode: LISP; Syntax: COMMON-LISP; Base: 10.; Package: XIT -*-
;;;_____________________________________________________________________________
;;;
;;;                       System: XIT
;;;                       Module: Toplevel
;;;                       (Version 1.0)
;;;
;;; Copyright (c): Forschungsgruppe DRUID, Juergen Herczeg
;;;                Universitaet Stuttgart
;;;
;;; File: /usr/local/lisp/xit/kernel/toplevel.lisp
;;; File Creation Date: 6/23/89 10:31:37
;;; Last Modification Time: 06/17/93 14:02:59
;;; Last Modification By: Hubertus Hohl
;;;
;;;
;;; Changes (worth to be mentioned):
;;; ================================
;;; 09/03/90 [Kalle] make-toplevel generates now a top-level-shell (with
;;;                  setable shell-title) around the toplevel-window
;;;  
;;; 09/07/90 [Kalle] make-toplevel generalized to parameterize
;;;                  toplevel-class and toplevel-display
;;;
;;; 11/19/90 [Hubertus] adapted EVAL-TOPLEVEL-LOOP for multiple values
;;;
;;; 01/29/1991 (Juergen) the argument toplevel-window for the functions toplevel-loop 
;;;                      and eval-toplevel-loop now defaults to *toplevel*
;;;
;;; 02/12/1991 (Matthias) the global vars *, **, *** are set within
;;;                       eval-toplevel-loop as usually.
;;; 04/25/1991 (Matthias) moved identify-window-with-mouse from identifier.lisp
;;;                       into this file
;;;                       Note that window is selected now by button-p r e s s
;;;
;;; 05/13/1991 (Juergen)  New functions start-event-loop and 
;;;                       start-eval-event-loop have been added
;;;                       They are called from toplevel-loop and 
;;;                       eval-toplevel-loop, respectively.
;;;
;;; 05/13/1991 (Juergen)  The tag quit-toplevel-loop has been changed
;;;                       to quit-event-loop.
;;;
;;; 05/13/1991 (Juergen)  eval-toplevel-loop is now a function and no longer
;;;                       a method.
;;;
;;; 05/13/1991 (Juergen)  keyword toplevel-display of function make-toplevel
;;;                       has been changed to display.
;;;                       New keyword shell-class added.
;;; 09/16/1991 (Matthias) New func: (identify-window *toplevel* ...)
;;;
;;; 10/24/1991 (Hubertus) New gf SPECIFY-REGION-WITH-MOUSE for interactively 
;;;                       specifying a region by rubberbanding.
;;;
;;; 10/29/1991 (Juergen)  New function setup-toplevel which asks for the 
;;;                       x server host, opens a display, and makes a toplevel.
;;; 11/18/1991 (Matthias) New function stop-event-loop, does it gracefully.
;;; 02/10/1992 (Matthias) New mixin for toplevel-window: focus-mixin
;;;                       New: Control-C in toplevel is now handled by focus
;;;                       facility via key-press
;;; 02/11/1992 (Matthias) new method: accept-focus-p
;;;
;;; 02/23/1992 (Hubertus) added method confirm-terminating-application, which is 
;;;                       called whenever the user tries to ``delete the toplevel
;;;                       window'' (e.g. by selecting "QUIT" from a window manager
;;;                       menu). This method should return T, iff the system or
;;;                       user decides to complete the deletion. 
;;;                       As part of the confirmation, this method should also
;;;                       perform application-dependent operations to save its
;;;                       internal state, if needed.
;;;
;;; 12/15/1992 (Juergen)  identify-window has been extended to provide a menu 
;;;                       containing the windows in the selected object's
;;;                       window hierarchy when the right mouse button
;;;                       is used for selection.
;;;
;;; 01/21/1993 (Juergen)  New functions make-shell which is used used to
;;;                       create shells, e.g. by make-toplevel, and
;;;                       shell-class which returns the default shell class
;;;                       for a window class.
;;;
;;; 01/21/1993 (Juergen)  New global variable *multiple-shells-p* which
;;;                       determines whether windows are to be created 
;;;                       within a private shell or in the global toplevel.
;;;
;;; 01/25/1993 (Juergen)  New functions event-loop and eval-loop, which
;;;                       subsume start-event-loop, toplevel-loop, 
;;;                       eval-toplevel-loop, and eval-toplevel-loop
;;;                       depending on whether *toplevel* is opened or not.
;;;
;;; 01/26/1993 (Juergen)  New function toplevel-initialized-p has been
;;;                       introduced in anology to display-initialized-p.
;;;                       These functions should be used instead of
;;;                       (boundp *display*), (boundp *toplevel*), etc.
;;; 06/17/1993 (Hubertus) specialized method specify-region-with-mouse on 
;;;                       composite instead of toplevel-window 
;;;                       (needed for own shells)
;;;
;;;__________________________________________________________________________

(in-package :xit)

;____________________________________________________________________________

;____________________________________________________________________________
;
;                              Root Support
;____________________________________________________________________________

(defmethod root-p ((self root))
  t)

(defmethod root-p ((self basic-contact))
  nil)

;____________________________________________________________________________
;
;                              Shell Support
;____________________________________________________________________________

(defmethod shell-class ((class-name symbol))
  (if (subtypep class-name 'popup-window)
      'override-shell
    'top-level-shell))

(defun make-shell (&key (display *display*)
			(shell-class 'top-level-shell)
			(shell-title "XIT Shell")
			shell-init-list)
  (declare (special *display*))
  ;;(prog1
      (apply #'make-contact shell-class
	     :parent display			
	     :width 10 :height 10 ;; this is needed, when shell client is not
	                           ;; mapped (hack?)
	     :wm-title shell-title
	     `(,@shell-init-list
	       :cursor "arrow"
	       :background "white"
	       :wm-protocols-used (:wm_take_focus
				   ;; :wm_save_yourself
				   :wm_delete_window) 
	       :wm-keyboard-input :on
	       :wm-icon-title ,shell-title))
    ;;(update-state display))
      )

(defmethod shell-p ((self shell))
  t)

(defmethod shell-p ((self basic-contact))
  nil)

(defmethod client-window ((self shell) &key &allow-other-keys)
  (first (composite-children self)))

(defmethod toplevel-window ((self shell))
  (first (composite-children self)))

(defmethod delete-child :after ((self shell) contact &key)
  (declare (ignore contact))
  (with-slots (children) self
    ;; destroy the shell when it has no more children
    (unless children
      (destroy self))))

(defmethod change-layout :after ((shell shell) &optional client)
  (when client
    (with-slots (display (shell-state state)) shell
      (with-slots ((client-state state)) client
	(if (eq shell-state :mapped)
	    (case client-state
	      (:withdrawn (setf (contact-state shell) :withdrawn))
	      (:managed (setf (contact-state shell) :withdrawn)
			(setf (contact-state client) :withdrawn)))
	  (when (eq client-state :mapped)
	    (setf (contact-state shell) :mapped)))))))

#||
;; the following may be necessary

(defmethod realize :after ((shell shell))
  (let ((client (client-window self)))
    (when client
    (with-slots ((shell-state state)) shell
      (with-slots ((client-state state)) client
	(if (eq shell-state :mapped)
	    (when (member client-state '(:managed :withdrawn))
	      (setf (contact-state shell) :withdrawn))
	  (when (eq client-state :mapped)
	    (setf (contact-state shell) :mapped))))))))
||#

;;;____________________________________________
;;;
;;; Handling Interclient messages from the WM
;;;____________________________________________

;;; 02/23/1992 (Hubertus)  TODO: 
;;;(defevent top-level-shell (:wm_save_yourself) saving-state)

(defevent top-level-shell (:wm_delete_window) check-terminating-application)

(defmethod check-terminating-application ((shell top-level-shell))
  (with-slots (cluei::children) shell
    (when cluei::children
      (check-terminating-application (first cluei::children)))))

(defmethod check-terminating-application ((self basic-contact))
  ;; should return T, iff application has been terminated
  (when (confirm-terminating-application self)
    (destroy (contact-parent self)) ;; destroy the shell    
      t))
    
(defmethod confirm-terminating-application ((self basic-contact))
  ;; to be filled by applications
  T)

;;; 
;;; Some Notes on Input Focus Management: 02/23/1992 (Hubertus)
;;; (also see X11 Interclient Communication Conventions)
;;; 
;;; There are four models of input handling:
;;;
;;; `No Input' - client never expects keyboard input.
;;; `Passive Input' - client expects keyboard input, but never explicitely 
;;;                   sets the input focus.
;;; `Locally Active Input' - client expects keyboard input and explicitely sets 
;;;                          the input focus, but does only so, when one of its 
;;;                          windows already has the focus.
;;; `Globally Active Input' - client expects keyboard input and explicitely sets 
;;;                          the input focus, even when it is in windows the client ;;;                          does not own.
;;;
;;; Currently, XIT uses the `Locally Active Input' model of input handling.
;;; (indicated by :wm-keyboard :on and wm_take_focus present in WM_PROTOCOLS).

;____________________________________________________________________________
;
;                               Toplevel Window
;____________________________________________________________________________

(defvar *multiple-shells-p* nil
        "determines whether windows are created in private shells")

(defcontact toplevel-window (token-event-handler-mixin focus-mixin
			     interaction-window composite)
     ((name :initform :toplevel)
      (reactivity :initform '((:keyboard "C-c: Quit event loop"))))
     (:documentation "class for windows on the toplevel"))

(define-resources
  (* toplevel-window x) 150
  (* toplevel-window y) 150
  (* toplevel-window width) 900
  (* toplevel-window height) 700
  (* toplevel-window background) .25    ;; 25%gray
  (* toplevel-window cursor) "X_cursor")

(defmethod toplevel-window ((self toplevel-window))
  self)

(defmethod toplevel-p ((self toplevel-window))
  t)

(defmethod check-terminating-application ((self toplevel-window))
  ;; should return T, iff application has been terminated
  (when (confirm-terminating-application self)
    (let ((display (contact-display self)))
      (destroy (contact-parent self)) ;; destroy the shell    
      (stop-event-loop display)
      t)))

;;; The following is needed very urgently when no key-event is defined for
;;; toplevel-windows
(defmethod accept-focus-p ((contact toplevel-window))
  "Returns non-nil when CONTACT is willing to become the keyboard input focus"
  (declare (values boolean))
  (and (cluei::viewable-p contact)
       (cluei::sensitive-p contact)))

(defmethod key-press ((self toplevel-window) key)
  (case key
    (#\Control-\c (stop-event-loop (contact-display self)))))

(defun make-toplevel (&key (display *display*)
			   (shell-class)
			   (shell-title "XIT Toplevel")
			   (toplevel-class 'toplevel-window)
			   toplevel-init-list
			   shell-init-list)
  (declare (special *display* *shell* *toplevel*))
  (let* ((shell (make-shell :display display
			    :shell-class (or shell-class
					     (shell-class toplevel-class))
			    :shell-title shell-title
			    :shell-init-list shell-init-list))
	 (toplevel (apply #'make-contact toplevel-class
			  :parent shell toplevel-init-list)))
    ;; bind globals
    (setq *shell* shell)
    (setq *toplevel* toplevel)
    ;; the following is needed for CLX windows created by toplevel hooks
    (update-state display)
    (call-open-toplevel-hooks toplevel)
    (update-state display)
    (process-all-events display)
    toplevel))

#||
(defun make-toplevel (&key (display *display*)
			   (shell-class 'top-level-shell)
			   (shell-title "XIT Toplevel")
			   (toplevel-class 'toplevel-window)
			   toplevel-init-list
			   shell-init-list)
  (declare (special *toplevel* *display* *shell*))
  (setq *shell* (apply #'make-contact shell-class
			      :parent display			
			      :width 10 :height 10 ; don't know if necessary
			      :wm-protocols-used '(:wm_take_focus
						   ;; :wm_save_yourself
						   :wm_delete_window) 
			      :wm-keyboard-input :on
			      :wm-title shell-title
			      :wm-icon-title shell-title
			      shell-init-list))
  (setq *toplevel* (apply #'make-contact toplevel-class
			  :parent *shell* toplevel-init-list))
  (update-state display)  ; this is needed for CLX windows created by toplevel hooks
  (call-open-toplevel-hooks *toplevel*)
  (update-state display)
  (process-all-events display)
  *toplevel*)
||#

(defun toplevel-initialized-p (&optional toplevel)
  (declare (special *toplevel*))
  (let ((top (or toplevel
		 (and (boundp '*toplevel*)
		      *toplevel*))))
    (and top
	 (display-initialized-p (contact-display top))
	 (not (destroyed-p top)))))

(defun setup-toplevel ()
  (query-x-server-host)
  (open-toplevel-display)
  (make-toplevel))
  

(defun make-and-loop-toplevel ()
  (make-toplevel)
  (toplevel-loop))

(defun start-event-loop (&optional (display *display*))
  (declare (special *display*))
  (format t "~%-- Starting Event Loop --~%")
  (catch 'quit-event-loop
    (loop (process-next-event display)))
  (format t "~%-- Stopping Event Loop --~%")
  display)

(defvar *eval-loop-timeout* 0.1
  "Timeout in seconds used for process-next-event in the eval-loop")
;; 0.1 has been derived from experience (formerly 1).
;; Low values enable timer events in short intervals;
;; very small values (~0) increase cpu load of lisp process.
;; High values produce longer sleep times.

(defun start-eval-event-loop (&optional (display *display*))
  (declare (special *display*))
  (format t "~%-- Starting Event and Read Loop --~%~%<eval:> ")
  (catch 'quit-event-loop
    (loop (cond ((process-next-event display *eval-loop-timeout*))
		((listen)	   
		 (format t "~{~S~%~}~&<eval:> "
			 (let ((result
				(multiple-value-list (eval (read)))))
			   (setq *** **)
			   (setq ** *)
			   (setq * (car result))
			   result))))))
  (format t "~%-- Stopping Event and Read Loop --~%")
  display)

(defun stop-event-loop (&optional (display *display*))
  (declare (special *display*))
  (ignoring-errors (ungrab-pointer display)
		   (throw 'quit-event-loop nil)))

(defun toplevel-loop (&optional (toplevel *toplevel*))
  (declare (special *toplevel*))
  (change-window-cursor toplevel "arrow")
  (start-event-loop (contact-display toplevel))
  (change-window-cursor toplevel "X_cursor")
  toplevel)

(defun eval-toplevel-loop (&optional (toplevel *toplevel*))
  (declare (special *toplevel*))
  (change-window-cursor toplevel "arrow")
  (start-eval-event-loop (contact-display toplevel))
  (change-window-cursor toplevel "X_cursor")
  toplevel)

(defun event-loop ()
  (if (toplevel-initialized-p)
    (toplevel-loop)
    (start-event-loop)))

(defun eval-loop ()
  (if (toplevel-initialized-p)
    (eval-toplevel-loop)
    (start-eval-event-loop)))

;__________________________________________________________________________
; 
;                          Specifying Regions
;__________________________________________________________________________

;;; 10/24/1991 (Hubertus)  
;;; Interactively specifying a region by rubberbanding.
;;;

(defmethod specify-region-with-mouse ((toplevel composite) &key
				      confine-to
				      resize-only?
				      (initial-region (region 0 0 0 0))
				      (minimum-width 0)
				      (minimum-height 0)
				      (maximum-width most-positive-fixnum)
				      (maximum-height most-positive-fixnum)
				      (line-style :solid)
				      (line-width 1)
				      (button-action :button-press))
  
  "Allows the user to specify a region with the mouse (relative to the 
   window specified by CONFINE-TO). After the mouse is grabbed 
   rubber-banding occurs until a mouse button is pressed.
   If the left button is pressed, the new region is returned. The user may 
   abort by pressing the middle button, returning NIL.
   To move the opposite corner of the region too, hold down the SHIFT key.
   The region is restricted to the window specified by CONFINE-TO (default
   TOPLEVEL). 
   If an INITIAL-REGION is specified, it is used to store the resulting values.
   MINIMUM-xxx and MAXIMUM-XXX specify constraints on the region size.
   If RESIZE-ONLY? is specified as non-NIL, the upper left corner of the region
   remains fixed."

  (declare (values (or null region))
	   (special *inversion-pixel*))
  (setq confine-to (or confine-to toplevel))
  (maxf (region-w initial-region) minimum-width)
  (minf (region-w initial-region) maximum-width)
  (maxf (region-h initial-region) minimum-height)
  (minf (region-h initial-region) maximum-height)
  (with-slots (display) toplevel
    (using-gcontext (gc :drawable confine-to
			:subwindow-mode :include-inferiors
			:function BOOLE-XOR
			:foreground *inversion-pixel*
			:line-width line-width
			:line-style line-style
			:dashes nil)
      (let ((width (region-w initial-region))
	    (height (region-h initial-region))
	    (x-pos (region-x initial-region))
	    (y-pos (region-y initial-region)))
	(warp-pointer confine-to (+ x-pos width) (+ y-pos height))
	(process-all-events display)
	(dragging-mouse
	 (confine-to :cursor "sizing"
		     :drag (eq button-action :button-release)
		     :abort-events '((:button-press :button-2 *))
		     :mouse-documentation
		     (format nil "Mouse-L: Position corner of rectangle;  ~
                                        Mouse-M: Abort~
                                        ~:[;   Hold down Shift to move opposite corner too.~;.~]"
			     resize-only?))
	 (:before () (draw-rectangle-inside confine-to gc x-pos y-pos width height))
	 (:dragged (state x y) (draw-rectangle-inside confine-to gc x-pos y-pos width height)
			(if (and (not resize-only?)
				   (member state
					   '#.`(,(make-state-mask :shift :button-1)
						,(make-state-mask :shift))
					   :test #'=))
			      ;; shift hold down, move opposite corner too

			      (setq x-pos (- x width)
				    y-pos (- y height))
			    (setq width (min maximum-width
					     (max minimum-width
						  (- x x-pos)))
				  height (min maximum-height
					      (max minimum-height
						   (- y y-pos)))))
			(draw-rectangle-inside confine-to gc x-pos y-pos width height))
	 (:abort () (draw-rectangle-inside confine-to gc x-pos y-pos width height)
		 nil)
	 (:after () (draw-rectangle-inside confine-to gc x-pos y-pos width height)
		 (setf (region-x initial-region) x-pos
		       (region-y initial-region) y-pos
		       (region-w initial-region) width
		       (region-h initial-region) height)
		 initial-region))))))

#|| Alt und (hin-)laenglich:
(defmethod specify-region-with-mouse ((toplevel toplevel-window) &key
				      confine-to
				      resize-only?
				      (initial-region (region 0 0 0 0))
				      (minimum-width 0)
				      (minimum-height 0)
				      (maximum-width most-positive-fixnum)
				      (maximum-height most-positive-fixnum)
				      (line-style :solid)
				      (line-width 1)
				      (button-action :button-press))
  
  "Allows the user to specify a region with the mouse (relative to the 
   window specified by CONFINE-TO). After the mouse is grabbed 
   rubber-banding occurs until a mouse button is pressed.
   If the left button is pressed, the new region is returned. The user may 
   abort by pressing the middle button, returning NIL.
   To move the opposite corner of the region too, hold down the SHIFT key.
   The region is restricted to the window specified by CONFINE-TO (default
   TOPLEVEL). 
   If an INITIAL-REGION is specified, it is used to store the resulting values.
   MINIMUM-xxx and MAXIMUM-XXX specify constraints on the region size.
   If RESIZE-ONLY? is specified as non-NIL, the upper left corner of the region
   remains fixed."

  (declare (values (or null region))
	   (special *inversion-pixel*))
  (setq confine-to (or confine-to toplevel))
  (maxf (region-w initial-region) minimum-width)
  (minf (region-w initial-region) maximum-width)
  (maxf (region-h initial-region) minimum-height)
  (minf (region-h initial-region) maximum-height)
  (with-slots (display) toplevel
    (using-gcontext (gc :drawable confine-to
			:subwindow-mode :include-inferiors
			:function BOOLE-XOR
			:foreground *inversion-pixel*
			:line-width line-width
			:line-style line-style
			:dashes nil)
      (let ((width (region-w initial-region))
	    (height (region-h initial-region))
	    (x-pos (region-x initial-region))
	    (y-pos (region-y initial-region)))
	(warp-pointer confine-to (+ x-pos width) (+ y-pos height))
	(process-all-events display)
	(if (eq :success
		(grab-pointer confine-to
			      '(:button-press :button-release :pointer-motion) 
			      :owner-p t
			      :confine-to confine-to
			      :sync-pointer-p t
			      :cursor (convert confine-to "sizing" 'cursor)
			      :time nil))
	    (with-mouse-documentation ("Mouse-L: Position corner of rectangle;  ~
                                        Mouse-M: Abort~
                                        ~:[;   Hold down Shift to move opposite corner too.~;.~]"
				       resize-only?)
	      (draw-rectangle-inside confine-to gc x-pos y-pos width height)
	      (catch 'abort
		(unwind-protect
		    (flet ((button-action (code)
			     (case code
			       (1 (draw-rectangle-inside confine-to gc x-pos y-pos width height)
				  T)
			       (2 ;; abort if middle button released
				(draw-rectangle-inside confine-to gc x-pos y-pos width height)
				(discard-current-event display)
				(throw 'abort nil))
			       (otherwise nil))))
		      (allow-events display :async-pointer)
		      (event-case (display :discard-p t :force-output-p t)
			(motion-notify (x y event-window state)
			(draw-rectangle-inside confine-to gc x-pos y-pos width height)
			(multiple-value-bind (confine-to-x confine-to-y)
			    (contact-translate event-window x y confine-to)
			  (if (and (not resize-only?)
				   (member state
					   '#.`(,(make-state-mask :shift :button-1)
						,(make-state-mask :shift))
					   :test #'=))
			      ;; shift hold down, move opposite corner too
			      (setq x-pos (- confine-to-x width)
				    y-pos (- confine-to-y height))
			    (setq width (min maximum-width
					     (max minimum-width
						  (- confine-to-x x-pos)))
				  height (min maximum-height
					      (max minimum-height
						   (- confine-to-y y-pos))))))
			(draw-rectangle-inside confine-to gc x-pos y-pos width height)
			nil) 
			(button-press (code)
			 (when (eq button-action :button-press)
			   (button-action code)))
			(button-release (code)
			 (when (eq button-action :button-release)
			   (button-action code)))
			))
		  (ungrab-pointer display))
		(setf (region-x initial-region) x-pos
		      (region-y initial-region) y-pos
		      (region-w initial-region) width
		      (region-h initial-region) height)
		initial-region))
	  (progn
	    (ungrab-pointer display)
	    nil))))))
||#
