;;; -*- Mode: LISP; Syntax: COMMON-LISP; Base: 10.; Package: XIT -*-
;;;_____________________________________________________________________________
;;;
;;;                       System: XIT
;;;                       Module: Drag&Drop
;;;                       Version: 1.0
;;;
;;; Copyright (c): Forschungsgruppe DRUID, Juergen Herczeg
;;;                Universitaet Stuttgart
;;;
;;; File: /usr/local/lisp/xit/kernel/drag-and-drop.lisp
;;; File Creation Date: 02/26/93 09:18:53
;;; Last Modification Time: 07/05/93 16:25:07
;;; Last Modification By: Juergen Herczeg
;;;
;;;
;;; Changes (worth to be mentioned):
;;; ================================
;;;
;;;_____________________________________________________________________________

(in-package :xit)

;_______________________________________________________________________________

(defvar *drag-and-drop-default-type* nil) ;; (member :move :reparent nil)

;;(defvar *drag-and-drop-copy-p* nil)

(defmethod drag-window-p ((self interaction-window))
  (reactivity-entry self :drag-event))

(defmethod drag-window-p (self)
  (declare (ignore self))
  nil)

(defmethod drop-window-p ((self interaction-window))
  (reactivity-entry self :drop-event))

(defmethod drop-window-p (self)
  (declare (ignore self))
  nil)

(defmethod in-drop-window-p ((self interaction-window))
  (with-slots (parent) self
    (or (drop-window-p self)
	(in-drop-window-p parent))))

(defmethod in-drop-window-p (self)
  (declare (ignore self))
  nil)

(defmethod drag-and-drop-window ((self interaction-window)
				 &key
				 (in-bounds-p *move-window-with-mouse-in-bounds*)
				 confine-to constraint cursor
				 (mouse-position :warp)
				 (mouse-documentation
				  "Drag window with mouse. Any key aborts.")
				 (abort-events '((:key-press * *)))
				 (feedback? nil feedback?-p)
				 (test nil test-p)
				 type
				 (drop-on-part-p nil)
				 (action '(:drag :drop :default))
				 fail-action)
  (mouse-feedback-off self) ;; due to problem in move-window method
  (multiple-value-bind (x y drop-window found-p)
      (move-windows-with-mouse (contact-root self)
			       (list self)
			       :style :drag-and-drop
			       :confine-to confine-to
			       :in-bounds-p in-bounds-p
			       :constraint constraint
			       :cursor cursor
			       :mouse-position mouse-position
			       :mouse-documentation mouse-documentation
			       :abort-events abort-events
			       :feedback? (if feedback?-p
					      feedback?
					    (or type test))
			       :type type
			       :test test
			       ;;(if test-p test
			       ;;   (unless (drag-window-p self) #'drop-window-p))
			       )
    (if found-p
	(if (and (not drop-on-part-p)
		 (or (eq self drop-window)
		     (ancestor-p drop-window self)))
	    (drag-and-drop-action self (and found-p drop-window) x y :default)
	  (drag-and-drop-action self (and found-p drop-window) x y action))
      (when fail-action
	(funcall fail-action drop-window)))))

(define-call-action :drag-and-drop (&rest values)
  `(drag-and-drop-window *self* ,@values))
    
(defmethod drag-and-drop-action ((drag-window interaction-window)
				 drop-window
				 x y
				 (action (eql nil)))
  (values nil drop-window x y)) ;; nothing to do

(defmethod drag-and-drop-action ((drag-window interaction-window)
				 drop-window
				 x y
				 (action (eql :default)))
  (unless (eq *drag-and-drop-default-type* :default)
    (drag-and-drop-action drag-window drop-window x y
			  *drag-and-drop-default-type*)))
  
(defmethod drag-and-drop-action ((drag-window interaction-window)
				 drop-window
				 x y
				 (action (eql :move)))
  (with-slots (parent) drag-window
    (let ((new-x x)
	  (new-y y))
      (when (and drop-window (not (eq drop-window parent)))
	(multiple-value-bind (drop-x drop-y)
	    (contact-translate drop-window x y parent)
	  (setq new-x drop-x
		new-y drop-y)))
      (move-window drag-window new-x new-y)))
  (values :move drop-window x y))
  
(defmethod drag-and-drop-action :around ((drag-window interaction-window)
					 (drop-window composite)
					 x y
					 (action (eql :reparent)))
  (if (or (eq drop-window drag-window)
	  (eq drop-window (contact-parent drag-window))
	  (ancestor-p drop-window drag-window))
	(drag-and-drop-action drag-window drop-window x y :move)  
    (call-next-method)))
  
(defmethod drag-and-drop-action ((drag-window interaction-window)
				 (drop-window composite)
				 x y
				 (action (eql :reparent)))
  (setf (contact-parent drag-window) drop-window)
  (move-window drag-window x y)
  (values :reparent drop-window x y))
   
(defmethod drag-and-drop-action ((drag-window interaction-window)
				 (drop-window layouted-window)
				 x y
				 (action (eql :reparent)))
  (with-final-layout drop-window
    (setf (contact-parent drag-window) drop-window)
    (move-window drag-window x y))
  (values :reparent drop-window x y))
  
(defmethod drag-and-drop-action ((drag-window interaction-window)
				 drop-window
				 x y
				 (action (eql :reparent)))
  (drag-and-drop-action drag-window drop-window x y :move))
 
(defmethod drag-and-drop-action ((drag-window interaction-window)
				 drop-window
				 x y
				 (action (eql :copy)))
  (with-slots (parent) drag-window
    (let ((new-x x)
	  (new-y y))
      (when (and drop-window (not (eq drop-window parent)))
	(multiple-value-bind (drop-x drop-y)
	    (contact-translate drop-window x y parent)
	  (setq new-x drop-x
		new-y drop-y)))
      (copy-and-move-window drag-window new-x new-y)))
  (values :copy drop-window x y))
  
(defmethod drag-and-drop-action :around ((drag-window interaction-window)
					 (drop-window composite)
					 x y
					 (action (eql :copy-and-reparent)))
  (if (or (eq drop-window drag-window)
	  (eq drop-window (contact-parent drag-window))
	  (ancestor-p drop-window drag-window))
	(drag-and-drop-action drag-window drop-window x y :copy)  
    (call-next-method)))
  
(defmethod drag-and-drop-action ((drag-window interaction-window)
				 (drop-window composite)
				 x y
				 (action (eql :copy-and-reparent)))
  (copy-window-to-parent drag-window drop-window x y)
  (values :copy-and-reparent drop-window x y))
  
(defmethod drag-and-drop-action ((drag-window interaction-window)
				 drop-window
				 x y
				 (action (eql :copy-and-reparent)))
  (drag-and-drop-action drag-window drop-window x y :copy))

(define-event-key :drag-event ())

(define-event-key :drop-event ())

(define-event-key :drag-and-drop
  (:mouse-documentation-prefix "Mouse-M:"
   :default-mouse-documentation "Drag&drop window")
  (:single-middle-button :default-actions drag-and-drop-window))

(defmethod send-drag-event ((drag-window basic-contact)
			    &optional drop-window x y)
  (call-event-action drag-window :drag-event drop-window x y))
 
(defmethod send-drop-event ((drop-window basic-contact)
			    &optional drag-window x y)
  (call-event-action drop-window :drop-event drag-window x y))
  
(defmethod drag-and-drop-action ((drag-window interaction-window)
				 drop-window 
				 x y
				 (action (eql :drag)))
  (when (drag-window-p drag-window)
    (send-drag-event drag-window drop-window x y)
    (values :drag drop-window x y)))

(defmethod drag-and-drop-action ((drag-window interaction-window)
				 (drop-window interaction-window)
				 x y
				 (action (eql :drop)))
  (when (in-drop-window-p drop-window)
    (send-drop-event drop-window drag-window x y)
    (values :drop drop-window x y)))

(defmethod drag-and-drop-action ((drag-window interaction-window)
				 drop-window
				 x y
				 (action (eql :drop)))
  nil)

(defmethod drag-and-drop-action ((drag-window interaction-window)
				 drop-window
				 x y
				 (action function))
  (funcall action drag-window drop-window x y)
  (values :action action drop-window x y))
 
(defmethod drag-and-drop-action ((drag-window interaction-window)
				 drop-window
				 x y
				 (action list))
  ;; perform all actions in the list until the first returns t
  (dolist (act action)
    (let ((results
	   (multiple-value-list
	       (drag-and-drop-action drag-window drop-window x y act))))
      (when (first results)
	(return (values-list results))))))
