; -*-Emacs-Lisp-*-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
; File:         xlisp-mode.el
; RCS:          $Header: xlisp-mode.el,v 1.1 89/08/29 02:59:18 mayer Exp $
; Description:  Stuff for running XLISP in a subshell
; Author:       Niels Mayer, HPLabs
; Created:      Thu Jun  8 18:16:00 1989
; Modified:     Tue Jul 11 01:13:49 1989 (Niels Mayer) mayer@hplnpm
; Language:     Emacs-Lisp
; Package:      N/A
; Status:       Experimental (Do Not Distribute)
;
; (c) Copyright 1988, Hewlett-Packard Company, all rights reserved.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(require 'shell)			; I'm assuming this is the standard
					; 18.54 version of shell.el.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Code to run 'xlisp' process as a gnuemacs subshell. This is essentially
;;; the same as the code to run a common lisp process as a gnuemacs subshell.
;;; However, with this code, you may run other lisps alongside xlisp.
;;; (code stolen & mutated from emacs/lisp/shell.el)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar inferior-xlisp-mode-map nil)
(if inferior-xlisp-mode-map
    nil
  (setq inferior-xlisp-mode-map (copy-alist shell-mode-map))
  (lisp-mode-commands inferior-xlisp-mode-map)
  (define-key inferior-xlisp-mode-map "\e\C-x" 'xlisp-send-defun))

(defvar inferior-xlisp-program "xlisp"
  "*Program name for invoking an inferior Lisp with `run-xlisp'.")

(defvar inferior-xlisp-load-command
  "(load \"%s\" :verbose nil :print t)\n"
  "*Format-string for building a Xlisp expression to load a file.
This format string should use %s to substitute a file name
and should result in a Xlisp expression that will command the inferior Xlisp
to load that file.  The default works acceptably on most Lisps.
The string \"(progn (load \\\"%s\\\" :verbose nil :print t) (values))\\\n\"
produces cosmetically superior output for this application,
but it works only in Common Lisp.")

(defvar inferior-xlisp-prompt "^.*>:? *$"
  "*Regexp to recognize prompts from the inferior Lisp.
Default is right for Franz Lisp and kcl.")

(defun inferior-xlisp-mode ()
  "Major mode for interacting with an inferior Xlisp process.
Runs a Xlisp interpreter as a subprocess of Emacs, with Xlisp I/O
through an Emacs buffer.  Variable inferior-xlisp-program controls
which Xlisp interpreter is run.  Variables inferior-xlisp-prompt
and inferior-xlisp-load-command can customize this mode for different
Lisp interpreters.

Commands:
DELETE converts tabs to spaces as it moves back.
TAB indents for Lisp; with argument, shifts rest
 of expression rigidly with the current line.
Meta-Control-Q does TAB on each line starting within following expression.
Paragraphs are separated only by blank lines.  Semicolons start comments.

Return at end of buffer sends line as input.
Return not at end copies rest of line to end and sends it.

The following commands imitate the usual Unix interrupt and
editing control characters:
\\{shell-mode-map}

Entry to this mode calls the value of xlisp-mode-hook with no arguments,
if that value is non-nil.  Likewise with the value of shell-mode-hook.
xlisp-mode-hook is called after shell-mode-hook.

You can send text to the inferior Xlisp from other buffers
using the commands process-send-region, process-send-string
and \\[xlisp-send-defun]."
  (interactive)
  (kill-all-local-variables)
  (setq major-mode 'inferior-xlisp-mode)
  (setq mode-name "Inferior Xlisp")
  (setq mode-line-process '(": %s"))
  (lisp-mode-variables t)
  (use-local-map inferior-xlisp-mode-map)
  (make-local-variable 'last-input-start)
  (setq last-input-start (make-marker))
  (make-local-variable 'last-input-end)
  (setq last-input-end (make-marker))
  (run-hooks 'shell-mode-hook 'xlisp-mode-hook))

(defun run-xlisp ()
  "Run an inferior Xlisp process, input and output via buffer *xlisp*."
  (interactive)
  (switch-to-buffer (make-shell "xlisp" inferior-xlisp-program))
  (inferior-xlisp-mode))

(defun xlisp-send-defun (no-display-flag)
  "Send the current defun to the Xlisp process made by M-x run-xlisp.
With argument, suppress redisplay and scrolling of the *xlisp* buffer.
Variable `inferior-xlisp-load-command' controls formatting of
the `load' form that is set to the Xlisp process."
  (interactive "P")
  (or (get-process "xlisp")
      (error "No current xlisp process"))
  (save-excursion
   (end-of-defun)
   (let ((end (point))
	 (filename (format "/tmp/emlisp%d.lsp" (process-id (get-process "xlisp")))))
     (beginning-of-defun)
     (write-region (point) end filename nil 'nomessage)
     (process-send-string "xlisp" (format inferior-xlisp-load-command filename)))
   (if (not no-display-flag)
       (let* ((process (get-process "xlisp"))
	      (buffer (process-buffer process))
	      (w (or (get-buffer-window buffer) (display-buffer buffer)))
	      (height (window-height w))
	      (end))
	 (save-excursion
	   (set-buffer buffer)
	   (setq end (point-max))
	   (while (progn
		    (accept-process-output process)
		    (goto-char (point-max))
		    (beginning-of-line)
		    (or (= (point-max) end)
			(not (looking-at inferior-xlisp-prompt)))))
	   (setq end (point-max))
	   (vertical-motion (- 4 height))
	   (set-window-start w (point)))
	 (set-window-point w end)))))

(defun xlisp-send-defun-and-go ()
  "Send the current defun to the inferior Xlisp, and switch to *xlisp* buffer."
  (interactive)
  (xlisp-send-defun t)
  (switch-to-buffer "*xlisp*"))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; This code sets up xlisp-mode when visiting a file with extension .lsp.
;;; This is exactly the same as gnuemacs' normal lisp mode; I just wanted
;;; to avoid conflict with other lisps that you might be using under gnuemacs,
;;; such as common lisp.
;;; The following was stolen&mutated from emacs/lisp/lisp-mode.el
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar xlisp-mode-map ())
(if xlisp-mode-map
    ()
  (setq xlisp-mode-map (make-sparse-keymap))
  (define-key xlisp-mode-map "\e\C-x" 'xlisp-send-defun)
  (define-key xlisp-mode-map "\C-c\C-l" 'run-xlisp)
  (lisp-mode-commands xlisp-mode-map))

(defun xlisp-mode ()
  "Major mode for editing Xlisp code.
Commands:
Delete converts tabs to spaces as it moves back.
Blank lines separate paragraphs.  Semicolons start comments.
\\{xlisp-mode-map}
Note that `run-xlisp' may be used either to start an inferior Xlisp job
or to switch back to an existing one.

Entry to this mode calls the value of xlisp-mode-hook
if that value is non-nil."
  (interactive)
  (kill-all-local-variables)
  (use-local-map xlisp-mode-map)
  (setq major-mode 'xlisp-mode)
  (setq mode-name "Xlisp")
  (lisp-mode-variables t)
  (set-syntax-table lisp-mode-syntax-table)
  (run-hooks 'xlisp-mode-hook))

;;
;; All files with extension .lsp are assumed to be XLISP files, to be edited in
;; xlisp-mode.
;; Note -- this overrides a binding in loaddefs.el which will put files with
;; extension .lsp in lisp-mode
;;
(setq auto-mode-alist
      (append '(("\\.lsp$" . xlisp-mode)
		)
	      auto-mode-alist))
