/* zortstuf.c - Zortech-C specific routines */

#include "xlisp.h"
#include "osdefs.h"
#include <string.h>
#include <dos.h>
#include <stdlib.h>

#define LBSIZE 200

unsigned _stack = 16384;		/* set up reasonable stack */

/* external variables */
extern LVAL s_unbound,s_dosinput,true;
extern FILE *tfp;


/* local variables */
static char lbuf[LBSIZE];
static int lpos[LBSIZE];
static int lindex;
static int lcount;
static int lposition;
static long rseed = 1L;

/* forward declarations */
void xinfo(void);
void xflush(void);
int  xgetc(void);
void xputc(int ch);
void setraw(void);
void unsetraw(void);


/* osinit - initialize */
VOID osinit(banner)
  char *banner;
{
#ifdef BUFFERED
		setvbuf(stdout,NULL,_IOFBF,256);
#endif
	printf("%s\n",banner);
	lposition = 0;
	lindex = 0;
	lcount = 0;
	setraw();
}

/* osfinish - clean up before returning to the operating system */
VOID osfinish()
{
		unsetraw();
}

/* oserror - print an error message */
VOID oserror(msg)
  char *msg;
{
	printf("error: %s\n",msg);
}

/* osrand - return a random number between 0 and n-1 */
int osrand(n)
  int n;
{
	long k1;

	/* make sure we don't get stuck at zero */
	if (rseed == 0L) rseed = 1L;

	/* algorithm taken from Dr. Dobbs Journal, November 1985, page 91 */
	k1 = rseed / 127773L;
	if ((rseed = 16807L * (rseed - k1 * 127773L) - k1 * 2836L) < 0L)
		rseed += 2147483647L;

	/* return a random number between 0 and n-1 */
	return ((int)(rseed % (long)n));
}

/* osaopen - open an ascii file */
FILE *osaopen(name,mode)
  char *name,*mode;
{
	return (fopen(name,mode));
}

/* osbopen - open a binary file */
FILE *osbopen(name,mode)
  char *name,*mode;
{
	char bmode[10];
	strcpy(bmode,mode); strcat(bmode,"b");
	return (fopen(name,bmode));
}

/* osclose - close a file */
int osclose(fp)
  FILE *fp;
{
	return (fclose(fp));
}

/* ostgetc - get a character from the terminal */
int ostgetc()
{
	int ch;
	union REGS regs;
	struct SREGS segregs;

	/* check for a buffered character */
	if (lcount--)
		return (lbuf[lindex++]);

	/* get an input line */

	if (getvalue(s_dosinput) != NIL) {
#ifdef BUFFERED
		fflush(stdout);
#endif
		lindex = 2;
		lbuf[0] = LBSIZE - 2;
		regs.x.ax = 0x0A00;
		regs.x.dx = FP_OFF(lbuf);
		segregs.ds = FP_SEG(lbuf);
		intdosx(&regs,&regs,&segregs);
		putchar('\n');
		lcount = lbuf[1];
		lbuf[lcount+2] = '\n';
		if (tfp) fwrite(&lbuf[2],1,lcount+1,tfp);
		return (lbuf[lindex++]);
	}
	else {

	for (lcount = 0; ; )
		switch (ch = xgetc()) {
		case '\r':
				lbuf[lcount++] = '\n';
				xputc('\r'); xputc('\n'); lposition = 0;
				if (tfp) fwrite(lbuf,1,lcount,tfp);
				lindex = 0; lcount--;
				return (lbuf[lindex++]);
		case '\010':
		case '\177':
				if (lcount) {
					lcount--;
					while (lposition > lpos[lcount]) {
						xputc('\010'); xputc(' '); xputc('\010');
						lposition--;
					}
				}
				break;
		case '\032':
				xflush();
				return (EOF);
		default:
				if (ch == '\t' || (ch >= 0x20 && ch < 0x7F)) {
					lbuf[lcount] = ch;
					lpos[lcount] = lposition;
					if (ch == '\t')
						do {
							xputc(' ');
						} while (++lposition & 7);
					else {
						xputc(ch); lposition++;
					}
					lcount++;
				}
				else {
					xflush();
					switch (ch) {
					case '\003':	xltoplevel();	/* control-c */
					case '\007':	xlcleanup();	/* control-g */
					case '\020':	xlcontinue();	/* control-p */
					case '\032':	return (EOF);	/* control-z */
					case '\024':	xinfo();		/* control-t */
									return ostgetc();
					default:		return (ch);
					}
				}
		}}
}

/* ostputc - put a character to the terminal */
VOID ostputc(ch)
  int ch;
{
	/* check for control characters */

	oscheck();

	/* output the character */
	if (ch == '\n') {
		xputc('\r'); xputc('\n');
/*		lposition = 0; */
	}
	else {
		xputc(ch);
/*		lposition++; */
   }

   /* output the character to the transcript file */
   if (tfp)
		fputc(ch,tfp);
}

/* osflush - flush the terminal input buffer */
VOID osflush()
{
	lindex = lcount = lposition = 0;
}

/* oscheck - check for control characters during execution */
VOID oscheck()
{
	int ch;

	if ((ch = (bdos(6,0xFF,0) & 0xff)) != 0)
		switch (ch) {
		case '\002':	/* control-b */
			xflush();
			xlbreak("BREAK",s_unbound);
			break;
		case '\003':	/* control-c */
			xflush();
			xltoplevel();
			break;
		case '\023':	/* control-s */
			xgetc();	/* paused -- get character and toss */
			break;
		case '\024':	/* control-t */
			xinfo();
			break;
		}
}

/* xinfo - show information on control-t */
static VOID xinfo()
{
	extern int nfree,gccalls;
	extern long total;
	char buf[80];
	sprintf(buf,"\n[ Free: %d, GC calls: %d, Total: %ld ]",
			nfree,gccalls,total);
	errputstr(buf);
#ifdef BUFFERED
	fflush(stdout);
#endif
}

/* xflush - flush the input line buffer and start a new line */
static VOID xflush()
{
	osflush();
	ostputc('\n');
}

/* xgetc - get a character from the terminal without echo */
static int xgetc()
{
#ifdef BUFFERED
	fflush(stdout);
#endif
	return (bdos(7,0,0) & 0xFF);
}

/* xputc - put a character to the terminal */
static void xputc(ch)
  int ch;
{
#ifdef BUFFERED
	putchar(ch);
	if (ch == '\n') fflush(stdout);
#else
	bdos(6,ch,0);
#endif
}

/* xsystem - execute a system command */
LVAL xsystem()
{
	char *cmd;
	int ok;
	
	if (moreargs())
		cmd = (char *)getstring(xlgastring());
	else
		cmd = getenv("COMSPEC");
	xllastarg();
	unsetraw();
	ok = system(cmd);
	setraw();
	return (ok == 0 ? true : cvfixnum((FIXTYPE)errno));
}

/* xgetkey - get a key from the keyboard */
LVAL xgetkey()
{
	xllastarg();
	return (cvfixnum((FIXTYPE)xgetc()));
}

static unsigned savestate;
static unsigned char savebrk;

#ifdef GRAPHICS
static unsigned char origmode;
static unsigned ourmode1=0, ourmode2=0;

static VOID setmode(ax,bx)
int ax,bx;
{
	union REGS regs;
	regs.x.ax = ax;
	regs.x.bx = bx;
	int86(0x10, &regs, &regs);
}

#endif

/* setraw -- set raw mode */
static VOID setraw()
{
	union REGS regs;
	
	regs.x.ax = 0x4400; /* get device status */
	regs.x.bx = 1;
	intdos(&regs,&regs);
	regs.h.dh = 0;
	savestate = regs.x.dx;
	regs.x.ax = 0x4401;
	regs.h.dl |= 0x20;
	intdos(&regs,&regs);

	regs.x.ax = 0x3300; /* get ctrl-break status */
	intdos(&regs,&regs);
	savebrk = regs.h.dl;
	regs.x.ax = 0x3301;
	regs.h.dl = 0;
	intdos(&regs,&regs);

#ifdef GRAPHICS
	regs.x.ax = 0x0f00; /* get mode */
	int86(0x10, &regs, &regs);
	origmode = regs.h.al;
	if (ourmode1 != 0)	/* mode was changed -- use it */
		setmode(ourmode1,ourmode2);
#endif
}

/* unsetraw -- restore original mode */
static VOID unsetraw()
{
	union REGS regs;
	
	regs.x.ax = 0x4401;
	regs.x.bx = 1;
	regs.x.dx = savestate;
	intdos(&regs,&regs);
	regs.x.ax = 0x3301;
	regs.h.dl = savebrk;
	intdos(&regs,&regs);
	
#ifdef GRAPHICS
	if ((ourmode1 !=0) && (ourmode2 != origmode))
		setmode(origmode,0);
#endif
}


/* ossymbols - enter os specific symbols */
VOID ossymbols()
{
}

#ifdef GRAPHICS

static union REGS regin, regout;
static int xpos=0, ypos=0;
static int Xmax=-1, Ymax=-1;
static unsigned char drawvalue=15;

static LVAL draw(int x, int y, int x2, int y2)

{
	int xStep,yStep,xDist,yDist;
	int i, t8, t9, t10;
	
#ifdef BUFFERED
	fflush(stdout);
#endif

	if ((x < 0) | (x > Xmax) | (y < 0) | (y > Ymax) |
		(x2 < 0)| (x2 > Xmax)  | (y2 < 0) | (y2 > Ymax))
			return (NIL);

	x -= x2;	 /* cvt to distance and screen coordiate (right hand) */
	y2 = Ymax - y2;
	y = (Ymax - y) - y2;
	
	if (x < 0) {	/* calculate motion */
		xStep = -1;
		xDist = -x;
	}
	else {
		xStep = 1;
		xDist = x;
	}
	if (y < 0) {
		yStep = -1;
		yDist = -y;
	}
	else {
		yStep = 1;
		yDist = y;
	}
	
	regin.x.ax = drawvalue + 0x0c00;	/* write graphic pixel command */
	
	regin.x.cx = x2;		/* initial coordinates */
	regin.x.dx = y2;

	int86(0x10,&regin,&regout); /* initial draw */

	
	if (yDist == 0) {
		i = xDist;
		while (i--) { 
			regin.x.cx += xStep;
			int86(0x10,&regin,&regout);
		}
	}
	else if (xDist == yDist) {
		i = xDist;
		while (i--) { 
			regin.x.cx += xStep;
			regin.x.dx += yStep;
			int86(0x10,&regin,&regout);
		}
	}
	else if (xDist == 0) {
		i = yDist;
		while (i--) {
			regin.x.dx += yStep;
			int86(0x10,&regin,&regout);
		}
	}
	else if (xDist > yDist) {
		t8 = 2*yDist;
		t10 = 2*yDist - xDist;
		t9 = 2*(yDist - xDist);
		i = xDist;
		while (i--) {
			regin.x.cx += xStep;
			if (t10 < 0) {
				t10 += t8;
			}
			else {
				regin.x.dx += yStep;
				t10 += t9;
			}
			int86(0x10,&regin,&regout);
		}
	}
	else {
		t8 = 2*xDist;
		t10 = 2*xDist - yDist;
		t9 = 2*(xDist - yDist);
		i = yDist;
		while (i--) {
			regin.x.dx += yStep;
			if (t10 < 0) {
				t10 += t8;
			}
			else {
				regin.x.cx += xStep;
				t10 += t9;
			}
			int86(0x10,&regin,&regout);
		}
	}
	return (true);
}


/* xmode -- set display mode */
/* called with either ax contents, or ax,bx,xsize,ysize */
LVAL xmode()
{
	LVAL arg;

	arg = xlgafixnum();
	ourmode1 = (int) getfixnum(arg);
	
	if (moreargs()) {
		arg = xlgafixnum();
		ourmode2 = (int) getfixnum(arg);
		arg = xlgafixnum();
		Xmax = (int) getfixnum(arg) - 1;	/* max x coordinate */
		arg = xlgafixnum();
		Ymax = (int) getfixnum(arg) - 1;	/* max y coordinate */
		xllastarg();
	}
	else {
		ourmode2 = 0;
		switch (ourmode1) {
		case 4:
		case 5:
		case 13: Xmax = 319;
				 Ymax = 199;
				 break;
		case 6:
		case 14: Xmax = 639;
				 Ymax = 199;
				 break;
		case 16: Xmax = 639;
				 Ymax = 349;
				 break;
		case 18: Xmax = 639;	/* added VGA mode */
				 Ymax = 479;
				 break;
		default: Xmax = Ymax = -1; /* not a graphic mode */
				 break;
		}
	}

	
	setmode(ourmode1,ourmode2); /* set mode */
	return (true);
}

/* xcolor -- set color */

LVAL xcolor()
{
	LVAL arg;
	
	arg = xlgafixnum();
	xllastarg();
	
	drawvalue = (char) getfixnum(arg);
	
	return (arg);
}

/* xdraw -- absolute draw */

LVAL xdraw()
{
	LVAL arg = true;
	int newx, newy;
	
	while (moreargs()) {
		arg = xlgafixnum();
		newx = (int) getfixnum(arg);
	
		arg = xlgafixnum();
		newy = (int) getfixnum(arg);
		
		arg = draw(xpos,ypos,newx,newy);
		
		xpos = newx;
		ypos = newy;
	}
	return (arg);
}

/* xdrawrel -- absolute draw */

LVAL xdrawrel()
{
	LVAL arg = true;
	int newx, newy;
	
	while (moreargs()) {
		arg = xlgafixnum();
		newx = xpos + (int) getfixnum(arg);
	
		arg = xlgafixnum();
		newy = ypos + (int) getfixnum(arg);
		
		arg = draw(xpos,ypos,newx,newy);
		
		xpos = newx;
		ypos = newy;
	}
	return (arg);
}

/* xmove -- absolute move, then draw */

LVAL xmove()
{
	LVAL arg;
	
	arg = xlgafixnum();
	xpos = (int) getfixnum(arg);
	
	arg = xlgafixnum();
	ypos = (int) getfixnum(arg);
	
	return (xdraw());
}

/* xmoverel -- relative move */

LVAL xmoverel()
{
	LVAL arg;
	
	arg = xlgafixnum();
	xpos += (int) getfixnum(arg);

	arg = xlgafixnum();
	ypos += (int) getfixnum(arg);

	return (xdrawrel());
}

#endif
