/* -*-C-*-
********************************************************************************
*
* File:         unixstuff.c
* RCS:          $Header: unixstuff.c,v 1.4 90/10/29 13:28:19 mayer Exp $
* Description:  UNIX-Specific interfaces for XLISP
* Author:       David Michael Betz; Niels Mayer
* Created:      
* Modified:     Wed Nov  7 17:09:31 1990 (Niels Mayer) mayer@hplnpm
* Language:     C
* Package:      N/A
* Status:       X11r4 contrib tape release
*
* WINTERP 1.0 Copyright 1989 Hewlett-Packard Company (by Niels Mayer).
* XLISP version 2.1, Copyright (c) 1989, by David Betz.
*
* Permission to use, copy, modify, distribute, and sell this software and its
* documentation for any purpose is hereby granted without fee, provided that
* the above copyright notice appear in all copies and that both that
* copyright notice and this permission notice appear in supporting
* documentation, and that the name of Hewlett-Packard and David Betz not be
* used in advertising or publicity pertaining to distribution of the software
* without specific, written prior permission.  Hewlett-Packard and David Betz
* make no representations about the suitability of this software for any
* purpose. It is provided "as is" without express or implied warranty.
*
* HEWLETT-PACKARD AND DAVID BETZ DISCLAIM ALL WARRANTIES WITH REGARD TO THIS
* SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
* IN NO EVENT SHALL HEWLETT-PACKARD NOR DAVID BETZ BE LIABLE FOR ANY SPECIAL,
* INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
* LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
* OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
* PERFORMANCE OF THIS SOFTWARE.
*
* See ./winterp/COPYRIGHT for information on contacting the authors.
* 
* Please send modifications, improvements and bugfixes to mayer@hplabs.hp.com
* Post XLISP-specific questions/information to the newsgroup comp.lang.lisp.x
*
********************************************************************************
*/
static char rcs_identity[] = "@(#)$Header: unixstuff.c,v 1.4 90/10/29 13:28:19 mayer Exp $";


#include "xlisp.h"

#ifdef WINTERP
#include <fcntl.h>		/* needed for fcntl(2) calls */
#endif

/******************************************************************************
 * Prim_POPEN - start a process and open a pipe for read/write 
 * (code stolen from xlfio.c:xopen())
 *
 * syntax: (popen <command line> :direction <direction>)
 *                <command line> is a string to be sent to the subshell (sh).
 *                <direction> is either :input (to read from the pipe) or
 *                                      :output (to write to the pipe).
 *                                      (:input is the default)
 *
 * Popen returns a stream, or NIL if files or processes couldn't be created.
 * The  success  of  the  command  execution  can be checked by examining the 
 * return value of pclose. 
 *
 * Added to XLISP by Niels Mayer
 ******************************************************************************/
LVAL Prim_POPEN()
{
  extern LVAL k_direction, k_input, k_output;
  char *name,*mode;
  FILE *fp;
  LVAL dir;

  /* get the process name and direction */
  name = (char *) getstring(xlgastring());
  if (!xlgetkeyarg(k_direction, &dir))
    dir = k_input;
  
  /* get the mode */
  if (dir == k_input)
    mode = "r";
  else if (dir == k_output)
    mode = "w";
  else
    xlerror("bad direction",dir);
  
  /* try to open the file */
  return ((fp = popen(name,mode)) ? cvfile(fp) : NIL);
}


/******************************************************************************
 * Prim_PCLOSE - close a pipe opened by Prim_POPEN().
 * (code stolen from xlfio.c:xclose())
 *
 * syntax: (pclose <stream>)
 *                  <stream> is a stream created by popen.
 * returns T if the command executed successfully, otherwise, 
 * returns the exit status of the opened command.
 *
 * Added to XLISP by Niels Mayer
 ******************************************************************************/
LVAL Prim_PCLOSE()
{
  extern LVAL true;
  LVAL fptr;
  int  result;

  /* get file pointer */
  fptr = xlgastream();
  xllastarg();

  /* make sure the file exists */
  if (getfile(fptr) == NULL)
    xlfail("file not open");

  /* close the pipe */
  result = pclose(getfile(fptr));

  if (result == -1)
    xlfail("<stream> has not been opened with popen");
    
  setfile(fptr,NULL);

  /* return T if success (exit status 0), else return exit status */
  return (result ? cvfixnum(result) : true);
}


/******************************************************************************
 * Prim_SYSTEM - run a process, sending output (if any) to stdout/stderr
 *
 * syntax: (system <command line>)
 *                 <command line> is a string to be sent to the subshell (sh).
 *
 * Returns T if the command executed succesfully, otherwise returns the 
 * integer shell exit status for the command.
 *
 * Added to XLISP by Niels Mayer
 ******************************************************************************/
LVAL Prim_SYSTEM()
{
  extern LVAL true;
  extern int sys_nerr;
  extern char *sys_errlist[];
  extern int errno;
  LVAL command;
  int  result;
  char temptext[1024];

  /* get shell command */
  command = xlgastring();
  xllastarg();
  
  /* run the process */
  result = system((char *) getstring(command));

  if (result == -1) {		/* if a system error has occured */
    if (errno < sys_nerr)
      (void) sprintf(temptext, "Error in system(3S): %s\n", sys_errlist[errno]);
    else
      (void) strcpy(temptext, "Error in system(3S): unknown error\n");
    xlfail(temptext);
  }

  /* return T if success (exit status 0), else return exit status */
  return (result ? cvfixnum(result) : true);
}


/******************************************************************************
 * (FSCANF-FIXNUM <stream> <scanf-format>)
 * This routine calls fscanf(3s) on a <stream> that was previously openend
 * via open or popen. It will not work on an USTREAM.
 * <scanf-format> is a format string containing a single conversion
 * directive that will result in an integer valued conversion.
 * %d, %u, %o, %x, %ld, %lu, %lo and %lx style conversions 
 * are acceptable for this routine.
 * WARNING: specifying a <scanf-format> that will result in the conversion
 * of a result larger than sizeof(long) will result in corrupted memory and
 * core dumps. 
 * 
 * This routine will return an FIXNUM if fscanf() returns 1 (i.e. if
 * the one expected conversion has succeeded. It will return NIL if the
 * conversion wasn't successful, or if EOF was reached.
 ******************************************************************************/
LVAL Prim_FSCANF_FIXNUM()
{
  LVAL  lval_stream;
  char* fmt;
  long  result;
  
  lval_stream = xlgastream();
  if (getfile(lval_stream) == NULL)
    xlerror("File not opened.", lval_stream);
  fmt = (char *) getstring(xlgastring());
  xllastarg();
  
  result = 0L;			/* clear it out hibits incase short is written */
  /* if scanf returns result <1 then an error or eof occured. */
  if (fscanf(getfile(lval_stream), fmt, &result) < 1)
    return (NIL);
  else
    return (cvfixnum((FIXTYPE) result));
}


/******************************************************************************
 * (FSCANF-STRING <stream> <scanf-format>)
 * This routine calls fscanf(3s) on a <stream> that was previously openend
 * via open or popen. It will not work on an USTREAM.
 * <scanf-format> is a format string containing a single conversion
 * directive that will result in a string valued conversion.
 * %s, %c, and %[...] style conversions are acceptable for
 * this routine.
 * WARNING: specifying a <scanf-format> that will result in the conversion
 * of a result larger than 1024 characters will result in corrupted
 * memory and core dumps.
 * 
 * This routine will return a string if fscanf() returns 1 (i.e. if
 * the one expected conversion has succeeded. It will return NIL if the
 * conversion wasn't successful, or if EOF was reached.
 ******************************************************************************/
LVAL Prim_FSCANF_STRING()
{
  LVAL lval_stream;
  char* fmt;
  char result[BUFSIZ];

  
  lval_stream = xlgastream();
  if (getfile(lval_stream) == NULL)
    xlerror("File not opened.", lval_stream);
  fmt = (char *) getstring(xlgastring());
  xllastarg();
  
  result[0] = result[1] = '\0';	/* if the conversion is %c, then fscanf
				   doesn't null terminate the string,
				   so do it just incase */

  /* if scanf returns result <1 then an error or eof occured. */
  if (fscanf(getfile(lval_stream), fmt, result) < 1)
    return (NIL);
  else
    return (cvstring(result));
}


/******************************************************************************
 * (FSCANF-FLONUM <stream> <scanf-format>)
 * This routine calls fscanf(3s) on a <stream> that was previously openend
 * via open or popen. It will not work on an USTREAM.
 * <scanf-format> is a format string containing a single conversion
 * directive that will result in an FLONUM valued conversion.
 * %e %f or %g are valid conversion specifiers for this routine.
 *
 * WARNING: specifying a <scanf-format> that will result in the conversion
 * of a result larger than sizeof(float) will result in corrupted memory and
 * core dumps. 
 * 
 * This routine will return a FLONUM if fscanf() returns 1 (i.e. if
 * the one expected conversion has succeeded. It will return NIL if the
 * conversion wasn't successful, or if EOF was reached.
 ******************************************************************************/
LVAL Prim_FSCANF_FLONUM()
{
  LVAL lval_stream;
  char* fmt;
  FILE * fp;
  float result;
  
  lval_stream = xlgastream();
  if (getfile(lval_stream) == NULL)
    xlerror("File not opened.", lval_stream);
  fmt = (char *) getstring(xlgastring());
  xllastarg();
  
  /* if scanf returns result <1 then an error or eof occured. */
  if (fscanf(getfile(lval_stream), fmt, &result) < 1)
    return (NIL);
  else
    return (cvflonum((FLOTYPE) result));
}


/******************************************************************************
 * (copy-array <src> <dest> [<pos>]) --> returns <dest>
 * This function copies from array <src> into the preallocated array <dest>
 * (allocate with 'make-array'). If the optional arg <pos> is given, then
 * elements from <src> will be written into <dest> at index <pos>, otherwise
 * <pos> defaults to 0. 
 *
 * This function was added to xlisp by Niels Mayer.
 ******************************************************************************/
LVAL Prim_COPY_ARRAY()
{
  register int size;
  register LVAL *src, *dest;
  LVAL src_array, dest_array, lval_pos;

  src_array = xlgavector();     /* get <src> */
  dest_array = xlgavector();    /* get <dest> */
  if moreargs()
    lval_pos = xlgafixnum();    /* get optional <pos> */
  else
    lval_pos = NIL;
  xllastarg();

  src = src_array->n_vdata;
  dest = dest_array->n_vdata;

  if (getsize(src_array) < getsize(dest_array)) /* which is shortest? */
    size = getsize(src_array);
  else
    size = getsize(dest_array);

  if (lval_pos) {
    int pos = getfixnum(lval_pos);
    int len = getsize(dest_array) - pos;
    if ((len <= 0) || (pos < 0))
      xlerror("Array position out of bounds.", lval_pos);    
    if (len < size)
      size = len;
    dest = dest + pos;
  }

  while (size--)
    *dest++ = *src++;

  return (dest_array);
}

/******************************************************************************
 * (array-insert-pos <array> <pos> <elt>) --> returns the new <array>
 * inserts <elt> at index <pos> in <array>. if <pos> < 0, then <elt> is
 * appended to the end of <array>.
 *
 * This function was added to xlisp by Niels Mayer.
 ******************************************************************************/
LVAL Prim_ARRAY_INSERT_POS()
{
  register int i;
  register LVAL *src, *dest;
  LVAL src_array, dest_array, elt, lval_position;
  int src_size, position;

  src_array = xlgavector();     /* get <array> */
  lval_position = xlgafixnum(); /* get <pos>, a fixnum */
  elt = nextarg();              /* get <elt>, which can be any lisp type */
  xllastarg();

  src_size = getsize(src_array);
  position = getfixnum(lval_position);
  if (position >= src_size)
    xlerror("Array insertion position out of bounds.", lval_position);
  dest_array = newvector(src_size + 1);

  src = src_array->n_vdata;
  dest = dest_array->n_vdata;

  if (position < 0) {           /* append <elt> to end of array */
    i = src_size;
    while (i--)
      *dest++ = *src++;
    *dest = elt;
  }
  else {                        /* insert <elt> at <position> */
    i = position;
    while (i--)
      *dest++ = *src++;
    *dest++ = elt;
    i = src_size - position;
    while (i--)
      *dest++ = *src++;
  }
  return (dest_array);
}

/******************************************************************************
 * (array-delete-pos <array> <pos>) --> returns the new <array>
 * deletes the element at index <pos> in <array>. If <pos>==-1, then it
 * will delete the last element in the array. 
 * Note that this function is destructive. It reuses the old <array>'s
 * elements.
 *
 * This function was added to xlisp by Niels Mayer.
 ******************************************************************************/
LVAL Prim_ARRAY_DELETE_POS()
{
  register int i;
  register LVAL *src, *dest;
  LVAL src_array, dest_array, lval_position;
  int src_size, position;

  src_array = xlgavector();     /* get <array> */
  lval_position = xlgafixnum(); /* get <pos>, a fixnum */
  xllastarg();

  src_size = getsize(src_array);
  position = getfixnum(lval_position);
  if (position >= src_size)
    xlerror("Array insertion position out of bounds.", lval_position);
  if ((src_size - 1) > 0)
    dest_array = newvector(src_size - 1);
  else
    return (NIL);

  src = src_array->n_vdata;
  dest = dest_array->n_vdata;

  if (position < 0) {           /* remove last element of array */
    i = src_size - 1;
    while (i--)
      *dest++ = *src++;
  }
  else {                        /* remove <elt> at <position> */
    i = position;
    while (i--)
      *dest++ = *src++;
    src++;                      /* don't copy the deleted elt */
    i = src_size - (position + 1);
    while (i--)
      *dest++ = *src++;
  }
  return (dest_array);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/
/* Origins:
 *
 * stuff.c  -- operating system specific routines 
 * -- Written by dbetz for XLISP 2.0 
 * -- Copied by EFJohnson from a BIX message 
 * -- Unix System V 
 *
 * AND
 *
 * From Ken Whedbee's unixstuff.c 
 * Revision:	Ken Whedbee  kcw@reef.cis.ufl.edu
 *
 *		1. Control characters will work now on BSD and S5
 *		2. Added Common Lisp timing functions
 *
 */

#include <signal.h>
#include <sys/types.h>
#include <sys/times.h>

#ifdef KCW_INPUT   /* hacked by KCW */
#ifdef BSD
#include <sys/ioctl.h>
struct sgttyb savetty;
struct sgttyb newtty;
#define stty(fd,arg)    (ioctl(fd, TIOCSETP, arg))
#define gtty(fd,arg)    (ioctl(fd, TIOCGETP, arg))
#else
#include <termio.h>
struct termio savetty;
struct termio newtty;
#define stty(fd,arg)    (ioctl(fd, TCGETA, arg))
#define gtty(fd,arg)    (ioctl(fd, TCSETAF, arg))
#endif
#endif

#define	LBSIZE	200

/* -- external variables */
extern FILE *tfp;

/* -- local variables */
static  long    rseed = 1L;
static  char    lbuf[LBSIZE];
static  int     lindex;
static  int     lcount;


/* -- osinit - initialize */
VOID osinit(banner)

char	*banner;
{
	printf("%s\n", banner );
	lindex	= 0;
	lcount	= 0;
#ifdef KCW_INPUT
	init_tty();		/* KCW */
#endif
}

/* -- osfinish - clean up before returning to the operating system */
VOID osfinish()
{
#ifdef KCW_INPUT   /* hacked by KCW */
        stty(0, &savetty);	/* KCW */
#endif
}


/* -- oserror - print an error message */
VOID oserror(msg)

char	*msg;

{
	printf( "error: %s\n", msg );
}


/* -- osrand - return a random number between 0 and n-1 */
int osrand(n)

int	n;

{
	long k1;

	/* -- make sure we don't get stuck at zero */
	if ( rseed == 0L ) rseed = 1L;

	/* -- algorithm taken from Dr Dobbs Journal, Nov. 1985, page 91 */
	k1 = rseed / 127773L;
	if ( ( rseed = 16807L * (rseed - k1 * 127773L) -k1 * 2836L) < 0L )
		rseed += 2147483647L;

	/* -- return a random number between 0 and n-1 */
	return( (int) (rseed & (long) n ) );
}


/* -- osaopen -- open an ascii file */
FILE	*osaopen( name, mode )
char	*name, *mode;
{
#ifdef WINTERP
  FILE* fp = fopen(name, mode);
  fcntl(fileno(fp), F_SETFD, 1); /* set close-on-exec */
  return (fp);
#else
	return( fopen( name, mode ) );
#endif /* WINTERP */
}


/* -- osbopen -- open a binary file */
FILE	*osbopen( name, mode )
char	*name, *mode;
{
#ifdef WINTERP
  FILE* fp = fopen(name, mode);
  fcntl(fileno(fp), F_SETFD, 1); /* set close-on-exec */
  return (fp);
#else
	return( fopen( name, mode ) );
#endif	/* WINTERP */
}


/* -- osclose -- close a file */
int	osclose( fp )
FILE	*fp;
{
	return( fclose( fp ) );
}

/* -- ostgetc - get a character from the terminal */

int	ostgetc()
{
#ifdef KCW_INPUT   /* hacked by KCW */
        char *xfgets();

	while(--lcount < 0 )
		{
		if ( xfgets(lbuf,LBSIZE,stdin) == NULL ) /* KCW */
			return( EOF );
		if ( tfp )
			fputs( lbuf, tfp );
		lcount = strlen( lbuf );
		lindex = 0;
		}

	return( lbuf[lindex++] );
#else
	while(--lcount < 0 )
		{
		if ( fgets(lbuf,LBSIZE,stdin) == NULL )
			return( EOF );
		if ( tfp )
			fputs( lbuf, tfp );
		lcount = strlen( lbuf );
		lindex = 0;
		}

	return( lbuf[lindex++] );
#endif
}

/* -- ostputc - put a character to the terminal */

VOID ostputc( ch )
int     ch;
{
#ifdef KCW_INPUT   /* hacked by KCW */
        char buf[1];

        buf[0] = ch;
        
        /* -- output the character */
/*        putchar( ch ); */
        write(1,buf,sizeof(buf)); /* NPM note: 1 == stdout's fd ... */

        /* -- output the char to the transcript file */
        if ( tfp )
                putc( ch, tfp ); /* NPM */
#else
	/* -- check for control characters */
	oscheck();
	
	/* -- output the character */
	putchar( ch );

	/* -- output the char to the transcript file */
	if ( tfp )
		putc( ch, tfp );
#endif
}


/* -- osflush - flush the terminal input buffer */
VOID osflush()
{
	lindex = lcount = 0; 
}

/* -- oscheck - check for control characters during execution */
VOID oscheck()
{
}


/* -- ossymbols - enter os-specific symbols */
ossymbols()
{
}


#ifdef KCW_INPUT   /* hacked by KCW */
static osx_check(ch)		/* added by KCW */
char ch;
{
     switch (ch) {
        case '\003':    
          xltoplevel(); /* control-c */
        case '\007':    
          xlcleanup();  /* control-g */
        case '\020':    
          xlcontinue(); /* control-p */
        case '\024':    /* control-t */
          xinfo();
          printf("\n ");
     }
}

/* xinfo - show information on control-t
   hacked by KCW */
static xinfo()
{
  extern int nfree,gccalls;
  extern long total;
  char buf[80], tymebuf[20];
  int tyme;

  tyme = time(0);
  strcpy(tymebuf, ctime(&tyme));
  tymebuf[19] = '\0';
  sprintf(buf,"\n[ %s Free: %d, GC calls: %d, Total: %ld ]",
    tymebuf, nfree,gccalls,total);
  errputstr(buf);
}

/* xflush - flush the input line buffer and start a new line */
static VOID xflush()
{
	osflush();
	ostputc('\n');
}

/* read_keybd -- read a character from the keyboard
   added by KCW */
char read_keybd()
{
   int nrd;
   char buf[1];

   nrd = read(0, buf, sizeof(buf)); /* NPM note: 0 == stdin fd */
   buf[nrd] = 0;

   if (buf[0] == 127) {		/* perform the BACKSPACE */
      stdputstr("\010");
      stdputstr(" ");
      stdputstr("\010");
   }
   else
      stdputstr(buf);

   return(buf[0]);
}

init_tty()		/* added KCW */
{
        /* extern sigcatch(); */
	extern onsusp();

        signal(SIGINT, xltoplevel); 
	signal(SIGQUIT, SIG_IGN);
	if (signal(SIGTSTP, onsusp) == SIG_DFL) {
		signal(SIGTSTP, onsusp);
	}
        if (gtty(0, &savetty) == -1)
        {
                printf("ioctl failed: not a tty\n");
                exit();
        }
#ifdef BSD
        newtty = savetty;
        newtty.sg_flags |= CBREAK;      /* turn off canonical mode */
                                        /* i.e., turn on cbreak mode */
        newtty.sg_flags &= ~ECHO;       /* turn off character echo */
#else
        newtty.c_lflag &= ~ICANON;	/* SYS 5 */
        newtty.c_lflag &= ~ECHO;
        newtty.c_cc[VMIN] = 1;
        newtty.c_cc[VTIME] = 1;
#endif
        /*
         * You can't request that it try to give you at least
         * 5 characters, nor set the timeout to 10 seconds,
         * as you can in the S5 example.  If characters come
         * in fast enough, though, you may get more than one.
         */
        if (stty(0, &newtty) == -1)
        {
                printf("cannot put tty into cbreak mode\n");
                exit();
        }
}

onsusp()		/* added KCW */
{
	/* ignore SIGTTOU so we dont get stopped if csh grabs the tty */
	signal(SIGTTOU, SIG_IGN);
	stty(0, &savetty);
	xflush();
	signal(SIGTTOU,SIG_DFL);

	/* send the TSTP signal to suspend our process group */
	signal(SIGTSTP, SIG_DFL);
	sigsetmask(0);
	kill(0, SIGTSTP);
	/* pause for station break */

	/* we re back */
	signal(SIGTSTP, onsusp);
	stty(0, &newtty);
}

char *xfgets(s, n, iop)		/* hacked fgets, KCW */
char *s;
register FILE *iop;
{
        register c;
        register char *cs;

        cs = s;
        while (--n>0 && (c = read_keybd()) != EOF) { 
             switch(c) {
                  case '\002' :                 /* CTRL-b */
                  case '\003' :                 /* CTRL-c */
                  case '\007' :                 /* CTRL-g */
                  case '\020' :                 /* CTRL-p */
                  case '\024' : osx_check(c);   /* CTRL-t */
                                n++;
                                break;

                  case 127    : n+=2;           /* BACKSPACE */
                                *cs--;
                                *cs = ' ';
                                break;
                 
                  default     : *cs++ = c;      /* character */
                }
                if (c=='\n') break;
        }
        if (c == EOF && cs==s) return(NULL); 
        *cs++ = '\0';
        return(s);
}
#endif


/***********************************************************************/
/**                                                                   **/
/**   Time Functions:  code from Tom Almy and Luke Tierney further    **/
/**                    hacked by KCW                                  **/
/**                                                                   **/
/***********************************************************************/

#include <unistd.h>		/* for SYSCONF(2) */
unsigned long ticks_per_second()
{
  static unsigned long hz = 0L;

  if (hz)
    return (hz);
  else {
    hz = (unsigned long) sysconf(_SC_CLK_TCK);		/* XPG3, POSIX.1, FIPS 151-1 */
    return (hz);
  }
}

unsigned long run_tick_count()		/* CPU time */
{
  struct tms tm;

  times(&tm);
  return((unsigned long) tm.tms_utime + tm.tms_stime );  
}

unsigned long real_tick_count() 	/* real time */
{                  
#ifdef HAVE_TIME_STAMP		/* NPM -- time_stamp is not a global value on HPUX, at least. Perhaps on SUN?? */
  extern long time_stamp;
  return((unsigned long) (60 * (time((unsigned long *) NULL) - time_stamp)));
#else
  return((unsigned long) (60 * time((unsigned long *) NULL))); /* NPM -- this makes no sense at all... */
#endif
}

LVAL xstime()
{
  extern LVAL xeval();
  extern char buf[];
  LVAL result;
  unsigned long tm, rtm;
  double dtm, rdtm;
  
  tm = run_tick_count();
  rtm = real_tick_count();
  result = xeval();
  tm = run_tick_count() - tm;
  rtm = real_tick_count() - rtm;
  dtm = (tm > 0) ? tm : -tm;
  rdtm = (rtm > 0) ? rtm : -rtm;
  sprintf(buf, "CPU %.2f sec., Real %.2f sec.\n", dtm / ticks_per_second(),
                                            rdtm / ticks_per_second());
  stdputstr(buf);
  return(result);
}

LVAL xs_get_internal_run_time()
{
  return(cvfixnum((FIXTYPE) run_tick_count()));
}

LVAL xs_get_internal_real_time()
{
  return(cvfixnum((FIXTYPE) real_tick_count()));
}
