/* metastuf.c - Metaware High-C specific routines */

#include "xlisp.h"
#include "osdefs.h"
#include <string.h>

#include <implement.cf>
#include <language.cf>

#pragma Global_aliasing_convention(_Private_routine_prefix "%r");

   pragma data(common,_Private_prefix "dosregs");
   typedef union {
		 struct {char L,H;} LH;	  /* Lower & Upper portions of register. */
		 unsigned R;			  /* Entire register. */
		 } Register;
   typedef struct {
	  Register AX,BX,CX,DX,SI,DI,DS,ES;
	  unsigned Flags;
	  } DOS_communication;

   DOS_communication Registers;
   pragma data;

#pragma Calling_convention(PASCAL);
   /* Use this for your own direct communication with MS-DOS. */
   extern void calldos();
   extern void callint(int interrupt);

#pragma Global_aliasing_convention();
#pragma Calling_convention(_DEFAULT_CALLING_CONVENTION);


#define LBSIZE 200

/* external variables */
extern LVAL s_unbound,s_dosinput,true;
extern FILE *tfp;

/* external functions */
extern int ssystem(char *cmd, char *tail);
extern void setdrawmode(int mode);
extern void unsetdrawmode(void);
#include<stdlib.h>



/* local variables */
static char lbuf[LBSIZE];
static int lpos[LBSIZE];
static int lindex;
static int lcount;
static int lposition;
static long rseed = 1L;

/* forward declarations */
void xinfo(void);
void xflush(void);
void xputc(int ch);
void setraw(void);
void unsetraw(void);
int  xgetc(void);

#ifdef BUFFERED
#define CHBSIZE 256
static char outbuf[CHBSIZE];
static char *outbufp = &outbuf[0];

void flushbuf()
{
	if (outbufp != &outbuf[0]) {
		Registers.AX.R = 0x4000;
		Registers.BX.R = 1;
		Registers.CX.R = outbufp - &outbuf[0];
		Registers.DX.R = (unsigned int) &outbuf[0];
		calldos();
		outbufp = &outbuf[0];
	}
}
#endif

long myftell(FILE *fp)	/* metaware's is broken */
{
	long pos;

	Registers.AX.R = 0x4201;
	Registers.BX.R = (unsigned int) fp->_fd;
	Registers.CX.R = 0;
	Registers.DX.R = 0;
	calldos();
	
	pos = (Registers.DX.R << 16) + ((Registers.AX.R) & 0xffff);
	
	if ((fp->_flag & _UNINITIALIZED) ||
		(fp->_cnt < 0))
			return pos;
	
	if (fp->_flag & _WROTE_LAST) {
		pos += BUFSIZ - fp->_cnt;
	}
	else {
		pos -= fp->_cnt;
	}
	return pos;
}

/* osinit - initialize */
VOID osinit(banner)
  char *banner;
{
	printf("%s\n",banner);
	lposition = 0;
	lindex = 0;
	lcount = 0;
	setraw();
}

/* osfinish - clean up before returning to the operating system */
VOID osfinish()
{
#ifdef BUFFERED
	flushbuf();
#endif
	unsetraw();
}

/* oserror - print an error message */
VOID oserror(msg)
  char *msg;
{
	printf("error: %s\n",msg);
}

/* osrand - return a random number between 0 and n-1 */
int osrand(n)
  int n;
{
	long k1;

	/* make sure we don't get stuck at zero */
	if (rseed == 0L) rseed = 1L;

	/* algorithm taken from Dr. Dobbs Journal, November 1985, page 91 */
	k1 = rseed / 127773L;
	if ((rseed = 16807L * (rseed - k1 * 127773L) - k1 * 2836L) < 0L)
		rseed += 2147483647L;

	/* return a random number between 0 and n-1 */
	return ((int)(rseed % (long)n));
}

/* osaopen - open an ascii file */
FILE *osaopen(name,mode)
  char *name,*mode;
{
	return (fopen(name,mode));
}

/* osbopen - open a binary file */
FILE *osbopen(name,mode)
  char *name,*mode;
{
	char bmode[10];
	strcpy(bmode,mode); strcat(bmode,"b");
	return (fopen(name,bmode));
}

/* osclose - close a file */
int osclose(fp)
  FILE *fp;
{
	return (fclose(fp));
}

/* ostgetc - get a character from the terminal */
int ostgetc()
{
	int ch;

	/* check for a buffered character */
	if (lcount--)
		return (lbuf[lindex++]);

	/* get an input line */
	if (getvalue(s_dosinput) != NIL) {
#ifdef BUFFERED
		flushbuf();
#endif
		lindex = 2;
		lbuf[0] = LBSIZE - 2;
		Registers.AX.R = 0x0A00;
		Registers.DX.R = (unsigned int) lbuf;
		calldos();
		putchar('\n');
		lcount = lbuf[1];
		lbuf[lcount+2] = '\n';
		if (tfp) fwrite(&lbuf[2],1,lcount+1,tfp);
		return (lbuf[lindex++]);
	}
	else {
	for (lcount = 0; ; )
		switch (ch = xgetc()) {
		case '\r':
				lbuf[lcount++] = '\n';
				xputc('\r'); xputc('\n'); lposition = 0;
				if (tfp) fwrite(lbuf,1,lcount,tfp);
				lindex = 0; lcount--;
				return (lbuf[lindex++]);
		case '\010':
		case '\177':
				if (lcount) {
					lcount--;
					while (lposition > lpos[lcount]) {
						xputc('\010'); xputc(' '); xputc('\010');
						lposition--;
					}
				}
				break;
		case '\032':
				xflush();
				return (EOF);
		default:
				if (ch == '\t' || (ch >= 0x20 && ch < 0x7F)) {
					lbuf[lcount] = ch;
					lpos[lcount] = lposition;
					if (ch == '\t')
						do {
							xputc(' ');
						} while (++lposition & 7);
					else {
						xputc(ch); lposition++;
					}
					lcount++;
				}
				else {
					xflush();
					switch (ch) {
					case '\003':	xltoplevel();	/* control-c */
					case '\007':	xlcleanup();	/* control-g */
					case '\020':	xlcontinue();	/* control-p */
					case '\032':	return (EOF);	/* control-z */
					case '\024':	xinfo();		/* control-t */
									return ostgetc();
					default:		return (ch);
					}
				}
		}}
}

/* ostputc - put a character to the terminal */
VOID ostputc(ch)
  int ch;
{
	/* check for control characters */
	oscheck();

	/* output the character */
	if (ch == '\n') {
		xputc('\r'); xputc('\n');
/*		lposition = 0; */
	}
	else {
		xputc(ch);
/*		lposition++; */
   }

   /* output the character to the transcript file */
   if (tfp)
		fputc(ch,tfp);
}

/* osflush - flush the terminal input buffer */
VOID osflush()
{
	lindex = lcount = lposition = 0;
}

/* oscheck - check for control characters during execution */
VOID oscheck()
{
#if 0
	int ch;

   Registers.AX.LH.H = 0xB;
   calldos();
   if (Registers.AX.LH.L) 
		   ch = xgetc();
   else 
		   return;		/* nothing read */
   switch (ch) {
#else
		Registers.AX.R = 0x0600;
		Registers.DX.LH.L = 0xff;
		calldos();
		if (Registers.AX.LH.L == 0) return;		/* no characters */

		switch (Registers.AX.LH.L) {
#endif

		case '\002':	/* control-b */
			xflush();
			xlbreak("BREAK",s_unbound);
			break;
		case '\003':	/* control-c */
			xflush();
			xltoplevel();
			break;
		case '\023':	/* control-s */
			xgetc();	/* paused -- get character and toss */
			break;
		case '\024':	/* control-t */
			xinfo();
			break;
		}
}

/* xinfo - show information on control-t */
static VOID xinfo()
{
	extern int nfree,gccalls;
	extern long total;
	char buf[80];
	sprintf(buf,"\n[ Free: %d, GC calls: %d, Total: %ld ]",
			nfree,gccalls,total);
	errputstr(buf);
#ifdef BUFFERED
	flushbuf();
#endif
}

/* xflush - flush the input line buffer and start a new line */
static VOID xflush()
{
	osflush();
	ostputc('\n');
}

/* xgetc - get a character from the terminal without echo */
static int xgetc()
{
#ifdef BUFFERED
	flushbuf();
#endif
		Registers.AX.LH.H = 0x7;
		calldos();
		return Registers.AX.LH.L;
}

/* xputc - put a character to the terminal */
static void xputc(ch)
  int ch;
{
#ifdef BUFFERED
	*outbufp++ = ch;
	if (ch == '\n' || outbufp == &outbuf[CHBSIZE]) flushbuf();
#else
   Registers.AX.LH.H = 6; Registers.DX.LH.L = ch;
   calldos();
#endif
}

static unsigned savestate;
static unsigned char savebrk;
#ifdef GRAPHICS
static unsigned char origmode;
static unsigned ourmode1=0, ourmode2=0;

static VOID setmode(ax,bx)
int ax,bx;
{
	Registers.AX.R = ax;
	Registers.BX.R = bx;
	callint(0x10);
}

#endif

/* setraw -- set raw mode */
static VOID setraw()
{
	Registers.AX.R = 0x4400;		/* get device status */
	Registers.BX.R = 1;
	calldos();
	Registers.DX.LH.H = 0;
	savestate = Registers.DX.R;
	Registers.AX.R = 0x4401;
	Registers.DX.LH.L |= 0x20;
	calldos();

	Registers.AX.R = 0x3300; /* get ctrl-break status */
	calldos();
	savebrk = Registers.DX.LH.L;
	Registers.AX.R = 0x3301;
	Registers.DX.LH.L = 0;
	calldos();

#ifdef GRAPHICS
	Registers.AX.R = 0x0f00;		/* get mode */
	callint(0x10);
	origmode = Registers.AX.LH.L;
	if (ourmode1 != 0)		/* mode was changed -- use it */
		setmode(ourmode1,ourmode2);
#endif
}

/* unsetraw -- restore original mode */
static VOID unsetraw()
{
	Registers.AX.R = 0x4401;
	Registers.BX.R = 1;
	Registers.DX.R = savestate;
	calldos();
	Registers.AX.R = 0x3301;
	Registers.DX.LH.L = savebrk;
	calldos();

#ifdef GRAPHICS
	if ((ourmode1 !=0) && (ourmode2 != origmode))
		setmode(origmode,0);
#endif
}


/* xsystem - execute a system command */
LVAL xsystem()
{
	char commandtail[128],*s;
	int Err;

	if (moreargs()) {
		strcpy(commandtail," /c ");
		s = (char *)getstring(xlgastring());
		strcat(commandtail,s);
		strcat(commandtail,"\r");
		commandtail[0] = strlen(commandtail) - 2;
		xllastarg();
	}
	else
		strcpy(commandtail,"\001 \r");

	unsetraw();
	Err = ssystem(getenv("COMSPEC"),commandtail);
	setraw();
	return ( Err == 0 ? true : cvfixnum((FIXTYPE)Err));
}

/* xgetkey - get a key from the keyboard */
LVAL xgetkey()
{
	xllastarg();
	return (cvfixnum((FIXTYPE)xgetc()));
}

/* ossymbols - enter os specific symbols */
VOID ossymbols()
{
}


#ifdef GRAPHICS

static int xpos=0, ypos=0;
static int Xmax = -1, Ymax=-1;
extern int bytesperline;
static unsigned char drawvalue=15;

extern void setpixel();

static LVAL draw(int x, int y, int x2, int y2)

{
	int xStep,yStep,xDist,yDist;
	int i, t8, t9, t10;
	
#ifdef BUFFERED
	flushbuf();
#endif

	if ((x < 0) | (x > Xmax) | (y < 0) | (y > Ymax) |
		(x2 < 0)| (x2 > Xmax)  | (y2 < 0) | (y2 > Ymax))
			return (NIL);

	x -= x2;	 /* cvt to distance and screen coordiate (right hand) */
	y2 = Ymax - y2;
	y = (Ymax - y) - y2;
	
	if (x < 0) {	/* calculate motion */
		xStep = -1;
		xDist = -x;
	}
	else {
		xStep = 1;
		xDist = x;
	}
	if (y < 0) {
		yStep = -1;
		yDist = -y;
	}
	else {
		yStep = 1;
		yDist = y;
	}
	
	setdrawmode(drawvalue);
	setpixel(x2,y2);
	
	if (yDist == 0) {
		i = xDist;
		while (i--) { 
			x2 += xStep;
			setpixel(x2,y2);
		}
	}
	else if (xDist == yDist) {
		i = xDist;
		while (i--) { 
			x2 += xStep;
			y2 += yStep;
			setpixel(x2,y2);
		}
	}
	else if (xDist == 0) {
		i = yDist;
		while (i--) {
			y2 += yStep;
			setpixel(x2,y2);
		}
	}
	else if (xDist > yDist) {
		t8 = 2*yDist;
		t10 = 2*yDist - xDist;
		t9 = 2*(yDist - xDist);
		i = xDist;
		while (i--) {
			x2 += xStep;
			if (t10 < 0) {
				t10 += t8;
			}
			else {
				y2 += yStep;
				t10 += t9;
			}
			setpixel(x2,y2);
		}
	}
	else {
		t8 = 2*xDist;
		t10 = 2*xDist - yDist;
		t9 = 2*(xDist - yDist);
		i = yDist;
		while (i--) {
			y2 += yStep;
			if (t10 < 0) {
				t10 += t8;
			}
			else {
				x2 += xStep;
				t10 += t9;
			}
			setpixel(x2,y2);
		}
	}
	unsetdrawmode();
	return (true);
}

/* xmode -- set display mode */
/* called with either ax contents, or ax,bx,xsize,ysize */
LVAL xmode()
{
	LVAL arg;

	arg = xlgafixnum();
	ourmode1 = (int) getfixnum(arg);
	
	if (moreargs()) {
		arg = xlgafixnum();
		ourmode2 = (int) getfixnum(arg);
		arg = xlgafixnum();
		Xmax = (int) getfixnum(arg) - 1;	/* max x coordinate */
		arg = xlgafixnum();
		Ymax = (int) getfixnum(arg) - 1;	/* max y coordinate */
		xllastarg();
	}
	else {
		ourmode2 = 0;
		switch (ourmode1) {
		case 4:
		case 5:
		case 13: Xmax = 319;
				 Ymax = 199;
				 break;
		case 6:
		case 14: Xmax = 639;
				 Ymax = 199;
				 break;
		case 16: Xmax = 639;
				 Ymax = 349;
				 break;
		case 18: Xmax = 639;	/* added VGA mode */
				 Ymax = 479;
				 break;
		default: Xmax = Ymax = -1; /* not a graphic mode */
				 break;
		}
	}

	setmode(ourmode1,ourmode2); /* set mode */
	bytesperline = (Xmax + 1) / 8;

	
	return (true);
}

/* xcolor -- set color */

LVAL xcolor()
{
	LVAL arg;
	
	arg = xlgafixnum();
	xllastarg();
	
	drawvalue = (char) getfixnum(arg);
	
	return (arg);
}

/* xdraw -- absolute draw */

LVAL xdraw()
{
	LVAL arg = true;
	int newx, newy;
	
	while (moreargs()) {
		arg = xlgafixnum();
		newx = (int) getfixnum(arg);
	
		arg = xlgafixnum();
		newy = (int) getfixnum(arg);
		
		arg = draw(xpos,ypos,newx,newy);
		
		xpos = newx;
		ypos = newy;
	}
	return (arg);
}

/* xdrawrel -- absolute draw */

LVAL xdrawrel()
{
	LVAL arg = true;
	int newx, newy;
	
	while (moreargs()) {
		arg = xlgafixnum();
		newx = xpos + (int) getfixnum(arg);
	
		arg = xlgafixnum();
		newy = ypos + (int) getfixnum(arg);
		
		arg = draw(xpos,ypos,newx,newy);
		
		xpos = newx;
		ypos = newy;
	}
	return (arg);
}

/* xmove -- absolute move, then draw */

LVAL xmove()
{
	LVAL arg;
	
	arg = xlgafixnum();
	xpos = (int) getfixnum(arg);
	
	arg = xlgafixnum();
	ypos = (int) getfixnum(arg);
	
	return (xdraw());
}

/* xmoverel -- relative move */

LVAL xmoverel()
{
	LVAL arg;
	
	arg = xlgafixnum();
	xpos += (int) getfixnum(arg);
	
	arg = xlgafixnum();
	ypos += (int) getfixnum(arg);
	
	return (xdrawrel());
}

#endif
