/* dldmem - xlisp dynamic memory management routines */
/*		Copyright (c) 1985, by David Michael Betz
		All Rights Reserved
		Permission is granted for unrestricted non-commercial use		*/

/* Modified memory management scheme such that array/string space is
   managed here rather than using malloc/free. The advantage of this is
   the array/string space gets compacted allowing better operation when
   available memory is tight or virtual memory is used. XSCHEME does this,
   but probably needs it more since Xscheme functions are kept as compiled
   code in arrays rather than lists. */

/* When this module is used rather than xldmem (and dlimage is used rather
   than xlimage) then ALLOC and EXPAND take an additional second argument
   for array segment allocation size and array segments to add, respectively.
   The ROOM report is changed to indicate array allocation statistics. */


#include "xlisp.h"
#include <stdlib.h>
#include <string.h>

/* node flags */
#ifdef JGC
#define MARK	0x20
#define LEFT	0x40
#else
#define MARK	1
#define LEFT	2
#endif

/* macro to compute the size of a segment */
#define segsize(n) (sizeof(SEGMENT)+((n)-1)*sizeof(struct node))

/* external variables */
extern LVAL obarray,s_gcflag,s_gchook,s_unbound,true;
extern LVAL xlenv,xlfenv,xldenv;
extern char buf[];

/* For vector memory management */
#define VSSIZE 10000

#define vsegsize(n) (sizeof(VSEGMENT)+((n)-1)*sizeof(LVAL))

#define btow_size(n)	(((n) + sizeof(LVAL) - 1) / sizeof(LVAL))

typedef struct vsegment {
	struct vsegment *vs_next;	/* next vector segment */
	LVAL *vs_free;				/* next free location in this segment */
	LVAL *vs_top;				/* top of segment (plus one) */
	LVAL vs_data[1];			/* segment data */
} VSEGMENT;

VSEGMENT *vsegments;	/* list of vector segments */
VSEGMENT *vscurrent;	/* current vector segment */
int vscount;			/* number of vector segments */
LVAL *vfree;			/* next free location in vector space */
LVAL *vtop;				/* top of vector space */


/* variables local to xldmem.c and xlimage.c */
SEGMENT *segs,*lastseg,*fixseg,*charseg;
int anodes,vnodes,nsegs,gccalls;
long nnodes,nfree,total;
LVAL fnodes;

/* forward declarations */
#ifdef ANSI
void compact_vector(VSEGMENT *vseg);
void findvmemory(int size);
void compact(void);
LVAL allocvector(int type, int size);
VSEGMENT *newvsegment(unsigned int n);
#ifdef JMAC
FORWARD LVAL Newnode(int type);
#else
FORWARD LVAL newnode(int type);
#endif
FORWARD VOID mark(LVAL ptr);
FORWARD VOID sweep(void);
FORWARD VOID findmem(void);
FORWARD int  addseg(void);

#else
FORWARD VOID compact_vector();
FORWARD VOID findvmemory();
FORWARD VSEGMENT *newvsegment();
FORWARD VOID compact();
FORWARD LVAL allocvector();
#ifdef JMAC
FORWARD LVAL Newnode();
#else
FORWARD LVAL newnode();
#endif
FORWARD VOID mark();
FORWARD VOID sweep();
FORWARD VOID findmem();
#endif

#ifdef JMAC
LVAL _nnode = 0;
FIXTYPE _tfixed = 0;
int _tint = 0;

#define	newnode(type) (((_nnode = fnodes) != NIL) ? \
 			((fnodes = cdr(_nnode)), \
 			 nfree--, \
 			 (_nnode->n_type = type), \
 			 rplacd(_nnode,NIL), \
 			 _nnode) \
 		    : Newnode(type))
 
#endif


/* xlminit - initialize the dynamic memory module */
VOID xlminit()
{
	LVAL p;
	int i;

	/* initialize our internal variables */
	segs = lastseg = NULL;
	nnodes = nfree = total = 0L;
	nsegs = gccalls = 0;
	anodes = NNODES;
	vnodes = VSSIZE;
	fnodes = NIL;

	/* initialize vector space */
	vsegments = vscurrent = NULL;
	vscount = 0;
	vfree = vtop = NULL;

	/* allocate the fixnum segment */
	if ((fixseg = newsegment(SFIXSIZE)) == NULL)
		xlfatal("insufficient memory");

	/* initialize the fixnum segment */
	p = &fixseg->sg_nodes[0];
	for (i = SFIXMIN; i <= SFIXMAX; ++i) {
		p->n_type = FIXNUM;
		p->n_fixnum = i;
		++p;
	}

	/* allocate the character segment */
	if ((charseg = newsegment(CHARSIZE)) == NULL)
		xlfatal("insufficient memory");

	/* initialize the character segment */
	p = &charseg->sg_nodes[0];
	for (i = CHARMIN; i <= CHARMAX; ++i) {
		p->n_type = CHAR;
		p->n_chcode = i;
		++p;
	}

	/* initialize structures that are marked by the collector */
	obarray = xlenv = xlfenv = xldenv = NIL;
	s_gcflag = s_gchook = NIL;

	/* allocate the evaluation stack */
	if ((xlstkbase = (LVAL **)malloc(EDEPTH * sizeof(LVAL *))) == NULL)
		xlfatal("insufficient memory");
	xlstack = xlstktop = xlstkbase + EDEPTH;

	/* allocate the argument stack */
	if ((xlargstkbase = (LVAL *)malloc(ADEPTH * sizeof(LVAL))) == NULL)
		xlfatal("insufficient memory");
	xlargstktop = xlargstkbase + ADEPTH;
	xlfp = xlsp = xlargstkbase;
	*xlsp++ = NIL;
}

/* cons - construct a new cons node */
LVAL cons(x,y)
  LVAL x,y;
{
	LVAL nnode;

	/* get a free node */
	if ((nnode = fnodes) == NIL) {
		xlstkcheck(2);
		xlprotect(x);
		xlprotect(y);
		findmem();
		if ((nnode = fnodes) == NIL)
			xlabort("insufficient node space");
		xlpopn(2);
	}

	/* unlink the node from the free list */
	fnodes = cdr(nnode);
	--nfree;

	/* initialize the new node */
	nnode->n_type = CONS;
	rplaca(nnode,x);
	rplacd(nnode,y);

	/* return the new node */
	return (nnode);
}

/* cvstring - convert a string to a string node */
LVAL cvstring(str)
  char *str;
{
	LVAL val;
	val = newstring(strlen(str)+1);
	strcpy(getstring(val),str);
	return (val);
}

/* newstring - allocate and initialize a new string */
LVAL newstring(size)
  int size;
{
	LVAL val;
	val = allocvector(STRING,btow_size(size));
	val->n_strlen = size;
	return (val);
}

/* cvsymbol - convert a string to a symbol */
LVAL cvsymbol(pname)
  char *pname;
{
	LVAL val;
	xlsave1(val);
	val = allocvector(SYMBOL,SYMSIZE);
	setvalue(val,s_unbound);
	setfunction(val,s_unbound);
	setpname(val,cvstring(pname));
	xlpop();
	return (val);
}

/* cvsubr - convert a function to a subr or fsubr */
#ifdef ANSI
LVAL cvsubr(LVAL (*fcn)(void), int type, int offset)
#else
LVAL cvsubr(fcn,type,offset)
  LVAL (*fcn)(); int type,offset;
#endif
{
	LVAL val;
	val = newnode(type);
	val->n_subr = fcn;
	val->n_offset = offset;
	return (val);
}

/* cvfile - convert a file pointer to a stream */
LVAL cvfile(fp)
  FILE *fp;
{
	LVAL val;
	val = newnode(STREAM);
	setfile(val,fp);
	setsavech(val,'\0');
#ifdef BETTERIO
	val->n_sflags = 0;
#endif
	return (val);
}

#ifdef JMAC
 
/* cvfixnum - convert an integer to a fixnum node */
LVAL Cvfixnum(n)
  FIXTYPE n;
{
	LVAL val;
	val = newnode(FIXNUM);
	val->n_fixnum = n;
	return (val);
}
#else
/* cvfixnum - convert an integer to a fixnum node */
LVAL cvfixnum(n)
  FIXTYPE n;
{
	LVAL val;
	if (n >= SFIXMIN && n <= SFIXMAX)
		return (&fixseg->sg_nodes[(int)n-SFIXMIN]);
	val = newnode(FIXNUM);
	val->n_fixnum = n;
	return (val);
}
#endif

/* cvflonum - convert a floating point number to a flonum node */
LVAL cvflonum(n)
  FLOTYPE n;
{
	LVAL val;
	val = newnode(FLONUM);
	val->n_flonum = n;
	return (val);
}

/* cvchar - convert an integer to a character node */
#ifdef JMAC
LVAL Cvchar(n)
  int n;
{
	xlerror("character code out of range",cvfixnum((FIXTYPE)n));
	return (NIL);	/* never really returns */
}
#else
LVAL cvchar(n)
  int n;
{
	if (n >= CHARMIN && n <= CHARMAX)
		return (&charseg->sg_nodes[n-CHARMIN]);
	xlerror("character code out of range",cvfixnum((FIXTYPE)n));
	return (NIL);	/* never really returns */
}
#endif

/* newustream - create a new unnamed stream */
LVAL newustream()
{
	LVAL val;
	val = newnode(USTREAM);
	sethead(val,NIL);
	settail(val,NIL);
	return (val);
}

/* newobject - allocate and initialize a new object */
LVAL newobject(cls,size)
  LVAL cls; int size;
{
	LVAL val;
	val = allocvector(OBJECT,size+1);
	setelement(val,0,cls);
	return (val);
}

/* newclosure - allocate and initialize a new closure */
LVAL newclosure(name,type,env,fenv)
  LVAL name,type,env,fenv;
{
	LVAL val;
	val = allocvector(CLOSURE,CLOSIZE);
	setname(val,name);
	settype(val,type);
	setenvi(val,env);
	setfenv(val,fenv);
	return (val);
}

#ifdef STRUCTS
/* newstruct - allocate and initialize a new structure node */
LVAL newstruct(type,size)
 LVAL type; int size;
{
	LVAL val;
	val = allocvector(STRUCT,size+1);
	setelement(val,0,type);
	return (val);
}
#endif


/* newvector - allocate and initialize a new vector */
LVAL newvector(size)
  int size;
{
	return (allocvector(VECTOR,size));
}

/* allocvector - allocate and initialize a new vector node */
LOCAL LVAL allocvector(type,size)
  int type,size;
{
	LVAL val,*p;
	int i;

	if (size > vnodes)
		xlabort("insufficient vector allocation unit size");
		
	xlsave1(val);
	val = newnode(type);

	/* initialize the vector node */
	val->n_type = type;
	val->n_vsize = size;
	val->n_vdata = NULL;

	/* add space for the backpointer */
	++size;
	
	/* make sure there's enough space */
	if ((vfree + size >= vtop) && !scanvmemory(size)) {
		findvmemory(size);
		if (vfree + size >= vtop)
			xlabort("insufficient vector space");
	}

	/* allocate the next available block */
	p = vfree;
	vfree += size;
	
	/* store the backpointer */
	*p++ = val;
	val->n_vdata = p;

	/* set all the elements to NIL */
	for (i = size; i > 1; --i)
		*p++ = NIL;

	/* return the new vector */
	xlpop();
	return (val);
}

/* scanvmemory - look for vector segment with enough space */
/* return success */
int scanvmemory(size)
  int size;
{
	VSEGMENT *vseg;
	if (vscurrent != NULL)
		vscurrent->vs_free = vfree;
	for (vseg = vsegments; vseg != NULL; vseg = vseg->vs_next)
		if (vseg->vs_free + size < vseg->vs_top) {
			vfree = vseg->vs_free;
			vtop = vseg->vs_top;
			vscurrent = vseg;
			return TRUE;
		}
	return FALSE;
}

/* findvmemory - find vector memory (used by 'xsimage.c') */
VOID findvmemory(size)
  int size;
{
	/* first try garbage collecting */
	gc();

	/* look for a vector segment with enough space */
	if (scanvmemory(size)) return;
	
	/* allocate a new vector segment and make it current */
	newvsegment(vnodes);
}

/* newvsegment - create a new vector segment */
VSEGMENT *newvsegment(n)
  unsigned int n;
{
	VSEGMENT *newseg;

	/* allocate the new segment */
	if ((newseg = (VSEGMENT *)calloc(1,vsegsize(n))) == NULL)
		return (NULL);

	if (vscurrent != NULL)
		vscurrent->vs_free = vfree;

	/* initialize the new segment */
	vfree = newseg->vs_free = &newseg->vs_data[0];
	vtop = newseg->vs_top = newseg->vs_free + n;
	newseg->vs_next = vsegments;
	vscurrent = vsegments = newseg;

	/* update the statistics */
	total += (long)vsegsize(n);
	++vscount;

	/* return the new segment */
	return (newseg);
}
 
/* newnode - allocate a new node */
#ifdef JMAC
LOCAL LVAL Newnode(type)
  int type;
{
	LVAL nnode;

	/* get a free node */
	findmem();
	if ((nnode = fnodes) == NIL)
		xlabort("insufficient node space");

	/* unlink the node from the free list */
	fnodes = cdr(nnode);
	nfree -= 1L;

	/* initialize the new node */
	nnode->n_type = type;
	rplacd(nnode,NIL);

	/* return the new node */
	return (nnode);
}
#else
LOCAL LVAL newnode(type)
  int type;
{
	LVAL nnode;

	/* get a free node */
	if ((nnode = fnodes) == NIL) {
		findmem();
		if ((nnode = fnodes) == NIL)
			xlabort("insufficient node space");
	}

	/* unlink the node from the free list */
	fnodes = cdr(nnode);
	nfree -= 1L;

	/* initialize the new node */
	nnode->n_type = type;
	rplacd(nnode,NIL);

	/* return the new node */
	return (nnode);
}
#endif

/* findmem - find more memory by collecting then expanding */
LOCAL VOID findmem()
{
	gc();
	if (nfree < (long)anodes)
		addseg();
}

/* gc - garbage collect (only called here and in xlimage.c) */
VOID gc()
{
	register LVAL **p,*ap,tmp;
	char buf[STRMAX+1];
	LVAL *newfp,fun;

	/* print the start of the gc message */
	if (s_gcflag && getvalue(s_gcflag)) {
		sprintf(buf,"[ gc: total %ld, ",nnodes);
		stdputstr(buf);
	}

	/* mark the obarray, the argument list and the current environment */
	if (obarray)
		mark(obarray);
	if (xlenv)
		mark(xlenv);
	if (xlfenv)
		mark(xlfenv);
	if (xldenv)
		mark(xldenv);

	/* mark the evaluation stack */
	for (p = xlstack; p < xlstktop; ++p)
		if ((tmp = **p) != 0)
			mark(tmp);

	/* mark the argument stack */
	for (ap = xlargstkbase; ap < xlsp; ++ap)
		if ((tmp = *ap) != 0)
			mark(tmp);

	/* compact vector space */
	compact();

		/* sweep memory collecting all unmarked nodes */
	sweep();

	/* count the gc call */
	++gccalls;

	/* call the *gc-hook* if necessary */
	if (s_gchook && ((fun = getvalue(s_gchook)) != 0) ) {
		newfp = xlsp;
		pusharg(cvfixnum((FIXTYPE)(newfp - xlfp)));
		pusharg(fun);
		pusharg(cvfixnum((FIXTYPE)2));
		pusharg(cvfixnum((FIXTYPE)nnodes));
		pusharg(cvfixnum((FIXTYPE)nfree));
		xlfp = newfp;
		xlapply(2);
	}

	/* print the end of the gc message */
	if (s_gcflag && getvalue(s_gcflag)) {
		sprintf(buf,"%ld free ]\n",nfree);
		stdputstr(buf);
	}
}

/* mark - mark all accessible nodes */
LOCAL VOID mark(ptr)
  LVAL ptr;
{
	register LVAL this,prev,tmp;
#ifdef JGC
	int i,n;
#else
	int type,i,n;
#endif

	/* initialize */
	prev = NIL;
	this = ptr;

	/* mark this list */
	for (;;) {

#ifdef JGC
  
/* descend as far as we can */
	while (!(this->n_type & MARK))
  
		/* check cons and symbol nodes */
		if (((i = (this->n_type |= MARK) & TYPEFIELD) == CONS)||
			(i == USTREAM)) {
			if ((tmp = car(this)) != 0) {
				this->n_type |= LEFT;
				rplaca(this,prev);
			}
			else if ((tmp = cdr(this)) != 0)
				rplacd(this,prev);
			else				/* both sides nil */
				break;
			prev = this;			/* step down the branch */
			this = tmp;
		}
		else {
			if (((i & ARRAY) != 0) && (this->n_vdata != 0))
				for (i = 0, n = getsize(this); i < n;)
					if ((tmp = getelement(this,i++)) != 0)
						if ((tmp->n_type & (ARRAY|MARK)) == ARRAY ||
							tmp->n_type == CONS ||
							tmp->n_type == USTREAM)
							mark(tmp);
						else tmp->n_type |= MARK;
						break;
		}

		/* backup to a point where we can continue descending */
		for (;;)

			/* make sure there is a previous node */
			if (prev) {
				if (prev->n_type & LEFT) {		/* came from left side */
					prev->n_type &= ~LEFT;
					tmp = car(prev);
					rplaca(prev,this);
					if ((this = cdr(prev)) != 0) {
						rplacd(prev,tmp);						
						break;
					}
				}
				else {							/* came from right side */
					tmp = cdr(prev);
					rplacd(prev,this);
				}
				this = prev;					/* step back up the branch */
				prev = tmp;
			}
#else
		/* descend as far as we can */
		while (!(this->n_flags & MARK))

			/* check cons and symbol nodes */
			if ((type = ntype(this)) == CONS || type == USTREAM ) { /* TAA fix*/
				if ((tmp = car(this)) != 0) {
					this->n_flags |= MARK|LEFT;
					rplaca(this,prev);
				}
				else if ((tmp = cdr(this)) != 0) {
					this->n_flags |= MARK;
					rplacd(this,prev);
				}
				else {							/* both sides nil */
					this->n_flags |= MARK;
					break;
				}
				prev = this;					/* step down the branch */
				this = tmp;
			}

			/* mark other node types */
			else {
				this->n_flags |= MARK;
				switch (type) {
				case SYMBOL:
				case OBJECT:
				case VECTOR:
				case CLOSURE:
#ifdef STRUCTS
				case STRUCT:
#endif
					if (this->n_vdata)
						for (i = 0, n = getsize(this); --n >= 0; ++i)
							if ((tmp = getelement(this,i)) != 0)
								mark(tmp);
					break;
				}
				break;
			}

		/* backup to a point where we can continue descending */
		for (;;)

			/* make sure there is a previous node */
			if (prev) {
				if (prev->n_flags & LEFT) {		/* came from left side */
					prev->n_flags &= ~LEFT;
					tmp = car(prev);
					rplaca(prev,this);
					if ((this = cdr(prev)) != 0) {
						rplacd(prev,tmp);						
						break;
					}
				}
				else {							/* came from right side */
					tmp = cdr(prev);
					rplacd(prev,this);
				}
				this = prev;					/* step back up the branch */
				prev = tmp;
			}
#endif
			/* no previous node, must be done */
			else
				return;
	}
}

/* compact - compact vector space */
LOCAL VOID compact()
{
	VSEGMENT *vseg;

	/* store the current segment information */
	if (vscurrent)
		vscurrent->vs_free = vfree;

	/* compact each vector segment */
	for (vseg = vsegments; vseg != NULL; vseg = vseg->vs_next)
		compact_vector(vseg);

	/* make the first vector segment current */
	if ((vscurrent = vsegments) != 0) {
		vfree = vscurrent->vs_free;
		vtop = vscurrent->vs_top;
	}
}

/* compact_vector - compact a vector segment */
LOCAL VOID compact_vector(vseg)
  VSEGMENT *vseg;
{
	register LVAL *vdata,*vnext,*vfree,vector;
	register int vsize;

	vdata = vnext = &vseg->vs_data[0];
	vfree = vseg->vs_free;
	while (vdata < vfree) {
		vector = *vdata;
#ifdef JGC
		if ((vector->n_type & TYPEFIELD) == STRING)
#else
		if (vector->n_type == STRING)
#endif
			vsize = btow_size(vector->n_vsize) + 1;
		else
			vsize = vector->n_vsize + 1;
#ifdef JGC
		if (vector->n_type & MARK) {
#else
		if (vector->n_flags & MARK) {
#endif
			if (vdata == vnext) {
				vdata += vsize;
				vnext += vsize;
			}
			else {
				vector->n_vdata = vnext + 1;
				while (vsize > 0) {
					*vnext++ = *vdata++;
					--vsize;
				}
			}
		}
		else
			vdata += vsize;
	}
	vseg->vs_free = vnext;
}

/* sweep - sweep all unmarked nodes and add them to the free list */
/*LOCAL*/ VOID sweep()
{
	SEGMENT *seg;
	LVAL p;
	int n;

	/* empty the free list */
	fnodes = NIL;
	nfree = 0L;

	/* add all unmarked nodes */
	for (seg = segs; seg; seg = seg->sg_next) {
		if (seg == fixseg || seg == charseg) 
#ifdef JGC
			{
			/* remove marks from segments */
			p = &seg->sg_nodes[0];
			for (n = seg->sg_size; --n >= 0;)
				(p++)->n_type &= ~MARK;
			continue;
		}
#else
			continue; /* don't sweep fixed segments */
#endif
		p = &seg->sg_nodes[0];
#ifdef JGC
		for (n = seg->sg_size; --n >= 0;)
			if (p->n_type & MARK)
				(p++)->n_type &= ~MARK;
			else {
				if (((ntype(p)&TYPEFIELD) == STREAM) && getfile(p)

#else
		for (n = seg->sg_size; --n >= 0; ++p)
			if (!(p->n_flags & MARK)) {
				if ((ntype(p) == STREAM) && getfile(p)
#endif
					&& getfile(p) != stdin
					&& getfile(p) != stdout
					&& getfile(p) != stderr)/* taa fix - dont close stdio */
					osclose(getfile(p));
				p->n_type = FREE;
				rplaca(p,NIL);
				rplacd(p,fnodes);
#ifdef JGC
				fnodes = p++;
				nfree++;
			}
#else
				fnodes = p;
				nfree += 1L;
			}

			else
				p->n_flags &= ~MARK;
#endif
	}
}

/* addseg - add a segment to the available memory */
LOCAL int addseg()
{
	SEGMENT *newseg;
	LVAL p;
	int n;

	/* allocate the new segment */
	if (anodes == 0 || (newseg = newsegment(anodes)) == NULL)
		return (FALSE);

	/* add each new node to the free list */
	p = &newseg->sg_nodes[0];
	for (n = anodes; --n >= 0; ++p) {
		rplacd(p,fnodes);
		fnodes = p;
	}

	/* return successfully */
	return (TRUE);
}

/* newsegment - create a new segment (only called here and in xlimage.c) */
SEGMENT *newsegment(n)
  int n;
{
	SEGMENT *newseg;

	/* allocate the new segment */
	if ((newseg = (SEGMENT *)calloc(1,segsize(n))) == NULL)
		return (NULL);
 
	/* initialize the new segment */
	newseg->sg_size = n;
	newseg->sg_next = NULL;
	if (segs)
		lastseg->sg_next = newseg;
	else
		segs = newseg;
	lastseg = newseg;

	/* update the statistics */
	total += (long)segsize(n);
	nnodes += (long)n;
	nfree += (long)n;
	++nsegs;

	/* return the new segment */
	return (newseg);
}
 
/* stats - print memory statistics */
#ifdef ANSI
static void stats(void)
#else
LOCAL VOID stats()
#endif
{
	VSEGMENT *vseg;		/* must calculate vector space used */
	long vnu = 0;
	if (vscurrent != NULL)
		vscurrent->vs_free = vfree;
	for (vseg = vsegments; vseg != NULL; vseg = vseg->vs_next)
			vnu += vseg->vs_free - &vseg->vs_data[0];


	sprintf(buf,"Nodes:        %ld\n",nnodes); stdputstr(buf);
	sprintf(buf,"Free nodes:   %ld\n",nfree);  stdputstr(buf);
	sprintf(buf,"Segments:     %d\n",nsegs);   stdputstr(buf);
	sprintf(buf,"Vector nodes: %ld\n",vnu);	   stdputstr(buf);
	sprintf(buf,"Vector segs:  %d\n",vscount); stdputstr(buf);
	sprintf(buf,"Allocate:     %d\n",anodes);  stdputstr(buf);
	sprintf(buf,"Vec Allocate: %d\n",vnodes);  stdputstr(buf);
	sprintf(buf,"Total:        %ld\n",total);  stdputstr(buf);
	sprintf(buf,"Collections:  %d\n",gccalls); stdputstr(buf);
}

/* xgc - xlisp function to force garbage collection */
LVAL xgc()
{
	/* make sure there aren't any arguments */
	xllastarg();

	/* garbage collect */
	gc();

	/* return nil */
	return (NIL);
}

/* xexpand - xlisp function to force memory expansion */
LVAL xexpand()
{
	LVAL num;
	FIXTYPE n,i;

	/* get the new number to allocate */
	if (moreargs()) {
		num = xlgafixnum();
		n = getfixnum(num);
	}
	else
		n = 1;

	if (moreargs()) {	/* allocate vector segments */
		num = xlgafixnum();
		i = getfixnum(num);
		xllastarg();
		for (; i > 0; i--)
				if (newvsegment(vnodes) == NULL) break;
	}
	else
		xllastarg();

	/* allocate more segments */
	for (i = 0; i < n; i++)
		if (!addseg())
			break;

	/* return the number of segments added */
	return (cvfixnum((FIXTYPE)i));
}

/* xalloc - xlisp function to set the number of nodes to allocate */
LVAL xalloc()
{
	int n,oldn;
	LVAL num;

	/* get the new number to allocate */
	num = xlgafixnum();
	n = (int) getfixnum(num);

	if (moreargs()) {	/* vector allocation */
		num = xlgafixnum();
		xllastarg();
		vnodes = (int) getfixnum(num);
	}
	else
		xllastarg();

	/* set the new number of nodes to allocate */
	oldn = anodes;
	anodes = n;

	/* return the old number */
	return (cvfixnum((FIXTYPE)oldn));
}

/* xmem - xlisp function to print memory statistics */
LVAL xmem()
{
	/* allow one argument for compatiblity with common lisp */
	if (moreargs()) xlgetarg();
	xllastarg();

	/* print the statistics */
	stats();

	/* return nil */
	return (NIL);
}

#ifdef SAVERESTORE
/* xsave - save the memory image */
LVAL xsave()
{
	char *name;

	/* get the file name, verbose flag and print flag */
	name = getstring(xlgetfname());
	xllastarg();

	/* save the memory image */
	return (xlisave(name) ? true : NIL);
}

#ifdef MSC6
/* no optimization which interferes with setjmp */
#pragma optimize("elg",off)
#endif

/* xrestore - restore a saved memory image */
LVAL xrestore()
{
	extern jmp_buf top_level;
	char *name;

	/* get the file name, verbose flag and print flag */
	name = getstring(xlgetfname());
	xllastarg();

	/* restore the saved memory image */
	if (!xlirestore(name))
		return (NIL);

	/* return directly to the top level */
	stdputstr("[ returning to the top level ]\n");
	longjmp(top_level,1);
}
#ifdef MSC6
#pragma optimize("",on)
#endif

#endif

