;;; -*- Mode:Lisp; Package:CLUE; Syntax:COMMON-LISP; Base:10; Lowercase:T -*-
;;;
;;;	------ CLUE Grapher ------ 1990
;;;
;;;	by Tomoru Teruuchi (Fuji Xerox)
;;;	written as a part of a project under the Jeida Common Lisp Committee 1990
;;;
;;;	The source code is completely in public domain.
;;;
;;;	The author(s), the organization the author belongs, Jeida,
;;;	Jeida CLC, Masayuki Ida (the chairman), and Aoyama Gakuin University
;;;	have no warranty about the contents, the results caused by the use,
;;;	and any other possible situations.
;;;
;;;	This codes is written as an experimentation.
;;;	This codes do not mean the best functinality the system supports.

;;;
;;; nodes-and-arcs.lisp

(in-package 'grapher :use '(lisp cluei xlib pcl))

(defvar *all-the-nodes* nil)

(defvar *next-node-index* 0
  "Unique index assigned to each node.")

(defclass node ()
  ((arcs :initform nil :accessor node-arcs)
   (xpos :accessor node-xpos
	 :initarg :xpos)
   (ypos :accessor node-ypos
	 :initarg :ypos)
   (selected-x-offset :accessor node-sxoff
		      :initform nil)
   (selected-y-offset :accessor node-syoff
		      :initform nil)
   (radius :initform :needs-calculation
	   :initarg :radius
	   :accessor node-radius)
   (label :initform nil
	  :initarg :label
	  :accessor node-label)
   (shape :initform :circle
	  :initarg :shape
	  :accessor node-shape)
   (unique :initform (incf *next-node-index*)
	   :accessor node-unique))
  )

(defmacro node-name (node)
  `(or (node-label ,node)(format nil "Unnaned node ~d" (node-unique ,node)))
)

(defmethod initialize-instance :after ((node node) &rest initargs)
  (declare (ignore initargs))
  (push node *all-the-nodes*))

(defmethod print-object ((node node) stream)
  (let ((name (node-name node)))
    (lcl::printing-random-object (node stream :typep :no-pointer)
      (write-string name stream))))

(defmacro stream-write-strings (stream contents &optional (ret-p nil))
  `(progn 
     (dolist (item (list ,@contents))
	   (stream-write-string ,stream item))
     (if ,ret-p (stream-fresh-line ,stream)))
)

(defmethod print-object--stream ((node node) stream &optional (non-return-p nil))
  (let ((name (node-name node)))
    (stream-write-strings stream ("#<Node " name "/ (" (format nil "~d" (node-xpos node)) ","
				   (format nil "~d" (node-ypos node)) ")>") (null non-return-p))
    )
)

(defmacro map-over-nodes ((node-var) &body body)
  `(dolist (,node-var *all-the-nodes*)
     ,@body))

(defmethod (setf node-label) :after (new-value (node node))
  (declare (ignore new-value))
  (setf (node-radius node) :needs-calculation))

(defmethod move ((node node) new-xpos new-ypos)
  (with-slots (xpos ypos) node
    (setf xpos new-xpos
	  ypos new-ypos)))

(defmethod delete-self ((node node))
  (dolist (arc (node-arcs node))
    (delete-self arc))
  (setf *all-the-nodes* (delete node *all-the-nodes*)))

;;;
;;; added for extension.
(defmacro draw-my-label (contact x y string &optional (erase-p nil) (gc nil))
  `(multiple-value-bind (width height)
			(string-width ,contact ,string)
			(draw-string ,contact ,string
				     (- ,x (floor width 2))
			 (+ ,y (floor height 2)) ,erase-p ,gc))
)

(defmethod present-self ((node node) window &optional (erase-p nil))
  (calculate-radius node window)
  (with-slots (xpos ypos radius label) node
    (draw-circle window xpos ypos radius erase-p)
    (draw-my-label window xpos ypos 
		(or label 
		    (format nil "~d" (node-unique node))
		    )
		erase-p)
    ))

(defmethod calculate-radius ((node node) window)
  (with-slots (radius label) node
    (when (and (stringp label)
	       (zerop (length label)))
      (setf label nil))
    (when (eq radius :needs-calculation)
      (setf radius (ceiling 
		    (+ (string-width window 
				  (or label
				      (format nil "~d"
					      (node-unique node)))
				  ) 10) 2)))
    radius))
;;;
;;; find node with its name
;;;
(defun find-node (label &optional (no-error nil))
  (let ((a-node nil))
    (dolist (current-node *all-the-nodes*)
	    (if (equal label (node-label current-node))
		(setq a-node current-node)))
    (if (null a-node)
	(dolist (current-node *all-the-nodes*)
		(if (equal (string-to-integer label)
			   (node-unique current-node))
		    (setq a-node current-node))))
    (if (and (null a-node) (null no-error))
	(stream-write-strings *i-stream* 
			      ("***Error*** : Node \"" label "\" does not exist!!!") t))
    a-node)
)

(defclass arc ()
  ((mark :initform nil
	 :accessor arc-mark)
   (label :initform nil         ;added for extension
	  :initarg :label       ;added for extension
	  :accessor arc-label)  ;added for extension
   (unique :initform nil        ;added for extension
	   :initarg :unique     ;added for extension
	   :accessor arc-unique);added for extension
   (width :initform nil         ;added for extension
	  :accessor arc-width)  ;added for extension
   (height :initform nil        ;added for extension
	   :accessor arc-height);added for extension
   (node1 :accessor arc-node1
	  :initarg :node1)
   (node2 :accessor arc-node2
	  :initarg :node2))
  )

(defmethod add-arc ((node node) (arc arc))
  (with-slots (arcs) node
    (push arc arcs)))

(defmethod remove-arc ((node node) (arc arc))
  (with-slots (arcs) node
    (setf arcs (delete arc arcs))))

(defmethod initialize-instance :after ((arc arc) &rest initargs)
  (declare (ignore initargs))
  (with-slots (node1 node2) arc
    (add-arc node1 arc)
    (add-arc node2 arc)
    (setf (arc-unique  arc) (format nil "~a <--> ~a" (node-name node1) (node-name node2))))
)


(defmethod print-object ((arc arc) stream)
  (with-slots (node1 node2) arc
    (lcl::printing-random-object (arc stream :typep)
      (format stream "~a <--> ~a" node1 node2))))

(defmethod print-object--stream ((arc arc) stream &optional (non-return-p nil))
  (with-slots (label unique) arc
	      (stream-write-strings stream 
				     ("#< Arc / "(if label label unique) ">") (null non-return-p)))
)

(defmacro map-over-arcs ((arc-var) &body body)
  (let ((mark-var (make-symbol "MARK"))
	(node-var (make-symbol "NODE")))
    `(let ((,mark-var (list nil)))
       (dolist (,node-var *all-the-nodes*)
	 (dolist (,arc-var (node-arcs ,node-var))
           (unless (eq (arc-mark ,arc-var) ,mark-var)
             ,@body
             (setf (arc-mark ,arc-var) ,mark-var)))))))


(defmethod delete-self ((arc arc))
  (with-slots (node1 node2) arc
    (remove-arc node1 arc)
    (remove-arc node2 arc)))

(defmethod find-arc ((node1 node)(node2 node))
  (with-slots (arcs) node1
	      (dolist (arc arcs)
		      (if (or (eq (arc-node1 arc) node2)
			      (eq (arc-node2 arc) node2))
			  (return arc))))
)

;;;
;;; This function is added for extension (mouse sensing)
;;;
(defun find-arc-box (x y)
  (let ((ret-arc nil))
  (dolist (a-node *all-the-nodes*)
	  (dolist (an-arc (node-arcs a-node))
		  (let ((node1 (arc-node1 an-arc))
			(node2 (arc-node2 an-arc)) midx midy 
			(width (arc-width an-arc))
			(height (arc-height an-arc)))
		    (setq midx (+ (node-xpos node1) (ceiling (- (node-xpos node2)
								(node-xpos node1)) 2)))
		    (setq midy (+ (node-ypos node1) (ceiling (- (node-ypos node2)
								(node-ypos node1)) 2)))
		    (if (and (<= (- midx (floor width 2)) x)
			     (>= (+ midx (ceiling width 2)) x)
			     (<= (- midy (floor height 2)) y)
			     (>= (+ midy (ceiling height 2)) y))
			(setq ret-arc an-arc)))))
  ret-arc)
)

(defmethod arc-name ((arc arc))    ;added for extension
  (with-slots (label unique) arc   ;added for extension
	      (if label label      ;added for extension
		unique))           ;added for extension
)                                  ;added for extension

(defmethod present-self ((arc arc) window &optional (erase-p nil))
  (with-slots (node1 node2) arc
    (multiple-value-bind (x1 y1 x2 y2)
        (find-edges-of-nodes (node-radius node1)
			     (node-xpos node1) (node-ypos node1)
			     (node-radius node2)
			     (node-xpos node2) (node-ypos node2))
      (draw-my-arc window x1 y1 x2 y2 arc erase-p)
      ;;renamed from draw-line(original) to draw-my-arc
      ;;because CLX defined draw-line already and
      ;;its parameters format is different(GC is required)
      ))
)

(defun find-edges-of-nodes (r1 xpos1 ypos1 r2 xpos2 ypos2)
  (let* ((dx (- xpos2 xpos1))
	 (dy (- ypos2 ypos1))
	 (length (isqrt (+ (* dx dx) (* dy dy)))))
    (values (+ xpos1 (ceiling (* dx r1) length))
	    (+ ypos1 (ceiling (* dy r1) length))
	    (- xpos2 (floor (* dx r2) length))
	    (- ypos2 (floor (* dy r2) length)))))
;;;
;;; This function is added for extension.
;;; This may calculate arc's label's edge position.
;;;
(defun find-edges-of-arcs (width height xpos1 ypos1 xpos2 ypos2)
  (let* ((dx (- xpos2 xpos1))(midx (+ xpos1 (ceiling dx 2)))
	 (dy (- ypos2 ypos1))(midy (+ ypos1 (ceiling dy 2)))
	 (half-height (ceiling height 2))
	 (half-width (ceiling width 2)))
    (if (or (eql dx 0) (> (abs (/ dy dx)) (/ height width)))
	(values (- midx (* (if (< dy 0) -1 1)(floor  (* half-height dx) dy)))
		(- midy (* (if (< dy 0) -1 1) half-height))
		(+ midx (* (if (< dy 0) -1 1)(ceiling (* half-height dx) dy)))
		(+ midy (* (if (< dy 0) -1 1) half-height)))
      (values (- midx (* (if (< dx 0) -1 1) half-width)) 
	      (- midy (* (if (< dx 0) -1 1) (floor (* half-width dy) dx)))
	      (+ midx (* (if (< dx 0) -1 1) half-width))
	      (+ midy (* (if (< dx 0) -1 1) (ceiling (* half-width dy) dx))))))
)

