;;;-*-Mode:LISP; Package:(PCL Lisp 1000); Base:10; Syntax:Common-lisp -*-
;;;
;;; *************************************************************************
;;; Copyright (c) 1985, 1986, 1987, 1988 Xerox Corporation.
;;; All rights reserved.
;;;
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted.  Any distribution of this
;;; software or derivative works must comply with all applicable United
;;; States export control laws.
;;; 
;;; This software is made available AS IS, and Xerox Corporation makes no
;;; warranty about the software, its performance or its conformity to any
;;; specification.
;;; 
;;; Any person obtaining a copy of this software is requested to send their
;;; name and post office or electronic mail address to:
;;;   CommonLoops Coordinator
;;;   Xerox PARC
;;;   3333 Coyote Hill Rd.
;;;   Palo Alto, CA 94304
;;; (or send Arpanet mail to CommonLoops-Coordinator.pa@Xerox.arpa)
;;;
;;; Suggestions, comments and requests for improvements are also welcome.
;;; *************************************************************************
;;;

(in-package 'pcl)

(defmethod class-prototype ((c standard-class))
; (or (slot-value--class c 'prototype)
;     (put-slot--class c 'prototype (make-instance c)))
  (or (slot-value c 'prototype)
      (setf (slot-value c 'prototype) (make-instance c))))

;;;
;;; class-direct-methods
;;; class-direct-generic-functions
;;;
;;; direct-methods is the primary value we store, direct-generic-functions
;;; is a derivative value that we also maintain.  For performance reasons
;;; we don't always keep the direct-generic-functions value up to date.
;;; 

(defmethod class-direct-methods ((class standard-class))
; (slot-value--class class 'direct-methods)
  (slot-value class 'direct-methods))

(defmethod (setf class-direct-methods) (nv (class standard-class))
  (with-slots (direct-methods direct-generic-functions)
	      class
    (without-interrupts
      (setf direct-methods nv
	    direct-generic-functions ()))))

(defmethod class-direct-generic-functions ((class standard-class))
  (with-slots (direct-generic-functions)
	      class
    (or direct-generic-functions
	(setf direct-generic-functions
	      (gathering ((result (collecting-once)))
		(dolist (m (class-direct-methods class))
		  (gather (method-generic-function m) result)))))))

#|

A class is 'fully defined' iff its class precedence list can be computed. This
is true when it and all of its superclasses are defined.

A class is 'defined' when a defclass form which defines that class has been
defined.

A class is 'undefined' when no class by that name exists.

  (defclass forward-referenced-standard-class (class) ())

  (defmethod make-instance ..
  (defmethod compute-class-precedence-list ..
  (defmethod compatible..
  (defmethod class-for-redefinition ..




when something changes, we walk down the tree passing the following information

where the changes actually happened and what it was,
whether the class at this point in the tree is fully defined,
whether the class at this point in the tree used to be fully defined


|#



;;; There are 6 cases:
;;;   F  --  F    | + 2 cases because a fully defined class can
;;;   F  --  NF   | either have instances or not have instances
;;;   NF --  NF
;;;   NF --  F
;;;
;;; 

;; n cases
;;; 1   a fully-defined class changes, all its subclasses are also fully
;;;     defined.  Just have to propagate info.
;;; 2   fully defined class with instances is changed to become not
;;;     fully defined.
;;; 3 a class which is not fully defined becomes fully defined because
;;;    some other class far above it becomes fully defined.
;;; 4  a not fully defined class with some defined subclasses is changed.
;;; 
(defmethod PROPAGATE-CLASS-UPDATE ((class standard-class)
				   new-fully-defined-p
				   old-fully-defined-p
				   changed-class
				   &rest key-arguments	  ;hidden argument
				   &key (its-direct-superclasses () supers-p)
					(its-options () options-p)
					(its-direct-slots  () slots-p))
  (declare (ignore its-direct-superclasses its-options its-direct-slots))

  (let ((old-cpl (class-precedence-list class))
	(new-cpl ()))
    
    (when new-fully-defined-p
      (cond (supers-p
	     (setq new-cpl (compute-class-precedence-list class))
	     (setf (class-precedence-list class) new-cpl)
	     (update-slots--class class)
	     (update-constructors--class class))
	    ((or options-p slots-p)
	     (update-slots--class class)
	     (update-constructors--class class))))
    
    
    ;; Propagate all the change information down through our subclasses.
    ;; For each subclass we also compute its new and old fully-defined-p
    ;; status.  The details of this computation are specific to PCL.
    (dolist (subclass (class-direct-subclasses class))
      (let ((sub-forward-referenced-supers
	      (class-forward-referenced-supers subclass))
	    sub-newly-defined-p
	    sub-oldly-defined-p)
	(cond ((null sub-forward-referenced-supers)
	       ;; The subclass used to be fully defined.  By definition,
	       ;; that means that we used to be fully defined.  It also
	       ;; means that if we just became not-fully-defined this
	       ;; subclass must now become not fully defined.
	       (setq sub-newly-defined-p new-fully-defined-p
		     sub-oldly-defined-p 't)
	       (when (not new-fully-defined-p)
		 (setf (class-forward-referenced-supers subclass) (list class))))
	      ((and (eq (car sub-forward-referenced-supers) class)
		    (null (cdr sub-forward-referenced-supers)))
	       ;; The only reason this subclass used to be not fully defined
	       ;; is because we used to be not fully defined.  That means
	       ;; that if we are still not fully defined so is this subclass
	       ;; and if we just became fully defined so does this subclass.
	       (setq sub-newly-defined-p new-fully-defined-p
		     sub-oldly-defined-p old-fully-defined-p)
	       (when new-fully-defined-p
		 (setf (class-forward-referenced-supers subclass) ())))
	      (t
	       ;; The general case is where there were multiple reasons
	       ;; why this subclass used to be not-fully-defined.  That
	       ;; means it stays not fully defined, but we may add or
	       ;; remove ourselves as a reason.
	       (setq sub-newly-defined-p nil
		     sub-oldly-defined-p nil)
	       (setf (class-forward-referenced-supers subclass)
		     (if new-fully-defined-p
			 (delete class sub-forward-referenced-supers)
			 (pushnew class sub-forward-referenced-supers)))))
	
	(apply #'propagate-class-update subclass
					sub-newly-defined-p
					sub-oldly-defined-p
					changed-class
					key-arguments)))
    
    (when new-fully-defined-p
      (cond (supers-p
	     (when (eq class changed-class)
	       (update-method-inheritance class
					  old-cpl
					  (class-precedence-list class)))))
      (setf (class-all-default-initargs class)
	    (collect-all-default-initargs class new-cpl)))
    
    ))
  
(defun update-slots--class (class)
  (let* ((cpl (class-precedence-list class))
	 (local-slots (class-local-slots class))
	 (slots ())
	 (instance-slots ())
	 (class-slots ()))

    ;; If I saved accessor/reader prefixes somewhere, I could save time
    ;; here.  Also, if merge actually kept track of whether something
    ;; changed that would save time.
    
    (merge-accessor/reader-prefixes local-slots (class-options class))
;   (check-accessor/reader-compatibility local-slots)

    (setq slots (order-slotds class
			      (collect-slotds class local-slots cpl)
			      cpl))
  
    (dolist (slot slots)
      (case (slotd-allocation slot)
	(:instance (push slot instance-slots))
	(:class (push slot class-slots))))
    (setq instance-slots (nreverse instance-slots)
	  class-slots (nreverse class-slots))

    (update-slot-accessors--class class slots)
      
    ;; If there is a change in the shape of the instances then the
    ;; old class is now obsolete.  
    (let* ((new-instance-slots-layout (mapcar #'slotd-name instance-slots))
	   (owrapper
	     (class-wrapper class))
	   (nwrapper
	     (if (or (null owrapper)
		     (not (equal new-instance-slots-layout
				 (wrapper-instance-slots-layout owrapper))))
		 (make-class-wrapper class)
		 owrapper)))
      
      (setf (class-no-of-instance-slots class) (length instance-slots))
      (setf (class-slots class) slots)
      (setf (class-wrapper class) nwrapper)
      (setf (wrapper-instance-slots-layout nwrapper)
	    new-instance-slots-layout)
      (setf (wrapper-class-slots nwrapper)
	    (mapcar #'(lambda (slotd)
			(cons (slotd-name slotd)
			      (funcall (slotd-initfunction slotd))))
		    class-slots))

      (when (and owrapper (neq owrapper nwrapper))
	(invalidate-wrapper owrapper)))))

(defun update-slot-accessors--class (class slots)
  (update-slot-accessors--class-1 class
				  slots
				  (class-slots class)))

(defun update-slot-accessors--class-1 (class slotds old-slotds)
  (dolist (slotd slotds)
    (let* ((slot-name (slotd-name slotd))
	   (old-slotd (dolist (o old-slotds)
			(when (eq slot-name (slotd-name o)) (return o))))
	   (forcep (and old-slotd
			(not (equal (slotd-type old-slotd)
				    (slotd-type slotd)))))
;          (old-accessors (and old-slotd (slotd-accessors old-slotd)))
	   (old-readers (and old-slotd (slotd-readers old-slotd)))
	   (old-writers (and old-slotd (slotd-writers old-slotd))))
      (update-slot-accessors--class-2
	class slotd forcep (slotd-readers slotd) old-readers :reader)
      (update-slot-accessors--class-2
	class slotd forcep (slotd-writers slotd) old-writers :writer))))

(defun update-slot-accessors--class-2 (class slotd forcep new old acc/rea)
  (flet ((get-gf (name) (ensure-generic-function name)))

    (dolist (gf-name old)
      (when (or forcep (not (memq gf-name new)))
	(ecase acc/rea
	  (:reader
	    (remove-reader-method class slotd (get-gf gf-name)))
	  (:writer
	    (remove-writer-method class slotd (get-gf gf-name))))))

    (dolist (gf-name new)
      (when (or forcep (not (memq gf-name old)))
	(ecase acc/rea
	  (:reader
	    (add-reader-method class slotd (get-gf gf-name)))
	  (:writer
	    (add-writer-method class slotd (get-gf gf-name))
	    (when (and (listp gf-name)
		       (eq (car gf-name) 'setf))
	      (do-standard-defsetf-1 (cadr gf-name)))))))))



(defun update-constructors--class (class)
  (let ((options (class-options class))
	(old-constructors (class-constructors class))
	(new-constructors ()))
    (dolist (option options)
      (when (and (listp option) (eq (car option) ':constructor))
	(push (cdr option) new-constructors)))
    ;; First get rid of any constructors which don't appear in the new
    ;; constructors.  Don't need to compare the old and new definitions
    ;; of this constructor, just get rid of it if it shouldn't have a
    ;; definition according to the new options.
    (dolist (old old-constructors)
      (unless (assq (car old) new-constructors)
	(fmakunbound (car old))))

    ;; Now define all the new constructors.  As an optimization (and
    ;; an important one for that matter) check to see if there was an
    ;; old definition of this constructor which was the same as the
    ;; new definition and if so don't bother doing the new definition.    
    (dolist (new new-constructors)
      (unless (equal new (assq (car new) old-constructors))
	(let ((constructor (apply #'make-constructor class (cdr new))))
	  (setq constructor (set-function-name constructor (car new)))
	  (setf (symbol-function (car new)) constructor))))

    (setf (class-constructors class) new-constructors)))


(defun collect-all-default-initargs (class cpl)
  (declare (ignore class))
  (labels ((walk (tail)
	     (if (null tail)
		 nil
		 (append (class-default-initargs (car tail))
			 (walk (cdr tail))))))
    (let ((initargs (walk cpl)))
      (delete-duplicates initargs :test #'eq :key #'car :from-end t))))

;;;
;;; CLASS-FOR-REDEFINITION old-class proto-class name ds-options slotds
;;; protocol: class definition
;;; 
;;; When a class is being defined, and a class with that name already exists
;;; a decision must be made as to what to use for the new class object, and
;;; whether to update the old class object.  For this, class-for-redefinition
;;; is called with the old class object, the prototype of the new class, and
;;; the name ds-options and slotds corresponding to the new definition.
;;; It should return the class object to use as the new definition.  It is
;;; OK for this to be old-class if that is appropriate.
;;; 
(defmethod class-for-redefinition ((old-class standard-class)
				   (proto-class standard-class)
				   name
				   local-supers
				   local-slot-slotds
				   extra)
  (declare (ignore name local-supers local-slot-slotds extra))
  old-class)



(defmethod update-method-inheritance ((class standard-class) old-cpl new-cpl)
  (let ((already-done ()))
    (unless (equal old-cpl new-cpl)
      (let ((shared-tail (shared-tail old-cpl new-cpl)))
	(dolist (old old-cpl)
	  (unless (memq old shared-tail)
	    (setq already-done
		  (update-method-inheritance-1 old already-done))))
	(dolist (new new-cpl)
	  (unless (memq new old-cpl)
	    (setq already-done
		  (update-method-inheritance-1 new already-done))))))))

(defun update-method-inheritance-1 (class already-done)
  (let ((methods (class-direct-methods class)))
    (dolist (method methods)
      (let ((gf (method-generic-function method)))
	(if (null gf)
	    (error "A method still on a class, has no generic function?")
	    (unless (memq gf already-done)
	      (push gf already-done)
	      (invalidate-generic-function gf)))))
    already-done))

(defun shared-tail (list1 list2)
  (let ((l1 (length list1))
	(l2 (length list2)))
    (cond ((< l1 l2)
	   (setq list2 (nthcdr (- l2 l1) list2)))
	  ((< l2 l1)
	   (setq list1 (nthcdr (- l1 l2) list1))))
    (do ((shared-tail list1)
	 (t1 list1 (cdr t1))
	 (t2 list2 (cdr t2)))
	((null t1) shared-tail)
      (if (eq (car t1) (car t2))
	  ()
	  (setq shared-tail (cdr t1))))))

(defmethod collect-slotds ((class standard-class) local-slots cpl)    
  (let ((slots ()))
    (labels ((collect-one-class (local-slots pos)
	       (setq local-slots (copy-list local-slots))
	       ;; For each of the slots we have already found, get the
	       ;; slot description this class has for a slot by that
	       ;; name or NIL if this class has no direct-slot by that
	       ;; name.
	       (dolist (slot slots)
		 (let ((hit (dolist (ls local-slots)
			      (when (eq (slotd-name ls) (car slot))
				(return ls)))))
		   (when hit (setq local-slots (delq hit local-slots)))
		   (push hit (cdr slot))))
	       ;; For any remaining direct-slots this class has, create
	       ;; a new entry in slots.  Add a bunch of trailing NILs
	       ;; to the entry to represent the classes that didn't
	       ;; have direct slots for this slot.
	       (dolist (ls local-slots)
		 (push (list* (slotd-name ls)
			      ls
			      (make-list pos :initial-element nil))
		       slots)))
	     (collect-cpl (cpl-tail)
	       (cond ((null cpl-tail) 0)
		     (t
		      (let ((pos (1+ (collect-cpl (cdr cpl-tail)))))
			(collect-one-class (class-local-slots (car cpl-tail))
					   pos)
			pos)))))

      (collect-one-class local-slots (collect-cpl (cdr cpl)))

      
      ;; Now use compute-effective-slotd to condense all the slot
      ;; descriptions for slots of the same name into one slot
      ;; description for that slot.
      (mapcar #'(lambda (descriptions)
		  (compute-effective-slotd class (cdr descriptions)))
	      slots))))

(defmethod order-slotds ((class standard-class) slotds cpl)
  (let ((superclass-slots (reverse (mapcar #'class-slots (cdr cpl)))))
    (flet ((superclass-slot-ordering (slotd)
	     ;; If a slot with this name appears in one of our supers,
	     ;; return two values:
	     ;;   1  the class-slots of the most general class this
	     ;;      slot appears in
	     ;;   2  a tail of the first value such that the its
	     ;;      first element is the relevant slotd
	     ;;
	     ;; The way to think of these two values is that they specify
	     ;; the first class which included this slot AND the position
	     ;; within instances of that class the slot appeared.
	     ;;
	     (dolist (order superclass-slots)
	       (let ((p (member slotd order
				:test #'(lambda (a b)
					  (and (eq (slotd-name a)
						   (slotd-name b))
					       (eq (slotd-allocation a)
						   (slotd-allocation b)))))))
		 (when p (return (values order p)))))))
      (sort slotds
	#'(lambda (x y)
	    (cond ((eq (slotd-allocation x) (slotd-allocation y))
		   (let (x-class-slots x-tail y-class-slots y-tail)
		     (multiple-value-setq (x-class-slots x-tail)
					  (superclass-slot-ordering x))
		     (multiple-value-setq (y-class-slots y-tail)
					  (superclass-slot-ordering y))
		     (cond ((null y-class-slots) 't)
			   ((null x-class-slots) 'nil)
			   ((eq x-class-slots y-class-slots)
			    (tailp y-tail x-tail))
			   (t
			    (memq y-class-slots
				  (memq x-class-slots
					superclass-slots))))))
		  ((eq (slotd-allocation x) ':instance) 't)
		  (t nil)))))))

(defmethod COMPUTE-EFFECTIVE-SLOTD ((class standard-class) slotds)
  (let* ((unsupplied *slotd-unsupplied*)
	 (name unsupplied)
	 (keyword unsupplied)
	 (initfunction unsupplied)
	 (initform unsupplied)
	 (initargs nil)
	 (allocation unsupplied)
	 (type unsupplied)
;	 (accessors (and (car slotds)
;			 (slotd-accessors (car slotds))))
	 (readers   (and (car slotds)
			 (slotd-readers (car slotds))))
	 (writers   (and (car slotds)
			 (slotd-writers (car slotds)))))

    (dolist (slotd slotds)
      (when slotd
	(when (eq name unsupplied)
	  (setq name (slotd-name slotd)
		keyword (slotd-keyword slotd)))
	(when (eq initform unsupplied)
	  (setq initform (slotd-initform slotd))
	  (setq initfunction (slotd-initfunction slotd)))
	(when (eq allocation unsupplied)
	  (setq allocation (slotd-allocation slotd)))
	(setq initargs (append (slotd-initargs slotd) initargs))
	(let ((slotd-type (slotd-type slotd)))
	  (setq type (cond ((eq type unsupplied) slotd-type)
			   ((eq slotd-type unsupplied) type)
			   ((subtypep type slotd-type) type)
			   (t `(and ,type ,slotd-type)))))))
    
;   (when (eq initform unsupplied)
;     (setq initfunction nil))
    (when (eq type unsupplied)
      (setq type 't))
    (when (eq allocation unsupplied)
      (setq allocation :instance))
    
    (make-slotd class
		:name name
		:keyword keyword
		:initform initform
		:initfunction initfunction
		:initargs initargs
		:allocation allocation
		:type type
;		:accessors accessors
		:readers readers
		:writers writers)))

(defmethod compute-class-precedence-list ((root standard-class))
  (let ((*cpl* ())
	(*root* root)
	(*must-precede-alist* ()))
    (declare (special *cpl* *root* *must-precede-alist*))
    ;; We start by computing two values.
    ;;   CPL
    ;;     The depth-first left-to-right up to joins walk of the supers tree.
    ;;     This is equivalent to breadth-first left-to-right walk of the
    ;;     tree with all but the last occurence of a class removed from
    ;;     the resulting list.  This is in fact how the walk is implemented.
    ;;
    ;;   MUST-PRECEDE-ALIST
    ;;     An alist of the must-precede relations. The car of each element
    ;;     of the must-precede-alist is a class, the cdr is all the classes
    ;;     which either:
    ;;       have this class as a local super
    ;;      or
    ;;       appear before this class in some other class's local-supers.
    ;;
    ;;     Thus, the must-precede-alist reflects the two constraints that:
    ;;       1. A class must appear in the CPL before its local supers.
    ;;       2. Order of local supers is preserved in the CPL.
    ;;
    (labels
         ((must-move-p (element list &aux move)
	    (dolist (must-precede (cdr (assq element *must-precede-alist*)))
	      (when (setq move (memq must-precede (cdr list)))
		(return move))))
	  (find-farthest-move (element move)
	    (let ((closure (compute-must-precedes-closure element)))
	      (dolist (must-precede closure)
		(setq move (or (memq must-precede move) move)))
	      move))
	  (compute-must-precedes-closure (class)
	    (let ((closure ()))
	      (labels ((walk (element path)
			 (when (memq element path)
			   (class-ordering-error
			     *root* element path *must-precede-alist*))
			 (dolist (precede
				   (cdr (assq element
					      *must-precede-alist*)))
			   (unless (memq precede closure)
			     (pushnew precede closure)
			     (walk precede (cons element path))))))
		(walk class nil)
		closure))))
      
      (walk-supers *root*)			;Do the walk
      ;; For each class in the cpl, make sure that there are no classes after
      ;; it which should be before it.  We do this by cdring down the list,
      ;; making sure that for each element of the list, none of its
      ;; must-precedes come after it in the list. If we find one, we use the
      ;; transitive closure of the must-precedes (call find-farthest-move) to
      ;; see where the class must really be moved. We use a hand-coded loop
      ;; so that we can splice things in and out of the CPL as we go.
      (let ((tail *cpl*)
	    (element nil)
	    (move nil))
	(loop (when (null tail) (return))
	      (setq element (car tail)
		    move (must-move-p element tail))
	      (cond (move
		     (setq move (find-farthest-move element move))
		     (setf (cdr move) (cons element (cdr move)))
		     (setf (car tail) (cadr tail)) ;Interlisp delete is OK
		     (setf (cdr tail) (cddr tail)) ;since it will never be
						   ;last element of list.
		     )
		    (t
		     (setq tail (cdr tail)))))
	(copy-list *cpl*)))))

(defun walk-supers (class &optional precedence)
  (declare (special *cpl* *root* *must-precede-alist*))
  (let ((elem (assq class *must-precede-alist*)))
    (if elem
	(setf (cdr elem) (union (cdr elem) precedence))
	(push (cons class precedence) *must-precede-alist*)))
  (let ((rsupers (reverse (cons class (class-local-supers class)))))
    (iterate ((sup (list-elements rsupers))
	      (pre (list-tails (cdr rsupers)))
	      (temp (eachtime nil)))
      ;; Make sure this element of supers is OK.
      ;;  Actually, there is an important design decision hidden in
      ;;  here.  Namely, at what time should symbols in a class's
      ;;  local-supers be changed to the class objects they are
      ;;  forward referencing.
      ;;   1. At first allocate-instance (compute-class-precedence-list)?
      ;;   2. When the forward referenced class is first defined?
      ;;  This code does #1.
      (cond ((classp sup))
	    ((and (symbolp sup)
		  (setq temp (find-class sup t)))
	     ;; This is a forward reference to a class which is
	     ;; now defined.  Replace the symbol in the local
	     ;; supers with the actual class object, and set sup.
	     (nsubst temp sup (class-local-supers class))
	     (setq sup temp))
	    ((symbolp sup)
	     (error "While computing the class-precedence-list for ~
                             the class ~S.~%~
                             The class ~S (from the local supers of ~S) ~
                             is undefined."
		    (class-name *root*) sup (class-name class)))
	    (t
	     (error "INTERNAL ERROR --~%~
                             While computing the class-precedence-list for ~
                             the class ~S,~%~
                             ~S appeared in the local supers of ~S."
		    *root* sup class)))
      (walk-supers sup pre))
    (unless (memq class *cpl*) (push class *cpl*))))

(defun class-ordering-error (root element path must-precede-alist)
  (setq path (cons element (reverse (memq element (reverse path)))))
  (flet ((pretty (class) (or (class-name class) class)))
    (let ((explanations ()))
      (do ((tail path (cdr tail)))
	  ((null (cdr tail)))
	(let ((after (cadr tail))
	      (before (car tail)))
	  (if (memq after (class-local-supers before))
	      (push (format nil
			    "~% ~A must precede ~A -- ~
                              ~A is in the local supers of ~A."
			    (pretty before) (pretty after)
			    (pretty after) (pretty before))
		    explanations)
	      (dolist (common-precede
			(intersection
			  (cdr (assq after must-precede-alist))
			  (cdr (assq before must-precede-alist))))
		(when (memq after (memq before
					(class-local-supers common-precede)))
		  (push (format nil
				"~% ~A must precede ~A -- ~
                                  ~A has local supers ~S."
				(pretty before) (pretty after)
				(pretty common-precede)
				(mapcar #'pretty
					(class-local-supers common-precede)))
			explanations))))))
      (error "While computing the class-precedence-list for the class ~A:~%~
              There is a circular constraint through the classes:~{ ~A~}.~%~
              This arises because:~{~A~}"
	     (pretty root)
	     (mapcar #'pretty path)
	     (reverse explanations)))))

(defmethod compute-method-inheritance-list ((class standard-class)
					    local-supers)
  (declare (ignore local-supers))
  (compute-class-precedence-list class))

(defmethod compatible-meta-class-change-p (class proto-new-class)
  (eq (class-of class) (class-of proto-new-class)))

(defmethod check-super-metaclass-compatibility ((class t) (new-super t))
  (unless (eq (class-of class) (class-of new-super))
    (error "The class ~S was specified as a~%super-class of the class ~S;~%~
            but the meta-classes ~S and~%~S are incompatible."
	   new-super class (class-of new-super) (class-of class))))

(defun classp (x)
  (and (iwmc-class-p x) (typep--class x 'standard-class)))




;;;
;;; make-instances-obsolete can be called by user code.  It will cause the
;;; next access to the instance (as defined in 88-002R) to trap through the
;;; update-instance-for-redefined-class mechanism.
;;; 
(defmethod make-instances-obsolete ((class standard-class))
  (invalidate-wrapper (class-wrapper class)))

;;;
;;; obsolete-instance-trap is the internal trap that is called when we see
;;; an obsolete instance.  The times when it is called are:
;;;   - when the instance is involved in method lookup
;;;   - when attempting to access a slot of an instance
;;;
;;; It is not called by class-of, wrapper-of, or any of the low-level instance
;;; access macros.
;;;
;;; Of course these times when it is called are an internal implementation
;;; detail of PCL and are not part of the documented description of when the
;;; obsolete instance update happens.  The documented description is as it
;;; appears in 88-002R.
;;;
;;; This has to return the new wrapper, so it counts on all the methods on
;;; obsolete-instance-trap-internal to return the new wrapper.  It also does
;;; a little internal error checking to make sure that the traps are only
;;; happening when they should, and that the trap methods are computing
;;; apropriate new wrappers.
;;; 
(defun obsolete-instance-trap (wrapper instance)
  (unless (zerop (wrapper-cache-no wrapper))
    (error "Internal error.~@
            Got an OBSOLETE-INSTANCE-TRAP, but wrapper-cache-no isn't 0."))
  (let ((new-wrapper (obsolete-instance-trap-internal (wrapper-class wrapper)
						      wrapper
						      instance)))
    (when (zerop (wrapper-cache-no new-wrapper))
      (error "Internal Error.~@
              wrapper-cache-no is still 0 after OBSOLETE-INSTANCE-TRAP."))
    new-wrapper))

;;;
;;; local  --> local        transfer 
;;; local  --> shared       discard
;;; local  -->  --          discard
;;; shared --> local        transfer
;;; shared --> shared       discard
;;; shared -->  --          discard
;;;  --    --> local        added
;;;  --    --> shared        --
;;; 
(defmacro obsolete-instance-trap-1 (wrapper-fetcher slots-fetcher)
  `(let* ((guts (allocate-instance class))
	  (nwrapper (class-wrapper class))
	  (olayout (wrapper-instance-slots-layout owrapper))
	  (nlayout (wrapper-instance-slots-layout nwrapper))
	  (oslots (,slots-fetcher instance))
	  (nslots (,slots-fetcher guts))
	  (oclass-slots (wrapper-class-slots owrapper))
	  (added ())
	  (discarded ())
	  (plist ()))

     ;; Go through all the old local slots.
     (iterate ((oslot (list-elements olayout))
	       (opos (interval :from 0)))
       (let ((npos (posq oslot nslots)))
	 (if npos
	     (setf (svref nslots npos) (svref oslots opos))
	     (progn (push oslot discarded)
		    (setf (getf plist oslot) (svref oslots opos))))))

     ;; Go through all the old shared slots.
     (iterate ((oclass-slot-and-val (list-elements oclass-slots)))
       (let ((name (car oclass-slot-and-val))
	     (val (cdr oclass-slot-and-val)))
	 (let ((npos (posq name nlayout)))
	   (if npos
	       (setf (svref nslots npos) (cdr oclass-slot-and-val))
	       (progn (push name discarded)
		      (setf (getf plist name) val))))))

     ;; Go through all the new local and shared slots to compute
     ;; the added slots.
     (dolist (nslot nlayout)
       (unless (or (memq nslot olayout)
		   (assq nslot oclass-slots))
	 (push nslot added)))

     (without-interrupts
       (setf (,wrapper-fetcher instance) nwrapper)
       (setf (,slots-fetcher instance) nslots))

     (update-instance-for-redefined-class instance
					  added
					  discarded
					  plist)
     nwrapper))

(defmethod obsolete-instance-trap-internal ((class standard-class)
					    owrapper
					    instance)
  (obsolete-instance-trap-1 iwmc-class-class-wrapper
			    iwmc-class-static-slots))

(defmethod obsolete-instance-trap-internal ((class funcallable-standard-class)
					    owrapper
					    instance)
  (obsolete-instance-trap-1 funcallable-instance-wrapper
			    funcallable-instance-static-slots))



;;;
;;;
;;;
(defmacro change-class-internal (wrapper-fetcher slots-fetcher)
  `(let* ((old-class (class-of instance))
	  (copy (allocate-instance old-class))
	  (guts (allocate-instance new-class))
	  (new-wrapper (,wrapper-fetcher guts))
	  (old-wrapper (class-wrapper old-class))
	  (old-layout (wrapper-instance-slots-layout old-wrapper))
	  (new-layout (wrapper-instance-slots-layout new-wrapper))
	  (old-slots (,slots-fetcher instance))
	  (new-slots (,slots-fetcher guts))
	  (old-class-slots (wrapper-class-slots old-wrapper)))

    ;;
    ;; "The values of local slots specified by both the class Cto and
    ;; Cfrom are retained.  If such a local slot was unbound, it remains
    ;; unbound."
    ;;     
    (iterate ((new-slot (list-elements new-layout))
	      (new-position (interval :from 0)))
      (let ((old-position (position new-slot old-layout :test #'eq)))
	(when old-position
	  (setf (svref new-slots new-position)
		(svref old-slots old-position)))))

    ;;
    ;; "The values of slots specified as shared in the class Cfrom and
    ;; as local in the class Cto are retained."
    ;;
    (iterate ((slot-and-val (list-elements old-class-slots)))
      (let ((position (position (car slot-and-val) new-layout :test #'eq)))
	(when position
	  (setf (svref new-slots position) (cdr slot-and-val)))))

    ;; Make the copy point to the old instance's storage, and make the
    ;; old instance point to the new storage.
    (without-interrupts
      (setf (,slots-fetcher copy) old-slots)
      
      (setf (,wrapper-fetcher instance) new-wrapper)
      (setf (,slots-fetcher instance) new-slots))

    (update-instance-for-different-class copy instance)
    instance))

(defmethod change-class ((instance object)
			 (new-class standard-class))
  (unless (iwmc-class-p instance)
    (error "Can't change the class of ~S to ~S~@
            because it isn't already an instance whose metaclass is~%~S."
	   instance
	   new-class
	   'standard-class))
  (change-class-internal iwmc-class-class-wrapper
			 iwmc-class-static-slots))

(defmethod change-class ((instance object)
			 (new-class funcallable-standard-class))
  (unless (funcallable-instance-p instance)
    (error "Can't change the class of ~S to ~S~@
            because it isn't already an instance whose metaclass is~%~S."
	   instance
	   new-class
	   'funcallable-standard-class))
  (change-class-internal funcallable-instance-wrapper
			 funcallable-instance-static-slots))


;in high.lisp
;(defmethod change-class ((instance t) (new-class symbol))
;  (change-class instance (find-class symbol)))





(defun named-object-print-function (instance stream
				    &optional (extra nil extra-p))
  (declare (ignore depth))
  (printing-random-thing (instance stream)
    (if extra-p					
	(format stream "~A ~S ~:S"
		(capitalize-words (class-name (class-of instance)))
		(slot-value instance 'name)
		extra)
	(format stream "~A ~S"
		(capitalize-words (class-name (class-of instance)))
		(slot-value instance 'name)))))