;;; -*- Mode:Common-Lisp; Package:Yes-Way; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************

;-------------------------------------------------------------------------------

;;; Summary window code.

(defun message-window-width ()
  *message-window-width*
)

(defun message-window-height ()
  *message-window-height*
)

(defun message-window-font-map ()
  *message-window-font-map*
)

(defun make-yw-right-button-menu ()
  (make-instance 'w:menu
                 :Label "Which Operation"
		 :Item-list *right-button-menu-commands*
                 :pop-up t
		 :Columns 1
  )
)

(defun default-message-pane-edges ()
  (list 0 0 *message-window-width* *message-window-height*)
)

(defun max-offset (x-offset y-offset)
  (destructuring-bind (left top right bottom) (Default-Message-Pane-Edges)
    (values
      (if (= x-offset 0)
	  nil
          (truncate (/ (- (send tv:default-screen :Inside-Width) (- right left))
		       x-offset
		    )
          )
      )
      (if (= y-offset 0)
	  nil
	  (truncate
	    (/ (- (send tv:default-screen :Inside-Height) (- bottom top))
	       y-offset
	    )
	  )
      )
    )
  )
)

(defun frequencies (list so-far)
"Returns a list of the frequencies of the different elements in List.  These
 are accumulated into So-Far.  For example, if List is '(a b c d a f e c a d b)
 then this function returns: '((e 0) (f 0) (d 1) (c 1) (b 2) (a 2)).  Note the
 frequencies are zero indexed.
"
  (if list
      (if (keywordp list)
	  nil
	  (frequencies (remove (first list) list :Test #'equal)
		       (cons (list (first list)
				   (tv:occurences (first list) (rest list))
			     )
			     so-far
		       )
	  )
      )
      so-far
  )
)

(defun compute-new-index (offset least max)
  (if (= offset 0)
      (first least)
      (if (< offset 0)
	  (if (<= (first least) 0)
	      0
	      (max 0
		 (- (* offset (- (first least) 1)))
	      )
	  )
	  (if (>= (first least) max)
	      (* max offset)
	      (max 0 (* offset (+ (first least) 1)))
	  )
      )
  )
)

(defun offsetting-window-edges
   (&optional (resource 'summary-windows)
    (x-offset *new-headers-window-x-offset*)
    (y-offset *new-headers-window-y-offset*)
   )
"Returns a new set of edges for a new window.  It uses the existing members of
 Resource to compute the new edges so that the new panes walk across the screen.
"
  (let ((existing (Map-yw-Resource-Return
		    #'(lambda (win &rest ignore)
			(multiple-value-list (send win :position))
		      )
		      resource
		    #'(lambda (window)
			(and (not (send window :User-Defined-Position-P))
			     (send window :Mailstreams)
			)
		      )
		  )
       )
      )
      (if existing
	  (let ((modded-x
		  (if (= x-offset 0)
		      (mapcar #'first existing)
		      (mapcar
			#'(lambda (x) (round (/ (first  x) (abs x-offset))))
			Existing
		      )
		  )
		)
		(modded-y
		  (if (= y-offset 0)
		      (mapcar #'second existing)
		      (mapcar
			#'(lambda (x) (round (/ (second  x) (abs y-offset))))
			Existing
		      )
		  )
		)
	       )
	       (multiple-value-bind (max-x max-y) (Max-Offset x-offset y-offset)
		 (let ((least-x (tv:Get-Least (Frequencies modded-x nil)))
		       (least-y (tv:Get-Least (Frequencies modded-y nil)))
		      )
		      (let ((new-x (Compute-New-Index x-offset least-x max-x))
			    (new-y (Compute-New-Index y-offset least-y max-y))
			   )
			   (values new-x new-y)
		      )
		 )
	       )
	  )
	  (values-list *default-ideal-message-display-pane-position*)
      )
  )
)

;;;**************************************************************************;;;
;;;  Clean-Break-Mixin                                                       ;;;
;;;    Prevent it from using a background window for error notification.     ;;;
;;;     IV:                                                                  ;;;
;;;       None                                                               ;;;
;;;     Components:                                                          ;;;
;;;       None                                                               ;;;
;;;     Methods:                                                             ;;;
;;;       :whopper :notice                                                   ;;;
;;;**************************************************************************;;;

(defflavor clean-break-mixin () ())


(defwhopper (clean-break-mixin :notice) (event &rest args)
  (case event
    (:error
       (if (loop for w = self then (tv:sheet-superior w) until (null w)
		 always (tv:sheet-exposed-p w))
	   ;; If window is visible, go ahead and use it.
	   (tv:wait-till-safe-for-error self 'tv:sheet-can-get-lock self)
	 ;; Otherwise must notify.
	 (or (let ((tv:process-is-in-error self))
	       (tv:wait-till-safe-for-error self 'tv:notify-possible-p self))
	     (progn
	       (tv:notify self "Process ~A got an error"
                       (tv:process-name current-process))
	       ;; If notifying for an error, remain "in error" until selected
	       (let ((tv:process-is-in-error self))
		 (process-wait
                   "Selected" #'(lambda (w) (eq tv:selected-window w)) self)
		 nil)))));)
    (otherwise (lexpr-continue-whopper event args))))

;(defmethod (clean-break-mixin :print-notification)) (&rest args)
;  (declare (arglist time string window-of-interest))
;  (lexpr-send self :print-notification-on-self args))

;===============================================================================

(defflavor message-display-pane
	   (mailstreams
	    (owner nil)
	    (icon nil)
	    (mouse-down-line nil)
	    (all-selected-lines nil)
	    (old-number-of-items 0)
	    (inits-to-undo nil)
	    (user-defined-position-p nil)
	    (item-list nil)
	    (screen-image nil)
	    (top-item nil)
	    (target-top-item nil)
	    (current-item nil)
	    (item-blinker nil)
	    (item-type-alist nil)
	   )
	   (tv:menu-execute-mixin
	    w:scroll-bar-mixin
	    tv:borders-mixin
	    w:scroll-bar-mixin
	    tv:window
	    tv:dont-select-with-mouse-mixin
	    tv:Essential-Mouse
	   )
  (:Default-Init-Plist
    :Label nil
    :Font-Map (message-window-font-map)
    :Activate-p t
    :Scroll-Bar 2
    :Scroll-bar-always-displayed nil
    :Height (message-window-height)
    :Width (message-window-width)
    :Expose-P t
    :Activate-P t
    :Deexposed-Typeout-Action :Permit
    :Save-Bits t
    :blinker-p nil
  )
  :Initable-Instance-Variables
  :Settable-Instance-Variables
  :Gettable-Instance-Variables
)

(defmethod (message-display-pane :after :init) (ignore)
  (setq item-blinker (tv:make-blinker self 'tv:hollow-rectangular-blinker
				      :visibility nil
		     )
  )
  (send self :Reposition nil)
)

(defmethod (message-display-pane :number-of-items) ()
  (length item-list)
)

(defmethod (message-display-pane :before :end-of-line-exception) ()
  (throw 'end-of-line t)
)

(defmethod (message-display-pane :end-of-page-exception) (&rest ignore)
  (setf (tv:sheet-end-page-flag self) 0)
  (throw 'end-of-page t)
)

(defmethod (message-display-pane :scroll-position) ()
  (values (or top-item 0) (send self :number-of-items) tv:line-height
	  (tv:sheet-number-of-inside-lines)
  )
)

;;;Edited by Tom Gruber            18 Feb 92  11:31
(defmethod (message-display-pane :scroll-to)
	   (new-top &optional (type :absolute) (force-redisplay nil) &aux delta)
  (or top-item (setq top-item 0))
  (and (eq type :relative) (setq new-top (+ top-item new-top)))
  (setq new-top (max (min new-top (- (send self :number-of-items)
				     (send self :screen-lines)
				  )
		     )
		     0
		)
  )
  (setq delta (- top-item new-top)) ;; positive delta means move higher
  (setq top-item new-top)
  ;; item number to top of screen, i.e. scroll down.
  (if (and (not force-redisplay) (< (abs delta) (send self :screen-lines)))
      (progn (send self :scroll-bitmap delta)
	     (send self :move-items delta)
	     (if (< delta 0)
		 ;;; scroll backwards, so redisplay at bottom of screen.
		 (send self :display
		       (max 0 (min (+ top-item
				      (- (send self :Screen-Lines) (- delta) 1)
				   )
				   (- (send self :number-of-items) 1)
			      )
		       )
		       (max 0 (min (+ top-item (- (send self :Screen-Lines) 1))
				   (- (send self :number-of-items) 1)
			      )
		       )
		 )
		 (send self :display
		       top-item
		       (max 0 (min (+ top-item (- delta 1))
				   (- (send self :number-of-items) 1)
			      )
		       )
		 )
	     )
      )
      (send self :Redisplay t t)
  )
  (send self :new-scroll-position)
)

(defmethod (message-display-pane :after :refresh) (&optional type)
  "When exposed, draw in the items."
  (and (or (not tv:restored-bits-p) (eq type :size-changed))
       (send self :redisplay t t)
  )
)


(defmethod (message-display-pane :redisplay)
	   (&optional (full-p nil) (force-p nil))
  (ignore full-p)
  (if (and full-p (not force-p))
      (send self :Scroll-To top-item)
      (progn (or top-item (setq top-item 0))
	     (send self :Flush-Screen-Image)
	     (send self :display top-item
		   (max 0 (min (+ top-item (send self :screen-lines))
			       (- (send self :number-of-items) 1)
			  )
		   )
	     )
      )
  )
)

(defmethod (Message-Display-Pane :Bottom-Possible-Item) ()
  (or top-item (setq top-item 0))
  (+ top-item (- (send self :Screen-Lines) 1))
)

(defmethod (Message-Display-Pane :Bottom-Item) ()
  (min (send self :Bottom-Possible-Item) (- (send self :Number-Of-Items) 1))
)

(defmethod (Message-Display-Pane :Pop-Up-By-N) (item n)
  "Called when n messages have been removed starting at Item."
  (let ((items-location (cons nil item-list)))
       (loop for loc on items-location
	     for index from 0
	     when (eq item (second loc))
	     do (setf (rest loc) (rest (nthcdr n loc)))
	        (setq item-list (rest items-location))
	        (if (and (>= index top-item)
			 (<= index (+ top-item (send self :Screen-Lines)))
		    )
		    ;;; Then we have deleted an on-screen item.
		    (if (zerop top-item)
			(send self :Display index (send self :Bottom-Item))
			(if (= (send self :Bottom-Item)
			       (send self :Bottom-Possible-Item)
			    )
			    ;; Then the screen is full anyway, so we don't
			    ;; need to scroll.
			    (send self :Display index (send self :Bottom-Item))
			    ;; The item at the bottom of the screen is the
			    ;; actual bottom item, so we can't pop up the items
			    ;; off the bottom of the window.  We therefore have
			    ;; to pop down items to fill the gap.
			           ;; Scroll up to show now visible items
			    (progn (send self :Scroll-To (- n) :Relative)
				   ;; Refresh from the pop-up point.
				   (send self :Display index
					 (+ index (send self :Screen-Lines))
				   )
			    )
;;; The old slow way.
;;;			    (send self :Scroll-To (- n) :Relative t)
			)
		    )
		    ;;; The item is off the screen, so don't scroll.
		    nil
		)
		(return nil)
       )
  )
)

(defmethod (Message-Display-Pane :Push-Down-By-N) (new-items message)
"Pushes Message down in the window by N slots because new-items have been
inserted above it."
  (let ((message-index (and message
			    (position message screen-image
				      :Test #'eq
;				      :Key 'object
				      :Key #'(lambda (x) (and x (Object x)))
				      :End (send self :Screen-Lines)
			    )
		       )
	)
       )
       (if message-index
	   (if (>= (length new-items)
		   (- (send self :Screen-Lines) message-index 1)
	       )
	       ;;; Then all messages below the new ones will be obscured so just
	       ;;; refresh the new items.
	       (send self :Display
		     (position (first new-items) item-list :Test #'eq)
		     (position (first (last new-items)) item-list :Test #'eq)
	       )
	       ;;; There should probably be some sort of bitblt thing
	       ;;; here for this.
	       (let ((message-position-in-item-list
		       (position message item-list :Test #'eq :Key 'object)
		     )
		    )
		    (send self :Display
			  (position (first new-items) item-list :Test #'eq)
			  (+ message-position-in-item-list
			     (- (send self :Screen-Lines) message-index 1)
			  )
		    )
	       )
	   )
	   (let ((start (position (first new-items) item-list :Test #'eq))
		 (end (position (first (last new-items)) item-list :Test #'eq))
		)
	        (and start end (send self :Display start end))
	   )
       )
  )
)

(defmethod (Message-Display-Pane :Flush-Line) (line)
  (setf (aref screen-image line) nil)
  (send self :clear-between-cursorposes
	0 (* tv:line-height line)
	(tv:sheet-inside-right self) (* tv:line-height line)
  )
)

(defmethod (Message-Display-Pane :Flush-Lines) (from-line to-line)
  (loop for index from from-line to to-line
	do (setf (aref screen-image index) nil)
  )
  (if (>= to-line from-line)
      (send self :clear-between-cursorposes 0 (* tv:line-height from-line)
	    (tv:sheet-inside-right self) (* tv:line-height to-line)
      )
      nil
  )
)

(defmethod (message-display-pane :flush-screen-image) ()
;  (loop for index from 0 below (length screen-image)
;	do (send self :Flush-Line index)
;  )
  (send self :Flush-Lines 0 (- (send self :Screen-Lines) 1))
)

(defmethod (message-display-pane :move-items) (by-lines)
  (loop for index from 0 below (length screen-image)
	for item = (aref screen-image index)
	when item
	do (incf (top    item) (* by-lines tv:line-height))
	   (incf (bottom item) (* by-lines tv:line-height))
  )
  (if (>= by-lines 0)
      (loop for index from (- (send self :Screen-Lines) 1) downto by-lines
	    do (setf (aref screen-image index)
		     (aref screen-image (- index by-lines))
	       )
	    finally (loop for index from 0 below by-lines
			  do (setf (aref screen-image index) nil)
		    )
      )
      (loop for index from (- by-lines) below (send self :Screen-Lines)
	    do (setf (aref screen-image (- index (- by-lines)))
		     (aref screen-image index)
	       )
	    finally (loop for index
			  from (- (send self :Screen-Lines) (- by-lines))
			  below (send self :Screen-Lines)
			  do (setf (aref screen-image index) nil)
		    )
      )
  )
  (if (< by-lines 0)
      (send self :Flush-Lines (- (send self :Screen-Lines) (- by-lines))
	    (- (send self :Screen-Lines) 2)
      )
;      (loop for index from (- (send self :Screen-Lines) (- by-lines))
;	    below (- (send self :Screen-Lines) 1)
;	    do (send self :Flush-Line index)
;      )
     (send self :Flush-Lines 0 (- by-lines 1))
;      (loop for index from 0 below by-lines
;	    do (send self :Flush-Line index)
;      )
  )
)

(defun foo (Window)
  (print :-------)
  (Loop for i from 0 below (min (length (send Window :Screen-Image)) 25)
	for item = (aref (send Window :Screen-Image) i)
	if item
	do (print (list i item (Object item) (top    item) (bottom item)))
	else do (print i)
  )
)

(defmethod (message-display-pane :scroll-bitmap) (by-lines)
  (if (< by-lines 0)
      (let ((offset (* -1 tv:line-height by-lines)))
	   (tv:prepare-sheet (self)
	     (send self :Bitblt tv:alu-setz 
		   (tv:sheet-inside-width self)
		   offset tv:100%-white 0 0 0 0
	     )
	     (send self :bitblt-within-sheet tv:alu-seta
		   (tv:sheet-inside-width self)
		   (- (tv:sheet-inside-height self) offset)
		   (tv:sheet-inside-left self)
		   (+ (tv:sheet-inside-top self) offset)
		   (tv:sheet-inside-left self)
		   (tv:sheet-inside-top self)
	     )
	     (send self :Bitblt tv:alu-setz 
		   (tv:sheet-inside-width self)
		   offset
		   tv:100%-white 0 0
		   0 (* (- (send self :Screen-Lines) by-lines) tv:line-height)
	     )
	   )
      )
      (let ((offset (* tv:line-height by-lines)))
	   (tv:prepare-sheet (self)
	     (send self :Bitblt tv:alu-setz 
		   (tv:sheet-inside-width self)
		   offset
		   tv:100%-white 0 0
		   0 (* (- (send self :Screen-Lines) by-lines) tv:line-height)
	     )
	     (bitblt tv:alu-seta
		     (tv:sheet-inside-width self)
		     (- (tv:sheet-inside-height self) offset)
		     tv:screen-array
		     (tv:sheet-inside-left self)
		     (tv:sheet-inside-top self)
		     tv:bit-array
		     (tv:sheet-inside-left self)
		     (+ (tv:sheet-inside-top self) offset)
	     )
	     (bitblt tv:alu-seta
		     (tv:sheet-inside-width self)
		     (- (tv:sheet-inside-height self) offset)
		     tv:bit-array
		     (tv:sheet-inside-left self)
		     (+ (tv:sheet-inside-top self) offset)
		     tv:screen-array
		     (tv:sheet-inside-left self)
		     (+ (tv:sheet-inside-top self) offset)
	     )
	     (send self :Bitblt tv:alu-setz 
		   (tv:sheet-inside-width self)
		   offset
		   tv:100%-white 0 0
		   0 0
	     )
	   )
      )
  )
)

(defmethod (message-display-pane :scroll-redisplay) (new-top delta &aux nlines)
  (tv:sheet-home self)
  (setq nlines (send self :screen-lines))
  (cond	((> delta 0)				;Scrolling forward
         ;; Make room for the new lines and then have :REDISPLAY
         ;; print them on the screen.
	 (setq delta (min delta nlines))
	 (setq top-item new-top)
	 (send self :redisplay (- nlines delta) nlines))
	((< delta 0)				;Scrolling backward
	 (setq delta (min (- delta) nlines))
	 (setq top-item new-top)
	 (send self :redisplay 0 delta)))
  (send self :new-scroll-position))

(defmethod (message-display-pane :display) (start end)
  "Arguments are item numbers -- assumes screen area already erased."
; (and (y-or-n-p "?") (cl:break))
  (or top-item (setq top-item 0))
  (let ((lines (send self :Screen-Lines)))
       (let ((to-fetch nil))
	    (loop with limit = (send self :number-of-items)
		  for screen-image-index
		      from (- start top-item) to (- end top-item)
		  for item-number from start to end
		  when (and (>= screen-image-index 0) (< item-number limit)
			    (< screen-image-index lines)
		       )
		  do (let ((message
			     (Object (send self :item-of-number item-number))
			   )
			  )
			  (if (is-present (cache-envelope message))
			      nil
			      (push message to-fetch)
			  )
		     )
	    )
	    (setq to-fetch (nreverse to-fetch))
	    (and to-fetch (get-missing-envelopes to-fetch))
	    (loop with limit = (send self :number-of-items)
		  for screen-image-index
		      from (- start top-item) to (- end top-item)
		  for item-number from start to end
		  when (and (>= screen-image-index 0) (< item-number limit)
			    (< screen-image-index lines)
		       )
		  do (tv:sheet-set-cursorpos
		       self 0 (* tv:line-height screen-image-index)
		     )
		     (send self :Print-Item
			   (send self :item-of-number item-number)
			   screen-image-index item-number
		     )
		  finally (if (>= item-number (- limit 1))
			      (send self :Clear-Below-Items
				    (max 0 screen-image-index)
			      )
			      nil
			  )
	    )
       )
  )
)

(defmethod (Message-Display-Pane :Clear-Below-Items) (start)
  (send self :Flush-Lines start (- (send self :Screen-Lines) 1))
;  (loop for index from start below (send self :Screen-Lines)
;	do (send self :Flush-Line index)
;  )
)


(defmethod (Message-Display-Pane :redisplay-selected-items) (items)
  (let ((indices
	  (loop for item in (list-if-not items)
		for index = (position item screen-image :Test #'eq)
		when index
		collect index
	  )
	)
       )
       (send self :display (+ top-item (apply #'min indices))
	     (+ top-item (apply #'max indices))
       )
  )
)

(defmethod (message-display-pane :print-item) (item line-no item-no)
  (ignore line-no item-no)
; (and (y-or-n-p "~S ?" item) (cl:break))
  (catch 'end-of-line
    (catch 'end-of-page
      (multiple-value-bind (x y) (send self :read-cursorpos)
	(setf (left item) x)
	(setf (top item) y)
         (tv:prepare-sheet (self)
	   (tv:%draw-rectangle (- (tv:sheet-inside-width self) x)
			       (min (- (tv:sheet-inside-height self) y)
				    (tv:sheet-line-height self)
			       )
			       (+ x (tv:sheet-inside-left self))
			       (+ y (tv:sheet-inside-top  self))
			       (tv:sheet-erase-aluf self) self
           )
	 )
      )
      (setf (aref screen-image line-no) item)
      (princ (object item) self)
    )
  )
  (multiple-value-bind (x y) (send self :read-cursorpos)
    (setf (right item) x)
    (setf (bottom item) (- (+ tv:line-height y) (send self :vsp)))
  )
)

;;;Edited by Tom Gruber            22 Jan 92  17:50
(defmethod (message-display-pane :After :print-item) (item line-no item-no)
  (ignore line-no item-no)
  (let ((message (Object item)))
       (if (Selected-P message)
	   (loop for type in (Selected-P message)
		 do (send self :inverse-video-item item type)
	   )
	   nil
       )
  )
)

(defmethod (message-display-pane :item-of-number) (number)
  (nth number item-list) ;;; Probably needs optimisation
)

;;;Edited by Tom Gruber            18 Feb 92  11:31
(defmethod (message-display-pane :after :new-scroll-position) (&rest ignore)
  (let ((not-displayed
          (loop for index from 0 below (send self :screen-lines)
                for item in (nthcdr top-item item-list)
                when (not (equal (aref screen-image index) item))
                collect index
          )
        )
       )
       (if not-displayed
           (send self :display (apply #'min not-displayed)
                 (apply #'max not-displayed)
           )
           nil
       )
  )
  (let ((blank-lines (- (send self :number-of-items) top-item
			(send self :screen-lines) -1
		     )
	)
       )
       (if (< blank-lines 0)
	   (progn (send self :clear-between-cursorposes
		     0 (- (tv:sheet-inside-bottom self)
			  (* -1 tv:line-height blank-lines)
		       )
		     (tv:sheet-inside-right self) (tv:sheet-inside-bottom self)
		  )
;		  (loop for line from 0 below (- blank-lines)
;			do (setf (aref screen-image (- (send self :screen-lines)
;						       line 1
;						    )
;			         )
;				 nil
;			   )
;		  )
	   )
	   nil
       )
  )
  (tv:mouse-wakeup)
)

(defmethod (message-display-pane :mouse-sensitive-item) (x y)
  (loop for index from 0 below (length screen-image)
	for item = (aref screen-image index)
	when item
	do (let ((Left (left item) )
		 (Top (top item))
		 (Right (right item))
		 (Bottom (bottom item))
		)
	        (if (and Left Top Right Bottom
			 (>= x left)
			 (<= x right)
			 (>= y top)
			 (<= y bottom)
		    )
		    (return (values item (type item) Left top
				    (- Right left)
				    (- Bottom top)
			    )
		    )
		    nil
		)
	   )
  )
)

(defmethod (message-display-pane :mouse-moves) (x y)
  (tv:mouse-set-blinker-cursorpos)
  (multiple-value-bind (item nil left top wid hei)
      (send self :mouse-sensitive-item x y)
    (and top (setq top (1- top)))
    (cond (item
	   (tv:blinker-set-cursorpos
	     item-blinker
	     Left
;	     Top
;	     (- left (tv:sheet-inside-left))
;	     (- top (tv:sheet-inside-top))
;	     (+ left (tv:sheet-inside-left))
	     (+ top (tv:sheet-inside-top))
	   )
	   (tv:blinker-set-size item-blinker wid hei)
	   (tv:blinker-set-visibility item-blinker t)
	   (setq current-item item))
	  (t
	   (tv:blinker-set-visibility item-blinker nil)
	   (setq current-item nil)))))

;-------------------------------------------------------------------------------

;;;Edited by Tom Gruber            23 Jan 92  13:10
(defmethod (Message-Display-Pane :clean-up) ()
  (setq User-Defined-Position-P nil)
  (send self :Forget-Items t)
  (send self :set-item-list nil)
  (if screen-image
      (loop for i from 0 below (array-dimension screen-image 0)
	    do (setf (aref screen-image i) nil)
      )
      nil
  )
  (setq top-item nil)
  (setq target-top-item nil)
  (send self :clear-window)
  (send self :set-filter :Recycled-Window)
  (setq mailstreams nil)
  (setq mouse-down-line nil)
  (setq all-selected-lines nil)
  (setq old-number-of-items 0)
  (tv:sheet-force-access (self)
    (send self :set-label "")
    (send self :bury)
  )
)

(defmethod (Message-Display-Pane :Reinitialise) (inits)
  (catch-error (send self :reinitialise-1 inits) t)
)

(defmethod (Message-Display-Pane :Reinitialise-1) (inits)
  (setq inits-to-undo
        (loop for (name value) on inits by #'cddr
	      append (if *reset-windows-on-deallocation*
			 (if (and (consp value) (equal :Values (first value)))
			     (list name
				   (list :Values
					 (multiple-value-list (send self name))
				   )
			     )
			     (list name (send self name))
			 )
			 nil
		     )
	      do
	      (send self :Expose) ;;; Make sure we're still exposed after any
	      ;;; previous inits.  Gruber seemed to have an output hold problem.
	      (if (and (consp value) (equal :Values (first value)))
		  (lexpr-send self :send-if-handles
			(intern (string-append "SET-" (symbol-name name))
				:Keyword
			)
			(second value)
		  )
		  (send self :send-if-handles
			(intern (string-append "SET-" (symbol-name name))
				:Keyword
			)
			value
		  )
	      )
	)
  )
)

(defmethod (Message-Display-Pane :After :Icon-For-Window) ()
  (let ((str (find-if #'stringp (send self :Label))))
       (if str (send icon :Set-Text str) nil)
  )
)

(defmethod (Message-Display-Pane :Icon-For-Window) ()
  (or icon
      (let ((new-icon (if (fboundp 'w:make-label-icon)
			  (funcall 'w:make-label-icon self)
			  nil
		      )
	    )
	   )
	   (setq icon new-icon)
	   icon
      )
  )
)

(defun ideal-message-display-pane-position (window)
  (ignore window)
  (Offsetting-Window-Edges)
)

(defmethod (Message-Display-Pane :visible-selected-messages) ()
  (loop for index from 0 below (array-dimension screen-image 0)
	for item = (aref screen-image index)
	for message = (if item (Message-From-Display-Item item) nil)
	when (and item (Selected-P message))
	collect message
  )
)

(defmethod (Message-Display-Pane :refresh-selected-messages) ()
  (loop for message in (send self :Visible-Selected-Messages)
	do (refresh-on-all-windows message)
  )
)

(defmethod (Message-Display-Pane :Resynch) (&optional (refresh-p t))
  ;;; Sometimes the top-item gets out of synch.  This should resynch it.
  (without-recursion ()
   (if (aref screen-image 0)
       (let ((top-number
	       (position (aref screen-image 0) item-list :test #'eq)
	     )
	    )
	    (if (and (numberp top-number) (numberp top-item))
	        (if (and (aref screen-image 0)
			 (not (equal top-number top-item))
		    )
		    (progn (send self :Reset-Top-Item top-number)
			   (send self :refresh-selected-messages)
		    )
		    (if refresh-p (send self :refresh-selected-messages) nil)
		)
		nil
	    )
       )
       nil
   )
  )
)

(defwhopper (Message-Display-Pane :Scroll-To) (&rest args)
; (ticl:with-metering ("lm:rice;meter-scroll.text" (not (eq sys:current-process tv:mouse-process))) ()
  (let ((filter (send self :Filter)))
       (if filter
	   (let ((*potential-messages-to-preempt*
		   (send filter :Computed-Order-Safe)
		 )
		)
		(declare (special *potential-messages-to-preempt*))
		(lexpr-continue-whopper args)
	   )
	   (tv:with-window-ops-on-bit-array (self)
	     (lexpr-continue-whopper args)
	   )
       )
  )
;  )
)

(defwhopper (Message-Display-Pane :redisplay-selected-items) (&rest args)
  (tv:with-window-ops-on-bit-array (self) (lexpr-continue-whopper args))
)

(defwhopper (Message-Display-Pane :set-edges) (&rest args)
  (tv:without-window-ops-on-bit-array (self) (lexpr-continue-whopper args))
)

(defmethod (message-display-pane :clear-item) (item)
"Clears an item on the screen."
  (multiple-value-bind (ignore ignore left top width height)
      (send self :item-details item)
    (if (and left top width height)
	(send self :bitblt tv:alu-seta width height tv:100%-white 0 0
	      (- left (tv:sheet-inside-left)) (- top (tv:sheet-inside-top))
	)
	nil
    )
  )
)


(defun size-from-spec (spec sheet)
  (typecase spec
    (cons (case (first spec)
	    (:Chars (* (second spec)
		       (let ((the-font (or (third spec) :Default)))
			    (tv:font-char-width
			      (tv:coerce-font the-font sheet)
			    )
		       )
		    )
	    )
	    (:pixels (second spec))
	    (otherwise (yw-error "~S is an illegal size spcifier.  ~
                        It should be of the form (:chars ...), ~
                        (:pixels ...), a number or null."
			spec
		       )
	    )
	  )
    )
    (number spec)
    (null nil)
    (otherwise (yw-error "~S is an illegal size spcifier.  ~
                        It should be of the form (:chars ...), ~
                        (:pixels ...), a number or null."
			spec
	       )
    )
  )
)


(defmethod (message-display-pane :inverse-video-item) (item type)
"Makes a specific type of inverse video change to the specified item."
  (declare (special *selected-message-display-stipple-alist*))
  (multiple-value-bind (ignore ignore left top width height)
      (send self :item-details item) ;;; These are in inside coords.
    (if (and left top width height)
        (let ((entry (or (find-if #'(lambda (X)
				      (equalp
					type (message-highlight-spec-name X)
				      )
				    )
				  *selected-message-display-stipple-alist*
			 )
			 (find-if #'(lambda (X)
				      (equalp :Otherwise
					      (message-highlight-spec-name X)
				      )
				    )
				  *selected-message-display-stipple-alist*
			 )
		     )
	      )
	     )
	     (let ((alu     (message-highlight-spec-alu     entry))
		   (stipple (message-highlight-spec-stipple entry))
		   (start   (message-highlight-spec-start   entry))
		   (end     (message-highlight-spec-end     entry))
		  )
		  (let ((real-start (or (Size-From-Spec start self) left)))
		       (let ((real-width
			       (- (or (and (Size-From-Spec end self)
					   (+ (Size-From-Spec end self)
					      (tv:sheet-inside-left)
					   )
				      )
				      (+ left width)
				  )
				  real-start
			       )
			     )
			    )
			    (send self :Bitblt alu real-width height
				  stipple 0 0
				  real-start top
;				  (- real-start (tv:sheet-inside-left))
;				  (- top (tv:sheet-inside-top))
			    )
		       )
		  )
	     )
	)
	nil
    )
  )
)

(defmethod (message-display-pane :highlight-message)
	   (message &optional (type t) (no-error-p nil))
  (if message
      (Add-Selected-Type message type)
      (if no-error-p
	  nil
	  (yw-error "~D is not a message." message)
      )
  )
)

(defmethod (message-display-pane :dehighlight-message)
	   (message &optional (type t) (no-error-p nil))
  (if message
      (Remove-Selected-Type message type)
      (if no-error-p
	  nil
	  (yw-error "~D is not a message." message)
      )
  )
)

(defmethod (message-display-pane :get-right-on-screen-item) (for-item)
  (let ((msg (message-from-display-item for-item)))
       (loop for i from 0 below (length screen-image)
	     when (let ((item (aref screen-image i)))
		       (and item
			    (eq msg (message-from-display-item-array
				      (aref item 0)
				    )
			    )
		       )
		  )
	     return (aref screen-image i)
       )
  )
)

(defmethod (Message-Display-Pane :top-message-on-screen) ()
  (let ((item (part-of-list (max top-item 0) 1 item-list)))
       (if (first item)
	   (message-from-display-item (first item))
	   nil
       )
  )
)

(defmethod (Message-Display-Pane :bottom-message-on-screen) ()
  (let ((item (part-of-list (max (+ (length screen-image) (- top-item 1)) 0) 1
			    item-list
	      )
	)
       )
       (if (or (first item) item-list)
	   (message-from-display-item
	     (or (first item) (first (last item-list)))
	   )
	   nil
       )
  )
)

(defmethod (message-display-pane :Reset-Top-Item) (to)
"We have decided that the top item visible on the screen is not reflected
by these top-item type slots so reset them to what reality tells us it should
be.
"
  (setq top-item to)
  (setq target-top-item to)
) 

(defmethod (message-display-pane :Screen-Lines) ()
  (or screen-image
      (setq screen-image
	    (make-array (ceiling (tv:sheet-inside-height tv:default-screen)
				 5
			)
	    )
      )
  )
  (floor (tv:sheet-inside-height self) tv:line-height)
)

(defmethod (message-display-pane :item-details) (item)
"Returns the relevant details about the item.  If we cannot find an item
to match then we return all nils.
"
  (declare (values mouse-info ignore left top width height))
  (if (or top-item target-top-item)
      (values (Mouse-Info item) nil (Left item) (Top item)
	      (- (Right item) (Left item))
	      (- (Bottom item) (Top item))
      )
      (values nil nil nil nil nil nil)
  )
)

(defmethod (message-display-pane :deexposed-mouse-buttons) (mask x y)
  "Call standard :MOUSE-BUTTONS method."
  (if *deexposed-mailer-mouse-behaviour-enabled*
      (send self :mouse-buttons mask x y)
      (send self :Expose)
  )
)

(defmethod (message-display-pane :deexposed-who-line-documentation-string) ()
  "Call normal :WHO-LINE-DOCUMENTATION-STRING."
  (if *deexposed-mailer-mouse-behaviour-enabled*
      (send self :who-line-documentation-string)
      nil
  )
)

(defparameter tv:*exposed-deexposed-blinker* nil)

;;; Copied from (DEFMETHOD (ESSENTIAL-SCROLL-MOUSE-MIXIN :MOUSE-MOVES) (X Y)
(defmethod (message-display-pane :deexposed-mouse-moves) (x y)
  "Call standard :MOUSE-BUTTONS method."
  (if yw:*deexposed-mailer-mouse-behaviour-enabled*
      (progn (tv:mouse-set-blinker-cursorpos)
	     (multiple-value-bind (item nil left top wid hei)
		 (send self :mouse-sensitive-item x y)
	       (and top (setq top (1- top)))
	       (if tv:exposed-p
		   nil
		   (let ((blinker
			   (rest (assoc :rectangle-blinker
					(send (send self :superior)
					      :mouse-blinkers
					)
				 )
			   )
			 )
			)
			(setq tv:*exposed-deexposed-blinker* blinker)
			(if blinker
			    (cond ((and item (eq self (tv:window-under-mouse))
					(progn (sleep 0.1)
					       (eq self (tv:window-under-mouse))
					)
				   )
				   (tv:blinker-set-cursorpos
				     blinker
				     (+ tv:x-offset
					(send self :left-margin-size)
					(- left (tv:sheet-inside-left)))
				     (+ tv:y-offset (send self :top-margin-size)
					(- top (tv:sheet-inside-top))))
				   (tv:blinker-set-size blinker wid hei)
				   (tv:blinker-set-visibility blinker t)
				   (setq current-item item))
				  (t
				   (tv:blinker-set-visibility blinker nil)
				   (setq current-item nil)
				  )
			    )
			)
		   )
	       )
	     )
      )
      nil
  )
)

(defun items-between (window new old)
  (let ((smaller (min (cache-msg# new) (cache-msg# old)))
	(larger  (max (cache-msg# new) (cache-msg# old)))
       )
       (loop with index = smaller
	     for message = (cache-entry-of index window)
	     for item = (and message
			     (find-if #'(lambda (item)
					  (equal window (window item))
					)
					(cache-all-items message)
			     )
			)
	     when item
	     collect item
	     until (equal index larger)
	     do (setq index (+ 1 index))
       )
  )
)

(defmethod (Message-Display-Pane :mouse-hold) (item buttons)
  (declare (special *top-level-mouse-blip-methods*))
  (let ((entry (find-if
		  #'(lambda (spec)
		      (and (or (not (top-level-blip-applicable-if spec))
			       (funcall (top-level-blip-applicable-if spec)
					(send self :Owner) self
			       )
			   )
			   (equal (top-level-blip-mouse-key spec)
				  (case buttons
				    (1 :mouse-l-hold)
				    (2 :mouse-m-hold)
				    (3 :mouse-r-hold)
				    (otherwise nil)
				  )
			   )
		      )
		    )
		    *top-level-mouse-blip-methods*
	       )
	)
	(*terminal-io* (send (send self :Owner) :Prompt-Window))
       )
       (catch-error
	 (if entry
	     (let ((lines (Items-Between
			    self (Object item) (object mouse-down-line)
			  )
		   )
		  )
	          (if (> (length lines) 1)
		      (loop for an-item in lines do
			    (let ((*message* (Object an-item)))
			         (declare (special *message*))
				 (send (send self :Owner)
				       (Top-Level-Blip-Starter-Method entry)
				       (Make-A-Sequence nil
					 :Owner (send self :Owner)
					 :Mailbox mailstreams
					 :Sequence-Specifier (list *message*)
				       )
				 )
			    )
		      )
		      nil
		  )
		  (Let ((to-remove
			  (set-difference All-Selected-Lines lines)
			)
		       )
		       (if (top-level-blip-undo-method entry)
			   (loop for an-item in to-remove do
				 (send (send self :Owner)
				       (top-level-blip-undo-method entry)
				       an-item
				 )
			   )
			   nil
		       )
		  )
	     )
	     nil
	 )
       )
  )
)

(defmethod (message-display-pane :after :mouse-moves) (x y)
  (send self :mouse-moves-internal x y)
)

(defmethod (message-display-pane :mouse-moves-internal) (x y)
  (let ((buttons (tv:mouse-buttons t)))
       (if (= buttons 0)
	   nil
	   (catch-error
	     (multiple-value-bind (item nil left top wid hei)
		 (send self :mouse-sensitive-item x y)
	       (ignore left top wid hei)
	       (if item
		   (let ((*mailer* (send self :Owner))
			 (*message* (Object item))
			)
		        (declare (special *message*))
			(if mouse-down-line
			    nil
			    (setq mouse-down-line item)
			)
			(pushnew item all-selected-lines)
			(send self :mouse-hold item buttons)
		   )
		   nil
	       )
	     )
	   )
       )
  )
)

(defun check-deexposed-blinkers ()
  (if tv:*Exposed-Deexposed-Blinker*
      (Let ((window (tv:window-under-mouse :deexposed-mouse-moves :active)))
	   (if (and window (send window :Operation-Handled-P
				 :Deexposed-Mouse-Moves
			   )
	       )
	       nil
	       (progn (tv:blinker-set-visibility
			tv:*Exposed-Deexposed-Blinker* nil
		      )
		      (setq tv:*Exposed-Deexposed-Blinker* nil)
	       )
	   )
      )
      nil
  )
)

(defmethod (message-display-pane :Select-Me) ()
  (send self :Maybe-Expose)
  (send owner :make-window-current self)
)

(defmethod (message-display-pane :Forget-Me) ()
  (send owner :Forget-Window self)
)

(defmethod (message-display-pane :mouse-click) (button x y &aux op)
  (declare (special *background-mouse-blip-methods*))
  (if mouse-down-line
      (process-run-function '(:Name "Wait for release")
			    #'(lambda (window)
				(process-wait "Wait for release"
				  #'(lambda () (= (tv:mouse-buttons t) 0))
				)
				(send window :Set-Mouse-Down-Line nil)
				(send window :Set-All-Selected-Lines nil)
			      )
			      self
      )
      nil
  )
  (if (loop for var
	    in *all-specials-that-cause-blips-from-background-mouse-clicks*
	    when (send self :In-Window-Process var)
	    return t
	    finally (return nil)
      )
      (send self :force-kbd-input
	    (list :Summary-Window
		  (list :Summary-Window Mailstreams self)
		  (send owner :Mailbox-Selector)
		  button
	    )
      )
      (multiple-value-bind (item type)
	  (send self :mouse-sensitive-item x y)
	(cond ((null item)
	       (let ((entry
		       (find-if
			 #'(lambda (X)
			     (and (char-equal
				    button
				    (background-blip-Mouse-Char X)
				  )
				  (or (not (background-blip-applicable-if x))
				      (funcall
					(background-blip-applicable-if x)
					(send self :Owner) self
				      )
				  )
			     )
			   )
			   *background-mouse-blip-methods*
		       )
		     )
		    )
		    (if entry
			(process-run-function
			  '(:Name "Message Display Background" :Priority 1)
			  #'(lambda (window)
			      (send window (background-blip-method entry))
			    )
			  self
			)
			nil
		    )
	       )
	      )
	      ((or (zerop (tv:char-mouse-clicks button))
		   (equal 1 (tv:char-mouse-clicks button))
	       )
	       ;;; This copied from (essential-scroll-mouse-mixin :mouse-click)
	       (cond
		 ((null item) nil)
		 ((progn (sleep 0.1) all-selected-lines) t)
		 ((or (null type)
		      (setq op
			  (first (cdr (assoc type item-type-alist :test #'eq)))
		      )
		  )
		  ;; psych out :BUTTONS --- Copy of code in
		  ;; (TV:BASIC-MENU :MOUSE-BUTTONS)
		  (cond
		    ((and (consp item)
			  (>= (length item) 3)
			  (eq (second item) :buttons))
		     (setq item (nth (char-mouse-button button) (third item)))))
		  (tv:blinker-set-visibility item-blinker nil)
		  (send self :Force-Kbd-Input
			(list (or type (first (mouse-info item)))
			      (mouse-info item) self button))
;		  (send self :execute (if op
;					  (list* nil op item)
;					  (mouse-info item)))
		  t)
		 (t (send self :force-kbd-input (list type item self button))
		    t)))
	      (t nil)
	)
      )
  )
)

;(defmethod (sheet :Inverse-Around :Handle-Mouse)
;	   (cont mt original-args &rest ignore)
;  (check-deexposed-blinkers)
;  (lexpr-funcall-with-mapping-table cont mt original-args)
;)

(defadvise tv:Mouse-Default-Handler (:watch-for-blinkers) ()
  (check-deexposed-blinkers)
  :Do-It
)

(defun get-any-mouse-sensitive-item (window)
  (multiple-value-bind (self-x self-y)
      (tv:sheet-calculate-offsets window nil)
    (send window :Mouse-Sensitive-Item
	  (- tv:mouse-x self-x) (- tv:mouse-y self-y)
    )
  )
)

(defmethod (Message-Display-Pane :iconify-me) ()
  (if (fboundp 'w:shrink-window) (funcall 'w:shrink-window self) nil)
  (send self :Bury)
)

(defmethod (Message-Display-Pane :read-me) ()
  (send owner :Read-Type-Command
	(or (send self :Filter) (simple-sequence :Sequence-All))
	:Read-Sequence
  )
)

(defmethod (Message-Display-Pane :right-button-menu) ()
  (let ((choice (w:menu-choose
		  *summary-window-background-right-button-menu-items*
		  :Label "Select and operation"
		  :Columns 1
		)
	)
       )
       (if choice
	   (send self choice)
	   (beep)
       )
  )
)

(defmethod (Message-Display-Pane :in-window-process) (symbol)
  (let ((sg (send (send owner :Process) :Stack-Group)))
       (without-interrupts (eh:symeval-in-stack-group symbol sg))
  )
)

(defmethod (message-display-pane :Who-line-documentation-string) ()
  (declare (special *top-level-mouse-blip-methods*
		    *background-mouse-blip-methods*
           )
  )
  (if (send self :In-Window-Process '*reading-a-mailbox*)
      "L, M, R: Select this mailbox/sequence R2: Bring up the system menu"
      (multiple-value-bind (item ignore) (Get-Any-Mouse-Sensitive-Item self)
	(if item
	    (if (send self :In-Window-Process '*reading-a-sequence*)
		"L, M, R: Select this message R2: Bring up the system menu"
		
		(nconc (loop for X in *top-level-mouse-blip-methods*
			     when (or (not (top-level-blip-applicable-if x))
				      (funcall (top-level-blip-applicable-if x)
					       (send self :Owner) self
				      )
				  )
			     nconc (List (Top-Level-Blip-Mouse-Key X)
					 (Top-Level-Blip-Doc-String X)
				   )
		       )
		      '(:Mouse-R-2 "Bring up the system menu"
			:Sort ""
			:Smart-Newlines ""
		       )
		)
	    )
	    (nconc (loop for X in *background-mouse-blip-methods*
			 when (or (not (background-blip-applicable-if x))
				  (funcall (background-blip-applicable-if x)
					   (send self :Owner) self
				  )
			      )
			 nconc (List (background-Blip-Mouse-Key X)
				     (background-Blip-Doc-String X)
			       )
		   )
		  '(:Mouse-R-2 "Bring up the system menu"
		    :Sort ""
		    :Smart-Newlines ""
		   )
	    )
	)
      )
  )
)

(defmethod (Message-Display-Pane :Reposition) (resize-p)
  (if resize-p
      (send self :Set-Edges 0 0 (message-window-width) (message-window-height))
      nil
  )
  (multiple-value-bind (x y) (ideal-message-display-pane-position self)
    (send self :Set-Position
	  (if (> (+ x (send self :Width)) (send (send self :Superior) :Width))
	      (- (send (send self :Superior) :Width) (send self :Width) 1)
	      x
	  )
	  (if (> (+ y (send self :Height)) (send (send self :Superior) :Height))
	      (- (send (send self :Superior) :Height) (send self :Height) 1)
	      y
	  )
    )
  )
)

(defun message-from-display-item-array (item)
  (third (tv:scroll-entry-mouse-info item))
)

(defun message-from-display-item (item)
;  (etypecase item
;    (yw-item (object item))
;    (cache item)
;  )
  ;;; This function was taking a bunch of time, so we rely on the Object
  ;;; method to barf if we can't handle this.
  (typecase item
    (cache item)
    (otherwise (object item))
  )
)

(defun part-of-list (from delta list)
; The following won't work because it fills things up with nils if it
; overflows.
;  (firstn delta (nthcdr from list))
  (let ((rest (nthcdr from list)))
       (loop for i from 1 to delta
	     for tail on rest
	     while tail collect (first tail)
       )
  )
)

(defmethod (Message-Display-Pane :prefetch-envelopes-on-screen) (scroll-pos)
  (let ((all-in-window
	  (mapcar 'Message-From-Display-Item
	   (part-of-list (max (- scroll-pos 1) 0)
			 (+ 1 (send self :screen-lines))
			 item-list
	   )
	  )
	)
       )
       (let ((filtered
	       (remove-if
		 #'(lambda (x)
		     (or (not x) (not (equal :Unbound (Cache-Envelope x))))
		   )
		   all-in-window
	       )
	     )
	    )
	    (if filtered
		(get-missing-envelopes filtered)
		nil
	    )
       )
  )
)

;(defmethod (Message-Display-Pane :Before :Scroll-To)
;	   (to &optional (type :Absolute))
;  (let ((target
;	  (case type
;	    (:absolute to)
;	    (:relative (+ to tv:top-item))
;	    (otherwise (yw-error "~A is an unknown type of scrolling" TYPE))
;	  )
;	)
;       )
;;       (send self :Prefetch-Envelopes-On-Screen target)
;;      (send self :Expose)  ;;; I don't think that we should do this.
;  )
;)

(defmethod (message-display-pane :maybe-redisplay-selected-item)
	   (message &optional (force-p nil))
  (declare (special *items-to-refresh-alist*))
  (ignore force-p)
  (if mailstreams
      (let ((the-item (send self :Find-Display-Item-For-Message message)))
	   (if the-item
	       (tv:sheet-force-access
		 (self :no-prepare)
		 (setq target-top-item top-item)
		 (catch 'tv:end-of-page
		   (dotimes (i (send self :screen-lines))
		     (cond
		       ((and (equal (aref screen-image i) the-item))
			(if (boundp '*items-to-refresh-alist*)
			    (let ((entry
				    (or (assoc self *items-to-refresh-alist*
					       :Test #'eq
					)
					(let ((new-entry (list self nil)))
					     (push new-entry
						   *items-to-refresh-alist*
					     )
					     new-entry
					)
				    )
				  )
				 )
				 (push the-item (second entry))
			    )
			    (let ((*force-redisplay* t))
				 (declare (special *force-redisplay*))
				 (send self :Redisplay-Selected-Items
				       (list the-item)
				 )
			    )
			)
		       )
		     )
		   )
		 )
	       )
	   )
      )
      nil
  )
)

(defmethod (Message-Display-Pane :Find-Display-Item-For-Message) (message)
  (let ((message-number (number-of message)))
       (if (and (numberp message-number)
		(numberp (send (cache-mailstream message) :Messagecnt))
		(<= message-number
		    (send (cache-mailstream message) :Messagecnt)
		)
	   )
	   (let ((items (cache-all-items message)))
		(let ((item (find-if #'(lambda (x)
					 (equal self (window x))
				       )
				       items
			    )
		      )
		     )
		     (if item
			 item
			 nil
		     )
		)
	   )
	   (values nil :Message-Number-Out-Of-Range)
       )
  )
)

(defun should-force-p (message mailstream force-p)
  (ignore mailstream)
  (or (not (is-present (cache-flags message)))
      (if (equal force-p :Flags-Changed)
	  (let ((f1 (cache-Flags-Used-For-Display message))
		(f2 (cache-flags message))
	       )
	       (not (and (equal (length f1) (length f2))
			 (equal nil (set-difference f1 f2))
			 (equal nil (set-difference f2 f1))
		    )
	       )
	  )
	  force-p
      )
  )
)

(defun make-sure-items-are-updated (mailstream)
"Loops through mailstream and makes sure that all of the window items are
showing the right thing.
"
  (let ((messages (send mailstream :Messagearray)))
       (if (arrayp messages)
	   (loop for cache being the array-elements of messages
		 when cache
		 do (let ((flags (cache-flags cache)))
			 (if (and (cache-header-display-string cache)
				  (not (equal (cache-flags-used-for-display
						cache 
					      )
					      flags
				       )
				  )
			     )
			     (flush-display-cache-for cache t)
			     nil
			 )
		    )
	   )
	   :Message-Array-No-There-Yet
       )
  )
)

(defun all-open-imap-streams-safe ()
  "Returns the really open subset of *all-open-imap-streams*."
  (loop for stream in *all-open-imap-streams*
	when (and (send stream :Mailbox)
		  (send stream :Messagearray)
	     )
	collect stream
  )
)

(defun ensure-all-windows-are-updated ()
"Makes sure that all mailstreams are updated appropriately."
  (mapcar 'Make-Sure-Items-Are-Updated (All-Open-Imap-Streams-Safe))
)

(defmethod (Message-Display-Pane :Flush-Display-For)
	   (message &optional (force-p t))
  (if (and message
	   (or force-p (not (cache-header-display-string message))
	       (search (the string *waiting-message-text*)
		       (the string (cache-header-display-string message))
		       :Test #'char-equal
		       :End2 (length *waiting-message-text*)
	       )
	   )
      )
      (progn (reset-display-items-for message)
	     (Send self :maybe-redisplay-selected-item message force-p)
      )
      ;;; The window might not have this message, since it might have been
      ;;; filtered out.
      (send self :maybe-redisplay-selected-item message force-p)
  )
)

(defmethod (Message-Display-Pane :Fully-Flush-Display-Cache)
	   (&optional (force-p t))
  (if (send self :Filter)
      (loop for message in (send (send self :Filter) :Numberise-Messages)
	    do (send self :Flush-Display-For message force-p)
      )
      (loop for stream in mailstreams
	    do (loop for number from 1 to (send stream :messagecnt)
		     do (send self :Flush-Display-For
			      (cache-entry-of number stream) force-p
			)
	       )
      )
  )
)

(defun refresh-all-message-displays ()
  (Apply-To-Summary-Windows
    #'(lambda (window)
	(let ((streams (send window :Mailstreams)))
	     (if (and streams
		      (loop for stream in streams always (send stream :Open-P))
		 )
		 (progn (send window :Fully-Flush-Display-Cache)
			(send window :Redisplay)
		 )
		 nil
	     )
	)
      )
  )
)

(defmethod (message-display-pane :insert-items) (from items)
  "Inserts items before the specified position."
  (let ((tail (nthcdr from item-list)))
       (if tail
	   (setf (rest tail) (append items (rest tail)))
	   (if item-list
	       (setq item-list (append item-list items))
	       (send self :set-item-list items)
	   )
       )
  )
)

(defmethod (Message-Display-Pane :Before :Kill) (&rest ignore)
  (if icon
      (progn (send icon :Kill)
	     (setq icon nil)
      )
      nil
  )
  (if owner
      (send owner :Forget-Window self)
      nil
  )
)

(defmethod (message-display-pane :Goto-Beginning) ()
  (send self :Scroll-To 0)
)

(defmethod (message-display-pane :Goto-End) ()
  (send self :Scroll-To (send self :number-of-items))
)


(defmethod (message-display-pane :go-up-by) (lines)
 (let ((top (if (<= top-item 0) 0 (max 0 (- top-item lines)))))
      (send self :Scroll-To top)
 )
)

(defmethod (message-display-pane :page-up) ()
 (send self :Go-Up-By (send self :screen-lines))
)

(defmethod (message-display-pane :line-up) ()
 (send self :Go-Up-By 1)
)

(defmethod (message-display-pane :go-down-by) (lines)
  (let ((len (send self :number-of-items)))
       (let ((bottom (if (>= top-item len)
			 len
			 (min len (+ top-item lines 1))
		     )
	     )
	    )
            (send self :Scroll-To (max 0 (- Bottom 1)))
       )
 )
)

(defmethod (message-display-pane :page-down) ()
  (send self :go-down-by (send self :screen-lines))
)

(defmethod (message-display-pane :line-down) ()
  (send self :go-down-by 1)
)

(defun items-before (items message sequence)
  (loop for item in items
	until (Message>= (Message-From-Display-Item item) message sequence)
	collect item
  )
)

(defun items-not-before (items message new-items sequence)
  (loop for item in items
	when (message>= (Message-From-Display-Item item)
			(message+ message (length new-items) sequence)
			sequence
	     )
	collect item
  )
)

(defun window-less-p (a b)
  (if *prefer-windows-with-visible-selected-messages-when-exposing-p*
      (let ((selected-in-a (send (first a) :Visible-Selected-Messages))
	    (selected-in-b (send (first b) :Visible-Selected-Messages))
	   )
	   (or (and selected-in-a (not selected-in-b))
	       (and (or (and selected-in-a selected-in-b)
			(and (not selected-in-a) (not selected-in-b))
		    )
		    (or (if (typep (first a) 'tv:top-label-mixin)
			    (> (third a) (third b))
			    (< (third a) (third b))
			)
			(and (= (third a) (third b))
			     (< (second a) (second b))
			)
		    )
	       )
	   )
      )
      (or (if (typep (first a) 'tv:top-label-mixin)
	      (> (third a) (third b))
	      (< (third a) (third b))
	  )
	  (and (= (third a) (third b))
	       (< (second a) (second b))
	  )
      )
  )
)

(defun Expose-In-Order
       (resource &optional (predicate #'(lambda (&rest ignore) t)) (force-p t))
  (let ((windows (Map-yw-Resource-Return
		    #'(lambda (win &rest ignore)
			(multiple-value-bind (left top) (send win :position)
			  (list win left top)
			)
		      )
		      resource
		      predicate
		  )
       )
      )
      (let ((sorted (sort (copy-list windows) 'window-less-p)))
	   (loop for win in (reverse sorted) do
		 (send (first win) (if force-p :Expose :Maybe-Expose))
	   )
      )
  )
)

;-------------------------------------------------------------------------------

;;; Message filtering.

(defflavor filtration-mixin
	   (filter)
	   ()
  (:Default-Init-Plist
  )
  :Initable-Instance-Variables
  :Settable-Instance-Variables
  :Gettable-Instance-Variables
)

(defmethod (Filtration-Mixin :after :clean-up) ()
  (setq filter nil)
)

(defmethod (Filtration-Mixin :Accept-Message-P) (message)
  (typecase filter
    (Message-Sequence (send filter :Accept-Message-P message))
    (null t)
    (otherwise
     (yw-warn "Internal error.  The filter ~S should be a message sequence."
	      filter
     )
    )
  )
)

(defmethod (Filtration-Mixin :Accept-Message-P-Given-Sequence)
	   (message sequence)
  (typecase filter
    (message-sequence
     (send filter :Accept-Message-P-Given-Sequence message sequence)
    )
    (null t)
    (otherwise
     (yw-warn "Internal error.  The filter ~S should be a message sequence."
	      filter
     )
    )
  )
)

;-------------------------------------------------------------------------------

(defflavor summary-window
	   ((item-list-lock (make-an-imap-lock)))
	   (filtration-mixin message-display-pane)
  (:Default-Init-Plist
  )
  :Initable-Instance-Variables
  :Settable-Instance-Variables
  :Gettable-Instance-Variables
)

(defun tv:expose-inferior-window-safely (window-to-expose owner owner-type)
  (multiple-value-bind (left top right bottom) (send window-to-expose :Edges)
    (if (typep owner owner-type)
	(multiple-value-bind (oleft otop oright obottom) (send owner :Edges)
	  (if (and (or (and (>  right oleft) (<= right oright))
		       (and (>= left  oleft) (< left  oright))
		       (and (<  left  oleft) (>  right oright))
		   )
		   (or (and (>  bottom otop) (<= bottom obottom))
		       (and (>= top    otop) (< top    obottom))
		       (and (<  top    otop) (>  bottom obottom))
		   )
	      )
	      nil
	      (send window-to-expose :Expose)
	  )
	)
	nil
    )
  )
)

(defmethod (Summary-Window :Maybe-Expose) ()
  (tv:expose-inferior-window-safely self owner 'mail-control-window)
)

(defmethod (summary-window :Print-Self) (stream &rest ignore)
  (format stream "#<Summary Window ~A>"
	  (catch-error (send self :make-label) nil)
  )
)

(defun trim-off (source strings)
  (if strings
      (let ((index (search (the string (first strings)) (the string source)
			   :Test #'char-equal
		   )
	    )
	   )
	   (trim-off (if index
			 (subseq source (+ index (length (first strings))))
			 source
		     )
		     (rest strings)
	   )
      )
      (let ((capitalised (string-trim *whitespace-chars*
				      (string-capitalize source :Spaces t)
			 )
	    )
	   )
	   (if (equal "" capitalised)
	       source
	       capitalised
	   )
      )
  )
)

(defun colonify-1 (sequence-start current-number list)
  (declare (optimize (speed 3) (safety 0)))
  (if list
      (if (equal (+ 1 current-number) (first list))
	  (colonify-1 sequence-start (+ 1 current-number) (rest list))
	  (cons (format nil "~D:~D"  sequence-start current-number)
		(Colonify list)
	  )
      )
      (list (format nil "~D:~D" sequence-start current-number))
  )
)
       
(defun colonify (numbers)
  (declare (optimize (speed 3) (safety 0)))
  (if numbers
      (if (rest numbers)
	  (let ((one (Number-Of (first numbers)))
		(two (Number-Of (second numbers)))
	       )
	       (if (equal (+ 1 one) two)
		   (colonify-1 one two (rest (rest numbers)))
		   (cons one (colonify (rest numbers)))
	       )
	  )
	  numbers
      )
      nil
  )
)

(defun colonify-numbers (list &optional (space-p t))
  (let ((non-message
	  (find-if-not #'(lambda (x) (or (numberp x) (typep x 'cache))) list)
	)
       )
       (if non-message
	   (let ((strs (mapcar #'make-label-from-filter list)))
	        (if space-p
		    (format nil "~A~{, ~A~}" (first strs) (rest strs))
		    (format nil "~A~{,~A~}"  (first strs) (rest strs))
		)
	   )
	   (let ((numbers (mapcar #'Number-Of list)))
	        (let ((result (colonify (sort numbers #'<))))
		     (if space-p
			 (format nil "~A~{, ~A~}" (first result) (rest result))
			 (format nil "~A~{,~A~}" (first result) (rest result))
		     )
		)
	   )
       )
  )
)

(defun mouse-sensitively-colonify-1
       (sequence-start current-number list mailstream)
  (declare (optimize (speed 3) (safety 0)))
  (if list
      (if (equal (+ 1 current-number) (first list))
	  (mouse-sensitively-colonify-1
	    sequence-start (+ 1 current-number) (rest list) mailstream
	  )
	  (cons (list (loop for i from sequence-start to current-number
			    collect (Cache-Entry-Of i mailstream)
		      )
		      nil
		      (format nil "~D:~D"  sequence-start current-number)
		)
		(mouse-sensitively-Colonify list mailstream)
	  )
      )
      (list (list (loop for i from sequence-start to current-number
			collect (Cache-Entry-Of i mailstream)
		  )
		  nil
		  (format nil "~D:~D" sequence-start current-number)
	    )
      )
  )
)

(defun mouse-sensitively-colonify (numbers mailstream)
  (declare (optimize (speed 3) (safety 0)))
  (if numbers
      (if (rest numbers)
	  (let ((one (Number-Of (first numbers)))
		(two (Number-Of (second numbers)))
	       )
	       (if (equal (+ 1 one) two)
		   (mouse-sensitively-colonify-1 one two (rest (rest numbers))
						  mailstream
                   )
		   (cons (list (Cache-Entry-Of one mailstream) nil one)
			 (mouse-sensitively-colonify
			   (rest numbers) mailstream
			 )
		   )
	       )
	  )
	  (list (list (Cache-Entry-Of (first numbers) mailstream)
		      nil (first numbers)
		)
	  )
      )
      nil
  )
)

(defun Mouse-Sensitively-Colonify-Numbers
       (mailstream output-stream list &optional (space-p t))
  (let ((non-message
	  (find-if-not #'(lambda (x) (or (numberp x) (typep x 'cache))) list)
	)
       )
       (if non-message
	   (let ((strs (mapcar 'make-label-from-filter list)))
	        (if space-p
		    (format output-stream "~A~{, ~A~}" (first strs) (rest strs))
		    (format output-stream "~A~{,~A~}"  (first strs) (rest strs))
		)
	   )
	   (let ((numbers (mapcar #'Number-Of list)))
	        (let ((result (Mouse-Sensitively-Colonify
				(sort numbers #'<) mailstream
			      )
		      )
		     )
		     (if space-p
			 (format output-stream "~~{, ~~}"
				 (first result) (rest result)
			 )
			 (format output-stream "~~{,~~}"
				 (first result) (rest result)
			 )
		     )
		)
	   )
       )
  )
)

(defun (:Property :Through :make-label-from-filter-function) (arg1 arg2)
  (format nil "~D:~D" arg1 arg2)
)

(defun (:Property :Sequence-Field :make-label-from-filter-function) (arg1 arg2)
  (format nil "Field ~A ~A" arg1 (Make-Label-From-Filter arg2))
)

(defun (:Property :Sequence-Length :make-label-from-filter-function) (arg1 arg2)
  (format nil "Length ~A ~D" arg1 arg2)
)

(defun make-label-from-filter (x)
  (clean-up-string
    (typecase x
      (keyword (Trim-Off (symbol-name X) '("SEQUENCE-" "GET-")))
      (cons (if (keywordp (first x))
		(if (get (first x) :Make-Label-From-Filter-Function)
		    (apply (get (first x) :Make-Label-From-Filter-Function)
			   (rest x)
		    )
		    (if (equal (length x) 3)
			(format nil "~A ~A ~A"
				(Make-Label-From-Filter (second x))
				(Make-Label-From-Filter (first x))
				(Make-Label-From-Filter (third x))
			)
			(format nil "~A~{ ~A~}"
				(Make-Label-From-Filter (first x))
				(mapcar #'Make-Label-From-Filter (rest x))
			)
		    )
		)
		(Colonify-numbers x)
	    )
      )
      (message-sequence
       (let ((spec (send x :Sequence-Specifier)))
	    (if (or (equal (length spec) 1)
		    (assoc (first spec) *simple-term-specifiers* :Test #'eq)
		)
		(make-label-from-filter (send x :Sequence-Specifier))
		(string-append
		  "[" (make-label-from-filter (send x :Sequence-Specifier)) "]"
		)
	    )
       )
      )
      (cache (format nil "~D" (cache-msg# x)))
      (string (quotify-string-with-whitespace-chars x))
      (otherwise (format nil "~A" x)) ;;; Took the space out.  Was that right?
    )
  )
)

(defun quotify-string-with-whitespace-chars (string)
  (if (string-search-set *whitespace-chars* string)
      (format nil "\"~A\"" string)
      string
  )
)

(defun clean-up-string (X)
  (string-trim " 	" x)
)

(defun merge-strings (strings)
  (if strings
      (if (rest strings)
	  (string-append (Clean-Up-String (first strings)) " "
			 (merge-strings (rest strings))
	  )
	  (Clean-Up-String (first strings))
      )
      ""
  )
)

(defun print-short-mailbox-name (name-or-stream &optional (readable-p nil))
  (cond ((not name-or-stream) "???")
	 ((consp name-or-stream)
	  (loop with outstring = ""
		for box in name-or-stream
		for rest on name-or-stream
		for result = (print-short-mailbox-name box readable-p)
		do (setq outstring (string-append outstring result))
		when (rest rest)
		do (setq outstring (string-append outstring ", "))
		finally (return outstring)
	  )
	 )
	 (t (let ((name (if (stringp name-or-stream)
			    name-or-stream
			    (send name-or-stream :Mailbox)
			)
		  )
		  (stream (if (stringp name-or-stream)
			      nil
			      name-or-stream
			  )
		  )
		 )
		 (let ((index (or (search ":" (the string name)
					  :Test #'char-equal
				  )
				  nil
			      )
		       )
		       (defname
			 (check-user-mailbox (Default-Mailbox-Directory))
		       )
		      )
		      (let ((host (if index (subseq name 0 index) ""))
			    (box  (if index (subseq name (+ 1 index)) name))
			    (default (subseq defname
				       (+ 1 (search ":" (the string defname)
						    :Test #'char-equal
					    )
				       )
				     )
			    )
			   )
			   (let ((host-object (net:parse-host host t)))
				(if host-object
				    (Print-Short-Mailbox-Name-Internal
				      host host-object default box name stream
				      readable-p
				    )
				    name
				)
			   )
		      )
		 )
	    )
	 )
  )
)

(defun Print-Short-Mailbox-Name-Internal
       (host host-object default box name stream readable-p)
  (let ((abrev
	  (if (or (equal host "") (not host-object)
		  (equal host-object (net:parse-host *user-host*))
	      )
	      :default
	      (send (net:parse-host host) :Send-If-Handles :Short-Name)
	  )
	)
	(short-box
	  (canonicalize-mailbox-name
	    (let ((index (search (the string default) (the string box)
				 :Test #'char-equal
			 )
		  )
		 )
		 (if index (subseq box (length default)) box)
	    )
	  )
	)
       )
       (let ((bboard-p (or (typep stream 'ip-nntp-stream)
			   (and host-object
				(equalp
				  *bboard-source-directory*
				  (pathname-directory
				    (fs:default-pathname (send stream :Mailbox))
				  )
				)
			   )
		       )
	     )
	    )
	    (if bboard-p
		(if (typep stream 'ip-nntp-stream)
		    (format nil "NetNews ~A" name)
		    (if readable-p
			(send (fs:default-pathname name) :String-For-Printing)
			(format nil "BBoard ~A"
			  (string-capitalize
			    (send (fs:default-pathname name) :Name) :Spaces t
			  )
			)
		    )
		)
		(Maybe-Add-Read-Only-Text
		  readable-p abrev short-box host stream
		)
	    )
       )
  )
)

(defun maybe-add-read-only-text (readable-p abrev short-box host stream)
  (if readable-p
      (if (equal :default abrev)
	  (Format nil "~A" short-box)
	  (Format nil "~A:~A" (if (and abrev (not (equal "" abrev))) abrev host)
		  short-box
	  )
      )
      (let ((r/o-text (if (and stream (send stream :Read-Only-P))
			  (tv:fontify-string " (read only)" 1)
			  ""
		      )
	    )
	   )
	   (if (equal :default abrev)
	       (Format nil "~A~A" short-box r/o-text)
	       (Format nil "~A:~A~A" short-box
		       (if (and abrev (not (equal "" abrev))) abrev host)
		       r/o-text
	       )
	   )
      )
  )
)

(defmethod (Summary-Window :make-label) (&optional (the-filter filter))
  (let ((*print-case* :Capitalize))
       (if the-filter
	   (apply #'string-append
		  (send (simple-sequence the-filter) :Make-Label)
		  "; "
		  (print-short-mailbox-name mailstreams)
		  (let ((number (send self :number-of-items)))
		       (if (and *show-number-of-messages-in-summary-labels-p*
				(> number 0)
			   )
			   (list (format nil "; ~D message~P" number number))
			  '("")
		       )
		  )
	   )
	   (string-append
	     (print-short-mailbox-name mailstreams)
	     (let ((number (send self :number-of-items)))
		  (if (and *show-number-of-messages-in-summary-labels-p*
			   (> number 0)
		      )
		      (format nil "; ~D message~P" number number)
		      ""
		  )
	     )
	   )
       )
  )
)

(defun number-of (message)
  (etypecase message
    (fixnum message)
    (cache (cache-msg# message))
  )
)

(defun Cache-Entry-Of (message source &optional (error-p t))
  (declare (optimize (speed 3) (safety 0)))
  (typecase message
    (cache message)
    (cons (Cache-Entry-Of (first message) source))
    (otherwise
      (typecase source
	(imap-stream-mixin
	 (etypecase message
	   (fixnum (map-elt (send source :messagearray) message source error-p))
	   (cache message)
	 )
	)
	(cache (cache-entry-of message (cache-mailstream source)))
	(multi-sequence
	 (if (typep message 'cache)
	     message
	     (ferror nil "Cannot coerce number and multi-sequence to cache.")
	 )
	)
	(message-sequence (cache-entry-of message (send source :Mailstream)))
	(Message-Display-Pane
	 (let ((mailstreams (send source :Mailstreams)))
	      (if (and (equal (length mailstreams) 1)
		       (loop for stream in mailstreams
			     always (send stream :Open-P)
		       )
		  )
		  (cache-entry-of message (first mailstreams))
		  (if error-p
		      (ferror nil "Cannot find cache entry.")
		      nil
		  )
	      )
	 )
	)
	(otherwise
	 (etypecase message
	   (fixnum (map-elt (send source :messagearray) message source error-p))
	   (cache message)
	 )
	)
      )
    )
  )
)

(defmethod (setf Cache-Entry-Of)
	   (new-entry (message fixnum) (source imap-stream-mixin))
  (assert (or (null new-entry) (typep new-entry 'cache)))
  (setf (aref (send source :messagearray) (- message 1)) new-entry)
)

(clos:defclass yw-item
	  ()
  ((object :initarg :object :accessor object)
   (window :initarg :window :accessor window)
   (type :initarg :type :accessor type :initform nil)
   (mouse-info :initarg :mouse-info :accessor mouse-info)
   (left :initarg :left :accessor left :initform nil)
   (top :initarg :top :accessor top :initform nil)
   (right :initarg :right :accessor right :initform nil)
   (bottom :initarg :bottom :accessor bottom :initform nil)
  )
)

(defmethod clos:print-object ((me Yw-Item) stream)
  (format stream "#<Item") 
  (if (clos:slot-boundp me 'Object)
      (catch-error (format stream " ~S" (Object me)) nil)
      (format stream " Unbound!!")
  )
  (format stream ">")
)

(defmethod (Summary-Window :make-item) (cache-entry sequence)
"Gets a window item for message number I.  Either finds or creates a new item."
  (declare (values item new-item-created-p))
  (ignore sequence)
  (let ((existing-items (cache-all-items cache-entry)))
       (or (loop for item in existing-items
		 when (eq (Window item) self)
		 return item
	   )
	   (let ((item (make-instance
			 'yw-item :object cache-entry
			 :window self
			 :mouse-info
			 (list :Message
			       (cache-mailstream cache-entry)
			       cache-entry
			 )
		       )
		 )
		)
		(Add-Item cache-entry item self)
		(values item t)
	   )
       )
  )
)

(defmethod (Summary-Window :Numberise-Messages-In-Order) ()
  (if filter
      (send filter :Numberise-Messages-In-Order)
      (yw-warn "Internal error.  There is no filter in window ~S" self)
  )
)

(defmethod (Summary-Window :scroll-to-show)
	   (message &optional (recentering-fraction 0.3))
  (let ((item (send self :Find-Display-Item-For-Message message)))
       (if item
	   (let ((number (position item item-list)))
		(if (and (>= (- number 1) top-item)
			 (< (- number 1) (+ top-item (send self :screen-lines)))
		    ) ; Then it is on screen
		    nil
		    (let ((top-line
			    (max 0 (- (- number 1)
				      (round
					(* recentering-fraction
					   (send self :screen-lines)
					)
				      )
				   )
			    )
			  )
			 )
			 (send self :Scroll-To top-line)
		    )
		)
		:message-found-in-this-window
	   )
	   nil
       )
  )
)

;;;Edited by Tom Gruber            10 Feb 92  11:16
(defmethod (summary-window :scroll-to-good-place)
	   (&optional (new-messages-at-end 0))
  (let ((str (first (last mailstreams))))
       (if (and str (send str :Open-P))
	   (let ((totalmsgs  (send str :messagecnt))
		 (recentmsgs (send str :recentcnt))
		 (the-length (send self :number-of-items))
		)
		(let ((scroll-pos
			(if (if (send str :bboard-p)
				(not
				  *scroll-to-most-recent-messages-for-bboards-p*
				)
				(or (and (not filter)
				     (or
				       (not *scroll-to-most-recent-messages-p*)
				       (<= recentmsgs (send self :screen-lines))
				     )
				    )
				    (and filter
					 (or (equal 0 new-messages-at-end)
                                             ;(null new-messages-at-end)
					     (< (+ new-messages-at-end
						   (send self :Bottom-Item)
						)
						(- (send self :Number-Of-Items)
						   1
						)
					     )
					 )
				    )
				)
			    )
			    (if filter
				;;; This should leave filter windows in the same
				;;; place, not move to the bottomn.
				(min the-length
				     (or top-item target-top-item
					 most-positive-fixnum
				     )
				     (- the-length (send self :screen-lines))
				)
				(- the-length (send self :screen-lines))
			    )
			    ;;; Just show new headers.
			    (if (or (> recentmsgs 0)
				    (> new-messages-at-end 0)
				)
				(max
				  (- totalmsgs
				     (min (send self :screen-lines)
					  (max new-messages-at-end recentmsgs)
				     )
				  )
				  (- totalmsgs
				     (max (send self :screen-lines)
					  (max new-messages-at-end recentmsgs)
				     )
				  )
				)
				(if (send str :bboard-p)
				    (send self :Number-Of-Items)
				    (- (send self :Number-Of-Items)
				       (send self :screen-lines)
				    )
				)
			    )
			)
		      )
		     )
		     (let ((real-scroll-pos (max 0 scroll-pos)))
			  (send self :Prefetch-Envelopes-On-Screen
				real-scroll-pos
			  )
			  (send self :Scroll-To real-scroll-pos)
			  real-scroll-pos
		     )
		)
	   )
	   nil
       )
  )
)

(defmethod (Summary-Window :After :change-of-size-of-margins)
	   (&rest ignore)
  (send self :Scroll-To-Good-Place)
)

(defmethod (Summary-Window :After :set-edges) (&rest ignore)
  (send self :Scroll-To-Good-Place)
)

(defwhopper (Summary-Window :clean-up) ()
  (with-item-list-lock (continue-whopper))
)

(defwhopper (Summary-Window :insert-items) (from items)
  (with-item-list-lock (continue-whopper from items))
)

;;;Edited by Tom Gruber            10 Feb 92  11:16
(defwhopper (summary-window :make-sequence-and-item-list-congruent) (&rest args)
  (with-item-list-lock nil (lexpr-continue-whopper args))
)


(defmethod (summary-window :forget-items) (&optional (recursion nil))
  (with-item-list-lock recursion
    (loop for item in item-list
	  do (Remove-Item (message-from-display-item item) item nil)
    )
  )
)

(defmethod (summary-window :make-sequence-and-item-list-congruent)
	   (&optional (sequence (or filter
				    (make-a-sequence
				      nil :Owner owner
				      :Mailbox mailstreams
				      :Sequence-Specifier '(:Sequence-All)
				    )
				)
		      )
	   )
  (loop with changed-p = nil
	with new-messages = 0
	with accepted-messages = (send sequence :Computed-Order-Safe)
	with top-location = (cons nil item-list)
	with location = top-location
	with new-messages-at-end = 0
	for item = (second location)
	for message = (and item (Message-From-Display-Item item))
	do ;(format t "~%-----~S~%     ~S" (first accepted-messages) message)
	   (cond ;;; We have run out to legal messages, so throw away anything
		 ;;; else.
		 ((not accepted-messages)
;		  (loop for this-item in (rest location)
;			for mess = (Message-From-Display-Item this-item)
;			do (remove-item mess this-item)
;		  )
		  (remove-items-at-end (rest location) t)
		  (setf changed-p t)
		  (setf (rest location) nil)
		 )
	         ((eq message (first accepted-messages))
		  ;;; Message matches.
		  (setq accepted-messages (rest accepted-messages))
		  (setq location (rest location))
		 )
		 ((not message)
		  ;;; There are no original messages left, but there are
		  ;;; more to be snipped in here.
		  (loop for mess in accepted-messages
			for count from 1
			do (setf accepted-messages (rest accepted-messages))
			   (incf new-messages)
			   (if (not message) (incf new-messages-at-end) nil)
			collect (send self :Make-Item mess sequence)
			        into new-items
			finally (let ((new-location (last new-items)))
				     (setf (rest location)
					   (append new-items (rest location))
				     )
				     (setf item-list (rest top-location))
				     (setf changed-p t)
				     (setf location new-location)
				)
		  )
		 )
		 ((member message accepted-messages :Test #'eq)
		  ;;; The next item represents a real message, but there are
		  ;;; more to be snipped in here.
		  (loop for mess in accepted-messages
			for count from 1
			until (eq mess message)
			do (setf accepted-messages (rest accepted-messages))
			   (incf new-messages)
			   (if (not message) (incf new-messages-at-end) nil)
			collect (send self :Make-Item mess sequence)
			        into new-items
			finally (let ((new-location (last new-items)))
				     (setf (rest location)
					   (append new-items (rest location))
				     )
				     (setf item-list (rest top-location))
				     (setf changed-p t)
				     (setf location new-location)
				     (send self :Push-Down-By-N
					   new-items message
				     )
				)
		  )
		 )
		 (t ;;; The item must be a now rejected message, so snip it out.
		    (remove-item message item)
		    (setf changed-p t)
		    (if (eq item (second location))
			(progn (setf (rest location) (rest (rest location)))
			       (setf item-list (rest top-location))
			)
			nil
		    )
		 )
	   )
	until (and (not accepted-messages) (not (rest location)))
	finally (setf item-list (rest top-location))
	        (return changed-p new-messages new-messages-at-end)
  )
)

(defmethod (summary-window :set-up)
	   (the-owner the-filter the-mailbox-name the-mailstreams
	    &optional (start-index (Cache-Entry-Of 1 (first the-mailstreams))
		       start-index-supplied-p
		      )
	    (replace-from-index start-index-supplied-p)
	    (force-scroll-p nil)
	   )
  (declare (special *disable-add-associated-filters*))
  (ignore the-mailbox-name replace-from-index)
  (check-type start-index cache)
  (loop for stream in the-mailstreams
	when (not (typep stream 'imap-stream-mixin))
	do (yw-warn "~S is not an IMAP stream." stream)
  )
  (setq filter 
        (if (and the-filter (not (associated-p the-filter)))
	    (copy-and-concretify-filter the-filter the-owner the-mailstreams
					nil
            )
	    the-filter
	)
  )
  (if filter (send the-filter :Set-Computed-Order :Undefined))
  (setq mailstreams the-mailstreams)
  (setq owner the-owner)
  (send self :Set-Io-Buffer
	(send (send owner :Get-Pane :Prompt-Window) :Io-Buffer)
  )
  (send self :Set-Label (send self :Make-Label))
  (send self :Set-Up-For-Sequence force-scroll-p)
)

;;;Edited by Tom Gruber            18 Feb 92  11:31
(defmethod (summary-window :Set-Up-For-Sequence)
	   (&optional (force-scroll-p nil))
  (multiple-value-bind (change-p new-messages new-messages-at-end)
	  (send self :Make-Sequence-And-Item-List-Congruent
		(or filter
		    (make-a-sequence nil :Owner owner
		       :Mailbox mailstreams
		       :Sequence-Specifier '(:Sequence-All)
		    )
		)
	  )
     (if (or (not top-item) force-scroll-p change-p)
	 (let ((window-was-empty (not top-item)))
	   (send self :Scroll-To-Good-Place new-messages-at-end)
	   (if (or window-was-empty
		   (and item-list (not (aref screen-image 0)))
		   (and (not item-list) (aref screen-image 0))
	       )
	       (send self :Refresh)
	       nil
	   )
	   (send self :Resynch)
	   (send self :Set-Label (send self :Make-Label))
	   ;;; display-item has a dummy nil at the front to take the tail
	   (setq old-number-of-items (send self :number-of-items))
	 )
	 nil
     )
     (values (> (send self :number-of-items) 0) new-messages)
  )
)

(defun message< (message-a message-b &optional (given-sequence nil))
  "Says whether message-a is before message-b given the messages in the streams
denoted by the sequence.  This ignores the order in which the sequence
numberises the messages."
  (let ((stream-a (cache-mailstream message-a))
	(stream-b (cache-mailstream message-b))
       )
       (if (eq stream-a stream-b)
	   (< (cache-msg# message-a) (cache-msg# message-b))
	   (progn (assert given-sequence)
		  (let ((all-streams (send given-sequence :Mailstreams)))
		       (let ((pos-a (position stream-a all-streams :Test #'eq))
			     (pos-b (position stream-a all-streams :Test #'eq))
			    )
			    (or (< pos-a pos-b)
				(and (= pos-a pos-b)
				     (< (cache-msg# message-a)
					(cache-msg# message-b)
				     )
				)
			    )
		       )
		  )
	   )
       )
  )
)

(defun Message<= (Message-a message-b given-sequence)
  (or (eq Message-a message-b) (message< message-a message-b given-sequence))
)

(defun message>= (message-a message-b given-sequence)
  (message< message-b message-a given-sequence)
)

(defun message+ (message increment wrt-sequence)
  (ignore message increment wrt-sequence)
  (if wrt-sequence
      (ferror nil "Not implemented yet.")
      (cache-entry-of (+ increment (cache-msg# message))
		      (cache-mailstream message)
      )
  )
)

(Defun Compute-New-Items-For-Summary-Window
       (existing filter start-index new-item-p mailstreams)
  (let ((new-items
	 (if existing
	     ;;; This is just an optimisation.  Filter messages in an
	     ;;; existing window rather than starting from scratch.
	     (let ((subseq (third (send filter :Sequence-Specifier))))
		  (loop for item in (rest (send existing :Display-Item))
			for cache = (message-from-display-item item)
			when (and (Message<= Start-index cache filter)
				  (send self
					:Accept-Message-P-Given-Sequence
					cache subseq
				  )
			     )
			collect (multiple-value-bind (item new-p)
				    (send self :Make-Item cache filter)
				  (if new-p (setq new-item-p t) nil)
				  item
				)
		  )
	     )
	     (if (and *display-ordered-sequences-in-the-order-specified*
		      filter
		 )
		 (loop for cache
		       in (send self :Computed-Order-Safe)
		       when (Message<= Start-index cache filter)
		       collect (multiple-value-bind (item new-p)
				   (send self :Make-Item cache filter)
				 (if new-p (setq new-item-p t) nil)
				 item
			       )
		 )
		 (loop for stream in mailstreams
		       nconc (loop for i
				   from (if (eq (cache-mailstream start-index)
						stream
					    )
					    (Number-Of start-index)
					    1
					) 
				   to (send stream :MessageCnt)
				   for cache = (Cache-Entry-Of i stream)
				   when (send self :accept-message-p cache)
				   collect (multiple-value-bind (item new-p)
					       (send self :Make-Item
						     cache filter
					       )
					     (if new-p (setq new-item-p t) nil)
					     item
					   )
			     )
		 )
	     )
	 )
	)
       )
       (values new-items new-item-p)
  )
)

(defmethod (summary-window :new-messages) (new-count on-stream)
  (ignore new-count)
  (if item-list
      (let ((last-message
              (loop for item in item-list
                    for tail on item-list
                    for message = (Message-From-Display-Item item)
                    until (eq on-stream (cache-mailstream message))
                    finally
                     ;;; Tail is the first message in the mailbox of interest.
                     (return
                       (loop with last
                             = (Message-From-Display-Item (first tail))
                             for item in tail
                             for message = (Message-From-Display-Item item)
                             while (eq on-stream (cache-mailstream message))
                             do (setq last message)
                             ;;; Returns the last message in this mailbox.
                             finally (return last)
                       )
                     )
              )
            )
           )
           (if last-message
               (if (>= (cache-msg# last-message) new-count)
                   ;;; Then we have already done the setup, so do nothing.
                   nil
                   (send self :Set-Up owner filter
                         (send owner :Find-Mailbox-Name mailstreams)
                         mailstreams (Message+ last-message 1 nil)
                   )
               )
               (ferror nil "Something has gone wrong.")
           )
      )
      (send self :Set-Up owner filter
            (send owner :Find-Mailbox-Name mailstreams) mailstreams
      )
  )
)

(Defmethod (summary-window :mailbox-expunged) (on-stream)
  (if (member on-stream (send self :Mailstreams) :Test #'eq)
      (progn (if filter
		 (send filter :invalidate-computed-order)
		 nil
	     )
	     (send self :Set-Up owner filter
		   (send owner :Find-Mailbox-Name mailstreams)
		   mailstreams (Cache-Entry-Of 1 (first mailstreams))
		   nil t
	     )
      )
      nil
  )
)

(defun get-tab-position (string index)
  (let ((stop (position #\ string :Test #'char-equal :Start (+ index 1))))
       (let ((num (catch-error
		    (yw-read-from-string string nil -1 :Start (+ index 1)
					 :End stop
		    )
		  )
	     )
	    )
	    (values (if (numberp num)
		        (if (equal -1 num)
			    nil
			    num
			)
			nil
		    )
		    stop
	    )
       )
  )
)


(defun fat-string (x)
  (make-array 1 :Element-Type 'sys:fat-char :Initial-element x)
)

(defmethod (tv:scroll-window :string-motion)
	   (string &optional (start 0) end (start-x 0))
  (loop for i
	from start
	to (or (and end (min end (- (length string) 1)))
	       (- (length string) 1)
	   )
	do
	(let ((font :Default))
	(if (member (aref string i) '(#\  #\tab) :Test #'char-equal)
	    (let ((x start-x))
	         (let ((font-width
			 (tv:font-char-width (tv:coerce-font font self))
		       )
		      )
		      (let ((steps (* font-width
				      (selector (aref string i) char-equal
					(#\Tab 8)
					(#\  1)
				      )
				   )
			    )
			   )
			   (setf start-x
				 (* (+ 1 (floor X steps)) steps)
			   )
		      )
		 )
	    )
	    (if (char-equal #\ (aref string i))
		(multiple-value-bind (tab-pos new-index)
		    (get-tab-position string i)
		  (setq i new-index)
		  (if (and tab-pos (> tab-pos start-x))
		      (setf start-x tab-pos)
		      nil
		  )
		)
		(if (> (+ (+  start-x tv:width)
			  (- (tv:sheet-inside-right self)
			     (tv:sheet-char-width self))))
		    (setq start-x
			  (tv:sheet-compute-motion
			    self start-x 0
			    (fat-string (aref string i))
			  )
		    )
		    nil
		)
	    )
	)
	)
  )
  start-x
)

(defmethod (summary-window :string-out) (string &optional (start 0) end color)
  (loop with stop
	  = (or (and end (min end (- (length string) 1))) (- (length string) 1))
	for i from start to stop
	for char = (aref string i)
	do
	;;; These are chars anyway, so we should be ok with eq here ;char-equal)
	(if (member char '(#\  #\tab) :Test #'eq)
	    (let ((x (tv:sheet-cursor-x self))
		  (font :Default)
		 )
	         (let ((font-width
			 (tv:font-char-width (tv:coerce-font font self))
		       )
		      )
		      (let ((steps (* font-width
				      (selector char char-equal
					(#\Tab 8)
					(#\  1)
				      )
				   )
			    )
			   )
			   (setf (tv:sheet-cursor-x self)
				 (* (+ 1 (floor X steps)) steps)
			   )
		      )
		 )
	    )
	    (if (char-equal #\ char)
		(multiple-value-bind (tab-pos new-index)
		    (get-tab-position string i)
		  (setq i new-index)
		  (if (and tab-pos (> tab-pos (tv:sheet-cursor-x self)))
		      (setf (tv:sheet-cursor-x self) tab-pos)
		      nil
		  )
		)
		(if (zerop (char-font char))
		    ;;; Optimisation to reduce the number of calls to
		    ;;; sheet-tyo and %draw-char.
		    (loop for j from (+ 1 i) to stop
			  for ch = (aref string j)
			  while (and (zerop (char-font ch))
				   (not (member ch '(#\  #\tab #\) :Test #'eq))
				)
			  finally (tv:sheet-string-out self string i j color)
			          (setq i (- j 1))
		    )
		    (if (> (+ (+ (tv:sheet-cursor-x self) tv:width)
			      (- (tv:sheet-inside-right self)
				 (tv:sheet-char-width self))))
			(tv:sheet-tyo self (aref string i) nil color)
			nil
		    )
		)
	    )
	)
  )
)

(defun user-summary-window-inits (filter name streams)
  (ignore filter name streams)
  *user-summary-window-inits*
)

(defun make-summary-window (filter name streams)
  (apply 'make-instance
	 'Summary-Window
	 :Filter filter
	 :MailStreams streams
	 (User-Summary-Window-Inits filter name streams)
  )
)

(defun reposition-summary-windows (&optional (resize-p nil))
  (apply-to-summary-windows
    #'(lambda (window) (send window :Reposition resize-p))
  )
)

(defun apply-to-summary-windows (function &rest args)
  (funcall 'map-yw-resource
	   #'(lambda (window &rest ignore) (apply function window args))
	   'Summary-Windows
  )
)

(defun allocate-yw-resource (name)
  (or (loop for entry in (get name 'resource)
	    for (thing used-p) = entry
	    when (not used-p)
	    do (setf (second entry) t)
	       (return thing)
      )
      (progn (setf (get name 'resource)
		   (append (get name 'resource)
			   (list (list (funcall (get name 'constructor)) nil))
		   )
	     )
	     (allocate-yw-resource name)
      )
  )
)

(defun clear-yw-resource (resource-name)
  (setf (get resource-name 'resource) nil)
)

(defun deallocate-yw-resource (resource-name Object)
  (loop for entry in (get resource-name 'resource)
	for (thing used-p) = entry
	when (eq Object thing)
	do (setf (second entry) nil)
	   (return thing)
	finally (ferror nil "~S is not in resource ~S" Object resource-name)
  )
)

(defun is-in-yw-resource (Object resource-name)
  (loop for entry in (get resource-name 'resource)
	for (thing used-p) = entry
	when (eq Object thing)
	do (return t)
  )
)

(defun map-yw-resource
   (function resource-name &optional (include-function #'(lambda (ignore) t))
    &rest extra-args
   )
  "Call FUNCTION on each object created in resource RESOURCE-NAME.
FUNCTION gets three args at each call: the object, whether the resource
believes it is in use, and RESOURCE-NAME."
  ;;Windows are the user's problem....
  (loop for entry in (get resource-name 'resource)
	for (Object in-use-p) = entry
	when (and object
		  (or (not include-function)
		      (funcall include-function object)
		  )
	     )
	do (apply function object in-use-p resource-name extra-args)
  )
)

(defun map-yw-resource-return
   (function resource-name &optional (include-function #'(lambda (ignore) t))
    &rest extra-args
   )
  "Call FUNCTION on each object created in resource RESOURCE-NAME.
FUNCTION gets three args at each call: the object, whether the resource
believes it is in use, and RESOURCE-NAME.  Return a list of the answers."
  ;;Windows are the user's problem....
  (loop for entry in (get resource-name 'resource)
	for (Object in-use-p) = entry
	when (and object
		  (or (not include-function)
		      (funcall include-function object)
		  )
	     )
	collect
	(apply function object in-use-p resource-name extra-args)
  )
)

(defun remove-from-yw-resource (resource-name object)
"Remove OBJECT the resource RESOURCE-NAME.  OBJECT should have been returned
by a previous call to ALLOCATE-RESOURCE.
"
  (loop for entry in (get resource-name 'resource)
	for (thing used-p) = entry
	when (eq Object thing)
	do (setf (get resource-name 'resource)
		 (remove entry (get resource-name 'resource)))
	   (return thing)
	finally (ferror nil "~S is not in resource ~S" Object resource-name)
  )
)

(defun (:Property Summary-Windows constructor) ()
  (make-summary-window nil nil nil)
)

;(defwindow-resource summary-windows ()
;  :Initial-Copies 0
;  :Constructor (make-summary-window nil nil nil)
;)

;-------------------------------------------------------------------------------

(Check-Window-Resource 'summary-windows 'summary-window)
;(Clear-Yw-Resource 'summary-windows)