;;; -*- Mode:Common-Lisp; Package:IMAP; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************

(defmacro define-server-special-variable (name value)
"Defines a new server special variable called Name with default value Value.
Server special vars are bound at the top level of the IMAP server.
"
 `(progn (setq *all-server-special-variables*
	       (remove (assoc ',name *all-server-special-variables*)
		       *all-server-special-variables*
	       )
	 )
	 (push '(,name ,value) *all-server-special-variables*)
	 (sys:record-source-file-name ',name 'defvar)
	 (eval-when (compile load eval) (proclaim '(special ,name)))
	 ',name
  )
)


(defmacro with-server-special-variables (&body body)
"Declares all server special vars as special."
  `(locally
     (declare (special ,@(mapcar #'first *all-server-special-variables*)))
     ,@body
   )
)

(defmacro binding-server-special-variables (&body body)
"Binds all server special vars to their default values around body."
 `(progw *all-server-special-variables*
    (With-Server-Special-Variables ,@body)
  )
)

(defmacro with-file-locked ((file &rest prelude-forms) &body body)
"Executes Body with the file lock of file locked.  Executes prelude
forms without-interrupts before it gets the lock.
"
 `(let ((.path. (fs:default-pathname ,file)))
       (flet ((_body_ () ,@body)
	      (_prelude_ () ,@prelude-forms)
	     )
	 (if (equal (pathname-host (translate-pathname .path.)) si:local-host)
	     (let ((file (fs:lookup-file (pathname-directory .path.)
					 (pathname-name .path.)
					 (pathname-type .path.)
					 (pathname-version .path.)
			 )
		   )
		  )
		  (if file
		      (progn (without-interrupts (_prelude_))
			     (fs:locking-recursively (fs:file-lock file)
			       (_body_)
			     )
		      )
		      (progn (_prelude_) (_body_))
		  )
	     )
	     (progn (_prelude_) (_body_))
	 )
       )
  )
)

(defmacro with-write-lock ((mailbox) &body body)
"Executes body with a write lock on the mailbox."
 `(with-lock ((yw:imap.lock-location (mail-file-lock ,mailbox))) ,@body)
)

(defmacro waiting-for-file-lock (&body body)
"Executes body in a wrapper that causes it to wait and retry later if is
gets a lock collision.  This is important because some file lock collision
conditions seem to cause lossage that we cannot properly process-wait for.
"
 `(flet ((body-function () ,@body))
    (condition-case (condition)
      (body-function)
      (fs:file-operation-failure
	(if (search "File is locked for" (send condition :Format-String)
		    :Test #'char-equal
	    )
	    (progn (sleep 10 "Wait for file unlock.")
		   (body-function)
	    )
	    (signal-condition condition)
	)
      )
    )
  )
)