;;; -*- Mode:Common-Lisp; Package:IMAP; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; This code was written by James Rice.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.

;;; The development of this software was assisted by the following grants:
;;; Biomedical Research Technology Program of the National Institutes
;;; of Health under grant RR-00785
;;; Information Systems Technologies office of the Defense Advanced
;;; Research Projects Agency under contract N00039-86-C0033.

;;; **********************************************************************

;-------------------------------------------------------------------------------

;;; The Fetch command.

(defun consolidate-fetch-key (key numbers)
"Translates a fetch key into the internal representation of the key and
 range numbers.
"
  (selector key string-equal
    (:All '((:Flags nil nil)
	    (:Internaldate nil nil)
	    (:Rfc822.Size nil nil)
	    (:Envelope nil nil)
	   )
    )
    (:Fast '((:Flags nil nil)
	     (:Internaldate nil nil)
	     (:Rfc822.Size nil nil)
	    )
    )
    (:Envelope	        '((:Envelope nil nil)))
    (:Flags		'((:Flags nil nil)))
    (:Internaldate	'((:Internaldate nil nil)))
    (:Rfc822.Header	`((:Rfc822.Header  ,@numbers)))
    (:Rfc822.Text	`((:Rfc822.Text    ,@numbers)))
    (:Rfc822.Text*	`((:Rfc822.Text*   ,@numbers)))
    (:Rfc822.Size	`((:Rfc822.Size    ,@numbers)))
    (:Rfc822		`((:Rfc822         ,@numbers)))
    (Otherwise (list (cons (intern (string-upcase key) 'keyword) numbers)))
  )
)

(defun consolidate-fetch-keys (keys)
"Turns a list of keys of the form (foo bar (3 4) baz) into 
 ((foo) (bar 3 4) (baz)).
"
  (if keys
      (let ((key (first keys))
	    (numbers (second keys))
	   )
	   (if (consp numbers)
	       (append (consolidate-fetch-key key numbers)
		       (consolidate-fetch-keys (rest (rest keys)))
	       )
	       (append (consolidate-fetch-key key '(nil nil))
		       (consolidate-fetch-keys (rest keys))
	       )
	   )
      )
      nil
  )
)

(defmethod process-request ((command (eql :Fetch)) stream)
"Parses the fetch types from the stream."
  (let ((line (read-line stream)))
       (with-input-from-string (stream line)
	 (let ((sequence (read-sequence stream))
	       (things-to-fetch (read-sexpression stream))
	      )
	      (list sequence
		    (if (consp things-to-fetch)
			(Consolidate-Fetch-Keys things-to-fetch)
			(let ((numbers
				(if (feature-enabled-p :Indexable.Fields)
				    (Read-maybe-number-range stream)
				    '(nil nil)
				)
			      )
			     )
			     (consolidate-fetch-key things-to-fetch numbers)
			)
		    )
	      )
	 )
       )
  )
)

(defmethod process-imap-server-command
	   (stream tag (command (eql :Fetch)) &rest arglist)
"Processes a fetch command.  The things to fetch have already been parsed.
Sends solicited data for all of the tihings we need to fetch.
"
  (declare (special *current-mailbox*))
  (ignore stream tag command arglist)
  (destructuring-bind (sequence things-to-fetch) arglist
    (with-write-lock (*current-mailbox*)
      (let ((errors nil))
	   (loop for message-number in sequence
		 for message = (The-Message message-number *current-mailbox*)
		 do (Send-Solicited-Command-Start
		      stream tag message-number :Fetch #\(
		    )
		    (loop for (thing from to) in things-to-fetch
			  for rest on things-to-fetch
			  when (equal :Error
				      (fetch-1 stream *Current-Mailbox*
					       Message thing from to
				      )
			       )
			  do (push thing errors)
			  when (rest rest)
			  do (Send-Solicited-Command-Component stream #\space)
		    )
		    (Send-Solicited-Command-Component stream #\))
		    (finish-solicited-command stream)
	   )
	   (if errors
	       (values :Bad (format nil "Fetch Completed with errors.  ~
                                        [~{~A~^ ~}] are illegal." errors
			    )
	       )
	       (values :Ok "Fetch Complete")
	   )
      )
    )
  )
)


(defmethod fetch-1 (stream mailbox message (thing (eql :Flags)) from to)
"Sends back the flags to the client."
  (ignore mailbox from to)
  (let ((flags (decode-flags (message-flags message))))
       (if flags
	   (Send-Solicited-Command-Component stream thing flags)
	   (Send-Solicited-Command-Component stream thing #\( #\))
       )
  )
)

(defmethod Fetch-1
	   (stream mailbox message (thing (eql :Internaldate)) from to)
"Sends the internal date to the client."
  (ignore mailbox from to)
  (Send-Solicited-Command-Component
    stream thing (message-internal-date message)
  )
)

(defmethod Fetch-1
	   (stream mailbox message (thing (eql :Rfc822.Size)) from to)
"Sends the size of the message to the client."
  (ignore mailbox from to)
  (Send-Solicited-Command-Component stream thing (message-length message))
)

(defmethod fetch-1 (stream mailbox message (thing (eql :Rfc822.Header)) from to)
"Sends the rfc header to the client."
  (ignore mailbox)
  (Send-Maybe-Bounded-String stream thing (message-header message) from to)
)

(defun auto.set.seen-p ()
  "Is true if we should automatically set the \\seen flag."
  (declare (special *selected-version*))
  (or (and (equal *selected-version* *imap2-version*)
	   (not (feature-enabled-p :~Auto.Set.Seen))
      )
      (feature-enabled-p :Auto.Set.Seen)
  )
)

(defmethod fetch-1 (stream mailbox message (thing (eql :Rfc822.text)) from to)
"Sends the rfc text to the client, maybe setting the seen flag."
  (Send-Maybe-Bounded-String stream thing (message-body message) from to)
  (if (Auto.Set.Seen-P)
      (progn (Set-Flag mailbox message :\\SEEN t)
	     (Fetch-1 stream mailbox message :Flags nil nil)
      )
      nil
  )
)

(defmethod fetch-1 (stream mailbox message (thing (eql :Rfc822.Text*)) from to)
"Sends the rfc text to the client, never setting the seen flag."
  (declare (special *selected-version*))
  (ignore mailbox)
  (if t ;;; (equal *selected-version* *imap2-version*) ;;;{!!!!}
      (Send-Maybe-Bounded-String stream thing (message-body message) from to)
      :Error
  )
)

(defmethod fetch-1 (stream mailbox message (thing (eql :Rfc822)) from to)
"Sends the rfc header and text to the client, maybe setting the seen flag."
  (Send-Maybe-Bounded-String
    stream thing (message-rfc822 message) from to
  )
  (if (Auto.Set.Seen-P)
      (progn (Set-Flag mailbox message :\\SEEN t)
	     (Fetch-1 stream mailbox message :Flags nil nil)
      )
      nil
  )
)

(defmethod fetch-1 (stream mailbox message (thing (eql :Rfc822*)) from to)
"Sends the rfc header and text to the client, never setting the seen flag."
  (declare (special *selected-version*))
  (ignore mailbox)
  (if t ;;; (equal *selected-version* *imap2-version*) ;;; {!!!!}
      (Send-Maybe-Bounded-String
	stream thing (message-rfc822 message) from to
      )
      :Error
  )
)

(defmethod fetch-1 (stream mailbox message (thing (eql :Rfc822.Size)) from to)
"Sends the rfc size to the client."
  (ignore mailbox from to)
  (Send-Solicited-Command-Component stream thing (message-length message))
)

(defun decode-address (address)
"Decodes an address object into an IMAP list representation."
  (declare (special *selected-version*))
  (if (equal *Selected-Version* *imap2-version*)
      (list (yw:address-personalname address)
	    (yw:address-routelist    address)
	    (yw:address-mailbox      address)
	    (yw:address-host         address)
      )
      (list (yw:address-personalname address)
	    (yw:address-routelist    address)
	    (yw:address-mailbox      address)
	    (yw:address-host         address)
	    (yw:address-comment      address)
      )
  )
)

(defun decode-addresses (addresses)
  "Decodes a bunch of addresses."
  (mapcar 'decode-address addresses)
)

(defun nil-if-null-string (entry)
"Returns NIL if entry is the null string."
  (if (equal "" entry)
      nil
      entry
  )
)

(defun decode-envelope (envelope)
"Given an envelope object decodes it into the IMAP list notation, including
any addresses that it may contain.
"
  (list (yw:envelope-date envelope)
	(yw:envelope-subject envelope)
	(decode-addresses (yw:envelope-from envelope))
	(decode-addresses (yw:envelope-sender envelope))
	(decode-addresses (yw:envelope-reply-to envelope))
	(decode-addresses (yw:envelope-to envelope))
	(decode-addresses (yw:envelope-cc envelope))
	(decode-addresses (yw:envelope-bcc envelope))
	(nil-if-null-string (yw:envelope-in-reply-to envelope))
	(nil-if-null-string (yw:envelope-messageid envelope))
  )
)

(defmethod fetch-1 (stream mailbox message (thing (eql :Envelope)) from to)
"Fetches an envelope for the client."
  (ignore mailbox from to)
  (Send-Solicited-Command-Component
    stream thing (decode-envelope (message-envelope message))
  )
)

(defmethod fetch-1 (stream mailbox message (thing (eql :Bcc)) from to)
"Fetches a BCC field for the client."
  (ignore mailbox from to)
  (Send-Solicited-Command-Component
    stream thing (decode-addresses (yw:envelope-bcc (message-envelope message)))
  )
)

(defmethod fetch-1 (stream mailbox message (thing (eql :Body)) from to)
"Fetches the message Body for the client."
  (ignore mailbox)
  (Send-Maybe-Bounded-String stream thing (message-body message) from to)
)

(defmethod fetch-1 (stream mailbox message (thing (eql :CC)) from to)
"Fetches a CC field for the client."
  (ignore mailbox from to)
  (Send-Solicited-Command-Component
    stream thing (decode-addresses (yw:envelope-cc (message-envelope message)))
  )
)

(defmethod fetch-1 (stream mailbox message (thing (eql :$CC)) from to)
"Fetches a $CC field for the client."
  (ignore mailbox from to)
  (Send-Solicited-Command-Component
    stream thing (decode-addresses (yw:envelope-$cc (message-envelope message)))
  )
)

(defmethod fetch-1 (stream mailbox message (thing (eql :From)) from to)
"Fetches a FROM field for the client."
  (ignore mailbox from to)
  (Send-Solicited-Command-Component
   stream thing (decode-addresses (yw:envelope-From (message-envelope message)))
  )
)

(defmethod fetch-1 (stream mailbox message (thing (eql :$From)) from to)
"Fetches a $FROM field for the client."
  (ignore mailbox from to)
  (Send-Solicited-Command-Component
    stream thing
    (decode-addresses (yw:envelope-$from (message-envelope message)))
  )
)

(defmethod fetch-1 (stream mailbox message (thing (eql :Header)) from to)
"Fetches a header for the client."
  (ignore mailbox)
  (Send-Maybe-Bounded-String stream thing (message-header message) from to)
)

(defmethod fetch-1 (stream mailbox message (thing (eql :Size)) from to)
"Fetches a size for the client."
  (ignore mailbox from to)
  (Send-Solicited-Command-Component
    stream thing (+ (message-length message)
		    (* (- (length (send stream :Eol-Sequence)) 1)
		       (Count-Lines-In (message-rfc822 message))
		    )
		 )
  )
)

(defmethod fetch-1 (stream mailbox message (thing (eql :Subject)) from to)
"Fetches a subject field for the client."
  (ignore mailbox)
  (Send-Maybe-Bounded-String
    stream thing (yw:envelope-subject (message-envelope message)) from to
  )
)

(defmethod fetch-1 (stream mailbox message (thing (eql :$Subject)) from to)
"Fetches a $Subject field for the client."
  (ignore mailbox)
  (Send-Maybe-Bounded-String
    stream thing (yw:envelope-$Subject (message-envelope message)) from to
  )
)

(defmethod fetch-1 (stream mailbox message (thing (eql :Text)) from to)
"Fetches the message text for the client."
  (if (feature-enabled-p :Encoding)
      (let ((header-field (Get-Value-Of-Field :Encoding message)))
	   (if header-field
	       (let ((parsed (Parse-Encoding-List header-field message)))
		    (let ((texts (loop for (key (number format)) in parsed
				       for spec in parsed
				       when (eq :Text format)
				       collect (Get-Encoded-Body-Field
						 spec mailbox message
					       )
				 )
			  )
			 )
		         (Send-Maybe-Bounded-String
			   stream thing (Concatenate-Strings-With-Nls texts)
			   from to
			 )
		    )
	       )
	       (fetch-1 stream mailbox message :Rfc822.Text from to)
	   )
      )
      (fetch-1 stream mailbox message :Rfc822.Text from to)
  )
)

(defmethod fetch-1 (stream mailbox message (thing (eql :To)) from to)
"Fetches a TO field for the client."
  (ignore mailbox from to)
  (Send-Solicited-Command-Component
    stream thing (decode-addresses (yw:envelope-To (message-envelope message)))
  )
)

(defmethod fetch-1 (stream mailbox message (thing (eql :$To)) from to)
"Fetches a $To field for the client."
  (ignore mailbox from to)
  (Send-Solicited-Command-Component
    stream thing (decode-addresses (yw:envelope-$To (message-envelope message)))
  )
)

;-------------------------------------------------------------------------------

(defun parsed-encoding-list (message)
  "Returns the parsed encoding list for Message."
  (let ((header-field (Get-Value-Of-Field :Encoding message)))
       (if header-field
	   (parse-encoding-list header-field message)
	   nil
       )
  )
)

(defun step-forward-n-lines (index count string)
"Moves Index into String forward by Count lines."
  (if (equal 0 count)
      (values index :Last-Line)
      (let ((nl-pos (position #\newline string :Test #'char= :Start index)))
	   (if nl-pos
	       (if (equal (+ 1 nl-pos) (length string))
		   (values nl-pos :Nl-At-End)
		   (step-forward-n-lines (+ 1 nl-pos) (- count 1) string)
	       )
	       (values (length string) :Run-Out-Of-Newlines)
	   )
      )
  )
)
      
(defun parse-encoded-body (wrt-encoding mailbox message)
  "Returns a parsed encoding alist that maps the keys mentioned in the
 encodings to the appropriate message body substrings."
  (ignore mailbox)
  (let ((index 0)
	(body (message-body message))
       )
       (loop for (key (line-count)) in wrt-encoding
	     for end-of-part = (if (numberp line-count)
				   (step-forward-n-lines index line-count body)
				   nil
			       )
	     collect (list key (nsubstring body index end-of-part))
	     do (setq index (step-forward-n-lines end-of-part 1 body))
       )
  )
)

(defun Get-Encoded-Body-Field
       (encoding-spec mailbox message &optional (all-encodings nil))
  "Gets an encoded field named in the encoding-spec."
  (destructuring-bind (concrete-key format-and-options) encoding-spec
    (ignore format-and-options)
    (if (yw:is-present (message-parsed-body message))
	(let ((entry (assoc concrete-key (message-parsed-body message))))
	     (if entry
		 (second entry)
		 nil
	     )
	)
	(progn (setf (message-parsed-body message)
		     (parse-encoded-body
		       (or all-encodings (Parsed-Encoding-List message))
		       mailbox message
	             )
	       )
	       (get-encoded-body-field encoding-spec mailbox message)
	)
    )
  )
)

(defmethod fetch-1 (stream mailbox message (thing symbol) from to)
"Fetches a concrete key value for Thing."
  (ignore mailbox)
  (if (feature-enabled-p :Encoding)
      (let ((parsed (parsed-encoding-list message)))
	   (if (assoc thing parsed :Test #'eq)
	       (Send-Maybe-Bounded-String
		 stream thing
		 (get-encoded-body-field (assoc thing parsed :Test #'eq)
					 mailbox message
                 )
		 from to
               )
	       (Send-Maybe-Bounded-String
		 stream thing (get-value-string-of-field thing message) from to
	       )
	   )
      )
      (Send-Maybe-Bounded-String
	stream thing (get-value-string-of-field thing message) from to
      )
  )
)

;-------------------------------------------------------------------------------

(defun split-into-args (string start)
"Splits a string of the form \"foo, bar, baz\" into '(\"foo\" \"bar\" \"baz\")"
  (let ((index (string-search-set '(#\,) string start)))
       (cons (string-append
	       #\(
	       (string-trim yw:*whitespace-chars* (subseq string start index))
	       #\)
	     )
	     (if index
		 (split-into-args string (+ 1 index))
		 nil
	     )
       )
  )
)

(defun parse-encoding-list (header-field message)
"Parses the encoding field into a list."
  (multiple-value-bind (parsed found-p) (get header-field :Parsed-Encoding)
    (if (and found-p (yw:is-present parsed))
	parsed
        (let ((args (Split-Into-Args
		      (Get-Value-String-Of-Field :Encoding message) 0
		    )
	      )
	      (*package* (find-package 'keyword))
	      (*read-base* 10.)
	     )
	     (let ((list (loop for arg
			       in (loop for tail on args
					for arg in args
					collect (if (rest tail)
						    arg
						    (if (numberp (first arg))
							arg
							(cons 0 arg)
						    )
						)
				  )
			       collect
			       (list (gensym "ENCODING-")
				     (catch-error (read-from-string arg) nil)
			       )
			 )
		   )
		  )
	          (setf (get header-field :Parsed-Encoding) list)
	     )
	     (parse-encoding-list header-field message)
	)
    )
  )
)

(defmethod fetch-1 (stream mailbox message (thing (eql :Encoding)) from to)
  "Fetches the Generic Encoding key."
  (ignore mailbox)
  (if (feature-enabled-p :Encoding)
      (let ((header-field (Get-Value-Of-Field thing message)))
	   (if header-field
	       (let ((parsed (parse-encoding-list header-field message)))
		    (Send-Solicited-Command-Component
		      stream thing (loop for (key (number . things)) in parsed
					 collect (cons key things)
				   )
		    )
	       )
	       (Send-Maybe-Bounded-String stream thing nil nil nil)
	   )
      )
      ;;; Otherwise treat this as a Concrete Key.
      (Send-Maybe-Bounded-String
	stream thing (get-value-string-of-field thing message) from to
      )
  )
)

;-------------------------------------------------------------------------------

;;; The STORE command.
(defmethod process-request ((command (eql :Store)) stream)
"Parses the STORE request from the client."
  (let ((sequence (read-sequence stream))
	(keyword (read-keyword stream))
       )
       (case keyword
	((:+Flags :-Flags :Flags)
	 (list sequence keyword (yw:list-if-not (Read-Sexpression stream)))
	)
	((:Rfc822.text :Rfc822.Header)
	 (list sequence keyword (Read-String stream))
        )
	;;; This must be some generic, concrete or canonical key.
	(otherwise (list sequence keyword (read-sexpression stream)))
       )
  )
)

(defmethod process-imap-server-command
	   (stream tag (command (eql :Store)) &rest arglist)
"Processes the STORE command from the client."
  (declare (special *current-mailbox*))
  (ignore stream tag command arglist)
  (destructuring-bind (sequence store-type argument) arglist
    (with-write-lock (*current-mailbox*)
      (loop for message-number in sequence
	    for message = (The-Message message-number *current-mailbox*)
	    do (store-1 store-type stream argument tag *Current-Mailbox*
			Message message-number
	       )
      )
    )
    (values :Ok "Store Complete")
  )
)

(defmethod store-1 ((store-type (eql :Flags)) stream argument
		    tag mailbox message message-number
		   )
"Processes the STORE FLAGS command."
  (store-1-flags store-type stream argument tag mailbox message message-number)
)

(defmethod store-1 ((store-type (eql :+Flags)) stream argument
		    tag mailbox message message-number
		   )
"Processes the STORE +FLAGS command."
  (store-1-flags store-type stream argument tag mailbox message message-number)
)

(defmethod store-1 ((store-type (eql :-Flags)) stream argument
		    tag mailbox message message-number
		   )
"Processes the STORE -FLAGS command."
  (store-1-flags store-type stream argument tag mailbox message message-number)
)

(defun Store-1-Flags
       (store-type stream argument tag mailbox message message-number)
"Processes the STORE FLAGS, +FLAGS and -FLAGS commands."
  (if (equal :Flags store-type)
      (loop for flag in (message-flags message) do (setf (second flag) nil))
      nil
  )
  (loop for flag in argument do
	(Set-Flag mailbox message (net:intern-as-keyword flag)
		  (not (equal store-type :-flags))
	)
  )
  (say-flags-have-changed-for tag mailbox message-number message stream)
)


(defun say-flags-have-changed-for (tag mailbox message-number message stream)
"Tells the client that the flags have changed"
  (Send-Solicited-Command-Start stream tag message-number :Store #\()
  (fetch-1 stream mailbox message :Flags nil nil)
  (Send-Solicited-Command-Component stream #\))
  (finish-solicited-command stream)
)

;-------------------------------------------------------------------------------

;;; Store of some Concrete keys.

(defun Header-Changed-Actions
       (stream tag mailbox message message-number &optional (changed-field nil))
  (setf (mail-file-modified-p mailbox) t)
  (Send-Solicited-Command-Start stream tag message-number :Store #\()
  (Send-Solicited-Command-Component stream #\()
  (fetch-1 stream mailbox message :Rfc822.Size nil nil)
  (Send-Solicited-Command-Component stream #\space)
  (fetch-1 stream mailbox message :Envelope nil nil)
  (Send-Solicited-Command-Component stream #\space)
  (if changed-field
      (progn (fetch-1 stream mailbox message changed-field nil nil)
	     (Send-Solicited-Command-Component stream #\space)
      )
      nil
  )
  (fetch-1 stream mailbox message :Rfc822.Header nil nil)
  (Send-Solicited-Command-Component stream "))")
  (finish-solicited-command stream)
)

(defmethod store-1 ((store-type (eql :Rfc822.Header)) stream argument
		    tag mailbox message message-number
		   )
"Processes the STORE RFC822.HEADER command."
  (reset-message-header
    message message-number mailbox
    argument (mail-file-format mailbox)
  )
  (header-changed-actions stream tag mailbox message message-number)
)

(defmethod store-1 ((store-type (eql :Rfc822.Text)) stream argument
		    tag mailbox message message-number
		   )
"Processes the STORE RFC822.TEXT command."
  (reset-message-body
    message mailbox argument (mail-file-format mailbox)
  )
  (setf (mail-file-modified-p mailbox) t)
  (Send-Solicited-Command-Start stream tag message-number :Store #\()
  (Send-Solicited-Command-Component stream #\()
  (fetch-1 stream mailbox message :Rfc822.Size nil nil)
  (Send-Solicited-Command-Component stream #\space)
  (fetch-1 stream mailbox message :Rfc822.Text nil nil)
  (Send-Solicited-Command-Component stream "))")
  (finish-solicited-command stream)
)

;-------------------------------------------------------------------------------

;;; Store of Generic Keys.

(defmethod store-1 ((store-type (eql :BCC)) stream argument
		    tag mailbox message message-number
		   )
"Processes the STORE BCC command."
  (store-generic-key
    store-type stream argument tag mailbox message message-number
  )
)

(defmethod store-1 ((store-type (eql :Body)) stream argument
		    tag mailbox message message-number
		   )
"Processes the STORE Body command."
  (Store-1 :Rfc822.Text stream argument tag mailbox message message-number)
)

(defmethod store-1 ((store-type (eql :CC)) stream argument
		    tag mailbox message message-number
		   )
"Processes the STORE CC command."
  (store-generic-key
    store-type stream argument tag mailbox message message-number
  )
)

(defmethod store-1 ((store-type (eql :Encoding)) stream argument
		    tag mailbox message message-number
		   )
"Processes the STORE Encoding command.  Note: this is here because this is
magical when the Encoding feature is enabled.
"
  (if (feature-enabled-p :Encoding)
      ;;; This must be a new encoding list so do magic things.
      (let ((header-field
	      (or (Get-Value-Of-Field store-type message)
		  ;;; Creates a new header field for the encoding
		  (progn (store-generic-key
			   store-type stream "" tag mailbox message
			   message-number
			 )
			 (Get-Value-Of-Field store-type message)
		  )
	      )
	    )
	   )
	   (setf (get header-field :Parsed-Encoding) argument)
	   (restringify-encoding-field message)
      )
      (store-generic-key
	store-type stream argument tag mailbox message message-number
      )
  )
)

(defmethod store-1 ((store-type (eql :From)) stream argument
		    tag mailbox message message-number
		   )
"Processes the STORE From command."
  (store-generic-key
    store-type stream argument tag mailbox message message-number
  )
)

(defmethod store-1 ((store-type (eql :Header)) stream argument
		    tag mailbox message message-number
		   )
"Processes the STORE Header command."
  (Store-1 :Rfc822.Header stream argument tag mailbox message message-number)
)

(defmethod store-1 ((store-type (eql :Subject)) stream argument
		    tag mailbox message message-number
		   )
"Processes the STORE Subject command."
  (store-generic-key
    store-type stream argument tag mailbox message message-number
  )
)

(defmethod store-1 ((store-type (eql :Text)) stream argument
		    tag mailbox message message-number
		   )
"Processes the STORE Text command."
  ;;; {!!!!}  What do we need to do about EOL and encoding here?
  (Store-1 :Rfc822.Text stream argument tag mailbox message message-number)
)

(defmethod store-1 ((store-type (eql :To)) stream argument
		    tag mailbox message message-number
		   )
"Processes the STORE To command."
  (store-generic-key
    store-type stream argument tag mailbox message message-number
  )
)

(defun store-generic-key
	   (key stream argument tag mailbox message message-number)
"Processes the STORE of a generic key."
  (Set-Value-Of-Field key message argument)
  (reset-message-header
    message message-number mailbox
    (message-parsed-header message) (mail-file-format mailbox)
  )
  (header-changed-actions stream tag mailbox message message-number key)
)

;-------------------------------------------------------------------------------

;;; Store of Canonical keys.

(defmethod store-1 ((store-type (eql :$CC)) stream argument
		    tag mailbox message message-number
		   )
"Processes the STORE $CC command."
  (store-generic-key
    :CC stream argument tag mailbox message message-number
  )
)

(defmethod store-1 ((store-type (eql :$From)) stream argument
		    tag mailbox message message-number
		   )
"Processes the STORE $From command."
  (store-generic-key
    :From stream argument tag mailbox message message-number
  )
)

(defmethod store-1 ((store-type (eql :$Subject)) stream argument
		    tag mailbox message message-number
		   )
"Processes the STORE $Subject command."
  (store-generic-key
    :Subject stream argument tag mailbox message message-number
  )
)

(defmethod store-1 ((store-type (eql :$To)) stream argument
		    tag mailbox message message-number
		   )
"Processes the STORE $To command."
  (store-generic-key
    :To stream argument tag mailbox message message-number
  )
)

;-------------------------------------------------------------------------------

;;; Store concrete keys.

(defun concatenate-strings-with-nls (strings)
"Turns a list of strings into a string putting nls between them."
  (if (rest strings)
      (string-append (first strings) #\newline
		     (concatenate-strings-with-nls (rest strings))
      )
      (first strings)
  )
)

(defun count-lines-in (string)
  "Returns the number of lines in the string."
  (+ 1 (count #\newline string :Test #'char=))
)

(defun restringify-encoding-field (message)
  "Takes the encoding field in Message and turns it back into an RFC1154
 Encoding: header field."
  (let ((header-field (Get-Value-Of-Field :Encoding message))
	(parsed (parsed-encoding-list message))
	(*print-case* :Capitalize)
	(*print-base* 10.)
       )
       ;;; Reset length attributes.
       (loop for (key encoding-data) in parsed
	     for value = (assoc key (message-parsed-body message) :Test #'eq)
	     do (setf (first encoding-data) (count-lines-in value))
       )
       (let ((new-string
	       (with-output-to-string (*standard-output*)
		 (format t "Encoding: ")
		 (loop for (key encoding-data) in parsed
		       for rest on parsed
		       for value
		         = (assoc key (message-parsed-body message) :Test #'eq)
		       when value
		       do (if (eq rest parsed)
			      nil
			      (format t ", ")
			  )
			  (if (rest rest)
			      (format t "~{~A~^ ~}" encoding-data)
			      (format t "~{~A~^ ~}" (rest encoding-data))
			  )
		 )
	       )
	     )
	    )
	    (send header-field :Set-String new-string)
       )
  )
)

(Defun update-message-body-from-encodings (message mailbox parsed)
  "Takes the encoded body parts and turns them back into a new message body."
  (let ((new-body
	  (Concatenate-Strings-With-Nls
	    (loop for (key encoding-data) in parsed
		  for value
		  = (assoc key (message-parsed-body message) :Test #'eq)
		  collect (if value (list value) "")
	    )
	  )
	)
       )
       (restringify-encoding-field message)
       (reset-message-body
	 message mailbox new-body (mail-file-format mailbox)
       )
       (setf (mail-file-modified-p mailbox) t)
  )
)

(defmethod store-1 ((store-type symbol) stream argument
		    tag mailbox message message-number
		   )
"Processes the STORE command of a random concrete key.  Note:  we have to
 do magic things in the event of having a parsed encoding."
  (flet ((concrete ()
	   (store-generic-key
	     store-type stream argument tag mailbox message message-number
	   )
	 )
	)
        (if (feature-enabled-p :Encoding)
	    (let ((header-field (Get-Value-Of-Field store-type message)))
		 (if (get header-field :Parsed-Encoding)
		     (let ((parsed (parsed-encoding-list message)))
			  (let ((entry (assoc store-type parsed :Test #'eq)))
			       (if entry
				   ;;; Then we are replacing an encoded
				   ;;; component.
				   (progn (setf (second entry) argument)
					  (update-message-body-from-encodings
					    message mailbox parsed
					  )
				   )
				   ;;; This isn't an encoding so just concrete.
				   (concrete)
			       )
			  )
		     )
		     ;;; Then the encoding has not been parsed so
		     ;;; we know we are just a concrete key.
		     (concrete)
		 )
	    )
	    (concrete)
	)
  )
)


;-------------------------------------------------------------------------------

