;;; -*- Mode:Common-Lisp; Package:W; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.  Where functionality implemented herein replicates
;;; similarly named functionality on Symbolics machines, this code was
;;; developed solely from the interface specification in the documentation
;;; or through guesswork, never by examination of Symbolics source code.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.
;;; **********************************************************************

;;; This software developed by:
;;;	Rich Acuff
;;; at the Stanford University Knowledge Systems Lab in Nov '87.
;;;
;;; This work was supported in part by:
;;;	NIH Grant 5 P41 RR00785-15

;;; Code to allow windows to be "shrunk" to icons that can be
;;; independantly positioned and clicked on to reactivate the window
;;; being represented.

;;; Since I want this to work with the current system tools, and I don't
;;; want to be reimplementing all the window flavors in the system, I'm
;;; maintaining a data structure that maps from windows to ICON objects.

;;;  These are the messages that are sent to windows if they handle them
;;;
;;; :SHRINK-TO-ICON	-- The window does everything
;;; :ICON-FOR-WINDOW	-- The window gives us its icon (may create it implicitly)
;;; :REMOVE-ICON	-- The icon should no longer be associated with the window
;;; :MAKE-ICON		-- Create an icon (if don't handle :ICON-FOR-WINDOW)

(require 'utilities)		      ;For object to object map functions

(export
  '(*window-to-icon-map* shrink-window *transforming-window-wait-time*
    icon?)
  )

(defvar *window-to-icon-map* (make-map)
   "Mapping from windows to ICON objects.  Currently an ALIST.")

;;; Primary functional interfaces

(defun shrink-window (window)
  "   `Shrink' WINDOW (ie. replace it with an iconic representation).
   If WINDOW has a :SHRINK-TO-ICON method, use it, else use
   SHRINK-RANDOM-WINDOW."
  (if (send window :operation-handled-p :shrink-to-icon)
      (send window :shrink-to-icon)
      (shrink-random-window window)
      )
  )

;(defun shrink-random-window (window)
;  "Shrink WINDOW.  Use ICON-FOR-WINDOW to get the icon."
;  (let ((icon (icon-for-window window)))
;    (when icon
;      (send window :deactivate)
;      (draw-window-transformation window icon)
;      (send icon :expose)
;      )
;    )
;  )

(DEFUN shrink-random-window (window)
  "Shrink WINDOW.  Use ICON-FOR-WINDOW to get the icon."
  ;;; Rewriten by JPR to use find-right-sheet.  There were cases where the
  ;;;  previous recursing strategy didn't work, e.g. for CYC desktops.
  (let ((window (w:find-right-sheet window)))
       (LET ((icon (w:icon-for-window window)))
	    (WHEN icon
	      (send window :deactivate)
	      (SEND window :bury)
	      (w:draw-window-transformation window icon)
	      (SEND window :deexpose :default :clean) ;deexpose and wipe things.
	      (SEND icon :expose)
	      ;; jwz:
	      (tv:add-to-previously-selected-windows window t)
	    )
       )
  )
)

(defun icon-for-window (window)
  "   Return an ICON object for representing WINDOW.  Calls
   LOOK-FOR-ICON to find any pre-existing icon.  If there is no
   icon, calls :MAKE-ICON method if there is one, else uses
   MAKE-LABEL-ICON."
  (let ((icon (look-for-icon window)))
    ;; Make a new one if there wasn't one around.
    (unless icon
      (setq icon
	    (if (send window :operation-handled-p :make-icon)
		(send window :make-icon)
		(make-label-icon window)))
      (map-set *window-to-icon-map* window icon))
    icon
    )
  )

(defun look-for-icon (window)
  "   Return an icon object that represents WINDOW if there is one.
   Calls :ICON-FOR-WINDOW if WINDOW handles it, else uses the map."
  (if (send window :operation-handled-p :icon-for-window)
      (values (send window :icon-for-window))
      (values (map-lookup *window-to-icon-map* window))
      )
  )

;;;Edited by Acuff                 30 Nov 87  19:15
;;;Edited by Acuff                 4 Dec 87  19:31
;;;Edited by Mueller               23 Jan 91 18:05
(defun make-label-icon (window &optional (icon-flavor 'text-icon))
  "   Make a text icon for WINDOW based on WINDOW's label.  If there is
   no label, use the name.  TEXT-ICON-FLAVOR, if specified, should be a
   symbol nameing a flavor which includes W:TEXT-ICON."
  (let ((string (if (send window :operation-handled-p :label)
		    (if (stringp (third (send window :label)))
			(third (send window :label))
			;;; Fix put in as a result of Clint Hyde bug report.
			;;; The string is usually the sixth of the label spec.
			;;; I don't know under what circumstances it is the
			;;; third, but I left the references to Third in just
			;;; in case.
			(if (stringp (sixth (send window :label)))
			    (sixth (send window :label))
			    nil))
		    nil))
	(icon (make-instance icon-flavor ;FritzM. 1/23/91
			     :window-being-represented window
			     :superior (send window :superior)
			     )
	      )
	)
    (unless (typep string 'string)
      (setq string (send window :name)))
    (send icon :set-text string)
    ;; Start with the icon at the same place as the window
    (multiple-value-bind (x y)
	(send window :position)
      (when (send icon :set-position x y :verify)
	(send icon :center-around x y)))
    (push icon *windows-to-leave-exposed*)
    icon
    )
  )

;;; Icon flavors

;;; JWZ
(defvar *text-icon-default-font* :default "The default font for text-icons.")
(defvar *text-icon-default-border-width* 2 "The default border size for text icons.")

;;; jwz: added initialization of borders and font.
;;;
(defflavor text-icon
	   (string)
	   (icon-window-mixin borders-mixin stream-mixin minimum-window)
  :inittable-instance-variables
  (:default-init-plist :blinker-p nil
		       :deexposed-typeout-action :expose
		       :borders *text-icon-default-border-width*
		       :font-map (vector *text-icon-default-font*)
		       )
  (:documentation "A simple text icon")
  )

(defmethod (Text-Icon :After :Init) (plist)
  (let ((text (getf (first plist) :Text)))
       (if text (send self :Set-Text Text))
  )
)

(defmethod (text-icon :set-text) (new-string)
  "Setup SELF to show NEW-STRING."
  (check-arg new-string 'stringp "a string")
  (setq string new-string)
  (let ((was-active? (send self :active-p)))
    (send self :deactivate)
    (multiple-value-bind (sup-width sup-height)
	(send (send self :superior) :size)
      (send self :set-size sup-width sup-height)
      (multiple-value-bind (ignore y ignore max-x)
	  (send self :compute-motion string 0 nil 0 0 t)
	(send self :set-size (+ max-x 8) (+ y 8))
	(when was-active? (send self :activate))
	(when (send self :exposed-p)
	  (send self :refresh)
	  )
	)
      )
    )
  )

(defmethod (text-icon :after :refresh) (&rest ignore)
  "Redraw our string after refresh."
  (when (boundp-in-instance self 'string)
    (send self :string-out-explicit string 4 4
	  nil nil
	  (send self :current-font)
	  alu-seta
	  )
    )
  )

;;; From JWZ.
;;; Giving icons a control menu for changing their name, font, etc.
(defmethod (w:text-icon :parameter-menu) ()
  "Pop up a CVV menu letting the user change the name and font of this icon."
  (let* ((icon-name w:string)
	 (icon-font w:current-font)
	 (icon-bw (send self :border-margin-width)))
    (declare (special icon-name icon-font icon-bw))
    (w:choose-variable-values '((icon-name "Name" :string)
				(icon-font "Font" :font)
				(icon-bw "Margin Width" :fixnum))
			      :label "Icon Parameters")
    (unless (position (setq icon-font (tv:font-evaluate icon-font)) w:font-map :test #'eq)
      (send self :set-font-map (vector icon-font)))
    (send self :set-current-font icon-font)
    (send self :set-text icon-name)
    (send self :set-border-margin-width icon-bw))
  nil)

;;;*****************************************************************************
;;;*****************************************************************************

;;;New flavor, allowing pictorial icons. CRH 10/89. drawn from some earlier
;;;work. Mostly pinched from above defflavor.

(defflavor graphics-icon
	   ((picture nil)	   ;the actual picture. a bitmap array.
	    (size-of-picture nil)  ;Size not picture.  May be < size of bitmap
	   )
	   (icon-window-mixin
	    borders-mixin
	    label-mixin
	    stream-mixin
	    minimum-window
	   )
  :Inittable-Instance-Variables
  (:Gettable-Instance-Variables picture size-of-picture)
  (:Settable-Instance-Variables picture size-of-picture)
  (:special-instance-variables picture)
  (:default-init-plist :blinker-p nil
                       :Label nil
    		       :deexposed-typeout-action :Expose
		       :Save-Bits t
		       :borders 2)
  (:documentation "A simple Graphics Icon.
Picture can be a bit map, in which case this is used, a pathname for a file
saved using w:write-bit-array-file, in which case the file is read and that
bitmap is used, or a list of the form:
  (<pathname> <symbol>)

where <pathname> is a pathname for a xld file from which to load the bitmap
and <symbol> is the symbol that names the bitmap once the file has been loaded.
")
)

;;; After init added by JPR on 11/13/89 16:55:13.
(defmethod (graphics-icon :After :Init) (init-plist)
  (send self :Setup (first init-plist))
)

(defmethod (Graphics-Icon :After :Set-Picture) (to)
  (ignore to)
  (send self :Setup nil)
)

(defmethod (Graphics-Icon :After :Set-Size-Of-Picture) (to)
  (ignore to)
  (send self :Setup nil)
)

(defmethod (Graphics-Icon :Setup) (init-plist)
  (typecase picture
    ((or string pathname)
     (setq picture (w:read-bit-array-file picture))
    )
    (cons ;; Must be a xld file spec.
     (si:load-if (first picture))
     (setq picture (symbol-value (second picture)))
    )
    (otherwise nil)
  )
  (if (not size-of-picture)
      (setq size-of-picture (reverse (array-dimensions picture)))
  )
  (if (not (member :Size init-plist))
      (lexpr-send self :set-inside-size size-of-picture)
  )
)


(defmethod (Graphics-Icon :refresh-bitmap) ()
  "Draws the designated picture onto the icon-window."
  (send self :Clear-Window)
  ;;; Use tv:alu-add just in case we're colour.
  (bitblt tv:alu-add (first size-of-picture) (second size-of-picture)
	  picture 0 0
	  screen-array
	  (send self :left-margin-size) (send self :Top-Margin-Size)
  )
)

(defmethod (graphics-icon :after :expose) (&rest ignore)
  (send self :Refresh-Bitmap)
)

(defmethod (graphics-icon :after :refresh) (&rest ignore)
  (send self :Refresh-Bitmap)
)

;-------------------------------------------------------------------------------

;;; The primary mixin

;;; tv:show-partially-visible-mixin added by JPR on a suggestion from JWZ.
;;;Edited by Mueller               23 Jan 91 18:05
(defflavor icon-window-mixin
	   (window-being-represented)
	   (tv:show-partially-visible-mixin) ;FritzM. 1/23/91
  :inittable-instance-variables
  :gettable-instance-variables
  :settable-instance-variables
  )

;;;Edited by Acuff                 20 Nov 87  16:20
(defmethod (icon-window-mixin :after :init) (&rest ignore)
  "Error checking on the window this icon is to represent."
  (unless (and (typep window-being-represented 'sheet)
	       (not (icon? window-being-represented)))
    (error
      "Can't make an ICON window for ~A since it's not a non-Icon window."
      window-being-represented
      )
    )
  )

(defmethod (icon-window-mixin :expand-from-icon) ()
  "Deactivate this icon, activating the window we represent.  Select it
   if it's selectable."
  (send self :deactivate)
  (when window-being-represented
    (draw-window-transformation self window-being-represented)
    (send window-being-represented :expose)
    (send window-being-represented :send-if-handles :mouse-select)
    )
  )

(defmethod (icon-window-mixin :deexposed-mouse-buttons) (mask x y)
  "R-2 => sys menu, else :DEEXPOSED-MOUSE-CLICK method."
  (let ((buttons (mouse-character-button-encode mask)))
    (if (= buttons #\mouse-r-2)
        (mouse-call-system-menu)
        ;;else
        (send self :deexposed-mouse-click buttons x y)))
  )

(defmethod (icon-window-mixin :deexposed-mouse-click) (button x y)
  "L1: Expose, L2: Expand, M: Move/Drag, R: System Menu"
  (case button
    (#\mouse-L-1 (send self :expose))
    (#\mouse-L-2 (process-run-function "Expand" self :expand-from-icon))
    ((#\mouse-M-1 #\mouse-M-2)
     (if (and (boundp '*dragging-enabled?*)
	      *dragging-enabled?*
	      (char= #\mouse-m-1 button))
	 (drag-window-from self x y)
	 (process-run-function
	   '(:name "Set Window Position" :priority 1)
	   #'(lambda (window) (mouse-set-window-position window))
	   self)
	 )
     )
    ((#\mouse-R-1 #\mouse-R-2) (mouse-call-system-menu))
    )
  )

;;; jwz: modified to invoke the icon menu on mouse-R, and the system menu on mouse-R-2.
;;;
(defmethod (icon-window-mixin :mouse-click) (button x y)
  "L: Expand, M: Drag/Move, R: Icon Menu, R2: System Menu"
  (case button
    ((#\mouse-L-1 #\mouse-L-2)
     (process-run-function "Expand" self :expand-from-icon))
    ((#\mouse-M-1 #\mouse-M-2)
     (if (and (boundp '*dragging-enabled?*)
	      *dragging-enabled?*
	      (char= #\mouse-m-1 button))
	 (drag-window-from self x y)
	 (process-run-function
	   '(:name "Set Window Position" :priority 1)
	   #'(lambda (window) (mouse-set-window-position window))
	   self)))
    (#\mouse-R-1 (process-run-function "Icon Menu" self :icon-menu))
    (#\mouse-R-2 (mouse-call-system-menu))))


(defmethod (icon-window-mixin :who-line-documentation-string) ()
  `(:mouse-L-1 "Expand to Window"
    ,@(if (and (boundp '*dragging-enabled?*)
	       *dragging-enabled?*)
	  '(:mouse-M-hold "Drag this icon")
	  '(:mouse-M-1 "Move This Icon")
	  )
    :mouse-R-1 "Icon Menu"  ;;JWZ
    :mouse-R-2 "System Menu")
  )

(defmethod (icon-window-mixin :deexposed-who-line-documentation-string) ()
  `(:mouse-L-1 "Expose This Icon"
    :mouse-L-2 "Expand to Window"
    ,@(if (and (boundp '*dragging-enabled?*)
	       *dragging-enabled?*)
	  '(:mouse-M-hold "Drag this icon")
	  '(:mouse-M-1 "Move This Icon")
	  )
    :mouse-R-1 "System Menu")
  )

;;;Edited by Acuff                 4 Dec 87  19:31
(defmethod (icon-window-mixin :before :kill) (&rest ignore)
  "Disassociate self from other windows."
  (when window-being-represented
    (send window-being-represented :send-if-handles :remove-icon self)
    (map-remove *window-to-icon-map* window-being-represented)
    (setq *windows-to-leave-exposed* (delete self *windows-to-leave-exposed*))
    (setq window-being-represented nil)
    )
  )

;;; Support utility functions

(defmethod (sheet :before :expose) (&rest ignore)
  "If the window being exposed has an icon up, take it down."
  (let ((icon (look-for-icon self)))
    (when (and icon (send icon :active-p))
      (send icon :deactivate)
      (draw-window-transformation icon self)
      )
    )
  )

;;; From JWZ.  Varified by JPR.
(defvar *icon-menu-items*
 '(("Select" :eval (send window :select)
    :documentation "Select the window represented by this icon.")
   ("Expose" :eval (send window :expose)
    :documentation "Expose the window represented by this icon.")
   ("Kill" :eval (progn (send window :kill) (send icon :kill))
    :documentation "Kill the window represented by this icon (and this icon).")
   ("Edit Icon" :eval (send icon :send-if-handles :parameter-menu)
    :documentation "Change the name and/or font of this icon.")
   ("Inspect Window" :eval (inspect window) :documentation
    "Invoke the inspector on the window represented by this icon.")
   ("Inspect Icon" :eval (inspect icon)
    :documentation "Invoke the inspector on this icon."))
)

(defmethod (w:icon-window-mixin :icon-menu) ()
  "Pop up a menu of operations on this icon."
  (let* ((icon self)
	 (window (send icon :window-being-represented)))
    (declare (special icon window))
    (w:menu-choose *icon-menu-items* :label "Icon Menu")))

;-------------------------------------------------------------------------------

(defun icon? (thing)
  "Non-NIL if THING is an ICON for some window."
  (typep thing 'icon-window-mixin)
  )

(defvar *transforming-window-wait-time* .2
  "How many seconds to wait with window transform lines up.")

(defun draw-window-transformation (from-window to-window)
  "Draw `zap' lines indicating FROM-WINDOW is being transformed somehow
   to TO-WINDOW.  Windows must have same superior, else nothing is done."
  ;; Vars are {To | From}{Left | Right | Top | Bottom}
  (multiple-value-bind (fl ft)
      (send from-window :position)
    (multiple-value-bind (tl tt)
	(send to-window :position)
      (let ((fr (+ fl (send from-window :width)))
	    (fb (+ ft (send from-window :height)))
	    (tr (+ tl (send to-window :width)))
	    (tb (+ tt (send to-window :height)))
	    (superior (send from-window :superior))
	    )
	(when (eq superior (send to-window :superior))
	  ;; I don't like using %DRAW-LINE, but it seems to be the only way...
	  (lock-sheet (superior)
	    (prepare-sheet (superior)
	      (dotimes (i 2)
		(tv:%draw-line fl ft tl tt alu-xor t superior)
		(tv:%draw-line fr ft tr tt alu-xor t superior)
		(tv:%draw-line fl fb tl tb alu-xor t superior)
		(tv:%draw-line fr fb tr tb alu-xor t superior)
		(when (= i 0) (sleep *transforming-window-wait-time*))
		)
	      )
	    )
	  )
	)
      )
    )
  )

;;; Support for background windows

(require 'backgrounds "TOOLS:TOOLS;WINDOW-SYSTEM-ADDITIONS")

(defun shrink-all-windows-over-background ()
  (when *background*
    (let ((screen (sheet-get-screen *background*)))
      (loop for inf in (send screen :inferiors)
	    unless (or (icon? inf)
		       (member inf *windows-to-leave-exposed*)
		       (eq inf *background*))
	    do (shrink-window inf)
	    until (eq inf *background*)
	    )
      )
    )
  )

(defun shrink-up-windows (&optional arg)
  (case arg
    (nil (shrink-all-windows-over-background))
    (1 (clean-up-windows 1))
    (2 (let ((sw selected-window))
	 (shrink-all-windows-over-background)
	 (send sw :mouse-select)))
    (3 (clean-up-windows nil))
    (otherwise (shrink-all-windows-over-background))
    )
  )

(add-terminal-key #\z 'shrink-up-windows
   ''("Shrink all windows not on W:*WINDOWS-TO-LEAVE-EXPOSED*,"
      "   1 arg means bury instead,"
      "   2 arg means shrink all but the currently selected window,"
      "   3 arg means bury all but the currently selected window.")
   )

;;; Interface to the window manager system menu

(require 'window-manager-system-menu)

(pushnew `("Shrink" :window-op shrink-window-op
	   :documentation "Shrink this window to an Icon")
	 *window-manager-system-menu-items*
	 :test #'(lambda (x y) (string-equal (car x) (car y))))

(clear-resource 'window-manager-system-menu)

(defun shrink-window-op (window m-x m-y)
  (declare (ignore m-x m-y))
  (shrink-window (find-right-sheet window))
  )

(unless (assoc "Shrink Windows" *background-menu-items* :test #'string-equal)
  (setq *background-menu-items*
	(nconc *background-menu-items* 
	       '(("Shrink Windows" :eval (w:shrink-up-windows)
		  :documentation
		  "Shrink all windows not on USER:*WINDOWS-TO-LEAVE-EXPOSED*."
		  ))
	       )
	)
  )

;-------------------------------------------------------------------------------

;;; The following from JWZ.

(sys:advise W:DESELECT-AND-MAYBE-BURY-WINDOW :after :iconify nil
  "When there is a background window around, iconify the window as well as burying it."
  ;; This advice is redefined by "background-image-on-screen" which nukes w:*background*.
  (when (and (boundp 'w:*background*) w:*background*)
    (w:shrink-window (first sys:arglist))))


;-------------------------------------------------------------------------------

(provide 'window-icons)
