;;; -*- Mode:Common-Lisp; Package:SYS; Base:10 -*-

(defmacro untrace (&rest fns)
  "Untrace one or more functions.  With no arg, untrace all traced functions.
If you supply :Query as one of the args then you are queried about whether to
untrace the function."
  (let ((query? (member :query fns)))
       (setq fns (remove :query fns))
      `(loop for spec in ,(if fns `',fns '(trace))
	     when (or ,(not query?) (y-or-n-p "Untrace: ~A" spec))
	     collect (untrace-1 spec ,(not (null fns)))
       )
  )
)

(defun sys:trace-print
       (depth direction function print-args-flag extras-1 extras-2)
  (declare (special arglist values))
  (with-standard-io-environment
  (terpri *trace-output*)
  (do ((n (* 2 trace-level) (1- n)))
      ((not (> n 2))
       nil)
    (write-char #\SPACE *trace-output*))
  (princ "(" *trace-output*)
  (prin1 depth *trace-output*)
  (princ " "  *trace-output*)
  (princ direction *trace-output* )
  (princ " "  *trace-output*)
  (prin1 function *trace-output*)
  (if (and (consp function) (equal :method (first function)))
      (progn (princ " [Self = "  *trace-output*)
	     (prin1 self *trace-output*)
	     (princ "]"  *trace-output*)
      )
      nil
  )
  (let ((stuff (if (eq direction 'enter)
		   arglist
		   values)))
    (when (and stuff print-args-flag)
      (princ " :-" *trace-output*)
      (do ((tail stuff (cdr tail)))
	  ((atom tail)
	   (when tail
	     (princ " . " *trace-output*)
	     (prin1 tail *trace-output*)))
	(write-char #\SPACE *trace-output*)
	(prin1 (car tail) *trace-output*))))
  (when extras-1
    (princ "  \\\\" *trace-output*)
    (dolist (e extras-1)
      (princ " " *trace-output*)
      (prin1 (eval e) *trace-output*)))
  (when extras-2
    (princ "  //" *trace-output*)
    (dolist (e extras-2)
      (princ " " *trace-output*)
      (prin1 (eval e) *trace-output*)))
  (princ ")" *trace-output*)
  )
)

(defmacro trace ( &rest specs)
  "Trace one or more functions.  The default tracing action is to print
the function's name and arguments when it is called and print its name
and value(s) when it returns.
Each trace spec can take any of the following forms:
A symbol which is a function name, with no options;
A list (:FUNCTION <function-spec> <option-1> <option-2> ...); or
A list ((<function-1> <function-2> ...) <option-1> <option-2> ...)
The following are valid trace options:
:BREAK <pred> (or :EXITBREAK <pred>) causes a breakpoint to be entered
before (or after) the function is executed if <pred> is NON-NIL.
:ERROR causes the debugger to be invoked before the function is executed. 
:STEP causes the function to be single-stepped whenever it is called.
:ENTRYCOND <pred> (or :EXITCOND <pred>) causes trace information to
be printed on function entry (or exit) only if <pred> is NON-NIL.
:COND <pred> specifies both :ENTRYCOND and :EXITCOND together.
:WHEREIN <function> causes the function to be traced only when called 
from <function>.
:ARGPDL <pdl> specifies a symbol <pdl> which will contain the recent history
of the function.  <pdl> should be examined from within a breakpoint.
:ENTRYPRINT <form> (or :EXITPRINT <form>) evaluates <form> and prints the value
before (or after) the function is executed.
:PRINT <form> specifies both :ENTRYPRINT and :EXITPRINT together.
:ENTRY <list> (or :EXIT <list>) specifies a list of forms whose values are
printed on function entry (or exit).
:ARG prints the function's name and arguments on function entry.
:VALUE prints the function's returned value(s)  on function exit.
:BOTH  specifies both :ARG and :VALUE together. This is the default"
  (cond
    ((null specs) `traced-functions)
    (t `(mapcan 'trace-1 ',specs))))


(defun trace-action-for-maybe-setf-method
       (spec fcn clos-method-prompt flavor-method-prompt short-prompt
	function-to-call continuation functionify
       )
  (ignore spec)
  (let ((set-name `(setf ,(function-name fcn))))
       (if (and fcn
		(symbolp (function-name fcn))
		(fdefinition-safe set-name)
		(tv:generic-function-p-safe
		  (fdefinition-safe set-name)
		)
		(y-or-n-p "~&Trace (setf ~S)?" (function-name fcn))
	   )
	   (trace-action-for-maybe-method
	     set-name set-name clos-method-prompt flavor-method-prompt
	     short-prompt function-to-call continuation functionify
	   )
	   nil
       )
  )
)

(defun trace-action-for-maybe-method
       (spec fcn clos-method-prompt flavor-method-prompt short-prompt
	function-to-call continuation functionify
       )
  (let ((result (trace-action-for-maybe-method-internal
		  spec fcn clos-method-prompt flavor-method-prompt
		  short-prompt function-to-call continuation functionify
		)
	)
       )
       (trace-action-for-maybe-setf-method
	 spec fcn clos-method-prompt flavor-method-prompt short-prompt
	 function-to-call continuation functionify
       )
       result
  )
)

(defun trace-action-for-maybe-method-internal
       (spec fcn clos-method-prompt flavor-method-prompt short-prompt
	function-to-call continuation functionify
       )
  (let ((choice nil))
    (cond ((and fcn (tv:generic-function-p-safe
		      (sys:fdefinition-safe (function-name fcn)))
		(tv:generic-function-methods-safe
		  (sys:fdefinition-safe (function-name fcn)))
		(setq choice (fquery '(:Type :Tyi :Choices
					     (((t "Yes") #\Y #\space #\y)
					      ((:Selective "Selective") #\S #\s)
					      ((nil "No") #\N #\n)))
				     clos-method-prompt
				     (function-name fcn))))
	   (loop for meth in (tv:generic-function-methods-safe
			       (sys:fdefinition-safe (function-name fcn)))
		 for name = (function-name (tv:method-function-safe meth))
		 for new-spec = (if functionify
				    (let ((spec (copy-tree spec)))
				      (cond ((atom spec) `(:Function ,name))
					    ((eq (car spec) :function)
					     (setf (second spec) name))
					    ((atom (car spec))
					     (setf (first spec)
						   `(:Function ,name)))
					    (t (ferror
						 nil "Illegal trace????"))))
				    name)
		 when (or (eq t choice) (y-or-n-p short-prompt name))
		 do (funcall function-to-call new-spec)))
	  ((and (keywordp fcn)
		(zwei:list-methods-internal fcn)
		(setq choice (fquery '(:Type :Tyi :Choices
					     (((t "Yes") #\Y #\space #\y)
					      ((:Selective "Selective") #\S #\s)
					      ((nil "No") #\N #\n)))
				     flavor-method-prompt
				     fcn)))
	   (loop for meth in (zwei:list-methods-internal fcn)
		 for name = (function-name (third meth))
		 for new-spec = (if functionify
				    (let ((spec (copy-tree spec)))
				      (cond ((atom spec) `(:Function ,name))
					    ((eq (car spec) :function)
					     (setf (second spec) name))
					    ((atom (car spec))
					     (setf (first spec)
						   `(:Function ,name)))
					    (t (ferror
						 nil "Illegal trace????"))))
				    name)
		 when (or (eq t choice) (y-or-n-p short-prompt name))
		 do (funcall function-to-call new-spec)))
	  (t (funcall continuation))))
)

(advise trace-1 :around :Check-For-Generic-Functions nil
  (let ((fcn nil)
	(spec (first arglist)))
    (cond
      ((atom spec) (setq fcn spec))
      (t
       (cond
	 ((eq (car spec) :function) (setq fcn (cadr spec)
					  spec (cdr spec)))
	 ((atom (car spec)) (setq fcn (car spec)))
	 ((validate-function-spec spec) (SETQ fcn spec spec (LIST spec)))
	 ((validate-function-spec (CAR SPEC)) (SETQ FCN (CAR SPEC)))
	 (t (return
	      (loop for fcn in (car spec)
		    nconc (sys:trace-1 `(:function ,fcn ,@(cdr spec)))))))))
    (trace-action-for-maybe-method
      spec fcn "~&Trace methods of ~S separately? "
      "~&Trace methods named by ~S separately? " "~&Trace ~S"
      'sys:trace-1 #'(lambda () :Do-It) t)))

(advise untrace-1 :around :Check-For-Generic-Functions nil
  (let ((fcn nil)
	(spec (first arglist)))
    (cond
      ((atom spec) (setq fcn spec))
      (t
       (cond
	 ((eq (car spec) :function) (setq fcn (cadr spec)
					  spec (cdr spec)))
	 ((atom (car spec)) (setq fcn (car spec)))
	 ((validate-function-spec spec) (SETQ fcn spec spec (LIST spec)))
	 ((validate-function-spec (CAR SPEC)) (SETQ FCN (CAR SPEC)))
	 (t (return
	      (loop for fcn in (car spec)
		    nconc (sys:trace-1 `(:function ,fcn ,@(cdr spec)))))))))
    (trace-action-for-maybe-method
      spec fcn "~&UnTrace methods of ~S separately? "
      "~&UnTrace methods named by ~S separately? " "~&UnTrace ~S"
      'sys:untrace-1 #'(lambda () :Do-It) nil)))

;-------------------------------------------------------------------------------

zwei:
(DEFCOM COM-TRACE "Trace or untrace a function.
Reads the name of the function from the mini-buffer (the top of the kill
ring has the \"current\" function from the buffer) then pops up a menu
of trace options.  With an argument, omits menu step" ()
  (LET ((FCN (READ-FUNCTION-NAME "Trace" (RELEVANT-FUNCTION-NAME (POINT) NIL T T T) nil))) ;;; JPR.  Changed to allow keyword here.
    (COND (*NUMERIC-ARG-P*
	   (SI:EVAL-ABORT-TRIVIAL-ERRORS
	     (IF (ATOM FCN)
		 `(TRACE (,FCN))
		 `(TRACE (:FUNCTION ,FCN)))))
	  ((FBOUNDP 'TV:TRACE-VIA-MENUS)
	   (TV:TRACE-VIA-MENUS FCN))
	  (T
	   (FORMAT T "~A will be traced. However, since trace menu support
is not currently loaded, using any of the trace options will
require executing the TRACE function from a Listener. To stop
the trace, execute the UNTRACE function from a Listener." FCN)
	   (SI:EVAL-ABORT-TRIVIAL-ERRORS (IF (ATOM FCN)
					     `(TRACE (,FCN))
					     `(TRACE (:FUNCTION ,FCN)))))))
  DIS-NONE)


;-------------------------------------------------------------------------------


tv:
(DEFUN tv:TRACE-VIA-MENUS (&OPTIONAL FCN)
  "This function is invoked in the momentary menu process when the
user clicks TRACE and in the editor process by the editor's Trace
command.  If the function isn't supplied as an argument the user is
asked for it."
  (USING-RESOURCE (TRACE-POP-UP-WINDOW POP-UP-TEXT-WINDOW)
     (USING-RESOURCE (TRACE-POP-UP-MENU TRACE-POP-UP-MENU)
	(FUNCALL TRACE-POP-UP-WINDOW :SET-LABEL "Trace")
	(FUNCALL TRACE-POP-UP-WINDOW :SET-SIZE
		 (min 1000			;Accomadate small mac screens  PMH 3/11
		      (tv:sheet-width (tv:sheet-superior trace-pop-up-window))) 
		 (min 300			;Accomadate small mac screens  PMH 3/11
		      (floor (tv:sheet-height (tv:sheet-superior trace-pop-up-window)) 2)))
	(FUNCALL TRACE-POP-UP-WINDOW :CENTER-AROUND MOUSE-X MOUSE-Y)
	(WINDOW-CALL (TRACE-POP-UP-WINDOW :DEACTIVATE)
	 (UNWIND-PROTECT (LET
			  ((BLINKER (CAR (SHEET-BLINKER-LIST TRACE-POP-UP-WINDOW)))
			   (*terminal-io* TRACE-POP-UP-WINDOW)
			   (*standard-output* TRACE-POP-UP-WINDOW)
			   (*standard-input* TRACE-POP-UP-WINDOW)
			   (*query-io* TRACE-POP-UP-WINDOW))
			  (COND
			    ((NULL FCN)
			     ;; Make sure blinker is blinking.
			     (BLINKER-SET-VISIBILITY BLINKER :BLINK)
			     (FORMAT TRACE-POP-UP-WINDOW
				     "Type in name of function to be traced or untraced.
  Abort quits.~%")
			     (DO ((*TERMINAL-IO* TRACE-POP-UP-WINDOW)
				  (*STANDARD-INPUT* TRACE-POP-UP-WINDOW)
				  (*STANDARD-OUTPUT* TRACE-POP-UP-WINDOW))
				 (NIL)
			       (SETQ FCN (READ))
                               (setq fcn (dwimify-arg-package fcn 'function-spec))     ;!
			       (IF (FDEFINEDP FCN)
				 (RETURN ())
				 (FORMAT T " ;not a defined function, try again~%")))))
			  (FUNCALL TRACE-POP-UP-MENU :MOVE-NEAR-WINDOW TRACE-POP-UP-WINDOW)
			  (DO ((FORM (IF (ATOM FCN)
				       `(TRACE (,FCN))
				       `(TRACE (:FUNCTION ,FCN))))
			       (CHOICE)
			       (OPTION)
			       (ARG))
			      (NIL)
			       ;; Put the current status on the text window.
			    (FUNCALL TRACE-POP-UP-WINDOW :CLEAR-SCREEN)
			    ;; 76 is width in characters.
			    (GRIND-TOP-LEVEL FORM 76 TRACE-POP-UP-WINDOW)
			    ;; Not listening to the keyboard any more, shut off blinker.
			    (BLINKER-SET-VISIBILITY BLINKER NIL)
			    ;; Get input from the menu.
			    (SETQ CHOICE (FUNCALL TRACE-POP-UP-MENU :CHOOSE)
				  OPTION (FIRST CHOICE))
			    (send TRACE-POP-UP-WINDOW :Send-If-Handles :Clear-Input)
			    (send TRACE-POP-UP-MENU :Send-If-Handles :Clear-Input)
			    (COND
			      ((NULL CHOICE))            ;Try again if outside menu
			      ((EQ OPTION 'UNTRACE) (EVAL `(UNTRACE ,FCN)) (RETURN ()))
			      ((EQ OPTION 'QUIT) (RETURN ()))
			      ((EQ OPTION 'DO-IT) (EVAL FORM) (RETURN ()))
			      (T
			       ;; let's make it smarter so that we can undo choices.
			       (LET ((PREVIOUS-ENTRY-POSITION
				      (POSITION (SECOND CHOICE) (THE LIST (SECOND FORM)) :TEST
						#'EQ)))
				 (COND
				   ((AND PREVIOUS-ENTRY-POSITION
				       (OR (EQ (SECOND CHOICE) :BREAK)
					  (EQ (SECOND CHOICE) :EXITBREAK))
				       (CDDR CHOICE))
				    ;; Then we should either remove it from
				    ;; the list if it is the same or just
				    ;; change the argument.
				    (IF (EQ (THIRD CHOICE)
					 (NTH (1+ PREVIOUS-ENTRY-POSITION) (SECOND FORM)))
				     ;; Remove the choice
				      (IF (EQ PREVIOUS-ENTRY-POSITION 0)
					(SETF (SECOND FORM) (CDDR (SECOND FORM)))
					;;ELSE
					(SETF (NTHCDR PREVIOUS-ENTRY-POSITION (SECOND FORM))
					      (NTHCDR (+ 2 PREVIOUS-ENTRY-POSITION)
						      (SECOND FORM))))
				      ;;ELSE
				      ;; Fix the choice
				      (SETF (NTH (1+ PREVIOUS-ENTRY-POSITION) (SECOND FORM))
					    (THIRD CHOICE))))
				   ((AND PREVIOUS-ENTRY-POSITION (NULL OPTION))
				    ;; Delete the choice.
				    (IF (EQ PREVIOUS-ENTRY-POSITION 0)
				      (SETF (SECOND FORM) (CDR (SECOND FORM)))
				      (SETF (NTHCDR PREVIOUS-ENTRY-POSITION (SECOND FORM))
					    (NTHCDR (1+ PREVIOUS-ENTRY-POSITION) (SECOND FORM)))))
				   ((NULL OPTION)
				    ;; Just add the object.
				    (SETF (SECOND FORM) (APPEND (SECOND FORM) (CDR CHOICE))))
				   (T
				    ;; Needs an arg, get it.
				    (FORMAT TRACE-POP-UP-WINDOW "~2%~A:~%" OPTION)
				    ;; Turn on blinker.
				    (BLINKER-SET-VISIBILITY BLINKER :BLINK)
				    (LET ((*TERMINAL-IO* TRACE-POP-UP-WINDOW)
					  (*STANDARD-INPUT* TRACE-POP-UP-WINDOW)
					  (*STANDARD-OUTPUT* TRACE-POP-UP-WINDOW)
					  (FLAG))
				      (MULTIPLE-VALUE-SETQ (ARG FLAG)
					(FUNCALL *TERMINAL-IO* :RUBOUT-HANDLER
						 '((:FULL-RUBOUT :FULL-RUBOUT))
						 #'READ-FOR-TOP-LEVEL))
				      (UNLESS FLAG
				       ;; If previous entry replace the argument
				       ;; otherwise add on new argument.
					(IF (NULL PREVIOUS-ENTRY-POSITION)
					  (SETF (SECOND FORM)
						(APPEND (SECOND FORM) (CDR CHOICE) (CONS ARG ())))
					  (SETF (NTH (1+ PREVIOUS-ENTRY-POSITION) (SECOND FORM))
						ARG)))))))))))
	   (FUNCALL TRACE-POP-UP-MENU :DEACTIVATE))))))