;;; -*- Mode:Common-Lisp; Package:ZWEI; Fonts:CPTFONT,CPTFONTB; Base:10 -*-

;;;
;;; Copyright (c) 1987 Douglas Johnson.  All rights reserved. 
;;; 
; Copyright (C) 1980, Massachusetts Institute of Technology
; Copyright (C) 1984, Texas Instruments Incorporated. All rights reserved.


;***********************************************************************************
;
;        This file contains the ZWEI interface into the spelling 
;        facilities.
;
;***********************************************************************************



;  This table defines the syntax of a word for spelling correction.
;  A word is a-z, A-Z, and  single quote, delimited by anything else."


(DEFPARAMETER *SPELLING-WORD-SYNTAX-TABLE*
   (MAKE-SYNTAX-TABLE
    '((32 WORD-DELIMITER) WORD-DELIMITER;040 space
      WORD-DELIMITER;041 !
      WORD-DELIMITER;042 "
      WORD-DELIMITER;043 #
      WORD-DELIMITER;044 $
      WORD-DELIMITER;045 %
      WORD-DELIMITER;046 &
      WORD-ALPHABETIC;047 '
      WORD-DELIMITER;050 (
      WORD-DELIMITER;051 )
      WORD-DELIMITER;052 *
      WORD-DELIMITER;053 +
      WORD-DELIMITER;054 ,
      WORD-DELIMITER;055 -
      WORD-DELIMITER;056 .
      WORD-DELIMITER;057 /
      (10 WORD-DELIMITER);Digits
      WORD-DELIMITER;072 :
      WORD-DELIMITER;073 ;
      WORD-DELIMITER;074 <
      WORD-DELIMITER;075 =
      WORD-DELIMITER;076 >
      WORD-DELIMITER;077 ?
      WORD-DELIMITER;100 @
      (26 WORD-ALPHABETIC);Upper case letters
      WORD-DELIMITER;133 [
      WORD-DELIMITER;134 \
      WORD-DELIMITER;135 ]
      WORD-DELIMITER;136 ^
      WORD-DELIMITER;137 _
      WORD-DELIMITER;140 `
      (26 WORD-ALPHABETIC);Lower case letters
      (133 WORD-DELIMITER)))
   "This defines the syntax of a word for purposes of the spelling correction.") 


(DEFSUBST CURRENT-WORD (BP)
  "Return the BP that begins the current word."
  (FORWARD-WORD (FORWARD-WORD (FORWARD-CHAR BP -1 T) 1 T) -1 T)) 


(DEFMACRO WITH-SPELLING-WORD-SYNTAX (&BODY BODY)
  "Execute BODY in spelling word mode.
The spelling word syntax table is temporarily installed as the word syntax table."
  `(LET ((*MODE-WORD-SYNTAX-TABLE* *SPELLING-WORD-SYNTAX-TABLE*))
     ,@BODY)) 



(DEFCOM COM-CORRECT-SPELLING
	"Correct the spelling for the entire buffer or just the region if one exists."
	()
  (SPELLER:EDITOR-INIT-SPELLER)
  (LET ((CORRECTION-FUNCTION (IF (MEMBER (SEND *INTERVAL* :MAJOR-MODE) '(COMMON-LISP-MODE ZETALISP-MODE))
				 'CORRECT-SPELLING-LISP
				 'CORRECT-SPELLING-INTERVAL))
	(REDISPLAY DIS-ALL))
  (WITH-SPELLING-WORD-SYNTAX
    (PRESERVE-POINT 
      (COND ((WINDOW-MARK-P *WINDOW*)
	     (TYPEIN-LINE "Correcting region spelling...")
	     (SETF (WINDOW-MARK-P *WINDOW*) nil)	;1Get rid of region display*
	       (REDISPLAY *WINDOW* :RELATIVE 0)
	       (CATCH-ALL
		 (UNLESS (FUNCALL CORRECTION-FUNCTION (POINT) (MARK))
		   (SETF REDISPLAY DIS-NONE))))

	    (T   (TYPEIN-LINE "Correcting Spelling...")
		 (CATCH-ALL
		 (UNLESS (FUNCALL CORRECTION-FUNCTION *interval*)
		   (SETF REDISPLAY DIS-NONE)))))))
  (TYPEIN-LINE-MORE " Done.")
  REDISPLAY))



;(DEFCOM COM-CORRECT-WORD-SPELLING
;	"Correct the spelling of the word at point." ()
;  (SPELLER:EDITOR-INIT-SPELLER)	
;  (LET ((DISPLAY-CODE DIS-NONE))
;    (WITH-SPELLING-WORD-SYNTAX
;	    (LET* ((BP1 (CURRENT-WORD (POINT)))
;		      (BP2 (FORWARD-WORD BP1 1 T))
;		      (WORD (STRING-INTERVAL BP1 BP2)))
;		 (TYPEIN-LINE (STRING-APPEND "Correcting the spelling of " #\" WORD #\" "...")) 
;		 (CATCH 'SPELLER:ABORT
;		 (IF (SPELLER:EDITOR-CHECK-WORD-P WORD)
;		     (TYPEIN-LINE-MORE "2 *OK")
;		     (LET ((NEW-WORD (SPELLER:EDITOR-CORRECT-WORD WORD)))
;		       (UNLESS (STRING-EQUAL WORD NEW-WORD)	
;			 (CASE-REPLACE BP1 BP2  NEW-WORD)
;			 (SETQ DISPLAY-CODE DIS-TEXT)))))))
;  DISPLAY-CODE))

(DEFCOM COM-CORRECT-WORD-SPELLING
	"Correct the spelling of the word at point." ()
  ;;; Modded by JPR so that the right thing happens with M-$ if
  ;;; there is a region
  (SPELLER:EDITOR-INIT-SPELLER)
  (if (window-mark-p *window*)
      (com-correct-spelling)
      (LET ((DISPLAY-CODE DIS-NONE))
	(WITH-SPELLING-WORD-SYNTAX
	  (LET* ((BP1 (CURRENT-WORD (POINT)))
		 (BP2 (FORWARD-WORD BP1 1 T))
		 (WORD (STRING-INTERVAL BP1 BP2)))
	    (TYPEIN-LINE (STRING-APPEND "Correcting the spelling of "
					#\" WORD #\" "...")) 
	    (CATCH 'SPELLER:ABORT
	      (IF (SPELLER:EDITOR-CHECK-WORD-P WORD)
		  (TYPEIN-LINE-MORE " OK")
		  (LET ((NEW-WORD (SPELLER:EDITOR-CORRECT-WORD WORD)))
		    (UNLESS (STRING-EQUAL WORD NEW-WORD)	
		      (CASE-REPLACE BP1 BP2  NEW-WORD)
		      (SETQ DISPLAY-CODE DIS-TEXT)))))))
	DISPLAY-CODE)))

(defvar *count-words-checked* nil
"When true a count of the number of words is displayed as a row of
 dots and plusses.  Dots indicate 100 words, plusses come every 500.
"
)

(DEFUN CORRECT-SPELLING-INTERVAL (START-BP &OPTIONAL END-BP (reposition-end-bp-p t))
  "Corrects the spelling of an group of words specified by 
   START-BP and END-BP as the start and end of the interval
   or by START-BP as an interval.  Returns t if any words were
   changed, nil otherwise."
;;; reposition-end-bp-p arg added by JPR on 11/16/89 13:36:38
  (LET ((CHANGED NIL))
  (GET-INTERVAL START-BP END-BP NIL)
  (SETQ START-BP (COPY-BP START-BP)
	END-BP (if reposition-end-bp-p
		   (FORWARD-WORD (FORWARD-WORD END-BP -1 T) 1 T)
		   end-bp))
  (WITH-UNDO-SAVE  ("Spelling Correction" START-BP END-BP T)
    (LET ((*REGION-MARKING-MODE* :REVERSE-VIDEO)	;reverse video misspelled words
;;; Modded here by JPR 21 Jan 88.
	  (word-count 0))
      (DO* ((BP1 (CURRENT-WORD START-BP))
	    (BP2 (let ((next (FORWARD-WORD BP1 1 T)))
		   (if (or reposition-end-bp-p
			   (bp-< next end-bp))
		        next
			;; jwz: copy the bp passed in, since we stomp it.
			(copy-bp end-bp))))
	    (WORD))
	   (NIL)
	(1if* (bp-< bp2 bp1)
	    (1psetq* bp1 bp2
		    bp2 bp1))
	(SETQ WORD (STRING-INTERVAL BP1 BP2))
	(UNLESS (SPELLER:EDITOR-CHECK-WORD-P WORD)
	  (SETF CHANGED T)
	  (MOVE-BP (POINT) BP1)
	  (MOVE-BP (MARK) BP2)
	  (SETF (WINDOW-MARK-P *WINDOW*) T)
	  (REDISPLAY *WINDOW* :ABSOLUTE)   ;display word in center of screen
	  (LET ((NEW-WORD (SPELLER:EDITOR-CORRECT-WORD WORD)))
	    (UNLESS (STRING-EQUAL WORD NEW-WORD)
	      ;;; jwz: took out deletion loop.
	      (CASE-REPLACE BP1 BP2  NEW-WORD))))
;;; Modded here by JPR 21 Jan 88.
	(setq word-count (+ 1 word-count))
	(if (and *count-words-checked*
		 (equal (mod word-count 100) 99)
	    )
	    (if (equal (mod word-count 500) 499)
		(princ "+" *query-io*)
		(princ " " *query-io*)
	    )
	    nil
	)
	(SETQ BP2 (FORWARD-WORD BP2 1))
	(WHEN (OR (NULL BP2) (BP-< END-BP BP2)) (RETURN))
	(SETQ BP1 (FORWARD-WORD BP2 -1)))
      (SETF (WINDOW-MARK-P *WINDOW*) nil)  ;Get rid of region display
      ))
  CHANGED))


(DEFUN CORRECT-SPELLING-LISP (START-BP &OPTIONAL END-BP (reposition-end-bp-p t))
  "Corrects the spe2l*ling of LISP strings and comments in the 
   interval specified by START-BP and END-BP as the start and
   end of the interval or by START-BP as an interval.  
   This function assumes LISP code in the buffer.2 Returns t if anything 
   was changed.  NIL otherwise.*"
;;; reposition-end-bp-p arg added by JPR on 11/16/89 13:36:38
  (LET ((CHANGED NIL))
  (GET-INTERVAL START-BP END-BP NIL)
  (SETQ START-BP (COPY-BP START-BP)
	END-BP (1if* reposition-end-bp-p
		    (FORWARD-WORD (FORWARD-WORD END-BP -1 T) 1 T)
		    end-bp))
  (WITH-UNDO-SAVE  ("Spelling Correction" START-BP END-BP T)
    (LET ((*REGION-MARKING-MODE* :REVERSE-VIDEO)	;reverse video misspelled words
	  ;;; Modded here by JPR 21 Jan 88.
	  (word-count 0))
      (DO* ((BP1 (CURRENT-WORD START-BP))	
	    (BP2 (1let* ((next (FORWARD-WORD BP1 1 T)))
		   (1if* (1or* reposition-end-bp-p
			   (bp-< next end-bp))
		        next
			;; jwz: copy the bp passed in, since we stomp it.
			(copy-bp end-bp))))
	    (WORD))
	   (NIL)
	(1if* (bp-< bp2 bp1)
	    (1psetq* bp1 bp2
		    bp2 bp1))
	(MULTIPLE-VALUE-BIND (STRING NIL COMMENT)
	    (LISP-BP-SYNTACTIC-CONTEXT BP1)
	  (WHEN (OR STRING COMMENT) 
	    (SETQ WORD (STRING-INTERVAL BP1 BP2))
	    (UNLESS (SPELLER:EDITOR-CHECK-WORD-P WORD)
	      (SETF CHANGED T)
	      (MOVE-BP (POINT) BP1)
	      (MOVE-BP (MARK) BP2)
	      (SETF (WINDOW-MARK-P *WINDOW*) T)
	      (REDISPLAY *WINDOW* :ABSOLUTE)	;display word in center of screen
	      (LET ((NEW-WORD (SPELLER:EDITOR-CORRECT-WORD WORD)))
		(UNLESS (STRING-EQUAL WORD NEW-WORD)
		  (CASE-REPLACE BP1 BP2  NEW-WORD)))))
;;; Modded here by JPR 21 Jan 88.
	  (setq word-count (+ 1 word-count))
	  (if (and *count-words-checked*
		   (equal (mod word-count 100) 99)
	      )
	      (if (equal (mod word-count 500) 499)
		  (princ "+" *query-io*)
		  (princ " " *query-io*)
	      )
	      nil
	  )
	  (SETQ BP2 (FORWARD-WORD BP2 1))
	  (WHEN (OR (NULL BP2) (BP-< END-BP BP2)) (RETURN))
	  (SETQ BP1 (FORWARD-WORD BP2 -1)))
	(SETF (WINDOW-MARK-P *WINDOW*) nil))))	;Get rid of region display
  CHANGED))





;
;  Add commands to correct spelling to command table.
;
#-elroy #!z (SET-COMTAB *STANDARD-COMTAB* 
	    '(#/META-$       COM-CORRECT-WORD-SPELLING
	      #/M-C-$  COM-CORRECT-SPELLING)
		   (MAKE-COMMAND-ALIST '(COM-CORRECT-SPELLING
					  COM-CORRECT-WORD-SPELLING)))

#+elroy (SET-COMTAB *STANDARD-COMTAB* 
	    '(#\META-$       COM-CORRECT-WORD-SPELLING
	      #\M-C-$  COM-CORRECT-SPELLING)
		   (MAKE-COMMAND-ALIST '(COM-CORRECT-SPELLING
					  COM-CORRECT-WORD-SPELLING)))




;-------------------------------------------------------------------------------

;;; The following code was all written by James Rice at the Knowledge Systems Laboratory of Stanford University.

(defvar *global-document* :Unbound
"The global spelling checking dictionary."
)
(defvar *selected-local-dictionary* nil
"The dictionary selected in the current document."
)

(defvar speller:*modified-dictionaries* nil
"A list of all the dictionaries that have been modified this session."
)

(defvar speller:*read-only-dictionary-names* '("Main")
"A list of all of the dictionaries that the user cannot add names to."
)

(defvar *all-document-dictionaries* nil)

(defvar *all-global-dictionary-names* '("Main"))

(defun dictionary-named (name)
  (let ((dict
	  (nth (position name (send *global-document* :dictionary-list)
			 :key #'(lambda (x) (send x :name))
			 :Test #'string-equal
	       )
	       (send *global-document* :dictionary-list)
	  )
	)
       )
       (if (not dict) (ferror nil "Cannot find a dictionary called ~S" name))
       (cons (send dict :name) dict)
  )
)

(defun paths-match (a b)
  (fs:pathname-equal (fs:merge-pathnames a (default-dict-path))
		     (fs:merge-pathnames b (default-dict-path))
  )
)

(defun dictionary-with-path (path)
  (let ((path (fs:merge-pathnames path (default-dict-path))))
       (let ((index (position path (send *global-document* :dictionary-list)
			      :key #'(lambda (x) (send x :path))
			      :Test #'paths-match
		    )
	     )
	    )
	    (if index
		(let ((dict (nth index (send *global-document* :dictionary-list))))
		     (cons (send dict :name) dict)
		)
		(ferror nil "Cannot find a dictionary matching pathname ~S" path)
	    )
	    
       )
  )
)


(defun try-to-coerce-into-pathname (string)
  (catch-error (fs:parse-pathname string) nil)
)


(defun coerce-dictionary (dict)
  (typecase dict
    (speller:dictionary dict)
    (string (if (try-to-coerce-into-pathname dict)
		(coerce-dictionary
		  (rest (dictionary-with-path (try-to-coerce-into-pathname dict)))
		)
		(coerce-dictionary (rest (dictionary-named dict)))
	    )
    )
    (pathname (coerce-dictionary (rest (dictionary-with-path dict))))
    (symbol (coerce-dictionary (symbol-name dict)))
    (otherwise (ferror nil "~S cannot be coerced into a dictionary." dict))
  )
)

(defun all-global-dictionaries ()
  (mapcar #'rest (mapcar #'dictionary-named *all-global-dictionary-names*))
)


(defun find-current-document ()
  (speller:editor-init-speller)
  (if (and (send *interval* :operation-handled-p :get)
	   (send *interval* :operation-handled-p :name)
      )
      (or (send *interval* :get :current-document)
	  (let ((new (make-instance 'speller:document-dictionary
				    :name (send *interval* :name)
		     )
		)
	       )
	       (send *interval* :putprop new :current-document)
	       (send new :set-dictionary-list
		     (all-global-dictionaries)
	       )
	       (pushnew new *all-document-dictionaries*)
	       new
	  )
      )
      *global-document*
  )
)

;(let ((compiler:compile-encapsulations-flag t))
;     (advise process-command-char :around :set-up-dictionary-for-buffer nil
;       ;;; Bind up speller:*current-document* and
;       ;;; zwei:*selected-local-dictionary*
;       ;;; in the current document so that they always point to the right things
;       ;;; for the current buffer.
;       (let ((doc (find-current-document)))
;	    (let ((speller:*current-document* doc)
;		  (zwei:*selected-local-dictionary*
;		    (send *interval* :get :selected-local-dictionary)
;		  )
;		 )
;		 :do-it
;	    )
;       )
;     )
;)

(1defun* speller-command-hook (&rest ignore)
"A simple hook to make sure that we always have the right vars bound."
  (let ((doc (find-current-document)))
       (1setq* speller:*current-document* doc)
       (1setq* zwei:*selected-local-dictionary*
	      (send *interval* :get :selected-local-dictionary)
       )
  )
)

;;; Define the command hook.
(defprop speller-command-hook 11 command-hook-priority)
(command-hook 'speller-command-hook '*command-hook*)

;;; Add speller vars to closure variables.
;;; It doesn't really matter if these aren't bound for each
;;; window but it's a bit cleaner if they are.
(1setq* editor-closure-variables
       (remove-duplicates
	 (1append* '((speller:*current-document* nil)
		    (zwei:*Selected-Local-Dictionary* nil)
		   )
		   Editor-closure-variables
	 )
	 :Test #'1equal*
       )
)

(defun remove-all (exclusions from)
"Removes all instances of Exclusions from From."
  (if exclusions
      (remove-all (rest exclusions)
		  (remove (first exclusions) from
			  :key #'first :test #'string-equal
		  )
      )
      from
  )
)

(defun all-names-for (&optional (doc speller:*current-document*))
"Returns a list of all of the names of the dictionaries in doc."
  (mapcar #'(lambda (dic) (send dic :name)) (send doc :dictionary-list))
)

(defun dictionary-name-alist
       (&optional (exclusions nil) (doc speller:*current-document*))
"Returns an alist of Name . Dictionary for all of the dictionaries in Doc,
 except all of those named by Exclusions.
"
  (remove-all exclusions
	      (remove-duplicates
		(mapcar #'(lambda (dic) (cons (send dic :name) dic))
			  (if (consp doc)
			      doc
			      (send doc :dictionary-list)
			  )
		)
		:test #'equalp
	      )
  )
)


(defun create-new-dictionary (&key (path nil) (name-p nil))
"Creates a new dictionary.  If Name-p it T then the user is prompted for the
 name of the dictionary.  If Path is provided then this is set as the path for
 the dictionary.
"
  (let ((name (if name-p
		  (string-capitalize (completing-read-from-mini-buffer
				       "Name for new dictionary:" nil t
				     )
		  )
		  "Local"
	      )
	)
       )
       (if (and name-p
		(assoc name (dictionary-name-alist nil *global-document*)
		       :test #'string-equal
		)
	   )
	   (progn (typein-line "Dictionary \"~A\" already exists." name)
		  (barf)
	   )
	   (let ((instance (make-instance 'speller:dictionary :name name)))
	        (if path (send instance :set-path path) nil)
		(send *global-document* :add-dictionary instance)
		instance
	   )
       )
  )
)


(defun find-a-dictionary-for (path all-dictionaries create-p name-p)
"Finds a dictionary which matches Path in All.  If there is no match and
 Create-P then a new one is created.  If Name-p is T then any new dictionary
 will be created with a name which is got from the user.
"
  (if all-dictionaries
      (if (string-equal path (send (first all-dictionaries) :path))
	  (values (first all-dictionaries) nil)
	  (find-a-dictionary-for path (rest all-dictionaries) create-p name-p)
      )
      (if create-p
	  (values (create-new-dictionary :path path :name-p name-p) t)
	  (values nil nil)
      )
  )
)

(defun default-dict-path ()
"A default pathname for a dictionary file."
  (send (if (boundp '*major-mode*)
	    (fs:default-pathname (pathname-defaults))
	    (fs:default-pathname)
	)
        :new-pathname :type "DICTIONARY"
	:version nil
  )
)

(defmethod (speller:dictionary :set-path) (to)
"Becuase Path is not a settable IV."
  (setq speller:path to)
)

(defmethod (speller:dictionary :set-name) (to)
"Becuase Name is not a settable IV."
  (setq speller:name to)
)

;-------------------------------------------------------------------------------

(defun save-dictionary (dictionary)
  (speller:editor-init-speller)
  (let ((dict (coerce-dictionary dictionary)))
       (send dict :save-yourself)
  )
)

(defun unmod-dictionary (dict)
  (send dict :unmod-yourself t)
)

(defcom com-save-dictionary
  "Saves a spelling checker dictionary to a file which is prompted for." ()
  (speller:editor-init-speller)
  (let ((dict (completing-read-from-mini-buffer
			  "Name of dictionary to save:"
			  (dictionary-name-alist
			    speller:*read-only-dictionary-names*
			    speller:*modified-dictionaries*
			  )
			  nil
	      )
	)
       )
       (send (rest dict) :save-yourself)
  )
  dis-none
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-save-dictionary))
)

;-------------------------------------------------------------------------------

(defun load-dictionary (path)
  (speller:editor-init-speller)
  (multiple-value-bind (dictionary new-p)
      (find-a-dictionary-for
	path (send *global-document* :dictionary-list) t nil
      )
    (ignore new-p)
    (send dictionary :load)
    (typein-line "Dictionary \"~A\" Loaded." (send dictionary :name))
  )
)

(defcom com-load-dictionary
  "Loads a spelling checker dictionary from a file which is prompted for." ()
  (speller:editor-init-speller)
  (let ((path (send (read-defaulted-pathname
		      "From file:" (default-dict-path)
		      nil :newest :write
	            )
		    :string-for-printing
	      )
	)
       )
       (load-dictionary path)
  )
  dis-none
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-load-dictionary))
)

;-------------------------------------------------------------------------------
 
(defun select-dictionary (dictionary)
  "Selects a dictionary."
  (speller:editor-init-speller)
  (let ((dict (coerce-dictionary dictionary)))
       (setq *selected-local-dictionary* dict)
       (send *interval* :putprop *selected-local-dictionary*
	     :selected-local-dictionary
       )
       (typein-line "Dictionary \"~A\" Selected."
		    (send *selected-local-dictionary* :name)
       )
  )
)

(defcom com-select-dictionary
" Selects a spelling checker dictionary as the current one.
 This overrides the mechanism which prompts which dictionary
 a new word is to be added to.  With numeric arg deselects the
 current dictionary.
" ()
  (speller:editor-init-speller)
  (if *numeric-arg-p*
      (progn (setq *selected-local-dictionary* nil)
	     (typein-line "Selected Dictionary disabled.")
      )
      (let ((dict (completing-read-from-mini-buffer
		    (if *selected-local-dictionary*
			(format nil "Name of dictionary to select (Default ~A):"
				(send *selected-local-dictionary* :name)
			)
			"Name of dictionary to select:"
		    )
		    (dictionary-name-alist
		      speller:*read-only-dictionary-names*
		    )
		    nil
		  )
	    )
	   )
	   (select-dictionary (rest dict))
      )
  )
  dis-none
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-select-dictionary))
)

;-------------------------------------------------------------------------------

(defun revert-dictionary (dictionary)
  (speller:editor-init-speller)
  (let ((dict (coerce-dictionary dictionary)))
       (send dict :load)
       (typein-line "Dictionary \"~A\" reverted." (send dict :name))
  )
)

(defcom com-revert-dictionary-to-buffer
" Reverts a spelling checker dictionary." ()
  (speller:editor-init-speller)
  (let ((dict (completing-read-from-mini-buffer
		"Name of dictionary to revert:"
	        (dictionary-name-alist
		  speller:*read-only-dictionary-names*
		  *global-document*
		)
		nil
	      )
	)
       )
       (revert-dictionary (rest dict))
  )
  dis-none
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-revert-dictionary-to-buffer))
)

;-------------------------------------------------------------------------------

(defun add-dictionary (dictionary)
  (speller:editor-init-speller)
  (let ((dict (coerce-dictionary dictionary)))
       (let ((doc (find-current-document)))
	    (send doc :add-dictionary dict)
	    (typein-line "Dictionary \"~A\" added to ~A." (send dict :name)
			 (send doc :name)
	    )
       )
  )
)

(defcom com-add-dictionary-to-buffer
" Adds a spelling checker dictionary to the current buffer's
 list of dictionaries.
" ()
  (speller:editor-init-speller)
  (let ((dict (completing-read-from-mini-buffer
		"Name of dictionary to add:"
	        (dictionary-name-alist
		  (append speller:*read-only-dictionary-names*
			  (all-names-for)
		  )
		  *global-document*
		)
		nil
	      )
	)
       )
       (add-dictionary (rest dict))
  )
  dis-none
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-add-dictionary-to-buffer))
)

;-------------------------------------------------------------------------------

(defun add-global-dictionary (dictionary)
  (speller:editor-init-speller)
  (let ((dict (coerce-dictionary dictionary)))
       (pushnew (send dict :name) *all-global-dictionary-names*
		:test #'string-equal
       )
       (mapcar #'(lambda (doc)
;		   (typein-line "Dictionary \"~A\" added to ~A."
;				(send dict :name) (send doc :name)
;		   )
		   (send doc :add-dictionary dict)
		 )
		 *all-document-dictionaries*
       )
       (typein-line "Dictionary \"~A\" added to all buffers." (send dict :name))
  )
)

(defcom com-add-global-dictionary
" Adds a spelling checker dictionary to all buffers."
  ()
  (speller:editor-init-speller)
  (let ((dict (completing-read-from-mini-buffer
			  "Name of dictionary to add:"
			  (dictionary-name-alist
			    (append speller:*read-only-dictionary-names*
				    *all-global-dictionary-names*
				    (all-names-for)
			    )
			    *global-document*
			  )
			  nil
	      )
	)
       )
       (add-global-dictionary (rest dict))
  )
  dis-none
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-add-global-dictionary))
)

;-------------------------------------------------------------------------------

(defun remove-global-dictionary (dictionary)
  (speller:editor-init-speller)
  (let ((dict (coerce-dictionary dictionary)))
       (setq *all-global-dictionary-names*
	     (remove (send dict :name) *all-global-dictionary-names*
		     :test #'string-equal
	     )
       )
       (mapcar #'(lambda (doc)
;		   (typein-line "Dictionary \"~A\" removed from ~A."
;				(send dict :name) (send doc :name)
;		   )
		   (send doc :remove-dictionary dict)
		 )
		 *all-document-dictionaries*
       )
       (typein-line "Dictionary \"~A\" removed from all buffers."
		    (send dict :name)
       )
  )
)

(defcom com-remove-global-dictionary
" Removes a spelling checker dictionary from all buffers."
  ()
  (speller:editor-init-speller)
  (let ((dict (completing-read-from-mini-buffer
		"Name of dictionary to remove:"
	        (dictionary-name-alist
		  (append speller:*read-only-dictionary-names*
			  (set-difference (all-names-for *global-document*)
					  *all-global-dictionary-names*
					  :test #'string-equal
				  
			  )
		  )
		  *global-document*
		)
		nil
	      )
	)
       )
       (remove-global-dictionary (rest dict))
  )
  dis-none
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-remove-global-dictionary))
)

;-------------------------------------------------------------------------------

(defun edit-a-dictionary (dict)
"Edits dict."
  (let ((new-buffer (read-buffer-name "Buffer for dictionary:" t t)))
       (send new-buffer :Select)
       (let ((stream (interval-stream (send new-buffer :First-Bp)
				      (send new-buffer :Last-Bp)
		     )
	     )
	    )
	    (unwind-protect
	      (let ((mode-line ";;; -*- Mode:Text; Package:User -*-"))
		   (format stream "~a~%" mode-line)
		   (format stream "~&Name - ~S" (send dict :name))
		   (format stream "~&Path - ~S" (send dict :path))
		   (format stream "~&(Word  Suffixes)")
		   (maphash #'(lambda (key value &rest ignore)
				(if (string-equal value "nil")
				    (format stream "~&~A" key)
				    (format stream "~&~A	~A" key value)
				)
			      )
			      (send dict :spell-hash)
		   )
		   (move-bp (point) (send new-buffer :First-Bp))
		   (reparse-buffer-mode-line new-buffer)
	      )
	      (close stream)
	    )
       )
 )
)


(defcom com-edit-dictionary
" Edits a dictionary in a buffer specified by the user.  To save the buffer
 Use M-x Save Buffer As Dictionary.
"
  (speller:editor-init-speller)
  (let ((dict (rest (completing-read-from-mini-buffer
		      "Name of dictionary to edit:"
		      (dictionary-name-alist
			speller:*read-only-dictionary-names*
			*global-document*
		      )
		      nil
		    )
	      )
	)
       )
       (edit-a-dictionary dict)
  )
  dis-text
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-edit-dictionary))
)


;-------------------------------------------------------------------------------



(defcom com-save-buffer-as-dictionary
" Saves a dictionary buffer as a dictionary file.  It gets the path to save
 the dictionary to from the Path item in the buffer.
"
  (speller:editor-init-speller)
  (let ((buffer (read-buffer-name "Buffer to save as a dictionary:"
		  (first (history-contents (send *window* :buffer-history))) nil
		)
	)
       )
       (let ((stream (interval-stream (send buffer :First-Bp)
				      (send buffer :Last-Bp)
		     )
	     )
	    )
	    (unwind-protect
	      (let ((name (progn (read stream) (read stream) (read stream)))
		    (path (progn (read stream) (read stream) (read stream)))
		   )
		   (read-line stream)
		   (speller:builder stream path name)
		   (typein-line "Dictionary \"~A\" saved to ~A." name path)
	      )
	      (close stream)
	    )
       )
  )
  dis-none
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-save-buffer-as-dictionary))
)

;-------------------------------------------------------------------------------

(defcom com-remove-dictionary-from-buffer
  "Removes a spelling checker dictionary from the current buffer." ()
  (speller:editor-init-speller)
  (let ((dict (completing-read-from-mini-buffer
			  "Name of dictionary to remove:"
			  (dictionary-name-alist)
			  nil
	      )
	)
       )
       (send speller:*current-document* :remove-dictionary (rest dict))
       (typein-line "Dictionary \"~A\" removed from ~A." (first dict)
		    (send speller:*current-document* :name)
       )
  )
  dis-none
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-remove-dictionary-from-buffer))
)

;-------------------------------------------------------------------------------

(defun create-dictionary ()
  (com-create-dictionary)
)

(defcom com-create-dictionary
  "Creates a new spelling checker dictionary." ()
  (speller:editor-init-speller)
  (let ((dict (create-new-dictionary :name-p t)))
       (typein-line "Dictionary \"~A\" Created." (send dict :name))
  )
  dis-none
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-create-dictionary))
)

;-------------------------------------------------------------------------------

(defun menu-of-local-dictionaries (label names)
" Pops up a menu of all of the dictionaries named in the Alist Names, with the
 label Label.
"
  (let ((names (mapcar #'first names)))
       (1prog1* (w:menu-choose
		 names :Label label :columns (if (< (length names) 20) 1 nil)
	       )
	       (1send* zwei:*window* :Mouse-Select) ;; JPR.  Menu tends to deselect us.
       )
  )
)

(1defvar* *prefer-user-defined-local-dictionaries-over-local-p* t
"When true will not give the dictionary called LOCAL as an option for word addition
if there is a local dictionary that the user has already provided.
"
)

(defun get-local-dict-name (dictionary-name)
"Gets the name of a local dictionary.  If dictionary-name is not \"Local\" then
 this is assumed to be the name you want.  If it is \"Local\" then the user
 is prompted for the name of a local dictionary either by menu or by minibuffer
 as appropriate from the list of current dictionaries.
"
  (if (string-equal dictionary-name "Local")
      (let ((locals
	      (1if* *prefer-user-defined-local-dictionaries-over-local-p*
		   (1remove* (1assoc* "Local" (dictionary-name-alist speller:*read-only-dictionary-names*) :Test #'string-equal)
			    (dictionary-name-alist speller:*read-only-dictionary-names*)
			    :Test #'equalp
		   )
		   (dictionary-name-alist speller:*read-only-dictionary-names*)
	       )
	    )
	   )
	   (case (length locals)
	     (0 nil)
	     (1 (first (first locals)))
	     (otherwise
	       (or (if speller:*prompt-in-minibuffer*
		       (first (completing-read-from-mini-buffer
				"Name of dictionary for insertion:"
				locals nil
			      )
		       )
		       (menu-of-local-dictionaries
			 "Name of dictionary for insertion:" locals
		       )
		   )
		   dictionary-name
	       )
	     )
	   )
      )
      dictionary-name
  )
)

;-------------------------------------------------------------------------------

(DEFCOM COM-LIST-dictionaries "Print a list of the all dictionaries, their sizes and whether they have been modified." ()
  (LET* ((STAR-FLAG NIL)
	 (EQV-FLAG NIL)
	 (CIRCLE-PLUS-FLAG NIL)
	 (MAX-SIZE (MIN 120 (- (SEND *STANDARD-OUTPUT* :SIZE-IN-CHARACTERS) 50)))
	 (SIZE-POS (MIN (MAX (+ (FIND-MAXIMUM-BUFFER-NAME-LENGTH MAX-SIZE) 3)
				20)
			   (+ MAX-SIZE 2))))
    (FORMAT T "~&Dictionaries in ZWEI:~%  Dictionary name:~vTSize:~2%"
	    SIZE-POS)
    (DOLIST (DICTIONARY (send *global-document* :dictionary-list))
      (multiple-value-bind (name flag)
	  (send DICTIONARY :NAME-FOR-DISPLAY MAX-SIZE)
	(WRITE-CHAR (COND ((send dictionary :read-only-p)
			   (SETQ EQV-FLAG T)
			   #\)		  ; means read-only
			  ((send DICTIONARY :modified-p)
			   (SETQ STAR-FLAG T)
			   #\*)		  ;* means has unsaved changes.
			  (T
			   #\SPACE))	  ;blank if unmodified.
		    *STANDARD-OUTPUT*)
	(WRITE-CHAR #\SPACE *STANDARD-OUTPUT*)
	(IF FLAG
	    (SETQ CIRCLE-PLUS-FLAG FLAG))
	(LET ((size (hash-table-count (send DICTIONARY :spell-hash))))
	   (SEND *STANDARD-OUTPUT* :ITEM 'Dictionary DICTIONARY
		 "~A~vT~a"
		 NAME SIZE-POS size))
	(TERPRI *STANDARD-OUTPUT*)))
    (TERPRI *STANDARD-OUTPUT*)
    (AND STAR-FLAG
	 (PRINC "* means dictionary modified.  " *STANDARD-OUTPUT*))
    (AND EQV-FLAG
	 (PRINC " means read-only.  " *STANDARD-OUTPUT*))
    (AND CIRCLE-PLUS-FLAG
	 (PRINC "  means name truncated." *STANDARD-OUTPUT*))
    (AND (OR STAR-FLAG
	     EQV-FLAG
	     CIRCLE-PLUS-FLAG)
	 (TERPRI *STANDARD-OUTPUT*))
    DIS-NONE))

(tools:add-to-item-type-alist
  '(dictionary add-dictionary
	       "L: Add this dictionary, R: Menu of Add, Edit, Save, Unmod."
	       ("Add" :Value add-dictionary
		:Documentation "Add this dictionary to the current buffer."
	       )
	       ("Edit" :Value edit-a-dictionary
		:Documentation "Edit this dictionary."
	       )
	       ("Save" :Value save-dictionary
		:Documentation "Save this dictionary."
	       )
	       ("Unmod" :Value unmod-dictionary
		:Documentation "Mark this dictionary as unmodified."
	       )
   )
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-list-dictionaries))
)

;-------------------------------------------------------------------------------

(defun turn-path-into-defaults (path)
"Turns Path into a list of defaults suitable for merging."
  (list (list :host      (send path :host))
	(list :directory (send path :directory))
	(list :name      (send path :name))
	(list :type      (send path :type))
  )
)

(defun read-pathname (prompt default)
"Read a pathname with Prompt and Default.  This is a safe version, which will
 work even outside Zmacs.
"
  (if (boundp '*major-mode*)
      (zwei:read-defaulted-pathname prompt default nil :newest :write)
      (if default
	  (prompt-and-read (list :pathname :defaults
				 (turn-path-into-defaults default)
			   )
			   "~A (Default: ~A)"
			   prompt default
	  )
	  (prompt-and-read :pathname "~A" prompt)
      )
  )
)

(defun maybe-save-changed-dictionaries-p ()
"True if any dictionaries should be saved because they have been modified."
  (remove nil (mapcar #'(lambda (dict) (send dict :modified-p))
		      (send *global-document* :dictionary-list)
	      )
  )
)

(defun save-changed-dictionaries ()
"Saves all modified dictionaries, asking the user if they should be saved."
  (mapcar #'(lambda (dict) (send dict :maybe-save-yourself))
	    (send *global-document* :dictionary-list)
  )
)

(Add-Initialization "Save changed dictionaries"
  `(progn (speller:editor-init-speller)
	  (if (maybe-save-changed-dictionaries-p)
	      (if (with-timeout (3600 (format *query-io* " - timeout proceeding") t)
		    (y-or-n-p "There are modified Spelling-Checker dictionaries, ~
                               do you want to save any of them? [Timeout 1 Min -> yes]"
                    )
                  )
		  (save-changed-dictionaries)
		  nil
	      )
	      nil
	  )
   )
  '(:Logout :Normal)
)

;-------------------------------------------------------------------------------

;;; Added by JPR.  01/09/90 14:04:37


(defminor com-auto-spell-check-mode
	  auto-spell-check-mode
	  "Spell"
	  3
	  "Causes the speller to be invoked on each word you type."
	  ()
  (command-hook 'auto-spell-check-hook *command-hook*)
)

(set-comtab *standard-comtab* nil
	    (make-command-alist '(com-auto-spell-check-mode))
)

(defprop auto-spell-check-hook 10 command-hook-priority) 

(defvar *chars-that-induce-automatic-spell-checking* '(#\space #\tab #\newline))

(defun auto-spell-check-hook (char)
  (if (and (not (eq *interval* (window-interval *mini-buffer-window*)))
	   (member char *Chars-That-Induce-Automatic-Spell-Checking*
		   :Test #'char-equal
	   )
	   ;; jwz: not in send-mail mode, or after the end of the headers.
	   (or (not (member 'mail-mode *mode-list* :test #'eq :key #'car))
	       (let* ((headers-end-bp (do* ((line (bp-line (interval-first-bp *interval*))
						  (line-next line)))
					   ((zerop (length line))
					    (create-bp line 0)))))
		 (or (null headers-end-bp) (bp-< headers-end-bp (point)))))
      )
      (spell-check-last-word (point))
      nil
  )
)

(defun spell-check-last-word (bp)
  (let ((before (forward-word bp -1 t)))
       (speller:editor-init-speller)
       (let ((correction-function
	       (if (member (send *interval* :major-mode)
			   '(common-lisp-mode zetalisp-mode)
		   )
		   'correct-spelling-lisp
		   'correct-spelling-interval
	       )
	     )
	     (redisplay dis-text)	; jwz: dis-text is enough - dis-all causes flicker.
	    )				;      we could probably get away with dis-line.
	    (let ((bp (copy-bp (point) :moves)))
		 (with-spelling-word-syntax
		   (catch-all
		     (unless (funcall correction-function before (point) nil)
		       (setf redisplay dis-none)
		     )
		   )
		 )
	         (move-bp (point) bp)
	    )
	    ;(redisplay *window* :relative redisplay)	; jwz: this doesn't make sense.
	    (must-redisplay *window* redisplay)
	    (redisplay *window*)
       )
  )
)


;-------------------------------------------------------------------------------

;;; The following by Jamie Zawinski.

;;; Viewing dictionaries.

(defcom com-show-dictionary
	"Prompt for the name of a dictionary, and dump its words to standard output."
	()
  (let* ((dict (cdr (completing-read-from-mini-buffer "Show which dictionary?"
		      (dictionary-name-alist speller:*read-only-dictionary-names* *global-document*)
		      nil)))
	 (hash (send dict :spell-hash))
	 (sort-p (if (> (hash-table-count hash) 300)		; If it's really big, ask before sorting.
		     (y-or-n-p "Dictionary ~S has ~D entries.  Sort it first?" (send dict :name) (hash-table-count hash))
		     t)))
    (format t "Words in ~S:~2%" (send dict :name))
    (flet ((dump (word suffix)
	     (format t "~& ~A~{ ~15,15t~A~}" word (delete nil (speller:add-suffixes word suffix)))))
      (if sort-p
	(let* ((list (sort (maphash-return #'cons hash) #'string< :key #'car)))
	  (dolist (cons list) (dump (car cons) (cdr cons))))
	(maphash #'dump hash))))
  (format t "~2%Done.")
  DIS-NONE)

(set-comtab *STANDARD-COMTAB* '() (make-command-alist '(com-show-dictionary)))

;;; Prompting for suffixes.

(defconstant ALL-SUFFIX-FLAGS "vnxhygjdtrzspm"
  "A string of all the suffix flags that are defined.")


(defun query-about-suffixes (word)
  "Returns two values, a word and a string of the suffixes which may be applied to it.
  Gets these via a pop-up multiple-choice menu."
  (1send* zwei:*window* :Mouse-Select) ;; JPR.  Menu tends to deselect us.
  (cond ((not (fquery `(:help-function ,#'(lambda (&rest ignore)
					    (format t "~2&A ``complex'' word is one which may have suffixes applied to ~
							it, or which is itself the suffixed form of another word.~2%")))
		      "Is ~S a complex word? " word))
	 (values word ""))
	(t
	 (let* ((real-word
		  (read-string-from-minibuffer "Type the root word:" word (length word)
		    "If this word is a suffixed form, as ``locally'' is to ``local'', then type the un-suffixed word.")))
	   (when (string= real-word "") (setq real-word word))
	   (let* ((alist '())
		  (defaults '()))
	     (dotimes (i (length ALL-SUFFIX-FLAGS))
	       (let* ((flag (schar ALL-SUFFIX-FLAGS i))
		      (mod-word (speller:add-suffix-to-word real-word (string flag)))
		      (menu-item (and mod-word `(,mod-word :value ,flag))))
		 (when mod-word
		   (push menu-item alist)
		   (when (speller:check-word-p mod-word) (push menu-item defaults)))))
	     (setq alist (nreverse alist))
	     (let* ((selected (w:menu-choose alist
					     :highlighting t
					     :highlighted-items defaults
					     :menu-margin-choices '(:DOIT)
					     :label (format nil "Select the forms of ~S which are valid." real-word))))
	       (1send* *window* :Mouse-Select) ;; JPR.  Menu tends to deselect us.
	       (values real-word (coerce selected 'simple-string))))))))


(defvar *query-about-suffices-on-new-words-p* t
"When true asks you about whether new words are
 complex words etc."
)

(defwhopper (speller:document-dictionary :add-word) (dictionary-name word suffix)
  "If a suffix was not passed in (NIL instead of a string) then prompt the user."
  (if (or (not *query-about-suffices-on-new-words-p*) suffix)
      (continue-whopper dictionary-name word suffix)
      (multiple-value-bind (root-word suffixes) (query-about-suffixes word)
	(continue-whopper dictionary-name root-word suffixes))))



;;; Editing an arbitrary string in the minibuffer.

(defvar *read-string-comtab* (make-comtab :comtab-keyboard-array '((#\Help . com-read-string-help))
					     :comtab-indirect-to *mini-buffer-comtab*))

(defun read-string-from-minibuffer (prompt &optional (initial-completion "") (initial-pos (length initial-completion))
						     *read-string-help*)
  (declare (special *read-string-help*))
  (multiple-value-bind (ignore ignore interval)
		       (edit-in-mini-buffer *read-string-comtab* initial-completion initial-pos (list prompt))
    (string-interval interval)))


(defcom com-read-string-help "Help from within READ-STRING-FROM-MINIBUFFER." ()
  (declare (special *read-string-help*))
  (format t "~&You are typing to the minibuffer.  Return or End will exit.~2%")
  (when (and (boundp '*read-string-help*) *read-string-help*)
    (princ *read-string-help*))
  (terpri)
  DIS-NONE)

;-------------------------------------------------------------------------------