;;; -*- Mode:Common-Lisp; Package:EH; Base:10 -*-
;;; This software developed by:
;;;	James Rice
;;; at the Stanford University Knowledge Systems Lab in 1986, 1987.
;;;
;;; This work was supported in part by:
;;;	DARPA Grant F30602-85-C-0012

;;;----------------------------------------------------------------------
;;;  Much of this file is derived from code licensed from Texas Instruments
;;;  Inc.  Since we'd like them to adopt these changes, we're claiming
;;;  no rights to them, however, the following restrictions apply to the
;;;  TI code:

;;; Your rights to use and copy Explorer System Software must be obtained
;;; directly by license from Texas Instruments Incorporated.  Unauthorized
;;; use is prohibited.

;;;                           RESTRICTED RIGHTS LEGEND

;;;Use, duplication, or disclosure by the Government is subject to
;;;restrictions as set forth in subdivision (b)(3)(ii) of the Rights in
;;;Technical Data and Computer Software clause at 52.227-7013.
;;;
;;;                     TEXAS INSTRUMENTS INCORPORATED.
;;;                              P.O. BOX 2909
;;;                           AUSTIN, TEXAS 78769
;;;                                 MS 2151
;;;
;;; Copyright (C) 1986,1987 Texas Instruments Incorporated. All rights reserved.
;;;----------------------------------------------------------------------


(load-tools '(:KSL-Patches
	      :36xx-Explorer
	      :Utilities
	      :General-Inspector
	      :Window-Debugger-Enhancements
	     )
)
;(setq *source-code-debugging-enabled* t)
;(setq *source-code-debugging-enabled* nil)
;(setq *print-marked-form* t)
;(setq *catch-errors-p* nil)
;(setq *print-function-names-that-failed-to-scd* t)
;(setq *still-source-code-debug-for-warnings* nil)
;-------------------------------------------------------------------------------
;;; Debug variables...

(defparameter *print-marked-form* nil)
(defparameter *catch-errors-p* t)
(defparameter *print-names-of-source-code-debugged-forms* nil
"When true it prints the names of source code debugged forms as they are
 compiled.
"
)
(defparameter *print-function-names-that-failed-to-scd* nil)
(defparameter *still-source-code-debug-for-warnings* t)
;-------------------------------------------------------------------------------
;;; Vars...

;-------------------------------------------------------------------------------
;;; Prepare correct compile/load environment.

(defvar compiler:*source-code-pc* :Unbound
  "The pc in the current stack frame."
)

(defparameter *remember-enabled* nil)

(defvar *source-code-debugging-enabled* nil
"Is true when source code debugging is switched on."
)

(eval-when (compile)
  (setq *remember-enabled*
	(if (boundp '*source-code-debugging-enabled*)
	    *source-code-debugging-enabled*
	    nil
	)
  )
  (setq *source-code-debugging-enabled* nil)
)

;-------------------------------------------------------------------------------

(defparameter *source-code-debugging-stepping-mode-enabled* nil
"When true the scd inserts stepping code, else just debug point locating stuff."
)

(defparameter *grinding-debugged-source-code* nil
"True when we are in an inspect window and grinding out source code debugger
 type source code.  Its value is either nil, when not grinding or the list
 (marked-region matched-code)
"
)

(defparameter *source-code-index-counter* nil
"A counter which is bumped for each new source code debugging point."
)

(defun within-function-name (x)
"Given an advise-within call this returns the function spec for it, i.e.
 (:within foo bar).
"
  (list :Within (second x) (third x))
)

(defun advise-putprop (name value property)
"Records a property for some advise.  Note, we can't function-spec-putprop,
since the name of the advise is the same as the function and this would result
in a setf (get.
"
  (setf (getf (sys:dbis-plist (sys:get-debug-info-struct name t)) property)
	value
  )
)

(defvar *ignorable-warnings* '(:debugger-breakpoint)
"A list of the names of compiler:warn warning types, which will not cause the
 source code debugger compiler to punt and try the normal compiler.
"
)

(defvar *source-code-debuggable-forms*
       `((defun second 3)
	 (defsubst second 3)
	 (defmacro second 3)
	 (defmethod make-method-name get-method-body)
	 (advise second 5 advise-putprop)
	 (advise-within within-function-name 6)
	)
"A list of items defining those forms which can be source code debugged e.g.
 (defun second 2).  The first is the name of the form, the second is a function
 which returns the name of the function, given the defining form, the third is a number such that nthcdr of it and the defining form returns the body (after the
 arglist but before the decls/docstring.
"
)

(defparameter *marked-p* nil
"When true we are already inside the source code debugger during compilation,
 we should therefore not enter it again.
"
)

;-------------------------------------------------------------------------------
;;; Defstructs

;;; A defstruct holding a list of PC numbers and the form that thuse pc numbers
;;; are around.  Instances of this defstruct are stored on the function's plist
;;; to label the source code.

(defstruct (numbered-component :named)
  numbers
  code
)

(defun (:property numbered-component named-structure-invoke)
       (message record &rest args)
"A message handler for Numbered-Components."
  (case message
    (:which-operations '(:which-operations :print-self))
    (:print-self
     (catch-error
       (format (first args) "#<~	, ~S>" (numbered-component-numbers record) nil
	       (numbered-component-code record)
       )
       nil
     )
    )
    (otherwise (ferror nil "Illegal message ~S send to a numbered component."
		       message
	       )
    )
  )
)

;;; A defstruct holding some code which has been marked with debugging code
;;; and also its marked form, which is stored on the functions's plist to be
;;; used by the debugger.

(defstruct (marked-code :named)
  code
  marked
)

(defun (:property marked-code named-structure-invoke)
       (message record &rest args)
"A message handler for Marked-Codes."
  (case message
    (:which-operations '(:which-operations :print-self))
    (:print-self
     (catch-error
       (format (first args) "#<Code ~ ~>"
	       (marked-code-code record) t
	       (marked-code-marked record) t
       )
       nil
     )
    )
    (otherwise (ferror nil "Illegal message ~S send to a numbered component."
		       message
	       )
    )
  )
)

;-------------------------------------------------------------------------------
;;; Macros...

(defmacro multiple-value-prog2 (form value-form &rest body)
"Like multiple-value-prog1 only for prog2."
 `(multiple-value-prog1
    (progn ,form ,value-form)
    ,@body
  )
)


(defmacro code-and-marked ((trans num) form &body body)
"Defined trans and num to be the transformed code and the numbered code
 resulting from the evaluation of form, which returns a Marked-Code.  Body is
 executed in this context.
"
  `(let ((result ,form))
        (declare (unspecial result))
	(let ((,trans (marked-code-code   result))
	      (,num   (marked-code-marked result))
	     )
	     (declare (unspecial ,trans ,num))
	     ,@body
	)
   )
)

;-------------------------------------------------------------------------------
;;; The rest.
(defun make-method-name (spec)
"Given (foo :bar) from a defmethod, returns (:method foo :bar)."
  (if (consp (second spec))
      (cons :method (second spec)) ;;; Flavors method.
      #+CLOS
      (let ((qualifiers
	      (loop for x in (rest (rest spec)) while (not (consp x)) collect x)
	    )
	    (args (loop for arg in (find-if #'listp spec)
			until (member arg lambda-list-keywords)
			collect (if (consp arg) (second arg) t)
		  )
	    )
	   )
	  `(,(zwei:method-section-name-for-buffer)
	    ,(second spec)
	    ,@qualifiers
	    ,args
	   )
      )
  )
)

(defun source-code-debugging-enabled ()
"True when source code debugging is switched on."
  (and *source-code-debugging-enabled*
       (not (string-equal "SOURCE-CODE-DEBUGGING"
			  (pathname-name sys:fdefine-file-pathname)
	    )
       )
       (or (equal *source-code-debugging-enabled* :really)
	   (>= (compiler:opt-safety compiler:optimize-switch)
	       (compiler:opt-speed compiler:optimize-switch)
	   )
       )
  )
)

(defun get-method-body (spec)
  (if (consp (second spec))
      3 ;;; Flavors method.
      #+CLOS
      (let ((index (position (find-if #'listp spec) spec)))
	   (+ 1 index)
      )
  )
)

(defun enlist (marked-code)
"Given a marked code it returns its elements as a two-list."
  (list (marked-code-code marked-code) (marked-code-marked marked-code))
)

(defun map-for-values1 (predicate function list args)
"Called by map-for-values, actually does the work."
  (if (consp (rest list)) ;;; Watch out for dotted lists.
      (if (or (not predicate) (not (funcall predicate list)))
	  (let ((result (enlist (apply function (first list) args))))
	       (mapcar #'cons result
		       (map-for-values1 predicate function (rest list) args)
	       )
	  )
	  (Enlist (make-marked-code :Code list :marked nil))
      )
      (if (rest list)
	  (let ((result2 (enlist (apply function (first list) args)))
		(result3 (enlist (apply function (rest  list) args)))
	       )
	       (mapcar #'cons result2 result3)
	  )
	  (mapcar #'list (enlist (apply function (first list) args)))
      )
  )
)

(defun map-for-values (function list &rest args)
"A mapping function which applies function to elements of list and args.  The
 values of these functions are expected to be instances of marked-codes.  They
 are combined into a larger marked code, whose elements are lists resulting from
 the function applications.
"
  (if list
      (let ((body (map-for-values1 nil function list args)))
	   (make-marked-code :code (first body) :marked (second body))
      )
      (make-marked-code :code nil :marked nil)
  )
)

(defun map-for-values-with-stop-predicate (predicate function list &rest args)
"A mapping function which applies function to elements of list and args.  The
 values of these functions are expected to be instances of marked-codes.  They
 are combined into a larger marked code, whose elements are lists resulting from
 the function applications.
"
  (if list
      (let ((body (map-for-values1 predicate function list args)))
	   (make-marked-code :code (first body) :marked (second body))
      )
      (make-marked-code :code nil :marked nil)
  )
)

(defun new-number ()
"Returns a new unique PC index number and bumps the pc counter."
  (setq *source-code-index-counter*
	(+ 1 *source-code-index-counter*)
  )
  *source-code-index-counter*
)

;-------------------------------------------------------------------------------

(defflavor scd-stepper-common-lisp-cerror
	   (interesting-frame)
	   (common-lisp-cerror)
  :Initable-Instance-Variables
)

(defmethod (scd-stepper-common-lisp-cerror :find-current-frame) (sg)
  "A special version for stepper conditions, that points the stack at the
   frame we're really interested in, not the stepper code."
  ;; Returns four values, which are used to set *ERROR-LOCUS-FRAME*,
  ;; *CURRENT-FRAME*, *INNERMOST-VISIBLE-FRAME*
  ;; and *INNERMOST-FRAME-IS-INTERESTING*.
  (declare (values error-locus-frame current-frame innermost-visible-frame
		   innermost-frame-is-interesting))
  (let* ((frame (sg-top-frame sg)) 
	 (*innermost-visible-frame* frame))
    (declare (special *innermost-visible-frame*)) 
    ;; skip frame if a :ERROR-REPORTER function
    (do ((rp (sg-regular-pdl sg)))
	((not (let ((f (function-name (rp-function-word rp frame))))
		(if (symbolp f) (get f :error-reporter)))))
      ;; skip frames foothold -> fh-applier -> signal-condition
      ;; this is on the stack for microcode errors
      (and (eq (function-name (rp-function-word rp frame))
	       'signal-condition)
	   (eq (function-name (rp-function-word rp (sg-next-frame sg frame)))
	       'fh-applier)
	   (eq (function-name
		 (rp-function-word rp (sg-previous-nth-frame sg frame -2)))
	       'foothold)
	   (return (setq frame (sg-previous-nth-frame sg frame -3))))
      (setq frame (sg-next-frame sg frame)))
    (values frame interesting-frame ;(sg-out-to-interesting-frame sg frame)
	    *innermost-visible-frame* nil)))

(defmethod (exit-trap-error :after :find-current-frame) (ignore)
  (case *Source-Code-Debugging-Stepping-Mode-Enabled*
    (:End-Of-Function (setq *source-code-debugging-stepping-mode-enabled* t))
    (:Otherwise nil)
  )
)

;-------------------------------------------------------------------------------

;;; Stepper code.

(defvar *stepper-commands*
	'((#\space step-to-next)
	  (#\c-b step-to-debugger)
	  (#\resume proceed-to-end-of-this-function)
	  (#\c-resume proceed-switching-off-stepper)
	  (#\c-e edit-current-step-function)
	  (#\m-. edit-current-step-function)
	  (#\page step-clear-screen)
	  (#\c-l step-clear-screen)
	  (#\help step-help-command)
	 )
"The command alist for the SCD stepper."
)

(defun step-help-command (function sg fp frame)
  "The help command in the stepper."
  (ignore function sg fp frame)
  (format *error-output*
	  "~2%The following commands are defined:~%Key~15TChar~30TBinding"
  )
  (loop for (key function) in *Stepper-Commands*
	do (format *error-output* "~&~c~15T~S~30T~~A~"
		   key key (or (documentation function 'function) "")
	   )
  )
  (format *error-output* "~2%")
)

(Defun step-clear-screen (function sg rp frame)
  "Clears the screen and redisplays the
current frame's data."
  (ignore function sg rp frame)
  (send *error-output* :send-if-handles :Clear-Screen)
  (Info-About-Current-Frame 1)
  (terpri *error-output*)
  nil
)

(defun step-to-next (function sg rp frame)
  "Steps on to the next stepping point."
  (ignore function sg rp frame)
  t
)

(defun edit-current-step-function (function sg rp frame)
  "Edits the definition of the function
of the current frame."
  (ignore sg rp frame)
  (tv:edit function)
  nil
)

(defun proceed-to-end-of-this-function (function sg rp frame)
  "Proceeds to the end of the function currently
being executed. When you get there a breakpoint
is entered."
  (ignore rp)
  (format *error-output* "~&Proceed to end of ~S" function)
  (set-trap-on-exit sg frame t)
  (setq *source-code-debugging-stepping-mode-enabled* :end-of-function)
  t
)

(defun step-to-debugger (function sg rp frame)
  "Enters the debugger at the current state."
  (ignore function sg rp)
  (signal-proceed-case ((value)
			'scd-stepper-common-lisp-cerror
			:continue-format-string "Return to stepper"
			:format-string "Break into debugger from stepper"
			:Interesting-Frame frame
			:format-args nil)
    (:continue value)
  )
  nil
)

(defun proceed-switching-off-stepper (function sg rp frame)
  "Proceed switching off the stepper."
  (ignore function sg rp frame)
  (setq *Source-Code-Debugging-Stepping-Mode-Enabled* nil)
  t
)

(defun scd-stepper-cmd-loop (format-string &rest args)
  (info-about-current-frame)
  (apply #'format *error-output* format-string args)
  (loop for char = (progn (format *debug-io* "SCD> ") (read-char *debug-io*))
	for entry = (assoc char *Stepper-Commands*)
	do (if entry
	       (if (multiple-value-bind (function sg rp frame)
			 (find-interesting-frame-from-where-we-are -2)
		     (funcall (second entry) function sg rp frame)
		   )
		   (return nil)
		   nil
	       )
	       (beep) ;;; Command not found
	   )
  )
)

(defun info-about-current-frame (&optional (n-frames-back 0))
  (multiple-value-bind (function sg rp frame)
      (find-interesting-frame-from-where-we-are (- -3 n-frames-back))
    (ignore rp)
    (print-function-and-args sg frame)
    (let ((old #'sg-previous-frame))
         (letf ((#'sg-previous-frame
		 #'(lambda (sg this-frame &optional (innermost nil))
		     (if innermost
			 (funcall old sg this-frame innermost)
			 (funcall old sg this-frame frame)
		     )
		   )
		)
	       )
	       (print-source-code-or-disassembly
		 function (rp-location-counter-offset rp frame)
		 (get-stack-frame sg frame)
	       )
	 )
    )
  )
)

(Defun scd-step-before (number)
  (if (equal *source-code-debugging-stepping-mode-enabled* t)
      (scd-stepper-cmd-loop "~&Before:~%" number)
      nil
  )
)

(defun scd-step-after (number values)
  (if (equal *source-code-debugging-stepping-mode-enabled* t)
      (scd-stepper-cmd-loop "~&After :  ~{~S~^, ~}~%" values number)
      nil
  )
)

(defun scd-step-after-single-value (number value)
  (if (equal *source-code-debugging-stepping-mode-enabled* t)
      (scd-stepper-cmd-loop "~&After :  ~S~%" value number)
      nil
  )
)

(defun find-interesting-frame-from-where-we-are (&optional (previous-nth 1))
  (let ((sg (send current-process :stack-group)))
       (let ((innermost (sg-innermost-frame sg))
	     (rp (sys:sg-regular-pdl sg))
	    )
	    (let ((*current-frame*
		    (sg-previous-nth-frame sg innermost previous-nth
					   (sg-innermost-frame sg)
                    )
		  )
		 )
		 (let ((function (eh:rp-function-word rp eh:*current-frame*)))
		      (values function sg rp eh:*current-frame*)
		 )
	    )
       )
  )
)

(defmacro scd-wrapper (code number-1 number-2)
"Encapsulates the hair of the SCD code in a simple macro.  The numbers are
 the scd PC values for before and after the execution of Code.
"
  (if *source-code-debugging-stepping-mode-enabled* 
     `(progn
	(setq compiler:*source-code-pc* ,number-1)
	(scd-step-before ,number-1)
	(let ((.scd-stepper-values. (multiple-value-list ,code)))
	     (setq compiler:*source-code-pc* ,number-2)
	     (scd-step-after ,number-2 .scd-stepper-values.)
	     (values-list .scd-stepper-values.)
	)
      )
     `(multiple-value-prog2
	(setq compiler:*source-code-pc* ,number-1)
	,code
	(setq compiler:*source-code-pc* ,number-2)
      )
  )
)

(defmacro scd-body-wrapper (&body body)
"Used for the bodies of functions and such."
 `(let ((compiler:*source-code-pc* 0))
       (declare (special compiler:*source-code-pc*))
       ,@body
  )
)

(define-setf-method Scd-Wrapper (code number-1 number-2)
"Define a setf method for Scd-Wrapper so that scd on forms that setf things will
expand in a reasonable way.
"
  (let ((store-variable (gensym)))
       (Values nil
	       nil
	       (List store-variable)
	       (if *source-code-debugging-stepping-mode-enabled* 
		  `(progn
		     (setq compiler:*source-code-pc* ,number-1)
		     (scd-step-before ,number-1)
		     (let ((.scd-stepper-value. (setf ,code ,store-variable)))
			  (setq compiler:*source-code-pc* ,number-2)
			  (Scd-Step-After-Single-Value
			    ,number-2 .scd-stepper-value.
			  )
			  .scd-stepper-value.
		     )
		   )
		  `(prog2 ;multiple-value-
		     (setq compiler:*source-code-pc* ,number-1)
		     (setf ,code ,store-variable)
		     (setq compiler:*source-code-pc* ,number-2)
		   )
	       )
	       code
       )
  )  
)

(defun make-mark (code &optional numbered-code)
"Given a peice of code it marks it with PC incrementing code at either end.
 An instance of Marked-Code is returned with this marked code and the debugging
 info Numbered-Component.
"
  (let ((number-1 (new-number))
	(number-2 (new-number))
       )
       (make-marked-code
	 :Code `(scd-wrapper ,code ,number-1 ,number-2)
	 :marked
	 (make-numbered-component :numbers (list number-1 number-2)
				  :code (or numbered-code code)
	 )
       )
  )
)

;-------------------------------------------------------------------------------
; Walker code marker.
;-------------------------------------------------------------------------------

(defvar *all-symbol-macrolet-bindings* nil)

(defvar *scd-in-cw* nil)

(defun get-subforms-1 (form)
  (if (consp form)
      (progn (Get-Subforms (first form))
	     (Get-Subforms-1 (rest form))
      )
      (Get-Subforms form)
  )
)

(defun get-subforms (form)
  (declare (special *subforms*))
  (If (consp form)
      (cond ((member (first form) '(declare quote)) nil)
	    ((equal (first form) 'the)
	     (get-subforms (third form))
	    )
	    (t (pushnew form *Subforms*)
	       (Get-subforms-1 form)
	    )
      )
      (if (symbolp form) (pushnew form *Subforms*) Nil)
  )
)

(defun walker-function-processor (X)
  (declare (special *subforms* *all-forms-to-mark*))
  (if (member X *Subforms*)
      (pushnew x *all-forms-to-mark*) ;(Format t "~&Fn  - ~S" x))
      nil
  )
  (values x nil)
)

(defpackage scd-special-symbols)

(defun walker-var-processor (X)
  (declare (special *all-forms-to-mark* compiler:*loc*))
  (if (assoc X *All-Symbol-Macrolet-Bindings*)
      (progn (pushnew x *all-forms-to-mark*) ;(format t "~&Var - ~S" x)
	     (setf (first compiler:*loc*)
		   (list :Mark-This-One
			 (intern (let ((*package* nil)) (format nil "~S" x))
				 'scd-special-symbols
			 )
		   )
	     )
	     (values (first compiler:*loc*) nil)
      )
      (values x nil)
  )
)

(advise compiler:code-walk :Around :Foo nil
  (if (and *Scd-In-Cw*
	   (Not (equal (second arglist) 'walker-function-processor))
      )
      (Let ((cw-Fns (second arglist))
	    (cw-Vars (third arglist))
	   )
	   (setf (second arglist)
		 #'(lambda (X) (walker-function-processor X) (funcall cw-Fns X))
	   )
	   (setf (third arglist)
		 #'(lambda (X) (funcall cw-vars X) (walker-var-processor X))
	   )
      )
      nil
  )
  :Do-It
)

(advise ticlos:symbol-macrolet :Around :Foo nil
  (let-if *Scd-In-Cw*
	  ((*All-symbol-macrolet-bindings*
	     (append (second (first arglist))
		     (if (boundp '*all-symbol-macrolet-bindings*)
			 *all-symbol-macrolet-bindings*
			 nil
		     )
	     )
	    )
	   )
	   (let ((results (multiple-value-list :Do-It)))
	        (values-list results)
	   )
  )
)

(defun mark-all-forms-in (code to-mark)
  (if (consp code)
      (if (equal (first code) :Mark-This-One)
	  (make-mark (read-from-string (symbol-name (second code))))
	  (let ((new-code
		  (Map-For-Values-With-Stop-Predicate
		    #'(lambda (list)
			(and (equal (first list) '&aux)
			     (equal (second list)
				    '(compiler:*source-code-pc* 0)
			     )
			)
		      )
		    #'Mark-All-Forms-In code to-mark
		  )
		)
	       )
	       (if (member code to-mark)
		   (Code-And-Marked (code marked) new-code
		     (make-mark code marked)
		   )
		   new-code
	       )
	  )
      )
      (make-marked-code :code code :marked code)
  )
)

(defun Mark-Code (code)
  (let ((*scd-in-cw* t))
       (let ((*subforms* nil)
	     (*all-forms-to-mark* nil)
	     (*code-to-use* (copy-tree code))
	    )
	    (declare (special *code-to-use* *subforms* *all-forms-to-mark*))
	    (if *Scd-In-Cw* (Get-subforms *code-to-use*))
	    (Compiler:code-walk
	      *code-to-use* 'walker-function-processor 'walker-var-processor t
	    )
	    (mark-all-forms-in *code-to-use* *all-forms-to-mark*)
       )
  )
)

;-------------------------------------------------------------------------------
;-------------------------------------------------------------------------------
(defun body-from-numbered-form1 (form number match match-code)
"Called by body-from-numbered-form and actually does the work."
  (if (consp form)
      (multiple-value-bind (code new-match new-match-code)
	  (body-from-numbered-form1 (first form) number match match-code)
	(multiple-value-bind (tail-code tail-match tail-match-code)
	    (body-from-numbered-form1 (rest form) number
		 (if new-match new-match match)
		 (if new-match new-match-code match-code)
	    )
	  (values (cons code tail-code)
		  (if new-match new-match (if tail-match tail-match match))
		  (if new-match
		      new-match-code
		      (if tail-match tail-match-code match-code)
		  )
	  )
	)
      )
      (if (typep form 'numbered-component)
	  (let ((new-match (if (member number (numbered-component-numbers form))
			       form
			       match
			   )
		)
	       )
	       (multiple-value-bind
		 (numbered-code numbered-match numbered-match-code)
		   (body-from-numbered-form1 (numbered-component-code form)
						number new-match match-code
		   )
	         (values numbered-code
		         (if new-match new-match numbered-match)
			 (if new-match numbered-code numbered-match-code)
		 )
	       )
	  )
	  (values form match match-code)
      )
  )
)

(defun body-from-numbered-form (form &optional (number 0))
"Is passed a form and a PC index to it.  It returns three values; the code
 that represents the form, the marked region as specified by Number and the
 matching region of the first value which is Eq to the bit represented by
 Number.
"
  (body-from-numbered-form1 form number nil nil)
)

(defun post-process-transformed (index defn decls docs transformed)
"Given a peice of transformed code returns the function definition for it."
  `(,@(firstn index defn)
    ,@(if docs (list docs) nil)
    ,@decls 
    (let ((compiler:*source-code-pc* 0))
        ,@transformed
    )
   )
)


(defun post-process-marked (index defn decls docs numbered)
"Given a piece of marked code to be stashed on a function's plist it returns a
 suitable function definition for this purpose.
"
  `(,@(firstn index defn)
    ,@(if docs (list docs) nil)
    ,@decls
    ,@numbered
   )
)

(defun putise
       (name put-function form
	&optional (prop-name :source-code-debugger-numbered-source)
       )
"Given a function name and form it returns code to put form onto the Prop-Name
 property of Name's plist.
"
  (if (and (consp name) (equal (first name) :internal))
     `(setf (getf (sys:dbis-plist (sys:get-debug-info-struct ',name t))
		  ,prop-name
	    )
	    ,form
      )
     `(,put-function ',name ',form ,prop-name)
  )
)

(defun process-function-definition (name defn env counter)
"Given a function definition it parses its body and such-like, returning the
 values: decls, docs, marked code and new source code pc-index.
"
  (ignore name)
  (let ((*source-code-index-counter* (or counter 0))
	(*expansion-environment* env)
	(index (third (assoc (first defn)
			     *source-code-debuggable-forms*
		      )
	       )
	)
       )
       (multiple-value-bind (body decls docs)
	   (parse-body
	     (nthcdr (if (functionp index) (funcall index defn) index) defn)
	     *Expansion-Environment*
	   )
	 (Values decls docs (map-for-values #'mark-code body)
		 *source-code-index-counter*
	 )
      )
  )
)


(defun decls-say-I-can (decls)
"True if the decls sayt that it is reasonable to apply source code debugging
 to the body in question.
"
  (if decls
      (let ((optimisers (assoc 'optimize (rest (first decls)))))
	   (if optimisers
	       (let ((save compiler:optimize-switch))
		    (compiler:declare-optimize (rest optimisers))
		    (multiple-value-prog1 (source-code-debugging-enabled)
					  (setq compiler:optimize-switch save)
		    )
	       )
	       (decls-say-I-can (rest decls))
	   )
      )
      t
  )
)

(defun mark-function-definition
       (name put-function defn
	&optional (env sys:*macroexpand-environment*) (counter nil)
       )
"Takes a function definition, whose name is Name and marks its definition.
 Returns values of the marked function definition and the marks to stash on the
 plist of the function.  If it finds that it is not allowed to do source code
 debugging it returns the original function and nil.
"
  (let ((index (third (assoc (first defn)
			     *source-code-debuggable-forms*
		      )
	       )
	)
       )
       (let ((index (if (functionp index) (funcall index defn) index)))
	    (multiple-value-bind (decls docs new)
		(process-function-definition name defn env counter)
	      (if (decls-say-I-can decls)
		  (code-and-marked (transformed numbered) new
		    (let ((form1 (post-process-transformed
				   index defn decls docs transformed
				 )
			  )
			  (form2 (post-process-marked
				   index defn decls docs numbered
				 )
			  )
			 )
			 (if *print-marked-form* (pprint form1))
			 (if *print-marked-form* (pprint form2))
			 (values form1 (putise name put-function form2))
		    )
		  )
		  (values defn nil)
	      )
	    )
       )
  )
)

;;; Binds *source-code-debugging-enabled-here* for the duration of this loading
;;; of the file.  If it is modified to non-nil then it notifies that source
;;; code debugging was enabled i that file.

(let ((compiler:compile-encapsulations-flag t))
     (advise load :around :source-code-debugging nil
       (let ((*source-code-debugging-enabled-here* nil))
	    (declare (special *source-code-debugging-enabled-here*))
	    (let ((results (multiple-value-list :Do-It)))
		 (if *source-code-debugging-enabled-here*
		     (tv:notify tv:selected-window
				"Source code debugging enabled in file: ~A"
				(first arglist)
		     )
		     nil
		 )
		 (values-list results)
	    )
       )
     )
)

;;; Binds *source-code-debugging-enabled-here* for the duration of this loading
;;; of the file.  If it is modified to non-nil then it notifies that source
;;; code debugging was enabled i that file.

(let ((compiler:compile-encapsulations-flag t))
     (advise sys:fasload-1 :around :source-code-debugging nil
       (let ((*source-code-debugging-enabled-here* nil))
	    (declare (special *source-code-debugging-enabled-here*))
	    (let ((results (multiple-value-list :Do-It)))
		 (if *source-code-debugging-enabled-here*
		     (tv:notify tv:selected-window
				"Source code debugging enabled in file: ~A"
				(first arglist)
		     )
		     nil
		 )
		 (values-list results)
	    )
       )
     )
)

(defmacro maybe-catch-error (*catch-errors-p* &body forms)
"Catches errors in Forms if *catch-errors-p* is true."
 `(if ,*catch-errors-p*
      (catch-error (progn ,@forms) nil)
      (progn ,@forms)
  )
)

(defun mark-and-return-name (name put-function defn do-it-fun driver-closure)
"Given a function name and its definition it marks it and returns a suitable
 form to be compiled, i.e. the transformed code, the marked version, a setq
 to warn on file loading and the name of the function.
"
  (declare (special arglist))
  (multiple-value-bind (transformed marked)
      (mark-function-definition name put-function defn)
    (setf (first arglist)
	  `(progn ,transformed ,marked
		 (setq eh:*source-code-debugging-enabled-here* t)
		 ',(if marked
		       (if *print-names-of-source-code-debugged-forms*
			   (print `(:Debug ,name))
			   `(:Debug ,name)
		       )
		       name
		   )
	  )
    )
    (setf (second arglist) driver-closure)
    (let ((*marked-p* t)) (funcall do-it-fun))
  )
)

(defun make-debug-code (defn do-it-fun driver-closure)
"Called by an advise on Compile-Driver.  Checks to see whether it should
 do source code debugging and, if it should, compiles the thing with source
 code debugging decoration.  do-it-fun is the continuation passed in from the
 advise.
"
  (declare (special arglist *ignorable-warnings*
		    *still-source-code-debug-for-warnings*
		    *source-code-debuggable-forms* *catch-errors-p*
		    *source-code-debugging-enabled-here* *marked-p*
	   )
  )
  (multiple-value-bind (results error-p)
    (maybe-catch-error *catch-errors-p*
      (let ((name (funcall
		    (second (assoc (first defn) *source-code-debuggable-forms*))
		    defn
		  )
	    )
	    (put-function
	       (or (fourth (assoc (first defn) *source-code-debuggable-forms*))
		   'sys:function-spec-putprop
	       )
	    )
	    (old-warn #'compiler:warn)
	    (compiler:warn-on-errors nil)
	   )
	   (letf ((#'compiler:warn
		   #'(lambda (type &rest args)
		       (if (or (member type *ignorable-warnings*)
			       *still-source-code-debug-for-warnings*
			   )
			   (apply old-warn type args)
			   (apply #'cerror "Simply Proceed" (rest args))
		       )
		     )
		  )
		 )
		 (Mark-And-Return-Name
		   name put-function defn do-it-fun driver-closure
		 )
	   )
      )
    )
    (if error-p
	(let ((*marked-p* t))
	     (if *Print-Function-Names-That-Failed-To-Scd*
		 (compiler:warn 'source-code-debug-problem
				:probable-error
				"~&~S failed to source code debug compile."
				(firstn 3 defn)
		 )
		 nil
	     )
	     (Setf (first arglist) defn)
	     (funcall do-it-fun)
	)
	(values-list results)
    )
  )
)


;;; A wrapper around compile-driver which deals with source code debugging.
(let ((compiler:compile-encapsulations-flag t))
     (advise compiler:compile-driver :around :source-code-debugging nil
       (let ((do-it-fun #'(lambda () :do-it)))
	    (if (and (source-code-debugging-enabled)
		     (not *marked-p*)
		)
		(let ((defn (first arglist)))
		     (if (and (consp defn)
			      (symbolp (first defn))
			      (assoc (first defn)
				     *source-code-debuggable-forms*
			      )
			 )
			 (make-debug-code defn do-it-fun (second arglist))
			 :Do-It
		     )
		)
		:Do-It
	    )
       )
     )
)

(defparameter *scd-anyway* nil)

(advise compile-file :around :check-source-code-debugging nil
  (if (and (boundp '*source-code-debugging-enabled*)
	   *source-code-debugging-enabled*
	   (or *scd-anyway*
	       (with-timeout
		 (3600 (format *query-io* " - continuing without SCD.") nil)
		 (y-or-n-p "Source Code Debugging is currently enabled.  ~
			    Compile with source code debugging?"
		 )
	       )
	   )
      )
      :Do-It
      (let ((*source-code-debugging-enabled* nil))
	   (declare (special *source-code-debugging-enabled*))
	   :Do-It
      )
  )
)


(defun tv:current-pc (sg frame)
"Given a stack group and a frame number it returns the current value of the
 source code PC in that frame.
"
  (if (symeval-in-stack-group 'compiler:*source-code-pc* sg frame)
      (symeval-in-stack-group 'compiler:*source-code-pc* sg frame)
      0
  )
)

;-------------------------------------------------------------------------------

(defflavor show-stack-frame () (tv:inspection-data)
 (:Documentation "A flavor to show stack frames in the old way.")
)


(defmethod (show-stack-frame :middle-button-result) ()
  tv:data
)

(defmethod (show-stack-frame :format-concisely) (stream)
"Formats old type stack frames concisely."
  (non-source-code-stack-frame-print stream tv:data)
)

(defmethod (show-stack-frame :generate-item-specialized) (window)
"A specialised form of generate-item for stack frames."
  (let* ((sg (tv:stack-frame-stack-group tv:data))
	 (rp (eh:sg-regular-pdl sg))
	 (frame (tv:stack-frame-frame-number tv:data))
	 (function (eh:rp-function-word rp frame)))
    (cond ((consp function)
	   (send self :object-list function))
	  ((typep function 'compiled-function)
	   (tv:fef-display-list
	     function window (eh:rp-exit-pc rp frame)
	     (list nil nil nil nil fonts:hl12b 
		   (tv:stack-frame-function-name tv:data)))))))

(defmethod (show-stack-frame :generate-item) ()
"A stub for generate-item."
  nil
)

(defmethod (show-stack-frame :handle-mouse-click) (blip flavor-inspector)
"A mouse click handler for show-stack-frames."
  (let ((object (tv:find-inspection-object (send flavor-inspector :kbd-input))))
       (selector (fourth blip) =
	 (#\mouse-l-1 (send flavor-inspector :inspect-info-left-click))
	 (#\mouse-l-2 (send flavor-inspector :inspect-info-left-2-click object))
	 (#\mouse-m-1 (send flavor-inspector :inspect-info-middle-click))
	 (t (beep))
       )
  )
)

;;; Record the transformation between stack-frames and show-stack-frames.
tv:
(defperspective :disassembled-code (x show-x)
  :already-this-type-function (typep show-x 'eh:show-stack-frame)
  :show-x-type-for-perspective show-stack-frame
  :This-Perspective-Applicable-Function
    (and eh:*source-code-debugging-enabled*
	 (typep x 'tv:stack-frame)
	 (tv:debug-source-for-function
	   (tv:rp-function-word
	     (sys:sg-regular-pdl (tv:stack-frame-stack-group x))
	     (tv:stack-frame-frame-number x)
	   )
	 )
    )
  :menu-item-name "Disassembled Code"
  :New-Inspect-Function
    (tv:allocate-data 'eh:Show-stack-frame x)
  :Priority 5
)

;-------------------------------------------------------------------------------
(defflavor show-source-code () (tv:inspection-data)
 (:Documentation "A flavor to show a function's source code.")
)


(defmethod (show-source-code :middle-button-result) ()
  tv:data
)

(defmethod (show-source-code :format-concisely) (stream)
"Formats old type stack frames concisely."
  (format stream "Source code for ~" (list (function-name tv:data) t))
)

(defmethod (show-source-code :generate-item-specialized) (window)
"A specialised form of generate-item for stack frames."
  (let ((debugs (tv:debug-source-for-function tv:data)))
       (multiple-value-bind (code marked-region matched-code)
	  (funcall 'eh:body-from-numbered-form debugs)
	 (let ((*grinding-debugged-source-code*
		 (list marked-region matched-code)
	       )
	       (*current-pc* 0)
	      )
	      (declare (special *grinding-debugged-source-code*
				*current-pc*
		       )
	      )
	      (let ((results (multiple-value-list
			       (send window :object-list code)
			     )
		    )
		   )
		   (values-list
		     (append (list (first results) (second results)
				   (third results)
				   (fourth results)
				   `(:font fonts:hl12bi :String
				     ,(format nil "~s's Source code"
					      (function-name tv:data)
				      )
				    )
			     )
			     (nthcdr 5 results)
		     )
		   )
	      )
	 )
       )
  )
)

(defmethod (show-source-code :handle-mouse-click) (blip flavor-inspector)
"A mouse click handler for show-stack-frames."
  (let ((object (tv:find-inspection-object (send flavor-inspector :kbd-input))))
       (selector (fourth blip) =
	 (#\mouse-l-1 (send flavor-inspector :inspect-info-left-click))
	 (#\mouse-l-2 (send flavor-inspector :inspect-info-left-2-click object))
	 (#\mouse-m-1 (send flavor-inspector :inspect-info-middle-click))
	 (t (beep))
       )
  )
)


tv:
(defperspective :function-source-code (x show-x)
  :already-this-type-function (typep show-x 'eh:show-source-code)
  :show-x-type-for-perspective show-source-code
  :This-Perspective-Applicable-Function
    (and (or (and (compiled-function-p x)
		  (tv:debug-source-for-function x)
	     )
	     (and (symbolp x)
		  (sys:fdefinition-safe x)
		  (tv:debug-source-for-function (sys:fdefinition-safe x))
	     )
	 )
	 (fboundp 'eh:source-code-debugging-enabled)
	 (funcall 'eh:source-code-debugging-enabled)
    )
  :menu-item-name "Source Code"
  :New-Inspect-Function
    (tv:allocate-data 'eh:Show-source-code
		      (if (symbolp x) (sys:fdefinition-safe x) x)
    )
  :Priority 11
)


(defun non-source-code-stack-frame-print (stream frame)
"Prints out a stack-frame without source code debugging info."
  (let* ((frame-number (tv:stack-frame-frame-number frame))
	 (rp (eh:sg-regular-pdl (tv:stack-frame-stack-group frame)))
	 (function (eh:rp-function-word rp frame-number))
	 (pc (and (typep function 'compiled-function)  
		  (eh:rp-exit-pc rp frame-number)))
	 (*print-length* 5)
	 (*print-level* 3))
    (si:printing-random-object (frame stream :no-pointer)
      (format stream
	      "Stack-Frame ~ ~[PC=~D~;microcoded~;interpreted~]"
	      (list (function-name function) t)
	      (cond (pc 0)
		    ((typep function 'microcode-function) 1)
		    (t 2))
	      pc)
      )
    )
)

(defun source-code-stack-frame-print (stream frame)
"Prints out a stack-frame with source code debugging info."
  (let* ((frame-number (tv:stack-frame-frame-number frame))
	 (rp (sys:sg-regular-pdl (tv:stack-frame-stack-group frame)))
	 (function (eh:rp-function-word rp frame-number))
	 (pc (and (typep function 'compiled-function)  
		  (eh:rp-exit-pc rp frame-number)))
	 (*print-length* 5)
	 (*print-level* 3))
    (source-code-stack-frame-print-1 function stream frame frame-number pc)
  )
)

(defun source-code-stack-frame-print-1 (function stream frame frame-number pc)
  (if (and (source-code-debugging-enabled)
	   (tv:debug-source-for-function function)
	   (typep stream 'debugger-history-pane)
      )
      (si:printing-random-object (frame stream :no-pointer)
	(if (equal (aref (send stream :font-map)
			 tv:*font-for-selected-region*
		   )
		   (aref (send stream :font-map) 0)
	    )
	    (tv:set-font-map-if-you-must stream)
	    nil
	)
	(format stream
		"Stack-Frame ~ PC=~D ["
		(list (function-name function) t)
		pc
	)
	(let ((scd-pc (tv:current-pc
			(tv:stack-frame-stack-group frame)
			frame-number
		      )
	      )
	     )
	     (if (numberp scd-pc)
		 (if (evenp scd-pc)
		     (send stream :string-out
			   (tv:fontify-string
			     "After" (+ 1 tv:*font-for-selected-region*)
			   )
		     )
		     (send stream :string-out
			   (tv:fontify-string
			     "During" tv:*font-for-selected-region*
			   )
		     )
		 )
		 nil
	     )
	)
	(princ "]" stream)
      )
      (non-source-code-stack-frame-print stream frame)
  )
)

(defun print-out-a-stack-frame
    (message record &rest args)
"A new message handler for Stack-Frames."
  (case message
    (:which-operations '(:which-operations :print-self))
    (:print-self (source-code-stack-frame-print (first args) record))
    (otherwise (ferror nil "Illegal message ~S send to a stack frame."
		       message
	       )
    )
  )
)

;;; Put the new message handler on Stack-Frame's plist.
(putprop 'tv:stack-frame 'print-out-a-stack-frame 'named-structure-invoke)

;-------------------------------------------------------------------------------

(defun turn-source-code-debugging-off (&optional (silent-p t))
"Turns off source code debugging.  Unless Silent-p it tells you that it has done
 so.
"
  (setq eh:*source-code-debugging-enabled* nil)
  (proclaim '(optimize (safety 1) (speed 1)))
  (if (not silent-p) (zwei:typein-line "Source-Code Debugging Disabled.") nil)
)

(defun turn-source-code-debugging-on  (&optional (silent-p nil))
"Turns on source code debugging.  Unless Silent-p it tells you that it has done
 so.
"
  (setq eh:*source-code-debugging-enabled* t)
  (tv:enable-general-inspector)
  (proclaim '(optimize (safety 2) (speed 1)))
  (if (not silent-p) (zwei:typein-line "Source-Code Debugging Enabled.") nil)
)

zwei:
(defcom zwei:com-source-code-debugging
 "Toggle source-code debugging on or off." ()
  (if eh:*source-code-debugging-enabled*
      (eh:turn-source-code-debugging-off nil)
      (eh:turn-source-code-debugging-on  nil)
  )
  zwei:Dis-None
)

;;; Record new command.
(zwei:set-comtab zwei:*standard-comtab* nil
	    (zwei:make-command-alist '(zwei:com-source-code-debugging))
)

zwei:
(defcom zwei:com-source-code-debugging-stepper-mode
 "Toggle source-code debugging stepper mode on or off." ()
  (setq eh:*Source-Code-Debugging-Stepping-Mode-Enabled*
	(not eh:*Source-Code-Debugging-Stepping-Mode-Enabled*)
  )
  (format *query-io* "~&Source code debugging stepper mode ~A"
	  (if eh:*Source-Code-Debugging-Stepping-Mode-Enabled*
	      "enabled"
	      "disabled"
	  )
  )
  zwei:Dis-None
)

;;; Record new command.
(zwei:set-comtab zwei:*standard-comtab* nil
		 (zwei:make-command-alist
		   '(zwei:com-source-code-debugging-stepper-mode)
		 )
)

;-------------------------------------------------------------------------------
;;; Add SCD to the non window debugger.

;;; The following code was written by Jamie Zawinski.

(export '*ucl-debugger-source-code-fonts*)
(defvar *ucl-debugger-source-code-fonts* '(fonts:cptfont fonts:cptfontb)
  " A list of two fonts.  When the Source Code Debugging is on, and you are 
 examining the frame of a function that has been compiled with SCD, then the 
 commands which normally show disassembly will show the source code instead.
 The code will be shown in the first font, with currently executing part of the
 code in the second font.")

(profile:define-profile-variable *ucl-debugger-source-code-fonts* (:error)
  :cvv-type :list :documentation "The fonts in which the non-window-based debugger should show source-code.")


(DEFUN get-stack-frame (sg frame-number)
  "Given a stack group and a stack frame number, return a stack frame object."
  ;; This is based on an inspector function that did this, but also mucked about with an inspector pane.
  (let (function (label "") code)
    (setq function (rp-function-word (sg-regular-pdl sg) frame-number))
    ;; Set label to function name for frame
    ;; but print nothing for interpreted code
    (if (not (listp function))
        (let ((name (function-name function)))
          (setq label (cond ((stringp name) name)
                            ((symbolp name) (symbol-name name))
                            (t (format nil "~S" name))))))
    (setq code (tv:make-stack-frame :stack-group sg
				    :frame-number frame-number
				    :function-name label))))


(defun show-all-of-it (sg frame &optional no-disassembled-code)
  "Print function, args, locals, diassembled code."
   (let* ((rp (sg-regular-pdl sg))
	  (function (rp-function-word rp frame))
	  (pc-now (rp-location-counter-offset rp frame))
	  rest-arg-printed)
     ;; Print function name and last pc
     (print-function sg frame)
     (terpri) (terpri)
     ;; Print the arguments, including the rest-arg which is the first local
     (setq rest-arg-printed (print-frame-args sg frame 0))
     ;; Print the rest of the locals
     (print-frame-locals sg frame 0 rest-arg-printed)
     ;; Print disassembled (or source) code if we want it
     (unless no-disassembled-code
       (print-source-code-or-disassembly
	 function pc-now (get-stack-frame sg frame)))))


(defun print-source-code-or-disassembly (real-function pc-now sf)
  "If the given function has source-code debugging information, and SCD
 is turned on, then show its source code.  If the stream pointed at by
 *standard-output* is a SHEET, the write the current PC in font #2.
  If SCD is turned off or unavailable, show disassembly."
  (let* ((sg (tv:stack-frame-stack-group sf))
	 (rp (sg-regular-pdl sg))
	 (frame (tv:stack-frame-frame-number sf))
	 (function (rp-function-word rp frame)))
    (cond ((consp function)
	   (terpri)
	   (pprint function))
	  ((typep function 'compiled-function)
	   (let ((debugs (tv:debug-source-for-function function)))
	     (if (and debugs (eh:source-code-debugging-enabled))
		 (multiple-value-bind (code marked-region matched-code)
				      (eh:body-from-numbered-form
					debugs (tv:current-pc sg frame))
		   (let ((eh:*grinding-debugged-source-code*
			   (list marked-region matched-code))
			 (eh:*current-pc* (tv:current-pc sg frame)))
		     (declare (special eh:*current-pc*))
		     (let* ((nlines
			      (max *disassemble-instruction-count*
				   ;;; don't show absurdly few
				   (cond ((send *standard-output*
						:Operation-Handled-P
						:size-in-characters)
					  (multiple-value-bind (nil nlines)
					      (send *standard-output*
						    :size-in-characters)
					    (multiple-value-bind
					      (nil y-pos)
						(send *standard-output*
						      :Read-Cursorpos
						      :character)
					      ;;; Leave 1 line for prompt,
					      ;;; 2 for extra terpris.
					      (- nlines y-pos 3))))
					 (t 0))))
			    (width-in-chars
			      (1- (or (send *standard-output* :Send-If-Handles
					    :size-in-characters) 80)))
			    (strings (get-code-strings-in-region
				       code width-in-chars nlines)))
		       (terpri *standard-output*)
		       (let* ((old-font-map (send *standard-output*
						  :send-if-handles :font-map)))
			 ;;
			 ;; If the stream handles the :font-map message, then
			 ;; we temporarily swap in a new font map for drawing
			 ;; the code.
			 ;; Then we pprint the appropriate section, and swap
			 ;; back the old font map.
			 (unwind-protect
			     (let* ((n 0))
			       (when old-font-map
				 (send *standard-output* :Set-Font-Map
				       *ucl-debugger-source-code-fonts*))
			       (dolist (string strings)
				 (cond (old-font-map (send *standard-output*
							   :Fat-String-Out
							   string)
						     (terpri *standard-output*))
				       (t (write-line string
						      *standard-output*)))
				 (when (>= (incf n) nlines) (return))))
			   (when old-font-map
			     (send *standard-output*
				   :set-font-map old-font-map))))
		       )))
		 (print-disassembled-code real-function pc-now)))))))


(defun get-code-strings-in-region (code width-in-chars nlines)
  "Returns a list of fat-strings describing the given source-level-debuggable
 code.  There will not be more than NLINES of these strings, and they will be
 centered around the currently-executing code (that is, the code which will
 print in bold)."
  ;;
  ;; This is really gross.  I assume that there is some way to find out which
  ;; string represents the current PC by looking at the second value of
  ;; tv:grind-into-list, but I couldn't find it.  So, we look at the characters
  ;; in the strings, and decide that the lines representing the current PC are
  ;; the ones that contain characters with nonzero fonts.  Ack.  Pffft.
  ;;
  (let* ((strings (tv:grind-into-list code width-in-chars nil))
	 (length (length strings))
	 (rest strings)
	 (first-with-font-pos 0)
	 last-with-font-pos)
    (when (<= length nlines) (return-from get-code-strings-in-region strings))
    (flet ((zero-font (char)
	     "True if the font attribute of the character is zero."
	     (declare (character char) (optimize (speed 3)))
	     (zerop (char-font char))))
      ;; CDR down the list, stopping when we reach a string with a fonted char.
      (do* ()
	   ((null rest))
	  (when (typep (car rest) '(array sys:fat-char))
	    (unless (every #'zero-font (car rest)) (return))
	    (pop rest)
	    (incf first-with-font-pos)))
      (setq last-with-font-pos first-with-font-pos)
      ;; CDR down the list further, stopping when we reach a string
      ;; without any fonted chars.
      (do* ()
	   ((null rest))
	(unless (typep (car rest) '(array sys:fat-char)) (return))
	(when (every #'zero-font (car rest)) (return))
	(pop rest)
	(incf last-with-font-pos)))
    ;; Subseq into the whole list, centering the ``fonted window''.
    (let* ((center (round (+ first-with-font-pos last-with-font-pos) 2))
	   ;;; The midpoint of the fonted area.
	   (start (max 0 (- center (ceiling nlines 2))))
	   (end (min length (+ start nlines))))
      (when (< (- end start) nlines)
	(if (zerop start)
	    (incf end (- nlines (- end start)))
	    (decf start (- nlines (- end start)))))
      (subseq strings start end))))


;-------------------------------------------------------------------------------
;;; Patches to the code walker.
;-------------------------------------------------------------------------------

;;; By JPR.
(defun compiler:cw-local-function (exp)
  (declare (special eh:*all-forms-to-mark*))
  (if eh:*scd-in-cw*
      (pushnew exp eh:*all-forms-to-mark*)
      nil
  )
  (compiler:cw-eval-args exp)
)

compiler:
(defun compiler:cw-lambda-expression (exp)
  ;;  7/08/85 DNG - Fixed to recognize CLI:[NAMED-]LAMBDA as well as
  ;;                GLOBAL:[NAMED-]LAMBDA.
  ;;  7/22/86 DNG - Don't call CW-EXPRESSION on a function spec.
  ;; 12/31/86 DNG - Fix to handle (MACRO . (LAMBDA ...)).
  ;;  4/25/88 DNG - Modified for use with CODE-WALK.
  (declare (special eh:*code-to-use*))
  (let ((*local-variables* *local-variables*))
    (cond ((member (car exp)
		   '(global:lambda global:subst cli:lambda cli:subst)
		   :test #'eq
	   )
	   ;;; Addition here by JPR.
	   (if (and eh:*scd-in-cw*
		    (sys:is-in-tree exp eh:*code-to-use*)
		    (not (equal '(compiler:*source-code-pc* 0)
				(first (last (second exp)))
			 )
		    )
	       )
	       (if (member '&aux (second exp))
		   (nconc (second exp) '((compiler:*source-code-pc* 0)))
		   (nconc (second exp) '(&aux (compiler:*source-code-pc* 0)))
	       )
	       nil ;;; Do nothing
	   )
	   (let* ((expansion (cw-serial-binding (second exp) t))
		  (body-expansion (cw-clause (cddr exp))))
	     (if cw-return-expansion-flag
		 (list* (car exp)
			expansion
			body-expansion))))
	  ((member (car exp) '(global:named-lambda global:named-subst
						   named-lambda named-subst)
		   :test #'eq)
	   (let* ((expansion (cw-serial-binding (third exp) t))
		  (body-expansion (cw-clause (cdddr exp))))
	     (if cw-return-expansion-flag
		 (list* (first exp)
			(second exp)
			expansion
			body-expansion))))
	  ((eq (car exp) 'macro)
	   (if cw-return-expansion-flag
	       (cons 'macro (cw-lambda-expression (cdr exp)))
	     (cw-lambda-expression (cdr exp))))
	  ((lambda-macro-call-p exp)
	   (cw-lambda-expression
	     (lambda-macro-expand exp)))
	  ((validate-function-spec exp)
	   exp)
	  (t
	   ;; This is something invalid which will get a warning later.
	   (cw-expression exp)))))


Compiler:
(defsubst compiler:cw-eval-args (exp)
  (if cw-return-expansion-flag
      (if eh:*Scd-In-Cw*
	  ;;; This clause put in by JPR to allow side-effectig on the original
	  ;;; form
	  (cons (car exp)
		(loop for exp on (rest exp) collect
		      (let ((*loc* exp))
			   (declare (special *loc*))
			   (cw-expression (first exp))
		      )
		)
	  )
	  (cons (car exp) (mapcar #'cw-expression (cdr exp))))
    (if eh:*Scd-In-Cw*
	;;; This clause put in by JPR to allow side-effectig on the original
	;;; form
	(loop for exp on (rest exp) do
	      (let ((*loc* exp))
		   (declare (special *loc*))
		   (cw-expression (first exp))
	      )
	)
	(mapc #'cw-expression (cdr exp)))))

;;; This function not modified but must be recompiled to see the
;;; new version of the above defsubst.
compiler:
(defun compiler:cw-expression (exp &optional skip-handler &aux tem stop)
  ;; 10/18/86 DNG - Use si:args-desc instead of arglist to check for &quote args.
  ;;  1/28/87 DNG - Don't bind cw-function-environment to nil when expanding a local macro. [SPR 3088]
  ;; 12/11/87 DNG - Allow ALL-FUNCTIONS-TO-CHECK-FOR to be T to cause all to be returned.
  ;;		Modify update of ALL-FUNCTIONS to check for (SYMBOLP (CAR EXP))
  ;;		and use :TEST #'EQ for efficiency.
  ;;		Add :TEST #'EQ to the second PUSHNEW call for efficiency.
  ;;  4/26/88 DNG - Updated to support new interface function CODE-WALK.
  ;;  3/15/89 DNG - Use GET-FROM-FRAME-LIST.
  ;;  4/11/89 DNG - Add use of CW-EXTRA-ENVIRONMENT .
  ;;  4/18/89 DNG - Add use of WARN-ON-ERRORS; this is needed within SYMBOL-MACROLET.
  ;;  6/22/89 DNG - Fix to include CW-EXTRA-ENVIRONMENT in the environment 
  ;;		passed to MACROEXPAND-1.  Also use WITH-INTERPRETER-ENVIRONMENT instead 
  ;;		of WITH-STACK-LIST*.
  ;;  7/11/89 DNG - Don't invoke *cw-function-handler* on a local function.
  (typecase exp
    (symbol (unless (or (null exp)
			skip-handler
			(member exp *local-variables* :test #'eq))
	      (multiple-value-setq (exp stop)
		(funcall *cw-var-handler* exp))))
    (cons (unless (or skip-handler
		      (not (symbolp (car exp)))
		      (member (car exp) *local-functions* :test #'eq))
	    (case (car exp)
	      ( quote )
	      ( function (unless (member (second exp) *local-functions* :test #'equal)
			   (multiple-value-setq (exp stop)
			     (funcall *cw-function-handler* exp))))
	      ( t (unless (eq *cw-form-handler* #'identity)
		    (multiple-value-setq (exp stop)
		      (funcall *cw-form-handler* exp))))) )))
  (cond ((or (atom exp) stop)
	 exp)
	((consp (car exp))
	 ;; Explicit lambda-expression
	 (if cw-return-expansion-flag
	     (cons (cw-lambda-expression (car exp))
		   (mapcar #'cw-expression (cdr exp)))
	   (progn (cw-lambda-expression (car exp))
		  (mapc #'cw-expression (cdr exp)))))
	((nsymbolp (car exp))
	 (cw-eval-args exp))
	((setq tem (get-from-frame-list (locf (symbol-function (car exp)))
					cw-function-environment nil))
	 (if (eq (car-safe tem) 'macro)
	     ;; Local definition is a macro.  Call its expander.
	     (sys:with-interpreter-environment (si:*macroexpand-environment*
						     nil cw-function-environment cw-extra-environment)
	       (cw-expression (funcall (cdr tem) exp
				       si:*macroexpand-environment*)))
	   ;; Local definition is not a macro.  Assume it evals its args.
	     ;;; Mod here by JPR.  Used to be (cw-eval-args exp)
	   (cw-local-function exp)))
	((setq tem (get (car exp) 'cw-handler))
	 ;; special form with its own way of doing this.
	 (funcall tem exp))
	;;kludge to deal with &quote. Blech
	((and (fboundp (car exp))
	      (nth-value 3 (si:args-desc (car exp))))
	 (let ((quoted nil)
	       (tem (arglist (car exp) t)))
	   (flet ((frob (arg) (do ((x (pop tem) (pop tem)))
				  ((not (member x lambda-list-keywords :test #'eq))
				   (if quoted arg (cw-expression arg)))
				(cond ((eq x '&quote) (setq quoted t))
				      ((eq x '&eval) (setq quoted nil))))))
	     (if cw-return-expansion-flag
		 (cons (car exp) (mapcar #'frob (cdr exp)))
	       (mapc #'frob (cdr exp))))))
	((multiple-value-bind (v1 v2)
	     (sys:with-interpreter-environment (env nil cw-function-environment cw-extra-environment)
	       (if (eq (second (first eh:*condition-handlers*))
		       'warn-on-errors-condition-handler)
		   ;; If already within the WARN-ON-ERRORS in PRE-OPTIMIZE, need to handle 
		   ;; errors here so they get reported accurately instead of having 
		   ;; PRE-OPTIMIZE report a problem with the top-level macro.
		   (block warn
		     (WARN-ON-ERRORS ('MACRO-EXPANSION-ERROR "Error expanding macro ~S:" (car exp))
		       (return-from warn (macroexpand-1 exp env)))
		     ;; here if there was an error.
		     (return-from cw-expression `(ERROR-MACRO-EXPANDING ',exp)))
		 (macroexpand-1 exp env)))
	   (setq tem v1)
	   v2)
	 ;; Macro call.
	 (cw-expression tem))
	(t
	 (cw-eval-args exp))))

;-------------------------------------------------------------------------------

;;; Revert compile/load environment.
(eval-when (compile)
  (setq *source-code-debugging-enabled* *remember-enabled*)
)

;-------------------------------------------------------------------------------
;;; Patch

(defun eh:print-frame-locals (sg frame indent &optional dont-print-rest-arg)
  "Print the locals in FRAME, indenting lines by INDENT chars."
  (declare (special *error-print-level* *error-print-length*))
  (let* ((rp (sg-regular-pdl sg))
	 (function (rp-function-word rp frame)))
    ;; Print locals in frame
    (let ((*print-level* *error-print-level*)
	  (*print-length* *error-print-length*))
    (multiple-value-bind (rest-arg-value rest-arg-p lexpr-call)
	(sg-rest-arg-value sg frame)
      (ignore lexpr-call)
      (do ((n-locals (fef-number-of-locals function))
	       (local-idx (sys:rp-local-offset sg rp frame))
	       (i 0 (1+ i)))
	      ((>= i n-locals))
	    (cond ((not (and rest-arg-p (zerop i)))
		   (format t "~&~VTLocal ~D (~A): "
			   indent i (local-name function i))
		   (multiple-value-bind (value error-p)
		       (catch-error (aref rp (+ i local-idx)) nil)
		     (if error-p
			 (progn (princ "?") (terpri) (return nil))
			 (prin1 value))))))
      (if (and rest-arg-p (not dont-print-rest-arg))
	  (format t "~&Rest arg: ~S" rest-arg-value))))))
