;;; -*- Mode:Common-Lisp; Package:FILE-SYSTEM; Base:10; Fonts:(TVFONT) -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.  Where functionality implemented herein replicates
;;; similarly named functionality on Symbolics machines, this code was
;;; developed solely from the interface specification in the documentation
;;; or through guesswork, never by examination of Symbolics source code.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.
;;; **********************************************************************

;;; This software developed by:
;;;	James Rice
;;; at the Stanford University Knowledge Systems Lab in Jun '87.
;;;
;;; This work was supported in part by:
;;;	DARPA Grant F30602-85-C-0012

;;; This file contains the definition of the function search-and-replace,
;;; Which searches for strings in files and optionally replaces them with
;;; strings.

;;; Written by JPR on 16 June 87.

(load-tools '(map-over-files))

(proclaim '(optimize (speed 3) (safety 0)))

(defvar *search-and-replace-area*
	(sys:make-area
	  :name 'search-and-replace
	  :representation :structure
	  :gc :temporary
	  :force-temporary t
        )
)

(sys:reset-temporary-area *search-and-replace-area*)

(let ((*standard-output* 'si:null-stream))
     (declare (special *standard-output*))
     (mapcar #'(lambda (pkg)
		 (unintern 'ticl:search-and-replace (symbol-package pkg))
	       )
	       (where-is "SEARCH-AND-REPLACE")
     )
)

(defun Compare-String-Full
       (search-pattern case-sensitive-p line line-to
	line-start
       )
"
 Looks for Search-pattern in Line, starting at Line-start.  The search pattern
 can contain the wildcards.
"
  (if (listp search-pattern)
      (if search-pattern
	  (let ((search (first search-pattern)))
	       (case search
		 (:Multiple
		  (Compare-String-Full
		    (rest search-pattern) case-sensitive-p line
		    line-to line-start
		  )
		 )
		 (:Single
		  (multiple-value-bind (this-beginning this-end)
		    (Compare-String-Full
		      (rest search-pattern) case-sensitive-p line
		      line-to (+ 1 line-start)
		    )
		    (if (equal this-beginning (+ 2 line-start))
			(values line-start this-end)
			nil
		    )
		  )
		 )
		 (otherwise
		  (multiple-value-bind (this-beginning this-end)
		    (Compare-String-Optimised
		      search case-sensitive-p line line-to line-start
		    )
		    (if (not this-beginning)
			nil
			(if (rest search-pattern)
			    (multiple-value-bind (beginning end)
				(Compare-String-Full
				  (rest search-pattern) case-sensitive-p line
				  line-to this-end
				)
			      (if beginning
				  (values this-beginning end)
				  nil
			      )
			    )
			    (values this-beginning this-end)
			)
		    )
		  )
		 )
	       )
	  )
	  (values line-start line-start)
      )
      (Compare-String-Optimised
	search-pattern case-sensitive-p line line-to line-start
      )
  )
)


(defun print-fill-char-for (line index hpos fillchar)
"
 Prints out a fill char starting at the position on the line Hpos.  Tabs are
 taken into account.
"
  (if (eql (aref line index) #\Tab)
      (let ((target (- 8 (mod hpos 8))))
	   (loop for i from 0 to (- target 1)
		 do (format t fillchar)
	   )
	   (- target hpos)
      )
      (progn (format t fillchar)
	     1
      )
  )
)

(defun print-fill-chars-for (header line line-start start fillchar hpos)
"
 Prints out a collection of fill chars starting at the position on the line
 Hpos.  Tabs are taken into account.
"
  (let ((line (string-append header line)))
       (loop for i from 0 to (- start 1)
	     do (setq hpos
		      (+ hpos
			 (print-fill-char-for line (+ line-start i)
					      hpos fillchar
			 )
		      )
		)
       )
  )
  hpos
)


(defvar *line-print-out-function* 'print-out-line
"The function to use to print out a line and line number."
)

(defun print-out-line (stream header line file line-number index)
"The default line printing function."
  (ignore file line-number index)
  (format stream "~&~A" header)
  ;;; Print the line like this because sometimes it is an art-8b, not a string.
  (loop for i from 0 below (array-active-length line) do
	(format stream "~C" (aref line i))
  )
)

(defun Print-Underline-For-Internal
       (header line file line-number start end printed-p index)
"
 Prints out Line if printed-p is nil and then prints out under it a row of
 spaces and then a pair of uparrows showing where a match was found.
"
  (if printed-p
      nil
      (funcall *Line-Print-Out-Function* t header line file line-number index)
  )
  (format t "~&")
  (let ((hpos (print-fill-chars-for header line 0 start " " 0)))
       (format t "")
       (if (> (- (+ 1 end) start) 1)
	   (progn (print-fill-chars-for header line (+ 1 start)
					(- (+ 1 end) start 2) "_" (+ 1 hpos)
		  )
		  (format t "")
	   )
	   nil
       )
  )
)


(defun Print-Underline-For
       (file line line-from line-to start end printed-p line-number)
"
 Prints out Line if printed-p is nil and then prints out under it a row of
 spaces and then a pair of uparrows showing where a match was found.
"
  (let ((sys:default-cons-area *search-and-replace-area*))
       (let ((line (subseq line line-from line-to))
	     (header
	       (if line-number
		   (format nil "[~6D]" line-number)
		   ""
	       )
	     )
	     (start (- start line-from))
	     (end (- end line-from))
	    )
	    (print-underline-for-internal
	      header line file line-number
	      (if line-number (+ (length header) start) start)
	      (if line-number (+ (length header) end) end)
	      printed-p start
	    )
       )
  )
)


(defun perform-substitution
       (file line line-from line-to start end replace-string confirm-p silent-p
	line-number echo-replacement-p search-pattern case-sensitive-p
       )
"
 Substitutes Replace-String into Line to replace the substring from Start to
 End.  If Confirm-P is true then the user is asked whether the substitution
 should happen.  If Silent-p is true then the match is not printed out.
"
  (declare (values new-line modified-p new-from new-to next-index))
  (let ((replace-string
	  (if (consp replace-string)
	      (compute-replace-string search-pattern replace-string
				      case-sensitive-p line line-to start
              )
	      replace-string
	  )
	)
       )
       (let ((replace-length (length (the string replace-string))))
	    (if (eql replace-length (+ 1 (- end start)))
		(progn (if (and silent-p
			       (not (equal :Just-Print-Matching-Lines silent-p))
			   )
			   nil
			   (print-underline-for file line line-from line-to
						start end nil line-number
			   )
		       )
		       (if (or (not confirm-p) (y-or-n-p "Replace?"))
			   (progn (copy-array-portion
				    replace-string 0 replace-length line start
				    (+ start replace-length)
				  )
				  (values line t line-from line-to
					  (+ replace-length start)
				  )
			   )
			   (values line nil line-from line-to end)
		       )
		)
		(let ((new-line (make-array
				  (+ (- start line-from)
				     (- line-to (+ 1 end))
				     replace-length
				  )
				  :element-type 'string-char
				  :area *search-and-replace-area*
				)
		      )
		     )
		     (copy-array-portion
		       line line-from start new-line 0 (- start line-from)
		     )
		     (copy-array-portion
		       replace-string 0 replace-length new-line
		       (- start line-from) (+ (- start line-from)
					      replace-length)
		     )
		     (Copy-array-portion
		       line (+ 1 end) line-to new-line
		       (+ (- start line-from) replace-length)
		       (length (the string new-line))
		     )
		     (if (and silent-p
			      (not (equal :Just-Print-Matching-Lines silent-p))
			 )
			 nil
			 (print-underline-for file line line-from line-to
					      start end nil line-number
			 )
		     )
		     (let ((replace-p (if confirm-p (y-or-n-p "Replace?") t)))
			  (if (and replace-p echo-replacement-p)
			      (Print-Underline-For
				file new-line 0 (length (the string new-line))
				(- start line-from)
				(+ (- start line-from) replace-length)
				nil line-number
			      )
			      nil
			  )
			  (if replace-p
			     (values new-line t 0
				     (length (the string new-line))
				     (+ (- start line-from) replace-length)
			     )
			     (values line nil line-from line-to end)
			  )
		     )
		)
	    )
       )
  )
)

(defun compute-replace-string-1 
       (search-pattern replace-pattern case-sensitive-p line line-to line-start)
"Computes the replace string when we are patter matching.
The replace spec list is a list of either literal strings
to insert or two-lists which specify the start and end of
a subsequence in the source string to include.
"
  (declare (values start-of-match end-of-match replace-spec-list))
  (if search-pattern
      (let ((search (first search-pattern)))
	   (case search
	     (:Multiple
	      (multiple-value-bind (this-beginning this-end results)
	        (Compute-replace-string-1
		  (rest search-pattern) (rest replace-pattern)
		  case-sensitive-p line line-to line-start
		)
		(values this-beginning this-end
			(cons (list line-start this-beginning) results)
		)
	      )
	     )
	     (:Single
	      (multiple-value-bind (this-beginning this-end results)
		(Compute-replace-string-1
		  (rest search-pattern) (rest replace-pattern)
		  case-sensitive-p line line-to (+ 1 line-start)
		)
		(ignore this-beginning)
	        (values line-start this-end
			(cons (list line-start (+ 1 line-start)) results)
		)
	      )
	     )
	     (otherwise
	      (multiple-value-bind (this-beginning this-end)
		(Compare-String-Optimised
		  search case-sensitive-p line line-to line-start
		)
	        (if (rest search-pattern)
		    (multiple-value-bind (beginning end results)
			(Compute-replace-string-1
			  (rest search-pattern) (rest replace-pattern)
			  case-sensitive-p line line-to (+ 1 this-end)
			)
		      (if beginning
			  (values this-beginning end
				  (cons (first replace-pattern) results)
			  )
			  nil
		      )
		    )
		    (values this-beginning this-end replace-pattern)
		)
	      )
	     )
	   )
      )
      (values line-start line-start nil)
  )
)

(defun compute-replace-string
       (search-pattern replace-pattern case-sensitive-p line line-to line-start)
"Computes a replacement string given a pattern to replace for."
  (multiple-value-bind (match-start match-end substs)
      (Compute-Replace-String-1 Search-pattern replace-pattern case-sensitive-p
				line line-to line-start
      )
    (let ((match-length (loop for item in substs sum
			      (if (consp item)
				  (- (second item) (first item))
				  (length item)
			      )
			)
	  )
	 )
         (let ((string (make-array match-length :element-type 'string-char
				   :area *search-and-replace-area*
		       )
	       )
	       (index 0)
	      )
	      (loop for item in substs do
		    (if (consp item)
			(let ((delta (- (second item) (first item))))
			     (copy-array-portion
			       line (first item) (second item) string index
			       (+ index delta)
			     )
			     (setq index (+ index delta))
			)
			(let ((delta (length (the string item))))
			     (copy-array-portion
			       item 0 delta string index (+ index delta)
			     )
			     (setq index (+ index delta))
			)
		    )
	      )
	      (values string match-start match-end)
	 )
    )
  )
)

(defsubst fast-string-search (key string from to key-from key-to)
  (let ((key-len (- key-to key-from)))
    (cond
     ((= key-from key-to) (and (<= from to) from))
     (t (setq to (1+ (- to key-len)));Last position at which key may start +1
	(prog (ch1)
	      (cond
		((minusp to) (return ())))
	      (setq ch1 (aref key key-from))
	   loop
	      (or (setq from (%string-search-char ch1 string from to))
		  (return ()))
	      (and (%string-equal key key-from string from key-len)
		   (return from))
	      (setq from (1+ from))
	      (go loop))))))

;(defun xxx (str ignore)
;  (if (search "(UNLESS *close-data" str :Test #'char-equal)
;      (cl:break)))

sys:
(defmethod (sys:buffered-line-input-stream :Line-In-Super-Fast) ()
  (declare (special *new-line-delimiter*))
  (let ((string-index 0.)
	(new-string-index)
	(new-buffer-index)
	(old-string-p nil)
	(old-index nil)
	(cr-index)
	(old-cr-index))
    (unless readstring
      (setf readstring (get-readstring)))
    (loop
      (unless (and stream-input-buffer
		    (< stream-input-index stream-input-limit))
	;Out of input, get some more
	(if old-index
	    (progn (setq old-string-p t)
		   (copy-array-portion
		     stream-input-buffer old-index stream-input-limit
		     readstring 0 (- stream-input-limit old-index))
		   (setf (fill-pointer readstring)
			 (- stream-input-limit old-index))))
	(unless (send self :setup-next-input-buffer)
	  (setf (fill-pointer readstring) string-index)
;	  (Xxx readstring :a)
	  (return (values readstring t 0 string-index))))
      (setf old-cr-index cr-index)
      (setf cr-index
	    (%string-search-char #\newline ;;; *new-line-delimiter*
		 stream-input-buffer stream-input-index stream-input-limit))
      (if cr-index
	  (setq new-buffer-index (1+ cr-index)
		new-string-index
		(+ string-index (- cr-index stream-input-index))) 
	  (setq new-buffer-index stream-input-limit
		new-string-index
		(+ string-index (- stream-input-limit stream-input-index))))
      (when (> new-string-index (array-total-size readstring))
	(adjust-array readstring new-string-index))
      (setq old-index stream-input-index)
      (setq stream-input-index new-buffer-index
	    string-index new-string-index)
      (when cr-index
	(if old-string-p
	    (let ((old-end (fill-pointer readstring)))
	         (let ((new-end (- (+ old-end cr-index) old-index)))
		      (setq old-string-p nil)
		      (if (>= new-end (array-total-size readstring))
			  (setq readstring
				(adjust-array readstring (+ 100 new-end))
			  )
			  nil
		      )
;		      (Xxx readstring :b1)
		      (setf (fill-pointer readstring) new-end)
		      (copy-array-portion
			stream-input-buffer old-index cr-index ;;; old-index used to be 0.  JPR.
			readstring old-end new-end
		      )
;		      (Xxx readstring :b2)
		      (return (values readstring nil 0 new-end))))
	    (progn ;(Xxx readstring :c)
		   (return (values stream-input-buffer nil
			    old-index cr-index))))))))


(defun Compare-String-Optimised
       (search-string case-sensitive-p line to start)
"
 Looks for Search-string in Line starting from Start, perhaps case sensitively.
 It returns the values; start index and end index of the pattern or nil, nil
 if it is not found.
"
  (let ((found-p
	  (let ((alphabetic-case-affects-string-comparison case-sensitive-p))
	       (fast-String-Search search-string line start to
				   0 (array-active-length search-string)
               )
          )
	)
       )
       (if found-p
	   (values found-p
		   (- (+ found-p (length (the string search-string))) 1)
	   )
	   (values nil nil)
       )
  )
)


(defun search-for-string-in-line
  (file search-strings case-sensitive-p line line-from line-to silent-p
   line-number wild-p
  )
"
 Searches for Search-String in Line, perhaps case sensitively.  If silent-p is
 nil then the places where the matches are found are printed out.
"
  (let ((printed-p nil))
       (loop for search-string in search-strings do
	     (if (and wild-p (consp search-string))
		 (loop for index from line-from
		       to (- line-to 1)
		       until (>= index line-to) do
		       (multiple-value-bind (beginning end)
			    (Compare-String-Full
			      search-string case-sensitive-p
			      line line-to index
			    )
			    (if beginning
				(progn
				  (if (and silent-p
				       (not (equal :Just-Print-Matching-Lines
						   silent-p
					    )
				       )
				      )
				      nil
				      (Print-Underline-For
					file line line-from line-to beginning
					end printed-p line-number
				      )
				  )
				  (setq printed-p t)
				  (setq index beginning)
				)
				(return nil)
			    )
		       )
		 )
		 (loop for index from line-from
		       until (>= index line-to)
		       do (multiple-value-bind (beginning end)
			    (compare-string-optimised
			      search-string case-sensitive-p line
			      line-to index
			    )
			    (if beginning
				(progn
				  (if (and silent-p
				       (not (equal :Just-Print-Matching-Lines
						   silent-p
					    )
				       )
				      )
				      nil
				      (Print-Underline-For
					file line line-from line-to beginning
					end printed-p line-number
				      )
				  )
				  (setq printed-p t)
				  (setq index beginning)
				)
				(return nil)
			    )
			  )
		 )
	     )
       )
       printed-p
  )
)


(defun search-for-string-in-line-and-replace
       (file search-strings replace-strings case-sensitive-p line line-from
	line-to line-start confirm-p silent-p line-number wild-p
	echo-replacement-p
       )
"
 Searches for Search-String and replaces it with Replace-String in Line.
 If Case-Sensitive-P is true then the comparison of Search-String with Line is
 case sensitive.  If Confirm-P is true then the userr is askewd before a change
 is made.  If Silent-P is true then the changes are not printed out.
"
  (let ((modified-p nil))
       (loop for search-string in search-strings
	     for replace-string in replace-strings
	     for optimise-p = (and (not wild-p) (not (consp search-string)))
	     do
	     (loop for index from line-start
		   to line-to do
		   (multiple-value-bind (start end)
		       (if optimise-p
			   (compare-string-optimised
			     search-string case-sensitive-p line line-to index
			   )
			   (compare-string-full
			     search-string case-sensitive-p line line-to
			     index
			   )
		       )
		     (if start
			 (multiple-value-bind (new-line maybe-modified new-from
					       new-to next-index
					      )
			     (perform-substitution
			       file line line-from line-to start end
			       replace-string confirm-p silent-p line-number
			       echo-replacement-p search-string case-sensitive-p
			     )
			   (setq line new-line)
			   (setq line-from new-from)
			   (setq line-to new-to)
			   (if maybe-modified
			       (progn (setq modified-p t)
				      (setq line-start next-index)
				      (setq index next-index)
			       )
			       (setq index next-index)
			   )
			 )
			 (return nil)
		     )
		   )
	     )
       )
       (if modified-p
	   (multiple-value-bind (new-modified new-line new-from new-to)
	       (search-for-string-in-line-and-replace
		 file search-strings replace-strings case-sensitive-p
		 line line-from line-to line-start confirm-p silent-p
		 line-number wild-p echo-replacement-p
	       )
	     (if new-modified
		 (values t new-line new-from new-to)
		 (values t line line-from line-to)
	     )
	   )
	   (values nil line line-from line-to)
       )
  )
)
           


(defun search-with-replace-in-file
       (file search-string case-sensitive-p replace-string
	confirm-p silent-p line-numbers-p wild-p echo-replacement-p
       )
"
 Searches for Search-String and replaces all occurences of it with
 Replace-String, either unconditionally or if Confirm-p is true then
 if the user says so.  If Case-Sensitive-P is true then the comparison
 of the strings is case sensitive.
"
  (let ((modified-p nil)
	(line-number 0)
	(out-path nil)
	(new-file (send (send file :new-version :newest) :string-for-printing))
       )
       (with-open-file (instream file :Direction :Input)
	 (with-open-file (outstream file
				    :Direction :Output
				    :If-Exists :New-Version
			 )
	   (setq out-path (send outstream :truename))
	   (loop with line with eof-p with from with to
		 do (multiple-value-setq (line eof-p from to)
		      (send instream :line-in-super-fast)
		    )
		 until (and eof-p (string-equal line ""))
		 do (setq line-number (+ 1 line-number))
		    (multiple-value-bind (modified new-line new-from new-to)
			(search-for-string-in-line-and-replace
			  new-file search-string replace-string
			  case-sensitive-p line from to from confirm-p silent-p
			  (if line-numbers-p line-number nil) wild-p
			  echo-replacement-p
			)
			(if modified
			    (setq modified-p t)
			    nil
			)
		       (send outstream :line-out new-line new-from new-to)
		       (sys:reset-temporary-area *search-and-replace-area*)
		    )
	   )
	   (close outstream :Abort (not modified-p))
         )
       )
       (if modified-p out-path nil)
  )
)


(defun stretch-replaces (searches replaces)
"Returns a list of replace strings as long as searches."
  (if (equal nil searches)
      nil
      (if (equal nil (rest replaces))
	  (cons (first replaces) (stretch-replaces (rest searches) replaces))
	  (cons (first replaces)
		(stretch-replaces (rest searches) (rest replaces))
	  )
      )
  )
)



(defun Search-Without-Replace-In-File
  (file search-strings case-sensitive-p silent-p
   line-numbers-p wild-p search-not
  )
  (with-open-file (instream file :Direction :Input)
    (let ((found-p nil)
	  (line-number 0)
	 )
	 (loop with line with eof-p with from with to
	       do (multiple-value-setq (line eof-p from to)
		    (send instream :line-in-super-fast)
		  )
	       until (or (and eof-p (string-equal line ""))
			 (and silent-p
			      (not (equal :just-print-matching-lines silent-p))
			      found-p
			 )
		     )
	       do (setq line-number (+ 1 line-number))
		  (if (search-for-string-in-line file search-strings
						 case-sensitive-p line from to
						 silent-p
						 (if line-numbers-p
						     line-number
						     nil
						 )
						 wild-p
		      )
		      (setq found-p t)
		  )
	 )
	 (if search-not
	     (if found-p nil file)
	     (if found-p file nil)
	 )
    )
  )
)

(defvar *wildcard-char-quoting-character* #\\
"The character used to preceed a wildchar in a search
 string to make it a literal.
"
)

(defun parse-search-string (string wildcards search-start this-string-start)
"Turns search strings into lists that specify the strings to search for.  For
example: the string \"Hello*Jim?what's up\" will translate into
  (\"Hello\" :Multiple \"Jim\" :single \"what's up\")
Wild chars can be quoted so that \"Hello\*Jim?what's up\" will translate
into (\"Hello*Jim\" :single \"what's up\").
"
  (let ((position (string-search-set wildcards string search-start)))
       (if position
	   (if (or (= position 0)
		   (and (> position 0)
		        (not (equal *wildcard-char-quoting-character*
				    (aref string (- position 1))
			     )
			)
		   )
	       )
	       (let ((part (subseq (the string string)
				   this-string-start position
			   )
		     )
		     (body
		       (cons (if (equal (first wildcards)
					(aref string position)
				 )
				 :Multiple
				 :Single
			     )
			     (Parse-Search-String
			       string wildcards (+ 1 position)
			       (+ 1 position)
			     )
		       )
		     )
		    )
		    (if (equal part "") body (cons part body))
	       )
	       (Parse-Search-String
		 (string-append
		   (subseq (the string string) this-string-start (- position 1))
		   (subseq (the string string) position)
		 )
		 wildcards (- (+ 1 position) this-string-start) 0
	       )
	   )
	   (if (equal this-string-start 0)
	       string
	       (let ((part (subseq (the string string) this-string-start)))
		    (if (equal part "")
			nil
			(list (subseq (the string string) this-string-start))
		    )
	       )
	   )
       )
  )
)

(defun search-and-replace-in-file
       (file search-strings case-sensitive-p replace-strings
	confirm-p silent-p line-numbers-p echo-replacement-p
	search-not wild-p
       )
"
 This is the function used by search-and-replace as an argument to
 Map-Over-Files.  It decides whether to do a search and replace or
 simply a search and does the appropriate thing.
"
  (if replace-strings
      (if (search-without-replace-in-file
	    file search-strings case-sensitive-p
	    t nil wild-p search-not
	  )
	  (search-with-replace-in-file
	    file search-strings case-sensitive-p
	    replace-strings
	    confirm-p silent-p line-numbers-p wild-p
	    echo-replacement-p
	  )
	  nil
      )
      (search-without-replace-in-file
	file search-strings case-sensitive-p silent-p
	line-numbers-p wild-p search-not
      )
  )
)


(defun list-of-strings (something)
"Returns a list of strings, whether something is a listof strings or a string."
  (typecase something
    (string (list something))
    (list (mapcar #'(lambda (x)
		      (if (stringp x) x (ferror nil "~S should be a string." x))
		    )
		    something
	  )
    )
    (otherwise (ferror nil "~S should be a string." something))
  )
)

(defvar *default-search-and-replace-wildcards* '(#\* #\?)
"The wildcards to use by default for search-and-replace operations.
The first element is the multi-char wildcard, the second is the single char
wildcard.
"
)

(defun ticl:search-and-replace
     (search-string files &Key
      (replace-string nil)
      (confirm-replace-p t)
      (exclude nil)
      (case-sensitive-p nil)
      (line-numbers-p t)
      (silent-p nil)
      (wildcards *default-search-and-replace-wildcards*)
      (confirm-files-p nil)
      (remote-host nil)
      (Remote-Bindings nil)
      (echo-replacement-p nil)
      (search-not nil)
     )
"
 Searches over the files in Files for Search-String.

 Search-String     - Can have two sorts of wildcards; \"?\", which is a single
                     character wildcard, and \"*\", which is a multi
                     character wildcard.  This argument can be a string or a
                     list of strings.

 Files             - Is a string/pathname or list of strings/pathnames, denoting
                     the files to be searched.

 Returns either a) the list of these files in which the search string was found
 if no Replace-String is specified or b) the list of files for which a
 replacement was performed if Replace-String is specified.

 Takes the following keyword args:-
 Replace-String    - If there is a Replace-String the replacement is made
                     either unconditionally or according to Confirm-Replace-P.
                     If this is a list of strings then the strings are matched
                     for replacement with the search strings by their position
                     in the list.  The last element in the list of replace
                     strings is used for replacement for all of those elements
                     in the list of search strings, which do not have
                     associated replace strings.  The replace string can be
                     wildcarded in the same way that the search string is, but
                     the shape of the pattern must be congruent with that of
                     the search string.  For example, you could search and
                     and replace the following patterns:
                          \"Hello*Jim\" -> \"Good*Bye\"
                          \"Hello*Jim\" -> \"Goodbye\" ; because no wildcards
                     but not the pattern:
                          \"Hello*Jim\" -> \"Good?Bye\"
 Confirm-Replace-P - Replacements of strings are made unconditionally if this
                     is Nil, otherwise the user is asked to confirm that the
                     substitution should take place.
 Exclude           - A filespec, like that given to Map-Over-Files.  The files
                     specified are not Search-And-Replaced.
 Case-Sensitive-P  - If this is non-nil then string comparisons are case
                     sensitive, otherwise they are case insensitive.
 Line-Numbers-P    - If true then line numbers are printed out before the
                     matching lines.
 Silent-P          - If Silent-p is true then the matches are not printed, nor
                     are the names of the files searched.  This cannot be
                     non-nil if Confirm-Replace-P is true.  If this is
                     :Just-Print-Matching-Lines then only the mayching lines are
                     echoed, and the file names searched are not.
 Wildcards         - A list of two characters to be used as wildcard characters.
                     The first is the multiple character wildcard, the second is
                     the single character wildcard.  Note: wildcard chars can be
                     quoted in search strings using
                     *wildcard-char-quoting-character* (default =#\\).  If you
                     are just typing a call to search-and-replace to a listener
                     then you'll need to quote the \ char, i.e. type in
                     \"Hello\\*Jim\" in order for the reader to read the string
                     \"Hello\*Jim\", which will search for the literal string
                     \"Hello*Jim\".
 Confirm-Files-P   - If true then the user is prompted before each file is
                     searched.
 Echo-replacement-p- When true causes the strings that have been replaced to
                     be echoed after the replacement.
 Search-Not        - If true will find those files not including the string.
 Remote-Host       - The same as the remote-host argument for map-over-files.
 Remote-Bindings   - The same as the remote-bindings argument to map-over-files.

"
  (if (not search-string) (ferror nil "No search string provided."))
  (check-type wildcards cons)
  (check-type (first wildcards) character)
  (check-type (second wildcards) character)
  (assert (>= (length (the string search-string)) 1) (search-string)
          "Illegal search string ~S provided." search-string
  )
  (assert (not (and search-not replace-string)) (search-not replace-string)
          ":Search-Not argument cannot be used with :Replace-String."
  )
  (assert (not (and silent-p replace-string confirm-replace-p))
	  (silent-p replace-string confirm-replace-p)
	  "Silent-p and Confirm-Replace-P arguments are incompatible."
  )
  (let ((search-strings (list-of-strings search-string))
	(short-replace-strings (list-of-strings replace-string))
       )
       (let ((replace-strings
	       (if replace-string
		   (stretch-replaces search-strings short-replace-strings)
		   nil
	       )
	     )
	    )
	    (let ((wild-p (find-if
			    #'(lambda (search-string)
				(string-search-set wildcards search-string)
			      )
			    search-strings
			  )
		  )
		 )
		 (let ((parsed-search-strings
			 (if wild-p
			     (mapcar
			       #'(lambda (string)
				   (Parse-Search-String string wildcards 0 0)
				 )
			       search-strings
			     )
			     search-strings
			 )
		       )
		       (parsed-replace-strings
			 (mapcar #'(lambda (string)
				     (Parse-Search-String string wildcards 0 0)
				   )
				   replace-strings
			 )
		       )
		      )
		      (if parsed-replace-strings
			  (loop for s in parsed-search-strings
				for r in parsed-replace-strings
				when (consp r)
				do (assert (and (consp s)
						(equal (length s) (length r))
						(loop for s1 in s
						      for r1 in r
						      always
							(or (and (stringp s1)
								 (stringp r1)
							    )
							    (eq s1 r1)
							)
						)
					   )
					   ()
					   "Search and replace strings ~S and ~
                                           ~S are not congruent."
					   search-string replace-string
				   )
			  )
			  nil
		      )
		      (remove nil
			      (map-over-files #'search-and-replace-in-file files
					      :exclude exclude
					      :log (not silent-p)
					      :confirm confirm-files-p
					      :Remote-Host remote-host
					      :Remote-Bindings Remote-Bindings
					      :Arguments
					        (list parsed-search-strings
						      case-sensitive-p
						      parsed-replace-strings
						      confirm-replace-p
						      silent-p
						      line-numbers-p
						      echo-replacement-p
						      search-not wild-p
						)
			      )
		      )
		 )
	    )
       )
  )
)

;-------------------------------------------------------------------------------

(defvar zwei:*search-and-replace-directory* nil)

(defun find-tag-table (name)
  (rest (assoc name zwei:*zmacs-tag-table-alist* :test #'equal))
)

(defun new-versionify (path)
  (let ((path (fs:default-pathname path)))
       (if (equal (probe-file path)
		  (probe-file (send path :new-version :newest))
	   )
	   (send path :new-version :newest)
	   path
       )
  )
)

(defun make-suitable-tag-table (matches name)
  (let ((newestised (mapcar 'new-versionify matches)))
       (zwei:select-file-list-as-tag-table newestised name)
  )
)

(defun load-up-this-file (file)
  (let ((buffer (zwei:load-file-into-zmacs (new-versionify file) nil t)))
       (zwei:make-buffer-current buffer)
  )
)

(defun load-up-this-file-for-line (file-and-line)
  (let ((file (first file-and-line))
	(line (second file-and-line))
	(index (third file-and-line))
       )
       (let ((buffer (zwei:load-file-into-zmacs (new-versionify file) nil t)))
	    (zwei:make-buffer-current buffer)
	    (zwei:move-bp (zwei:point) (zwei:interval-first-bp buffer))
	    (zwei:down-real-line (max 0 (- line 1)))
	    (zwei:move-bp (zwei:point) (zwei:bp-line (zwei:point)) index)
       )
  )
)

(defun maybe-add-item-type-1 (window)
  (let ((item-types (send window :item-type-alist)))
       (if (or (not (assoc 'edit-this-file item-types))
	       (not (assoc 'edit-this-line item-types))
	   )
	   (send window :Set-Item-Type-Alist
		 (nconc
		    item-types
		    (list (list 'edit-this-file 'Load-Up-This-File
				"L: Edit this file."
			  )
			  (list 'edit-this-line 'load-up-this-file-for-line
				"L: Go to this line."
		          )
		    )
		 )
	   )
	   nil
       )
  )
)

(defun maybe-add-item-type ()
  (let ((str (sys:follow-syn-stream *standard-output*)))
       (maybe-add-item-type-1 str)
  )
)


(defvar *Search-Files-Default-File-Name* :Wild
"The default file name to search for with search-and-replace in zmacs."
)

(defvar *Search-Files-Default-File-Type* :Lisp
"The default file type to search for with search-and-replace in zmacs."
)

(defvar *Search-Files-Default-File-Version* :Newest
"The default file version to search for with search-and-replace in zmacs."
)

(defun itemise-file-name-internal (stream control-string newline-p args)
  (let ((str (if (streamp (sys:follow-syn-stream stream))
		 (sys:follow-syn-stream stream)
		 (sys:follow-syn-stream *standard-output*)
	     )
	)
       )
       (if (send str :operation-handled-p :item)
	   (progn (if newline-p (format str "~&") nil)
		  (lexpr-send str :item 'edit-this-file
		    (first args) control-string args
		  )
	   )
	   (apply 'format str control-string args)
       )
  )
)

(defun itemise-file-name (stream control-string &rest args)
  (Itemise-File-Name-Internal stream control-string t args)
)

(defun double-up-~s (string)
  (let ((index (position #\~ string :Test #'char=)))
       (if index
	   (string-append
	     (subseq string 0 index)
	     "~~"
	     (double-up-~s (nsubstring string (+ index 1)))
	   )
	   string
       )
  )
)

(defun itemise-line (stream header line file line-number index)
  (let ((str (if (streamp (sys:follow-syn-stream stream))
		 (sys:follow-syn-stream stream)
		 (sys:follow-syn-stream *standard-output*)
	     )
	)
       )
       (if (send str :operation-handled-p :item)
	   (progn (format str "~&~A" header)
		  (send str :item 'edit-this-line
			(list file line-number index)
			(double-up-~s
			  (with-output-to-string (str)
			    (loop for i from 0 below (array-active-length line)
				  do (format str "~C" (aref line i))
			    )
			  )
			)
		  )
	   )
	   (Print-Out-Line stream header line file line-number index)
       )
  )
)

(defun server-itemise-line (stream &rest args)
  (Server-Form-Reply stream 'fs:*line-print-out-function* args)
)

(defun system-source-files (system-name)
  (Si:system-source-files system-name si:*source-file-types* nil t)
)

(defun get-replace-string ()
  (let ((default (or (and (boundp 'zwei:*tags-query-replace-to*)
			  zwei:*tags-query-replace-to*
		     )
		     zwei:*zmacs-tags-search-key-string*
		 )
	)
       )
       (if default
	   (let ((string
		   (zwei:typein-line-readline
		     "Replace some occurrences of \"~A\" with (default \"~A\"):"
		     zwei:*zmacs-tags-search-key* default
		   )
		 )
		)
		(if (equal "" string) default string)
	   )
	   (zwei:typein-line-readline
	     "Replace some occurrences of \"~A\" with:"
	     zwei:*zmacs-tags-search-key*
	   )
       )
  )
)

(defun print-out-matching-files (files make-tag-table-p)
  (format t "~%Matching files :- ")
  (loop for path in files do
	(format t "~%	")
	(Itemise-File-Name-Internal *standard-output* "~A" nil
				    (list (send path :String-For-Printing))
	)
  )
  (if make-tag-table-p
      (let ((name (format nil "Files matching search for ~S"
			  zwei:*zmacs-tags-search-key*
		  )
	    )
	   )
	   (make-suitable-tag-table files name)
	   (setq zwei:*zmacs-current-tag-table* (find-tag-table name))
	   (format *query-io* "~&Tag table \"~A\" built." name)
      )
      nil
  )
)

(defvar zwei:*zmacs-search-files-silently-p* nil
"When true, searches files silently."
)

(defun Zmacs-Search-Files
       (remote-p replace-p &optional
	(files-to-prompt-for nil) (make-tag-table-p t) (search-not-p nil)
       )
  (multiple-value-setq
    (zwei:*zmacs-tags-search-function* zwei:*zmacs-tags-search-key*)
      (zwei:get-extended-string-search-strings
	nil "Search for string:" zwei:*search-mini-buffer-comtab*
      )
  )
  (if replace-p
      (setq zwei:*tags-query-replace-to* (get-replace-string))
  )
  (setq zwei:*zmacs-tags-search-key-string*
	zwei:*extended-string-search-last-string*
  )
  (if files-to-prompt-for
      (setq zwei:*search-and-replace-directory* files-to-prompt-for)
      (setq zwei:*search-and-replace-directory*
	    (zwei:read-defaulted-pathname "In Files"
	      (or (ucl:first-if-list zwei:*search-and-replace-directory*)
		  (make-pathname :Defaults (zwei:default-pathname)
				 :Name    *Search-Files-Default-File-Name*
				 :Type    *Search-Files-Default-File-Type*
				 :Version *Search-Files-Default-File-Version*
		  )
	      )
	    )
      )
  )
  (Maybe-Add-Item-Type)
  (Let ((result
	  (let ((fs:*log-print-function* 'Itemise-File-Name)
		(fs:*line-print-out-function* 'Itemise-Line)
	       )
	       (ticl:search-and-replace
		zwei:*zmacs-tags-search-key-string*
		zwei:*search-and-replace-directory*
		:Replace-String (if replace-p zwei:*tags-query-replace-to* nil)
		:Search-Not search-not-p
		:Silent-P (or search-not-p zwei:*zmacs-search-files-silently-p*)
		:Remote-Host
		(if remote-p
		    (pathname-host zwei:*search-and-replace-directory*)
		    nil
		)
		:Remote-Bindings
		  '((fs:*line-print-out-function* 'Server-Itemise-Line))
	       )
	  )
	)
       )
       (if replace-p
	   (if result
	       (format t "~%Modified files :- ~{~%  ~A~}"
		       (mapcar #'(lambda (x) (send x :String-For-Printing))
			       result
		       )
	       )
	       (format t "~%No files modified")
	   )
	   (if result
	       (print-out-matching-files result make-tag-table-p)
	       (format t "~%No files matched")
	   )
       )
       result
  )
)

zwei:
(defcom zwei:com-search-files "Searches in files for a string." ()
  (fs:Zmacs-Search-Files nil nil)
  dis-none
)

(zwei:set-comtab zwei:*standard-comtab* nil
		 (zwei:make-command-alist '(zwei:com-search-files))
)

zwei:
(defcom zwei:com-search-files-not "Searches for files not containing a string."
  ()
  (fs:Zmacs-Search-Files nil nil nil t t)
  dis-none
)

(zwei:set-comtab zwei:*standard-comtab* nil
		 (zwei:make-command-alist '(zwei:com-search-files-not))
)

zwei:
(defcom zwei:com-remote-search-files
	"Searches in files for a string on a remote host." ()
  (fs:Zmacs-Search-Files t nil)
  dis-none
)

(zwei:set-comtab zwei:*standard-comtab* nil
		 (zwei:make-command-alist '(zwei:com-remote-search-files))
)

(defvar zwei:*default-system* nil)

zwei:
(defcom zwei:com-search-system-files
	"Searches the files in a system for a string." ()
  (let ((system (zwei:read-system-name "System to search:" *default-system*)))
       (setq *Default-System* (sys:find-system-named system))
       (Fs:Zmacs-Search-Files nil nil (fs:system-source-files system))
  )
  dis-none
)

(zwei:set-comtab zwei:*standard-comtab* nil
		 (zwei:make-command-alist '(zwei:com-search-system-files))
)

zwei:
(defcom zwei:com-search-and-replace-in-files
	"Searches in files for a string and replaces it with another." ()
  (fs:Zmacs-Search-Files nil t)
  dis-none
)

(zwei:set-comtab
  zwei:*standard-comtab* nil
  (zwei:make-command-alist '(zwei:com-search-and-replace-in-files))
)


zwei:
(defcom zwei:com-search-and-replace-in-system-files
 "Searches the files in a system for a string and replaces it with another." ()
  (let ((system (zwei:read-system-name "System to search:" *Default-System*)))
       (setq *Default-System* (sys:find-system-named system))
       (fs:Zmacs-Search-Files nil t (fs:system-source-files system))
  )
  dis-none
)

(zwei:set-comtab zwei:*standard-comtab* nil
  (zwei:make-command-alist '(zwei:com-search-and-replace-in-system-files))
)

zwei:
(defcom zwei:com-remote-search-and-replace-in-files
"Searches in files on a remote host for a string and replaces it with another."
  ()
  (fs:Zmacs-Search-Files nil t)
  dis-none
)

(zwei:set-comtab
  zwei:*standard-comtab* nil
  (zwei:make-command-alist '(zwei:com-remote-search-and-replace-in-files))
)

;-------------------------------------------------------------------------------

(export 'search-and-replace 'ticl)