;;; -*- Mode:Common-Lisp; Package:TV; Base:10; Fonts:(TVFONT TR10B TR10I) -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.  Where functionality implemented herein replicates
;;; similarly named functionality on Symbolics machines, this code was
;;; developed solely from the interface specification in the documentation
;;; or through guesswork, never by examination of Symbolics source code.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may be distributed only to users who hold valid TI
;;; software licenses.
;;; **********************************************************************

;-------------------------------------------------------------------------------

;;; Definition of a simple pipe stream.

(1defflavor* pipe-stream ((buffer (1make-string* 100000))
			 (read-pointer 0)
			 (write-pointer 0)
			 (open-p t)
			 1function*
			 process
			 (lock (1list* nil))
			)
	    (sys:input-stream sys:output-stream)
  :Initable-Instance-Variables
  :Gettable-Instance-Variables
  :Settable-Instance-Variables
)

(1defmethod* (pipe-stream :After :Init) (1ignore*)
  (1setq* process (1process-run-function*
		  '(:Name "Pipe" :Priority -1)
		  #'(lambda (me)
		      (1funcall* (1send* me :Function) me)
		      (1send* me :Close)
		    )
		    self
		)
  )
)

(1defmethod* (pipe-stream :Tyo) (1char*)
  (1if* (1>*= write-pointer (1-* (1array-active-length* buffer) 1))
      (1progn* (1process-wait*
		"Wait to input into pipe"
		#'(lambda (me)
		    (1>*= (1send* me :Read-Pointer)
			(1send* me :Write-Pointer)
		    )
		  )
		  self
	      )
	      (1with-lock* ((1first* lock))
		(1setq* read-pointer 0)
		(1setq* write-pointer 0)
	      )
	      (1send* self :Tyo char)
      )
      (1with-lock* ((1first* lock))
	(1setf* (1aref* buffer write-pointer) char)
	(1incf* write-pointer)
      )
  )
)

(1defmethod* (pipe-stream :Close) (&rest ignore)
  (1setq* open-p nil)
  (1send* process :Kill)
  (1setq* process nil)
)

(1defmethod* (pipe-stream :UnTyi) (char)
  (1ignore* char)
  (1decf* read-pointer)
)

(1defmethod* (pipe-stream :Tyi) (&rest args)
  (1if* open-p
      (1if* (1<* read-pointer write-pointer)
	  (1with-lock* ((1first* lock))
	    (1prog1* (1aref* buffer read-pointer)
		    (1incf* read-pointer)
	    )
	  )
	  (1progn* (1process-wait* "Wait for pipe input"
				  #'(lambda (me)
				      (1or* (1not* (1send* me :Open-P))
					   (1<* (1send* me :Read-Pointer)
					       (1send* me :Write-Pointer)
					   )
				      )
				    )
				    self
		  )
		  (1lexpr-send* self :Tyi args)
	  )
      )
      (1if* (1<* read-pointer write-pointer)
	  (1with-lock* ((1first* lock))
	    (1prog1* (1aref* buffer read-pointer)
		    (1incf* read-pointer)
	    )
	  )
	  nil
      )
  )
)

;-------------------------------------------------------------------------------

;;; Postscript handling code.

(1defvar* 2*Rendering-Type** :Explorer
"The way that we are going to render the output."
)

(1defvar* 2*Draw-Alu-To-Use** Nil)

(1defvar* 2*Erase-Alu-To-Use** Nil)

(1defvar* 2*Something-Drawn-On-This-Page-P**) 

(1defconstant* 2*Dpi-Of-Explorer-Screen** 72)

(1defvar* 2*Dpi-Of-Postscript-Device** 72)

(1defvar* 2*Height-Of-Page-In-Inches** 11)

(1defvar* 2*Width-Of-Page-In-Inches** 8.5)

(1defvar* 2*Use-Gray-Scale-Instead-Of-Dashed-Lines-P** T)
(1defvar* 2*Use-Gray-Scale-Instead-Of-Gray-Scale-Bit-Arrays-P** T)
(1defvar* 2*Margin-Width** 0.5) ;;; in inches

(1defvar* 2*Postscript-Stream**)

(1Defmacro* 2With-Ps-Coords* (&body body)
  (1let* ((functions
	 `(,@(1if* (sys:is-in-tree 'ps-x body)
	        '((ps-x (screen-x)
			(1+* (1/* (1** 2*Margin-Width** 2*Dpi-Of-Postscript-Device**)
			       *Postscript-scale-factor*
			    )
			    (1** Scale screen-x)
			)
		  )
		 )
		 nil
	      )
	   ,@(1if* (sys:is-in-tree 'ps-y body)
	        '((ps-y (screen-y)
			(1+* (1/* (1** 2*Margin-Width** 2*Dpi-Of-Postscript-Device**)
			       *Postscript-scale-factor*
			    )
			    (1-* height-of-page (1** scale screen-y))
			)
		  )
		 )
		 nil
	     )
	  )
	)
       )
      `(1locally* (1declare* (1special* *postscript-scale-factor*))
	 (1let* ((scale (1float* (1/* 2*Dpi-Of-Postscript-Device**
				 2*Dpi-Of-Explorer-Screen**
			     )
		      )
	       )
	       (Height-of-page (1/* (1** (1-* 2*Height-Of-Page-In-Inches**
					 (1** 2 2*Margin-Width**)
				     )
				     2*Dpi-Of-Postscript-Device**
				  )
				  *Postscript-scale-factor*
			       )
	       )
	       (Width-of-page (1/* (1** (1-* 2*Width-Of-Page-In-Inches**
					(1** 2 2*Margin-Width**)
				    )
				    2*Dpi-Of-Postscript-Device**
				 )
				 *Postscript-scale-factor*
			       )
	       )
	      )
	      (1ignore* width-of-page height-of-page scale)
	      
	      (1flet* ,functions ,@body)
	 )
       )
  )
)

;-------------------------------------------------------------------------------

(1defflavor* 2Rendering-Mixin*
	   ()
	   ()
  (:Required-Flavors sheet)
  (:Required-Methods :Logical-Left-Edge
		     :Logical-Top-Edge
		     :Logical-Right-Edge
		     :Logical-Bottom-Edge
		     :X-Pl-Offset
		     :Y-Pl-Offset
		     :Scroll-To
		     :Scroll-To-Literally
		     :Inside-Width
		     :Inside-Height
		     :Left-Margin-Size
		     :Top-Margin-Size
		     :Right-Margin-Size
		     :Bottom-Margin-Size
  )
  :Abstract-Flavor
)

(1defmethod* (2Rendering-Mixin* :Define-Postscript-Procedures) ()
  (1format 2*Postscript-Stream*** "~%%---------- Procedure Definitions ~%")
  (1format 2*Postscript-Stream***
	    "~%%--- Define cliprectangle procedure (height width x y)~
             ~%/cliprectangle~
             ~%  {newpath moveto dup 0 rlineto exch 0 exch sub 0 exch rlineto~
             ~%   0 exch sub 0 rlineto closepath clip newpath} def~%"
  )
  (1format 2*Postscript-Stream***
	    "~%%--- Define black line procedure (from-x from-y to-x to-y)~
             ~%/blackline~
             ~%  {moveto lineto stroke} def~%"
  )
  (1format 2*Postscript-Stream***
	    "~%%--- Define Gray line procedure (gray from-x from-y to-x to-y)~
             ~%/grayline~
             ~%  {gsave moveto lineto setgray stroke grestore} def~%"
  )
  (1format 2*Postscript-Stream***
	    "~%%--- Define white line procedure (from-x from-y to-x to-y)~
             ~%/whiteline~
             ~%  {gsave moveto lineto 1.0 setgray stroke grestore} def~%"
  )
  (1format 2*Postscript-Stream***
	    "~%%--- Define rectangle procedure (gray height width x y)~
             ~%/rectangle~
             ~%  {gsave newpath moveto dup 0 rlineto exch 0 exch sub 0 exch~
             ~%   rlineto 0 exch sub 0 rlineto closepath setgray~
             ~%   fill grestore} def~%"
  )
  (1format 2*Postscript-Stream***
	    "~%%--- Define blackrectangle procedure (height width x y)~
             ~%/blackrectangle~
             ~%  {gsave newpath moveto dup 0 rlineto exch 0 exch sub 0 exch~
             ~%   rlineto 0 exch sub 0 rlineto closepath 0 setgray~
             ~%   fill grestore} def~%"
  )
  (1format 2*Postscript-Stream***
	    "~%%--- Define whiterectangle procedure (height width x y)~
             ~%/whiterectangle~
             ~%  {gsave newpath moveto dup 0 rlineto exch 0 exch sub 0 exch~
             ~%   rlineto 0 exch sub 0 rlineto closepath 1 setgray~
             ~%   fill grestore} def~%"
  )
  (1format 2*Postscript-Stream***
	    "~%%--- Define stringoutexplicit procedure~
             ~%          (string gray font-size font textwidth string~
             ~%           x y font-size font)~
             ~%/stringoutexplicit~
             ~%  {gsave findfont exch scalefont setfont moveto stringwidth ~
             ~%   pop div dup scale findfont exch scalefont setfont~
             ~%   setgray show grestore} def~%"
  )
  (1format 2*Postscript-Stream*** "~%%---------- End of Procedure Definitions ~%")
)

(1defmethod* (2Rendering-Mixin* :Draw-Postscript-Rectangle)
     (rectangle-width rectangle-height x y color &optional (1comment* nil))
  (1setq* 2*Something-Drawn-On-This-Page-P** T)
  (2With-Ps-Coords*
    (1format 2*Postscript-Stream*** "~%  ~$ ~$ ~$ ~$ ~$ rectangle~A"	
	   (1cond* ((1floatp* color) color) ; Colour already picked.
		 ((<= white color black)
		  (1float* (/ (- black color) black));Choose a percentage of gray.
		 )
		 (t white)		    ;Give up and make it white.
	   )
	   (1** scale rectangle-height)
	   (1** scale rectangle-width) (ps-x x) (ps-y y)
	   (1if* comment (1format* nil "  %---- ~A" comment) "")
    )
  )
)

(1defun-method* 2Draw-Alu-From-Window* 2Rendering-Mixin* ()
  (1or* *draw-alu-to-use* (1send* self :Char-Aluf))
)

(1defun-method* 2Erase-Alu-From-Window* 2Rendering-Mixin* ()
  (1or* *erase-alu-to-use* (1send* self :Erase-Aluf))
)

(1defmethod* (2Rendering-Mixin* :Output-Postscript-Bitmap)
	     (bitmap-array x y &optional (start-x 0) (start-y 0)
	      (image-width (1array-dimension* bitmap-array 1))
	      (image-height (1array-dimension* bitmap-array 0))
	     )
  (1let* ((procedure-name (1send* self :Procedurise-Bitmap bitmap-array start-x
			       start-y image-width image-height
			)
	)
       )
       (1format 2*Postscript-Stream***
		 "~%  ~D ~D ~D ~D whiterectangle~
                  ~%  gsave~
                  ~%  ~D ~D translate~
                  ~%  ~D ~D scale~
                  ~%  0 setgray~
                  ~%  ~D ~D 1 [~D 0 0 -~D 0 ~D] {~A} image~
                  ~%  grestore"
		 image-height image-width x y ;; whiterectangle
		 x (1-* y image-height)         ;; translate
		 image-width image-height     ;; scale
		 image-width image-height
		 image-width image-height image-height
		 procedure-name               ;; image
       )
  )
)

(1defmethod* (2Rendering-Mixin* :Procedurise-Bitmap)
	     (bitmap-array start-x start-y image-width image-height)
  (1declare* (1special* *bitmap-procedures*))
  "Returns the name of a procedure that represents the bitmap Bitmap-Array."
  (1let* ((image-bit 0)
        (image-hex 0)
	(bit-add nil)
	(*print-base* 16)
	(procedure-name (1gensym* "BITMAP-PROCEDURE-"))
	(entry (1gethash*
		 (1list* bitmap-array start-x start-y image-width image-height)
		 *bitmap-procedures*
	       )
	)
	(bytes-on-line 0)
       )
       (1if* entry
	   (1symbol-name* entry)
	   (1progn* (1format 2*Postscript-Stream***
			     "~%/~A  %--- Define procedure to print bitmap ~S~
                              ~% <" (1symbol-name* procedure-name)
			      bitmap-array
                   )
		   (1do* (( y start-y (1+* y 1)))
		       ((1or* (1>*= y image-height)
			     (1>=* y (1array-dimension* bitmap-array 0)))
			 nil)
		     (1do* (( x start-x (1+* x 1)))
			  ((1or* (>1=* x image-width)
			     (1>=* x (1array-dimension* bitmap-array 1)))
			   nil)
		       (1setf* bit-add (1aref* bitmap-array y x))
		       (1if* (1=* bit-add 0) (1setf* bit-add 1) (1setf* bit-add 0))
		       (1setf* image-hex (1+* image-hex  bit-add ))
		       (1setf* image-bit (1+* image-bit 1))
		       (1if* (1equal* image-bit 8)
			   (1progn*
			     (1if* (1<* image-hex 16)
				 (1princ* "0" 2*Postscript-Stream**)
				 nil
			     )
			     (1setf* image-bit 0)
			     (1princ* image-hex 2*Postscript-Stream**)
			     (1princ* " " 2*Postscript-Stream**)
			     (1incf* bytes-on-line)
			     (1if* (1equal* 0 (1mod* bytes-on-line 25))
				 (1progn* (1setq* bytes-on-line 0)
					 (1format 2*Postscript-Stream*** "~%  ")
				 )
				 nil
			     )
			     (1setf* image-hex 0)
			   )
			   nil
		       )
		       (1setf* image-hex (1lsh* image-hex 1))
		     )
		   )
		   (1format 2*Postscript-Stream*** "> def~%")
		   (1setf* (1gethash*
			   (1list* bitmap-array start-x start-y
				 image-width image-height
			   )
			   *bitmap-procedures*
			 )
			 procedure-name
		   )
		   (1symbol-name* procedure-name)
	   )
       )
  )
)

(1defmethod* (2Rendering-Mixin* :Draw-Postscript-Line)
	     (from-x from-y to-x to-y alu &optional (gray nil) (1comment* nil))
  (1setq* 2*Something-Drawn-On-This-Page-P** T)
  (2With-Ps-Coords*
    (1if* (1equal* (2Draw-Alu-From-Window*) alu)
        (1if* gray
	    (1format 2*Postscript-Stream*** "~%  ~$ ~$ ~$ ~$ ~$ grayline~A"
		      gray (ps-x from-x) (ps-y from-y)
		      (ps-x   to-x) (ps-y   to-y)
		      (1if* comment (1format* nil "  %---- ~A" comment) "")
	    )
	    (1format 2*Postscript-Stream*** "~%  ~$ ~$ ~$ ~$ blackline~A"
		      (ps-x from-x) (ps-y from-y)
		      (ps-x   to-x) (ps-y   to-y)
		      (1if* comment (1format* nil "  %---- ~A" comment) "")
	    )
	)
	(1format 2*Postscript-Stream*** "~%  ~$ ~$ ~$ ~$ whiteline~A"
		  (ps-x from-x) (ps-y from-y)
		  (ps-x   to-x) (ps-y   to-y)
		  (1if* comment (1format* nil "  %---- ~A" comment) "")
	)
    )
  )
)

(1defun* 2Postscriptify-String* (1string* &optional (start 0))
  (1let* ((index (1string-search-set* '(#\\ #\( #\)) string start)))
       (1if* index
	   (1string-append* (1subseq* string start index) "\\"
			    (1aref* string index)
			    (2Postscriptify-String* string (1+* index 1))
	   )
	   (1subseq* string start)
       )
  )
)

(1defvar* 2*Paint-In-White-P** Nil)

(1defun* gsave ()
  (1format 2*Postscript-Stream*** "~&  gsave")
)

(1defun* grestore ()
  (1format 2*Postscript-Stream*** "~&  grestore")
)

(defvar *mac-font-to-explorer-font-mapping-table*
	(if (sys:mx-p)
	    (let ((result nil))
	         (1mapatoms*
		   #'(lambda (sym)
		       (if (and (boundp sym)
				(typep (symbol-value sym) 'font)
				(not (equal sym (font-name (symbol-value sym))))
			   )
			   (push (list (font-name (symbol-value sym)) sym)
				 result
			   )
			   nil
		       )
		     )
		   'fonts
		 )
		 result
	    )
	)
)


(defun maybe-explorerify-mac-font (font)
  (declare (special mac:*mac-font-translation-table*))
  (check-type font font)
  (if (sys:mx-p)
      (or (second (assoc (font-name font)
			 *mac-font-to-explorer-font-mapping-table*
			 :test #'eq
		  )
	  )
	  (font-name font)
      )
      (font-name font)
  )
)

(1defmethod* (2Rendering-Mixin* :Postscript-String)
	     (1string* start-pos-x start-pos-y screen-font)
  ;;; Note:  start-pos-x and start-pos-y are in internal coordinates.
  (coerce-font screen-font self)
  (2With-Ps-Coords*
    (1setq* 2*Something-Drawn-On-This-Page-P** T)
    (1let* ((font (1or* (1second* (1assoc* (1symbol-name*
				      (maybe-explorerify-mac-font screen-font)
				    )
				    printer:*explorer-postscript-font-map*
				    :Test #'1string-equal*
			  )
		    )
		    (1ferror* nil "Cannot find a postscript font for ~S"
			     screen-font
		    )
		)
	  )
	  (ps-string (2Postscriptify-String* string))
	 )
	 (1multiple-value-bind* (1ignore* text-height ignore text-width)
	     (1send* self :Compute-Motion string 0 nil 0 0 nil 0
		   nil nil nil screen-font
	     )
	   (1if* (1not* 2*Paint-In-White-P**)
	       (1send* self :Draw-Postscript-Rectangle
			text-width
			(1+* (font-char-height screen-font)
			   text-height
			)
			start-pos-x
			start-pos-y
			white "White-out area behind text"
	       )
	       nil
	   )
	   (1format 2*Postscript-Stream***
	     "~&  (~A) ~D ~D /~A ~D (~A) ~$ ~$ ~D /~A stringoutexplicit"
	     ps-string (1if* 2*Paint-In-White-P** 1 0) (1second* font)
	     (1first* font) text-width ps-string (ps-x start-pos-x)
	     (ps-y (1+* (font-baseline screen-font) start-pos-y))
	     (1second* font) (1first* font)
	   )
	 )
    )
  )
)

(1defmethod* (2Rendering-Mixin* :Compute-Page-Boundaries-1)
  (x scale logical-top-edge logical-right-edge logical-bottom-edge
   height-of-page-in-screen-pixels width-of-page-in-screen-pixels
   pages-from-left left-page-number right-page-number
   vertical-number-of-pages horizontal-number-of-pages
  )
  (loop for y from logical-top-edge to logical-bottom-edge
	by (/ height-of-page-in-screen-pixels scale)
	for pages-from-top from 1
	for page-number = (+ (* (- pages-from-left 1)
				 vertical-number-of-pages
			      )
			      pages-from-top
			  )
	for top-page-number = (- pages-from-top 1)
	for bottom-page-number = (+ 1 pages-from-top)
	collect
	  (list (round x)
		(round y)
		(round
		  (min logical-right-edge
		    (+ x (/ width-of-page-in-screen-pixels scale))
		  )
		)
		(round
		  (min logical-bottom-edge
		    (+ y (/ height-of-page-in-screen-pixels scale))
		  )
		)
		page-number
		(if (> left-page-number 0)
		    (- page-number vertical-number-of-pages)
		    nil
		)
		(if (> top-page-number 0) (- page-number 1) nil)
		(if (< (- right-page-number 1)
		       horizontal-number-of-pages
		    )
		    (+ page-number vertical-number-of-pages)
		    nil
		)
		(if (< (- bottom-page-number 1) vertical-number-of-pages)
		    (+ page-number 1)
		    nil
		)
	  )
  )
)

(1defmethod* (2Rendering-Mixin* :Compute-Page-Boundaries)
	     (&optional (scale 1.0) (landscape-p nil)
	      (logical-left-edge   (1send* self :Logical-Left-Edge))
	      (logical-top-edge    (1send* self :Logical-Top-Edge))
	      (logical-right-edge  (1send* self :Logical-Right-Edge))
	      (logical-bottom-edge (1send* self :Logical-Bottom-Edge))
	     )
"Scale of 0.5 means half linear dimensions."
  (1declare* (1values* page-coordinate-specs width-in-pages height-in-pages))
  (1let* ((height-of-page-in-screen-pixels
	  (1-* (1** (1if* landscape-p
		     2*Width-Of-Page-In-Inches**
		     2*Height-Of-Page-In-Inches**
		 )
		 2*Dpi-Of-Explorer-Screen**
	     )
	     ;;; Allow 1/2" space at each end.
	     (1round* (1** 2 2*Margin-Width** 2*Dpi-Of-Explorer-Screen**))
	  )
	)
        (width-of-page-in-screen-pixels
	  (1-* (1** (1if* landscape-p
		     2*Height-Of-Page-In-Inches**
		     2*Width-Of-Page-In-Inches**
		 )
		 2*Dpi-Of-Explorer-Screen**
	     )
	      ;;; Allow 1/2" space at each end.
	     (1round* (1** 2 2*Margin-Width** 2*Dpi-Of-Explorer-Screen**))
	  )
	)
       )
       (1let* ((lists
	      (1loop* for x
		    from logical-left-edge
		    to logical-right-edge
		    by (1/* width-of-page-in-screen-pixels scale)
		    for pages-from-left from 1
		    for left-page-number = (1-* pages-from-left 1)
		    for right-page-number = (1+* 1 pages-from-left)
		    for vertical-number-of-pages
		      = (1ceiling* (1-* logical-bottom-edge logical-top-edge)
				 (1/* height-of-page-in-screen-pixels scale)
			)
		    for horizontal-number-of-pages
		      = (1ceiling* (1-* logical-right-edge logical-left-edge)
				 (1/* width-of-page-in-screen-pixels scale)
			)
		    Collect
		       (1send* self :Compute-Page-Boundaries-1 X scale
			 logical-top-edge logical-right-edge logical-bottom-edge
			 height-of-page-in-screen-pixels
			 width-of-page-in-screen-pixels pages-from-left
			 left-page-number right-page-number
			 vertical-number-of-pages horizontal-number-of-pages
		       )
	      )
	     )
	    )
	    (1values* (1apply* #'1append* lists) (1length* lists)
		     (1length* (1first* lists))
	    )
       )
  )
)

(1defmethod* (2Rendering-Mixin* :Maximum-Number-Of-Pages)
	     (&optional (scale 1.0) (landscape-p nil))
"Takes a desired scale factor and whether we are in landscape mode or not
and returns three values;
  - The maximum number of pages that will be used to print in this scale.
    Fewer pages might be printed if there are blank pages.
  - The width in pages of the logical rectangle of pages that would
    be generated.
  - The height in pages of the logical rectangle of pages that would be
    generated.
Thus, if we have values (27 3 9) then we will print, at most, 27 pages in a
grid three wide and 9 high.
"
  (1declare* (1values* maximum-number-of-pages width-in-pages height-in-pages))
  (1multiple-value-bind* (pages the-width the-height)
      (1send* self :Compute-Page-Boundaries scale landscape-p)
    (1values* (1length* pages) the-width the-height)
  )
)

(1defvar* 2*Dont-Clip-P** Nil)

(1defmethod* (2Rendering-Mixin* :Draw-Simple-Border)
	     (left top right bottom x-window-offset y-window-offset)
  (1format 2*Postscript-Stream*** "~%%---- Start of Border")
  (1let* ((the-width  (1+* 1 (1-* right left))) ;; Go one pixel outside edges.
        (the-height (1+* 1 (1-* bottom top)))
       )
       (1send* self :Draw-Line
	      (1-*         -1 x-window-offset)
	      (1-*         -1 y-window-offset)
	      (1-*  the-width x-window-offset)
	      (1-*         -1 y-window-offset))
       (1send* self :Draw-Line
	      (1-*  the-width x-window-offset)
	      (1-*         -1 y-window-offset)
	      (1-*  the-width x-window-offset)
	      (1-* the-height y-window-offset))
       (1send* self :Draw-Line
	      (1-*         -1 x-window-offset)
	      (1-*         -1 y-window-offset)
	      (1-*         -1 x-window-offset)
	      (1-* the-height y-window-offset))
       (1send* self :Draw-Line
	      (1-*         -1 x-window-offset)
	      (1-* the-height y-window-offset)
	      (1-*  the-width x-window-offset)
	      (1-* the-height y-window-offset))
  )
  (1format 2*Postscript-Stream*** "~%% ---- End of Border")
)

(1defmethod* (2Rendering-Mixin* :Specify-Clipping-Region)
	     (left top right bottom x-window-offset y-window-offset)
  (2With-Ps-Coords*
    (1format 2*Postscript-Stream***
	   "~%  newpath % --- Start of clipping region~
	    ~%   ~D ~D moveto~
	    ~%   ~D ~D lineto~
	    ~%   ~D ~D lineto~
	    ~%   ~D ~D lineto~
	    ~%  closepath clip newpath"
	      (ps-x (1-* x-window-offset))
	      (ps-y (1-* y-window-offset))
	      (ps-x (1-* right left x-window-offset))
	      (ps-y (1-* y-window-offset))
	      (ps-x (1-* right left x-window-offset))
	      (ps-y (1-* bottom top y-window-offset))
	      (ps-x (1-* x-window-offset))
	      (ps-y (1-* bottom top y-window-offset))
    )
  )
)

(1defmethod* (rendering-mixin :New-Page)
	     (&key (notify-p t) (force-p nil) (page-spec nil))
  (1declare* (1special* *page-number*))
  (2Grestore*)
  (1if* (1or* force-p 2*Something-Drawn-On-This-Page-P**)
      (1Progn* (1incf* *page-number*)
	      (1if* page-spec
		  (1let* ((*Postscript-scale-factor* 1.0))
		       (1declare* (1special* *Postscript-scale-factor*))
		       (1apply* '2Print-Out-Adjoining-Page-Numbers*
			       self page-spec
		       )
		  )
		  nil
	      )
	      (1format 2*Postscript-Stream***
		"~%showpage   %-----------------------~
		 --------------------------- Page ~D~%"
		*page-number*
	      )
	      (1if* notify-p
		  (tv:notify tv:selected-window
			     "Page ~D postscriptified"
			     *page-number*
		  )
		  nil
	      )
      )
      nil
  )
  (1setq* 2*Something-Drawn-On-This-Page-P** Nil)
)

(1defmethod* (2Rendering-Mixin* :Setup-Postscript) (landscape-p)
  (1declare* (1special* *Postscript-scale-factor*))
  (1format 2*Postscript-Stream*** "%!")
  (1if* landscape-p
      (1format 2*Postscript-Stream***
		"~%  90 rotate  % --- Landscape format.  Rotate origin~
		 ~%  0 -~D translate  % --- slide it back onto the page."
		(1+* (1send* self :Inside-Height)
		   (1/* (1** 2 2*Margin-Width** 2*Dpi-Of-Postscript-Device**)
		       *Postscript-scale-factor*
		   )
		)
      )
      nil
  )
  (1send* self :Define-Postscript-Procedures)
)

(1defmethod* (2Rendering-Mixin* :compute-all-pages)
	     (whole-graph-p postscript-scale-factor landscape-p)
  (1if* whole-graph-p
      (1send* self :Compute-Page-Boundaries
	     postscript-scale-factor landscape-p
      )
      (1send* self :Compute-Page-Boundaries
	    postscript-scale-factor landscape-p
	    (1send* self :X-Pl-Offset)
	    (1send* self :Y-Pl-Offset)
	    (1+* (1send* self :X-Pl-Offset)
		(1send* self :inside-width)
	    )
	    (1+* (1send* self :Y-Pl-Offset)
		(1send* self :inside-height)
	    )
      )
  )
)

(1defmacro* 2With-Postscript-Environment* 
	    ((Whole-graph-p postscript-scale-factor landscape-p) &body body)
 `(1let* ((2*Dont-Clip-P** ,Whole-graph-p)
        (*print-circle* nil)
	(*print-pretty* nil)
	(*postscript-scale-factor* ,postscript-scale-factor)
	(Old-x-offset (1send* self :X-Pl-Offset))
	(old-y-offset (1send* self :Y-Pl-Offset))
	(*page-number* 0)
	(all-pages (1send* self :Compute-All-Pages ,whole-graph-p
			  ,postscript-scale-factor ,landscape-p
		   )
	)
	(*bitmap-procedures*
	  (1if* (1boundp* '*bitmap-procedures*)
	      (1locally* (1declare* (1special* *bitmap-procedures*))
		        *bitmap-procedures*
	      )
	      (1make-hash-table* :Test #'1equal*)
	  )
	)
       )
       (1declare* (1special* *page-number* *postscript-scale-factor*
			  *bitmap-procedures*
                 )
       )
       (1ignore* old-x-offset old-y-offset all-pages)
      ,@body
  )
)

(1defmethod* (2Rendering-Mixin* :Prepare-New-Page)
	     (left top right bottom postscript-scale-factor border-p
	      x-window-offset y-window-offset
	     )
  (2Gsave*)
  (1format 2*Postscript-Stream*** "~%  ~D ~D scale"
	    postscript-scale-factor postscript-scale-factor
  )
  (1if* border-p
      (1send* self :Draw-Simple-Border left top right bottom
	     x-window-offset y-window-offset
      )
      nil
  )
  (1send* self :Specify-Clipping-Region left top right bottom
	 x-window-offset y-window-offset
  )
)

(1defun* print-out-adjoining-page-numbers
  (sheet page-number left-number top-number right-number bottom-number)
  (1send* sheet :String-Out-Explicit-Internal-Coords
	 (1format* nil "~A" page-number)
	 0 (1-* (1+* (tv:font-char-height fonts:tvfont) 5)) fonts:tvfont tv:alu-seta
  )
  (1if* left-number
      (1send* sheet :String-Out-Explicit-Internal-Coords
        (1format* nil "~A" left-number)
	(1-* (1+* (sheet-string-length sheet (1format* nil "~A" left-number)) 5))
	(1floor* (1+* (1** (1-* 2*Height-Of-Page-In-Inches** (1** 2 2*Margin-Width**))
		      2*Dpi-Of-Postscript-Device**
		   )
		   (Font-char-height fonts:tvfont)
	       )
	       2
	)
	fonts:tvfont tv:alu-seta
      )
      nil
  )
  (1if* top-number
      (1send* sheet :String-Out-Explicit-Internal-Coords
        (1format* nil "~A" top-number)
	(1floor* (1+* (1** (1-* 2*Width-Of-Page-In-Inches** (1** 2 2*Margin-Width**))
		      2*Dpi-Of-Postscript-Device**
		   )
		   (Sheet-string-length sheet (1format* nil "~A" top-number))
	       )
	       2
	)
	(1-* (1+* (Tv:font-char-height fonts:tvfont) 5)) fonts:tvfont tv:alu-seta
      )
      nil
  )
  (1if* right-number
      (1send* sheet :String-Out-Explicit-Internal-Coords
	(1format* nil "~A" right-number)
	(1+* (1** (1-* 2*Width-Of-Page-In-Inches** (1** 2 2*Margin-Width**))
	       2*Dpi-Of-Postscript-Device**
	    )
	    20
	)
	(1floor* (1+* (1** (1-* 2*Height-Of-Page-In-Inches** (1** 2 2*Margin-Width**))
		      2*Dpi-Of-Postscript-Device**
		   )
		   (Font-char-height fonts:tvfont)
	       )
	       2
	)
	fonts:tvfont tv:alu-seta
      )
      nil
  )
  (1if* bottom-number
     (1send* sheet :String-Out-Explicit-Internal-Coords
       (1format* nil "~A" bottom-number)
       (1floor* (1+* (1** (1-* 2*Width-Of-Page-In-Inches** (1** 2 2*Margin-Width**))
		     *Dpi-Of-Postscript-Device*
		  )
		  (sheet-string-length sheet (1format* nil "~A" bottom-number))
	      )
	      2
       )
       (1+* (1** (1-* 2*Height-Of-Page-In-Inches** (1** 2 2*Margin-Width**))
	      2*Dpi-Of-Postscript-Device**
	   )
	   20
       )
       fonts:tvfont tv:alu-seta
     )
     nil
  )
)

(defun 2Draw-Line-Clip-Visibility-Given-Width-Height*
       (point-x point-y width height &aux visibility)
;;; Modified from TI code for tv:2Draw-Line-Clip-Visibility*
  (setq visibility (cond ((<  point-x 0) 1)
			 ((>= point-x width) 2)
			 (t 0)
		   )
  )
  (cond ((<  point-y 0)      (logior 4 visibility))
	((>= point-y height) (logior 8 visibility))
	(t visibility)
  )
)


(1defmacro* with-postscript-page-environment ((left top right bottom) &body body)
 `(1let* ((2*Rendering-Type** :Postscript))
       (1let-if* (1not* (1boundp* '2*Something-Drawn-On-This-Page-P**))
	       ((2*Something-Drawn-On-This-Page-P** nil))
	       (1letf* ((#'draw-line-clip-visibility
		       #'(lambda (x y)
			   (draw-line-clip-visibility-given-width-height
			     x y (1-* ,right ,left) (1-* ,bottom ,top)
			   )
			 )
		      )
		     )
		     ,@body
	       )
       )
   )
)

(1defmethod* (2Rendering-Mixin* :Generate-Postscript-1)
     (border-p whole-graph-p notify-p postscript-scale-factor landscape-p
      number-pages-p window-offset
     )
  (1declare* (1values* pages-printed total-pages-including-blanks))
  (destructuring-bind (x-window-offset y-window-offset) window-offset
    (2With-Postscript-Environment*
      (Whole-graph-p postscript-scale-factor landscape-p)
	 (1send* self :Setup-Postscript landscape-p)
	 (1loop* for (left top right bottom . page-spec) in all-pages
	       
	       do (with-postscript-page-environment (left top right bottom)
		    (1send* self :Prepare-New-Page left top right bottom
			   postscript-scale-factor border-p
			   x-window-offset y-window-offset
		    )
		    (1unwind-protect* (1send* self :Scroll-To-Literally left top)
		      (1send* self :New-Page
			    :Notify-P (1and* notify-p (1rest* all-pages))
			    :Force-P nil
			    :Page-Spec (1if* number-pages-p page-spec nil)
		      )
		    )
		  )
	 )
	 (1send* self :Scroll-To old-x-offset old-y-offset t t t)
	 (1send* self :Refresh :Complete-Redisplay)
	 (1values* *page-number* (1length* all-pages))
    )
  )
)

(1defmethod* (2Rendering-Mixin* :Find-Landmark-Scale-Factors) (landscape-p)
"Returns a list of the form ((<sf> (val1 val2 val3)) (<sf2>...)), where
sf is a scale factor and the values are the values generated by
:Maximum-Number-Of-Pages.  Each entry in the list corresponds to the maximum
scale factor that will support a given number of pages.
"
  (1let* ((current
	  (1multiple-value-list*
	    (1send* self :Maximum-Number-Of-Pages 1.0 landscape-p)
	  )
	)
       )
       (1let* ((all (1list* (1list* 1.0 current))))
	    (1loop* for sf from 1.0 downto 0.01 by 0.01
		  for new
		    = (1multiple-value-list*
			(1send* self :Maximum-Number-Of-Pages sf landscape-p)
		      )
		  when (1not* (1equal* new current))
		  do (1push* (1list* sf new) all)
		     (1setq* current new)
	    )
	    all
       )
  )
)

(1defun* 2Print-Graph*
  (window
   &key (scale-factor 1.0)
        (just-compute-size-p nil)
	(find-landmark-scale-factors-p nil)
        (landscape-p nil)
	(whole-graph-p t)
	(border-p t)
	(number-pages-p whole-graph-p)
	(notify-p print-p)
	(printer (printer:get-default-printer))
        (print-p t)
	(1stream* *standard-output*)
	(1pathname* nil)
	(gray-scale-dashed-lines-p
	  2*Use-Gray-Scale-Instead-Of-Dashed-Lines-P**
	)
	(gray-scale-bit-arrays-p
	  2*Use-Gray-Scale-Instead-Of-Gray-Scale-Bit-Arrays-P**
	) 
	(window-offset '(0 0))
	
  )
"Prints the graph on the window.  Takes a whole bunch of keyword args:
   Scale-factor - controls the ratio of the size on the screen to the
              size on the page, 1.0 means 1:1 scale, 0.5 means put twice as
              much (linearly) on a page.
   Just-Compute-Size-p - when true does not print the graph but rather computes
              the maximum number of pages that the graph would occupy if printed
              with the specified scale factor and paper orientation
              (landscape-p).  When this arg is true the function returns three
              values:
                 - The maximum number of pages that will be used to print
                   in this scale.  Fewer pages might be printed if there
                   are blank pages.
                 - The width in pages of the logical rectangle of pages
                   that would be generated.
                 - The height in pages of the logical rectangle of pages
                   that would be generated.
              Thus, if we have values (27 3 9) then we will print, at most,
              27 pages in a grid three wide and 9 high.
   Find-Landmark-Scale-Factors-P - when true causes the grapher not to print
              the graph but rather to return an alist that maps the maximum
              scale factor that will generate a specific number of pages
              for the given paper orientation to a list of the form
              (max width height), which are the three values that would be
              returned if you had chosen :Just-Compute-Size-p t.  Thus,
                 ((0.25 (1 1 1) (0.3 (2 1 2) (1.0 (6 2 3)) would
              indicate that if you only want to generate a graph with 2 pages
              (1 wide and two high) then you should pick a scale factor that 
              is at most 0.3.
   Landscape-p - controls the orientation of the paper.  When true, comes out
              in landscape format.
   Whole-Graph-p - controls whether to print all of the graph, even if it is
              beyond the bounds of the physical window on the screen.  When NIL,
              it prints only the visible part of the graph.
   Border-p - controls whether a border is printed on each page of the graph
              or not.
   Number-pages-p - When true causes the grapher to number the pages it
              generates and to print the page numbers of the adjoining
              pages in the margins.  Thus, you get coordinates to help you
              join up the graph when you've finished printing.
   Notify-p - When true notifies you after each page has been converted for
              printing.  This can be a lengthy process for large graphs, so it's
              nice to see what's going on.
   Printer - must be the name of a :postscript-printer type of printer.
   print-p  - if true then it prints the resulting graph to Printer.
              If it is not then the rendered graph is printed to Stream.
   Stream - see print-p
   pathname - if supplied then the rendered graph is printed to a
              file of that name.
   Gray-Scale-Dashed-Lines-P - when true, it makes dashed lines come out as
              lines in gray scales with the shade of gray depending on the
              courseness of the dashes.  If this arg is false then the lines
              come out dashed just as they are on the screen.
              The gray scale way is quite a bit faster.
   Gray-Scale-Bit-Arrays-P - if true then bitblting an array which is known
              to be an array such as tv:12%-gray will result in a 12% gray
              rectangle being generated.  This should have better resolution
              in general than scaling the actual bitmap.
   Window-offset - defines the x, y offset to apply to the clipping
              window in screen coordinates.  This allows you to print things
              that are outside the window, i.e. the label.
"
  (1declare* (1unspecial* scale-factor))
  (1cond* (just-compute-size-p
	  (1send* window :Maximum-Number-Of-Pages scale-factor landscape-p)
	)
	(find-landmark-scale-factors-p
	 (1send* window :Find-Landmark-Scale-Factors landscape-p)
	)
	(t (1send* window :Generate-Postscript
		  :Border-P Border-p
		  :Whole-Graph-P whole-graph-p
		  :Print-P print-p
		  :Stream stream
		  :Pathname pathname
		  :Notify-P notify-p
		  :Gray-Scale-Dashed-Lines-P gray-scale-dashed-lines-p
		  :Gray-Scale-Bit-Arrays-P gray-scale-bit-arrays-p
		  :Postscript-Scale-Factor scale-factor
		  :Landscape-P landscape-p
		  :Number-Pages-P number-pages-p
		  :Window-Offset window-offset
		  :Printer printer
	   )
	)
  )
)

(1defmethod* (2Rendering-Mixin* :Generate-Postscript)
	     (&key (border-p t) (whole-graph-p nil) (print-p t)
	           (1stream* *standard-output*) (1pathname* nil)
		   (notify-p print-p)
		   (gray-scale-dashed-lines-p
		     2*Use-Gray-Scale-Instead-Of-Dashed-Lines-P**
		   )
		   (gray-scale-bit-arrays-p
		     2*Use-Gray-Scale-Instead-Of-Gray-Scale-Bit-Arrays-P**
		   )
		   (Postscript-scale-factor 1.0)
		   (landscape-p nil)
		   (number-pages-p whole-graph-p)
		   (window-offset '(0 0))
		   (printer (printer:get-default-printer))
	     )
"Generates postscript for Self.  Takes a whole bunch of keyword args:
   print-p  - if true then it prints the resulting postscript to Printer.
              If it is not then the postscript is printed to Stream.
   Stream - see print-p
   pathname - if supplied then the postscript is printed to a file of that name.
   Gray-Scale-Dashed-Lines-P - when true, it makes dashed lines come out as
              lines in gray scales with the shade of gray depending on the
              courseness of the dashes.  If this arg is false then the lines
              come out dashed just as they are on the screen.
              The gray scale way is quite a bit faster.
  Postscript-scale-factor - controls the ratio of the size on the screen to the
              size on the page, 1.0 means 1:1 scale, 0.5 means put twice as
              much (linearly) on a page.
  Gray-Scale-Bit-Arrays-P - if true then bitblting an array which is known
              to be an array such as tv:12%-gray will result in a 12% gray
              rectangle being generated.  This should have better resolution
              in general than scaling the actual bitmap.
  Printer - must be the name of a :postscript-printer type of printer.
  Window-offset - defines the x, y offset to apply to the postscript clipping
              window in screen coordinates.  This allows you to print things
              that are outside the window, i.e. the label.
"
  (1assert* (1not* (1and* print-p pathname)) (print-p pathname)
          "You cannot select the print-p and pathname option at the same time."
  )
  (1let* ((2*Use-Gray-Scale-Instead-Of-Dashed-Lines-P**
	  Gray-scale-dashed-lines-p
	)
        (2*Use-Gray-Scale-Instead-Of-Gray-Scale-Bit-Arrays-P**
	  gray-scale-bit-arrays-p
	)
       )
       (1flet* ((body (me)
	       (1send* me :Generate-Postscript-1 border-p whole-graph-p
		      notify-p postscript-scale-factor
		      landscape-p number-pages-p window-offset
	       )
	      )
	     )
	  (1if* print-p
	      (1let* ((results nil))
		   (1if* (1not* (1equal* :Postscript-Printer
				    (1get* (get-printer-device printer) :Type)
			    )
		       )
		       (1ferror* nil "~S is not the name of a postscript-printer."
			        printer
		       )
		       nil
		   )
		   (1with-open-stream*
		      (*standard-output*
			 (1let* ((me self))
			      (1make-instance* 'pipe-stream :Function
				#'(lambda (1stream*)
				    (1let* ((2*Postscript-Stream** stream))
					 (1setq* results
					       (1multiple-value-list* (body me))
					 )
				    )
				  )
			      )
			 )
		       )
		     (Printer:print-stream *standard-output* :Printer-Name
					   printer
                     )
		   )
		   (1values-list* results)
	      )
	      (1if* pathname
		  (1with-open-file*
		    (2*Postscript-Stream** pathname :Direction :Output)
		    (body self)
		  )
		  (1let* ((2*Postscript-Stream** stream)) (body self))
	      )
	  )
       )
  )
)

(1defvar* 2*Gray-Array-To-Grayness-Alist**
	 `((,tv:100%-black 1.0)
	   (,tv:88%-gray   0.88)
	   (,tv:75%-gray   0.75)
	   (,tv:66%-gray   0.66)
	   (,tv:50%-gray   0.50)
	   (,tv:33%-gray   0.33)
	   (,tv:25%-gray   0.25)
	   (,tv:hes-gray   0.15)
	   (,tv:12%-gray   0.12)
	   (,tv:100%-white 0.0)
	  )
)
  
(1Defwhopper* (2Rendering-Mixin* :Bitblt)
	      (alu wid hei from-array from-x from-y to-x to-y)
  (1ecase* 2*Rendering-Type**
    (:Explorer
     (1continue-whopper* alu wid hei from-array from-x from-y to-x to-y)
    )
    (:Postscript
     (1if* (1and* 2*Use-Gray-Scale-Instead-Of-Gray-Scale-Bit-Arrays-P**
	       (assoc from-array 2*Gray-Array-To-Grayness-Alist** :test #'1eq*)
	 )
	 (1progn* (1send* self :Draw-Postscript-Rectangle wid hei to-x to-y white
			 "White out behind bitblt area"
		  )
		  (1send* self :Draw-Postscript-Rectangle wid hei to-x to-y
			 (1-* 1 (1second* (assoc from-array
					 2*Gray-Array-To-Grayness-Alist**
					 :Test #'1eq*
				       )
			      )
			 )
			 "Use gray rectangle as bitblt"
		  )
	 )
	 (2With-Ps-Coords*
	   (1setq* 2*Something-Drawn-On-This-Page-P** T)
	   (1send* self :Output-Postscript-Bitmap from-array
		  (ps-x to-x) (ps-y to-y)
		 from-x from-y wid hei
	   )
	 )
     )
    )
  )
)

(1defwhopper* (2Rendering-Mixin* :String-Out-Explicit)
	      (1string* start-pos-x start-pos-y x-limit y-limit screen-font alu
	       &rest args
	      )
  (1ecase* 2*Rendering-Type**
    (:Explorer (1lexpr-continue-whopper*
		 string start-pos-x start-pos-y x-limit y-limit
		 screen-font alu args
	       )
    )
    (:Postscript (1send* self :Postscript-String string
		        (1-* start-pos-x (1send* self :Left-Margin-Size))
			(1-* start-pos-y (1send* self :Top-Margin-Size)) screen-font
		 )
    )
  )
)

(1defwhopper* (2Rendering-Mixin* :String-Out-Centered-Explicit)
	      (1string* &optional (left (sheet-inside-left))
		       y-pos right (y-limit (sheet-inside-right))
		       (screen-font (1send* self :Current-Font))
		       (alu (1send* self :Char-Aluf))
		       (start 0) (end nil)
	       &rest args
	      )
  (1ecase* 2*Rendering-Type**
    (:Explorer (1lexpr-continue-whopper*
		 string left y-pos right y-limit
		 screen-font alu start end args 
	       )
    )
    (:Postscript
     (1let* ((string-length (tv:sheet-string-length self string start end)))
          (1let* ((start-x (1/* (1-* (1-* (1or* right (sheet-inside-width)) left)
			        string-length
			    )
			    2
			 )
		)
	       )
	       (1send* self :Postscript-String (1subseq* string start end)
		      (1-* start-x (1send* self :Left-Margin-Size))
		      (1-* y-pos (1send* self :Top-Margin-Size))
		      screen-font
	       )
	  )
     )
    )
  )
)

(1defmethod* (2Rendering-Mixin* :String-Out-Explicit-Internal-Coords)
	      (1string* x y &rest args)
  (1lexpr-send* self :String-Out-Explicit-Within-Region string
	       (1-* x (sheet-inside-left self)) (1-* y (sheet-inside-top self)) args
  )
)

(1defwhopper* (2Rendering-Mixin* :String-Out-Explicit-Within-Region)
	      (1string* start-pos-x start-pos-y screen-font alu 
		      &optional
		      (min-x (1send* self :Left-Margin-Size))
		      (min-y (1send* self :Top-Margin-Size))
		      (max-x (1-* (1send* self :Width)
				(1send* self :Right-Margin-Size)
			     )
		      )
		      (max-y (1-* (1send* self :Height)
				(1send* self :Bottom-Margin-Size)
			     )
		      )
		      (start 0)
		      end
		      (multi-line-line-height
			(1+* (send self :Vsp)
			    (let ((the-font screen-font))
			         (coerce-font the-font self)
				 (font-char-height the-font)
			    )
			)
		      )
	      )
  (1declare* (1special* printer:*explorer-postscript-font-map*))
  (1ecase* 2*Rendering-Type**
    (:Explorer (1continue-whopper*
		 string start-pos-x start-pos-y screen-font alu min-x min-y
		 max-x max-y start end multi-line-line-height
	       )
    )
    (:Postscript
;     (2Gsave*)
;     (1send* self :Specify-Clipping-Region min-x min-y max-x max-y)
     (1send* self :Postscript-String string
	    (1-* start-pos-x (1send* self :Left-Margin-Size))
	    (1-* start-pos-y (1send* self :Top-Margin-Size)) screen-font
     )
;     (2Grestore*)
    )
  )
)


(defwhopper (rendering-mixin :string-out-up)
	    (string &optional (start 0) (end nil)
		    (color (if (color-system-p self)
			       (sheet-foreground-color self)
			       -1
			   )
		    )
	   )
  (1ecase* 2*Rendering-Type**
    (:Explorer (1continue-whopper* string start end color))
    (:Postscript
     (multiple-value-bind (x y) (send self :read-cursorpos)
       (loop for index from start below (or end (length string))
	     for y-index from y by (send self :line-height)
	     do (send self :string-out-explicit-within-region
		      (string (aref string index)) x y-index
		      (send self :current-font) tv:alu-seta
		)
       )
     )
    )
  )
)

;-------------------------------------------------------------------------------

(1defflavor* 2Tv-Colon-Graphics-Mixin-Rendering-Mixin*
	   ()
	   ()
  (:Required-Flavors 2Rendering-Mixin*)
)

(1defwhopper* (2Tv-Colon-Graphics-Mixin-Rendering-Mixin* :Draw-Rectangle)
	      (rectangle-width rectangle-height x y
	       &optional (alu (2Draw-Alu-From-Window*))
	       (color (1send* self :foreground-color))
	      )
  (1ecase* 2*Rendering-Type**
    (:Explorer (1continue-whopper* rectangle-width rectangle-height
				    x y alu color
               )
    )
    (:Postscript (1send* self :Draw-Postscript-Rectangle
		       rectangle-width rectangle-height x y color
		       ":Draw-rectangle"
		 )
    )
  )
)

(1defwhopper* (2Tv-Colon-Graphics-Mixin-Rendering-Mixin* :Draw-Line)
	      (from-x from-y to-x to-y
		      &optional (alu (2Draw-Alu-From-Window*))
		      (draw-end-point t)
		      (color (1if* (color-system-p self)
				 (1send* self :foreground-color)
				 -1
			     )
		      )
	       )
  (1ecase* 2*Rendering-Type**
    (:Explorer (1continue-whopper* from-x from-y to-x to-y alu draw-end-point
				    color
               )
    )
    (:Postscript (1send* self :Draw-Postscript-Line
		        from-x from-y to-x to-y alu nil ":Draw-Line"
		 )
    )
  )
)

(1defwhopper* (2Tv-Colon-Graphics-Mixin-Rendering-Mixin* :Draw-Dashed-Line)
	      (from-x from-y to-x to-y &optional (alu char-aluf)
	    (dash-spacing 20) &rest args)
  (1ecase* 2*Rendering-Type**
    (:Explorer (1lexpr-continue-whopper*
		 from-x from-y to-x to-y alu dash-spacing args
	       )
    )
    (:Postscript
     (1if* 2*Use-Gray-Scale-Instead-Of-Dashed-Lines-P**
	 (1send* self :Draw-Postscript-Line
	        from-x from-y to-x to-y alu
		(1max* 0.0 (1min* (1-* 1.0 (1/* dash-spacing 100.0)) 1.0))
		":Draw-Dashed-Line as a gray line"
	 )
	 (1lexpr-continue-whopper*
	   from-x from-y to-x to-y alu dash-spacing args
	 )
     )
    )
  )
)

;-------------------------------------------------------------------------------

(1defflavor* 2W-Colon-Graphics-Mixin-Rendering-Mixin*
	   ()
	   ()
  (:Required-Flavors 2Rendering-Mixin*)
)

(1defwhopper* (2W-Colon-Graphics-Mixin-Rendering-Mixin* :Draw-Filled-Rectangle)
	      (left top rectangle-width rectangle-height
	    &optional
	    (color (if (color-system-p self)
		       (tv:sheet-foreground-color self)
		       black
		   )
	    )
	    &rest args
	    )
  (1ecase* 2*Rendering-Type**
    (:Explorer (1lexpr-continue-whopper*
		 left top rectangle-width rectangle-height color args
               )
    )
    (:Postscript (1send* self :Draw-Postscript-Rectangle
		       rectangle-width rectangle-height left top color
		       ":Draw-Filled-Rectangle"
		 )
    )
  )
)

(1defwhopper* (2W-Colon-Graphics-Mixin-Rendering-Mixin* :Draw-Line)
	      (from-x from-y to-x to-y &optional (thickness 1)
		      (color (if (color-system-p self)
				 (tv:sheet-foreground-color self)
				 black
			     )
		      )
		      (alu       w:normal)
		      (draw-end-point t)
		      (texture w:*default-texture*)
	      )
  (1ecase* 2*Rendering-Type**
    (:Explorer (1continue-whopper* from-x from-y to-x to-y thickness color
				    alu  draw-end-point texture
               )
    )
    (:Postscript (1send* self :Draw-Postscript-Line
		        from-x from-y to-x to-y alu nil ":Draw-Line"
		 )
    )
  )
)

(1defwhopper* (2W-Colon-Graphics-Mixin-Rendering-Mixin* :Draw-Dashed-Line)
	      (from-x from-y to-x to-y
		      &optional (thickness 1)
		      (color (if (color-system-p self)
				 (tv:sheet-foreground-color self)
				 black
			     )
		      )
		      (alu w:normal)
		      (dash-spacing 20)
		      space-literally-p
		      (offset 0)
		      (dash-length (floor dash-spacing 2))
		      (texture w:*default-texture*)
	      )
  (1ecase* 2*Rendering-Type**
    (:Explorer (1continue-whopper*
		 from-x from-y to-x to-y thickness color alu dash-spacing
		 space-literally-p offset dash-length texture
	       )
    )
    (:Postscript
     (1if* 2*Use-Gray-Scale-Instead-Of-Dashed-Lines-P**
	 (1send* self :Draw-Postscript-Line
	        from-x from-y to-x to-y alu
		(1max* 0.0 (1min* (1-* 1.0 (1/* dash-spacing 100.0)) 1.0))
		":Draw-Dashed-Line as a gray line"
	 )
	 (1continue-whopper*
	   from-x from-y to-x to-y thickness color alu dash-spacing
	   space-literally-p offset dash-length texture
	 )
     )
    )
  )
)

;-------------------------------------------------------------------------------

(1defflavor* 2Rendering-Stream*
	   ()
	   (tv:window)
  (:Default-Init-Plist :Save-Bits t :deexposed-typeout-action :Permit)
  :abstract-flavor
)

(1defflavor* 2Tv-Colon-Graphics-Mixin-Rendering-Stream*
	   ()
	   (2Tv-Colon-Graphics-Mixin-Rendering-Mixin*
	    2Rendering-Mixin*
	    2Rendering-Stream*
	   )
)

(1defflavor* 2W-Colon-Graphics-Mixin-Rendering-Stream*
	   ()
	   (2W-Colon-Graphics-Mixin-Rendering-Mixin*
	    2Rendering-Mixin* 
	    2Rendering-Stream*
	   )
)

;-------------------------------------------------------------------------------
;;; Required methods.

(1defmethod* (2Rendering-Stream* :X-Pl-Offset) () 0)
(1defmethod* (2Rendering-Stream* :Y-Pl-Offset) () 0)
(1defmethod* (2Rendering-Stream* :Logical-Left-Edge) () 0)
(1defmethod* (2Rendering-Stream* :Logical-Right-Edge) () (sheet-inside-width))
(1defmethod* (2Rendering-Stream* :Logical-Top-Edge) () 0)
(1defmethod* (2Rendering-Stream* :Logical-Bottom-Edge) () (sheet-inside-height))
(1defmethod* (2Rendering-Stream* :Scroll-To) (&rest ignore)
  (1send* self :Refresh :Complete-Redisplay)
)

;-------------------------------------------------------------------------------

(1defwhopper* (2Rendering-Stream* :Set-Cursorpos) (x y &optional (unit :Pixel))
  (1ecase* 2*Rendering-Type**
    (:Explorer (1continue-whopper* x y unit))
    (:Postscript (1continue-whopper* x y unit)
		 (1let* ((x (1ecase* unit
			    (:Pixel x)
			    (:Character (1** x (1send* self :Char-Width)))
			  )
		       )
		       (y (1ecase* unit
			    (:Pixel y)
			    (:Character (1** y (1send* self :Line-Height)))
			  )
		       )
		      )
		      (2With-Ps-Coords*
			(1format* 2*postscript-stream**
				  "~%  ~$ ~$ moveto  % --- Set-cursorpos"
				  (ps-x x) (ps-y y)
			)
		      )
		 )
    )
  )
)

(1defwhopper* (2Rendering-Stream* :String-Out)
	      (1string* &optional (start 0) (end nil) (color nil))
  (1ecase* 2*Rendering-Type**
    (:Explorer   (1continue-whopper* string start end color))
    (:Postscript
      (1let* ((the-font (1send* self :Current-Font)))
	  (1multiple-value-bind*
	    (x y) (1send* self :Read-Cursorpos)
	    (let ((x (+ x (sheet-inside-left self)))
		  (y (+ y (sheet-baseline-adj self) (sheet-inside-top  self)))
		 )
		 (1multiple-value-bind* (final-x final-y)
		     (1send* self :compute-motion
			    string 0 nil x y nil 0 nil nil
			    nil the-font
		     )
		   (1ignore* final-x)
		   (1if* (1or* (1>* final-y y)
			   (string-search-set '(#\tab) string)
		       )
		       (1loop* for i from 0
			     below (1or* end (1length* string)) do
			     (1send* self :Tyo (1aref* string i))
		       )
		       (1send* self :String-Out-Explicit-Within-Region
			      string x y the-font tv:alu-seta
		       )
		   )
		 )
	    )
	  )
     )
    )
  )
)

(1defwhopper* (2Rendering-Stream* :Line-Out)
	      (1string* &optional (start 0) (end nil) (color nil))
  (1ecase* 2*Rendering-Type**
    (:Explorer   (1continue-whopper* string start end color))
    (:Postscript (1loop* for i from 0 below (1or* end (1length* string)) do
		       (1send* self :Tyo (1aref* string i))
		 )
		 (1terpri* self)
    )
  )
)

(1defwhopper* (2Rendering-Stream* :Tyo) (ch &optional (font nil) (color nil))
  (1ecase* 2*Rendering-Type**
    (:Explorer   (1continue-whopper* ch font color))
    (:Postscript (1cond* ((1char-equal* ch #\newline)
			 (1send* self :Set-Cursorpos 0
			        (1+* (1send* self :Line-Height)
				    (1send* self :Cursor-Y)
				)
				:Pixel
			 )
			)
		        ((1char-equal* ch #\page) (1send* self :New-Page))
			(t (1let* ((the-font (1or* font (1send* self :Current-Font)))
				 (the-string
				   (1typecase* ch
				     (fixnum (1string* (1int-char* ch)))
				     (otherwise (1string* ch))
				   )
				 )
				)
			        (1multiple-value-bind*
				  (x y) (1send* self :Read-Cursorpos)
				  (let ((x (+ x (sheet-inside-left self)))
					(y (+ y (sheet-inside-top  self)))
				       )
				       (1send* self
					     :String-Out-Explicit-Within-Region
					     the-string
					     x y the-font
					     tv:alu-seta
				       )
				       (1multiple-value-bind* (new-x new-y)
					   (1send* self :compute-motion
						  the-string 0 nil x y nil
						  0 nil nil
						  nil the-font
					   )
					 (1send* self :Set-Cursorpos new-x new-y
						:Pixel
					 )
				       )
				  )
				)
			   )
			)
		  )
    )
  )
)

;-------------------------------------------------------------------------------


(1defwindow-resource* 2Tv-Colon-Rendering-Streams*
	()
  :Make-Window (2Tv-Colon-Graphics-Mixin-Rendering-Stream*)
  :Initial-Copies 0
)

(1defwindow-resource* 2W-Colon-Rendering-Streams*
	()
  :Make-Window (2W-Colon-Graphics-Mixin-Rendering-Stream*)
  :Initial-Copies 0
)

;-------------------------------------------------------------------------------

(1defflavor* postscript-stream
	   (2Rendering-Stream*
	    (resource '2Tv-Colon-Rendering-Streams*)
	    (rendering-type :Postscript)
	    left
	    top
	    right
	    bottom
	    (postscript-scale-factor 1)
	    (landscape-p nil)
	    (pages nil)
	    (bitmap-procedures (1make-hash-table* :Test #'1equal*))
	    (something-drawn-on-this-page-p nil)
	    (page-number 0)
	    postscript-stream
	   )
	   (sys:output-stream)
  :Initable-Instance-Variables
  (:Init-Keywords :Stream :Pathname)
)

(1defmethod* (2Postscript-Stream* :After :Init) (plist)
  (1setq* 2Rendering-Stream* (1allocate-resource* resource))
  (1cond* ((1getf* (1first* plist) :Pathname)
	  (1setq* 2Postscript-Stream*
		(1open* (1getf* (1first* plist) :Pathname) :Direction :Output)
	  )
	 )
	 ((1getf* (1first* plist) :Stream)
	  (1setq* 2Postscript-Stream* (1getf* (1first* plist) :Stream))
	 )
	 (t (1ferror* nil "No postscript stream specified."))
  )
  (1let* ((2*Postscript-Stream** postscript-stream)
        (*bitmap-procedures* bitmap-procedures)
       )
       (1declare* (1special* *bitmap-procedures*))
       (2With-Postscript-Environment*
	 (nil postscript-scale-factor landscape-p)
	 (1send* self :Setup-Postscript landscape-p)
	 (1setq* pages all-pages)
	 (1multiple-value-setq*
	   (left top right bottom)
	   (1values-list* (1first* all-pages))
	 )
	 (1send* 2Rendering-Stream* :Prepare-New-Page left top right bottom
	        postscript-scale-factor nil 0 0
         )
       )
  )
)

(1defmethod* (2Postscript-Stream* :Compute-All-Pages) (1ignore* scale ignore)
  (1let* ((2*Postscript-Stream** postscript-stream)
        (*postscript-scale-factor* scale)
       )
       (1declare* (1special* *postscript-scale-factor*))
       (1or* pages
	   (2With-Ps-Coords* (1list* (1list* 0 0 width-of-page height-of-page)))
       )
  )
)

(1defmacro* defpostscript-method ((flavor name))
 `(1defmethod* (,flavor ,name) (&rest args)
    (1let* ((2*Postscript-Stream** postscript-stream)
	  (*bitmap-procedures* bitmap-procedures)
	  (2*Something-Drawn-On-This-Page-P** something-drawn-on-this-page-p)
	 )
         (1Declare* (1special* *bitmap-procedures*))
	 (2With-Postscript-Environment*
	   (nil postscript-scale-factor landscape-p)
	   (2With-Postscript-Page-Environment*
	     (left top right bottom)
	     (1unwind-protect*
		 (1lexpr-send* 2Rendering-Stream* ,name args)
	       (1setq* something-drawn-on-this-page-p
		     2*Something-Drawn-On-This-Page-P**
	       )
	     )
	   )
	 )
    )
  )
)

(1defmethod* (2Postscript-Stream* :X-Pl-Offset) () 0)
(1defmethod* (2Postscript-Stream* :Y-Pl-Offset) () 0)
(2Defpostscript-Method* (2Postscript-Stream* :Tyo))
(2Defpostscript-Method* (2Postscript-Stream* :String-Out))
(2Defpostscript-Method* (2Postscript-Stream* :Line-Out))
(2Defpostscript-Method* (2Postscript-Stream* :Draw-Line))
(2Defpostscript-Method* (2Postscript-Stream* :Draw-Dashed-Line))
(2Defpostscript-Method* (2Postscript-Stream* :Draw-Rectangle))
(2Defpostscript-Method* (2Postscript-Stream* :Close))
(2Defpostscript-Method* (2Postscript-Stream* :Setup-Postscript))
(2Defpostscript-Method* (2Postscript-Stream* :Draw-Filled-Rectangle))

(1defmethod* (2Postscript-Stream* :Before :Close) (&rest ignore)
  (1send* self :New-Page)
)

(1defmethod* (2Postscript-Stream* :New-Page) (&rest ignore)
  (1let* ((2*Something-Drawn-On-This-Page-P**
	  (1or* (1and* (1boundp* '2*Something-Drawn-On-This-Page-P**)
		    2*Something-Drawn-On-This-Page-P**
	      )
	      something-drawn-on-this-page-p
	  )
	)
        (*page-number* page-number)
	(2*Postscript-Stream** postscript-stream)
	(*postscript-scale-factor* postscript-scale-factor)
       )
       (1declare* (1special* *page-number* *postscript-scale-factor*))
       (1unwind-protect*
	   (1Send* rendering-stream :New-Page nil)
	 (1setq* page-number *page-number*)
       )
       (1send* 2Rendering-Stream* :Prepare-New-Page left top right bottom
	      postscript-scale-factor nil 0 0
       )
  )
)

(1defmethod* (2Postscript-Stream* :After :Close) (&rest ignore)
  (1deallocate-resource* resource 2Rendering-Stream*)
  (1setq* 2Rendering-Stream* :Closed)
  (1send* 2Postscript-Stream* :Close)
)
