;;; -*- Mode:Common-Lisp; Package:TV; Base:10 -*-


(defvar *button* "Button")

(defvar *process-wait-strings-to-var-alist*
	'(("Button" *button*))
)

(defvar *Messy-Interactor-Default-Edges* :mouse)

(defvar *Messy-History-Default-Edges* :mouse)

(defvar *Messy-Inspector-Default-Edges* :mouse)

(defvar *Messy-Menu-Default-Edges* nil)

;-------------------------------------------------------------------------------


(defflavor shared-io-buffer-mixin ((io-buffer nil)) ()
  (:initable-instance-variables io-buffer)
)

(defmethod (shared-io-buffer-mixin :before :init) (ignore)
  ;; make sure that there is an io buffer.
  (or io-buffer (setq io-buffer (make-default-io-buffer)))
)

(defmethod (shared-io-buffer-mixin :create-pane)
	   (superior name flavor &rest options)
  (let ((pane (let ((*Button* (let ((*print-case* :Capitalize))
				   (format nil "~A" name)
			      )
		    )
		   )
		   (Apply #'make-window flavor :io-buffer io-buffer options)
	      )
	)
       )
       (push (list name pane) (send self :Internal-Panes))
       (send pane :Set-Superior superior)
       pane
  )
)

;-------------------------------------------------------------------------------

(defflavor messy-constraints-mixin
	   ((panes nil)
	    (internal-panes nil)
	    (constraints nil)
	   )
	   (shared-io-buffer-mixin)
  :Settable-Instance-Variables
)

(defmethod (Messy-Constraints-Mixin :After :Init) (ignore)
  (loop for (pane-name . flavor-inits) in panes do
	(lexpr-send self :create-pane (send self :Superior)
		    pane-name flavor-inits
        )
  )
)

(defmethod (Messy-Constraints-Mixin :After :Set-Superior) (to)
  (loop for (pane-name pane) in internal-panes do
	(send pane :Set-Superior to)
  )
)

(defmethod (Messy-Constraints-Mixin :Add-New-Pane) (pane)
  (push (list (gensym) pane) internal-panes)
)

(defmethod (Messy-Constraints-Mixin :Configuration) ()
  nil
)

(defmethod (Messy-Constraints-Mixin :Set-Configuration) (to)
  (ignore to)
  nil
)

(defmethod (Messy-Constraints-Mixin :Get-Pane) (name)
  (second (assoc name internal-panes))
)

(defmethod (Messy-Constraints-Mixin :Forget-Pane) (pane)
  (let ((internal (loop for entry in internal-panes
			when (equal pane (second entry))
			return entry
	          )
	)
       )
       (if internal
	   (progn (setq internal-panes (remove internal internal-panes))
		  (setq panes (remove (assoc (first internal) panes) panes))
	   )
	   nil
       )
  )
)

(defmethod (Messy-Constraints-Mixin :After :Kill) (&rest args)
  (loop for (pane-name pane) in internal-panes do
	(lexpr-send pane :Kill args)
  )
)

(defwhopper (messy-constraints-mixin :select-pane) (pane)
  (or (equal (send pane :Superior) (send self :Superior))
      (continue-whopper pane))
  (send self :set-selection-substitute pane))

(defmethod (messy-constraints-mixin :Exposed-Panes) ()
  (loop for (name pane) in internal-panes
	when (assoc name panes)
	collect pane
  )
)

;-------------------------------------------------------------------------------

(defparameter messy-inspector-menu-cmds
	     '(:All-Fl&Cl
	       Add-pane-cmd
	       Arglist-Cmd
	       break-cmd
	       Inspect-Compile-Cmd
	       DBG-Sg-CMD
	       delete-all-cmd
	       Document-Something-Cmd
	       inspect-edit-cmd
	       Inspect-Eval-Cmd
	       end-cmd
	       :documentation-cmd
	       :help-on-syntax
	       Inspect-MacroExpand-Cmd
	       :lisp-mode-cmd
	       modify-cmd
	       modify-print-cmd
	       refresh-cmd
	       trace-cmd
	      )
"A list of the names of all of the menu commands to be provided in the general
 inspector.
"
)

(defcommand Add-pane-cmd nil
  '(:description  "Add a new inspector pane to this inspector."
    :names ("Add Inspector") :keys ()
   )
  (declare (special frame))
  (send frame :Add-Inspectors 1)
)

(pushnew 'Messy-Inspector-Menu-Specifier *all-menu-specifiers*)

(defparameter Messy-Inspector-Menu-Specifier
  '((Messy-inspector-menu Messy-inspector-menu-cmds))
"A list of lists.  Each element in the list is a two-list.  The first element is
 the name of the menu, the second is the name of a list of the names of the 
 commands to go into the menu named by the first.
"
)

(defparameter Messy-Inspector-Command-Tables
  '((messy-inspector-cmd-table
     all-flavor-inspector-commands
     "Messy Inspector Commands"
     Messy-inspector
    )
   )
"The command tables used by the Messy inspector.  This is a list of lists.
 Each element in the list has the elements a) The name of the command table,
 b) the name of a list of all of the command names to be in that table,
 c) a string for the name of the command table and d) the flavor of frame
 for which the commands should work.
"
)

(defparameter *all-messy-menu-specifiers* '(Messy-Inspector-Menu-Specifier)
"A list of all of the names of the command menus in the general inspectoir
 frame.
"
)

(defparameter Messy-Inspector-Menu-Specifier
  '((messy-inspector-menu messy-inspector-menu-cmds))
"A list of lists.  Each element in the list is a two-list.  The first element is
 the name of the menu, the second is the name of a list of the names of the 
 commands to go into the menu named by the first.
"
)

(defparameter *All-Messy-Command-Table-Names*
	      (cons 'Messy-Inspector-Command-Tables *all-command-table-names*)
"The names of all of the command tables used by the general inspector."
)

(putprop 'messy-inspector *all-menu-specifiers* :all-menu-specifiers)
(putprop 'messy-inspector *All-Messy-Command-Table-Names*
	 :All-Command-Table-Names
)

(defflavor Messy-Inspector
	   ()
	   (basic-general-inspector messy-constraints-mixin)
  :Settable-Instance-Variables
  (:Default-Init-Plist
    :Number-Of-Inspectors 1
    :Borders nil
    :Edges '(0 0 10 10)
    :Label nil
    :Expose-P t
    :menu-panes '((menu messy-inspector-menu))
    :All-Command-Table-Names *All-Messy-Command-Table-Names*
    :All-Menu-Specifiers *All-Messy-Menu-Specifiers*
    :Active-Command-Tables nil
    :All-Command-Tables nil
  )
  :Settable-Instance-Variables
  :Gettable-Instance-Variables
  :Initable-Instance-Variables
)

(build-command-table 'messy-inspector-cmd-table 'messy-inspector
  '(:All-Fl&Cl
    Add-pane-cmd
    :help-on-syntax
    :help-on-inspected-data
    :end-cmd
    :options-menu
    :trace-method
    :fi-doc-cmd
    config-toggle-cmd
    mode
    ;;These are Inspector commands we are able to borrow.
    delete-all-cmd
    refresh-cmd
    page-up-cmd
    page-down-cmd
    page-to-top
    page-to-bottom
    break-cmd)
  :init-options '(:name "Messy Inspector Commands"))

(defmethod (Messy-Inspector :Before :Init) (ignore)
  (setq all-command-table-names *All-Messy-Command-Table-Names*)
  (setq All-Menu-Specifiers *All-Messy-Menu-Specifiers*)
)

(defmethod (Messy-Inspector :After :Init) (ignore)
  (setq ucl:Active-command-tables
	(mapcar #'first
		(symbol-value (first (send self :all-command-table-names)))
        )
  )
  (setq ucl:all-command-tables (get-all-command-table-names (type-of self)))
)

(Defmethod (Messy-Inspector :After :Expose) (&rest ignore)
  (process-run-function "Expose Messy Inspector"
    'expose-messy-inspector self
  )
)

(defmethod find-superior-for-new-inspect-pane ((me messy-inspector))
  (send me :Superior)
)

(defmethod (Messy-Inspector :Non-Inspect-Pane-Flavors) ()
  (declare (values interactor-flavor history-flavor menu-flavor))
  (values 'Messy-Interactor-Pane 'Messy-Inspector-History-Window
	  'Messy-Inspector-Menu-Pane
  )
)

(defmethod (messy-inspector :non-inspect-panes) ()
"Returns a list of the non inspect panes for self.  This is used at init time."
  (multiple-value-bind (interactor-flavor history-flavor menu-flavor)
      (send self :Non-Inspect-Pane-Flavors)
    (list `(interactor ,interactor-flavor
	    :pseudo-frame ,self
	    :label nil
	    :more-p nil
	    :Edges-From ,*Messy-Interactor-Default-Edges*
	    :font-map  ,(list (first *Inspector-Font-Map*)
			      (second *Inspector-Font-Map*)
			)
	    :who-line-message
	    "To inspect something use the <Mode> menu option to get the right input behaviour and then type what you want to inspect.
  Press HELP for a help menu, META-HELP for help on typed expressions.  R2: System Menu."
	   )
	  `(history ,history-flavor
	    :pseudo-frame ,self
	    :Edges-From ,*Messy-History-Default-Edges*
	    :line-area-mouse-doc
	    (:mouse-l-1 "Inspect the indicated data"
	     :mouse-m-1 "Remove it from the Inspector"
	    )
	    :normal-mouse-documentation
	    (:mouse-l-1 "Select data to inspect"
	     :mouse-m-2 "Lock/Unlock inspector pane"
	     :mouse-r-2 "System Menu"
	    )
	   )
	  `(menu ,menu-flavor
	    :pseudo-frame ,self
	    :Edges-From ,*Messy-Menu-Default-Edges*
	   )
    )
  )
)

(defmethod (Messy-Inspector :Inspect-Pane-Flavors) ()
  (declare (values (typeout-inspect-pane-flavor normal-inspect-pane-flavor)))
  (values 'messy-inspect-pane-with-typeout 'messy-inspect-pane)
)

(defmethod (messy-inspector :set-up-inspectors) (noi)
"Sets up Noi inspect panes for self.  This is done at frame init time."
  (multiple-value-bind (typeout-inspect-pane-flavor normal-inspect-pane-flavor)
      (send self :Inspect-Pane-Flavors)
    (dotimes (i noi)
      (let ((name1 (intern (format () "INSPECTOR-~D" I) "TV")))
	  (push `(,name1 ,(if (= i (1- noi))
			      typeout-inspect-pane-flavor
			      normal-inspect-pane-flavor)
		  :Pseudo-frame ,self
		  :Edges-From ,*Messy-Inspector-Default-Edges*
		  :current-object-who-line-message
		  ,(function (lambda (current-object)
			       (cond
				 ((equal current-object '(nil))
				  (send (tv:window-under-mouse) :help-string)
				 )
				 ((typep current-object 'flavor-operation-mixin)
				  `(:mouse-l-1 "Select data to inspect"
				    :Mouse-M-1
				     "Help on currently displayed data"
				    :mouse-m-2 "Lock/Unlock inspector pane"
				    :mouse-r-1
				    ,(or (catch-error
					   (format
					     () "Menu of operations on ~s"
					     (flavor-or-class-name
					       (send current-object :data)
					     )
					   )
					   nil
					 )
					 ""
				      )
				  )
				 )
				 (t '(:mouse-l-1
				       "Choose an item to inspect"))))))
		(send self :panes))
	  (push name1 inspectors)
       )
    )
  )
)

(defmethod (messy-inspector :add-inspectors) (&optional (noi 1))
"Adds Noi inspect panes to self."
  (multiple-value-bind (typeout-inspect-pane-flavor normal-inspect-pane-flavor)
      (send self :Inspect-Pane-Flavors)
    (loop for i from (length inspectors) below (+ (length inspectors) noi) do
      (let ((name1 (intern (format () "INSPECTOR-~D" I) "TV")))
	   (let ((pane-spec
		   `(,name1 ,(if (= i (1- noi))
			      typeout-inspect-pane-flavor
			      normal-inspect-pane-flavor)
		     :Pseudo-frame ,self
		     :Edges-From ,*Messy-Inspector-Default-Edges*
		     :current-object-who-line-message
		     ,(function (lambda (current-object)
				  (cond
				    ((equal current-object '(nil))
				     (send (tv:window-under-mouse) :help-string)
				    )
				    ((typep current-object
					    'flavor-operation-mixin
				     )
				     `(:mouse-l-1 "Select data to inspect"
				       :Mouse-M-1
				       "Help on currently displayed data"
				       :mouse-m-2 "Lock/Unlock inspector pane"
				       :mouse-r-1
				       ,(or (catch-error
					      (format
						() "Menu of operations on ~s"
						(flavor-or-class-name
						  (send current-object :data)
						)
					      )
					      nil
					    )
					    ""
					 )
				     )
				    )
				    (t '(:mouse-l-1
					  "Choose an item to inspect"))))))))
	     (push pane-spec panes)
	     (let ((pane (lexpr-send self :Create-Pane
				     (send self :Superior) pane-spec
			 )
		   )
		  )
		  (setq inspectors (append inspectors (list pane)))
	     )
	   )
       )
    )
  )
)

;-------------------------------------------------------------------------------

(defun make-sure-can-be-exposed (sheet)
  (if (typep sheet 'tv:sheet)
      (let ((superior (send sheet :Superior)))
	   (multiple-value-bind (left top right bottom) (send sheet :Edges)
	     (if (tv:sheet-within-sheet-p sheet superior)
		 t
		 (send sheet :Set-Edges
		       (min (+ (sheet-inside-width superior)
			       (sheet-inside-left superior)
			    )
			    (max 0 (sheet-inside-left superior)
				 left
			    )
		       )
		       (min (+ (sheet-inside-height superior)
			       (sheet-inside-top superior)
			    )
			    (max 0 (sheet-inside-top superior)
				 top
			    )
		       )
		       (min (+ (sheet-inside-width superior)
			       (sheet-inside-left superior)
			    )
			    (max 0 (sheet-inside-left superior)
				 right
			    )
		       )		 
		       (min (+ (sheet-inside-height superior)
			       (sheet-inside-top superior)
			    )
			    (max 0 (sheet-inside-top superior)
				 bottom
			    )
		       )
		 )
	     )
	   )
      )
      :Argument-Not-A-Sheet
  )
)

(defun expose-messy-inspector (inspector)
  (assert (fboundp 'tv:expose-inferior-window-safely) ()
	  "Expose code not loaded."
  )
  (if inspector
      (progn (send inspector :Bury)
	     (let ((interactor (send inspector :Get-Pane 'interactor)))
	          (if interactor
		      (progn (make-sure-can-be-exposed interactor)
			     (send interactor :Expose)
			     (loop for (pane-name pane)
				   in (send inspector :Internal-Panes)
				   unless (or (equal pane-name 'interactor)
					      ;(typep pane 'w:menu)
					  )
				   do (make-sure-can-be-exposed pane)
				      (tv:expose-inferior-window-safely
					pane interactor
					'inspector-interaction-pane
				      )
			     )
			     (send interactor :mouse-select)
		      )
		      nil
		  )
	     )
      )
      nil
  )
)

;-------------------------------------------------------------------------------

(defflavor pseudo-frame-mixin
	   (pseudo-frame)
	   ()
  :Settable-Instance-Variables
  :Gettable-Instance-Variables
  :Initable-Instance-Variables
  (:Default-Init-Plist :Expose-P t :Activate-p t)
)

(defmethod (Pseudo-frame-Mixin :Before :Expose) (&rest ignore)
  (make-sure-can-be-exposed self)
)

(defmethod Find-Inspector-Window-1 ((pane Pseudo-frame-Mixin) the-superior)
  (ignore the-superior)
  (send pane :Pseudo-frame)
)

(defmethod (Pseudo-frame-Mixin :After :Kill) ()
  (send pseudo-frame :Forget-pane self)
)

;-------------------------------------------------------------------------------

(defflavor messy-interactor-pane
	   ()
	   (Pseudo-frame-Mixin inspector-interaction-pane)
  :Settable-Instance-Variables
  :Gettable-Instance-Variables
  :Initable-Instance-Variables
)

(defmethod (Messy-Interactor-Pane :After :expose) (&rest ignore)
  (let ((inspector (find-inspector-window self)))
       (if (typep inspector 'Messy-Inspector)
	   (tv:without-recursion (Expose-Messy-Inspector inspector))
	   nil
       )
  )
)

(defmethod (Messy-Interactor-Pane :After :Kill) (&rest ignore)
  (catch-error (send pseudo-frame :Kill) nil)
)

;-------------------------------------------------------------------------------

(defflavor abstract-messy-inspect-pane
	   ()
	   ()
  (:Required-Flavors inspect-window)
)

(defmethod (Abstract-Messy-Inspect-Pane :After :Kill) ()
  (let ((iframe (find-inspector-window self)))
       (let ((panes (send iframe :inspectors)))
	    (send iframe :Set-Inspectors (remove self panes))
       )
  )
)

(defmethod (abstract-messy-inspect-pane :toggle-lock) ()
  (let ((panes (send (find-inspector-window self) :inspectors))
	(iframe (find-inspector-window self))
       )
       (let ((num-of-locked-panes
	       (loop for el in panes
		     counting (send el :locked-p) into x
		     finally (return x)
	       )
	     )
	     (lock-x (- (send self :width) 50.))
	     (lock-y 3.)
	     (num-panes (length panes))
	    )
	    (cond (locked-p
		   (setq locked-p nil)
		   (w:prepare-sheet (self)
		     (w:draw-char
		       (send (send iframe :superior)
			     :parse-font-descriptor 'fonts:icons)
		       98. lock-x lock-y w:alu-andca self
		     )
		   )
		  )
		  ((or (= 1 num-panes) (>= num-of-locked-panes (- num-panes 1)))
		   (beep)
		  )
		  (t (setq locked-p t)
		     (w:prepare-sheet (self)
		       (w:draw-char
			 (send (send iframe :superior)
			       :parse-font-descriptor 'fonts:icons)
			 98. lock-x lock-y w:alu-xor self
		       )
		     )
		  )
	    )
       )
  )
)

(defflavor messy-inspect-pane-with-typeout
	   ()
	   (Abstract-Messy-Inspect-Pane
	    Pseudo-frame-Mixin
	    general-inspect-pane-with-typeout
	   )
  :Settable-Instance-Variables
  :Gettable-Instance-Variables
  :Initable-Instance-Variables
  (:Default-Init-Plist :Save-Bits t)
)

(defflavor Messy-Inspect-Pane
	   ()
	   (Abstract-Messy-Inspect-Pane Pseudo-frame-Mixin General-Inspect-pane)
  :Settable-Instance-Variables
  :Gettable-Instance-Variables
  :Initable-Instance-Variables
  (:Default-Init-Plist :Save-Bits t)
)


;-------------------------------------------------------------------------------

(defflavor messy-inspector-menu-pane
	   ()
	   (Pseudo-frame-Mixin inspector-menu-pane)
  :Settable-Instance-Variables
  :Gettable-Instance-Variables
  :Initable-Instance-Variables
  (:Default-Init-Plist
    :Save-Bits t
    :Minimum-Width 80
    :Minimum-Height 315
    :Columns 1
  )
)

(defmethod (Messy-Inspector-Menu-Pane :After :Init) (init-plist)
  (lexpr-send self :Set-Edges
    (let ((minimum-width (or (get init-plist :minimum-width) 80))
	  (minimum-height (or (get init-plist :minimum-height) 315))
	 )
         (multiple-value-list
	   (mouse-specify-rectangle-set-sheet
	     () () () () superior minimum-width minimum-height
	   )
	 )
    )
  )
)

(build-menu 'messy-inspector-menu
	    'messy-inspector
  :default-item-options `(:font ,*general-inspector-menu-item-font*)	
  :item-list-order messy-inspector-menu-cmds
)

(defflavor messy-inspector-history-window
	   ()
	   (Pseudo-frame-Mixin general-inspector-history-window)
  :Settable-Instance-Variables
  :Gettable-Instance-Variables
  :Initable-Instance-Variables
  (:Default-Init-Plist :Save-Bits t)
)

(defmethod (messy-inspector-history-window :After :Kill) (&rest ignore)
  (catch-error (send pseudo-frame :Kill) nil)
)

;-------------------------------------------------------------------------------

(w:add-system-key
  #\m-I
  'messy-inspector
  "Messy Inspector - examine complex data structures."
  t
)


;-------------------------------------------------------------------------------

(advise mouse-specify-rectangle-set-sheet :Around :Set-Process-Wait-Messages nil
  (let ((old #'process-wait))
       (letf ((#'process-wait
	       #'(lambda (whostate &rest args)
		   (let ((entry (assoc whostate
				       *Process-Wait-Strings-To-Var-Alist*
				       :Test #'string-equal
				)
			 )
			)
		        (apply old
			       (if entry (symbol-value (second entry)) whostate)
			       args
			)
		   )
		 )
	      )
	     )
	     :Do-It
       )
  )
)
