;;; -*- Mode:Common-Lisp; Package:TV; Base:10; Fonts:(TVFONT) -*-


;;; This software developed by:
;;;	James Rice
;;; at the Stanford University Knowledge Systems Lab in 1986, 1987.
;;;
;;; This work was supported in part by:
;;;	DARPA Grant F30602-85-C-0012

;;;----------------------------------------------------------------------
;;;  Much of this file is derived from code licensed from Texas Instruments
;;;  Inc.  Since we'd like them to adopt these changes, we're claiming
;;;  no rights to them, however, the following restrictions apply to the
;;;  TI code:

;;; Your rights to use and copy Explorer System Software must be obtained
;;; directly by license from Texas Instruments Incorporated.  Unauthorized
;;; use is prohibited.

;;;                           RESTRICTED RIGHTS LEGEND

;;;Use, duplication, or disclosure by the Government is subject to
;;;restrictions as set forth in subdivision (b)(3)(ii) of the Rights in
;;;Technical Data and Computer Software clause at 52.227-7013.
;;;
;;;                     TEXAS INSTRUMENTS INCORPORATED.
;;;                              P.O. BOX 2909
;;;                           AUSTIN, TEXAS 78769
;;;                                 MS 2151
;;;
;;; Copyright (C) 1986,1987 Texas Instruments Incorporated. All rights reserved.
;;;----------------------------------------------------------------------

#||

;;; How to write your own perspective:
;;; ==================================

;;; The following comment has some hints on how to make your own inspector
;;; perspectives.  There's a detailed worked example.  Clearly, the whole
;;; of the inspector's code is filled with worked examples of one sort or
;;; another, so if the example below doesn't seem to be quite enough, just
;;; look for the code that implements the sort of thing that you want it to
;;; look like that already exists.  As a reasonable heuristic, if you like
;;; the way that CLOS classes are inspected the look for a flavor called
;;; show-clos-class or some such.  Just flavor inspect the dependent flavors
;;; of tv:inspection-data and look for something that approximates what
;;; you're looking for.

;;; Let's pretend that we have a class of thing for which we
;;; want to have special support in the inspector.  Let's define
;;; it as:

(defflavor rule (name knowledge-source body) ()
  :Initable-Instance-Variables
  :Settable-Instance-Variables
  :Gettable-Instance-Variables
)

;;; We'll make rules with the DefRule form:

(defmacro defrule (name (knowledge-source) &body body)
 `(progn (setf (get ',name :rule)
	       (make-instance 'rule
			      :Name ',name
			      :knowledge-source ',knowledge-source
			      :Body ',body
	       )
         )
	 (def ,name)
	',name
  )
)

;;; Now let's define a rule:

(defrule rabbit-from-a-hat (magic) some clever rule implementation)

;;; Now we wnat to look at rules specially.  This will mean that, by
;;; default, when the inspector sees a Rule it will have our new behaviour,
;;; but when we Mouse-L-2 on it we'd like to be able to see it as the
;;; flavors instance that it really is.

;;; What's more we know that there's a mapping from symbols to rules
;;; (the rule is stored on the PList of the rule's name under the :Rule
;;; property).  We should therefore extend the perspectives for symbols
;;; so that any defined rules will appear as options.

;-------------------------------------------------------------------------------

;;; First we'll define the way to inspect rules.  This is done by defining
;;; a new flavor of inspection data.

(defflavor show-rule () (generic-middle-button-mixin inspection-data))

;;; Inspection-data is the basic flavor that will deal with how to display
;;; the rule.  Generic-Middle-Button-Mixin will give some reasonable behaviour
;;; for when we get middle buttoned on.

;;; The flavor Show-Rule will have one instance variable, called Data,
;;; that will contain the rule instance.

;;; Now we have to define a number of methods for the above flavor to control
;;; the way it's displayed and the way it acts.

;;; First we'll define what gets returned when we're middle buttoned on.
;;; This makes sure that we get the rule.

(defmethod (show-rule :middle-button-result) ()
"Returns the Rule."
  data
)

(defmethod (show-rule :format-concisely) (stream)
"Prints out the rule for the history window and such like.
"
  ;; Note, here we make a special case for history windows so that we'll be
  ;; printed out as "Rule foo" and foo will be mouse-sensitive and will point to
  ;; the name of the rule.  This is just icing on the cake.  When we're not in
  ;; the history window we don't need to say "Rule" because we'll know from
  ;; the context of the display what it is.  We aren't slashifying here because
  ;; we really don't want to have to worry about this sort of thing.  The reason
  ;; for haveing "Rule" in the history window is to be able to distinguish
  ;; between this rule and the symbol of the same name.
  (if (in-history-window-p stream)
      (format stream "Rule ~" (list (send data :Name) nil (send data :Name)))
      (format stream "~A" (send data :Name))
  )
)

;;; Now, let's deal with making the display for the rule.  This is controlled
;;; by the :Generate-Item method, which returns two values: the mouse-sensitive
;;; items for the body of the display and then the specification for the
;;; title of the display.

;;; This method can be as complicated as you make it.  However, there are a
;;; number of simple things to make each item out of.
;;; *blank-line-item* - This leaves a blank line.
;;; *one-space-item* - leaves a space.
;;; "string" - Just prints a string without any special fonts or
;;;    mouse-sensitivity.
;;; (:colon <number>) - prints a colon and tabs to column <number>.  This is
;;;    useful in tabulating things like defstruct slots.
;;; (:font <number> "string") - will display String in font <number> non-mouse
;;;    sensitively.
;;; (:item1 <type> value &optional print-fn) - will make a mouse-sensitive
;;;    item of type <type> whose value is Value and whose printing behaviour
;;;    can be modified by supplying Print-Fn.
;;;    The most important values for <Type> are 'instance, which is used for
;;;    displaying things like Show-Rule, and 'Named-Structure-Value, which is
;;;    a convenient way to display normal values.
;;;    print-fn must be a function with arglist:
;;;    (instance stream &optional (level 0)).  Instance is the thing being
;;;    printed, stream is the stream to print it to and Level is the depth of
;;;    nesting that we're at.  A good function to call inside such a print-fn
;;;    is inspection-data-print-item-concisely, which takes the same args.
;;;    For instance, defining a print-fn as below would make sure that in this
;;;    case show-rules are printed as Knowledge-source-name:rule-name.  The
;;;    resulting mouse-sensitive item will still point to the original
;;;    show-rule.

;;; A significant thing to note here is the function Allocate-Data.  This
;;; will create/find in its cache an instance of (in this case) show-rule
;;; that is representing the data item.  This you should always use
;;; (allocate-data 'show-rule <rule-instance>) to get an instance of show-rule,
;;; your should never call make-instance.

(defun print-rule-verbosely (instance stream &optional (level 0))
  (ignore level)
  (let ((name (format nil "~A:~A"
		      (send (send instance :Data) :Knowledge-Source)
		      (send (send instance :Data) :Name)
	      )
	)
	(*print-escape* nil)
       )
       (format stream "~A" name)
  )
)

;;; Now let's define that method:

(defmethod (show-rule :generate-item) ()
  (values
    ;; This is the list of inspector items.
    `(,*blank-line-item*
      ((:font 1 "Details of ")
       (:item1 instance ,(allocate-data 'show-rule data) print-rule-verbosely))
      ,*blank-line-item*
      ((:font 1 "Knowledge Source")
       (:Colon 30)
       ,(string (send data :Knowledge-Source)))
      ((:Font 1 "Body")
       (:Colon 30)
       (:Item1 named-structure-value ,(send data :Body))))
    ;; This is the title for the window.
    `(:font fonts:hl12bi :string ,(format nil "Rule ~A" (send data :name)))))

;;; Now let's define a :Help method so that when the user middle buttons on
;;; the display he'll get something useful.  If we'd defined any menus on
;;; the right button this might be a good place to talk about them.

(defmethod (show-rule :help) ()
  (format nil "You're currently looking at a rule called ~S" (send data :name)))

;-------------------------------------------------------------------------------

;;; This is all very well, but what if we wan't to be able to use the modify
;;; command on sundry properties of the rule, for instance , we might want to
;;; be able to smash the knowledge source with which it is associated.

;;; The following will show you how to do this:

;;; First we must define a new sort of inspection type for the knowledge source
;;; itself.  This will have associated with it a function that will know how to
;;; do the modify operation.

;;; Let's call this type Rule-Slot:

(defun (:property Rule-slot set-function)
       (item new-value object)
  ;;; Note: Object is our Show-Rule instance, so we must point to the rule
  ;;; it contains.
  (let ((slot (third (second item))))
       (set-in-instance object slot new-value)
  )
)

;;; Rule slot names are only mouse sensitive when they're being modified.
(defprop Rule-slot t only-when-modify) 

;;; Now we'll have to change our generate item method so that we get to see
;;; the new item types:  Note, the symbols Knowledge-source and Body are the
;;; names of the slots in Rules.

(defmethod (show-rule :generate-item) ()
  (values
    ;; This is the list of inspector items.
    `(,*blank-line-item*
      ((:font 1 "Details of ")
       (:item1 instance ,(allocate-data 'show-rule data) print-rule-verbosely))
      ,*blank-line-item*
      ((:Item1 Rule-Slot Knowledge-Source)
       (:Colon 30)
       ,(string (send data :Knowledge-Source)))
      ((:Item1 Rule-Slot Body)
       (:Colon 30)
       (:Item1 named-structure-value ,(send data :Body))))
    ;; This is the title for the window.
    `(:font fonts:hl12bi :string ,(format nil "Rule ~A" (send data :name)))))

;;; This works fine in a simplistic case but, as you'll see if you try it,
;;; we've lost the nice fonts and printing for when we print out the
;;; attribute names.  Let's fix it:

(defun print-rule-slot-name (slot stream &optional (level 0))
  (ignore level)
  (format stream "~A" (string-capitalize (symbol-name slot) :spaces t))
)

(defmethod (show-rule :generate-item) ()
  (values
    ;; This is the list of inspector items.
    `(,*blank-line-item*
      ((:font 1 "Details of ")
       (:item1 instance ,(allocate-data 'show-rule data) print-rule-verbosely))
      ,*blank-line-item*
      ((:font 1 (:Item1 Rule-Slot Knowledge-Source print-rule-slot-name))
       (:Colon 30)
       ,(string (send data :Knowledge-Source)))
      ((:font 1 (:Item1 Rule-Slot Body print-rule-slot-name))
       (:Colon 30)
       (:Item1 named-structure-value ,(send data :Body))))
    ;; This is the title for the window.
    `(:font fonts:hl12bi :string ,(format nil "Rule ~A" (send data :name)))))

;;; That's better.  Note we're allowed to make compound items like
;;; (:font 1 (:item1 ...)).
;-------------------------------------------------------------------------------

;;; Now we have to tell the inspector about the new perspectives.

(Defperspective :rule (x show-x)
  :show-x-type-for-perspective Show-rule
  :This-Perspective-Applicable-Function
    (and (not (typep show-x 'Show-Rule))
         (or (and (symbolp x) (get x :rule)) (typep x 'Rule))
    )
  :menu-item-name "Rule"
  :New-Inspect-Function
    (allocate-data 'Show-Rule (if (symbolp x) (get x :rule) x))
  :Priority 11
)

;;; The representation of Rule will automatically get a default perspective
;;; to inspect it as a generic flavors instance.  This has a priority of 10,
;;; so we just make sure that the perspectiev defined above has a higher
;;; perspective and we're well away.

;-------------------------------------------------------------------------------

;;; Now as a final complication, maybe we'd like to have a right button menu
;;; for show-rules.  This might do interesting things to the knowledge base.

;;; First let's define a function to put up the menu:

(defparameter *rule-menu-options*
	    '(("Frob Knowledge Source" :Value :Frob-Knowledge-Source
	       :Documentation "Frob in some way with the knowledge source."
	      )
	     )
)

(defun select-rule-operation (Rule window)
"Given a rule and the inspector window, selects something to do with the rule
 and does it.
"
  (let ((choice
	   (ucl::smart-menu-choose
	     *rule-menu-options* :label
	     (format nil "Operations on ~A" (send Rule :Name))
	   )
	)
       )
       (if choice
	   (send window choice rule)
	   nil
       )
  )
)

;;; The definition above requires that we define a method to implement the
;;; :frob-knowledge-source option.

(defmethod (general-inspector :Frob-Knowledge-Source) (Rule)
"Just a dummy definition, since I don't know what to frob in this toy example."
  (print (send Rule :Knowledge-Source))
)

;;; Now we have to trap the mouse-click:

(defwrapper (show-rule :handle-mouse-click)
	    ((blip inspector) &body body)
"Makes sure that show-rule things can have a right button menu."
  `(if (= (fourth blip) #\Mouse-r-1)
       (select-rule-operation data inspector)
       . ,body
   )
)

;;; Finally we should make sure that the who-line mouse doc knows about
;;; the new menu.

(defmethod (show-rule :who-line-doc) (ignore &optional ignore)
"Returns a who-line doc string for show-rules."
  '(:Mouse-L-1 "Inspect Rule"
    :Mouse-M-1 "Set *"
    :Mouse-R-1 "Menu of Rule operations"
   )
)

;;; Gosh.  That's the end of this example.


||#

;-------------------------------------------------------------------------------
;;; This is where things really start.
;-------------------------------------------------------------------------------

(eval-when (compile load)
  (load-tools '(:Development-Tool-Consistency-Enhancements
	        :Inspector-Enhancements
	       )
  )
)

;-------------------------------------------------------------------------------
(defvar *general-inspector-menu-item-font*
	(if (eq sys:(processor-type microcode-type-code) :micro-explorer)
	    fonts:cptfont
	    fonts:tvfont
	)
"The font for the command menu in general inspector frames."
)

(defvar *general-inspector-number-of-menu-pane-rows*
        (if (equal *general-inspector-menu-item-font*
		   (if (eq (sys:processor-type sys:microcode-type-code)
			   :micro-explorer
		       )
		       fonts:cptfont fonts:tvfont
		   )
	    )
	    4
	    6
	)
"The number of rows for the command menu in general inspector frames."
)

(defvar *General-Inspector-Configuration* :three-panes
"Default configuration for the General Inspector"
)

(defparameter *all-command-table-names*
	      '(General-Inspector-Command-Tables
		Inspector-Command-Tables
		Flavor-Inspector-Command-Tables
	       )
"The names of all of the command tables used by the general inspector."
)

(defparameter Flavor-Inspector-Command-Tables
  '((flavor-inspector-cmd-table
     all-flavor-inspector-commands
     "Flavor Inspector Commands"
     flavor-inspector
    )
   )
"The command tables used by the Flavor inspector.  This is a list of lists.
 Each element in the list has the elements a) The name of the command table,
 b) the name of a list of all of the command names to be in that table,
 c) a string for the name of the command table and d) the flavor of frame
 for which the commands should work.
"
)

(defparameter Inspector-Command-Tables
  '((inspector-menu-cmd-table
     inspector-menu-cmds
     "Inspector menu commands"
     inspect-frame
    )
    (inspector-other-cmd-table
     inspector-non-menu-cmds
     "Inspector menu commands"
     inspect-frame
    )
   )
"The command tables used by the Inspector.  This is a list of lists.
 Each element in the list has the elements a) The name of the command table,
 b) the name of a list of all of the command names to be in that table,
 c) a string for the name of the command table and d) the flavor of frame
 for which the commands should work.
"
)

(defparameter General-Inspector-Command-Tables
	      (append Flavor-Inspector-Command-Tables Inspector-Command-Tables)
"The command tables used by the General-Inspector.  This is a list of lists.
 Each element in the list has the elements a) The name of the command table,
 b) the name of a list of all of the command names to be in that table,
 c) a string for the name of the command table and d) the flavor of frame
 for which the commands should work.
"
)

(putprop 'general-inspector *all-command-table-names* :all-command-table-names)

(defparameter *all-menu-specifiers* '(General-Inspector-Menu-Specifier)
"A list of all of the names of the command menus in the general inspectoir
 frame.
"
)

(defparameter General-Inspector-Menu-Specifier
  '((general-inspector-menu general-inspector-menu-cmds))
"A list of lists.  Each element in the list is a two-list.  The first element is
 the name of the menu, the second is the name of a list of the names of the 
 commands to go into the menu named by the first.
"
)

(putprop 'general-inspector *all-menu-specifiers* :all-menu-specifiers)

(defparameter *all-prompts*
	      (list '("Inspect: " :Set-Inspect nil
		      General-Inspector-Command-Tables
		     )
		    '("> " :set-> nil General-Inspector-Command-Tables)
		    `(,(if (clos-p)
			   "Flavor\/Class\/Method: "
			   "Flavor\/Method: "
		       )
		      :set-flavor/class
		      ,(if (clos-p)
			   '(method-specs class-instance class-names
			     clos-method-specs flavor-instance flavor-names
			     ucl:command-names
			    )
			   '(method-specs flavor-instance flavor-names
			     ucl:command-names
			    )
		       )
		       General-Inspector-Command-Tables
		     )
	      )
"A list which maps prompt names to command table specifications.  Each element
 in the list represents a new prompt and hence typein mode.  The element is a
 list, whose elements are as follows: a) A string to print as the font, b)
 the name of a method to call to select that typein mode and c) the name
 of the command tables to use for that typein mode.
"
)

(defparameter all-flavor-inspector-commands
	     '(:all-fl&cl :help-on-syntax
	       :help-on-inspected-data :end-cmd
	       :options-menu :trace-method :fi-doc-cmd :toggle-config-cmd
	       mode delete-all-cmd refresh-cmd
	       page-to-top page-to-bottom break-cmd
	      )
"A list of the names of all of the commands to be provided in the general
 inspector.
"
)

(defparameter general-inspector-menu-cmds
	     '(:all-fl&cl
	       Arglist-Cmd
	       break-cmd
	       Inspect-Compile-Cmd
	       :toggle-config-cmd
	       DBG-Sg-CMD
	       delete-all-cmd
	       Document-Something-Cmd
	       inspect-edit-cmd
	       Inspect-Eval-Cmd
	       end-cmd
	       :documentation-cmd
	       :help-on-syntax
	       Inspect-MacroExpand-Cmd
	       :lisp-mode-cmd
	       modify-cmd
	       modify-print-cmd
	       refresh-cmd
	       trace-cmd
	      )
"A list of the names of all of the menu commands to be provided in the general
 inspector.
"
)

;-------------------------------------------------------------------------------


(defmethod (show-method :method-from-show-method) ()
"Extracts a method from a show-method object.  The method is in the second slot
 in the method table entry.
"
  (second data)
)

(defmethod (show-method-details :method-from-show-method-details) ()
"Extracts a method from a show-method-details object.  The method is in the
 second slot in the method table entry.
"
  (second aux-data)
)


(defflavor general-inspector-history-window () (inspect-history-window)
  (:default-init-plist
    :line-area-mouse-doc
      '(:mouse-l-1 "Inspect the indicated object"
	:mouse-m-1 "Set *"
       )
    :normal-mouse-documentation
      '(:mouse-l-1 "Inspect the indicated object"
       )
  )
)

(defmethod (general-inspector-history-window :who-line-documentation-string) ()
"A who line doc string method for the history window."
  (let ((frame (Find-Inspector-Window self)))
       (if (send frame :inspector-typein-p)
	   (send (send frame :get-pane 'interactor)
		 :who-line-documentation-string
	   )
	   (if sensitive-history-item
	       (cond
		 ((let ((item (get-mouse-sensitive-item)))
		       (when (and
       ;;; xxxx (si:send-if-handles superior :inspection-data-active?)
			       (typep item 'inspection-data))
			 (send item :who-line-doc nil)
		       )
		  )
		 )
		 ((or modify-mode (and (not setting-mode) (key-state :hyper)))
		  '(:mouse-r-2 "System Menu")
		 )
		 (setting-mode
		  '(:mouse-l-1 "Set with this value" :mouse-r-1 "Abort")
		 )
		 (t (send self :get-normal-mouse-documentation)))
	       (cond
		 ((or modify-mode (and (not setting-mode) (key-state :hyper)))
		  '(:mouse-r-2 "System Menu")
		 )
		 (setting-mode
		  '(:mouse-l-1
		     "Select a value to set with" :mouse-r-2 "System Menu"
		   )
		 )
		 (t (send self :get-normal-mouse-documentation))
	       )
	   )
       )
  )
)

(defflavor general-inspect-window ()
	   (inspect-window)
  (:documentation
"This is a generalised version of the old inspector.  It allows the user to do
 normal inspection things, flavor inspection and class inspection all in one
 tool.  It also supports a perspectives mechanism whereby the user can view
 certain data structures in a number of different ways.  Middle button clicks
 cause things to be set to = and * and echoed in the interactor.  L2 invokes
 the perspectives mechanism.
"
  )
)

(defmethod (general-inspect-window :after :init) (ignore)
"Record the new middle button in the who-line doc.  We can't do this in the
 default init plist because the inspect-window flavor sets it in its init
 method.
"
  (setq normal-mouse-documentation
      '(:mouse-l-1 "Inspect list item"
	:mouse-m-1 "Set *"
	:mouse-m-2 "Lock/Unlock inspector pane"
       )
  )
)

(defflavor general-inspect-pane () (general-inspect-window inspect-pane)
  (:documentation
    "The flavor of inspector window used by the general inspector."
  )
)

(defflavor general-inspect-pane-with-typeout ()
	   (general-inspect-window inspect-pane-with-typeout)
  (:documentation
    "The flavor of inspector window used by the general inspector's main pane."
  )
)

(defvar *Inhibit-Inspection-Data* nil)

(Defun (:property isolated-instance-slot set-function) (item new-value object)
  (let* ((slot (send (third (second item)) :data))
         (message-name (intern (string-append "SET-" slot) "")))
    (if (get-handler-for message-name object)
        (catch-error (send object message-name new-value) t)
        (set-in-instance object slot new-value)))) 

(defprop isolated-instance-slot nil only-when-modify) 

(defvar *all-inspection-data-item-types* '(instance isolated-instance-slot)
"All of the item types that act like Instance for inspection data."
)


(defun expand-as-plist-p (object slot-name slot-value)
  (ignore object)
  (and (consp slot-value)
       (let ((string
	       (typecase slot-name
		 (string slot-name)
		 (symbol (symbol-name slot-name))
		 (otherwise nil)
	       )
	     )
	    )
	    (and string
		 (or (search "PROPERTY-LIST" string :Test #'char=)
		     (search "PLIST" string :Test #'char=)
		 )
	    )
       )
  )
)

(defun expand-as-alist-p (object slot-name slot-value)
  (ignore object)
  (and (consp slot-value)
       (let ((string
	       (typecase slot-name
		 (string slot-name)
		 (symbol (symbol-name slot-name))
		 (otherwise nil)
	       )
	     )
	    )
	    (and string (search "ALIST" string :Test #'char=))
       )
  )
)

(defvar *expand-slot-alist*
	`((Expand-As-PList-P "PList" get-items-for-plist)
	  (Expand-As-AList-P "AList" get-items-for-alist)
	 )
)

(defun maybe-expand-names-and-values-for-slot
  (results object slot-bound-p slot-name slot-value window
   &optional (prefix "    ")
  )
  (If (and *show-plists-and-alists-for-show-slots-p* slot-bound-p)
      (loop for (test title expander) in *Expand-Slot-Alist*
	    when (funcall test object slot-name slot-value)
	    do (if title (push `((:Font 1 ,title)) results) nil)
	       (loop for item
		     in (funcall expander object slot-value window prefix)
		     do (push item results)
	       )
      )
      nil
  )
  results
)

;;;Edited by James Rice            9 Mar 90  14:07
(defmethod (general-inspect-window :object-instance) (obj)
  (let ((maxl -1)
        result flavor)
    ;;If the instance to inspect is an instance of INSPECTION-DATA and our
    ;;superior's INSPECTION-DATA-ACTIVE? is T, let the instance generate
    ;;the inspection item.  This is used in special-purpose inspectors
    ;;such as the flavor inspector.
    (if (and (or (not (boundp '*inhibit-inspection-data*))
		 (not *inhibit-inspection-data*)
	     )
	     (typep obj 'inspection-data))
	;;; fix put in here by JPR to support more sophisticated item
	;;; generation.  This is particularly important for item generators
	;;; that need to be able to point to the window, not just the flavor
	;;; inspector.
	(if (send obj :operation-handled-p :generate-item-specialized)
	    (send obj :generate-item-specialized self)
	    ;; Window flavor added by JPR.  08/01/90 11:05:06
	    (multiple-value-bind (text-items inspector-label window-flavor)
		(send obj :generate-item)
	      (values text-items () 'inspect-printer () inspector-label
		      nil window-flavor)))
        ;;Otherwise inspect the flavor instance in the normal fashion.
        (progn
          (setq flavor (si:instance-flavor obj))
          (setq result
                (list '("")
                      `("An object of flavor " (:item1 flavor ,(type-of obj))
                        ".  Function is "
			(:item1 flavor-function ,(si:instance-function obj)))))
          (if flavor
	      (let ((flavors
		      (mapcar #'(lambda (fl) (get fl 'sys:flavor))
			 (if *sort-components-when-displaying-instances*
			     (sort (copy-list (si:flavor-depends-on-all flavor))
				   *Instance-Component-Sort-Comparator*
			     )
			     (si:flavor-depends-on-all flavor)))))
		(loop for flav in flavors do
		      (let ((ivars (si:flavor-local-instance-variables flav)))
			(loop for iv in ivars do
			      (setq maxl
				(max (flatsize (ucl:first-if-list iv)) maxl)))))
		(loop for flav in flavors
		      for ivars =
		        (if *sort-ivs-when-displaying-instances*
			    (sort
			      (copy-list
				(si:flavor-local-instance-variables flav))
			      *Instance-Component-Sort-Comparator*)
			    (si:flavor-local-instance-variables flav))
		    do
		    (if (and ivars
			  *separate-components-when-displaying-instances*
			)
			(push `((:font 1 "IVs from ")
				(:item1 instance
					,(allocate-data 'show-flavor flav)))
			      result)
			nil)
		    (loop for iv in ivars
			  for sym = (ucl:first-if-list iv)
			  for slot-bound-p = (boundp-in-instance obj sym)
			  for slot-value = (and slot-bound-p
						(symeval-in-instance obj sym)
					   )
			  for print-value = (if slot-bound-p
					       `(:item1 instance-value
						 ,slot-value)
					       "unbound")
			  do
			  (push
			    `((:item1 isolated-instance-slot
				      ,(allocate-data
					 'show-isolated-instance-variable
					 sym (list flav flavor obj)))
			      (:colon ,(+ 2 maxl))
			      ,print-value)
			    result)
			  (if (equal sym 'si::hash-array)
			      (let ((window-items
				      (make-window-items-for-hash-table
					(send obj :hash-array) 'identity nil)))
				(dolist (element window-items)
				  (push element result))))
			  (setq result
				(maybe-expand-names-and-values-for-slot
				  result obj slot-bound-p sym slot-value
				  self ; = window
				)
			  )
		    )
		    (if (and ivars
			  *separate-components-when-displaying-instances*
			)
			(push *blank-line-item* result)
			nil)
		    ))
	      (let ((ivars (%p-contents-offset
			     (%p-contents-as-locative-offset obj 0)
			     %instance-descriptor-bindings)))
		(do ((bindings ivars (cdr bindings))
		     (i 1 (1+ i)))
		    ((null bindings))
		  (setq maxl (max (flatsize (car bindings)) maxl)))
		(do ((bindings ivars (cdr bindings))
		     (sym)
		     (i 1 (1+ i)))
		    ((null bindings))
		  (setq sym (car bindings))
		  (push
		    `((:item1 instance-slot ,sym) (:colon ,(+ 2 maxl))
		      ,(if (= dtp-null (%p-data-type (%instance-loc obj i)))
			   "unbound"
			   `(:item1 instance-value ,(%instance-ref obj i))))
		    result)
		  (if (equal (first bindings) 'si::hash-array)
		      (let ((window-items
			      (make-window-items-for-hash-table
				(send obj :hash-array) 'identity nil)))
			(dolist (element window-items)
			  (push element result)))))))
          (nreverse result)))))

;-------------------------------------------------------------------------------

(defflavor show-isolated-instance-variable
	   () (generic-middle-button-mixin auxiliary-data-mixin inspection-data)
  ;; first of aux-data is the defining flavor second is the flavor of
  ;; the calling instance.
)

(defmethod (show-isolated-instance-variable :format-concisely) (stream)
  (if (in-history-window-p stream)
      (let ((flav (allocate-data 'show-flavor (first aux-data))))
	   (format stream "~ IV of ~"
		   (list data t data) (list flav t flav)
	   )
      )
      (format stream "~s" data)
  )
)

(defwhopper (show-isolated-instance-variable :who-line-doc) (&rest args)
  (let ((result (lexpr-continue-whopper args))
	(stringl (format () "Inspect ~s, the definer of ~s"
			 (si:flavor-name (first aux-data)) data
		 )
        )
	(stringr (format () "Inspect IVs of ~S"
			 (si:flavor-name (second aux-data))
		 )
        )
	(stringm2 (format () "Set MAR to this IV"))
       )
       (if (member :mouse-r-1 result)
	   (setf (getf result :mouse-r-1) stringr)
	   (setq result (cons :mouse-r-1 (cons stringr result)))
       )
       (if (and (fboundp 'set-mar-for-location)
		(typep (third aux-data) 'instance)
	   )
	   (if (member :meta-mouse-m result)
	       (setf (getf result :Meta-Mouse-M) stringm2)
	       (setq result (cons :Meta-Mouse-M (cons stringm2 result)))
	   )
	   nil
       )
       (if (member :mouse-l-1 result)
	   (setf (getf result :mouse-l-1) stringl)
	   (setq result (cons :mouse-l-1 (cons stringl result)))
       )
       result
   )
) 


(defwhopper (show-isolated-instance-variable :handle-mouse-click)
	    (blip flavor-inspector)
  (let ((mouse-char (fourth blip)))
       (case mouse-char
	 (#\Mouse-l      
	  (send flavor-inspector :inspect-thing 'show-flavor (first aux-data))
	 )
	 (#\Meta-Mouse-m      
	  (if (and (fboundp 'set-mar-for-location)
		   (typep (third aux-data) 'instance)
	      )
	      (funcall 'set-mar-for-location
		       (locate-in-instance (third aux-data) data)
	      )
	      (beep)
	  )
	 )
	 (#\Mouse-r    
	  (send flavor-inspector :inspect-thing 'show-instance-variables
		(second aux-data)
          )
	 )
	 (otherwise (continue-whopper blip flavor-inspector))
       )
  )
)


;-------------------------------------------------------------------------------

(DEFMETHOD (BASIC-INSPECT :OBJECT-NAMED-STRUCTURE)    ;!added hash tables
           (OBJ &AUX (MAXL -1) ALIST DEFSTRUCT-ITEMS RESULT NSS D)
  (SETQ NSS (NAMED-STRUCTURE-P OBJ))
  (PUSH `("Named structure of type " (:ITEM1 NAMED-STRUCTURE-P ,NSS)) RESULT)
  (PUSH '("") RESULT)
  (COND
    ((SETQ D (GET NSS 'SI::DEFSTRUCT-DESCRIPTION))
     (SETQ ALIST (SI::DEFSTRUCT-DESCRIPTION-SLOT-ALIST D))
     (DO ((L ALIST (CDR L)))
	 ((NULL L) NIL)
       (SETQ MAXL (MAX (FLATSIZE (CAAR L)) MAXL)))
        ;; For a named structure, each line contains the name and the value
     
     (DO ((L ALIST (CDR L)))
	 ((NULL L) NIL)
       (PUSH `(;(:ITEM1 NAMED-STRUCTURE-SLOT ,(CAAR L))
	       (:Item1 isolated-defstruct-slot
		       ,(allocate-data 'Show-Defstruct-Slot
				       (CAAR L) (list nss obj)))
	       (:COLON ,(+ 2 MAXL))
	       (:ITEM1 NAMED-STRUCTURE-VALUE
		       ,(CATCH-ERROR
			  (FUNCALL (SI:DEFSTRUCT-SLOT-DESCRIPTION-REF-MACRO-NAME
				     (CDAR L))
				   OBJ)
			  NIL)))
	     RESULT)))
    ((SETQ DEFSTRUCT-ITEMS (GET NSS 'SI::DEFSTRUCT-ITEMS))
     (DOLIST (ELT DEFSTRUCT-ITEMS)
       (SETQ MAXL (MAX (FLATSIZE ELT) MAXL)))
     ;;; For a named structure, each line contains the name and the value
     (DOLIST (ELT DEFSTRUCT-ITEMS)
       (PUSH `((:ITEM1 NAMED-STRUCTURE-SLOT ,ELT)
	       (:COLON ,(+ 2 MAXL))
	       (:ITEM1 NAMED-STRUCTURE-VALUE
		       ,(CATCH-ERROR (FUNCALL ELT OBJ) NIL)))
	     RESULT))))
  (if (typep obj 'si:hash-table)                      ;!
      (progn (push '("") result)
	     (push '("Hash Array Elements") result)
	     (push '("") result)
	     (loop for element in (make-window-items-for-hash-table
				     obj 'identity nil
				  )
	      do (push element result)
	     )
	     (push '("") result)
      )
  )
  (IF (AND (ARRAYP OBJ) (ARRAY-HAS-LEADER-P OBJ))
      (SEND SELF :OBJECT-ARRAY OBJ T (NREVERSE RESULT))
      ;;;mention-leader is always T
      (VALUES (NREVERSE RESULT) OBJ 'INSPECT-PRINTER)))

;-------------------------------------------------------------------------------

(Defun (:property isolated-defstruct-slot set-function) (item new-value object)
  (let ((slot (send (third (second item)) :data)))
       (setf (clos:slot-value object slot) new-value)
  )
)

(defmethod clos:slot-value-using-class
	   ((class clos:structure-class) instance slot-name)
  (let ((dd (get (clos:class-name class) 'sys:defstruct-description)))
       (let ((alist (sys:defstruct-description-slot-alist dd)))
	    (let ((entry (assoc slot-name alist :Test #'eq)))
	         (if entry
		     (funcall (sys:defstruct-slot-description-ref-macro-name
				(rest entry)
			      )
			      instance
		     )
		     (clos:slot-missing class instance slot-name
					'clos:slot-value
		     )
		 )
	    )
       )
  )
)

(defmethod (setf clos:slot-value-using-class)
	   (new-value (class clos:structure-class) instance slot-name)
  (let ((dd (get (clos:class-name class) 'sys:defstruct-description)))
       (let ((alist (sys:defstruct-description-slot-alist dd)))
	    (let ((entry (assoc slot-name alist :Test #'eq)))
	         (if entry
		     (eval
		       `(setf (,(sys:defstruct-slot-description-ref-macro-name
				  (rest entry)
			        )
			       ,instance
			      )
			      ,new-value
			)
		     )
		     (clos:slot-missing class instance slot-name
					'clos:slot-value
		     )
		 )
	    )
       )
  )
)

(defprop isolated-defstruct-slot nil only-when-modify) 

(defflavor show-defstruct-slot
	   () (generic-middle-button-mixin auxiliary-data-mixin inspection-data)
  (:Documentation
    "aux-data is a list of the form (class-of-defstruct instance)."
  )
)

(pushnew 'isolated-defstruct-slot *All-Inspection-Data-Item-Types*)

(Defmethod (show-defstruct-slot :format-concisely) (stream)
  (if (in-history-window-p stream)
      (let ((class (first aux-data)))
	   (format stream "~ SLOT of ~"
		   (list data t data) (list class t class)
	   )
      )
      (format stream "~s" data)
  )
)

(defwhopper (show-defstruct-slot :who-line-doc) (&rest args)
  (let ((result (lexpr-continue-whopper args))
	(stringm2 (format () "Set MAR to this Slot"))
       )
       (if (and (fboundp 'set-mar-for-location)
		(member :meta-mouse-m result)
	   )
	   (setf (getf result :Meta-Mouse-M) stringm2)
	   (setq result (cons :Meta-Mouse-M (cons stringm2 result)))
       )
       result
   )
) 

(defmethod (locf ticlos:slot-value-using-class)
	   ((class clos:structure-class) object slot-name)
  (let ((desc (get (clos:class-name class) 'sys:defstruct-description)))
       (let ((entry
	       (assoc slot-name
		      (sys:defstruct-description-slot-alist desc) :Test #'eq
	       )
	     )
	    )
	    (if entry
		(locf (aref object (second entry)))
		(clos:slot-missing class object slot-name 'locf)
	    )
       )
  )
)

(defwhopper (show-defstruct-slot :handle-mouse-click)
	    (blip flavor-inspector)
  (let ((mouse-char (fourth blip)))
       (case mouse-char
	 (#\Mouse-l      
	  (send flavor-inspector :inspect-thing 'Show-Generic-Object-Thing
		(first aux-data)
	  )
	 )
	 (#\Meta-Mouse-m      
	  (if (fboundp 'set-mar-for-location)
	      (funcall 'set-mar-for-location
		(locf (clos:slot-value (second aux-data) data))
	      )
	      (beep)
	  )
	 )
	 (otherwise (continue-whopper blip flavor-inspector))
       )
  )
)


;-------------------------------------------------------------------------------

(defun gi-doc-cmd ()
  "Display some brief documentation about each of the Inspector's panes."
  (declare (special frame))
  (si:with-help-stream
    (window :label "Documentation for General Inspector"
	    :superior tv:default-screen
    )
    (format window
	    "
                               FLAVOR/CLASS INSPECTOR HELP
  ------------------------------------------------------------------------------
                    *** OPTIONAL THIRD INSPECTION PANE ***

    Displays previously inspected item.
 -------------------------------------------------------------------------------
                        *** OPTIONAL SECOND PANE ***

    Displays previously inspected item.
 -------------------------------------------------------------------------------
                        *** MAIN INSPECTION PANE ***

    This pane displays the structure of the most recently inspected item.
    Specify objects to inspect by either:

      * Entering them into the Interaction Pane or,
      * Clicking Mouse-Left on the mouse sensitive elements of previously
        inspected items.
      * Clicking Mouse-Left-2 will either simply act like Mouse-Left or, if it
        can, will allow you to view the data in a different perspective.  This
        could take the form either of simply changing to a different perspective
        or of popping up a menu if there are a number of different known
        perspectives.
      * Clicking Mouse-Middle on an object will cause that object to be echoed
        in the interaction pane and will make that object the current value of
        both * and of =.  The previous values of * and ** will ripple through
        as appropriate.
      * Clicking Mouse-Middle-2 anywhere in the pane toggles the pane's locked
        status.  When locked, the inspected item in that pane will be frozen
        until unlocked.  Only two of the 3 panes may be locked.
      * Clicking Mouse-Right on items in this pane will put up a menu of
        appropriate operations, when there is such a menu.
 -------------------------------------------------------------------------------
                             *** HISTORY PANE ***

    This pane displays a list of the objects that have been inspected.

    To bring an object back into the Main Inspection Pane, click
    Mouse-Left on that object in this pane.

    To remove an item from the History Pane, position the mouse-cursor to the 
    left of the item until the cursor becomes a right-pointing arrow (this is
    the items \"line area\"). Now click Mouse-Middle.

 -------------------------------------------------------------------------------
                             *** COMMAND MENU ***

    Click Mouse-Left to select a command.

 -------------------------------------------------------------------------------
                           *** INTERACTION PANE *** 

    The behaviour of this pane is sensitive to the current Mode.  This is set
    by either the Mode menu command of the s-m keystroke.  Typeing to these
    modes will do the following:

      Inspect - will cause the VALUE of the expression that you type to be
          inspected.
      > - does not cause anything to be inspected at all.  It allows you to
          do normal lisp interaction.  This is useful when you want to evaluate
          lisp expressions using values seen inthe inspector as arguments. 
          Do this by setting the value of *, ** and *** by using the middle
          button mouse click mentioned above.
      Flavor/Class/Method - This is the most complicated input mode.  It allows
          You to enter a Flavor name, a Class name, a Flavors method, a CLOS
          method or a Generic Function.

          For Flavors input:
	  -- a flavor name to inspect, terminated by pressing the RETURN key
	  -- a method specification to inspect.  The syntax is
	       (Flavor-Name Method-Name)
	     or
	       (Flavor-Name Method-Type Method-Name)
	     or
	       Flavor-Name Method-Name
	     or    
	       Flavor-Name Method-Type Method-Name
	  
	     The last two types of expressions are terminated by pressing
             the RETURN key.
	     Method-Type is one of the following:
	       :AFTER :AND :AROUND :BEFORE :CASE :DEFAULT :OR :OVERRIDE :WRAPPER

          For CLOS input:	  
	  -- a class name to inspect, terminated by pressing the RETURN key
	  -- a method specification to inspect.  The syntax is
	       (Class-Name Generic-Function-Name)
	     or
	       (Class-Name Method-Type Generic-Function-Name)
	     or
	       (Generic-Function-Name Class-Name)
	     or
	       (Generic-Function-Name Method-Type Class-Name)
	     or
	       Class-Name Generic-Function-Name
	     or    
	       Class-Name Method-Type Generic-Function-Name
	     or
	       Generic-Function-Name Class-Name
	     or    
	       Generic-Function-Name Method-Type Class-Name
	  
	     The last four types of expressions are terminated by pressing
             the RETURN key.
	     Method-Type is one of the following:
	       :AFTER :BEFORE :CASE

	  While typing these expressions, you may press the SPACE Bar to
          complete a Class, Flavor or method name.  You may also use the Input
          Editor completion commands summarized below:
	  
	     CTRL-ESCAPE  -- Recognition Completion (same as the SPACE Bar)
	     CTRL-\/      -- List Recognition Completions
	     SUPER-ESCAPE -- Apropos Completions
                             (complete word as an inner substring)
	     SUPER-\/     -- List Apropos Completions
	     HYPER-ESCAPE -- Spelling Corrected Completion
                             (corrects minor typos)
	     HYPER-\/     -- List Spelling Corrected Completions

 -------------------------------------------------------------------------------
  ")
    (show-all-commands-for-frame frame window)))


;-------------------------------------------------------------------------------

(defflavor basic-general-inspector
  ((flavor-inspect-p nil)
   (all-command-table-names *all-command-table-names*)
   (all-menu-specifiers *all-menu-specifiers*)
   (all-prompts *all-prompts*)
  )
  (basic-flavor-inspector)
  (:Default-Init-Plist
    :Active-command-tables
    (mapcar #'first
	    (symbol-value (first (send self :all-command-table-names)))
    )
    :all-command-tables (get-all-command-table-names (type-of self))
    :menu-panes '((menu general-inspector-menu))
    :Typein-modes nil
    :Basic-help '(gi-doc-cmd)
    :Prompt (first (first (send self :all-prompts)))
    :print-results? 'general-inspector-print-values?
    :inspection-data-active? nil
  )
  :Gettable-instance-variables
  :Initable-instance-variables
  :Settable-instance-variables
)


(defflavor general-inspector ()
	   (basic-general-inspector Inspector-Framifying-Mixin)
)

;-------------------------------------------------------------------------------


(let ((compiler:compile-encapsulations-flag t))
     (advise (:method basic-flavor-inspector :before :expose) :around
	     :dont-run-if-general-inspector nil
       (if (typep self 'basic-general-inspector)
	   nil
	   :do-it
       )
     )
)

(defmethod (basic-general-inspector :before :expose) (&rest ignore)
"Make sure that our configuration is what we want it to be."
  (send self :set-configuration *General-Inspector-Configuration*)
)

(defparameter *debugging?* nil "Just used by me for debugging.")
;(setq *Debugging?* t)

(defmethod (basic-general-inspector :inspect-thing)
           (type thing &optional (aux-data nil aux-supplied?))
  (let ((inspected-thing
	  ;;; this is just in case we get really screwed up.
	  (if *debugging?*
	      (list type thing aux-data)
	      (inspect-real-value
	       `(:value
		 ,(if aux-supplied?
		    (allocate-data type thing aux-data)
		    (allocate-data type thing))
		 ,history)))))
    (inspect-flush-from-history inspected-thing history)
    (send history :append-item inspected-thing)
    (update-panes)))

(defmethod (basic-general-inspector :pseudo-update-*) (value)
"This is sort of like the normal update-*, only it doesn't do any of the
 frobbing around with the history window.  We don't want to do any of that if
 the user just middle buttons on something.
"
  (let ((items (send history :items)))
       (let ((nitems (+ 1 (if items (array-active-length items) 0))))
	    (setf (aref items (+ 1 nitems)) value)
            (if (>= nitems 1) (set-element-from-history '=   -1 nitems items))
            (if (>= nitems 1) (set-element-from-history '*   -1 nitems items))
	    (if (>= nitems 1) (set-element-from-history '**   1 nitems items))
	    (if (>= nitems 2) (set-element-from-history '***  2 nitems items))
       )
  )
)

(defmethod (basic-general-inspector :set-up-equal) (value)
"Sets up = and * to point to Value.  ** and *** ripple up."
  (declare (special =))
  (let ((*print-level* 3)
	(*print-length* 5)
       )
       (send self :pseudo-update-* value)
       (prin1 * *terminal-io*)
  )
  (send self :handle-prompt)
)

(defmethod (basic-general-inspector :inspect-info-middle-click)
	   (&optional something)
"Makes the value that was clicked on the current value of * and =."
  (let ((thing (if something
		   something
		   (inspect-real-value ucl:kbd-input)
	       )
	)
       )
       (send self :set-up-equal thing)
  )
)

(defun find-inspection-object (blip)
"Given a mouse blip extracts something from it to inspect.  Mouse blips
 have a number of different formats so we have to frob around a bit.
"
  (if (consp blip)
      (cond ((consp (rest blip))
	     (cond ((and (typep (second blip) 'inspection-data)) (second blip))
		   ((and (consp (rest (rest blip))) (typep (third blip) 'inspection-data))
		    (third  blip)
		   )
		   ((consp (second blip)) (find-inspection-object (second blip)))
		   ((and (consp (rest (rest blip))) (consp (rest (rest (rest blip))))
			 (typep (fourth blip) 'inspection-data)
		    )
		    (fourth blip)
		   )
		   ((equal :Value (first blip))
		    (allocate-data 'show-generic-object-thing (second blip))
		   )
		   ((and (consp (rest (rest blip))) (instancep (third blip)))
		    (send (third blip) :send-if-handles :current-object)
		   )
		   (t nil)
	      )
	    )
	    (t nil)
      )
      nil
  )
)

(defun method-p (x)
"Is true if x is a (flavors) method."
  (and (functionp x)
       (consp (function-name x))
       (or (equal :Method (first (function-name x)))
	   (equal 'ticlos:method (first (function-name x)))
       )
       (fdefinition-safe (function-name x))
  )
)

(deftype :method ()
"Is a method."
  `(satisfies method-p)
)

(deftype method-function ()
"Is a method."
  `(satisfies method-p)
)

(deftype type-specifier ()
  "The type which denotes type specifiers."
  `(satisfies type-specifier-p)
)

(defun data-from-method (method)
"Given a flavors method turns it into data suitable to ge given to allocate
 data, i.e. it finds the flavor and the method table entry and returns these
 as a list.
"
  (let ((method-spec (function-name method))
	(flavor (get (second (function-name method)) 'si:flavor))
       )
       (let ((mte (rest (rest (rest (assoc (first (last method-spec))
					   (sys:flavor-method-table flavor)
					   :Test #'eq
				    )
			      )
			)
		  )
	     )
	    )
	    (let ((entry (assoc method-spec mte :Test #'equal)))
		 (let ((method-data
			 (or entry
			     (list method-spec method
				   (si:method-plist method-spec)
			     )
			 )
		       )
		      )
		      (list flavor method-data)
		 )
	    )
       )
  )
)

(defun data-from-clos-method (method)
"Given a clos method turns it into data suitable to ge given to allocate data,
 i.e. it finds a class and returns it and the method as a list.
"
  (list (first (method-type-specifiers-safe method)) method)
)

(defun data-from-clos-generic-function (gf)
"Given a generic function turns it into data suitable to ge given to allocate
 data, i.e. it just returns a list of itself twice.
"
  (list (Function-Generic-Function-Safe gf) (Function-Generic-Function-Safe gf))
)

(defun data-from-class (instance)
"Maps an instance into the class of the instance."
  (list (class-named-safe (class-of-safe instance)))
)


(defwrapper (inspection-data :handle-mouse-click)
	    ((blip flavor-inspector) &body body)
"Supports the mouse-l-2 click as well as the others."
  `(let ((object (find-inspection-object (send flavor-inspector :kbd-input))))
        (if (and (= (fourth blip) #\mouse-l-2) object)
	    (send flavor-inspector :inspect-info-left-2-click
		  object
	    )
            . ,body
	)
   )
)

(defmethod (basic-general-inspector :around :handle-unknown-input) (cont mt ignore)
  (let (inspection-data)
    (cond
      ;;first see if they toggled a pane's locked status
      ((AND (CONSP ucl:kbd-input)
            (EQ (FIRST ucl::kbd-input) :MOUSE-BUTTON)
            (eql (SECOND ucl::kbd-input) #\MOUSE-M-2))
       (SEND (THIRD ucl::kbd-input) :toggle-lock))   
      ((AND (CONSP ucl:kbd-input)
            (eql (fourth ucl::kbd-input) #\MOUSE-M-2))
       (SEND (THIRD ucl::kbd-input) :toggle-lock))   
      ;;If not a blip, let UCL's method handle unknown input
      ((NEQ UCL::INPUT-MECHANISM 'UCL::UNKNOWN)
      ;?((OR (EQ UCL::INPUT-MECHANISM 'UCL::TYPEIN) (not (LISTP THING)))
       (FUNCALL-WITH-MAPPING-TABLE CONT MT :HANDLE-UNKNOWN-INPUT))
      ;;Blip contains an inspection-data instance and we are currently inspecting treating them specially.
      ((AND ;INSPECTION-DATA-ACTIVE?
            (OR
              ;;Blip in form (INSTANCE (:ITEM1 INSTANCE <inspection-data instance>) <window> <mouse button>).
              ;;These are the standard inspection-data blips from the inspection panes.
              (AND (member (FIRST UCL::KBD-INPUT)
			   *All-Inspection-Data-Item-Types* :Test #'eq)
                   (Eq (FIRST (SECOND UCL::KBD-INPUT)) :ITEM1)
                   (TYPEP (THIRD (SECOND UCL::KBD-INPUT)) 'INSPECTION-DATA)
                   (SETQ INSPECTION-DATA (THIRD (SECOND UCL::KBD-INPUT))))
              ;;Blip in form (:VALUE <inspection-data instance> <window> <mouse button>).  These blips come from
              ;;the inspection history and always have flavor information in them.
              (AND (EQ (FIRST UCL::KBD-INPUT) :VALUE)
                   (TYPEP (SECOND UCL::KBD-INPUT) 'INSPECTION-DATA)
                   (SETQ INSPECTION-DATA (SECOND UCL::KBD-INPUT)))
	      ))
       ;;Have the INSPECTION-DATA handle the mouse blip.  (Each type of info handles the various mouse buttons differently.)
       (SEND INSPECTION-DATA :HANDLE-MOUSE-CLICK UCL::KBD-INPUT SELF))
      ((and (eq (first ucl::kbd-input) :value)
	    (setq inspection-data (Map-Into-Show-X (second ucl::kbd-input) t)))
       (SEND INSPECTION-DATA :HANDLE-MOUSE-CLICK UCL::KBD-INPUT SELF))
      ((EQ (FIRST UCL::KBD-INPUT) :LINE-AREA)
       (SELECTOR (FOURTH UCL::KBD-INPUT) eql
         (#\MOUSE-L   (SEND SELF :INSPECT-INFO-LEFT-CLICK))
	 ;;; modeed here by JPR to support perspectives.
	 (#\MOUSE-L-2 (SEND SELF :INSPECT-INFO-LEFT-2-CLICK))
	 ;;; This only gets called when the line area mouse blinker is on
	 ;;; i.e. we only get removal in this special case, otherwise normal
	 ;;; mouse-m action applies.
         (#\MOUSE-M
          ;; Delete from line area
          (SEND HISTORY :FLUSH-OBJECT (INSPECT-REAL-VALUE UCL::KBD-INPUT))
          (SEND HISTORY :SET-CACHE NIL)
          ;;make sure the pane is unlocked if they deleted that item
          (LOOP for iw in inspectors
                when (EQ (INSPECT-REAL-VALUE UCL::KBD-INPUT) (SEND iw :current-object))
                do (SEND iw :set-locked-p nil))
          (UPDATE-PANES))
	 (T
          (SEND SELF :INSPECT-INFO-RIGHT-CLICK))))
      ;;Middle click on inspected Lisp object--inspect it, leaving source in one of the windows
      ((eql (FOURTH UCL::KBD-INPUT) #\MOUSE-M)
       (send self :set-up-equal (INSPECT-REAL-VALUE UCL::KBD-INPUT)))
      ((eql (FOURTH UCL::KBD-INPUT) #\MOUSE-L-2)
       ;;; modded here by JPR to support perspectives.
       (send self :inspect-info-left-2-click))
      ;; right Click on inspected Lisp Object-- inspect its function definition, or itself if no function.
      ((eql (FOURTH UCL::KBD-INPUT) #\MOUSE-R)
       (SEND SELF :INSPECT-INFO-RIGHT-CLICK))
      ((KEY-STATE :HYPER)
       ;; Hyper means modify the slot we are pointing at.
       (IF (OR (NULL (FIRST UCL::KBD-INPUT)) (NULL (GET (FIRST UCL::KBD-INPUT) 'SET-FUNCTION)))
           (FORMAT USER "~&Cannot set this component.")
           (PROGN
             (INSPECT-SET-SLOT UCL::KBD-INPUT USER HISTORY INSPECTORS)
             (UPDATE-PANES)))
       (SEND SELF :HANDLE-PROMPT))
      (T ;; Otherwise inspect UCL:KBD-INPUT.
       (SEND SELF :INSPECT-INFO-LEFT-CLICK)))))

(defmethod (basic-general-inspector :inspect-info-right-click) ()
  (beep))

(defun get-all-menu-specifiers (flavor-name)
"Gets the menu specifiers property from the flavor."
  (let ((menus (get flavor-name :all-menu-specifiers)))
       (if menus
	   (apply #'append (mapcar #'symbol-value menus))
	   (ferror nil
		   "~A frame does not have a :all-menu-specifiers property"
		   flavor-name
	   )
       )
  )
)

(defun get-all-command-tables (flavor-name)
"Gets the all-command-tables property from the flavor."
  (let ((names (get flavor-name :all-command-table-names)))
       (if names
	   (apply #'append (mapcar #'symbol-value names))
	   (ferror nil
		   "~A frame does not have a :all-command-table-names property"
		   flavor-name
	   )
       )
  )
)

(defun get-all-command-table-names (flavor-name &optional (error-p t))
"Gets the all-command-table-names property from the flavor."
  (let ((names (get flavor-name :all-command-table-names)))
       (if names
	   (apply #'append
		  (mapcar #'(lambda (a-name)
			      (mapcar #'first (symbol-value a-name))
			    )
			    names
		  )
	   )
	   (or (loop for flav
		     in (sys:flavor-depends-on (get flavor-name 'sys:flavor))
		     for result = (get-all-command-table-names flav nil)
		     when result
		     return result
	       )
	       (if error-p
		   (ferror nil
		    "~A frame does not have a :all-command-table-names property"
			   flavor-name
		   )
		   nil
	       )
	   )
       )
  )
)

(defmethod (basic-general-inspector :number-of-menu-pane-rows) ()
"Returns the number of menu pane rows to have in the general inspector."
  *general-inspector-number-of-menu-pane-rows*
)

(defstruct (inspector-constraint :Named)
  menu-string
  menu-doc-string
  number-of-inspectors
  constraint
)

(defvar *all-inspector-constraints*
       `(
	 ,#'(lambda (window inspectors noi)
	      (ignore noi)
	      (Make-Inspector-Constraint
		:Menu-String "One big inspector"
		:Menu-Doc-String "Just one king-sized inspect pane."
		:Number-Of-Inspectors 1
		:Constraint
		`(:one-pane
		   (,(car inspectors) menu history interactor)
		   ((interactor 4 :lines))
		   ((menu ,(send window :number-of-menu-pane-rows) :lines))
		   ((history 4 :lines))
		   ((,(car inspectors) :even)))))
	 ,#'(lambda (window inspectors noi)
	      (ignore noi)
	      (Make-Inspector-Constraint
		:Menu-String "Debug"
		:Menu-Doc-String
		"One small inspect pane and a huge interactor pane."
		:Number-Of-Inspectors 1
		:Constraint
		`(:debug
		   (,(car inspectors) menu history interactor)
		   ((interactor 35 :lines))
		   ((menu ,(send window :number-of-menu-pane-rows) :lines))
		   ((history 4 :lines))
		   ((,(car inspectors) :even)))))
	 ,#'(lambda (window inspectors noi)
	      (ignore inspectors noi)
	      (Make-Inspector-Constraint
		:Menu-String "Two panes one above the other"
		:Menu-Doc-String "Two panes of equal size, one above the other"
		:Number-Of-Inspectors 2
		:Constraint
		`(:two-horizontal-panes
		   ,(reverse `(interactor menu history inspector-0 inspector-1))
		   ((interactor 4 :lines))
		   ((menu ,(send window :number-of-menu-pane-rows) :lines))
		   ((history 4 :lines))
		   ((inspector-1 0.5))
		   ((inspector-0 :even)))))
	 ,#'(lambda (window inspectors noi)
	      (ignore inspectors noi)
	      (Make-Inspector-Constraint
		:Menu-String "Two panes one beside the other"
		:Menu-Doc-String "Two panes of equal size, one beside the other"
		:Number-Of-Inspectors 2
		:Constraint
		`(:two-vertical-panes
		   ,(reverse `(interactor menu history side-by-side))
		   ((interactor 4 :lines))
		   ((menu ,(send window :number-of-menu-pane-rows) :lines))
		   ((history 4 :lines))
		   ((side-by-side :horizontal (:even)
				  (inspector-0 inspector-1)
				  ((inspector-1 0.5))
				  ((inspector-0 :even)))))))
	 ,#'(lambda (window inspectors noi)
	      (ignore noi)
	      (Make-Inspector-Constraint
		:Menu-String "Three Panes"
	        :Menu-Doc-String "Three panes stacked one above the other."
	        :Number-Of-Inspectors 3
		:Constraint
		`(:three-panes
		   ,(reverse `(interactor menu history ,@(firstn 3 inspectors)))
		   ((interactor 5 :lines))
		   ((menu ,(send window :number-of-menu-pane-rows) :lines))
		   ((history 4 :lines))
		   ,(mapcar
		      #'(lambda (name1)
			  `(,name1 :limit (1 36 :lines)
			    ,(/ 0.3s0 (1- 3)) :lines))
		      (list (second inspectors) (third inspectors)))
		   ((,(car inspectors) :even)))))
	 ,#'(lambda (window inspectors noi)
	      (ignore inspectors noi)
	      (Make-Inspector-Constraint
		:Menu-String "Three Panes beside one another"
	        :Menu-Doc-String "Three strung out horizontally."
	        :Number-Of-Inspectors 3
		:Constraint
		`(:Three-Panes-beside-each-other
		   ,(reverse `(interactor menu history inspectors))
		   ((interactor 5 :lines))
		   ((menu ,(send window :number-of-menu-pane-rows) :lines))
		   ((history 4 :lines))
		   ((inspectors :Horizontal (:Even)
			       (inspector-0 inspector-1 inspector-2)
			       ((inspector-0 0.33))
			       ((inspector-1 0.50))
			       ((inspector-2 :Even)))))))
	 ,#'(lambda (window inspectors noi)
	      (ignore inspectors noi)
	      (Make-Inspector-Constraint
		:Menu-String "Three Panes Big on Left"
	        :Menu-Doc-String
	  "Three Panes, one big one on the left, two smaller ones on the right."
	        :Number-Of-Inspectors 3
		:Constraint
		`(:Three-Panes-1+2
		   ,(reverse `(interactor menu history inspectors))
		   ((interactor 5 :lines))
		   ((menu ,(send window :number-of-menu-pane-rows) :lines))
		   ((history 4 :lines))
		   ((inspectors :Horizontal (:Even)
			       (inspector-0 right-strip)
			       ((inspector-0 0.45))
			       ((right-strip :Vertical (:Even)
					     (inspector-2 inspector-1)
					     ((inspector-1 0.5))
					     ((inspector-2 :Even)))))))))
	 ,#'(lambda (window inspectors noi)
	      (ignore inspectors noi)
	      (Make-Inspector-Constraint
		:Menu-String "Four Panes"
	        :Menu-Doc-String "Four panes in a square."
	        :Number-Of-Inspectors 4
		:Constraint
		`(:four-panes
		   ,(reverse `(interactor menu history bottom-strip top-strip))
		   ((interactor 5 :lines))
		   ((menu ,(send window :number-of-menu-pane-rows) :lines))
		   ((history 4 :lines))
		   ((top-strip :Horizontal (0.5);(:Even)
			       (inspector-1 inspector-3)
			       ((inspector-1 0.5))
			       ((inspector-3 :Even))))
		   ((bottom-strip :Horizontal (:Even)
			       (inspector-0 inspector-2)
			       ((inspector-0 0.5))
			       ((inspector-2 :Even)))))))
	 ,#'(lambda (window inspectors noi)
	      (ignore inspectors noi)
	      (Make-Inspector-Constraint
		:Menu-String "Five panes"
	        :Menu-Doc-String
	  "Five Panes, one big one on the left, four smaller ones on the right."
	        :Number-Of-Inspectors 5
		:Constraint
		`(:Five-Panes-1+2+2
		   ,(reverse `(interactor menu history inspectors))
		   ((interactor 5 :lines))
		   ((menu ,(send window :number-of-menu-pane-rows) :lines))
		   ((history 4 :lines))
		   ((inspectors :Horizontal (:Even)
			       (inspector-0 middle-strip right-strip)
			       ((inspector-0 0.33))
			       ((middle-strip :Vertical (0.50)
					     (inspector-2 inspector-1)
					     ((inspector-2 0.5))
					     ((inspector-1 :Even))))
			       ((right-strip :Vertical (:Even)
					     (inspector-4 inspector-3)
					     ((inspector-4 0.5))
					     ((inspector-3 :Even)))))))))
	 ,#'(lambda (window inspectors noi)
	      (ignore inspectors noi)
	      (Make-Inspector-Constraint
		:Menu-String "Six Panes"
	        :Menu-Doc-String "Six panes in 3 vertical stripes."
	        :Number-Of-Inspectors 6
		:Constraint
		`(:six-panes
		   ,(reverse `(interactor menu history bottom-strip top-strip))
		   ((interactor 5 :lines))
		   ((menu ,(send window :number-of-menu-pane-rows) :lines))
		   ((history 4 :lines))
		   ((top-strip :Horizontal (0.5)
			       (inspector-1 inspector-3 inspector-5)
			       ((inspector-1 0.33))
			       ((inspector-3 0.5))
			       ((inspector-5 :Even))))
		   ((bottom-strip :Horizontal (:Even)
			       (inspector-0 inspector-2 inspector-4)
			       ((inspector-0 0.33))
			       ((inspector-2 0.5))
			       ((inspector-4 :Even)))))))
	)
"A list of functions that deal with constraints for general inspectors.
Each function takes (frame list-of-inspectors number-of-inspectors) as its
args.  It must return a defstruct instance of the type Inspector-Constraint.
"
)

(defmethod (basic-general-inspector :get-constraints) (noi)
"Gets the constraints for the general inspector, given the number of
 inspect panes.  This was abstracted from the :before :init method.
"
  (mapcar #'(lambda (item)
	      (Inspector-Constraint-Constraint
		(funcall item self inspectors noi)
	      )
	    )
	    *All-Inspector-Constraints*
  )
)


(defmethod (basic-general-inspector :choose-constraint-with-menu) ()
"Gets the constraints for the general inspector, given the number of
 inspect panes.  This was abstracted from the :before :init method.
"
  (w:menu-choose
    (mapcar #'(lambda (item)
		(let ((spec (funcall item self inspectors (length inspectors))))
		     (list (Inspector-Constraint-Menu-String spec)
			   :Value (first (Inspector-Constraint-Constraint spec))
			   :Documentation
			     (Inspector-Constraint-Menu-Doc-String spec)
		     )
		)
	      )
	      *All-Inspector-Constraints*
    )
    :label "Choose a new inspector configuration"
    :scrolling-p nil
  )
)


;;; TI code patched by JPR.
(DEFMETHOD (general-inspect-window :toggle-lock) ()   ;!
  (LET* ((iframe (Find-Inspector-Window self))
         (inspectors (SEND iframe :inspectors))
         ;(num-inspectors (LENGTH inspectors))
         (config (SEND iframe :configuration))
         (num-of-locked-panes (LOOP for el in inspectors
                                    counting (SEND el :locked-p) into x
                                    finally (RETURN x)))
         (lock-x (- (SEND self :width) 50.))
         (lock-y 3.))
    (COND (locked-p (SETQ locked-p nil)
                   (w:prepare-sheet (self)
		     (w:draw-char
		       (send (send iframe :superior)
			     :parse-font-descriptor 'fonts:icons)
		       98. lock-x lock-y w:alu-andca self)))
          (t
	   (let ((entry (find-if #'(lambda (spec)
				     (eq config
					 (first
					   (Inspector-Constraint-Constraint
					     (funcall spec iframe inspectors
						      (length inspectors)
					     )
					   )
					 )
				     )
				   )
				   *All-Inspector-Constraints*
			)
		 )
		)
	        (if (and entry
			 (< num-of-locked-panes
			    (- (Inspector-Constraint-Number-Of-Inspectors
				 (funcall entry iframe inspectors
					  (length inspectors)
				 )
			       )
			       1
			    )
			 )
		    )
		    (progn (SETQ locked-p t)
			   (w:prepare-sheet (self)
			     (w:draw-char
			       (send (send iframe :superior)
				     :parse-font-descriptor 'fonts:icons)
			       98. lock-x lock-y w:alu-xor self)))
		    (beep)))))))



(Defmethod (general-inspect-window :help-string) ()
"Returns the help string for the general inspector.  This was abstracted from
 the :before :init method.
"
  (if (send (Find-Inspector-Window self) :flavor-inspect-p)
      (if (clos-p)
          "Inspection Pane.  To inspect a flavor or class, type its name.  To inspect a method, type <flavor name> <method name>
Press HELP for a help menu, META-HELP for help on typed expressions.  R2: System Menu."
	  "Inspection Pane.  To inspect a flavor, type its name.  To inspect a method, type <flavor name> <method name>
Press HELP for a help menu, META-HELP for help on typed expressions.  R2: System Menu."
      )
      "Inspection Pane."
  )
)

(defmethod (basic-general-inspector :set-up-inspectors) (noi)
"Sets up Noi inspect panes for self.  This is done at frame init time."
  (dotimes (i noi)
    (let ((name1 (intern (format () "INSPECTOR-~D" i) "TV")))
	(push `(,name1 ,(if (zerop i)
			    'general-inspect-pane-with-typeout
			    'general-inspect-pane)
                :current-object-who-line-message
		,(function (lambda (current-object)
			     (cond
			       ((equal current-object '(nil))
				(send (tv:window-under-mouse) :help-string)
			       )
			       ((typep current-object 'flavor-operation-mixin)
				`(:mouse-l-1 "Select data to inspect"
				  :mouse-m-1 "Help on currently displayed data"
				  :mouse-m-2 "Lock/Unlock inspector pane"
				  :mouse-r-1
				  ,(or (catch-error
					 (format
					   () "Menu of operations on ~s"
					   (flavor-or-class-name
					     (send current-object :data)
					   )
					 )
					 nil
				       )
				       ""
				    )
				)
			       )
			       (t '(:mouse-l-1
				     "Choose an item to inspect"))))))
              (send self :panes))
	(push name1 inspectors)
     )
  )
  (setq inspectors (reverse inspectors))
)

(defmethod (basic-general-inspector :non-inspect-panes) ()
"Returns a list of the non inspect panes for self.  This is used at init time."
  (list `(interactor inspector-interaction-pane
	  :label nil
	  :more-p nil
	  :font-map  ,(list (first *Inspector-Font-Map*)
			    (second *Inspector-Font-Map*)
		      )
	  :who-line-message
	  "To inspect something use the <Mode> menu option to get the right input behaviour and then type what you want to inspect.
Press HELP for a help menu, META-HELP for help on typed expressions.  R2: System Menu."
	 )
	`(history general-inspector-history-window
          :line-area-mouse-doc
	  (:mouse-l-1 "Inspect the indicated data"
	   :mouse-m-1 "Remove it from the Inspector"
	  )
	  :normal-mouse-documentation
	  (:mouse-l-1 "Select data to inspect"
	   :mouse-m-2 "Lock/Unlock inspector pane"
	   :mouse-r-2 "System Menu"
	  )
	 )
	`(menu inspector-menu-pane)
  )
)

(defvar *maximum-number-of-inspect-panes* 6)

(defmethod (basic-general-inspector :before :init) (plist)
"Sets up a general inspector.  I don't like doing it like this.  I inherited it
 from the flavor inspector.  In fact we have to advisingly shadow the method
 for the flavor inspector in order to get it to work at all.
"
  (unless inspectors
	  (let ((noi (or (get plist :number-of-inspectors)
			 *Maximum-Number-Of-Inspect-Panes*
		     )
		)
	       )
	       (Send self :set-panes (send self :non-inspect-panes))
	       (send self :set-up-inspectors noi)
	       (send self :set-constraints (send self :get-constraints noi))
	  )
  )
)

(defcommand :toggle-config-cmd nil
  '(:description  "Select a new inspector pane configuration."
    :names ("Config") :keys (#\s-C)
   )
  (declare (special frame))
  (let ((new-cfg (send frame :Choose-Constraint-With-Menu)))
       (delaying-screen-management 
	 (cond (new-cfg
		(setq *general-inspector-configuration* new-cfg)
		(send frame :set-configuration new-cfg)
	       )
	 )
       )
  )
)


(defcommand (basic-general-inspector :documentation-cmd) ()
  '(:description 
    "Display some brief documentation about each of the Inspector's panes."
    :names ("Help")
    :keys (#\c-HELP)
   )
  (gi-doc-cmd)
)

(defun mode-undo-function (mode-spec)
"If a mode has an undo function to be executed when the mode is exited then this
 is the function that gets it.
"
  (sixth mode-spec)
)

(defun mode-setup-function (mode-spec)
"If a mode has an setup function to be executed when the mode is entered
 then this is the function that gets it.
"
  (fifth mode-spec)
)

(defcommand :lisp-mode-cmd nil	
  '(:description
    "Toggle between Lisp mode, Inspect mode and Flavor\/Class Inspect Mode...." 
    :names ("Mode")
    :keys (#\s-M)
   )
  (declare (special ucl::prompt frame))
  (multiple-value-bind (ignore current-index)
      (find ucl::prompt (send frame :all-prompts)
	    :key #'first :test #'string-equal
      )
       (let ((old (if current-index
		      (nth current-index (send frame :all-prompts))
		      nil
		  )
	     )
	     ;;; Finds the index for the current mode in the mode list and finds
	     ;;; the next one as appropriate.  It sets upt the typein modes
	     ;;; and all that sort of thing for the incoming mode and calls any
	     ;;; mode undo/setup functions that apply.
	     (selected (nth (if (or (not current-index)
				    (equal (+ 1 current-index)
					   (length (send frame :all-prompts))
				    )
			        )
			        0
			        (+ 1 current-index)
			    )
			    (send frame :all-prompts)
		        )
	     )
	    )
            (if (mode-undo-function old)
		(funcall (mode-undo-function old))
		nil
	    )
	    (setq ucl::prompt (first selected))
	    (send frame (second selected))
	    (send frame :set-typein-modes (third selected))
	    (send frame :set-active-command-tables
		  (mapcar #'first (symbol-value (fourth selected)))
	    )
	    (if (mode-setup-function selected)
		(funcall (mode-setup-function selected))
		nil
	    )
	    (send frame :handle-prompt)
       )
  )
)

(DEFCOMMAND MODE ()
   '(:DESCRIPTION "Toggle between Lisp mode and Inspect mode."
     :NAMES ("Mode")
     :KEYS (#\s-M))
   (DECLARE (SPECIAL UCL::TYPEIN-MODES UCL::PROMPT))
   ;;; Patch put in here by JPR.
   (:lisp-mode-cmd))


(defmethod (basic-general-inspector :set-inspect) ()
"Turns on inspect mode, switching off flavor inspecting."
  (setq flavor-inspect-p nil)
  (send self :set-inspection-data-active? nil)
)

(defmethod (basic-general-inspector :set->) ()
"Switches on lisp interaction mode."
  (setq flavor-inspect-p nil)
  (send self :set-inspection-data-active? nil)
)

(defmethod (basic-general-inspector :set-flavor/class) ()
"Switches on flavor/class/method input mode."
  (setq flavor-inspect-p t)
  (send self :set-inspection-data-active? t)
)

(build-menu 'general-inspector-menu
	    'general-inspector
  :default-item-options `(:font ,*general-inspector-menu-item-font*)	
  :item-list-order general-inspector-menu-cmds
)


(defun assure-is-a-frame (window)
"Looks up the superiors of window until it finds a frame."
  (if window
      (if (typep window 'basic-frame)
	  window
	  (assure-is-a-frame (Find-Inspector-Window window))
      )
      nil
  )
)


(defun find-or-create-inspect-window
       (flavor &optional (current-window nil) (expose-p t))
"Finds or creates an inspect window of the flavor Flavor."
  (let ((iwin (or current-window
		  (let ((old-window (find-window-of-flavor flavor)))
		       (if old-window
			   (assure-is-a-frame old-window)
			   nil
		       )
		  )
		  (make-window flavor)
	      )
	)
       )
       (if expose-p
	   (send iwin ':mouse-select)
	   nil
       )
       iwin
  )
)

(defun reinstall-commands (frame)
"Reinstalls all inspector commands just in case they have changed."
  (mapcar #'(lambda (table)
	      (build-command-table (first table) (fourth table)
		(symbol-value (second table))
		:init-options `(:name ,(third table))
	      )
	    )
	    (get-all-command-tables frame)
  )
  (mapcar #'(lambda (menu)
	      (build-menu (first menu) frame
		:default-item-options
		`(:font ,*general-inspector-menu-item-font*)
		:item-list-order (symbol-value (second menu))
	      )
	    )
	    (get-all-menu-specifiers frame)
  )
  nil
)

(defun set-element-from-history (symbol n nitems items)
"This is rather a complicated on.  It sets up something like the value of **
 from the inspector history.  Symbol is the thing to set.  N is the index to use
 into the history, for instance we would use 1 for *, 2 for **.  Nitems is the
 total number of items in the history and items is the items in the history (an
 array).  It gets the thing from the history and then transforms it into
 something suitable to put into the symbol.  For instance, if it is a show-x
 data thing, then it sees if it knows about :middle-button-result and :aux-data
 methods to find something suitable.
"
  (if (and (instancep (aref items (- nitems n)))
	   (send (aref items (- nitems n)) :send-if-handles :data)
      )
      (cond ((and (consp (aref items (- nitems n)))
		  (send (aref items (- nitems n)) :send-if-handles
			:middle-button-result
		  )
	     )
	     (set symbol
		  (car (send (aref items (- nitems n)) :middle-button-result))
	     )
	     t
	    )
	    ((and (consp (aref items (- nitems n)))
		  (send (aref items (- nitems n)) :send-if-handles :aux-data)
	     )
	     (set symbol (car (send (aref items (- nitems n)) :aux-data)))
	     t
	    )
	    ((eq 'ignore (send (aref items (- nitems n)) :data)) t)
	    (t (or (set symbol
			(send (aref items (- nitems n))
			      :send-if-handles :middle-button-result
			)
		   )
		   (set symbol
			(send (aref items (- nitems n))
			      :send-if-handles :aux-data
			)
		   )
		   (set symbol
			(let ((data (send (aref items (- nitems n)) :data)))
			     (typecase data
			       (si:flavor (si:flavor-name data))
			       (otherwise data)
			     )
			)
		   )
	       )
	    )
      )
      (set symbol (aref items (- nitems n)))
  )
)

(defmethod (basic-general-inspector :update-*) ()
  (let ((items (send history :items)))
       (let ((nitems (if items (array-active-length items) 0)))
            (if (>= nitems 1) (set-element-from-history '*   1 nitems items))
	    (if (>= nitems 2) (set-element-from-history '**  2 nitems items))
	    (if (>= nitems 3) (set-element-from-history '*** 3 nitems items))
       )
  )
)

(defun swap-system-keys
       (from to &optional (froms-window-type-was nil) (for-system nil))
"Moves the system key whose key stroke is From to the key To.  If From's window
 type was is specified then it checks to make sure that the window type of the
 system key definition on From is what you think is should be, i.e. it hasn't
 been redefined by soneone.  If it has been then you don't want to do the swap,
 since you'll be moving the wrong functionality onto To.  If For-System is
 specified, then it is the name of the system that defined the tool that's on
 the system key.  This is important, since the system definition can have a
 pointer to the system key that it thinks it should be using.  This means that
 unless this is updated you can loose after warm boots/band builds because
 these system keys are often reset in warm initializations.
"
  (let ((old (assoc from tv:*system-keys*))
	(target (assoc to tv:*system-keys*))
       )
       (if (rest target)
	   (format *error-output* "~&Target system-key ~A already defined." to)
	   (if (and froms-window-type-was
		    (not (equalp (second old) froms-window-type-was))
	       )
	       (format *error-output*
		       "~&System key ~A already seems to have been swapped."
		       from
	       )
	       (if (rest old)
		   (progn (tv:remove-system-key from)
			  (apply #'tv:add-system-key to (rest old))
			  (if for-system
			      (let ((system (sys:find-system-named for-system)))
				   (if (and (typep system 'sys:system)
					    (getf (sys:system-plist system)
						  :instance-type
					    )
				       )
				       (setf (getf (sys:system-plist system)
						   :default-system-key
					     )
					     to
				       )
				       nil
				   )
			      )
			      nil
			  )
		   )
		   nil
	       )
	       nil
	   )
       )
  )
)



(defun general-inspector-print-values? ()
  (declare (:self-flavor tv:basic-inspect-frame))
  (declare (special ucl:prompt ucl:input-mechanism history))
  (and (ucl:abnormal-command?)
       (if (string-equal ucl:prompt "> ")
           (progn
	     ;;; I see no reason why it should update panes here, even though
	     ;;; the patch to the inspector does.
	     ;;; The user can always hit refresh if he has typed something that
	     ;;; might affect something on the screen.
;            (update-panes)
             t
	   )
	   (unless (eq ucl:input-mechanism 'ucl:unknown)
	     (let ((thing (inspect-real-value `(:value ,(car \/) ,history))))
	          (inspect-flush-from-history thing history)
		  (send history :append-item thing)
		  (update-panes)
		  nil
	     )
	   )
       )
  )
)


;-------------------------------------------------------------------------------

;;; Patches....

;;RDA: fix this so exiting from INSPECT* with the END key works
;;; jwz: this was calling :bury instead of w:deselect-and-maybe-bury-window.
(DEFCOMMAND (flavor-inspector :end-cmd) ()
  '(:DESCRIPTION "Exit the Flavor Inspector."
    :NAMES ("Exit")
    :KEYS (#\END))
  (DECLARE (SPECIAL = inspect*-quit))
  (if (and (boundp 'inspect*-quit)
	   inspect*-quit
      )
      (SEND self :QUIT =) 
      (w:deselect-and-maybe-bury-window self)
  )
)

(Defmethod (show-flavor :format-concisely) (stream)
  (if (in-history-window-p stream)
      (format stream "Flavor ~s" (si:flavor-name data))
      (format stream "~s" (si:flavor-name data))
  )
)


(defun select-flavor-inspect (window)
"Keeps setting the input mode of window until it sees that it's in flavor/class
 input mode.
"
  (if (search "flavor" (send window :prompt) :test #'string-equal)
      nil
      (let ((frame window)
	    (ucl:prompt (send window :prompt))
	   )
	   (declare (special ucl::prompt frame))
	   (:lisp-mode-cmd)
	   (select-flavor-inspect window)
      )
  )
)

(defvar *show-as-much-of-a-string-on-a-line-as-you-can-p* t
"When true causes strings in the inspector to print out as far as they
can on one line.
"
)

(defvar *string-length-to-...-at* 30
"The length of a string at which to turn into ...s."
)

;;; TI code modified by JPR.
(DEFUN PRINT-ITEM-CONCISELY (ITEM STREAM &OPTIONAL (LEVEL 0) (princ-p nil))
  "Print ITEM on STREAM in a summarized fashion.
LEVEL is the depth in recursive calls to this function."
  ;;; Modded here by JPR to subsume inspection-data-print-item-concisely.
  ;;; Too many bits of the inspector call this function directly.
  (let ((*printing-mouse-sensitively* t))
       (declare (special *printing-mouse-sensitively*))
       (if (and (typep item 'inspection-data)
		;;; This test added by JPR.
		(or (and (or (typep (Find-Inspector-Window self)
				    'Basic-General-Inspector
			     )
			     (typep (Find-Inspector-Window self)
				    'eh:new-debugger-frame
			     )
			 )
		    )
		    (si:send-if-handles (Find-Inspector-Window self)
					:inspection-data-active?
		    )
		)
	   )
	   (send item :format-concisely stream)
	   (LET ((TYPE (DATA-TYPE ITEM)))
	     (IF (member TYPE '(DTP-LIST DTP-STACK-LIST))
		 (COND
		   ((EQ (CAR ITEM) 'QUOTE)
		    (SEND STREAM :TYO #\')
		    (SEND STREAM :ITEM1 (CADR ITEM)
			  :VALUE #'PRINT-ITEM-CONCISELY (1+ LEVEL)))
		   ((AND *PRINT-LEVEL* (>= LEVEL *PRINT-LEVEL*))
		    (SEND STREAM :STRING-OUT (SI:PTTBL-PRINLEVEL *READTABLE*)))
		   (T (DO ()
			  ((OR (ATOM ITEM) (NEQ (CAR ITEM) 'QUOTE)))
			(SETQ ITEM (CADR ITEM)))
		      (SEND STREAM :TYO (SI:PTTBL-OPEN-PAREN *READTABLE*))
		      (DO ((L ITEM (CDR L))
			   (FLAG NIL T)
			   (I 1 (1+ I)))
			  ((ATOM L)
			   (COND
			     (L (SEND STREAM :String-Out
				      (SI:PTTBL-CONS-DOT *READTABLE*))
				(SEND STREAM :ITEM1 L
				      :VALUE #'Print-Item-Concisely
				      (1+ LEVEL))))
			   (SEND STREAM :Tyo
				 (SI:PTTBL-CLOSE-PAREN *READTABLE*)))
			(AND FLAG
			     (SEND STREAM :TYO (SI:PTTBL-SPACE *READTABLE*)))
			(SEND STREAM :ITEM1 (CAR L)
			      :VALUE #'PRINT-ITEM-CONCISELY (1+ LEVEL))
			(COND
			  ((AND *PRINT-LENGTH* (>= I *PRINT-LENGTH*))
			   (SEND STREAM :String-Out
				 (SI:PTTBL-PRINLENGTH *READTABLE*))
			   (FUNCALL STREAM ':Tyo
				    (SI:PTTBL-CLOSE-PAREN *READTABLE*))	;?
			   (RETURN ()))))))
		 (PROGN
		   (CASE TYPE
			 (('compiled-function 'microcode-function)
			  (FUNCALL STREAM :STRING-OUT "#'"))
			 (DTP-ARRAY
			  (if (STRINGP ITEM)
			      (if *Show-As-Much-Of-A-String-On-A-Line-As-You-Can-P*
				  (let ((index (%string-search-char
						 #\NEWLINE item 0
						 (length item))))
				    (if index
					(setq item (string-append
						     (nsubstring item 0 index)
						     "..."))))
				  (if (or (AND (NOT (= LEVEL 0))
					       (> (ARRAY-ACTIVE-LENGTH ITEM)
						  *string-length-to-...-at*))
					  (%string-search-char #\NEWLINE
					       item 0 (length item)))
				      (setq item "..."))))))
		   (funcall (if princ-p 'princ 'prin1) ;;; JPR.
		     (CASE TYPE
			   (DTP-SYMBOL
			     (IF (POSITION #\NEWLINE
				       (THE STRING (STRING (SYMBOL-NAME ITEM)))
				       :TEST #'CHAR-EQUAL)
				 (INTERN (STRING-SUBST-CHAR #\SPACE #\NEWLINE
							    (SYMBOL-NAME ITEM))
					 (SYMBOL-PACKAGE ITEM))
				 ITEM))
			   (('COMPILED-FUNCTION 'MICROCODE-FUNCTION)
			    (si:get-debug-info-field
			      (si:get-debug-info-struct item) :name))
			   (OTHERWISE ITEM))
		     STREAM)))))))


;;; TI Code, modified by JPR.
(defun inspection-data-print-item-concisely (thing stream &optional (level 0))
      ;fi
  ;;;Send changed to Send-if-handles.  Frames, which are not inspect frames may
  ;;;still have this sort of window (debugger frames).  Such frames may not
  ;;;have the extra IVs to cope with this message.
  ;;; This binding added by JPR.
  (print-item-concisely thing stream level))


;;; Changed by JPR to allow for Mouse-L-2 behaviour.
(DEFMETHOD (BASIC-INSPECT :WHO-LINE-DOCUMENTATION-STRING) ()  ;fi
  (COND 
    ;;If mouse is over an item containing an instance of INSPECTION-DATA,
    ;;let the instance provide the who-line-doc.  This is used in special-purpose inspectors such as the flavor inspector.
    ((AND SENSITIVE-INSPECT-ITEM 
	  (LET ((item (get-mouse-sensitive-item)))
	    (WHEN (AND (LISTP item) (TYPEP (THIRD item) 'inspection-data))
	      (SEND (THIRD item) :who-line-doc SELF)))))
    (SENSITIVE-INSPECT-ITEM
     (IF DISPLAYING-LIST
	 (COND (MODIFY-MODE
		'(:mouse-l-1 "Modify list item" :mouse-r-1 "Abort"))
	       ((AND (NOT SETTING-MODE) (KEY-STATE :HYPER))
		'(:mouse-l-1 "Modify list item"))
	       (SETTING-MODE
		'(:mouse-l-1 "Set with this value" :mouse-r-1 "Abort"))
	       (T
		(send self :get-normal-mouse-documentation)))
	 (COND (MODIFY-MODE
		'(:mouse-l-1 "Modify slot" :mouse-r-1 "Abort"))
	       ((AND (NOT SETTING-MODE) (KEY-STATE :HYPER)) 
		'(:mouse-l-1 "Modify slot"))
	       (SETTING-MODE
		'(:mouse-l-1 "Set with this value" :mouse-r-1 "Abort"))
	       (T
		(send self :get-normal-mouse-documentation)))))
    (DISPLAYING-LIST
     (COND ((OR MODIFY-MODE (AND (NOT SETTING-MODE) (KEY-STATE :HYPER)))
	    '(:mouse-l-1 "Choose a list item to modify" :mouse-r-2 "System Menu"))
	   (SETTING-MODE
	    '(:mouse-l-1 "Choose a list item to set with" :mouse-r-2 "System Menu"))
	   (T
	    '(:mouse-l-1 "Choose a list item to inspect" :mouse-r-1 "System Menu"))))
    (T
     (COND ((OR MODIFY-MODE (AND (NOT SETTING-MODE) (KEY-STATE :HYPER))) 
	    '(:mouse-l-1 "Choose a slot to modify" :mouse-r-2 "System Menu"))
	   (SETTING-MODE
	    '(:mouse-l-1 "Choose a value to set with" :mouse-r-2 "System Menu"))
	   (T
	    (SEND SELF :FUNCALL-INSIDE-YOURSELF current-object-who-line-message current-object))))))


(defmethod (show-clos-class :who-line-doc)
	   (inspection-pane? &optional no-sensitive-item?)
  (cond
    (no-sensitive-item?
     `(:mouse-l-1 "Choose an item to inspect"
       :mouse-m-2 "Lock/Unlock inspector pane"
       :mouse-r-1 ,(format nil "Menu of operations on ~A"
			   (Class-Pretty-Name data))))
    (inspection-pane?
      '(:mouse-l-1 "Inspect this CLOS class"
		   ;;; JPR.
        :mouse-m-1 "Set *"
        :mouse-m-2 "Lock/Unlock inspector pane"
	:mouse-r-1 "Menu of other operations"))
    (t
     '(:mouse-l-1 "Inspect this CLOS Class"
		   ;;; JPR.
       :mouse-m-1 "Set *"
       :mouse-m-2 "Lock/Unlock inspector pane"
       :mouse-r-1 "Menu of other operations"))))

(defmethod (flavor-operation-mixin :who-line-doc)
	   (inspection-pane? &optional no-sensitive-item?)
  (cond
    (no-sensitive-item?
     `(:mouse-l-1 "Choose an item to inspect"
       :mouse-m-2 "Lock/Unlock inspector pane"
       :mouse-r-1 ,(format nil "Menu of operations on ~s"
			   (si:flavor-name data))))
    (inspection-pane?
      '(:mouse-l-1 "Inspect this flavor information"
		   ;;; JPR.
	:mouse-m-1 "Set *"
        :mouse-m-2 "Lock/Unlock inspector pane"
	:mouse-r-1 "Menu of other operations"))
    (t
     '(:mouse-l-1 "Inspect this flavor information"
		   ;;; JPR.
       :mouse-m-1 "Set *"
       :mouse-m-2 "Lock/Unlock inspector pane"
       :mouse-r-1 "Menu of other operations"))))


(defmethod (show-clos-method :who-line-doc) (ignore &optional ignore)
  '(:mouse-l-1 "Inspect method details"
		   ;;; JPR.
    :mouse-m-1 "Set *"
    :mouse-r-1 "Menu of method operations"))


(defmethod (show-clos-generic-function :who-line-doc) (ignore &optional ignore)
  '(:mouse-l-1 "Inspect generic function details"
		   ;;; JPR.
    :mouse-m-1 "Set *"
    :mouse-r-1 "Menu of generic function operations"))

;-------------------------------------------------------------------------------
;-------------------------------------------------------------------------------

(defun maybe-allocate-locative (something)
"Returns an item for the inspector to represent Something.  If Something is a
 cons then it is of the form (<value> <locative-to-value>).  If the locative
 item is :No-Locative then a :List-structure (non-modifiable) item is generated,
 otherwise a Show-Data-With-Locative item is generated.  If something is not a
 cons then it is itemised non mouse sensitively in font 2.
"
  (if (consp something)
      (if (equal :No-Locative (second something))
	  (list :item1 :list-structure (first something))
	  (list :item1 'show-data-with-locative
		(allocate-data 'show-data-with-locative (first something)
			       (second something)
		)
	  )
      )
      (fontify-string (format nil "~a" something) 2)
  )
)

(defun take-first-off-lists (lists)
"Is passed a list of lists of the form ((a b c) (d e f) (g h i)).  It returns
 the list ((a d g) (b e h) (c f i)).
"
  (if (first lists)
      (cons (mapcar #'first lists)
	    (take-first-off-lists (mapcar #'rest lists))
      )
      nil
  )
)

(defun join-value-list (value-list)
"Is passed a value list which will be itemised.  It returns a list like the
 value-list only any element which is of the form (:wide (...)) is appended
 into the list.  Thus (a b c (:wide (d e f)) g h) turns into (a b c d e f g h).
"
  (if value-list
      (if (and (consp (first value-list))
	       (equal :wide (first (first value-list)))
	  )
	  (append (second (first value-list))
		  (join-value-list (rest value-list))
	  )
	  (cons (first value-list) (join-value-list (rest value-list)))
      )
      nil
  )
)


(defmethod (basic-inspect :object-generic-locatised-things)
 (title obj top-value top-value-title first-element-specialp &rest values
  &aux (maxl -1) result
 )
" A generalised method which generates items for the inspector.
 Title - is a string used to describe Obj at the top of the display.
 Obj - is the thing which is actually being inspected.
 Top-Value - is either something to print out under the title, or :No
 Top-Value-Title - is a string used to describe Top-Value, unless Top-Values=:no
 First-Element-Specialp - Tells the method that the first list in Values is to
			  be treated specially (as row headers).
 Values - is a list of column specs for the columns of things to be printed.
          Each column spec is a list of the elements to be inspected.  Each
          element in the spec should be of the form:
		(<value> <locative-to-value>) - Modifiable mouse sensitive item.
		(<value> :No-Locative) - NonModifiable mouse sensitive item.
		<non-cons> - Non mouse sensitive value (error message)
"
  (push `(,title (:item1 named-structure-p ,obj)) result)
  (push '("") result)
  (if (not (equal :no top-value))
      (progn (push `(,top-value-title (:item1 named-structure-p ,top-value))
		   result)
	     (push '("") result)))
  (if (rest values)
      (if first-element-specialp
	  (let ((lists (take-first-off-lists (rest values))))
	       (do ((l (first values) (cdr l)))
		   ((null l) nil)
		 (setq maxl (max (flatsize l) maxl)))
	       (loop for a-name in (first values)
		     for value-list in lists
		     do
		     (push `((,@(maybe-allocate-locative a-name))
			     (:colon ,(+ 2 (min 30 maxl)))
			     ,@(apply #'append
				(loop for value
				      in (join-value-list value-list) collect
				  `((,@(maybe-allocate-locative value)) " ")
				)
			       )
			    )
			    result
		     )
	       )
	  )
	  (let ((lists (take-first-off-lists values)))
	       (loop for value-list in lists
		     do
		     (push `(,@(apply #'append
				(loop for value
				      in (join-value-list value-list) collect
				  `((,@(maybe-allocate-locative value)) " ")
				)
			       )
			    )
			    result
		     )
	       )
	  )
      )
      (loop for a-name in (first values) do
	    (push `((,@(maybe-allocate-locative a-name))) result)
      )
  )
  (values (nreverse result) obj 'inspect-printer)
)

(defmethod (basic-inspect :object-paired-thing)
           (title obj top-value names values
	    &optional (top-value-string "First value: ")
	   )
" A generalised method which generates items for the inspector for pairs of
 things.
 Title - is a string used to describe Obj at the top of the display.
 Obj - is the thing which is actually being inspected.
 Top-Value - is either something to print out under the title, or :No
 Top-Value-Sting - is a string used to describe Top-Value, unless Top-Values=:no
 First-Element-Specialp - Tells the method that the first list in Values is to
			  be treated specially (as row headers).
 Names and Values - Are column specs for the columns of things to be printed.
          Each column spec is a list of the elements to be inspected.  Each
          element in the spec should be of the form:
		(<value> <locative-to-value>) - Modifiable mouse sensitive item.
		(<value> :No-Locative) - NonModifiable mouse sensitive item.
		<non-cons> - Non mouse sensitive value (error message)
"
  (send self :object-generic-locatised-things
	title obj top-value top-value-string t names values
  )
)


(defun maybe-show-list-named-structure (list)
"Is passed a list.  If it thinks that it can inspect it as a named structure
 then it allocates a Show-List-Named-Structure item.  If it can't inspect it
 then it returns the list, barfing.
"
  (if (get (first list) 'si:defstruct-description)
      (allocate-data 'show-list-named-structure list)
      (progn (beep)
	     (format *query-io* "~&~S is not the name of a defstruct type."
		     (first list)
	     )
	     list
      )
  )
)

(defun get-type (prompt)
"Reads something from the user, prompting with Prompt."
  (declare (special user history = inspectors frame))
  (let-if (and (not (boundp 'user)) (boundp 'eh:*window-debugger*))
	  ((user (send eh:*window-debugger* :lisp-window))
	   (history (send eh:*window-debugger* :inspect-history-window))
	   (= nil)
	   (inspectors (list (send eh:*window-debugger* :inspect-window)))
	   (frame eh:*window-debugger*)
	  )
    (letf ((#'inspect-get-value-from-user #'my-inspect-get-value-from-user))
	  (format user prompt)
	  (multiple-value-bind (value punt-p)
	      (inspect-get-value-from-user user history inspectors)
	    (if punt-p
		(throw :Abort-tag nil)
		value
	    )
	  )
    )
  )
)


(defun can-be-coerced-to-type (something)
"If something can be coerced into a defstruct type name then it is, otherwise
 it returns nil.
"
  (cond ((and (symbolp something) (get something 'si:defstruct-description))
	 something
	)
	((and (symbolp something) (boundp something)
	      (not (equal something (symbol-value something)))
	 )
	 (can-be-coerced-to-type (symbol-value something))
	)
	((and (consp something) (catch-error (eval something) nil))
	 (can-be-coerced-to-type (eval something))
	)
	(t nil)
  )
)

      
(defun read-type-name ()
"Prompts the user for the name of a defstruct type and returns it."
  (let ((name (get-type "~&Name of structure type: ")))
       (if (can-be-coerced-to-type name)
	   (can-be-coerced-to-type name)
	   (progn (beep)
		  (format *query-io* "~&~S is not the name of a defstruct type."
			  name
		  )
		  (read-type-name)
	   )
       )
  )
)

(defun maybe-show-list-unnamed-structure (list)
"Is passed a list and allocates a Show-List-Unnamed-Structure item to inspect
 it as a structure which is asked of the user.
"
  (let ((type-name (read-type-name)))
       (allocate-data 'show-list-unnamed-structure list type-name)
  )
)

(defun maybe-show-list-offset-unnamed-structure (list)
"Is passed a list and allocates a Show-List-Offset-Unnamed-Structure item to
 inspect it as a structure which is asked of the user.
"
  (let ((type-name (read-type-name)))
       (allocate-data 'show-list-offset-unnamed-structure list type-name)
  )
)

;-------------------------------------------------------------------------------

(defflavor list-inspection-mixin
	   () (data-as-aux-data-mixin inspection-data)
  :Gettable-Instance-Variables
  :Abstract-Flavor
  (:Documentation :Mixin
"
 A mixin which allows the system to inspect lists from different perspectives.
 Flavors are built on this one which hold the list in their Data slots and
 know how to print them specially.
"
  )
)

(defmethod (list-inspection-mixin :handle-mouse-click) (blip flavor-inspector)
"A mouse click handler for list perspectives."
  (selector (fourth blip) =
    (#\mouse-l-1 (send flavor-inspector :inspect-info-left-click))
    (#\mouse-m-1 (send flavor-inspector :inspect-info-middle-click))
    (t (beep))
  )
)


(defwrapper (list-inspection-mixin :handle-mouse-click)
	    ((blip flavor-inspector) &body body)
"A mouse click handler for list perspectives that knows about l2 behaviour."
  `(let ((object (find-inspection-object (send flavor-inspector :kbd-input))))
        (if (and (= (fourth blip) #\mouse-l-2) object)
	    (send flavor-inspector :inspect-info-left-2-click
		  object
	    )
            . ,body
	)
   )
)

(defmethod (list-inspection-mixin :print-self) (stream depth slashify)
"A print method for all things built on list-inspection-mixin.  This means
 that when one of these is inspected the Data is printed at the top of the
 screen, not #<list-inspection-mixin...>.
"
  (ignore depth)
  (format stream "~" (list data t data) slashify)
)


;-------------------------------------------------------------------------------

(defflavor data-as-aux-data-mixin
	   () (inspection-data)
  :Abstract-Flavor
  (:Documentation :Mixin
"
 A mixin which provides an :aux-data method, which simply returns Data.  This is
 used so that Update-Panes sets * and such to the right thing.
"
  )
)

(defmethod (data-as-aux-data-mixin :aux-data) ()
"An :aux-data method, which simply returns Data.  This is
 used so that Update-Panes sets * and such to the right thing.
"
  data
)

;-------------------------------------------------------------------------------

(defflavor show-data-with-locative
	   (aux-data)
	   (list-inspection-mixin)
  :Initable-Instance-Variables
  (:Documentation
"
 A Show-X flavor which shows Data in a simple way but also has a locative to
 Data stored in Aux-Data.  This allows the user to Modify the data that this
 represents.
"
  )
)


(defmethod (show-data-with-locative :format-concisely) (stream)
"A simple print method for show data with locatives.  It just prints the Data."
  (format stream "~" (list data t data))
)

(defmethod (show-data-with-locative :locative) ()
"The locative is stored in the Aux-Data slot.  Perversely the :Aux-Data method
 actually returns Data.
"
  aux-data
)

(defmethod (show-data-with-locative :generate-item-specialized) (window)
"Generates an item for the Data."
  (inspect-object-display-list data window)
)

(defmethod (show-data-with-locative :match?) (thing locative)
"Will match to any existing allocated data eq to Thing."
  (ignore locative)
  (eq data thing)
)

(defun (:property show-data-with-locative set-function)
       (item new-value object)
"Finds a show-data-with-locative [(third (second item))] and sets its data
 slot, smashing the locative so that they both have the value New-Value.
"
  (ignore object)
  (let ((show-thing (third (second item))))
       (send show-thing :set-data new-value)
       (setf (contents (send show-thing :locative)) new-value)
       (send show-thing :data)
  )
)


(defmethod (basic-inspect :object-show-data-with-locative) (object)
"A method which is used by the top level inspect-setup-object-display-list to
 inspect this.  In fact it just inspects the data slot.
"
  (inspect-object-display-list (send object :data) self)
)

;-------------------------------------------------------------------------------

(defun split-up-list-into-pairs (list)
"Given a plist (:a 42 :b 20 :c) is returns the values:
 i)  ((:a <locative to :a>) (:b <locative to :b>) (:c <locative to :c>))
 ii) ((42 <locative to 42>) (20 <locative to 20>) 
     \"No value matching this key\")
"
  (if (consp list)
      (if (rest list)
	  (if (and (consp (rest list))
		   (or (consp (rest (rest list)))
		       (not (rest (rest list)))
		   )
	      )
	      (multiple-value-bind (names values)
		  (split-up-list-into-pairs (rest (rest list)))
		(values (cons (list (first  list) (locf (first  list))) names)
			(cons (list (second list) (locf (second list))) values)
		)
	      )
	      (if (consp (rest list))
		  (values (list (list (first  list) (locf (first  list)))
				(list (rest (rest list))
				      (locf (rest (rest list)))
				)
			  )
			  (list (list (second list) (locf (second list)))
			      " - Mal-formed PList (dotted before this element)"
			  )
		  )
		  (values (list (list (first list) (locf (first list)))
				(list (rest  list) (locf (rest  list)))
			  )
			  (list (list (rest list) (locf (rest list)))
			      " - Mal-formed PList (dotted before this element)"
			  )
		  )
	      )
	  )
	  (values (list (list (first  list) (locf (first  list))))
		  (list "No value matching this key")
	  )
      )
      (if list
	  (values (list (list list :no-locative))
		  (list "Mal-formed PList (dotted at the end)")
	  )
	  (values nil nil)
      )
  )
)

(defflavor show-plist
	   () (list-inspection-mixin)
  (:Documentation "A flavor of show-x, which displays a list as a Plist.")
)

(defmethod (show-plist :format-concisely) (stream)
"Prints in the history panel showing that it is a Plist representation of a
 list.
"
  (format stream "Plist ~" (list data t data))
)

(defmethod (show-plist :generate-item-specialized) (window)
"Generates items for Data so that it is displayed as a Plist."
  (multiple-value-bind (names values) (split-up-list-into-pairs data)
    (send window :object-paired-thing
	  "PList representation of: " data :no names values
    )
  )
)

;-------------------------------------------------------------------------------

(defflavor show-offset-plist
	   () (list-inspection-mixin)
  (:Documentation 
"
 A flavor of show-x, which displays a list as a Plist.  This differs from
 Show-Plist in that the display is offset by one.  This gives the user an
 alternative for aligning the keys with the values in the plist.
"
  )
)

(defmethod (show-offset-plist :format-concisely) (stream)
"Prints in the history panel showing that it is an offset Plist representation
 of a list.
"
  (format stream "Offset Plist ~" (list data t data))
)
	  
(defmethod (show-offset-plist :generate-item-specialized) (window)
"Generates items for Data so that it is displayed as an offset Plist."
  (multiple-value-bind (names values) (split-up-list-into-pairs (rest data))
    (send window :object-paired-thing
	  "Offset PList representation of: " data (first data) names values
    )
  )
)

;-------------------------------------------------------------------------------

(defun locatise-elements (list)
"Given the list (a b c)  it returns the list:
 ((a <locative to a>) (b <locative to b>) (c <locative to c>).
"
  (if (consp list)
      (cons (list (first list) (locf (first list)))
	    (if (or (consp (rest list)) (not (rest list)))
		(locatise-elements (rest list))
		(list "." (list (rest list) (locf (rest list))))
	    )
      )
      (if list ;;; dotted list
	  (list (list list :no-locative))
	  nil
      )
  )
)

(defun split-alist-up (list)
"Given an Alist of the form ((a 42) b (c 20 30)) [n.b. b is a bogus element] it
 returns the values:
 i)  ((a <locative to a>) (b <locative to b>) (c <locative to c>))
 ii) ((:wide (42 <locative to 42>)) \"This element was not a cons\"
      (:wide (20 <locative to 20>) (30 <locative to 30>)))
"
  (if (consp list)
      (multiple-value-bind (names values)
	  (if (or (consp (rest list)) (not (rest list)))
	      (split-alist-up (rest list))
	      (values (list (list (rest list) (locf (rest list))))
		      (list "Mal-formed AList (dotted at the end)")
	      )
	  )
	(if (consp (first list))
	    (if (consp (rest (first list)))
	        (values (cons (list (first (first list))
				    (locf (first (first list)))
			      )
			      names
			)
			(cons (list :wide
				    (locatise-elements (rest (first list)))
			      )
			      values
			)
		)
		(values (cons (list (first (first list))
				    (locf (first (first list)))
			      )
			      names
			)
			(cons (list :wide
				    (list "."
					  (list (rest (first list))
						(locf (rest (first list)))
					  )
				    )
			      )
			      values
			)
		)
	    )
	    (values (cons (list (first list) (locf (first list))) names)
		    (cons "This element was not a cons" values)
	    )
	)
      )
      (if list
	  (values (list (list list :no-locative))
		  (list "Mal-formed AList (dotted at the end)")
	  )
	  (values nil nil)
      )
  )
)


(defflavor show-alist
	   () (list-inspection-mixin)
  (:Documentation "A flavor of show-x, which displays a list as an Alist.")
)


(defmethod (show-alist :format-concisely) (stream)
"Prints in the history panel showing that it is an AList representation of a
 list.
"
  (format stream "AList ~" (list data t data))
)


(defmethod (show-alist :generate-item-specialized) (window)
"Generates items for Data so that it is displayed as an AList."
  (multiple-value-bind (names values)
      (split-alist-up data)
    (send window :object-paired-thing
	  "AList representation of: " data :no names values
    )
  )
)

;-------------------------------------------------------------------------------

(defflavor show-one-on-a-line
	   () (list-inspection-mixin)
  (:Documentation 
"
 A flavor of show-x, which displays a list in such a way that each element is
 printed on a different line.
"
  )
)

(defmethod (show-one-on-a-line :format-concisely) (stream)
"Prints in the history panel showing that it is a one element on each line
 representation of a list.
"
  (format stream "One-on-a-line ~" (list data t data))
)

(defun split-up-list-into-individual-elements (list)
"Given the list (:a :b :c), it returns the list:
 ((:a <locative to :a>) (:b <locative to :b>) (:c <locative to :c>))
"
  (if (consp list)
      (let ((rest (if (or (consp (rest list)) (not (rest list)))
		      (split-up-list-into-individual-elements (rest list))
		      (list "." (list (rest list) (locf (rest list))))
		  )
	    )
	   )
	   (cons (list (first list) (locf (first list))) rest)
      )
      (if list
	  (list "." (list list :no-locative))
	  nil
      )
  )
)


(defmethod (show-one-on-a-line :generate-item-specialized) (window)
"Generates items for Data so that it displays each element on a fresh line."
  (send window :object-generic-locatised-things
	"One-on-a-line representation of: " data :no nil nil
	(split-up-list-into-individual-elements data)
  )
)

;-------------------------------------------------------------------------------


(defun split-up-into-defstruct-slots (slots list type)
"Is passed a list of slot names and a list of elements, which are to be
 interpressed as elements of an instance of the structure with the slots. 
 The special cases of there being too many or too few slot names are catered
 for.  Given the lists: (:a :b :c) (:d :e :f :g) it returns the values:
 i)  ((:a :no-locative) (:b :no-locative)
      (:c :no-locative) \"Slot names exhausted\")
 ii) ((:d <locative to :d>) (:e <locative to :e>)
      (:f <locative to :f>) (:g <locative to :g>))
"
  (if (consp list)
      (if slots
	  (multiple-value-bind (names values)
	      (if (or (consp (rest list)) (not (rest list)))
		  (split-up-into-defstruct-slots (rest slots) (rest list) type)
		  (if (and (member type '(:List*))
			   (not (rest (rest slots)))
		      )
		      (values (list (list (second slots) :No-Locative))
			      (list (list (rest list)  (locf (rest list))))
		      )
		      (Values (list (list (first slots) :No-Locative)
				    (list (rest list) (locf (rest list)))
			      )
			      (list
				"******"
			     "Mal-formed defstruct (dotted before this element)"
			      )
		      )
		  )
	      )
	    (values (cons (list (first slots) :No-Locative)        names)
		    (cons (list (first list)  (locf (first list))) values)
	    )
	  )
	  (multiple-value-bind (names values)
	      (if (or (consp (rest list)) (not (rest list)))
		  (split-up-into-defstruct-slots slots (rest list) type)
		  (values (list "Slot names exhausted"
				(list (rest list) (locf (rest list)))
			  )
			  (list "******" "Mal-formed defstruct (dotted at end)")
	          )
	      )
	    (values (cons "Slot names exhausted"                    names)
		    (cons (list (first list)  (locf (first list))) values)
	    )
	  )
      )
      (if list
	  (if slots
	      (values (list (list (first slots) :No-Locative)
			    (list list :no-locative)
		      )
		      (list "******"
			    "Mal-formed defstruct (dotted before this element)"
		      )
	      )
	      (values (list "Slot names exhausted" (list list :no-locative))
		      (list "******" "Mal-formed defstruct (dotted at end)")
	      )
	  )
	  (if slots
	      (multiple-value-bind (names values)
		  (split-up-into-defstruct-slots (rest slots) (rest list) type)
		(values (cons (list (first slots) :No-Locative) names)
			(cons "List elements exhausted"          values)
		)
	      )
	      (values nil nil)
	  )
      )
  )
)

(defflavor show-list-named-structure
	   () (list-inspection-mixin)
  (:Documentation
    "A flavor of show-x, which displays a list as a named structure."
  )
)

(defmethod (show-list-named-structure :format-concisely) (stream)
"Prints in the history panel showing that it is a Named structure
 representation of a list, whose type is (first Data).
"
  (format stream "List as ~ instance: ~"
	  (list (first data) t (first data)) (list data t data)
  )
)

(defmethod (show-list-named-structure :generate-item-specialized) (window)
"Generates items for Data so that it is displayed as a Named Structure."
  (let ((slot-names
	  (mapcar #'first (si:defstruct-description-slot-alist
			    (get (first data) 'si:defstruct-description)
			  )
	  )
	)
	(type (si:defstruct-description-type
		(get (first data) 'si:defstruct-description)
	      )
	)
       )
       (multiple-value-bind (names values)
	   (split-up-into-defstruct-slots (rest slot-names) (rest data) type)
	 (send window :object-paired-thing
	       "Named Structure representation of: " data (first data)
	       names values "Type: "
	 )
       )
  )
)



;-------------------------------------------------------------------------------


(defflavor show-list-unnamed-structure
	   (aux-data)
	   (list-inspection-mixin)
  (:Documentation
"A flavor of show-x, which displays a list as a structure, whose name is
 specified by the user.
"
  )
  :Initable-Instance-Variables
)

(defmethod (show-list-unnamed-structure :format-concisely) (stream)
"Prints in the history panel showing that it is a Named structure
 representation of a list, whose type is Aux-Data.
"
  (format stream "List as ~ instance: ~"
	  (list aux-data t aux-data) (list data t data)
  )
)

(defmethod (show-list-unnamed-structure :match?) (thing type)
"Will match to any existing allocated data eq to Thing and whose Aux-Data
 matches too.
"
  (and (eq data thing) (eq type aux-data))
)

(defmethod (show-list-unnamed-structure :generate-item-specialized) (window)
"Generates items for Data so that it is displayed as a Structure, whose type is
 stored in Aux-Data.
"
  (let ((slot-names
	  (mapcar #'first (si:defstruct-description-slot-alist
			    (get aux-data 'si:defstruct-description)
			  )
	  )
	)
	(type (si:defstruct-description-type
		(get aux-data 'si:defstruct-description)
	      )
	)
       )
       (multiple-value-bind (names values)
	   (if (equal (second (get aux-data 'si:defstruct-description))
		      'sys:named-list
	       )
	       (split-up-into-defstruct-slots (rest slot-names) data type)
	       (split-up-into-defstruct-slots slot-names data type)
	   )
	 (send window :object-paired-thing
	       "Named Structure representation of: " data aux-data
	       names values "Type: "
	 )
       )
  )
)

;-------------------------------------------------------------------------------


(defflavor show-list-offset-unnamed-structure
	   (aux-data)
	   (list-inspection-mixin)
  (:Documentation
"A flavor of show-x, which displays a list as a structure, whose name is
 specified by the user, which is displayed offset to allow for a type name.
"
  )
  :Initable-Instance-Variables
)

(defmethod (show-list-offset-unnamed-structure :format-concisely) (stream)
"Prints in the history panel showing that it is a Named structure
 representation of a list, whose type is Aux-Data.
"
  (format stream "List as ~ instance (offset): ~"
	  (list aux-data t aux-data) (list data t data)
  )
)

(defmethod (show-list-offset-unnamed-structure :match?) (thing type)
"Will match to any existing allocated data eq to Thing and whose Aux-Data
 matches too.
"
  (and (eq data thing) (eq type aux-data))
)

(defmethod (show-list-offset-unnamed-structure :generate-item-specialized)
	   (window)
"Generates items for Data so that it is displayed as a Structure, whose type is
 stored in Aux-Data, which is offset to allow for an ignored type in the first
 of the list.
"
  (let ((slot-names
	  (mapcar #'first (si:defstruct-description-slot-alist
			    (get aux-data 'si:defstruct-description)
			  )
	  )
	)
	(type (si:defstruct-description-type
		(get aux-data 'si:defstruct-description)
	      )
	)
       )
       (multiple-value-bind (names values)
	   (if (equal (second (get aux-data 'si:defstruct-description))
		      'sys:named-list
	       )
	       (Split-Up-Into-Defstruct-Slots
		 (rest slot-names) (rest data) type
	       )
	       (split-up-into-defstruct-slots slot-names (rest data) type)
	   )
	 (send window :object-paired-thing
	       "Named Structure representation of: " data aux-data
	       names values "Type: "
	 )
       )
  )
)

;-------------------------------------------------------------------------------


(defvar *update-panes-base-ok* t
"When true this makes sure that when the inspector's panes are updated the
 item cache is not flushed.
"
)

(let ((compiler:compile-encapsulations-flag t))
     (advise tv:update-panes :around :base-ok nil
       (if *update-panes-base-ok*
	   (setq arglist (list t))
	   nil
       )
       :do-it
     )
)

(defvar *flush-cache-if-left-button-on-something* nil
"When true this makes sure that when you left click on something in the
 inspector its cache entry will be flushed so that it will be completely
 recomputed.  Keeping this as Nil saved a lot of time.  If it is set to :really
 then all cache entries are flushed.
"
)

;;; TI Code
(DEFMETHOD (BASIC-INSPECT-FRAME :inspect-info-left-click) ()  ;fi
  (LET ((thing (inspect-real-value ucl:kbd-input)))
    ;; First flush item we will be inspecting
    (inspect-flush-from-history thing history)
    (SEND history :APPEND-ITEM thing)
    ;;; Modded here by JPR.
    (if *flush-cache-if-left-button-on-something*
	(if (equal *flush-cache-if-left-button-on-something* :really)
	    (send history :set-cache nil)
	    (send history :flush-object-from-cache thing)))
    (update-panes)))


(defun item-key (x)
"Makes a key for sorting that we can sort with string-lessp. out of an
 itemised hash table entry.
"
  (format nil "~S" (third (first x)))
)


(defmethod (basic-inspect :object-hash-table) (obj)
"Inspects a hash table in the normal, useful way."
  
  (values (cons `("Hash Table Elements:")
		 (make-window-items-for-hash-table obj 'identity nil)
	  )
	  obj 'inspect-printer
  )
)

;-------------------------------------------------------------------------------

(defflavor show-hash-table
	   ()
	   (generic-middle-button-mixin inspection-data)
  (:documentation
"Shows a hash tables hash array elements.
"
  )
)

(defmethod (show-hash-table :Middle-Button-Result) ()
"Just returns the data slot."
  data
)

(defmethod (show-hash-table :format-concisely) (stream)
"Just prints it out."
  (format stream "~S elements" data)
)

(defmethod (show-hash-table :generate-item) (&aux result)
"Makes the inspector items for a hash-table."
  (push '("") result)
  (push '("Hash Array Elements") result)
  (push '("") result)
  (loop for element in (Make-Window-Items-For-Hash-Table data 'identity nil)
	do (push element result)
  )
  (push '("") result)
  (values (nreverse result)
	 `(:font fonts:hl12bi :string ,(format nil "~s" data))
  )
)

(defmethod (show-hash-table :help) ()
"Gives help when you middle button on a defstruct."
  (format nil "
The inspection pane you just selected is currently displaying the hash table
~S as its elements.  Mousing L2 on it should show it
to you in some other way.
"
	  data))

;-------------------------------------------------------------------------------

(defflavor show-hash-table-sorted
	   ()
	   (generic-middle-button-mixin inspection-data)
  (:documentation
"Shows a hash tables hash array elements.
"
  )
)

(defmethod (show-hash-table-sorted :Middle-Button-Result) ()
"Just returns the data slot."
  data
)

(defmethod (show-hash-table-sorted :format-concisely) (stream)
"Just prints it out."
  (format stream "~S with sorted elements" data)
)

(defmethod (show-hash-table-sorted :generate-item) (&aux result)
"Makes the inspector items for a hash-table."
  (push '("") result)
  (push '("Hash Array Elements") result)
  (push '("") result)
  (loop for element in (Make-Window-Items-For-Hash-Table data 'identity t)
	do (push element result)
  )
  (push '("") result)
  (values (nreverse result)
	 `(:font fonts:hl12bi :string ,(format nil "~s" data))
  )
)

(defmethod (show-hash-table-sorted :help) ()
"Gives help when you middle button on a defstruct."
  (format nil "
The inspection pane you just selected is currently displaying the hash table
~S as its elements sorted.  Mousing L2 on it should show it
to you in some other way.
"
	  data))

;-------------------------------------------------------------------------------

(defflavor show-generic-defstruct
	   ()
	   (generic-middle-button-mixin inspection-data)
  (:documentation
"A fall-back option perspective which allows defstruct instances that
 have their own magic inspector behaviour to be inspected also as the
 defstructs that implement them.
"
  )
)

(defmethod (show-generic-defstruct :format-concisely) (stream)
"Just prints it out but notes that it is a defstruct perspective."
  (format stream "~ as a Defstruct Instance" (list data t data))
)

(defmethod (show-generic-defstruct :generate-item-specialized)
           (window &aux (maxl -1) alist defstruct-items result nss d)
"Makes the inspector items for a defstruct.  The body of this was stolen
 from :object-named-structure (but with the hash-table stuff removed.
"
  (setq nss (named-structure-p data))
  (push `("Named structure of type " (:item1 named-structure-p ,nss)) result)
  (push '("") result)
  (cond
    ((setq d (get nss 'si::defstruct-description))
     (setq alist (si::defstruct-description-slot-alist d))
     (do ((l alist (cdr l)))
	 ((null l) nil)
       (setq maxl (max (flatsize (caar l)) maxl)))
     ;; For a named structure, each line contains the name and the value
     
     (loop for l on alist
	   for slot-name = (first (first l))
	   for slot-value =
	       (catch-error
		 (funcall
		   (si:defstruct-slot-description-ref-macro-name
		     (cdar l))
		   data)
		 nil)
	   do
       (push `((:item1 named-structure-slot ,slot-name)
	       (:colon ,(+ 2 maxl))
	       (:item1 named-structure-value ,slot-value))
	     result)
       (setq result (maybe-expand-names-and-values-for-slot
		      result data t slot-name slot-value window
	            )
       )
     )
    )
    ((setq defstruct-items (get nss 'si::defstruct-items))
     (dolist (elt defstruct-items)
       (setq maxl (max (flatsize elt) maxl)))
     ; For a named structure, each line contains the name and the value
     (dolist (elt defstruct-items)
       (push `((:item1 named-structure-slot ,elt)
	       (:colon ,(+ 2 maxl))
	       (:item1 named-structure-value
		       ,(catch-error (funcall elt data) nil)))
	     result))))
  (if (and (arrayp data) (array-has-leader-p data))
      (send window :object-array data t (nreverse result))
      ;mention-leader is always T
      (values (nreverse result) data 'inspect-printer)))

(defmethod (show-generic-defstruct :help) ()
"Gives help when you middle button on a defstruct."
  (format nil "
The inspection pane you just selected is currently displaying the defstruct
instance ~S simply as a defstruct instance.  Mousing L2 on it should show it
to you in some other way.
"
	  data))

;-------------------------------------------------------------------------------

(defflavor show-instance
	   ()
	   (generic-middle-button-mixin inspection-data)
  (:documentation
"A fall-back option perspective which allows instances that
 have their own magic inspector behaviour to be inspected also as the
 instances that implement them.
"
  )
)

(defmethod (show-instance :format-concisely) (stream)
"Just prints it out but notes that it is an instance perspective."
  (format stream "~ as an Instance" (list data t data))
)

(defmethod (show-instance :generate-item-specialized) (window)
"Makes the inspector items for an instance."
  (let ((*inhibit-inspection-data* t))
       (multiple-value-bind (items ignore ignore ignore title)
	  (send window
		(if (typep data 'any-sort-of-clos-instance)
		    :object-clos-instance
		    :object-instance
		)
		data
	  )
	  (values items title)
       )
  )
)

(defmethod (show-instance :help) ()
"Gives help when you middle button on an instance."
  (format nil "
The inspection pane you just selected is currently displaying the
instance ~S simply as an instance.  Mousing L2 on it should show it
to you in some other way.
"
	  data
  )
)

;-------------------------------------------------------------------------------

(defflavor show-package-description
	   ()
	   (generic-middle-button-mixin inspection-data)
  (:documentation
"A perspective to view packages as if you were evaluating describe-package
on them.
"
  )
)

(defmethod (show-package-description :format-concisely) (stream)
"Just prints it out but notes that it is an package-description perspective."
  (format stream "~ as an Package Description" (list data t data))
)

(defun princ-package-name (package stream &rest ignore)
"Princs out the name of a package."
  (format stream "~A" (package-name (send package :Data)))
)

(defmethod (show-package-description :generate-item) ()
"Makes the inspector items for an package-description."
  (let ((*nopoint t))
       (values
	 `(,*blank-line-item*
	   ,@(if (package-nicknames data)
		`(((:Font 1 "Nicknames") (:Colon 40)
		  ,(format nil "~{~A~^, ~}" (package-nicknames data))
		  )
		 )
		 nil
	     )
	   ((:Font 1
	     ,(format nil "~D symbols out of a max of ~D  Hash modulus = ~D.~&"
		      (sys:pack-number-of-symbols data)
		      (sys:pack-max-number-of-symbols data)
		      (sys:p-number-of-entries (sys:pack-symbol-table data))
	      )
	    )
	   )
	   ,@(if (package-use-list data)
		`(((:Font 1 "Packages used by this one") (:Colon 40)
		  ,@(loop for pack in (package-use-list data) append
		      `(,@(if (equal pack (first (package-use-list data)))
			      nil
			      '(", ")
			  )
			(:Item1 instance
				,(allocate-data 'Show-Package-Description pack)
				princ-package-name
			)
		       )
		    )
		  )
		 )
		 nil
	     )
	   ,@(if (package-used-by-list data)
		`(((:Font 1 "Packages that use this one") (:Colon 20)
		  ,@(loop for pack in (package-used-by-list data) append
		      `(,@(if (equal pack (first (package-used-by-list data)))
			      nil
			      '(", ")
			  )
			(:Item1 instance
				,(allocate-data 'Show-Package-Description pack)
				princ-package-name
			)
		       )
		    )
		  )
		 )
		 nil
	     )
	   ,@(if (package-shadowing-symbols data)
		`(((:Font 1 "Symbols shadowed in this package"))
		  ,@(loop for symbol in (package-shadowing-symbols data) collect
			`("	" (:Item1 t ,symbol))
		    )
		 )
		 nil
	     )
	   ,@(if (sys:pack-store-function data)
		`(((:Font 1 "Symbols are interned in this package using ")
		   (:Item1 t (sys:pack-store-function data))
		  )
		 )
		 nil
	     )
	   ,@(if (sys:pack-auto-export-p data)
		`(((:Font 1
		    "Symbols in this package are automatically exported"
		   )
		  )
		 )
		 nil
	     )
	   ,@(if (sys:pack-plist data)
		`(((:Font 1 "Additional properties of this package:"))
		  ,@(loop for (key value) on (sys:pack-plist data) by #'cddr
			  collect
			  `("	" (:Item1 t ,key) (:Colon 40) (:Item1 t ,value))
		    )
		 )
		 nil
	     )
	   )
	  `(:font fonts:hl12bi
		  :string ,(format nil "Package ~S" (package-name data)))
       )
  )
)

(defmethod (show-package-description :help) ()
"Gives help when you middle button on an Package Description."
  (format nil "
The inspection pane you just selected is currently displaying the
package of ~S as an package-description. Mousing L2 on it should
show it to you in some other way.
"
	  data
  )
)

;-------------------------------------------------------------------------------

(defflavor show-area
	   ()
	   (generic-middle-button-mixin inspection-data)
  (:documentation
"A perspective to view areas as if you were evaluating describe-area
on them.
"
  )
)

(defmethod (show-area :format-concisely) (stream)
"Just prints it out but notes that it is an area perspective."
  (format stream "~ as an Area Description" (list data t (sys:area-name data)))
)

(defun split-into-lines-1 (string start end result)
"Splits a string into a list of substrings, one per line."
  (declare (optimize (speed 3) (safety 0)))
  (let ((index (sys:%string-search-char #\newline string start end)))
       (let ((line (if (stringp string)
		       (nsubstring string start index)
		       (let ((str (make-string
				    (- (or index (array-active-length string))
				       start
				    )
				  )
			     )
			    )
			    (loop for from-index from start below (length str)
				  for to-index from 0
				  do (setf (aref str to-index)
					   (let ((ch (aref string from-index)))
					        (typecase ch
						  (character ch)
						  (fixnum (code-char ch))
						  (otherwise #\?)
						)
					   )
				     )
			    )
;			    (copy-array-portion string start index
;						str 0 (length str)
;			    )
			    str
		       )
		   )
	     )
	    )
	    (if index
		(Split-Into-Lines-1 string (+ 1 index) end (cons line result))
		(nreverse (cons line result))
	    )
       )
  )
)

(defun split-into-lines (string &optional (start 0) (end (length string)))
"Splits a string into a list of substrings, one per line."
  (split-into-lines-1 string start end nil)
)

(defmethod (inspection-data :generate-item-from-standard-output) (function)
"Makes the inspector items for an area."
  (let ((string
	  (with-output-to-string (*standard-output*) (funcall function data))
	)
       )
       (cons *blank-line-item*
	     (mapcar #'list (Split-Into-Lines string))
       )
  )
)

(defmethod (show-area :generate-item) ()
"Makes the inspector items for an area."
  (let ((*nopoint t))
       (values
	  (send self :Generate-Item-From-Standard-Output 'sys:describe-area)
	  `(:font fonts:hl12bi
		  :string ,(format nil "Area ~S" (sys:area-name data))
	   )
       )
  )
)

(defmethod (show-area :help) ()
"Gives help when you middle button on an Area Description."
  (format nil "
The inspection pane you just selected is currently displaying the
Area of ~S as an area description. Mousing L2 on it should
show it to you in some other way.
"
	  data
  )
)

;-------------------------------------------------------------------------------

(defflavor show-system
	   ()
	   (generic-middle-button-mixin inspection-data)
  (:documentation
"A perspective to view systems as if you were evaluating describe-system
on them.
"
  )
)

(defmethod (show-system :format-concisely) (stream)
"Just prints it out but notes that it is a system description perspective."
  (format stream "~ as an System Description"
	  (list data t (sys:system-name data))
  )
)

(defmethod (show-system :generate-item) ()
"Makes the inspector items for a system."
  (let ((*nopoint t))
       (values
	  (send self :Generate-Item-From-Standard-Output 'sys:describe-system)
	  `(:font fonts:hl12bi
		  :string ,(format nil "System ~S" (sys:system-name data))
	   )
       )
  )
)

(defmethod (show-system :help) ()
"Gives help when you middle button on a System Description."
  (format nil "
The inspection pane you just selected is currently displaying the
System ~S as an system description. Mousing L2 on it should
show it to you in some other way.
"
	  data
  )
)

;-------------------------------------------------------------------------------

(defflavor show-bit-array
	   ()
	   (generic-middle-button-mixin inspection-data)
  (:documentation
"A perspective to view bit-arrays as 2-d bit maps.
"
  )
)

(defmethod (show-bit-array :format-concisely) (stream)
"Just prints it out but notes that it is a bit-array perspective."
  (format stream "~ as an Bit-Array" (list data t data))
)

(defmethod (show-bit-array :generate-item-specialized)
     (window &optional (mention-leader (array-has-leader-p data)) initial-items)
"Makes the inspector items for an bit-array."
  (ignore window)
  (setq initial-items (append initial-items '((""))))
  (values nil (list data mention-leader initial-items)
	  'inspect-bit-array-printer 0
	  `(:font fonts:hl12bi
		  :string ,(format nil "~S as a bit array" data)
	   )
	  'inspect-bit-array-item-generator
  )
)

(defmethod (show-bit-array :help) ()
"Gives help when you middle button on a Bit-array."
  (format nil "
The inspection pane you just selected is currently displaying the
array ~S as a bit-array. Mousing L2 on it should
show it to you in some other way.
"
	  data
  )
)

;-------------------------------------------------------------------------------

(defflavor show-bit-array-sideways
	   ()
	   (generic-middle-button-mixin inspection-data)
  (:documentation
"A perspective to view bit-arrays as 2-d bit maps on their sides.
"
  )
)

(defmethod (show-bit-array-sideways :format-concisely) (stream)
"Just prints it out but notes that it is a bit-array perspective."
  (format stream "~ as an Bit-Array on its side" (list data t data))
)

(defmethod (show-bit-array-sideways :generate-item-specialized)
     (window &optional (mention-leader (array-has-leader-p data)) initial-items)
"Makes the inspector items for an bit-array."
  (ignore window)
  (setq initial-items (append initial-items '((""))))
  (values nil (list data mention-leader initial-items)
	  #'(lambda (item arg stream item-number)
	      (inspect-bit-array-printer item arg stream item-number t)
	    )
	  0
	  `(:font fonts:hl12bi
		  :string ,(format nil "~S as a bit array" data)
	   )
	  #'(lambda (msg &optional arg1)
	      (inspect-bit-array-item-generator msg arg1 t)
	    )
  )
)

(defmethod (show-bit-array-sideways :help) ()
"Gives help when you middle button on a Bit-array."
  (format nil "
The inspection pane you just selected is currently displaying the
array ~S as a bit-array on its side. Mousing L2 on it should
show it to you in some other way.
"
	  data
  )
)

;-------------------------------------------------------------------------------

;;; Copied and modified from inspect-array-item-generator
(defun inspect-bit-array-item-generator (msg &optional arg1 (sideways-p nil))
  (declare (:self-flavor basic-inspect))
  (case msg
    (:Number-Of-Items
     (+ (if (second print-function-arg)
	    (or (array-leader-length (first print-function-arg)) 0)
	    0)
	(length (third print-function-arg))
	(let ((dims (array-dimensions (first print-function-arg))))
	     (if sideways-p
		 (second dims)
		 (first dims)
	     )
	)
     )
    )
    (:Number-Of-Item
     (if (numberp arg1)
	 (+ arg1 (length (third print-function-arg)))
	 (position arg1 (the list (third print-function-arg)) :test #'eq)
     )
    )
    (:Item-Of-Number
     (if (< arg1 (length (third print-function-arg)))
	 (nth arg1 (third print-function-arg))
	 (- arg1 (length (third print-function-arg)))
     )
    )
  )
)


(defvar *display-art-1b-arrays-as-pictures-p* nil
"When true uses blobs and blanks for the bits in art-1bs."
)

;;; Copied and modified from inspect-array-printer
(defun inspect-bit-array-printer
       (item arg stream item-number &optional (sideways-p nil)
	&aux (obj (first arg))
	(leader-length-to-mention (if (second arg) (array-leader-length obj) 0))
	(font :Default)
       )
  "The print-function used when inspecting an array."
  ;; (FIRST ARG) is the array.  (SECOND ARG) is T to display the leader.
  ;; ITEM is usually a number.  A small number is an index in the leader.
  (declare (type stream stream))
  (cond ((not (numberp item))
	 (inspect-printer item obj stream item-number)
	)
        ((< item leader-length-to-mention)
	 ;;; Print out the array leader like normal.
	 (let ((pntr (locf (array-leader obj item))))
	      (send stream :item1 item 'leader-slot
		    #'(lambda (item stream) (format stream "Leader ~D" item)))
	      (format stream ":~12T ")
	      (if (%p-contents-safe-p pntr)
		  (send stream :item1 (array-leader obj item)
			:value #'print-item-concisely)
		  (format stream "#<~A ~O>"
			  (or (nth (%p-data-type pntr) q-data-types)
			      (%p-data-type pntr))
			  (%p-pointer pntr)
		  )
	      )
	 )
	)
        (t ;;; Normal array slot.
	   (let ((sheet (typecase stream
			  (sheet stream)
			  (otherwise (send stream :Superior))
			)
		 )
		)
	        (coerce-font font sheet)
	        (let ((item (- item leader-length-to-mention))
		      (char-width
			(floor (send sheet :inside-width)
			       (font-char-width font)
			)
		      )
		     )
		     (loop for column from 0 below
			   (if sideways-p
			       (first  (array-dimensions obj))
			       (second (array-dimensions obj))
			   )
			   while (< column char-width)
			   do
			   (send stream :Item1
				 (if sideways-p
				     (list obj column item)
				     (list obj item column)
				 )
				 'bit-array-element
				 'print-hex-concisely
			   )
		     )
		)
	   )
	)
  )
)

(defun print-hex-concisely (datum stream &optional (level 0))
"Prints the item  out as a hex number."
  (let ((*print-circle* nil))
       (if (and *display-art-1b-arrays-as-pictures-p*
		(equal (array-type (first datum)) 'art-1b)
	   )
	   (let ((value (aref (first datum) (second datum) (third datum))))
		(princ (if (equal 0 value) " " 1) stream)
	   )
	   (let ((*print-base* 16.))
		(print-item-concisely
		  (aref (first datum) (second datum) (third datum))
		  stream level
		)
	   )
       )
  )
)

(defun (:property bit-array-element set-function) (item new-value ignore)
  (eval `(setf (aref ,@(second item)) ',new-value))
)

;-------------------------------------------------------------------------------

(defflavor show-generic-object-thing
	   ()
	   (generic-middle-button-mixin inspection-data)
  (:documentation
"A fall-back option perspective which allows things to be inspected in the
 old way :object-foo and all that.
"
  )
)

(defmethod (show-generic-object-thing :format-concisely) (stream)
"Just prints it."
  (inspection-data-print-item-concisely data stream)
)

(defun only-first-line (string window font)
"Makes sure that String can be printed easily on window in font Font.  If there
 are newlines in String then only the first line is taken, also the substring
 of string which will fit onto about one line in font Font it taken.  The
 (maybe) shorter string is returned.
"
  (declare (type string string))
  (let ((index (search (string #\newline) string :Test #'string-equal)))
       (let ((shorter-string (if index (subseq string 0 index) string)))
	    (subseq (the string shorter-string) 0
		    (floor (/ (send window :Inside-Width)
			      (font-char-width font)
			   )
		    )
	    )
       )
  )
)
	   
(defmethod (show-generic-object-thing :generate-item-specialized) (window)
"Makes the inspector items for something."
  (let ((*inhibit-inspection-data* t))
       (let ((results (multiple-value-list
			(send window (generic-object-foo-method data) data)
		      )
	     )
	    )
	    (if (fifth results)
		(values-list results)
		(values-list
		 (append (list (first results) (second results) (third results)
			       (fourth results)
			       `(:font fonts:hl12bi :String
				       ,(Only-First-Line
					  ;;; If this is a defstruct, then
					  ;;; just show the #< form, since
					  ;;; we're already looking inside it.
					  (let ((*print-structure* nil))
					       (format nil "~s" data)
					  )
					  window fonts:hl12bi
					)
				)
			 )
			 (nthcdr 5 results)
		 )
		)
	    )
       )
  )
)

(defmethod (show-generic-object-thing :help) ()
"Gives help when you middle button on something."
  (format nil "
The inspection pane you just selected is currently displaying the
~S ~S.
"
	  (type-of data) data
  )
)

(defmethod (show-generic-object-thing :middle-button-result) ()
  data
)

;-------------------------------------------------------------------------------

(defflavor show-function () (inspection-data)
  (:documentation
"A different perspective on functions, which allow you
 to see all sorts of interesting things about them.
"
  )
)

(defmethod (show-function :format-concisely) (stream)
"Prints out the function name.  The whole #'fn expression point to the function
 inspected in this way.  The name itself points to the function name.
"
  (format stream "#'~" (list (function-name data) t (function-name data)))
)

(defmethod (show-function :handle-mouse-click) (blip flavor-inspector)
"A simple mouse click handler for functions."
  (selector (fourth blip) =
    (#\mouse-l-1 (send flavor-inspector :inspect-info-left-click))
    (#\mouse-l-2 (send flavor-inspector :inspect-info-left-2-click))
    (#\mouse-m-1 (send flavor-inspector :inspect-info-middle-click))
    (otherwise (beep))
  )
)

(defmethod (show-function :generate-item) ()
"An item generator for functions.  This is handled much like CLOS methods and
 generic functions and like flavors methods.  Any interesting seeming aspects of
 the functions are displayed.
"
   (values
     (multiple-value-bind
       (referenced-ivars referenced-keywords problem
	referenced-functions referenced-generic-functions args returned-values
	locals specials-referenced specials-bound
       )
         (ivars-and-messages-in-method data)
       (ignore problem)
      `(,*blank-line-item*
	((:font 1 "Details of ")
	 (:item1 instance
		 ,(allocate-data 'show-function data)))
	,*blank-line-item*
	((:font 1 "Source File:               ")
	,(if (si:function-spec-get (function-name data) :source-file-name)
	     (path-string (function-name data) 'defun)
	     (format nil "Not Defined")))
	(,(if returned-values
	      '(:font 1 "Arglist  Returned Values: ")
	      '(:font 1 "Arglist:                   "))
	 ("~:[~*()~;~S~]" ,args ,args)
	 ,@(when returned-values
	     `(("  ~s" ,returned-values))))
	,*blank-line-item*
	((:font 1 "Documentation:"))
	,@(let ((doc (documentation data)))
	    (if doc
		(break-string-into-lines doc)
		*no-items*))
	,*blank-line-item*
       ((:font 1 ,(if (iwmc-class-p-safe data)
		       ""
		       "Referenced Instance Variables:")))
	,@(referenced-instance-variables-details data referenced-ivars)
	,*blank-line-item*
	((:font 1 "Referenced Keywords (possibly messages passed):"))
	,@(referenced-keywords-details referenced-keywords)
	,*blank-line-item*
	((:font 1 "Referenced Generic Functions:"))
	,@(referenced-generic-functions-details
	     referenced-generic-functions
	  )
	,*BLANK-LINE-ITEM*
	((:FONT 1 "Referenced Functions:"))
	,@(Referenced-Functions-Details referenced-functions)
	,*blank-line-item*
	((:font 1 "Locals:"))
	,@(locals-details locals)
	,*blank-line-item*
	((:font 1 "Referenced Specials:"))
	,@(referenced-specials-details specials-referenced)
	,*blank-line-item*
	((:font 1 "Specials Bound:"))
	,@(bound-specials-details specials-bound)
	,*blank-line-item*
	((:font 1 "Macros Expanded:"))
	,@(macros-expanded-details data)
	,*blank-line-item*
	((:font 1 "Interpreted Definition:"))
	,@(interpreted-definition-details data)
	))
     `(:font fonts:hl12bi
	     :string ,(format nil "Function ~S" (function-name data)))))

(defmethod (show-function :help) ()
  (let ((name (function-name data)))
    (format nil "
The inspection pane you just selected is currently displaying sundry details
about the compiled function ~S.
"
	    name)))


;-------------------------------------------------------------------------------

;;; Make a resource for general inspectors.
(defwindow-resource general-inspect-frame-resource nil :make-window
                    (general-inspector :process nil :label "foo")
		    :reusable-when :deactivated
)

(reinstall-commands 'general-inspector)

;;; Puts the general inspector on System-I, and moves the old inspector onto
;;; sym-sh-I
;(swap-system-keys #\I #\ 'inspect-frame :inspector)

(w:add-system-key
  #\I
  'tv:general-inspector
  "Inspector - examine complex data structures."
  t
)

(w:remove-system-key #\O)

(defun enable-general-inspector ()
  nil ;;; Not supported anymore.
)

(let ((sys:inhibit-fdefine-warnings t))
     (defwindow-resource eh:debugger-frame nil
       :make-window (eh:new-debugger-frame)
       :reusable-when :deactivated
       :initial-copies 0
     )
)
(clear-resource 'eh:debugger-frame)


(defun disable-general-inspector ()
  nil  ;;; Not supported anymore.
)

;-------------------------------------------------------------------------------

(defvar *name-of-specifiers*
	'((si:flavor si:flavor-name)
	  (any-sort-of-clos-class class-name-safe)
	  (compiled-function function-name)
	  (closure function-name)
	  (package package-name)
	  (any-type-of-clos-method clos-method-name)
	  (any-sort-of-clos-slotd slotd-name-safe)
	 )
"An AList that maps types of objects into the functions that can extract their
names.
"
)

;;;Edited by James Rice            9 Mar 90  14:07
(defun find-name-of (x)
  (declare (values name-of-x found-name-p matching-type))
  (let ((entry (find-if #'(lambda (spec) (typep x (first spec)))
			*name-of-specifiers*
	       )
	)
       )
       (if entry
	   (values (funcall (second entry) x) t (first entry))
	   (values nil nil nil)
       )
  )
)

;-------------------------------------------------------------------------------

;;; Representations...

(defflavor basic-perspective
	   ((name nil)
	    (already-this-type-function nil)
	    (show-x-type-for-perspective nil)
	    (this-perspective-applicable-function nil)
	    (menu-item-name nil)
	    (new-inspect-function nil)
	    (menu-who-line-doc-string nil)
	    (side-effect-function nil)
	    (priority 0)
	    (prefer-over nil)
	   )
	   ()
  :Initable-Instance-Variables
  :Gettable-Instance-Variables
  :Settable-Instance-Variables
  (:Documentation
"The encapsulation of the behaviour that lets things have multiple
 perspectives.  Each instance knows about the mapping from one
 particular type to another.
"
  )
)

(defmethod (Basic-Perspective :Print-Self) (stream depth slashify)
"Justs prints self out so that we can see its name."
  (ignore depth slashify)
  (format stream "#<Perspective ~S>" name)
)

(defmethod (Basic-Perspective :After :Init) (ignore)
"Does a little error checking and makes sure that wee have useful values for
 the menu item name and the mouse doc string.
"
  (if (not name)
      (ferror nil "No name supplied for perspective.")
      nil
  )
  (if (not menu-item-name)
      (setq menu-item-name (format nil "~S" name))
      nil
  )
  (if (not menu-who-line-doc-string)
      (setq menu-who-line-doc-string menu-item-name)
      nil
  )
  (if (not (or new-inspect-function side-effect-function ))
      (ferror nil "No mapping function provided.")
      nil
  )
)

(defmethod (Basic-Perspective :menu-item) (x show-x)
"Given a piece of data that's being inspected and the instance of
 inspection-data that's representing it (or nil if not yet allocated),
 returns a menu item for the perspective self.  If the item name or doc string
 values are functions then these are called otherwise the strings are returned.
"
  (let ((title (if (functionp menu-item-name)
		   (funcall menu-item-name x show-x)
		   menu-item-name
	       )
	)
	(doc (if (functionp menu-who-line-doc-string)
		 (funcall menu-who-line-doc-string x show-x)
		 menu-who-line-doc-string
	     )
	)
       )
       (list title :Value self :Documentation
	     (format nil "~A [~S]" doc name)
       )
  )
)

(defmethod (Basic-Perspective :get-value-to-inspect) (x show-x)
"Given a piece of data that's being inspected and the instance of
 inspection-data that's representing it (or nil if not yet allocated),
 returns four values; either:
  a) the value to inspect (calling new-inspect-function) and T, to show that
     we should inspect.
  b) x and nil, in which case side-effect-function will have been called.
     The second value tells the inspector not to do anything about this.
  c) Self's name
  d) Self
"
  (declare (values value-to-inspect inspect-p selfs-name self))
  (if side-effect-function
      (progn (funcall side-effect-function x show-x)
	     (values (or x :processed) nil name self)
      )
      (values (funcall new-inspect-function x show-x) t name self)
  )
)

(defmethod (Basic-Perspective :applicable-p) (x show-x)
"Given a piece of data that's being inspected and the instance of
 inspection-data that's representing it (or nil if not yet allocated),
 returns true if this perspective is a good one, given that we're inspecting x
 and that x is currently viewed as show-x.  The selection is done by checking
 the already-this-type-function, if provided, then the
 This-Perspective-Applicable-Function if provided and finally, if the latter
 was not provided passes if show-x-type-for-perspective is not the same as
 show-x.
"
  (and (not (and already-this-type-function
		 (funcall already-this-type-function x show-x)
	    )
       )
       (or (and This-Perspective-Applicable-Function
		(funcall This-Perspective-Applicable-Function x show-x)
	   )
	   (and (not This-Perspective-Applicable-Function)
		show-x-type-for-perspective
		(not (equal show-x show-x-type-for-perspective))
	   )
       )
  )
)

(defvar *all-perspective-names* nil
"A list of the names of all of the perspectives that have been defined."
)

(defmacro defperspective
	  (name (&rest arglist)
	   &key
	   (flavor 'Basic-Perspective)
	   (priority 0)
	   (prefer-over nil)
	   (show-x-type-for-perspective nil)
	   (menu-item-name nil)
	   (menu-who-line-doc-string nil)
	   (already-this-type-function nil)
	   (this-perspective-applicable-function nil)
	   (new-inspect-function nil)
	   (side-effect-function nil)
	   (other-keys nil)
	  )
"The way that the user defines new perspectives.  A perspective is the
 encapsulation of the mapping from one way of inspecting something into
 another way, with sundry conditionality.  For instance, using defperspective
 we can express the mapping that says:
    if we're currently inspecting a method then one way of viewing it is
    to show its method details a la flavor inspector, but don't use this
    perspective if we're already looking at it this way.

 Defperspective takes a number of arguments and keyword arguments, which
 have the following meanings:

 Name - The name of the perspective.  This should be a symbol.  It will be
        included in the list *All-Perspective-Names*.  The perspective object
        that is created is stored on the :perspective property of this symbol.
 Arglist - this is the arglist that will be used in the functions that are
        created by defperspective, so that you can name the args.  The arglist
        must be a two-list.  The first element is the name to be given to the
        actual object to inspect and the second is the name to be given to
        the instance of inspection-data (show-foo) that is currently
        encapsulating the first arg.  An example of this argument might be:
            (defperspective :symbol-as-rule (symbol show-x) ...
 Flavor - the flavor of perspective object that is created.  Only significant
        if, for some reason you want to have different behaviour for your
        perspectives.  See the definition of tv:basic-perspective and its
        protocol if you want to see what a perspective object must be able
        to do.
 Priority - A number used to order perspectives.  This is significant if you
        want to make sure that one particular perspective comes out as being
        the default.  Most perspectives have a priority of 0, but the
        perspective that displays things just as generic data structures
        (:generic-data-structure) has priority 10, so your priority should
        be higher than this if you want to override this behaviour by default.
 Prefer-Over - If you don't like the idea of putting priority numbers on your
        perspectives then you can express relative priorities by the use of this
        arg.  It should be a list of the names of perspectives that you should
        use this one rather than, e.g. :prefer-over (:generic-data-structure).
 Show-X-Type-For-Perspective - is the name of the type of inspection-data
        allocated by this perspective.
 -----------------------

 Perspective functions
 =====================
  All functions called during perspectivisation or generated by defperspective
  have the same arglists: (item show-item).  This always the case (that's why
  the Arglist argument must have two items.  The first argument is always the
  thing that's being inspected and the second is either nil or the instance
  of inspection-data (show-foo) that's being used to represent the item.  The
  second arg is nil in the event of no perspective having been allocated yet.
  Two arguments can optionally be functions.

 Menu-Item-Name - This can be one of three values: defaulted, in which case
        if a menu is generated then the name of the perspective will be used;
        a string, in which case this will be used as the name of the item in
        the menu; or a function in which case this will be called as above
        and the result of this will be used as the menu item name.
 Menu-Who-Line-Doc-String - This can be one of three values: defaulted, in
        which case if a menu is generated then the value of Menu-Item-Name
        will be used; a string, in which case this will be used as the who
        line doc string for this item in the menu; or a function in which
        case this will be called as above and the result of this will be
        used as the who line doc string.

  The following arguments are functions generated by defperspective.  You can
  provide the body code for them.  See the definition of
  Already-This-Type-Function for an example of this:

 Already-This-Type-Function -  A function body which, when true says that
        the thing that we're already inspecting is aready of this type, so we
        don't want to use this perspective.  The args defined in the Arglist
        arg above are used in the construction of the function.  For example,
        if our perspective is intended to show flavors defstruct instances
        simply as generic defstructs then we might say:
          (Defperspective :flavor-as-defstruct (x show-x)
            :already-this-type-function
               (and (typep x 'si:flavor)
                    (typep show-x 'Show-Generic-Object-Thing))
        Note how in this case we're using the arglist (x show-x).  If the thing
        we're inspecting is a flavor and it's encapsulated within a
        Show-Generic-Object-Thing then we don't want this perspective.
 This-Perspective-Applicable-Function - A function body, which when true says
        that this perspective is, indeed applicable.  Example:
          (Defperspective :flavor-as-flavor-inspect (x show-x)
            :This-Perspective-Applicable-Function
              (or (typep x 'si:flavor) (and (symbolp x) (get x 'si:flavor)))
        Note how this perspective will work for for any flavor object or symbol
        that names a flavor.
 New-Inspect-Function - An optional function body which is called when the
        perspective mechanism has decided that this perspective is applicable.
        It must return the new thing to inspect, calling allocate data as
        appropriate.  This argument cannot be used with Side-Effect-Function.
        Example:
          (Defperspective :flavor-as-flavor-inspect (x show-x)
            :New-Inspect-Function
              (allocate-data 'show-flavor
                (if (typep x 'si:flavor) x (get x 'si:flavor)))
        Note, in this case we've decided that X can be either s flavor or a
        symbol that names a flavor.
 Side-Effect-Function - An optional function body which is called when the
        perspective mechanism has decided that this perspective is applicable.
        It must perform whatever side effects are needed to implement this
        perspective.  This argument cannot be used with New-Inspect-Function.
        Example:
          (defperspective :flavor-as-graph (x show-x)
            :Side-Effect-Function
              (inspect-graph-class
                (if (typep x 'si:flavor) x (get x 'si:flavor)))


 Other-keys - If the :flavor option is used then this might take other
        init-args.  Other-keys are used to encapsulate these.
"
 `(let ((instance (make-instance
		    ',flavor
		    :Name ,name
		    :Already-This-Type-Function
		     ,(and already-this-type-function
			  `#'(lambda ,arglist (ignore ,@arglist)
				     ,already-this-type-function
			     )
		      )
		    :show-x-type-for-perspective ',show-x-type-for-perspective
		    :this-perspective-applicable-function
		     ,(and this-perspective-applicable-function
			  `#'(lambda ,arglist (ignore ,@arglist)
			       ,this-perspective-applicable-function
			     )
		      )
		    :Menu-Item-Name ,menu-item-name
		    :New-Inspect-Function
		     ,(and new-inspect-function
			   `#'(lambda ,arglist (ignore ,@arglist)
				      ,new-inspect-function
			      )
		      )
		    :Menu-Who-Line-Doc-String ,menu-who-line-doc-string
		    :Side-Effect-Function
		     ,(and side-effect-function
			   `#'(lambda ,arglist (ignore ,@arglist)
				      ,side-effect-function
			      )
		      )
		    :Priority ,(and priority
				    `#'(lambda ,arglist (ignore ,@arglist)
					       ,priority
				       )
			       )
		    :Prefer-Over ',prefer-over
		    ,@other-keys
		  )
	)
       )
       (setf (get ',name :perspective) instance)
       (pushnew ',name *All-Perspective-Names*)
       (Def ,name)
  )
)


;;;Edited by James Rice            9 Mar 90  14:07
(Defperspective :name (x show-x)
  :show-x-type-for-perspective show-generic-object-thing
  :This-Perspective-Applicable-Function (Find-Name-Of x)
  :menu-item-name "Name"
  :Menu-Who-Line-Doc-String "Inspect the name of this object."
  :New-Inspect-Function
    (allocate-data 'show-generic-object-thing (Find-Name-Of x))
  :Priority -1
)

(Defperspective :locative (x show-x)
  :already-this-type-function
    (and (locativep x) (typep show-x 'Show-Data-With-Locative))
  :This-Perspective-Applicable-Function
    (locativep x)
  :menu-item-name "Locative"
  :Menu-Who-Line-Doc-String
    "As a locative"
  :New-Inspect-Function
   (allocate-data 'show-data-with-locative x x)
  :Prefer-Over '(:generic-data-structure)
)

(Defperspective :class-instance (x show-x)
  :already-this-type-function
    (and (class-p-safe x)
	 (ticlos-p)
	 (typep x 'clos:class)
	 (or (typep show-x 'Show-Generic-Object-Thing)
	     (typep show-x 'show-instance)
	 )
    )
  :This-Perspective-Applicable-Function
    (or (and (class-p-safe x)
	     (ticlos-p)
	     (typep x 'clos:class)
	)
	(and (symbolp x) (ticlos-p) (ticlos:class-named x t))
    )
  :menu-item-name "Class instance"
  :Menu-Who-Line-Doc-String
    "As an instance of the data structure used to implement the class"
  :New-Inspect-Function
   (allocate-data (if (instancep x)
		      'show-instance
		      'Show-Generic-Object-Thing
		  )
		  (if (class-p-safe x) x (ticlos:class-named x t))
   )
  :Prefer-Over '(:pcl-class-instance)
)

(Defperspective :Pcl-class-instance (x show-x)
  :already-this-type-function
    (and (class-p-safe x)
	 (pcl-p)
	 (pcl-class-p x)
	 (or (typep show-x 'Show-Generic-Object-Thing)
	     (typep show-x 'show-instance)
	 )
    )
  :This-Perspective-Applicable-Function
    (or (and (class-p-safe x)
	     (pcl-p)
	     (pcl-class-p x)
	)
	(and (symbolp x) (pcl-p) (pcl:find-class x nil))
    )
  :menu-item-name "PCL Class instance"
  :Menu-Who-Line-Doc-String
    "As an instance of the data structure used to implement the class"
  :New-Inspect-Function
   (allocate-data (if (instancep x)
		      'show-instance
		      'Show-Generic-Object-Thing
		  )
		  (if (class-p-safe x) x (pcl:find-class x nil))
   )
)

(Defperspective :Pcl-object-implementation (x show-x)
  :already-this-type-function
    (and (pcl-p)
	 (iwmc-class-p-safe x)
	 (typep show-x 'Show-Generic-defstruct)
    )
  :This-Perspective-Applicable-Function
    (and (pcl-p) (iwmc-class-p-safe x))
  :menu-item-name "PCL implementation using defstruct"
  :Menu-Who-Line-Doc-String
    "As an instance of the defstruct used to implement the object"
  :New-Inspect-Function
   (allocate-data 'Show-Generic-defstruct x)
  :Priority -1
)

(Defperspective :Pcl-generic-function-implementation (x show-x)
  :already-this-type-function
    (and (pcl-p)
	 (pcl:generic-function-p x)
	 (typep show-x 'Show-Generic-Object-Thing)
    )
  :This-Perspective-Applicable-Function
    (and (pcl-p) (pcl:generic-function-p x))
  :menu-item-name "PCL generic function implementation"
  :Menu-Who-Line-Doc-String
    "As an instance of the closure used to implement the generic function"
  :New-Inspect-Function
   (allocate-data 'Show-Generic-Object-Thing x)
  :Priority -1
)

(Defperspective :class (x show-x)
  :show-x-type-for-perspective show-clos-class
  :This-Perspective-Applicable-Function
    (and (or (and (class-p-safe x) (ticlos-p)
		  (typep x 'clos:class)
	     )
	     (and (symbolp x) (class-named-safe x t)
		  (ticlos-p) (ticlos:class-named x t)
	     )
	 )
	 (not (typep show-x 'show-clos-class))
    )
  :menu-item-name "Class"
  :Menu-Who-Line-Doc-String
    "Inspect this as a CLOS class, showing its inheritance hierarchy."
  :New-Inspect-Function
   (allocate-data 'show-clos-class
		  (if (class-p-safe x) x (ticlos:class-named x))
   )
  :Prefer-Over (:Pcl-Class :Class-Instance)
)

(Defperspective :pcl-class (x show-x)
  :show-x-type-for-perspective show-clos-class
  :This-Perspective-Applicable-Function
    (and (or (and (class-p-safe x) (pcl-p)
		  (pcl-class-p x)
	     )
	     (and (symbolp x) (class-named-safe x t)
		  (pcl-p) (pcl:find-class x nil)
	     )
	 )
	 (not (typep show-x 'show-clos-class))
    )
  :menu-item-name "PCL Class"
  :Menu-Who-Line-Doc-String
    "Inspect this as a PCL class, showing its inheritance hierarchy."
  :New-Inspect-Function
   (allocate-data 'show-clos-class (if (class-p-safe x) x (pcl:find-class x)))
  :Prefer-Over (:PCL-Class-Instance)
)

(Defperspective :flavor-instance (x show-x)
  :already-this-type-function
    (and (typep x 'si:flavor) (typep show-x 'Show-Generic-Object-Thing))
  :This-Perspective-Applicable-Function
    (or (typep x 'si:flavor) (and (symbolp x) (get x 'si:flavor)))
  :menu-item-name "Flavor instance"
  :Menu-Who-Line-Doc-String
    "Inspect this as the defstruct instance that implements the flavor."
  :New-Inspect-Function
   (allocate-data 'Show-Generic-Object-Thing
		  (if (typep x 'si:flavor) x (get x 'si:flavor))
   )
)

(Defperspective :flavor (x show-x)
  :show-x-type-for-perspective show-flavor
  :already-this-type-function (typep show-x 'Show-flavor)
  :This-Perspective-Applicable-Function
    (or (typep x 'si:flavor) (and (symbolp x) (get x 'si:flavor)))
  :menu-item-name "Flavor"
  :Menu-Who-Line-Doc-String
    "Inspect this as a Flavor, showing its inheritance hierarchy."
  :New-Inspect-Function
    (allocate-data 'show-flavor (if (typep x 'si:flavor) x (get x 'si:flavor)))
)

(Defperspective :normal-list (x show-x)
  :show-x-type-for-perspective show-generic-object-thing
  :This-Perspective-Applicable-Function (listp x)
  :menu-item-name "Unstructured list"
  :Menu-Who-Line-Doc-String "Show this list as a normal ground list."
  :New-Inspect-Function (allocate-data 'show-generic-object-thing x)
)

(Defperspective :plist (x show-x)
  :show-x-type-for-perspective show-plist
  :This-Perspective-Applicable-Function (listp x)
  :menu-item-name "PList"
  :Menu-Who-Line-Doc-String "Show it as if it was a PList."
  :New-Inspect-Function (allocate-data 'show-plist x)
)

(Defperspective :offset-plist (x show-x)
  :show-x-type-for-perspective show-offset-plist
  :This-Perspective-Applicable-Function (listp x)
  :menu-item-name "Offset PList"
  :Menu-Who-Line-Doc-String "Show it as if it was a PList, offset at the start."
  :New-Inspect-Function (allocate-data 'show-offset-plist x)
)

(Defperspective :alist (x show-x)
  :show-x-type-for-perspective show-alist
  :This-Perspective-Applicable-Function (listp x)
  :menu-item-name "AList"
  :Menu-Who-Line-Doc-String "Show it as if it was an AList."
  :New-Inspect-Function (allocate-data 'show-alist x)
)

(Defperspective :one-on-a-line (x show-x)
  :show-x-type-for-perspective show-one-on-a-line
  :This-Perspective-Applicable-Function (listp x)
  :menu-item-name "One on a line"
  :Menu-Who-Line-Doc-String "Each element of the line on a separate line."
  :New-Inspect-Function (allocate-data 'show-one-on-a-line x)
)

(Defperspective :named-structure (x show-x)
  :show-x-type-for-perspective show-list-named-structure
  :This-Perspective-Applicable-Function
   (and (listp x)
	(symbolp (first x))
	(get (first x) 'si:defstruct-description)
   )
  :menu-item-name "Named Structure"
  :Menu-Who-Line-Doc-String
    "As if it is a defstruct instance whose type is (first <list>)."
  :New-Inspect-Function (allocate-data 'show-list-named-structure x)
)

(Defperspective :unnamed-structure (x show-x)
  :show-x-type-for-perspective show-list-unnamed-structure
  :This-Perspective-Applicable-Function (listp x)
  :menu-item-name "Structure"
  :Menu-Who-Line-Doc-String
    "As if it is a defstruct instance whose type you specify."
  :New-Inspect-Function
    (let ((type-name (read-type-name)))
         (allocate-data 'show-list-unnamed-structure x type-name)
    )
)

(Defperspective :offset-unnamed-structure (x show-x)
  :show-x-type-for-perspective show-list-offset-unnamed-structure
  :This-Perspective-Applicable-Function (listp x)
  :menu-item-name "Offset Structure"
  :Menu-Who-Line-Doc-String
    "As if it is a defstruct instance whose type you specify (offset)."
  :New-Inspect-Function
    (let ((type-name (read-type-name)))
         (allocate-data 'show-list-offset-unnamed-structure x type-name)
    )
)

(Defperspective :hash-table (x show-x)
  :show-x-type-for-perspective Show-hash-table
  :This-Perspective-Applicable-Function
    (and (hash-table-p x) (not (typep show-x 'show-hash-table)))
  :menu-item-name "Hash Table elements"
  :New-Inspect-Function (allocate-data 'Show-Hash-Table x)
  :Prefer-Over (:Hash-Table-Instance)
)

(Defperspective :hash-table-sorted (x show-x)
  :show-x-type-for-perspective Show-hash-table-sorted
  :This-Perspective-Applicable-Function
    (and (hash-table-p x) (not (typep show-x 'show-hash-table-sorted)))
  :menu-item-name "Hash Table elements (sorted)"
  :New-Inspect-Function (allocate-data 'Show-Hash-Table-sorted x)
  :Priority -1
)

(Defperspective :hash-table-instance (x show-x)
  :show-x-type-for-perspective Show-generic-defstruct
  :This-Perspective-Applicable-Function
    (and (hash-table-p x) (not (typep show-x 'show-generic-defstruct)))
  :menu-item-name "Hash Table instance"
  :New-Inspect-Function (allocate-data 'Show-generic-defstruct x)
)

(Defperspective :package (x show-x)
  :show-x-type-for-perspective Show-Generic-Object-Thing
  :This-Perspective-Applicable-Function (and (symbolp x) (find-package x))
  :menu-item-name "Package"
  :Menu-Who-Line-Doc-String
    "Inspect this as the package named by this symbol."
  :New-Inspect-Function
    (allocate-data 'Show-Generic-Object-Thing (find-package x))
)

(Defperspective :package-description (x show-x)
  :show-x-type-for-perspective Show-Package-Description
  :This-Perspective-Applicable-Function
    (or (and (symbolp x) (find-package x))
	(and (packagep x) (not (typep show-x 'show-package-description)))
    )
  :menu-item-name "Package Description"
  :Menu-Who-Line-Doc-String
    "Inspect this as a package-description."
  :New-Inspect-Function
    (allocate-data 'Show-Package-Description (find-package x))
)

(Defperspective :area-description (x show-x)
  :show-x-type-for-perspective Show-Area
  :This-Perspective-Applicable-Function
    (or (and (symbolp x) (boundp x)
	     (and (fixnump (symbol-value x))
		  (>= (symbol-value x) 0)
		  (< (symbol-value x) (length #'area-name))
	     )
	)
	(and (fixnump x) (>= x 0) (< x (length #'area-name)))
    )
  :menu-item-name "Area Description"
  :Menu-Who-Line-Doc-String
    "Inspect this as an area-description."
  :New-Inspect-Function
    (allocate-data 'Show-Area (if (symbolp x) (symbol-value x) x))
)

(Defperspective :system (x show-x)
  :show-x-type-for-perspective Show-Generic-Object-Thing
  :This-Perspective-Applicable-Function
    (and (or (symbolp x) (stringp x)) (sys:find-system-named x t t))
  :menu-item-name "System"
  :Menu-Who-Line-Doc-String
    "Inspect this as a System."
  :New-Inspect-Function
    (allocate-data 'Show-Generic-Object-Thing (sys:find-system-named x t t))
)

(Defperspective :system-description (x show-x)
  :show-x-type-for-perspective Show-System
  :This-Perspective-Applicable-Function
    (or (and (or (symbolp x) (stringp x))
	     (sys:find-system-named x t t)
	)
	(and (typep x 'sys:system) (not (typep show-x 'show-system)))
    )
  :menu-item-name "System Description"
  :Menu-Who-Line-Doc-String
    "Inspect this as a System-Description."
  :New-Inspect-Function
    (allocate-data 'Show-System (sys:find-system-named x t t))
)

(Defperspective :bit-array (x show-x)
  :show-x-type-for-perspective Show-Bit-Array
  :This-Perspective-Applicable-Function
    (and (arrayp x) (equal (array-rank x) 2)
	 (member (array-type x) '(art-1b art-2b art-4b))
	 (not (typep show-x 'show-bit-array))
    )
  :menu-item-name "Bit Array"
  :Menu-Who-Line-Doc-String
    "Inspect this as a Bit Array."
  :New-Inspect-Function
    (allocate-data 'show-bit-array x)
)

(Defperspective :bit-array-sideways (x show-x)
  :show-x-type-for-perspective Show-Bit-Array-sideways
  :This-Perspective-Applicable-Function
    (and (arrayp x) (equal (array-rank x) 2)
	 (member (array-type x) '(art-1b art-2b art-4b))
	 (not (typep show-x 'show-bit-array-sideways))
    )
  :menu-item-name "Bit Array on its side"
  :Menu-Who-Line-Doc-String
    "Inspect this as a Bit Array displayed on its side."
  :New-Inspect-Function
    (allocate-data 'show-bit-array-sideways x)
)

(Defperspective :resource (x show-x)
  :show-x-type-for-perspective Show-Generic-Object-Thing
  :This-Perspective-Applicable-Function
    (and (symbolp x) (get x 'defresource))
  :menu-item-name "Resource"
  :Menu-Who-Line-Doc-String
    "Inspect this as the resource named by this symbol."
  :New-Inspect-Function
    (allocate-data 'Show-Generic-Object-Thing (get x 'defresource))
)

(defvar *types-to-exclude-from-generic-display*
  '(compiled-function-p
    hash-table
    method-function
    clos:class
    si:flavor
    si:hash-table
    locative
   )
)

(defvar *types-to-exclude-from-generic-defstruct-display*
  '(compiled-function-p
    hash-table
    method-function
    clos:class
    si:flavor
    si:hash-table
    locative
    sys:named-structure
    tv:stack-frame
   )
)

(Defperspective :Defstruct (x show-x)
  :show-x-type-for-perspective Show-generic-defstruct
  :This-Perspective-Applicable-Function
    (and (not (typep show-x 'show-generic-defstruct))
	 (named-structure-p x)
	 (not (member x *Types-To-Exclude-From-Generic-Defstruct-Display*
		      :Test #'(lambda (a b)
				(if (type-specifier-p b)
				    (typep a b)
				    (funcall b a)
				)
			      )
	      )
	 )
    )
  :menu-item-name "Defstruct"
  :New-Inspect-Function (allocate-data 'Show-generic-defstruct x)
)

(Defperspective :generic-data-structure (x show-x)
  :show-x-type-for-perspective Show-Generic-Object-Thing
  :already-this-type-function (typep show-x 'Show-Generic-Object-Thing)
  :This-Perspective-Applicable-Function
    (and (not (typep show-x 'Show-Generic-Object-Thing))
	 (not (member x *Types-To-Exclude-From-Generic-Display*
		      :Test #'(lambda (a b)
				(if (type-specifier-p b)
				    (typep a b)
				    (funcall b a)
				)
			      )
	      )
	 )
    )
  :Menu-item-name "Simply inspect"
  :New-Inspect-Function
    (allocate-data 'Show-Generic-Object-Thing x)
  :Priority 10
)

(Defperspective :defstruct-description (x show-x)
  :show-x-type-for-perspective Show-List-Unnamed-Structure
  :This-Perspective-Applicable-Function
    (and (symbolp x) (get x 'si:defstruct-description))
  :menu-item-name "Defstruct Description"
  :Menu-Who-Line-Doc-String
    "Inspect this as the defstruct description named by this symbol."
  :New-Inspect-Function
    (allocate-data 'show-list-unnamed-structure
		   (get x 'si:defstruct-description)
		   'si:defstruct-description
    )
)

(Defperspective :method-details (x show-x)
  :show-x-type-for-perspective show-method-details
  :This-Perspective-Applicable-Function
    (and (typep x 'Method-function)
	 (not (typep show-x 'show-method-details))
	 (not (typep show-x 'show-clos-method-details))
    )
  :menu-item-name "Method Details"
  :Menu-Who-Line-Doc-String
    "Inspect this method, showing interesting information about it, e.g. IVs referenced, specials bound etc.."
  :New-Inspect-Function
    (if (equal :method (first (function-name x)))
        (apply #'allocate-data 'show-method-details (Data-From-Method x))
	(apply #'allocate-data 'show-clos-method-details
	       (data-from-clos-method
		 (method-from-method-function-safe x)
	       )
	)
    )
  :Priority 1
)

(Defperspective :disassembled-function (x show-x)
  :show-x-type-for-perspective Show-Generic-Object-Thing
  :This-Perspective-Applicable-Function
    (or (and (typep x 'compiled-function)
	     (not (typep show-x 'Show-Generic-Object-Thing))
	)
	(and (symbolp x) (compiled-function-p (fdefinition-safe x)))
    )
  :menu-item-name "Disassembled Function"
  :New-Inspect-Function
    (allocate-data 'Show-Generic-Object-Thing
		   (if (symbolp x) (fdefinition-safe x) x)
    )
)

(Defperspective :function-details (x show-x)
  :already-this-type-function (typep show-x 'Show-function)
  :show-x-type-for-perspective show-function
  :This-Perspective-Applicable-Function
    (or (and (typep x 'compiled-function)
	     (not (fef-of-gf-p x))
	     (not (typep x 'method-function))
	)
	(and (symbolp x) (fboundp x)
	     (not (fef-of-gf-p (fdefinition-safe x t)))
	)
    )
  :menu-item-name "Function details"
  :Menu-Who-Line-Doc-String
    "Inspect this function, showing interesting information about it, e.g. specials bound, functions called etc.."
  :New-Inspect-Function
    (allocate-data 'Show-function (if (symbolp x) (fdefinition-safe x) x))
)

(Defperspective :generic-function-details (x show-x)
  :already-this-type-function
    (typep show-x 'show-clos-generic-function-details)
  :show-x-type-for-perspective show-clos-generic-function-details
  :This-Perspective-Applicable-Function
    (or (and (typep x 'compiled-function) (fef-of-gf-p x))
	(and (symbolp x) (fboundp x)
	     (fef-of-gf-p (fdefinition-safe x t))
	)
	(typep x 'any-type-of-clos-gf)
    )
  :menu-item-name "Generic Function details."
  :Menu-Who-Line-Doc-String
    "Inspect this generic function, showing interesting information about it, e.g. specials bound, functions called etc.."
  :New-Inspect-Function
    (let ((gf (if (symbolp x)
		  (function-generic-function-safe (fdefinition-safe x t))
		  (if (typep x 'any-type-of-clos-gf)
		      x
		      (function-generic-function-safe x)
		  )
	      )
	  )
	 )
         (allocate-data 'show-clos-generic-function-details gf gf)
    )
  :Prefer-Over (:Generic-Data-Structure :disassembled-function)
  :Priority (if (symbolp x) 9 11)
)

(Defperspective :clos-method-details (x show-x)
  :show-x-type-for-perspective show-clos-method-details
  :This-Perspective-Applicable-Function
    (and (typep x 'any-type-of-clos-method)
	 (not (typep show-x 'show-clos-method-details))
    )
  :menu-item-name "Method Details"
  :Menu-Who-Line-Doc-String
    "Inspect this method, showing interesting information about it, e.g. IVs referenced, specials bound etc.."
  :New-Inspect-Function
    (apply #'allocate-data 'show-method-details (Data-From-Clos-Method x))
)

(Defperspective :clos-method-function (x show-x)
  :show-x-type-for-perspective Show-Generic-Object-Thing
  :This-Perspective-Applicable-Function (typep x 'any-type-of-clos-method)
  :menu-item-name "Method function"
  :Menu-Who-Line-Doc-String
    "Inspect the method function of this method."
  :New-Inspect-Function
    (allocate-data 'Show-Generic-Object-Thing (method-function-safe x))
)

(Defperspective :generic-function-fef (x show-x)
  :show-x-type-for-perspective Show-Generic-Object-Thing
  :This-Perspective-Applicable-Function (typep x 'any-type-of-clos-gf)
  :menu-item-name "Generic Function disassembled"
  :Menu-Who-Line-Doc-String
    "Inspect the generic function in disassembled form."
  :New-Inspect-Function
    (allocate-data 'Show-Generic-Object-Thing (get-fef-from-object x))
)

;-------------------------------------------------------------------------------

(defun perspective-greaterp (a b x show-x)
  (or (member (send b :Name) (send a :Prefer-Over))
      (> (let ((pr (send a :Priority)))
	      (if (functionp pr)
		  (funcall pr x show-x)
		  pr
	      )
	 )
	 (let ((pr (send b :Priority)))
	      (if (functionp pr)
		  (funcall pr x show-x)
		  pr
	      )
	 )
      )
  )
)

(defun perspective-equalp (a b x show-x)
  (= (let ((pr (send a :Priority)))
	  (if (functionp pr)
	      (funcall pr x show-x)
	      pr
	  )
     )
     (let ((pr (send b :Priority)))
	  (if (functionp pr)
	      (funcall pr x show-x)
	      pr
	  )
     )
  )
)

(defun perspective-really-greater-p (a b x show-x)
  (or (perspective-greaterp a b x show-x)
      (and (not (perspective-greaterp b a x show-x))
	   (not (perspective-equalp a b x show-x))
      )
  )
)

(defun sort-perspectives (list x show-x)
"Given a list of perspectives, sorts them into priority order.  Priority is
 assessed on the basis first of the prefer-over slot
 and then the priority.
"
  (stable-sort
    list #'(lambda (a b) (Perspective-Really-Greater-P a b x show-x)) 
  )
)

(defun get-perspectives-for-inspection-data (something)
"Given a thing, which might be an instance of Inspection-data, returns a list of
 all the perspectives that apply to it, sorted into order of highest priority
 first.
"
  (let ((cache (send something :Perspective-Cache)))
       (if (equal :Unbound cache)
	   (let ((value (or (send something :middle-button-result)
			    something
			)
		 )
		 (show-x something)
		)
		(let ((sorted
			(sort-perspectives
			  (loop for name in *All-Perspective-Names*
				when (send (get name :Perspective)
					   :Applicable-P value show-x
				     )
				collect (get name :Perspective)
			  )
			  something show-x
			)
		      )
		     )
		     (send something :Set-Perspective-Cache sorted)
		     sorted
		)
	   )
	   cache
       )
  )
)

(defvar *perspective-cache-table* (make-hash-table :Test #'eq))

(defun get-perspectives (something)
"Given a thing, which might be an instance of Inspection-data, returns a list of
 all the perspectives that apply to it, sorted into order of highest priority
 first.
"
  (if (typep something 'inspection-data)
      (get-perspectives-for-inspection-data something)
      (let ((entry (gethash something *Perspective-Cache-Table* :not-there)))
	   (if (listp entry)
	       entry
	       (Let ((sorted
		       (sort-perspectives
			 (loop for name in *All-Perspective-Names*
			       for perspective = (get name :Perspective)
			       When (and perspective
					 (send perspective
					       :Applicable-P something nil
					 )
				    )
			       collect (get name :Perspective)
			 )
			 something nil
		       )
		     )
		    )
		    (setf (gethash something *Perspective-Cache-Table*) sorted)
		    sorted
	       )
	   )
      )
  )
)

(defun map-into-show-x (something &optional (no-menu-p nil) (only-of-type nil))
"Takes Something and maps it into something else to inspect.  If there is more
 than one available perspective it pops up a menu of applicable perspectives.
"
  (let ((value (if (typep something 'inspection-data)
		   (or (send something :middle-button-result)
		       something
		   )
		   something
	       )
        )
	(show-x (if (typep something 'inspection-data) something nil))
       )
       (let ((unfiltered-entries (Get-Perspectives something)))
	    (let ((entries
		    (if only-of-type
			(remove-if-not #'(lambda (x) (typep x only-of-type))
				       unfiltered-entries
			)
			unfiltered-entries
		    )
		  )
		 )
		 (if entries
		     (let ((selected
			     (if (and (not no-menu-p) (rest entries))
				 ;;; More than one.
				 (w:menu-choose
				   (mapcar
				     #'(lambda (x) (send x :Menu-Item x show-x))
				     entries
				   )
				   :Label "Which perspective?"
				   :Columns
				 )
				 (first entries)
			     )
			   )
			  )
			  (if selected
			      (send selected :Get-Value-To-Inspect value show-x)
			      nil
			  )
		     )
		     nil
		 )
	    )
       )
  )
)

(defun inspect (&optional (object nil objp)
		(in-perspective nil)
		(inspector-type 'General-Inspector))
  "Call the Inspector to inspect OBJECT.  Selects an Inspector window.
   If in-perspective is the name of a perspective that is applicable to Object
   then this perspective is used.  If In-perspective is :menu then a menu of
   perspectives will be popped up if there are more than one.  If
   in-perspective is :generic, then generic inspection will be used
   (no fancy flavor inspection or anything like that).
   The Inspector runs in its own process, so your special variable bindings
   will not be visible.
   If you want to see special variable bindings, use INSPECT*."
  (declare (special frame))
  (let ((iframe ;;; Modded here by JPR.
		(find-or-create-inspect-window
		  inspector-type
		  (if (and (boundp 'frame) (typep frame inspector-type))
		      frame
		      nil
		  )
		)
	)
        (top-item nil)
	(inspect*-quit nil)
	(*print-pretty* nil)
	(*PRINT-ARRAY* *print-array*)
	(*print-circle* *print-circle*)
	(*print-radix* *print-radix*)
	(*nopoint *nopoint)
	(*print-base* *print-base*)
	(*read-base* *read-base*)
	(*print-level* *print-level*)
	(*print-length* *print-length*))
    (declare (special top-item = inspect*-quit *print-pretty*))
    (declare (special *print-array* *print-circle* *print-radix* *nopoint
		      *print-base* *read-base* *print-level* *print-length*))
   
    (setq = nil)
    ;(send iframe :prepare-for-use object objp)))
    (if objp
	(Send iframe :inspect-object object in-perspective))))


(defun inspect* (&optional (object nil) (in-perspective nil) (frame nil)
		 (resource 'general-inspect-frame-resource)
		)
  "Call the Inspector to inspect OBJECT.  Selects an Inspector window.
   If in-perspective is the name of a perspective that is applicable to Object
   then this perspective is used.  If In-perspective is :menu then a menu of
   perspectives will be popped up if there are more than one.  If
   in-perspective is :generic, then generic inspection will be used
   (no fancy flavor inspection or anything like that).
   The Inspector runs in the calling process, so our special variable bindings
   are visible.
   If you type END in the inspector, the value of = will be returned from the 
   function INSPECT."
  (with-stack-list (env si:*interpreter-environment*
			si:*interpreter-function-environment*
			nil)
    (let ((top-item nil)				   ;RDA: added
	  (inspect*-quit t)
	  (*print-pretty* nil)
	  (*PRINT-ARRAY* *print-array*)
	  (*print-circle* *print-circle*)
	  (*print-radix* *print-radix*)
	  (*nopoint *nopoint)
	  (*print-base* *print-base*)
	  (*read-base* *read-base*)
	  (*print-level* *print-level*)
	  (*print-length* *print-length*)
	  )
      (declare (special top-item inspect*-quit *print-pretty*))
      (setq ucl:*env* env)
      (flet ((body (iframe)
	      (send iframe :inspect-object object in-perspective)
	      (window-call-with-selection-substitute (iframe :deactivate)
		(inspect-command-loop iframe)
	      )
	     )
	    )
            (if frame
		(body frame)
	        (eval `(using-resource (iframe ,resource default-screen)
			 (funcall ,#'body iframe)
		       )
		)
	    )
      )
    )
  )
)

(defun inspect-flavor (&optional (object nil objp)
		       (inspector-type 'General-Inspector))
  "Call the Inspector to inspect OBJECT.  Selects an Inspector window.
   Inspects Object as if it was a flavor using the  :Flavor-As-Flavor-Inspect
   perspective.
   The Inspector runs in its own process, so your special variable bindings will not be visible.
   If you type END or use the exit menu option the original object  will
   be returned.
   This function is obsolete now.  You should use
   (Inspect object :perspective...)"
  (if objp
      (Inspect object :Flavor inspector-type)
      (Inspect nil nil inspector-type)
  )
)


(defmethod (basic-general-inspector :inspect-object)
	   (object &optional (perspective nil))
"Inspects an object.  Looks at the perspective arg and does the right hing.
 Perspective can be any of :Menu (use a menu if appropriate),
 :generic (use generic inspection), nil (pick the best perspective [default])
 or the name of a perspective to use.
"
  (multiple-value-bind (show-x inspect-p)
      (case perspective
	(:Menu (map-into-show-x object))
	(:generic (values (allocate-data 'show-generic-object-thing object) t))
	(nil (map-into-show-x object t))
	(otherwise
	 (if (member perspective *All-Perspective-Names*)
	     (let ((object (if (typep object 'inspection-data)
			       (send object :Middle-Button-Result)
			       object
			   )
		   )
		   (show-x (if (typep object 'inspection-data) object nil))
		  )
	          (if (send (get perspective :Perspective) :Applicable-P
			    object show-x
		      )
		      (send (get perspective :Perspective)
			    :Get-Value-To-Inspect object show-x
		      )
		      (progn (cerror "Just inspect it anyway"
			       "Perspective ~S is not applicable to ~S."
			       perspective object
			     )
			     (Map-into-show-x object t)
		      )
		  )
	     )
	     (progn (cerror "Just inspect it anyway"
			    "~S is not the name of a perspective." perspective
		    )
		    (Map-into-show-x object t)
	     )
	 )
	)
      )
    (if inspect-p
	(let ((thing (inspect-real-value `(:value ,show-x ,history))))
	     (inspect-flush-from-history thing history)
	     (send history :append-item thing)
	     (update-panes)
	)
	nil
    )
  )
)

(defmethod (general-inspector-history-window :inspect-object)
           (object inspector &optional top-item-no -label- dont-propogate
	    from-window-debugger?)
  ;;First, remember current TOP-ITEM of inspector
  (multiple-value-bind (object inspect-p)
      (if (typep object 'inspection-data)
	  (values object t)
	  (map-into-show-x object t)
      )
    (if inspect-p
        (let ((disp (send inspector :current-display)))
	  (and disp (setf (fourth disp) (send inspector :top-item)))
	  (or
	    (dotimes (i (array-active-length items))
	      (cond
		((neq object (aref items i)))
		(dont-propogate (return t))
		(T (send self :delete-item i) (return nil))))
	    (send self :append-item object))
	  (send self :put-item-in-window object)
	  (let ((ce (cli:assoc object cache :test #'eq)))
	    (if from-window-debugger?	;..
		(push (setq ce (Inspect-Setup-Object-Display-List
				 object inspector top-item-no -label-)) cache)
		(or ce
		    (push (setq ce (Inspect-Setup-Object-Display-List
				     object inspector top-item-no -label-))
			  cache)))
	    (or ;(eq (cdr ce) disp) ;;;; JPR.
		(send inspector :setup-object ce))
	    ))
	nil)))


(defmethod (basic-general-inspector :inspect-info-left-2-click)
	   (&optional something)
"This is the generic left-2 click method that knows how to support the
 perspectives mechanism and inspect the thing that was selected.
"
  (let ((thing (if something
		   something
		   (inspect-real-value ucl:kbd-input)
	       )
	)
       )
       (multiple-value-bind (real-thing inspect-p)
	   (map-into-show-x thing)
	  (if inspect-p
	      (progn (inspect-flush-from-history real-thing history)
		     (send history :append-item real-thing)
		     (send history :set-cache nil)
		     (update-panes)
	      )
	      nil
	  )
       )
  )
)


(defun general-inspector-print-values? ()
  (declare (:self-flavor tv:basic-inspect-frame))
  (declare (special ucl:prompt ucl:input-mechanism history))
  (and (ucl:abnormal-command?)
       (if (string-equal ucl:prompt "> ")
           (progn
	     ;;; I see no reason why it should update panes here, even though
	     ;;; the patch to the inspector does.
	     ;;; The user can always hit refresh if he has typed something that
	     ;;; might affect something on the screen.
;            (update-panes)
             t
	   )
	   (unless (eq ucl:input-mechanism 'ucl:unknown)
	     (multiple-value-bind (thing inspect-p)
		 (map-into-show-x (car \/) t)
	       (if inspect-p
		   (let ((thing (inspect-real-value `(:value ,thing ,history))))
			(inspect-flush-from-history thing history)
			(send history :append-item thing)
			(update-panes)
			nil
		   )
		   nil
	       )
	     )
	   )
       )
  )
)


(defun perspective-doc-string-addition (thing perspectives existing-docs)
"Is passed the thing that the mouse is over, a list of perspectives for it
 and the rest of the doc string.  It modifies the doc string if there are
 any applicable perspectives so that a) it mentions #\mouse-l-2 and b)
 it enumerates the names of the applicable perspectives.
"
  (if (or (and (not (typep thing 'inspection-data))
	       (rest perspectives)
	  )
	  (and (typep thing 'inspection-data) perspectives)
      )
      (append (list (first existing-docs) (second existing-docs)
		    :mouse-l-2 "View another perspective"
		    :Documentation
		     (let ((names (mapcar #'(lambda (x) (send x :Name))
					  perspectives
				  )
			   )
			   (*print-case* :Capitalize)
			  )
			  (format nil "[~S~{, ~S~}]" (first names)
				  (rest names)
			  )
		     )
	      )
	      (rest (rest existing-docs))
      )
      existing-docs
  )
)

(defwrapper (inspection-data :who-line-doc)
	    (ignore &body body)
"Supports the mouse-l-2 click as well as the others."
  `(let ((result . ,body))
        (Perspective-Doc-String-Addition
	  self
	  (get-perspectives self) result
	)
   )
)

(defun allocated-perspectives (of)
  (if (typep of 'inspection-data)
      (Get-Perspectives of)
      (Get-Perspectives ; (allocate-data 'show-generic-object-thing of)
        (map-into-show-x of t)
      )
  )
)

(defmethod (basic-inspect :get-normal-mouse-documentation) ()
"Knows how to get the mouse doc string for inspect panes.  If the object that
 we're over has perspectives then these are mentioned in the mouse docs.
"
  (let ((item (if sensitive-inspect-item (get-mouse-sensitive-item) nil)))
       (multiple-value-bind (perspectives thing)
	   (if (and (consp item) (third item)
		    (equal :item1 (first item))
	       )
	       (if (allocated-perspectives (third item))
		   (values (allocated-perspectives (third item)) (third item))
		   (values nil (third item))
	       )
	       (if (and item (allocated-perspectives item))
		   (values (allocated-perspectives item) item)
		   (if (and (equal print-function-arg :list-structure)
			    (locativep item)
			    (%p-contents-safe-p item)
			    (allocated-perspectives (first item))
		       )
		       (values (allocated-perspectives (first item))
			       (first item)
		       )
		       nil
		   )
	       )
	   )
           (ignore perspectives)
           (append (send (if (typep thing 'inspection-data)
                             thing
                             (map-into-show-x thing t)
                         )
                         :who-line-doc
                         t (not thing)
                   )
		   '(:Allow-Override "")
                   normal-mouse-documentation
           )
       )
  )
)

(defmethod (general-inspector-history-window :get-normal-mouse-documentation) ()
"Knows how to get the mouse doc string for inspector histor panes.  If the
 object that we're over has perspectives then these are mentioned in the
 mouse docs.
"
  (let ((item (if sensitive-history-item (get-mouse-sensitive-item) nil)))
       (Perspective-Doc-String-Addition item (and item (Get-Perspectives item))
					normal-mouse-documentation
       )
  )
)

(defmethod (show-flavor :Middle-Button-Result) ()
"Just returns the data slot."
  data
)

(defmethod (show-function :Middle-Button-Result) ()
"Just returns the data slot."
  data
)

(defmethod (show-method :Middle-Button-Result) ()
"Returns the method we represent."
  (send self :Method-From-Show-Method)
)

(defmethod (show-generic-defstruct :Middle-Button-Result) ()
"Just returns the data slot."
  data
)

(defmethod (list-inspection-mixin :middle-button-result) ()
"Just returns the data slot."
  data
)

(defmethod (show-related-methods :middle-button-result) ()
"Just returns the data slot."
  data
)

(defmethod (show-method-call-tree :middle-button-result) ()
"Just returns the data slot."
  data
)

(defmethod (debug-flavor :Middle-Button-Result) ()
  data
)

(defmethod (show-undefined-flavor :Middle-Button-Result) ()
  data
)

(defmethod (show-value :Middle-Button-Result) ()
  data
)

;(defmethod (General-Inspector-History-Window :Before :Append-Item) (item)
;"This is just for debugging purposes.  It should not be called now."
;  (if (not (typep item 'inspection-data))
;      (dbg)
;  )
;)

(DEFMETHOD (BASIC-INSPECT-FRAME :inspect-info-left-click) ()  ;fi
  (LET ((thing (inspect-real-value ucl:kbd-input)))
    ;; First flush item we will be inspecting
    (multiple-value-bind (thing inspect-p)
	(if (typep thing 'inspection-data)
	    (values thing t)
	    (map-into-show-x thing t)
	)
      (if inspect-p
	  (progn (inspect-flush-from-history thing history)
		 (SEND history :APPEND-ITEM thing)
		 ;;; Modded here by JPR.
		 (if *flush-cache-if-left-button-on-something*
		     (if (equal *flush-cache-if-left-button-on-something*
				:Really
			 )
			 (send history :set-cache nil)
			 (send history :flush-object-from-cache thing)))
		 (update-panes))))))


(defmethod (inspection-data :Print-Self) (stream &rest ignore)
"Just a simple print method for inspection datas, so that we can see what
 we're inspecting.
"
  (if *Inhibit-Inspection-Data*
      (catch-error (format stream "#<~S ~S>"
			   (type-of self)
			   (send self :Send-If-Handles :Middle-Button-Result)
		   )
		   nil
      )
      (send self :Format-Concisely stream)
  )
)

;;; Patch
;;; TI code.

(DEFUN INSPECT-SET-SLOT (SLOT *TERMINAL-IO* HISTORY INSPECTORS)
  "Set the contents of SLOT to a value we obtain with the mouse or by reading.
SLOT is a blip produced by clicking on a mouse-sensitive item.
HISTORY should be the INSPECT-HISTORY-WINDOW;
we tell it to forget cached data on the slot."
  (LET ((SET-FUNCTION (GET (FIRST SLOT) 'SET-FUNCTION)))
    (SEND *TERMINAL-IO* :CLEAR-SCREEN)
    (FORMAT *TERMINAL-IO* "~&New value to set with:")
    (MULTIPLE-VALUE-BIND (NEW-VALUE PUNT-P)
	(INSPECT-GET-VALUE-FROM-USER *TERMINAL-IO* HISTORY INSPECTORS)
      (OR PUNT-P
	  (SEND SET-FUNCTION SLOT NEW-VALUE
		(if (typep (SEND (THIRD SLOT) :CURRENT-OBJECT) 'inspection-data)
		    (send (SEND (THIRD SLOT) :CURRENT-OBJECT) :Data)
		    (SEND (THIRD SLOT) :CURRENT-OBJECT)))))
    ;; We must recompute object we modified
    (SEND HISTORY :FLUSH-OBJECT-FROM-CACHE (SEND (THIRD SLOT) :CURRENT-OBJECT))
    (PROG1
      (SEND (THIRD SLOT) :TOP-ITEM)
      (SEND (THIRD SLOT) :SET-CURRENT-OBJECT (LIST NIL)))))



(DEFCOMMAND DELETE-ALL-CMD NIL			
            '(:DESCRIPTION  "Delete all inspected objects from history and inspection panes."
              :NAMES ("Delete") :KEYS (#\c-PAGE))
            (DECLARE (SPECIAL HISTORY inspectors))
            (SEND HISTORY :FLUSH-CONTENTS)
            (LOOP for iw in inspectors
                  do (SEND iw :set-locked-p nil))
	    ;;; These lines added by JPR.  We don't want any old inspection
	    ;;; datas or perspectives if we delete all.
	    ;;; Mind you, *inspection-data* should really
	    ;;; be an IV of the frame.
	    (setq *inspection-data* nil)
	    (clrhash *Perspective-Cache-Table*)
            (Update-PANES))


(defmethod (flavor-inspector :disassemble-clos-method) (method)
"Given a method, inspects its disassembled code."
  (inspect method :clos-method-function)
)

(defmethod (flavor-inspector :disassemble-generic-function)
	   (generic-function  class)
"Given a generic function disassembles it."
  (ignore class)
  (inspect generic-function :generic-function-fef)
)

;-------------------------------------------------------------------------------

;;; The following fixes extend the behaviour of who-line docs.
;;; there were a number of major restrictions:

;;; a) It expected to have everything in the right order order.  This is not the
;;;    case if mouse docs are added by whoppers and such.
;;; b) It assumed that you would only want line breaks if you were using the
;;;    no-comma option, which is bogus
;;; c) It failed to compute a reasonable place for a line break if you wanted
;;;    it to.
;;; d) It failed to allow you to override mouse docs.  Thus if you had a wrapper
;;;    that added mouse docs you had to go through the old list and remove any
;;;    mouse docs with the same name (and get it all back in the right order.

;;; Patch from WhoLin
(defvar *newline-thrown* nil)
(defvar *non-printing-command* nil)

;;; TI code.
(DEFUN DISPLAY-WHO-LINE-MOUSE-INFO (WHO-SHEET MOUSE-KEYWORD DOCUMENTATION COMMA)
  "Display mouse information in the who line documentation window."
  ;; Do the things which need to be done before writing out the documentation string.
  (LET ((PAGE-OVERFLOW-ENCOUNTERED T)) 
    (COND (STRING-TOGETHER
           ;; If we are stringing everything together and this is not the first line
           ;; then we need to output a comma to separate this string from the previous string.
	   (if *Newline-Thrown*
	       ;;; Patched here by JPR.
	       (Setq *Newline-Thrown* nil)
	       (if *Non-Printing-Command*
		   (setq *Non-Printing-Command* Nil)
		   (When (and NOT-FIRST? (not comma)
			      (not (get mouse-keyword 'non-printing-mouse-keyword)))
		     ;; if not-first? is T and the value of  :no-comma is nil, 
		     (SHEET-STRING-OUT WHO-SHEET ", ")))))   ;; then we output a comma.
          (T
           ;; If we are formatting the lines, then we need to position the cursor to the correct place.
           (APPLY 'SHEET-SET-CURSORPOS WHO-SHEET
                  (CASE MOUSE-KEYWORD
                        ((:MOUSE-ANY                 ) `(,LEFT-CLICK-LOC   ,MOUSE-SINGLE-LOC))
                        ((:MOUSE-1-1    :MOUSE-L-1   ) `(,LEFT-CLICK-LOC   ,MOUSE-SINGLE-LOC))
                        ((:MOUSE-1-2    :MOUSE-L-2   ) `(,LEFT-CLICK-LOC   ,MOUSE-DOUBLE-LOC))
			((:Meta-Mouse-L              ) `(,LEFT-CLICK-LOC   ,MOUSE-DOUBLE-LOC))
                        ((:MOUSE-1-HOLD :MOUSE-L-HOLD) `(,LEFT-CLICK-LOC   ,MOUSE-HOLD-LOC  ))
                        ((:MOUSE-2-1    :MOUSE-M-1   ) `(,MIDDLE-CLICK-LOC ,MOUSE-SINGLE-LOC))
                        ((:MOUSE-2-2    :MOUSE-M-2   ) `(,MIDDLE-CLICK-LOC ,MOUSE-DOUBLE-LOC))
			((:Meta-Mouse-M              ) `(,MIDDLE-CLICK-LOC ,MOUSE-DOUBLE-LOC))
                        ((:MOUSE-2-HOLD :MOUSE-M-HOLD) `(,MIDDLE-CLICK-LOC ,MOUSE-HOLD-LOC  ))
                        ((:MOUSE-3-1    :MOUSE-R-1   ) `(,RIGHT-CLICK-LOC  ,MOUSE-SINGLE-LOC))
                        ((:MOUSE-3-2    :MOUSE-R-2   ) `(,RIGHT-CLICK-LOC  ,MOUSE-DOUBLE-LOC))
			((:Meta-Mouse-R              ) `(,RIGHT-CLICK-LOC  ,MOUSE-DOUBLE-LOC))
                        ((:MOUSE-3-HOLD :MOUSE-R-HOLD) `(,RIGHT-CLICK-LOC  ,MOUSE-HOLD-LOC  ))))))
    ;; We change the font for the mouse prefix to distinguish the prefix from the mouse documentation.
    (SEND WHO-SHEET :SET-CURRENT-FONT *MOUSE-DOCUMENTATION-LINE-BUTTONS-STANDARD-FONT* T)
    ;;; Patched by JPR.
    (if (equal :Newline mouse-keyword)
        (progn (terpri who-sheet)
	        (setq *Newline-Thrown* t))
	(if (get mouse-keyword 'non-printing-mouse-keyword)
	    (setq *Non-Printing-Command* T) ;;; do nothing.
	    (Catch 'PAGE-OVERFLOW
	      (SHEET-STRING-OUT WHO-SHEET
				(OR (CADR (IF (EQ MOUSE-HANDEDNESS :LEFT)
					      (ASSOC MOUSE-KEYWORD
						     '((:MOUSE-ANY "L,M,R") (:ANY "L,M,R")
						       (:MOUSE-R-1 "L")   (:MOUSE-R-2 "L2")   (:MOUSE-R-HOLD "LH")
						       (:MOUSE-3-1 "L")   (:MOUSE-3-2 "L2")   (:Meta-Mouse-L "Meta-L")
						       (:MOUSE-M-1 "M")   (:MOUSE-M-2 "M2")   (:MOUSE-M-HOLD "MH")
						       (:MOUSE-2-1 "M")   (:MOUSE-2-2 "M2")   (:Meta-Mouse-M "Meta-M")
						       (:MOUSE-L-1 "R")   (:MOUSE-L-2 "R2")   (:MOUSE-L-HOLD "RH")
						       (:MOUSE-1-1 "R")   (:MOUSE-1-2 "R2")   (:Meta-Mouse-R "Meta-R")) :TEST #'EQ)
					      ;;ELSE
					      (ASSOC MOUSE-KEYWORD
						     '((:MOUSE-ANY "L,M,R") (:ANY "L,M,R")
						       (:MOUSE-R-1 "R")   (:MOUSE-R-2 "R2")   (:MOUSE-R-HOLD "RH")
						       (:MOUSE-3-1 "R")   (:MOUSE-3-2 "R2")   (:Meta-Mouse-R "Meta-R")
						       (:MOUSE-M-1 "M")   (:MOUSE-M-2 "M2")   (:MOUSE-M-HOLD "MH")
						       (:MOUSE-2-1 "M")   (:MOUSE-2-2 "M2")   (:Meta-Mouse-M "Meta-M")
						       (:MOUSE-L-1 "L")   (:MOUSE-L-2 "L2")   (:MOUSE-L-HOLD "LH")
						       (:MOUSE-1-1 "L")   (:MOUSE-1-2 "L2")   (:Meta-Mouse-L "Meta-L")) :TEST #'EQ)))
				    ;; If the caller specified an illegal mouse button
				    ;; then use the following string as the mouse prefix.
				    (format nil "Bad doc keyword ~S" mouse-keyword)))
	      (SHEET-STRING-OUT WHO-SHEET ": " 0
				(IF (STRING-EQUAL "" DOCUMENTATION)
				    ;; If the documentation for this button is empty then we do
				    ;; not want to have the space after the mouse prefix.  In
				    ;; this case there are two mouse buttons which do the same
				    ;; thing.  The next mouse button will have the documentation
				    ;; for this mouse button.  See the EDIT SCREEN menu item of
				    ;; the System Menu for an example of this.
				    ;; may 9-9-88 NOTE:
				    ;; Above reference to EDIT-SCREEN is/was no longer true.
				    ;; No doc on this "feature" exists and it seems to be
				    ;; pretty much worthless.
				    1
				    ;;ELSE
				    NIL))
	      (SEND WHO-SHEET :SET-CURRENT-FONT NEW-WHO-LINE-FONT T)
	      (SHEET-STRING-OUT WHO-SHEET DOCUMENTATION)
	      (SETQ PAGE-OVERFLOW-ENCOUNTERED NIL))))

    (IF PAGE-OVERFLOW-ENCOUNTERED
      (SETQ MAXIMUM-WHO-LINE-MOUSE-X (SHEET-INSIDE-WIDTH WHO-SHEET))
      ;;ELSE
      (WHEN (>= (SHEET-CURSOR-Y WHO-SHEET) MAXIMUM-WHO-LINE-MOUSE-Y)
	(SETQ MAXIMUM-WHO-LINE-MOUSE-Y (SHEET-CURSOR-Y WHO-SHEET)
	      MAXIMUM-WHO-LINE-MOUSE-X (SHEET-CURSOR-X WHO-SHEET))))))

(DEFUN DISPLAY-WHO-LINE-MOUSE-INFO-MAC ( MOUSE-KEYWORD DOCUMENTATION COMMA)
  "Display mouse information in the mac who line documentation window."
  ;; Do the things which need to be done before writing out the documentation string.
  (declare (special current-doc-line))
  (LET ((PAGE-OVERFLOW-ENCOUNTERED T)) 
    (COND (STRING-TOGETHER
           ;; If we are stringing everything together and this is not the first line
           ;; then we need to output a comma to separate this string from the previous string.
           (if *Newline-Thrown*
	       ;;; Patched here by JPR.
	       (Setq *Newline-Thrown* nil)
	       (if *Non-Printing-Command*
		   (setq *Non-Printing-Command* Nil)
		   (When (and NOT-FIRST? (not comma)
			      (not (get mouse-keyword 'non-printing-mouse-keyword)))
		     ;; if not-first? is T and the value of  :no-comma is nil, 
		     (SETF current-doc-line (STRING-APPEND current-doc-line ", "))))))
	  ;; then we output a comma.
          (T t))
    (if (equal :Newline mouse-keyword)
        (progn (setq current-doc-line (string-append current-doc-line #\space))
	       ;;; the space should really be a newline but the mx who line stuff is
	       ;;; just too broken for it.
	        (setq *Newline-Thrown* t))
	(if (get mouse-keyword 'non-printing-mouse-keyword)
	    (setq *Non-Printing-Command* T) ;;; do nothing.
    (CATCH 'PAGE-OVERFLOW
      (SETF current-doc-line (STRING-APPEND current-doc-line 
			       (OR (CADR (IF (EQ MOUSE-HANDEDNESS :LEFT)
					     (ASSOC MOUSE-KEYWORD
						    '((:MOUSE-ANY "L,M,R") (:ANY "L,M,R")
						      (:MOUSE-R-1 "L")   (:MOUSE-R-2 "L2")   (:MOUSE-R-HOLD "LH")
						      (:MOUSE-3-1 "L")   (:MOUSE-3-2 "L2"    (:Meta-Mouse-L "Meta-L"))
						      (:MOUSE-M-1 "M")   (:MOUSE-M-2 "M2")   (:MOUSE-M-HOLD "MH")
						      (:MOUSE-2-1 "M")   (:MOUSE-2-2 "M2")   (:Meta-Mouse-M "Meta-M")
						      (:MOUSE-L-1 "R")   (:MOUSE-L-2 "R2")   (:MOUSE-L-HOLD "RH")
						      (:MOUSE-1-1 "R")   (:MOUSE-1-2 "R2")   (:Meta-Mouse-R "Meta-R")) :TEST #'EQ)
					     ;;ELSE
					      (ASSOC MOUSE-KEYWORD
						    '((:MOUSE-ANY "L,M,R") (:ANY "L,M,R")
						      (:MOUSE-R-1 "R")   (:MOUSE-R-2 "R2")   (:MOUSE-R-HOLD "RH")
						      (:MOUSE-3-1 "R")   (:MOUSE-3-2 "R2")   (:Meta-Mouse-R "Meta-R")
						      (:MOUSE-M-1 "M")   (:MOUSE-M-2 "M2")   (:MOUSE-M-HOLD "MH")
						      (:MOUSE-2-1 "M")   (:MOUSE-2-2 "M2")   (:Meta-Mouse-M "Meta-M")
						      (:MOUSE-L-1 "L")   (:MOUSE-L-2 "L2")   (:MOUSE-L-HOLD "LH")
						      (:MOUSE-1-1 "L")   (:MOUSE-1-2 "L2")   (:Meta-Mouse-L "Meta-L")) :TEST #'EQ)
					      ))
				   ;; If the caller specified an illegal mouse button
				   ;; then use the following string as the mouse prefix.
				   "Bad doc keyword")))
;;;       (SHEET-STRING-OUT WHO-SHEET ": " 0
;;;			(IF (STRING-EQUAL "" DOCUMENTATION)
;;;                            ;; If the documentation for this button is empty then we do
;;;                            ;; not want to have the space after the mouse prefix.  In
;;;                            ;; this case there are two mouse buttons which do the same
;;;                            ;; thing.  The next mouse button will have the documentation
;;;                            ;; for this mouse button.  See the EDIT SCREEN menu item of
;;;                            ;; the System Menu for an example of this.
;;;                            1
;;;                            ;;ELSE
;;;                            NIL))
;;;      (SEND WHO-SHEET :SET-CURRENT-FONT NEW-WHO-LINE-FONT T)
;;;      (SHEET-STRING-OUT WHO-SHEET DOCUMENTATION)
      (SETF current-doc-line (STRING-APPEND current-doc-line ":" documentation))
      (WHEN (> (LENGTH current-doc-line) 120.)
	(funcall 'display-mac-mouse-documentation nil) )
      (SETQ PAGE-OVERFLOW-ENCOUNTERED NIL))))
    )
  )
;-------------------------------------------------------------------------------

(defun order-mouse-items (items)
"Given a list of mouse doc items e.g. (:mouse-r-2 \"system menu\") it orders
 them in mouse button order.
"
  (sortcar (copy-list items)
	 #'(lambda (x y)
	     (if (and (>= (length (the string (symbol-name x))) 8)
		       (>= (length (the string (symbol-name y))) 8)
		       (search "MOUSE-" (the string (symbol-name x))
			        :End2 1
		       )
		       (search "MOUSE-" (the string (symbol-name y))
			        :End2 1
		       )
		 )
		 (let ((key-x (aref (symbol-name x) 6))
		      (key-y (aref (symbol-name y) 6))
		     )
		     (if (equal key-x key-y)
			 (let ((num-x (aref (symbol-name x) 8))
			      (num-y (aref (symbol-name y) 8))
			     )
			     (< (char-int num-x) (char-int num-y))
			 )
			 (or (and (equal #\L key-x)
				   (member key-y '(#\M #\R))
			     )
			     (and (equal #\M key-x) (equal #\R key-y))
			 )
		     )
		  )
		  (string-lessp (the string (symbol-name x))
			       (the string (symbol-name y))
		  )
	      )
	    )
  )
)

(defun doc-size (spec intro-offset length-so-far)
"Returns the x y motion for a mouse doc spec like (:mouse-r-2 \"system menu\")
 in the mouse doc sheet.  Intro-offset is the motion for a string like
 \"M2: ,\" so that we allow for this.  Length-so-far is the length of the
 current output so that we can compute whether we throw a newline.
"
  (declare (special si:%mx-mouse-doc-max))
  (if (sys:mx-p)
      (if (>= (+ intro-offset length-so-far (length (second spec)))
	      si:%mx-mouse-doc-max
	  )
	  (values 0 1)
	  (values (+ intro-offset length-so-far (length (second spec))) 0)
      )
      (sheet-compute-motion
	who-line-documentation-window
	(+ intro-offset length-so-far) 0 (second spec) 0 nil nil 0 nil nil nil
	(GET-DEFAULT-FONT who-line-documentation-window)
      )
  )
)


(defun maybe-split-doc-spec (specs intro-offset length-so-far)
"Given a list of mouse doc specs of the form (:mouse-r-2 \"system menu\"),
 the length of a string like \"M2: ,\" and the current length of the line
 it returns a flattened version of the specs list with :newline \"\" inserted
 where the mouse doc printer should output a newline.
"
  (if specs
      (multiple-value-bind (x y)
	  (doc-size (first specs) intro-offset length-so-far)
	(if (and (not (equal 0 length-so-far)) (> y 0))
	    (append '(:Newline "")
		      (Maybe-Split-Doc-Spec specs intro-offset intro-offset)
	    )
	    (append (first specs)
		     (maybe-split-doc-spec (rest specs)
					   intro-offset (+ x intro-offset)
		     )
	    )
	)
      )
      nil
  )
)

(setf (get :Allow-Override 'non-printing-mouse-keyword) t)
(setf (get :Sort 'non-printing-mouse-keyword) t)
(setf (get :Smart-Newlines 'non-printing-mouse-keyword) t)


(defun magic-extra-who-line-stuff
       (should-allow-override should-sort smart-newlines new-state split-up)
    (if should-allow-override
       ;; we should uniqify.  This'll invokve consing a new list but
       ;; that won't cost too much.
       (progn (setq new-state (copy-list new-state))
	      (loop for (name value) on new-state by #'cddr
		    for here on new-state by #'cddr
		    for rest = (rest here)
		    do (let ((index (position name rest :Test #'eq)))
			   (if index
			        (setf (nthcdr index rest)
				      (nthcdr (+ 2 index) rest)
				)
				nil
			   )
		       )
	      )
       )
       nil
    )

    (if (or should-sort smart-newlines)
        (setq split-up
	      (loop for (name value) on new-state by #'cddr
		    collect (list name value)
	      )
	)
	nil
    )
    
    (if should-sort
       ;; ok, we know that this'll be a little expensive but that's ok.
       (progn (setq split-up (Order-Mouse-Items split-up))
	      (setq new-state (apply #'append split-up))
       )
       nil
    )

    (if smart-newlines
       (let ((intro-offset (doc-size '(ignore "M2: , ") 0 0)))
	   (setq new-state
		 (Maybe-Split-Doc-Spec split-up intro-offset intro-offset)
	   )
       )
       nil
    )
    (values should-allow-override should-sort smart-newlines new-state split-up)
)

(DEFUN PROCESS-WHO-LINE-DOCUMENTATION-LIST (WHO-SHEET NEW-STATE) 
  "This function displays who line mouse documentation from a keyword
list.  The list is organized in keyword, value pairs.  That is each odd
item is a keyword and the following item is the value for that keyword.
The keywords that are recognized are:

	:DOCUMENTATION	a general documentation string which will be displayed
			below the mouse button documentation lines.
	:KEYSTROKE	used to indicate that a particular keystroke corresponds
			to this who line documentation.  This can be either a
			a string or a character.  For effeciency reasons it is
			recommended that this be a string.
        :NO-COMMA       if in list, then items in who line documentation are not
                        delimited by a comma. Users are responsible for delimiters.
	:FONT		cause the following items in the list to be displayed in
			the specified font.  The font is not required to be in
			the font map for the who line documentation window.
        :ALLOW-OVERRIDE with :Allow-Override \"\" it wil allow elements in the
                        new state to override other ones.  Thus if the state is
                        (:Allow-Override \"\" :mouse-l-1 \"Hello\" :mouse-l-1 \"foo\")
                        then the result will be L: Hello, not L: Hello, L: foo.
        :SORT           if :sort \"\" is provided then it will sort the items into
                        a reasonable order.
        :NEWLINE        if :newline \"\" is provided then it will throw a newline
                        in the designated place.
        :SMART-NEWLINES if :Smart-Newlines \"\" is provided then it will attempt
                        to compute a good place to throw a newline, trying not
                        to wrap any of the mouse docs.
	:MOUSE-ANY	documentation for clicking ANY mouse button clicked once.
	:MOUSE-L-1	documentation for the left   mouse button clicked once.
	:MOUSE-L-2	documentation for the left   mouse button clicked twice.
	:Meta-MOUSE-L	documentation for the meta-left mouse button clicked once.
	:MOUSE-L-HOLD   documentation for the left   mouse button held down.
	:MOUSE-M-1	documentation for the middle mouse button clicked once.
	:MOUSE-M-2	documentation for the middle mouse button clicked twice.
	:Meta-MOUSE-M	documentation for the meta-middle mouse button clicked once.
	:MOUSE-M-HOLD   documentation for the middle mouse button held down.
	:MOUSE-R-1	documentation for the right  mouse button clicked once
	:MOUSE-R-2	documentation for the right  mouse button clicked twice.
	:Meta-MOUSE-R	documentation for the meta-right mouse button clicked once.
	:MOUSE-R-HOLD   documentation for the right  mouse button held down.
        

The documentation display varies based on how many who line
documentation lines are available.  If there are less than three lines all
of the mouse documentation is squeezed onto one line instead of being
displayed at separate locations."
  ;;; Note that the old forms of the mouse keywords are supported too.
  ;;; That is writing :MOUSE-1-1 instead of :MOUSE-L-1.  This is done
  ;;; only in case someone out there is using that form.  We want users
  ;;; to use the newer form because it makes for better documentation,
  ;;; that is why we do not advertise the old form in the documentation
  ;;; line for this function.
  (LET ((HAVE-DOCUMENTATION (OR (MEMBER  :DOCUMENTATION NEW-STATE :TEST #'EQ)
                                (MEMBER :KEYSTROKE     NEW-STATE :TEST #'EQ)))
			
	;; when comma is nil, we will NOT output a comma in documentation line. 
	(COMMA (MEMBER :NO-COMMA NEW-STATE :TEST #'EQ))
	;; says that we should sort the new state list.
	(should-sort (member :Sort new-state :Test #'eq))
	;; says that we should let items to the left of the state override the same named
	;; items to the right.
	(should-allow-override (member :allow-override new-state :Test #'eq))
	;; says try to find a good place to put in newlines rather than wrapping.
	(smart-newlines (member :smart-newlines new-state :Test #'eq))
	;; holds the list if it is split into pairs.
	(split-up nil)
       )
	;; We string the documentation components together in certain cases.  If there
	;; is only 1 or 2 lines then there isn't much choice.  If there are 3 lines,
	;; then we also have to have a :DOCUMENTATION component too.  If there are 4 or
	;; more lines then we can display the documentation in a 3 column format.
    (SETQ STRING-TOGETHER (OR t  ;; may 9-9-88
			      ;; 
			      ;; *** HACK ALERT - When (3)4 or more doc lines exist, special
			      ;; formatting was done to put the 9 possible keys in 3 rows
			      ;; of 3 columns. Problem with this is that long strings will
			      ;; get trashed ( overwritten ) even though the # of wholine
			      ;; lines is GREATER then before. This was a good idea that
			      ;; is just unworkable for the benefit it adds. We can't expect
			      ;; every doc-string to be tested for multiple conditions of
			      ;; who-line lines. As a result much of this code is now obsolete
			      (= NUMBER-OF-WHO-LINE-DOCUMENTATION-LINES 1)
                              (= NUMBER-OF-WHO-LINE-DOCUMENTATION-LINES 2)
                              (AND HAVE-DOCUMENTATION
                                   (= NUMBER-OF-WHO-LINE-DOCUMENTATION-LINES 3))))

    ;; Initialize constants for this function execution.
    (SETQ LEFT-CLICK-LOC 2
	  MIDDLE-CLICK-LOC (IF (NOT STRING-TOGETHER) (TRUNCATE (SHEET-INSIDE-WIDTH WHO-SHEET) 3))
	  RIGHT-CLICK-LOC (IF (NOT STRING-TOGETHER)
                              (- (SHEET-INSIDE-WIDTH WHO-SHEET) MIDDLE-CLICK-LOC))
	  MOUSE-SINGLE-LOC 2
	  MOUSE-DOUBLE-LOC (AND (NOT STRING-TOGETHER)
                                (IF (OR (MEMBER :MOUSE-L-1 NEW-STATE :TEST #'EQ)
                                        (MEMBER :MOUSE-M-1 NEW-STATE :TEST #'EQ)
                                        (MEMBER :MOUSE-R-1 NEW-STATE :TEST #'EQ)
                                        (MEMBER :MOUSE-1-1 NEW-STATE :TEST #'EQ)
                                        (MEMBER :MOUSE-2-1 NEW-STATE :TEST #'EQ)
                                        (MEMBER :MOUSE-3-1 NEW-STATE :TEST #'EQ))
                                    ;; We have single click info, put this on the second line.
                                    (+ MOUSE-SINGLE-LOC WHO-LINE-DOCUMENTATION-LINE-HEIGHT)
                                    ;;ELSE Don't have single click info, put this on the first line.
                                    MOUSE-SINGLE-LOC))
	  MOUSE-HOLD-LOC (AND
                           (NOT STRING-TOGETHER)
                           (IF (OR (MEMBER :MOUSE-L-2 NEW-STATE :TEST #'EQ)
                                   (MEMBER :MOUSE-M-2 NEW-STATE :TEST #'EQ)
                                   (MEMBER :MOUSE-R-2 NEW-STATE :TEST #'EQ)
                                   (MEMBER :MOUSE-1-2 NEW-STATE :TEST #'EQ)
                                   (MEMBER :MOUSE-2-2 NEW-STATE :TEST #'EQ)
                                   (MEMBER :MOUSE-3-2 NEW-STATE :TEST #'EQ))
                               ;; We have both double click info.  The hold info must on the line after that.
                               (+ MOUSE-DOUBLE-LOC WHO-LINE-DOCUMENTATION-LINE-HEIGHT)
                               ;;ELSE
                               (IF (NOT (= MOUSE-SINGLE-LOC MOUSE-DOUBLE-LOC))
                                   ;; There was single click info. put this after that.
                                   (+ MOUSE-SINGLE-LOC WHO-LINE-DOCUMENTATION-LINE-HEIGHT)
                                   ;; ELSE This is the only mouse documentation. put on the first line.
                                   MOUSE-SINGLE-LOC))))
    (multiple-value-setq  ;;; JPR
      (should-allow-override should-sort smart-newlines new-state split-up)
      (magic-extra-who-line-stuff
	should-allow-override should-sort smart-newlines new-state split-up
      )
   )
    (SETQ OLD-WHO-LINE-FONT        (GET-DEFAULT-FONT WHO-SHEET)
	  NEW-WHO-LINE-FONT        OLD-WHO-LINE-FONT
	  NOT-FIRST?               NIL
	  MAXIMUM-WHO-LINE-MOUSE-X 0
	  MAXIMUM-WHO-LINE-MOUSE-Y 0)


    ;; This loops through all of the non-documentation keywords.  We process them first so we can put the
    ;; documentation strings towards the bottom of the window.  If we didn't then we might intersperse them.
    (LOOP FOR DOC-SPEC = NEW-STATE THEN (CDDR DOC-SPEC)
          WHILE DOC-SPEC
          FOR OLD-KEY = NIL THEN KEY
          FOR KEY     = (FIRST  DOC-SPEC)
          FOR VALUE   = (SECOND DOC-SPEC)
          FINALLY (UNLESS (EQ (SHEET-CURRENT-FONT WHO-SHEET) OLD-WHO-LINE-FONT)
                    (SEND WHO-SHEET :SET-CURRENT-FONT OLD-WHO-LINE-FONT))
          DO
          (PROGN
            (WHEN (AND (NOT NOT-FIRST?) OLD-KEY)
              (SETQ NOT-FIRST? (AND (NOT (EQ OLD-KEY :FONT))
                                    (NOT (EQ OLD-KEY :KEYSTROKE))
				   ;; (NOT (EQ OLD-KEY :NO-COMMA))
                                    (NOT (EQ OLD-KEY :DOCUMENTATION)))))
            (IF (EQ KEY :FONT)
                (PROGN
                  ;; Change the current font.  The T argument says to
                  ;; change the font even if it isn't in the FONT-MAP.
                  (SEND WHO-SHEET :SET-CURRENT-FONT VALUE T)
                  (SETQ NEW-WHO-LINE-FONT VALUE))
                ;;ELSE
                (IF (AND (NOT (EQ KEY :KEYSTROKE))
                         (NOT (EQ KEY :DOCUMENTATION))
			 (not (eq key :no-comma)))
                    (DISPLAY-WHO-LINE-MOUSE-INFO WHO-SHEET KEY VALUE COMMA)))))

    (WHEN HAVE-DOCUMENTATION
      (SHEET-SET-CURSORPOS WHO-SHEET MAXIMUM-WHO-LINE-MOUSE-X MAXIMUM-WHO-LINE-MOUSE-Y)
      ;; If the mouse info wraps onto the last line available then we start the :DOCUMENTATION info
      ;; there.  Otherwise we put the :DOCUMENTATION on the next line.
      (SETQ NOT-FIRST? (AND string-together
			    (= (1+ (TRUNCATE MAXIMUM-WHO-LINE-MOUSE-Y WHO-LINE-DOCUMENTATION-LINE-HEIGHT))
                               NUMBER-OF-WHO-LINE-DOCUMENTATION-LINES)
                            (NOT (ZEROP MAXIMUM-WHO-LINE-MOUSE-X)))) 

      (SETQ NEW-WHO-LINE-FONT (GET-DEFAULT-FONT WHO-SHEET))
      (CATCH 'PAGE-OVERFLOW
	(WHEN (NOT NOT-FIRST?)
	  (SEND WHO-SHEET :FRESH-LINE))
	;; Now we loop through again to get all of the :DOCUMENTATION info.
	(LOOP FOR DOCUMENTATION-KEYWORD IN '(:DOCUMENTATION :KEYSTROKE) ;;;:no-comma)
	      DO
	      (LOOP FOR DOC-SPEC = NEW-STATE THEN (CDDR DOC-SPEC)
		    WHILE DOC-SPEC
		    WITH OLD-KEY = NIL
		    FOR KEY      = (FIRST  DOC-SPEC)
		    FOR VALUE    = (SECOND DOC-SPEC)
		    FINALLY (UNLESS (EQ (SHEET-CURRENT-FONT WHO-SHEET) OLD-WHO-LINE-FONT)
			      (SEND WHO-SHEET :SET-CURRENT-FONT OLD-WHO-LINE-FONT))
		    WHEN (OR (EQ KEY :FONT) (EQ KEY DOCUMENTATION-KEYWORD))
		    DO
		    (PROGN
		      (IF (EQ KEY :FONT)
			  (PROGN
			    ;; Change the current font.  The T argument says to
			    ;; change the font even if it isn't in the FONT-MAP.
			    (SEND WHO-SHEET :SET-CURRENT-FONT VALUE T)
			    (SETQ NEW-WHO-LINE-FONT VALUE))
			  ;;ELSE
			  (WHEN (NOT (EQ KEY :FONT))
				(when (and NOT-FIRST? (not comma))
				       (SHEET-STRING-OUT WHO-SHEET ",  ")
				       ;;else
				    (SETQ NOT-FIRST? T)))
			    (WHEN (EQ KEY :KEYSTROKE)
			      (SEND WHO-SHEET :SET-CURRENT-FONT *MOUSE-DOCUMENTATION-LINE-BUTTONS-STANDARD-FONT* T)
			      (SHEET-STRING-OUT WHO-SHEET "Keystroke: ")
			      ;; Make sure the value is a string.
			      (WHEN (OR (CHARACTERP VALUE) (INTEGERP VALUE))
				(SETQ VALUE (FORMAT NIL "~:C" VALUE)))
			      (SEND WHO-SHEET :SET-CURRENT-FONT NEW-WHO-LINE-FONT T))
			    (SHEET-STRING-OUT WHO-SHEET VALUE))
		      (SETQ OLD-KEY KEY))
		    ))))))

;-------------------------------------------------------------------------------


(DEFUN PROCESS-WHO-LINE-DOCUMENTATION-LIST-MAC (NEW-STATE)
  (declare (special mac-line current-doc-line))
  "This function is similar to PROCESS-WHO-LINE-DOCUMENTATION-LIST
except it displays who line mouse documentation on the MAC"
  (LET ((HAVE-DOCUMENTATION (OR (MEMBER  :DOCUMENTATION NEW-STATE :TEST #'EQ)
                                (MEMBER :KEYSTROKE     NEW-STATE :TEST #'EQ)))
			
	;; when comma is nil, we will NOT output a comma in documentation line. 
	(COMMA (MEMBER :NO-COMMA NEW-STATE :TEST #'EQ))
	;; says that we should sort the new state list.
	(should-sort (member :Sort new-state :Test #'eq))
	;; says that we should let items to the left of the state override the same named
	;; items to the right.
	(should-allow-override (member :allow-override new-state :Test #'eq))
	;; says try to find a good place to put in newlines rather than wrapping.
	(smart-newlines (member :smart-newlines new-state :Test #'eq))
	;; holds the list if it is split into pairs.
	(split-up nil)
       )
	;; We string the documentation components together in certain cases.  If there
	;; is only 1 or 2 lines then there isn't much choice.  If there are 3 lines,
	;; then we also have to have a :DOCUMENTATION component too.  If there are 4 or
	;; more lines then we can display the documentation in a 3 column format.
    (SETQ STRING-TOGETHER (OR (= NUMBER-OF-WHO-LINE-DOCUMENTATION-LINES 1)
                              (= NUMBER-OF-WHO-LINE-DOCUMENTATION-LINES 2)
                              (AND HAVE-DOCUMENTATION
                                   (= NUMBER-OF-WHO-LINE-DOCUMENTATION-LINES 3))))

    ;; Initialize constants for this function execution.
    (SETQ LEFT-CLICK-LOC 2
	  MIDDLE-CLICK-LOC 200.
	  RIGHT-CLICK-LOC (IF (NOT STRING-TOGETHER)
                              400.)
	  MOUSE-SINGLE-LOC 2
	  MOUSE-DOUBLE-LOC (AND (NOT STRING-TOGETHER)
                                (IF (OR (MEMBER :MOUSE-L-1 NEW-STATE :TEST #'EQ)
                                        (MEMBER :MOUSE-M-1 NEW-STATE :TEST #'EQ)
                                        (MEMBER :MOUSE-R-1 NEW-STATE :TEST #'EQ)
                                        (MEMBER :MOUSE-1-1 NEW-STATE :TEST #'EQ)
                                        (MEMBER :MOUSE-2-1 NEW-STATE :TEST #'EQ)
                                        (MEMBER :MOUSE-3-1 NEW-STATE :TEST #'EQ))
                                    ;; We have single click info, put this on the second line.
                                    (+ MOUSE-SINGLE-LOC WHO-LINE-DOCUMENTATION-LINE-HEIGHT)
                                    ;;ELSE Don't have single click info, put this on the first line.
                                    MOUSE-SINGLE-LOC))
	  MOUSE-HOLD-LOC (AND
                           (NOT STRING-TOGETHER)
                           (IF (OR (MEMBER :MOUSE-L-2 NEW-STATE :TEST #'EQ)
                                   (MEMBER :MOUSE-M-2 NEW-STATE :TEST #'EQ)
                                   (MEMBER :MOUSE-R-2 NEW-STATE :TEST #'EQ)
                                   (MEMBER :MOUSE-1-2 NEW-STATE :TEST #'EQ)
                                   (MEMBER :MOUSE-2-2 NEW-STATE :TEST #'EQ)
                                   (MEMBER :MOUSE-3-2 NEW-STATE :TEST #'EQ))
                               ;; We have both double click info.  The hold info must on the line after that.
                               (+ MOUSE-DOUBLE-LOC WHO-LINE-DOCUMENTATION-LINE-HEIGHT)
                               ;;ELSE
                               (IF (NOT (= MOUSE-SINGLE-LOC MOUSE-DOUBLE-LOC))
                                   ;; There was single click info. put this after that.
                                   (+ MOUSE-SINGLE-LOC WHO-LINE-DOCUMENTATION-LINE-HEIGHT)
                                   ;; ELSE This is the only mouse documentation. put on the first line.
                                   MOUSE-SINGLE-LOC))))

    (multiple-value-setq  ;;; JPR
      (should-allow-override should-sort smart-newlines new-state split-up)
      (magic-extra-who-line-stuff
	should-allow-override should-sort smart-newlines new-state split-up
      )
   )
    (SETQ OLD-WHO-LINE-FONT        nil ;(GET-DEFAULT-FONT WHO-SHEET)
	  NEW-WHO-LINE-FONT        nil ;OLD-WHO-LINE-FONT
	  NOT-FIRST?               NIL
	  MAXIMUM-WHO-LINE-MOUSE-X 0
	  MAXIMUM-WHO-LINE-MOUSE-Y 0)


    ;; This loops through all of the non-documentation keywords.  We process them first so we can put the
    ;; documentation strings towards the bottom of the window.  If we didn't then we might intersperse them.
    (LOOP FOR DOC-SPEC = NEW-STATE THEN (CDDR DOC-SPEC)
          WHILE DOC-SPEC
          FOR OLD-KEY = NIL THEN KEY
          FOR KEY     = (FIRST  DOC-SPEC)
          FOR VALUE   = (SECOND DOC-SPEC)
          FINALLY  t ;(UNLESS (EQ (SHEET-CURRENT-FONT WHO-SHEET) OLD-WHO-LINE-FONT)
                  ;  (SEND WHO-SHEET :SET-CURRENT-FONT OLD-WHO-LINE-FONT))
          DO
          (PROGN
            (WHEN (AND (NOT NOT-FIRST?) OLD-KEY)
              (SETQ NOT-FIRST? (AND (NOT (EQ OLD-KEY :FONT))
                                    (NOT (EQ OLD-KEY :KEYSTROKE))
				   ;; (NOT (EQ OLD-KEY :NO-COMMA))
                                    (NOT (EQ OLD-KEY :DOCUMENTATION)))))
            (unless (EQ KEY :FONT)
                (IF (AND (NOT (EQ KEY :KEYSTROKE))
                         (NOT (EQ KEY :DOCUMENTATION))
			 (not (eq key :no-comma)))
 		    (progn
		      (DISPLAY-WHO-LINE-MOUSE-INFO-MAC  KEY VALUE COMMA)
		      (WHEN (> (LENGTH current-doc-line) 120.)
			(funcall 'display-mac-mouse-documentation nil))))
		    )))

    (WHEN HAVE-DOCUMENTATION
      ;; If the mouse info wraps onto the last line available then we start the :DOCUMENTATION info
      ;; there.  Otherwise we put the :DOCUMENTATION on the next line.
      (SETQ NOT-FIRST? (AND STRING-TOGETHER
                            (= (1+ (TRUNCATE MAXIMUM-WHO-LINE-MOUSE-Y WHO-LINE-DOCUMENTATION-LINE-HEIGHT))
                               NUMBER-OF-WHO-LINE-DOCUMENTATION-LINES)
                            (NOT (ZEROP MAXIMUM-WHO-LINE-MOUSE-X)))) 

	(WHEN (NOT NOT-FIRST?)
	  (funcall 'display-mac-mouse-documentation nil)
     )
	;; Now we loop through again to get all of the :DOCUMENTATION info.
	(LOOP FOR DOCUMENTATION-KEYWORD IN '(:DOCUMENTATION :KEYSTROKE) ;;;:no-comma)
	      DO
	      (LOOP FOR DOC-SPEC = NEW-STATE THEN (CDDR DOC-SPEC)
		    WHILE DOC-SPEC
		    WITH OLD-KEY = NIL
		    FOR KEY      = (FIRST  DOC-SPEC)
		    FOR VALUE    = (SECOND DOC-SPEC)
		    FINALLY t
		    WHEN
		       (EQ KEY DOCUMENTATION-KEYWORD)
		    DO
		    (PROGN
		      (IF (EQ KEY :FONT)
			  t
			  ;;ELSE
			  (WHEN (NOT (EQ KEY :FONT))
				(when (and NOT-FIRST? (not comma))
				  (SETF current-doc-line (STRING-APPEND  current-doc-line ", "))
				       ;;else
				    (SETQ NOT-FIRST? T)))
			    (WHEN (EQ KEY :KEYSTROKE)
			      (SETF current-doc-line (STRING-APPEND current-doc-line "Keystroke: "))

			      ;; Make sure the value is a string.
			      (WHEN (OR (CHARACTERP VALUE) (INTEGERP VALUE))
				(SETQ VALUE (FORMAT NIL "~:C" VALUE)))
			      )
			    (SETF current-doc-line (STRING-APPEND current-doc-line value))
			    (setf mac-line NUMBER-OF-WHO-LINE-DOCUMENTATION-LINES)
			    (funcall 'display-mac-mouse-documentation nil)
   			    )
		      (SETQ OLD-KEY KEY))
		    )))))

;-------------------------------------------------------------------------------

;;; The following simply included to force recompilation to get #'s to work.

(DEFUN INSPECT-ARRAY-PRINTER (ITEM ARG STREAM ITEM-NUMBER &AUX (OBJ (CAR ARG))    ;!
  (LEADER-LENGTH-TO-MENTION (if (cadr arg) (array-leader-length obj) 0)))
  "The print-function used when inspecting an array."
  ;; (CAR ARG) is the array.  (CADR ARG) is T to display the leader.
  ;; ITEM is usually a number.  A small number is an index in the leader.
  ;; Numbers too big for that start moving through the array elements.
  ;;;Make sure base is consistent since sometimes this is called from the mouse process.
  
  ;(LET ((*PRINT-BASE* *INSPECT-PRINT-BASE*))
    (COND
      ((NOT (NUMBERP ITEM)) (INSPECT-PRINTER ITEM OBJ STREAM ITEM-NUMBER))
      ((< ITEM LEADER-LENGTH-TO-MENTION)
       (LET ((pntr (LOCF (ARRAY-LEADER obj item))))
         (SEND STREAM :ITEM1 ITEM 'LEADER-SLOT
             #'(LAMBDA (ITEM STREAM) (FORMAT STREAM "Leader ~D" ITEM)))
       (FORMAT STREAM ":~12T ")
       (IF (%P-CONTENTS-SAFE-P pntr)
           (SEND STREAM :ITEM1 (ARRAY-LEADER OBJ ITEM) :VALUE #'PRINT-ITEM-CONCISELY)
           (FORMAT STREAM "#<~A ~O>"
                   (OR (NTH (%P-DATA-TYPE pntr) Q-DATA-TYPES)
                       (%P-DATA-TYPE pntr))
                   (%P-POINTER pntr)))))
      (T
       (LET ((ITEM (- ITEM LEADER-LENGTH-TO-MENTION))
             (RANK (ARRAY-RANK OBJ))
             INDICES)
         (OR (= RANK 1) (SETQ INDICES (ARRAY-INDICES-FROM-INDEX OBJ ITEM)))
         (SEND STREAM :ITEM1 (CONS ITEM (IF (= RANK 1) ITEM INDICES)) 'ARRAY-SLOT
               #'(LAMBDA (DATUM STREAM) (FORMAT STREAM "Elt ~D" (CDR DATUM))))
         (FORMAT STREAM ":~9T ")
         (IF
          (OR (CDR (ASSOC (ARRAY-TYPE OBJ) ARRAY-BITS-PER-ELEMENT :TEST #'EQ))
              (%P-CONTENTS-SAFE-P (AP-1-FORCE OBJ ITEM)))
	  ;; Deal with data types that are objects, and with numeric arrays.
          (SEND STREAM :ITEM1 (AR-1-FORCE OBJ ITEM) :VALUE #'PRINT-ITEM-CONCISELY)
	  ;; Deal with data types that aren't really objects.
          (FORMAT STREAM "#<~A ~O>"
                  (OR (NTH (%P-DATA-TYPE (AP-1-FORCE OBJ ITEM)) Q-DATA-TYPES)
                      (%P-DATA-TYPE (AP-1-FORCE OBJ ITEM)))
                  (%P-POINTER (AP-1-FORCE OBJ ITEM))))))))

(DEFUN PRINT-FEF-CONSTANT (ITEM STREAM PREFIX)
  (PRINC PREFIX STREAM)
  (SEND STREAM :ITEM1 (FIRST ITEM) :VALUE #'PRINT-ITEM-CONCISELY)) 


(defmethod (inspection-data :handle-mouse-click) (blip flavor-inspector)
  (selector (fourth blip) =
    (#\mouse-l-1 (send flavor-inspector :inspect-info-left-click))
    (t (beep))
  )
)

;-------------------------------------------------------------------------------

;;; The following code supports multiple-window representations within an
;;; inspect frame.

;;; Modified TI code.
(defun inspect-setup-object-display-list (object window
					  &optional top-item label
					  &aux str)
  (let ((*window* window))
    (declare (special *window*))
    (multiple-value-bind
      (display-list arg alt-print-fun first-top-item obj-label
       item-generator window-flavor)
	;; Window flavor added by JPR
	(send window
	      (cond ((typep object 'stack-frame) :object-stack-frame)
		    ((named-structure-p object) :object-named-structure)
		    (t
		     (case (data-type object)
		       (dtp-instance :object-instance)
		       (dtp-array :object-array)
		       (dtp-list :object-list)
		       (dtp-stack-list :object-list)
		       (dtp-symbol :object-symbol)
		       (dtp-closure :object-closure)
		       (dtp-lexical-closure :object-lexical-closure)	;!!!
		       (dtp-function :object-fef)
		       (dtp-locative :object-locative)
		       (dtp-stack-group :object-stack-group)	;!
		       (otherwise :object-other))))
	      object)
      (list object
	    (or alt-print-fun 'inspect-printer)
	    arg display-list (or top-item first-top-item 0)
	    (or label
		obj-label
		(list nil nil nil nil (label-font (send window :label))
		      (if (consp object)
			  "a list"
			  (nsubstring (setq str (let ((*inhibit-inspection-data*
							nil))
						  (format nil "~S~%" object)))
				      0. (position #\Cr str)))))
	    item-generator
	    ;; window flavor added by JPR.
	    window-flavor))))

;;; Modified TI code.
(defun inspect-setup-object (object window &optional top-item)
  (let ((disp (inspect-setup-object-display-list object window top-item)))
       ;; Look to a better window to use than an inspector
       (let ((real-window (send window :find-actual-window disp object)))
	    (send real-window :setup (cdr disp))
	    (send window :set-current-object (car disp))
	    (if (or (not (equal window real-window))
		    (member window
			    (send (Find-Inspector-Window self) :Send-If-Handles
				  :Exposed-Panes)
		    )
		)
		(send real-window :expose) ;;; JPR.
		nil
	    )
	    disp
       )
  )
)

;;; Modified TI code.
(defmethod (basic-inspect :setup-object) (sl)
  ;; Look to a better window to use than an inspector
  (let ((real-window (send self :find-actual-window sl :not-here)))
       (send real-window :setup (cdr sl))
       (send self :set-current-object (car sl))
       (if (or (not (equal self real-window))
	       (member self (send (Find-Inspector-Window self) :Send-If-Handles
			      :Exposed-Panes))
	   )
	   (send real-window :expose) ;;; JPR.
	   nil
       )
       sl
  )
)

(defmethod find-superior-for-new-inspect-pane ((frame t))
  frame
)

(defmethod find-compute-edges-for-new-inspect-pane ((frame t) inspector-edges)
  inspector-edges
)

(defun get-inferior-of-right-shape-and-size (inspector flavor)
"Given an inspector pane and a flavor of a window finds or creates a window
of the right flavor that is the same size as the inspector.  Flavor can be
either a flavor name or a list of the form (flavor-name &rest init-plist).
"
  (let ((inspector-edges (multiple-value-list (send inspector :Edges)))
	(frame (Find-Inspector-Window inspector))
	(flavor (ucl:first-if-list flavor))
	(init-args (if (consp flavor) (rest flavor) nil))
       )
       (or (find-if #'(lambda (win)
			(and (typep win flavor)
			     (equal (multiple-value-list (send win :Edges))
				    inspector-edges
			     )
			)
		      )
		      (send frame :Inferiors)
	   )
	   (let ((new-pane
		   (apply 'make-instance flavor :Superior
			  (Find-Superior-For-New-Inspect-Pane frame)
			  :Edges-From (Find-Compute-Edges-For-New-Inspect-Pane
					frame inspector-edges
				      )
			  init-args
		   )
		 )
		)
	        (send self :Send-If-Handles :add-new-pane new-pane)
	        new-pane
	   )
       )
  )
)

(defmethod (basic-inspect :find-actual-window) (display-list &rest args)
"Finds the actual window to use for the display list."
  (ignore args)
  (destructuring-bind
    (object printer arg display-list-for-obj the-top-item label
     the-item-generator flavor
    )
    display-list
    (ignore object printer arg display-list-for-obj label the-top-item
	    the-item-generator
    )
    (if flavor
	(Get-inferior-of-right-shape-and-size self flavor)
	self
    )
  )
)

(defun ideal-window-for-inspector (inspector)
"Given an inspector pane looks at the cache entry associated with it to
determine which window should really be geographically where this one is.
"
  (let ((ce (assoc (send inspector :Current-Object)
		   (send (send (Find-Inspector-Window inspector) :History)
			 :Cache
		   )
		   :Test #'eq
	    )
	)
       )
       (send inspector :Find-Actual-Window ce)
  )
)

(defwhopper (basic-inspect :scroll-to) (line type)
"Makes sure that scrolling works properly in the event of the ideal window not
being the inspector.
"
  (let ((real-window (Ideal-Window-For-Inspector self)))
       (if (equal real-window self)
	   (continue-whopper line type)
	   (case type
	     (:Relative (send real-window :Scroll-Relative-From-Inspector
			      self (* line line-height)
			)
             )
	     (:Absolute (send real-window :Scroll-To-From-Inspector
			      self (* line line-height)
			)
	     )
	     (otherwise (beep))
	   )
       )
  )
)

(defmethod (general-inspect-window :After :setup) (SL)
  "Make sure that the inspector is exposed if we are empty."
  (if (and (consp sl)
	   (consp (fifth sl))
	   (string-equal "EMPTY" (sixth (fifth sl)))
	   (member self (send (Find-Inspector-Window self) :Send-If-Handles
			      :Exposed-Panes))
      )
      (process-run-function
	"Expose"
	#'(lambda (win) (send win :Expose))
	self
      )
      nil
  )
)

(defmethod (basic-general-inspector :after :expose) (&rest ignore)
"Put in a hook so that non in-constraint inferiors are exposed as appropriate
 when we are.
"
  (Make-Sure-Right-Windows-Are-Exposed self)
)

(defmethod Find-Inspector-Window-1 (pane (me t))
  (ignore pane)
  me
)

(defmethod find-inspector-window-1
	   (pane (me constraint-frame-with-shared-io-buffer))
  (ignore pane)
  me
)

(defun find-inspector-window (from-window)
  (let ((the-superior (send from-window :Superior)))
       (Find-Inspector-Window-1 from-window the-superior)
  )
)


(defun make-sure-right-windows-are-exposed (window)
"Exposes all non in-constraint inferiors that need to be."
  (loop for inf in (send window :Send-If-Handles :Exposed-Panes) do
	(if (typep inf 'basic-inspect)
	    (let ((ce (assoc (send inf :Current-Object)
			     (send (send window :History) :Cache)
			     :Test #'eq
		      )
		  )
		 )
	         (let ((ideal-window (send inf :Find-Actual-Window ce)))
		      (if (not (equal ideal-window inf))
			  (process-run-function
			    "Expose"
			    #'(lambda (ideal-window inf)
				(ignore inf)
				(send ideal-window :Expose)
			      )
			    ideal-window inf
			  )
			  nil
		      )
		 )
	    )
	    nil
	)
  )
)


;-------------------------------------------------------------------------------

;;; Miscellaneous fixes.

(defflavor flavor-operation-mixin ()
	   (generic-middle-button-mixin inspection-data)
)

(defmethod (show-all-methods-sorted :Middle-Button-Result) ()
  data
)


;-------------------------------------------------------------------------------

;;; Patches to TI patches.

#!C
; From file INSPECT.LISP#> DEBUG-TOOLS; MR-X:
#8R TV#:
(COMPILER-LET ((*PACKAGE* (FIND-PACKAGE "TV"))
                          (SI:*LISP-MODE* :COMMON-LISP)
                          (*READTABLE* COMMON-LISP-READTABLE)
                          (SI:*READER-SYMBOL-SUBSTITUTIONS* SYS::*COMMON-LISP-SYMBOL-SUBSTITUTIONS*))
  (COMPILER#:PATCH-SOURCE-FILE "general-inspector: general-inspector; general-inspector.#"

(DEFMETHOD (BASIC-INSPECT :OBJECT-INSTANCE) (OBJ)     ;fi + hash tables
  (LET ((maxl -1)
        result flavor class) ;*** TAC 6/15/89 - added local class variable
    ;;If the instance to inspect is an instance of INSPECTION-DATA and our superior's INSPECTION-DATA-ACTIVE? is T,
    ;;let the instance generate the inspection item.  This is used in special-purpose inspectors such as the flavor inspector.
    (IF (AND (SI:SEND-IF-HANDLES SUPERIOR :INSPECTION-DATA-ACTIVE?) (TYPEP OBJ 'INSPECTION-DATA))
        (if (send obj :operation-handled-p :generate-item-specialized)
	    (send obj :generate-item-specialized self)
	    (MULTIPLE-VALUE-BIND (TEXT-ITEMS INSPECTOR-LABEL)
		(SEND OBJ :GENERATE-ITEM)
	      (VALUES TEXT-ITEMS () 'INSPECT-PRINTER () INSPECTOR-LABEL)))
        ;;Otherwise inspect the flavor instance in the normal fashion.
        (PROGN
	  (WHEN (ticlos:clos-instance-p obj) ;*** TAC 6/15/89 - if CLOS instance then get it's class
	      (SETQ class (ticlos:class-of obj))) 
	  (SETQ FLAVOR (SI:INSTANCE-FLAVOR OBJ)) 
	  (IF class
	      (SETQ RESULT ;*** TAC 6/15/89 - different label for a class 
                (LIST '("")
                      `("An object of class " (:ITEM1 CLASS ,(TYPE-OF OBJ))
                        ".  Class object is " (:ITEM1 CLASS-OBJECT ,class)))) 
	      (SETQ RESULT ;*** TAC 6/15/89 - same label as always for a flavor 
                (LIST '("")
                      `("An object of flavor " (:ITEM1 FLAVOR ,(TYPE-OF OBJ))
                        ".  Function is " (:ITEM1 FLAVOR-FUNCTION ,(SI:INSTANCE-FUNCTION OBJ)))))) 
          (LET ((IVARS
                  (IF FLAVOR
                      (SI:FLAVOR-ALL-INSTANCE-VARIABLES FLAVOR)
                      (%P-CONTENTS-OFFSET (%P-CONTENTS-AS-LOCATIVE-OFFSET OBJ 0)
                                          %INSTANCE-DESCRIPTOR-BINDINGS))))
            (DO ((BINDINGS IVARS (CDR BINDINGS))
                 (I 1 (1+ I)))
                ((NULL BINDINGS))
              (SETQ MAXL (MAX (FLATSIZE (CAR BINDINGS)) MAXL)))
            ;(SETQ MAXL (MAX (FLATSIZE (%FIND-STRUCTURE-HEADER (CAR BINDINGS))) MAXL)))
            (DO ((BINDINGS IVARS (CDR BINDINGS))
                 (SYM)
                 (I 1 (1+ I)))
                ((NULL BINDINGS))
              (SETQ SYM (CAR BINDINGS))
              ;(SETQ SYM (%FIND-STRUCTURE-HEADER (CAR BINDINGS)))
              (PUSH
                `((:ITEM1 INSTANCE-SLOT ,SYM) (:COLON ,(+ 2 MAXL))
                  ,(IF (= dtp-null (%p-data-type (%instance-loc obj i)))
                       ;,(IF (= (%P-LDB-OFFSET %%Q-DATA-TYPE OBJ I) DTP-NULL)
                       "unbound"
                       `(:item1 instance-value ,(%instance-ref obj i))))
                ;`(:ITEM1 INSTANCE-VALUE ,(%P-CONTENTS-OFFSET OBJ I))))
                RESULT)
              (if (equal (First Bindings) 'Si::Hash-Array)
                  (let ((window-items (Make-Window-Items-For-Hash-Table
					(send obj :hash-array) 'identity nil)))
                    (dolist (element window-items) (push element result))))
              ))
          (NREVERSE RESULT)))))
))

;;; Ignores put in by JPR.  It was doing string-width using the pixel position
;;; instead of the length of the string.

#!C
; From file INSPECT.LISP#> DEBUG-TOOLS; SYS:
#8R TV#:
(COMPILER-LET ((*PACKAGE* (FIND-PACKAGE "TV"))
                          (SI:*LISP-MODE* :COMMON-LISP)
                          (*READTABLE* COMMON-LISP-READTABLE)
                          (SI:*READER-SYMBOL-SUBSTITUTIONS* SYS::*COMMON-LISP-SYMBOL-SUBSTITUTIONS*))
  (COMPILER#:PATCH-SOURCE-FILE "general-inspector:general-inspector;general-inspector.#"

(DEFMETHOD (BASIC-INSPECT :OBJECT-LIST) (LIST)
  (let ((*print-circle* t))			; DAB 12-11-89
    (MULTIPLE-VALUE-BIND (STRING-LIST ATOMIC-ITEMS LIST-ITEMS)
	(GRIND-INTO-LIST LIST
			 (1- (TRUNCATE (SHEET-INSIDE-WIDTH) CHAR-WIDTH))	;01-05-88 DAB Added 1- to fix losing the last char
			 T)
      ;; turn string-list into a list of elements, one for each line, of the form
      ;; (NIL contents-string atom-item-list line-contains-lozenged-characters-p).
      (DO ((L STRING-LIST (CDR L))
	   (AIS ATOMIC-ITEMS (CDR AIS)))
	  ((NULL L))
	(LET ((LOZENGED-CHARACTERS
		(DOTIMES (I (LENGTH (CAR L)))
		  (IF (>= (AREF (CAR L) I) 200) (RETURN T)))))
	  ;; Convert the start and end indices for each atom-item from characters to pixels.
	  ;; If this line contains no lozenged characters,
	  ;; this can be done by multiplying.  Otherwise, SHEET-STRING-LENGTH must be used.
	  (DOLIST (I (CAR AIS))
	    (SETF (THIRD I)
		  (+ (SHEET-INSIDE-LEFT)
		     (IF LOZENGED-CHARACTERS
			 (MULTIPLE-VALUE-BIND (IGNORE ignore z)
			     (SHEET-STRING-LENGTH SELF (CAR L) 0 #+Ignore (THIRD I))
			   (VALUES z))
			 (* (THIRD I) CHAR-WIDTH))))
	    (SETF (FOURTH I)
		  (+ (SHEET-INSIDE-LEFT)
		     (IF LOZENGED-CHARACTERS
			 (MULTIPLE-VALUE-BIND (IGNORE ignore z)
			     (SHEET-STRING-LENGTH SELF (CAR L) 0 #+Ignore (FOURTH I))
			   (VALUES z))
			 (* (FOURTH I) CHAR-WIDTH)))))
	  (RPLACA L (LIST NIL (CAR L) (CAR AIS) LOZENGED-CHARACTERS))))
      ;; Convert the starting and ending hpos of each list-item from characters to pixels
      ;; Must find the line which the start or end appears on
      ;; and see whether that line had any lozenged characters
      ;; to decide whether a multiplication is sufficient.
      (DOLIST (I LIST-ITEMS)
	(SETF (SECOND I)
	      (+ (SHEET-INSIDE-LEFT)
		 (LET ((LINE-DESC (NTH (THIRD I) STRING-LIST)))
		   (IF (FOURTH LINE-DESC)
                       (MULTIPLE-VALUE-BIND (IGNORE ignore z)
                           (SHEET-STRING-LENGTH SELF (SECOND LINE-DESC) 0 #+Ignore (SECOND I))
                         (VALUES z))
		       (* (SECOND I) CHAR-WIDTH)))))
	(SETF (FOURTH I)
	      (+ (SHEET-INSIDE-LEFT)
		 (LET ((LINE-DESC (NTH (FIFTH I) STRING-LIST)))
		   (IF (FOURTH LINE-DESC)
                       (MULTIPLE-VALUE-BIND (IGNORE ignore z)
                           (SHEET-STRING-LENGTH SELF (SECOND LINE-DESC) 0 #+Ignore (FOURTH I))
                         (VALUES z))
		       (* (FOURTH I) CHAR-WIDTH))))))
      (SETQ LIST-ITEMS
	    (SORT LIST-ITEMS
		  #'(LAMBDA (X Y)
		      (COND
			((< (THIRD Y) (THIRD X)) T)
			((> (THIRD Y) (THIRD X)) NIL)
			(T (> (SECOND X) (SECOND Y)))))))
      (DO ((LINE (1- (LENGTH STRING-LIST)) (1- LINE))
	   (CURRENT LIST-ITEMS))
	  ((< LINE 0))
	(DO ()
	    ((OR (NULL CURRENT) (<= (THIRD (CAR CURRENT)) LINE)))
	  (SETQ CURRENT (CDR CURRENT)))
	(RPLACA (CAR (NTHCDR LINE STRING-LIST)) CURRENT))
      (VALUES STRING-LIST :LIST-STRUCTURE 'INSPECT-LIST-PRINTER))))
))