;;; -*- Mode:Common-Lisp; Package:tv; Base:10; Fonts:(TVFONT) -*-

;;; This software developed by:
;;;	James Rice
;;; at the Stanford University Knowledge Systems Lab in 1986 and 1987.
;;;
;;; This work was supported in part by:
;;;	DARPA Grant F30602-85-C-0012

;;;----------------------------------------------------------------------
;;;  Most of this file derived from code licensed from Texas Instruments
;;;  Inc.  Since we'd like them to adopt these changes, we're claiming
;;;  no rights to them, however, the following restrictions apply to the
;;;  TI code:

;;; Your rights to use and copy Explorer System Software must be obtained
;;; directly by license from Texas Instruments Incorporated.  Unauthorized
;;; use is prohibited.

;;;                           RESTRICTED RIGHTS LEGEND

;;;Use, duplication, or disclosure by the Government is subject to
;;;restrictions as set forth in subdivision (b)(3)(ii) of the Rights in
;;;Technical Data and Computer Software clause at 52.227-7013.
;;;
;;;                     TEXAS INSTRUMENTS INCORPORATED.
;;;                              P.O. BOX 2909
;;;                           AUSTIN, TEXAS 78769
;;;                                 MS 2151
;;;
;;; Copyright (C) 1986,1987 Texas Instruments Incorporated. All rights reserved.
;;;----------------------------------------------------------------------

;;; Defines a facility whereby edit buffer streams appear in the who
;;; line.

zwei:
;;;Edited by Acuff                 19 Feb 88  9:32
(defun zwei:find-file-sheet ()
"This function takes no arguments and returns the who-line file sheet object.
 This is the sheet object in which the open file status gets printed.
"
  (if tv:who-line-screen
      (let ((panes (Send tv:who-line-screen :Inferiors)))
	   (first (remove-if-not
		    #'(Lambda (pane) (typep pane 'tv:Who-line-file-sheet))
		    panes
		  )
	   )
      )
      nil
  )
)


zwei:
;;;Edited by Acuff                 19 Feb 88  9:32
(defun zwei:find-length-of-node
    (node &Aux (current-bp (bp-line (Send node :First-bp))) (counter 1))
"This function is passed a node, which represents an Interval.  It returns the
 number of lines in that node.  It gets this number by counting them.
"
  (let ((finish (bp-line (Send node :Last-bp))))
       ;;RDA: Add the NULL test for VisiDoc
       (loop until (or (null current-bp) (eq current-bp finish)) do
	    (Setq counter (+ 1 counter))
	    (Setq current-bp (line-next current-bp))
       )
  )
  counter
)


zwei:
;;;Edited by Acuff                 19 Feb 88  9:32
(defmethod (zwei:interval-stream :truename) (&aux int)
"This method is a slightly modified version of the original because the original
 caused ZMacs to bomb out occasionally because the index was nil and not a
 number.
"
  (and (typep (setq int (bp-top-level-node
			    (create-bp *line*
				       (if (equal *index* nil) 0 *index*))))
	       'file-buffer)
	(fs:parse-pathname (buffer-name int) 'ed-buffer)))


zwei:
;;;Edited by Acuff                 19 Feb 88  9:32
(defmethod (zwei:Interval-Stream :After :Init) (ignore)
"This method sets up properties for the stream, which hold the length of the
 stream and the current position in the stream in lines and characters.  It
 then adds Self to the who line file sheet's list of streams.
"
  (Send **Interval** :Putprop (find-length-of-node **Interval**) :Stream-Length)
  (Send **Interval** :Putprop 0 :Position)
  (Send **Interval** :Putprop 0 :Characters)
)


zwei:
;;;Edited by Acuff                 19 Feb 88  9:32
(defmethod (zwei:interval-stream :After :Close) (&Rest Ignore)
"This daemon method removes the stream from the who line if the stream is
closed.  The stream is closed explicitly like this when it passes through an
unwind protect with a close."
  (if (find-file-sheet)
      (send (find-file-sheet) :delete-stream Self)
      nil
  )
)


;;;Edited by Acuff                 19 Feb 88  9:32
zwei:
(defmethod (zwei:interval-stream :After :tyi) (&optional ignore)
"This daemon method is used in order to keep treack of reading from this
 stream so that a reasonable message will get put out in the who line.
 When a new line is read the position counter is incremented.  When the end
 of the stream is found the stream is removed from the who line stream list.
 Otherwise the character count is incremented.
"
  (declare (special *Zmacs-Buffer-Streams-Displayed-In-Who-Line-P*))
  (if *Zmacs-Buffer-Streams-Displayed-In-Who-Line-P*
      (cond (zwei:*eof* nil)
	    ((eq zwei:*index* nil) nil)
	    ((< zwei:*index* zwei:*stop-index*) nil)
	    ((eq zwei:*line* zwei:*last-line*)
	     (if (find-file-sheet)
		 (send (find-file-sheet) :delete-stream Self)
	     )
	    )
	    (t (let ((characters (Send **Interval** :Get :Characters)))
		  (if (and (equal 0 characters) (find-file-sheet))
		      (send (find-file-sheet) :add-stream Self)
		      nil
		  )
		  (Send **Interval** :Putprop
			(+ (array-active-length *line*) characters) :Characters
		  )
	       )
	       (Send **Interval** :Putprop
		     (+ 1 (Send **Interval** :Get :Position)) :Position
	       )
	       (Send **Interval** :Putprop
		     (+ 1 (Send **Interval** :Get :Characters)) :Characters
	       )
	    )
      )
  )
)



zwei:
;;;Edited by Acuff                 19 Feb 88  9:32
(defun zwei:all-zmacs-frame-are-at-the-top-level ()
"Check to see whether all Zmacs processes are at the top level.  If they are
 then no interval streams can be active.
"
  (loop for process in si:all-processes do
    (if (equal 'zmacs-window-top-level
	       (catch-error (first (send process :Initial-form)) nil)
	)
	(if (member (send process :WhoState)
		    '("KEYBOARD" "STOP" "ARREST")
		    :Test #'String-Equal
	    )
	    nil
	    (return nil)
	)
	nil
    )
    finally (return t)
  )
)


zwei:
(defmethod (zwei:Interval-Stream :Who-Line-Information) ()
"This is the method which tells the who line about this stream.  It returns
 the name of the stream, the fact that it is an input stream, the character
 position and a percentage count into the stream.
"
  (if (all-zmacs-frame-are-at-the-top-level)
      (send (find-file-sheet) :delete-stream Self)
      nil
  )
  (let ((position (Send **Interval** :Get :Position))
	(length (Send **Interval** :Get :Stream-Length))
	(characters (Send **Interval** :Get :Characters))
       )
       (let ((percent (if (and length (not (zerop length)))
			  (round (* 80. position) length)
			  ;;; This has to be 80. because of some strange 
			  ;;; Base problem.
			  nil
		      )
	     )
	    )
	    (Values (send self :Truename) :input characters percent)
       )
  )
)


;-------------------------------------------------------------------------------

;;; Allow background process display in the who line.

(defun maybe-reset-run-sheet-font
       (&optional (who-sheet who-line-run-state-sheet))
"Resets the font map so that there's a small one in position 1 if there isn't
 one yet.
"
  (let ((map (send who-sheet :font-map)))
       (if (equal (aref map 0) (aref map 1))
	   (send who-sheet :set-font-map
		 (list (font-name (send who-sheet :current-font))
		       *background-process-display-font*
		 )
	   )
	   nil
       )
  )
)

;(maybe-reset-run-sheet-font)

(defun process-interestingness (process)
"Returns a number that says how interesting a process is.  The more interesting
 the higher the number.  Totally boring processes return 0.
"
  (let ((idle (send process :idle-time)))
       (if idle
	   (if (= idle 0)
	       (let ((use (send process :percent-utilization)))
		    (if (> use 5)
			(or use 0)
			0
		    )
	       )
	       0
	   )
	   0
       )
  )
)


(defun find-interesting-process ()
"Finds the most interesting process out of the processes in the system.  Returns
 either the most interesting process or nil if it can't find one.
"
  (let ((best-process nil)
	(best-use 0)
       )
       (loop for p in all-processes do
	     (let ((int (process-interestingness p)))
		  (if (> int best-use)
		      (progn (setq best-use int)
			     (setq best-process p)
		      )
		      nil
		  )
	     )
       )
       best-process
  )
)


(defmethod (si:process :real-idle-time) ()
"Returns the time that a process has been idle in seconds.  :Idle-time doesn't
 work for the current process.
"
  (if (null si:last-time-run)
      nil
      (/ (time-difference (time) si:last-time-run) 60.)
  )
)


(defun run-state-display-function (process)
  (format nil "~A: ~A"
	  (process-whostate process)
	  (the string (send process :name))
  )
)

(defun who-line-run-state-1 (who-sheet)
"A new version of who-line-run-state that allows the display
 of background processes.
"
  (if (and *show-background-processes-in-who-line*
	   (typep current-process 'si:process)
	   (let ((idle (send current-process :real-idle-time)))
		(and (numberp idle)
		     (>= idle *who-line-run-state-toggle-time*)
		)
	   )
	   (equalp who-line-run-state "Keyboard")
      )
      (unwind-protect
	  (let ((interesting-process (find-interesting-process)))
	       (if interesting-process
		   (progn (maybe-reset-run-sheet-font who-sheet)
			  (send who-sheet :set-current-font 1)
			  (who-line-string who-sheet
					   (if (typep interesting-process
						      'si:process
					       )
					       (run-state-display-function
						 interesting-process
					       )
					       interesting-process
					   )
			  )
		   )
		   (who-line-string who-sheet who-line-run-state)
	       )
	  )
	(send who-sheet :set-current-font 0)
      )
      (who-line-string who-sheet who-line-run-state)
  )
)


;;; Replace the old one.
(without-interrupts
  (let ((si:inhibit-fdefine-warnings t))
       (deff who-line-run-state 'who-line-run-state-1)
  )
)

;-------------------------------------------------------------------------------


;;; Allow two files in the who line file sheet.


(DEFFLAVOR WHO-LINE-FILE-SHEET
   ((CURRENT-STREAM (list nil NIL))                   ;The one being displayed.
    ;; This is an array rather than a list to avoid consing.
    (OPEN-STREAMS (MAKE-ARRAY 20. ':TYPE 'ART-Q-LIST ':LEADER-LIST '(0)))
    ;; A list with elements (chaos-connection from-machine contact-name).
    (SERVERS-LIST NIL)
    (DISPLAYED-PERCENT (list nil nil))
    (DISPLAYED-COUNT (list nil nil)))
   (WHO-LINE-MIXIN MINIMUM-WINDOW)
  (:init-keywords :open-streams)
  (:initable-instance-variables
    current-stream displayed-percent displayed-count
  )
  (:gettable-instance-variables
    current-stream displayed-percent displayed-count
  )
)

(defmethod (WHO-LINE-FILE-SHEET :after :init) (plist)
  (if (get plist :open-streams)
      (loop for x in (get plist :open-streams) do
	    (vector-push-extend x open-streams)
      )
      nil
  )
)

(defun the-open-streams (x)
  (if (member :release-6 *features*)
      *open-streams*
      (send x :open-streams)
  )
)

(DEFUN WHO-LINE-FILE-SHEET-COMPUTE-CURRENT-STREAM (&OPTIONAL (UPDATE-P T))
  (declare (:self-flavor WHO-LINE-FILE-SHEET))
  (if *Show-Two-Files-In-Who-Line*
      (DO ((I (1- (ARRAY-LEADER (the-OPEN-STREAMS self) 0)) (1- I))
	   (inPUT-WINNER NIL)
	   (other-inPUT-WINNER NIL)
	   (OUTPUT-WINNER NIL)
	   (other-OUTPUT-WINNER NIL)
	   (STREAM)
	   (DIRECTION))
	  ((MINUSP I)
	   (SETQ CURRENT-STREAM
		 (if (and input-winner OUTPUT-WINNER)
		     (list input-winner OUTPUT-WINNER)
		     (if input-winner
			 (if other-input-winner
			     (list input-winner other-input-winner)
			     (list input-winner nil)
			 )
			 (if (and OUTPUT-WINNER other-output-winner)
			     (list OUTPUT-WINNER other-OUTPUT-WINNER)
			     (list output-winner nil)
			 )
		     )
		 )
	   ))
	(SETQ STREAM (AREF (the-OPEN-STREAMS self) I))
	(MULTIPLE-VALUE-setq (NIL DIRECTION)
	  (FUNCALL STREAM ':WHO-LINE-INFORMATION))
	(case DIRECTION
	  ((:INPUT :BIDIRECTIONAL)
	    (if output-winner
		(RETURN (SETQ CURRENT-STREAM (list STREAM output-winner)))
		(if input-winner
		    (if other-input-winner
			nil
			(setq other-input-winner stream)
		    )
		    (setq input-winner stream)
		)
	    )
	  )
	  (:OUTPUT
	   (if output-winner
	       (if other-output-winner
		   nil
		   (SETQ other-OUTPUT-WINNER STREAM)
	       )
	       (SETQ OUTPUT-WINNER STREAM)))))
      (DO ((I (1- (ARRAY-LEADER (the-OPEN-STREAMS self) 0)) (1- I))
	   (OUTPUT-WINNER NIL) (STREAM) (DIRECTION))
	  ((MINUSP I)
	   (SETQ CURRENT-STREAM (list OUTPUT-WINNER nil)))
	(SETQ STREAM (AREF (the-OPEN-STREAMS self) I))
	(MULTIPLE-VALUE-SETQ (NIL DIRECTION)
			     (FUNCALL STREAM :WHO-LINE-INFORMATION))
	(CASE DIRECTION
	      ((:INPUT :BIDIRECTIONAL)
	       (RETURN (SETQ CURRENT-STREAM (list STREAM nil))))
	      (:OUTPUT
	       (OR OUTPUT-WINNER
		   (SETQ OUTPUT-WINNER STREAM)))))
  )
  (AND UPDATE-P (WHO-LINE-UPDATE))
  current-stream)

(DEFMETHOD (WHO-LINE-FILE-SHEET :DELETE-STREAM) (STREAM &AUX POS)
  (declare (special *OPEN-STREAMS* *CURRENT-STREAM* ))
  (COND ((SETQ POS (POSITION STREAM (THE LIST (G-L-P (the-OPEN-STREAMS self))) :TEST #'EQ))
	 (COND ((= POS (1- (ARRAY-LEADER (the-OPEN-STREAMS self) 0)))
		 (without-interrupts (VECTOR-POP (the-OPEN-STREAMS self))))
	       (T  (without-interrupts (SETF (AREF (the-OPEN-STREAMS self) POS) (VECTOR-POP (the-OPEN-STREAMS self))))))
	 (AND (or (EQ STREAM *CURRENT-STREAM*)
		  (not *current-stream*)
		  (and (consp *current-stream*)
		       (member stream *current-stream*)))
	      (WHO-LINE-FILE-SHEET-COMPUTE-CURRENT-STREAM t)))) ;;; Should this be a T ??? JPR.
)

(DEFMETHOD (WHO-LINE-FILE-SHEET :display-stream-for-who-line)
	   (max-chars stream &optional (home t) (force-p nil))
  (if stream
  (LET ((OLD-STREAM WHO-LINE-ITEM-STATE)
	(PATHNAME) (DIRECTION) (PERCENT) (COUNT))
    (MULTIPLE-VALUE-setq (PATHNAME DIRECTION COUNT PERCENT)
      (FUNCALL STREAM ':WHO-LINE-INFORMATION))
    (if (not (consp displayed-percent)) (setq displayed-percent (list nil nil)))
    (if (not (consp displayed-count))   (setq displayed-count   (list nil nil)))
    (COND ((and (not force-p)
		(or (AND (consp current-stream)
			 (let ((index (position stream current-stream)))
			      (and index
				   (EQ OLD-STREAM current-STREAM)
				   (EQ PERCENT (nth index DISPLAYED-PERCENT))
				   (EQ COUNT   (nth index DISPLAYED-COUNT)))))
		    (AND (EQ OLD-STREAM STREAM)
			 (EQ PERCENT DISPLAYED-PERCENT)
			 (EQ COUNT   DISPLAYED-COUNT))))
	   nil)
	  (T
	   (if home (SHEET-HOME SELF) nil)
;	   (OR (EQ OLD-STREAM STREAM)
	       (SHEET-CLEAR-EOL SELF)
;	       )
	   (SETQ WHO-LINE-ITEM-STATE current-STREAM)
	   (let ((index (position stream current-stream)))
		(setf (nth index DISPLAYED-PERCENT) PERCENT)
		(setf (nth index DISPLAYED-COUNT) COUNT))
	   (DISPLAY-FILE-TRANSFER SELF PATHNAME DIRECTION COUNT
				  PERCENT MAX-CHARS)
	   t)))))


(DEFMETHOD (WHO-LINE-FILE-SHEET :UPDATE)
	   (&AUX (MAX-CHARS (TRUNCATE (SHEET-INSIDE-WIDTH)
				      CHAR-WIDTH))
	    IDLE STRING)
  (COND ((and CURRENT-STREAM (remove nil current-stream))
	 (if (consp current-stream)
	     (if (second current-stream)
		 (progn (SHEET-HOME SELF)
			(let ((modified (send self :display-stream-for-who-line
					      (floor (/ max-chars 2))
					      (first current-stream) nil)))
			  (SHEET-TYO self #\SPACE)
			  (send self :display-stream-for-who-line
				(- (floor (/ max-chars 2)) 1)
				(second current-stream) nil modified)
			)
		 )
		 (send self :display-stream-for-who-line max-chars
		       (first current-stream))
	     )
	     (send self :display-stream-for-who-line max-chars current-stream)))
	((AND (NOT (NULL SERVERS-LIST))
	      (PROGN (PURGE-SERVERS)
		     (NOT (NULL SERVERS-LIST))))
	 (COND ((= (LENGTH SERVERS-LIST) 1)
		(COND ((NEQ WHO-LINE-ITEM-STATE (CAAR SERVERS-LIST))
		       (SHEET-HOME SELF)
		       (SHEET-CLEAR-EOL SELF)
		       (SETQ STRING (FORMAT NIL "~A serving ~A"
					    (CADDAR SERVERS-LIST)
					    (CADAR  SERVERS-LIST)))
		       (SHEET-STRING-OUT SELF STRING 0
					 (MIN (LENGTH (the string STRING))
					      MAX-CHARS))
		       (RETURN-ARRAY (PROG1 STRING (SETQ STRING NIL)))
		       (SETQ WHO-LINE-ITEM-STATE (CAAR SERVERS-LIST)))))
	       ((NEQ WHO-LINE-ITEM-STATE (LENGTH SERVERS-LIST))
		(SHEET-HOME      SELF)
		(SHEET-HOME      SELF)
		(SHEET-CLEAR-EOL SELF)
		(SETQ STRING (FORMAT NIL "~D Active Servers"
				     (LENGTH SERVERS-LIST)))
		(SHEET-STRING-OUT SELF STRING 0
				  (MIN (LENGTH (the string STRING))
				       MAX-CHARS))
		(RETURN-ARRAY (PROG1 STRING (SETQ STRING NIL)))
		(SETQ WHO-LINE-ITEM-STATE (LENGTH SERVERS-LIST)))))
	(SI:WHO-LINE-JUST-COLD-BOOTED-P
	  (COND ((NEQ WHO-LINE-ITEM-STATE 'COLD)
		 (SHEET-CLEAR SELF)
		 (SETQ WHO-LINE-ITEM-STATE 'COLD)
		 (SHEET-STRING-OUT SELF "Cold-booted"))))
	;; Wait for 5 minutes before displaying idle time.
	((>= (SETQ IDLE (TRUNCATE (TIME-DIFFERENCE
				   (TIME)
				   TV:KBD-LAST-ACTIVITY-TIME) 3600.))
	    5)
						;Display keyboard idle time
	 (LET ((OLD-IDLE WHO-LINE-ITEM-STATE))
	   (COND ((OR (NOT (NUMBERP OLD-IDLE)) (NOT (= OLD-IDLE IDLE)))
		  (SHEET-CLEAR SELF)
		  (WITHOUT-INTERRUPTS
		    (LET ((STRING (MAKE-IDLE-MESSAGE IDLE)))
		      (SHEET-STRING-OUT SELF STRING)
		      (RETURN-ARRAY STRING)))
		  (SETQ WHO-LINE-ITEM-STATE IDLE)))))
	((NEQ WHO-LINE-ITEM-STATE 'NULL)
	 (SHEET-CLEAR SELF)
	 (SETQ WHO-LINE-ITEM-STATE 'NULL))))


(defun maybe-remove-old-window (window)
  (send window :kill)
  (set-in-instance (sheet-superior window) 'inferiors (remove window (send (sheet-superior window) :inferiors)))
  (set-in-instance (sheet-superior window) 'exposed-inferiors (remove window (send (sheet-superior window) :exposed-inferiors)))
)

(DEFUN INITIALIZE-STATUS-LINE (&OPTIONAL (INITIAL-CREATION NIL))
  (LET ((CURRENT-CHARACTER-POSITION 0)
	(LAST-CHARACTER-POSITION    0)
	(STATUS-LINE-CHAR-WIDTH   (FONT-CHAR-WIDTH *STATUS-LINE-STANDARD-FONT*))
	(WHO-LINE-HEIGHT            (SHEET-HEIGHT      WHO-LINE-SCREEN))
	(WHO-LINE-LINE-HEIGHT       (SHEET-LINE-HEIGHT WHO-LINE-SCREEN)))
    ;; 18 or 20 characters of the date and time.
    (SETQ LAST-CHARACTER-POSITION
	  ;; The 12 hour clock needs 2 more characters than
	  ;; the 24 hour version to show the AM/PM indicator.
	  (+ 18 (IF (12-HOUR-CLOCK-P) 2. 0.)))
    (IF INITIAL-CREATION
	(SETQ NWATCH-WHO-LINE-SHEET
	      (WHO-LINE-FIELD :FLAVOR 'WHO-LINE-SHEET
			      :Name "Nwatch"
			      :VSP 0
			      :WHO-LINE-UPDATE-FUNCTION
			      (IF (12-HOUR-CLOCK-P)
				  'NWATCH-WHO-FUNCTION-12
				  'NWATCH-WHO-FUNCTION-24)
			      :HEIGHT WHO-LINE-LINE-HEIGHT
			      :LEFT 0
			      :RIGHT (* LAST-CHARACTER-POSITION
					STATUS-LINE-CHAR-WIDTH)
			      :BOTTOM WHO-LINE-HEIGHT))
	;;ELSE
	(PROGN
	  ;; Only update those things that can change.
	  (SEND NWATCH-WHO-LINE-SHEET :SET-SIZE
		(* LAST-CHARACTER-POSITION STATUS-LINE-CHAR-WIDTH)
		WHO-LINE-LINE-HEIGHT)
	  (SEND NWATCH-WHO-LINE-SHEET :SET-WHO-LINE-UPDATE-FUNCTION
		(IF (12-HOUR-CLOCK-P)
		    'NWATCH-WHO-FUNCTION-12
		    'NWATCH-WHO-FUNCTION-24))
	  ;; We need to clear out the extra state so that it gets initialized
	  ;; in the nwatch function to its proper value.
	  (SEND NWATCH-WHO-LINE-SHEET :SET-WHO-LINE-EXTRA-STATE NIL)))
    (SETQ CURRENT-CHARACTER-POSITION LAST-CHARACTER-POSITION)
    (SETQ LAST-CHARACTER-POSITION
	  (+ LAST-CHARACTER-POSITION
	     (if *Show-Two-Files-In-Who-Line*
		 (+ 1 (array-total-size user-id))
		 13.
	     ))) ;;; JPR
    ;; Length of user-id + 1 characters of user id or process.
    (IF INITIAL-CREATION
	(SETQ USER-ID-WHO-LINE-SHEET
	      (WHO-LINE-FIELD :FLAVOR 'WHO-LINE-SHEET
			      :Name "User Id"
			      :VSP 0
			      :WHO-LINE-UPDATE-FUNCTION
			      'WHO-LINE-USER-OR-PROCESS
			      :HEIGHT WHO-LINE-LINE-HEIGHT
			      :LEFT (* CURRENT-CHARACTER-POSITION
				       STATUS-LINE-CHAR-WIDTH)
			      :RIGHT (* LAST-CHARACTER-POSITION
					STATUS-LINE-CHAR-WIDTH)
			      :BOTTOM WHO-LINE-HEIGHT))
	;;ELSE
	;; Only the position can change; the size remains constant.
	(SEND USER-ID-WHO-LINE-SHEET :SET-POSITION
	      (* CURRENT-CHARACTER-POSITION STATUS-LINE-CHAR-WIDTH)
	      (- WHO-LINE-HEIGHT WHO-LINE-LINE-HEIGHT)))
    (SETQ CURRENT-CHARACTER-POSITION LAST-CHARACTER-POSITION)
    (SETQ LAST-CHARACTER-POSITION
	  (+ LAST-CHARACTER-POSITION
	     (if *Show-Two-Files-In-Who-Line*
		 *Chars-for-who-line-package*
		 10.))) ;;; JPR
    ;; *Chars-for-who-line-package* characters of package.
    (IF INITIAL-CREATION
	(SETQ CURRENT-PACKAGE-WHO-LINE-SHEET
	      (WHO-LINE-FIELD :FLAVOR 'WHO-LINE-SHEET
			      :Name "Current Package"
			      :VSP 0
			      :WHO-LINE-UPDATE-FUNCTION 'WHO-LINE-PACKAGE
			      :HEIGHT WHO-LINE-LINE-HEIGHT
			      :LEFT (* CURRENT-CHARACTER-POSITION
				       STATUS-LINE-CHAR-WIDTH)
			      :RIGHT (* LAST-CHARACTER-POSITION
					STATUS-LINE-CHAR-WIDTH)
			      :BOTTOM WHO-LINE-HEIGHT))
	;;ELSE
	;; Only the position can change; the size remains constant.
	(SEND CURRENT-PACKAGE-WHO-LINE-SHEET :SET-POSITION 
	      (* CURRENT-CHARACTER-POSITION STATUS-LINE-CHAR-WIDTH)
	      (- WHO-LINE-HEIGHT WHO-LINE-LINE-HEIGHT)))
    (SETQ CURRENT-CHARACTER-POSITION LAST-CHARACTER-POSITION)
    (SETQ LAST-CHARACTER-POSITION
	  (+ LAST-CHARACTER-POSITION
	     (if *Show-Two-Files-In-Who-Line*
		 *Chars-for-who-line-process-state*
		 21.))) ;;; JPR
    ;; *Chars-for-who-line-process-state* characters of process state.
    (IF INITIAL-CREATION
	(SETQ WHO-LINE-RUN-STATE-SHEET
	      (WHO-LINE-FIELD :FLAVOR 'WHO-LINE-SHEET
			      :Name "Run State"
			      :VSP 0
			      :WHO-LINE-UPDATE-FUNCTION 'WHO-LINE-RUN-STATE
			      :LEFT (* CURRENT-CHARACTER-POSITION
				       STATUS-LINE-CHAR-WIDTH)
			      :RIGHT (* LAST-CHARACTER-POSITION
					STATUS-LINE-CHAR-WIDTH)
			      :HEIGHT WHO-LINE-LINE-HEIGHT
			      :BOTTOM WHO-LINE-HEIGHT))
	;;ELSE
	;; Only the position can change; the size remains constant.
	(SEND WHO-LINE-RUN-STATE-SHEET :SET-POSITION 
	      (* CURRENT-CHARACTER-POSITION STATUS-LINE-CHAR-WIDTH)
	      (- WHO-LINE-HEIGHT WHO-LINE-LINE-HEIGHT)))
    (SETQ CURRENT-CHARACTER-POSITION LAST-CHARACTER-POSITION)
    (SETQ LAST-CHARACTER-POSITION
	  (+ LAST-CHARACTER-POSITION
	     (if *Show-Two-Files-In-Who-Line*
		 (- 60. (array-total-size user-id))
		 36.))) ;;; JPR
    ;; The remaining characters go to the file/idle/boot state.
    (IF INITIAL-CREATION
	(let ((old (if (and (boundp 'who-line-file-state-sheet)
			    (typep who-line-file-state-sheet
				   'who-line-file-sheet
			    )
		       )
		       who-line-file-state-sheet
		       nil
		   )
	      )
	     )
	     (let ((current-stream
		     (or (and old (send old :current-stream)    (list nil nil)))
		   )
		   (displayed-percent
		     (or (and old (send old :displayed-percent) (list nil nil)))
		   )
		   (displayed-count
		     (or (and old (send old :displayed-count)   (list nil nil)))
		   )
		  )
		  (SETQ WHO-LINE-FILE-STATE-SHEET
			(WHO-LINE-FIELD :FLAVOR 'WHO-LINE-FILE-SHEET
					:Name "File State"
					;;; JPR.
					:open-streams
					(if (and old (not (member :release-6 *features*)))
					    (send old :open-streams)
					    nil
					)
					:current-stream current-stream
					:displayed-percent displayed-percent
					:displayed-count displayed-count
					:VSP 0
					:LEFT (* CURRENT-CHARACTER-POSITION
						 STATUS-LINE-CHAR-WIDTH)
					:RIGHT MAIN-SCREEN-WIDTH
					:HEIGHT WHO-LINE-LINE-HEIGHT
					:BOTTOM WHO-LINE-HEIGHT))
		  (if (member :release-6 *features*)
		      (setf (array-leader (the-open-streams WHO-LINE-FILE-STATE-SHEET) 0) 0)
		      nil
		  ))
	     (if old (progn (send old :delete-all-streams)
			    (send who-line-file-state-sheet :update))))
	;;ELSE
	;; Only the left edge will change, but the only way
	;; we can do that is to respecify all of the edges.
	(SEND WHO-LINE-FILE-STATE-SHEET :SET-EDGES
	      (* CURRENT-CHARACTER-POSITION STATUS-LINE-CHAR-WIDTH)
	      (- WHO-LINE-HEIGHT WHO-LINE-LINE-HEIGHT)
	      MAIN-SCREEN-WIDTH
	      WHO-LINE-HEIGHT))))


(defun clear-all-symbols (name)
  (let ((symbols (let ((*standard-output* 'si:null-stream))
		      (declare (special *standard-output*))
		      (where-is name)
		 )
	)
       )
       (mapc #'(lambda (symbol) (unintern symbol (symbol-package symbol)))
	     symbols
       )
       nil
  )
)

(clear-all-symbols "RESET-WHO-LINE")

(DEFMETHOD (screen :putprop) (value key)
  (setf (get (LOCF PROPERTY-LIST) key) value)
)


(defun reset-who-line ()
"Change format of the who line if necessary."
 (without-interrupts
  (if *Show-Two-Files-In-Who-Line*
      (setq *status-line-standard-font* fonts:tvfont)
      (setq *status-line-standard-font* fonts:cptfont)
  )
  (if (intersection '(:release-4 :release-5) *features*)
      (setq tv:*screens-exposed-at-disk-save*
	    (remove-if #'who-line-screen-p
		       tv:*screens-exposed-at-disk-save*
	    )
      )
      nil
  )
  (if (member :release-6 *features*)
      (progn (maybe-remove-old-window NWATCH-WHO-LINE-SHEET)
	     (maybe-remove-old-window USER-ID-WHO-LINE-SHEET)
	     (maybe-remove-old-window CURRENT-PACKAGE-WHO-LINE-SHEET)
	     (maybe-remove-old-window WHO-LINE-RUN-STATE-SHEET)
	     (maybe-remove-old-window WHO-LINE-FILE-STATE-SHEET)
	     (initialize-status-line t)
      )
      (progn (setq who-line-screen nil)
	     (let ((old #'define-screen))
		  (letf ((#'define-screen
			  #'(lambda (flavor ignore &rest args)
			      (apply old flavor
				     (string-append "Who Line Screen" "-"
						    (symbol-name (gensym ""))
				     )
				     args
			      )
			    )
			 )
			)
			(who-line-setup)
		  )
	     )
      )
  )
  (if (intersection '(:release-4 :Release-5) *features*)
      (progn (if (intersection '(:Release-5) *features*)
		 (setf (tv:screen-screens-who-line-screen *initial-screen*)
		       who-line-screen
		 )
		 (mapcar #'(lambda (x)
			     (send x :set-screen-descriptor who-line-screen)
			   )
			   all-the-screens
		 )
	     )
	     (setq tv:*screens-exposed-at-disk-save*
		   (cons who-line-screen tv:*screens-exposed-at-disk-save*)
	     )
      )
      nil
  )
 )
)

(export 'reset-who-line 'tv)
(reset-who-line)
