;;;-*- Mode:Common-Lisp; Package:Compiler; Base:10 -*-

;;; **********************************************************************
;;; Copyright (c) 1990 Stanford University.
;;; Copyright is held by Stanford University except where code has been
;;; modified from TI source code.  In these cases TI code is marked with
;;; a suitable comment.  Where functionality implemented herein replicates
;;; similarly named functionality on Symbolics machines, this code was
;;; developed solely from the interface specification in the documentation
;;; or through guesswork, never by examination of Symbolics source code.

;;; All Stanford Copyright code is in the public domain.  This code may be
;;; distributed and used without restriction as long as this copyright
;;; notice is included and no fee is charged.  This can be thought of as
;;; being equivalent to the Free Software Foundation's Copyleft policy.

;;; TI source code may only be distributed to users who hold valid TI
;;; software licenses.
;;; **********************************************************************

;;; The mods in this file implement Compiler-Letf, which is a mixture of Letf
;;; and Compiler-Let.

(defun compiler-letf (bindings body)
  (ignore bindings body)
  (ferror nil "Shouldn't be called.")
)

(defun do-compiler-letf (bindings body)
  (let ((function `(lambda () (letf ,bindings ,body))))
       (funcall (compile nil function))
  )
)


;;; TI code.  Mods marked by JPR.
(DEFUN COMPILE-DRIVER (OFORM PROCESS-FN OVERRIDE-FN &OPTIONAL COMPILE-TIME-TOO (TOP-LEVEL-P T))
  ;;  8/01/84 DNG - updated from MIT patches 98.40 and 98.57.
  ;; 12/26/84 DNG - Save value of DEFCONSTANT in FILE-CONSTANTS-LIST.
  ;;  1/18/85 DNG - Use COMPILE-PROCLAIM.
  ;;  2/20/85 DNG - Evaluate saved value of DEFCONSTANT.
  ;; 10/23/85 DNG - Fix handling of top-level COMPILER-LET so that the bindings
  ;;		    are implicitely special.  [SPR 837]
  ;;  1/16/86 DNG - Give warning on obsolete DEFUN syntax.
  ;;  1/27/86 DNG - Do style checking on random top-level forms.
  ;;  3/03/86 DNG - Fix so that an IMPORT within an EVAL-WHEN is fasdumped
  ;;		before being evaluated [SPR 1204]; bind *EVALHOOK* to
  ;;		#'EVAL-FOR-TARGET around macro expansion to use target definitions.
  ;;  3/18/86 DNG - Call CHECK-USED-BEFORE-DEFINED for DEFF-MACRO.
  ;;  5/19/86 DNG - Add special handling for EXPORT, IMPORT, etc. in cold-load.
  ;;  6/24/86 DNG - Fix to recognize PATCH-SOURCE-FILE in COMPILER package instead of COMPILER2.
  ;;  7/25/86 DNG -
  ;;  7/30/86 DNG - Evaluate COMPILATION-DEFINE at both compile and load time; always
  ;;		try to evaluate the value of a DEFCONSTANT at compile time.
  ;;  8/07/86 DNG - Major changes to minimize differences between top-level forms and functions.
  ;;  8/15/86 DNG - Don't optimize when an override function is given [ie, eval buffer].
  ;;  9/26/86 DNG - Added call to OBJECT-OPERATION-WITH-WARNINGS .
  ;; 11/21/86 DNG - Don't establish warnings context for a DEFPROP.
  ;;  2/11/87 DNG - Fix to not error on name starting with #\D but less that 3 characters.
  "Compile or evaluate a top-level form from a file or buffer."
  (WHEN (AND COMPILER-WARNINGS-CONTEXT
	     (NULL SI:OBJECT-WARNINGS-OBJECT-NAME)
	     (CONSP OFORM)
	     (SYMBOLP (FIRST OFORM))
	     (CADR-SAFE OFORM)
	     (SYMBOLP (SECOND OFORM))
	     (LET ((NAME (SYMBOL-NAME (FIRST OFORM))))
	       (AND (>= (LENGTH NAME) 3)
		    (CHAR= (CHAR NAME 0) #\D)
		    (CHAR= (CHAR NAME 1) #\E)
		    (CHAR= (CHAR NAME 2) #\F)))
	     (NOT (EQ (FIRST OFORM) 'DEFPROP)))
    ;; A definition form that ZMACS knows how to find, so use it as a reference point
    ;; for reporting any errors within it.
    (RETURN-FROM COMPILE-DRIVER
      (OBJECT-OPERATION-WITH-WARNINGS ((SECOND OFORM))
	(COMPILE-DRIVER OFORM PROCESS-FN OVERRIDE-FN COMPILE-TIME-TOO TOP-LEVEL-P))))
  (LET ((FORM OFORM))
    (WHEN (AND OVERRIDE-FN
	       (FUNCALL OVERRIDE-FN FORM))
      (RETURN-FROM COMPILE-DRIVER NIL))
    (LET ((MACRO-CONS-AREA DEFAULT-CONS-AREA)
	  (P1VALUE 'TOP-LEVEL-FORM))
      (SETQ FORM (PRE-OPTIMIZE FORM T OVERRIDE-FN))) ; check style, expand macros, and optimize
    (WHEN (AND OVERRIDE-FN
	       (NOT (EQ FORM OFORM))
	       (FUNCALL OVERRIDE-FN FORM))
      (RETURN-FROM COMPILE-DRIVER NIL))

    (IF (ATOM FORM)
	(FUNCALL PROCESS-FN FORM 'RANDOM)
      ;; If this was a top-level macro, supply a good guess
      ;; for the function-parent for any DEFUNs inside the expansion.
      (LET ((LOCAL-DECLARATIONS LOCAL-DECLARATIONS)
	    (FN (FIRST FORM)))
	(COND ((AND (NEQ FORM OFORM) (SYMBOLP (CADR OFORM)))
	       (PUSH `(FUNCTION-PARENT ,(CADR OFORM) ,(CAR OFORM))
		     LOCAL-DECLARATIONS)) )
	(COND ((EQ FN 'EVAL-WHEN)
	       (LET ((TIMES (SECOND FORM)))
		 (UNLESS (AND (LISTP TIMES)
			      (LOOP FOR TIME IN TIMES
				    ALWAYS (MEMBER TIME '(GLOBAL:EVAL LOAD COMPILE CLI:EVAL
								      #+compiler:debug Lisp:compile)
						   :TEST #'EQ)))
		   (WARN 'EVAL-WHEN ':IMPOSSIBLE "~S invalid EVAL-WHEN times;
must be a list of EVAL, LOAD, and/or COMPILE."
			 TIMES))
		 (LET* ((COMPILE (OR (MEMBER 'COMPILE TIMES :TEST #'EQ)
				     #+compiler:debug
				     (MEMBER 'Lisp:COMPILE TIMES :TEST #'EQ)))
			(LOAD (MEMBER 'LOAD TIMES :TEST #'EQ))
			(EVAL (OR (MEMBER 'GLOBAL:EVAL TIMES :TEST #'EQ) 
				  (MEMBER 'CLI:EVAL TIMES :TEST #'EQ)))
			(EVAL-NOW (AND (OR COMPILE (AND COMPILE-TIME-TOO EVAL)) T)))
		   (DOLIST (FORM1 (CDDR FORM))
		     (IF LOAD
			 (COMPILE-DRIVER FORM1 PROCESS-FN OVERRIDE-FN EVAL-NOW NIL)
		       (IF EVAL-NOW
			   (FUNCALL PROCESS-FN FORM1 'DECLARE)
			 (RETURN) ))))))
	      ((EQ FN 'WITH-SELF-ACCESSIBLE) ; Why is this here???
	       (MAPC #'(LAMBDA (FORM)
			 (COMPILE-DRIVER FORM PROCESS-FN OVERRIDE-FN COMPILE-TIME-TOO NIL))
		     (CDDR FORM)))
	      ((EQ FN 'PROGN)
	       (MAPC #'(LAMBDA (FORM)
			 (COMPILE-DRIVER FORM PROCESS-FN OVERRIDE-FN COMPILE-TIME-TOO TOP-LEVEL-P))
		     (CDR FORM)))
	      ((AND (OR TOP-LEVEL-P COMPILE-TIME-TOO)
		    (MEMBER FN '(SPECIAL UNSPECIAL COMPILATION-DEFINE 
				 MAKE-PACKAGE IN-PACKAGE SHADOW SHADOWING-IMPORT
				 EXPORT UNEXPORT USE-PACKAGE UNUSE-PACKAGE IMPORT
				 REQUIRE)
			    :TEST #'EQ))
	       (COND ((AND SI:FILE-IN-COLD-LOAD
			   (MEMBER FN '(EXPORT UNEXPORT IMPORT SHADOWING-IMPORT SHADOW
					USE-PACKAGE UNUSE-PACKAGE)
				   :TEST #'EQ)
			   (EQL (LENGTH FORM) 2))
		      ;; For cold-load files, these operations need an explicit package
		      ;; argument because we can't be sure what *PACKAGE* will be at the
		      ;; time the form is actually executed.
		      (SETQ FORM (LIST (FIRST FORM) (SECOND FORM) (PACKAGE-NAME *PACKAGE*))))
		     )
	       (FUNCALL PROCESS-FN FORM 'SPECIAL))
	      ((EQ FN 'DECLARE)
	       (COMPILE-DECLARE (CDR FORM) PROCESS-FN))
	      ((EQ FN 'PROCLAIM)
	       (COMPILE-PROCLAIM (CDR FORM) PROCESS-FN))
	      ((EQ FN 'COMMENT) NIL)
	      ((EQ FN 'COMPILER:PATCH-SOURCE-FILE)
	       (COMPILE-DRIVER `(EVAL-WHEN (LOAD EVAL)
				  (SETQ SI:PATCH-SOURCE-FILE-NAMESTRING ,(CADR FORM)))
			       PROCESS-FN OVERRIDE-FN COMPILE-TIME-TOO TOP-LEVEL-P)
	       (MAPC #'(LAMBDA (FORM)
			 (COMPILE-DRIVER FORM PROCESS-FN OVERRIDE-FN COMPILE-TIME-TOO TOP-LEVEL-P))
		     (CDDR FORM))
	       (COMPILE-DRIVER `(EVAL-WHEN (LOAD EVAL)
				  (SETQ SI:PATCH-SOURCE-FILE-NAMESTRING NIL))
			       PROCESS-FN OVERRIDE-FN COMPILE-TIME-TOO TOP-LEVEL-P))
	      ((EQ FN 'COMPILER-LET)
	       (*EVAL `(COMPILER-LET ,(CADR FORM)
			 (COMPILE-DRIVER '(PROGN . ,(CDDR FORM))
					 ',PROCESS-FN ',OVERRIDE-FN
					 ',COMPILE-TIME-TOO
					 ',TOP-LEVEL-P))))
	      ((EQ FN 'COMPILER-LETF)  ;;; JPR.
	       (do-compiler-letf (second form)
		 `(COMPILE-DRIVER '(PROGN . ,(CDDR FORM))
				  ',PROCESS-FN ',OVERRIDE-FN
				  ',COMPILE-TIME-TOO
				  ',TOP-LEVEL-P)))
	      (COMPILE-TIME-TOO		   ; EVAL-WHEN (COMPILE LOAD) 
	       (FUNCALL PROCESS-FN FORM 'MACRO))
	      (T			   ; EVAL-WHEN (LOAD)
	       (FUNCALL PROCESS-FN FORM 'RANDOM))
	      ))))
  NIL)


;;; TI code.  Mods marked by JPR.
(DEFUN P1 (ORIGINAL-FORM &OPTIONAL DONT-OPTIMIZE)
  "Pass 1 compilation of a single Lisp form."
  ;; 12/27/84 - Improve EXPRESSION-SIZE update.
  ;; 12/28/84 - Don't increment use count of ignored variable.
  ;; 12/29/84 - Do increment use count of propagated variable.
  ;;  1/19/85 - NOTINLINE declaration forces call instead of 
  ;;		machine instruction and prevents DEFSUBST expansion.
  ;;  1/23/85 - Add check for cold load files.
  ;;  1/24/85 - Add use of P1-WITH-ANNOTATION.
  ;;  2/20/85 - Suppress constant folding on dead code.
  ;;  8/27/85 - Suprress T.R.E. on function defined by Misc-op.
  ;;  2/21/86 - Enable first arg of FUNCALL to be ephemeral closure.
  ;;  5/07/86 - Do NIL ==> (QUOTE NIL) without consing.
  ;;  6/16/86 - Check for higher level lexical variable before DEFCONSTANT to
  ;;		allow local shadowing with UNSPECIAL declaration. [SPR 2413]
  ;;  6/20/86 - Call EXPAND-LAMBDA directly instead of using P1LAMBDA.
  ;;  6/25/86 - Fix to handle (FUNCALL '#<DTP-FUNCTION ...> ...).
  ;;  7/02/86 - Change handling of non-local lexical variables.
  ;;  7/10/86 - Set SPECIAL-VAR-BIT in USED-VAR-SET on reference to free
  ;;		special variable; provide for inline expansion of local functions.
  ;;  7/17/86 - Allow inline expansion of local functions.
  ;;  7/25/86 - More changes for non-local variables.
  ;;  8/28/86 - Call to p1argc no longer passes result of getargdesc - just pass form
  ;;  9/09/86 - Increment use count of propagated BREAKOFF-FUNCTION.
  ;;  9/15/86 - Call MAYBE-INTEGRATE after POST-OPTIMIZE instead of before.
  ;;  9/16/86 - Record side-effects for arbitrary function calls.
  ;;  9/18/86 - Use FIX-FUNCALL-EVALUATION-ORDER on FUNCALL forms.
  ;;  9/20/86 - Add special handling for COMPILER-LET.
  ;;  9/24/86 - Pass saved ALLVARS as second arg to FIX-FUNCALL-EVALUATION-ORDER .
  ;; 10/18/86 - Permit tail recursion elimination of local functions.
  ;; 11/14/86 - Don't count BLOCK-FOR-PROG in EXPRESSION-SIZE.
  ;;  7/07/87 - Special handling for constants evaluated at load time. [SPR 4918]
  ;;  9/28/87 - Modified for Scheme. [Not included in this file until 3/15/89.]
  ;; 10/02/87 - Tail Recursion Elimination is always enabled in Scheme mode.
  ;;		Don't add special variable to FREEVARS when value is not being used.
  ;; 10/14/87 - Fixed bug in 9/28 change.
  ;; 11/14/87 - Add support for SCHEME:DEFINE-INTEGRABLE .
  ;;		Permit a FEF object to appear as the CAR of a form.
  ;; 11/21/87 - Permit keywords to be used as variable names in Scheme mode.
  ;; 12/19/87 - Fix use of symbol defined by SCHEME:DEFINE-INTEGRABLE in 
  ;;		function position.  Inline expansion of FUNCALL of a breakoff
  ;;		function.  Modified to facilitate tail recursion elimination on LETREC functions.
  ;;  1/09/88 - Add use of SCHEME:PCS-INTEGRATE-T-AND-NIL.
  ;;  2/10/88 - Add inherited vars argument to TAIL-RECURSION-ELIMINATION. [SPR 7113]
  ;; 12/16/88 - Fix to not optimize (FUNCALL 'symbol ...) when it has the same 
  ;;		name as a local function.
  ;;  4/22/89 - Update and uncomment the support for PCS-INTEGRATE-T-AND-NIL.
  ;;  4/25/89 - Add setting of COMPILAND-CONSTANTS-EXPANDED for SPR 6501.
  (DECLARE (OPTIMIZE (SAFETY 0) (SPEED 2)))
  (LET (FORM TM NEW-SIZE NEW-FORM INDECL HANDLER)
    (IF (ATOM ORIGINAL-FORM)
	(SETQ FORM ORIGINAL-FORM)
      (IF (AND (COMPILING-SCHEME-P)
	       (TYPECASE (CAR ORIGINAL-FORM)
		 ( SYMBOL (IF (LOOKUP-VAR (CAR ORIGINAL-FORM) VARS)
			      (NOT (ASSOC (CAR ORIGINAL-FORM) LOCAL-FUNCTIONS :TEST #'EQ))
			    (NOT (OR (FBOUNDP (CAR ORIGINAL-FORM))
				     (EQ (GET (CAR ORIGINAL-FORM) 'INTEGRABLE '|<Undefined>|)
					 '|<Undefined>|)))) )
		 ( CONS (NOT (MEMBER (CAAR ORIGINAL-FORM) SI:FUNCTION-START-SYMBOLS :TEST #'EQ)))
		 ( T T)))
	  (SETQ FORM (CONS 'FUNCALL ORIGINAL-FORM))
	(PROGN
	  (WHEN (ATOM (CAR ORIGINAL-FORM))
	    (SETQ INDECL (INLINE-DECL (CAR ORIGINAL-FORM))) )
	  (SETQ FORM (PRE-OPTIMIZE ORIGINAL-FORM T
				   (OR DONT-OPTIMIZE
				       (AND (EQ INDECL 'NOTINLINE)
					    (NULL (GETL (CAR ORIGINAL-FORM)
							'(P1 P2))) ) ) ))
	  (WHEN (AND (NOT (EQ FORM ORIGINAL-FORM))
		     (CONSP FORM)
		     (NOT (SYMBOLP (CAR FORM)))
		     (COMPILING-SCHEME-P))
	    (SETQ FORM (CONS 'FUNCALL FORM)))
	  ) ) )
    (SETQ NEW-SIZE (+ EXPRESSION-SIZE 1-IF-LIVE-CODE))
    (COND
      ((ATOM FORM)
       (SETQ EXPRESSION-SIZE NEW-SIZE)
       (RETURN-FROM P1
	 (COND ((EQ FORM 'NIL) '(QUOTE NIL)) ; avoid consing for this common special case
	       ((EQ FORM 'T)   '(QUOTE T))
	       ((OR (NOT (SYMBOLP FORM))
		    (AND (KEYWORDP FORM) (NOT (COMPILING-SCHEME-P))))
		(LIST 'QUOTE FORM))	  ; constant other than a DEFCONSTANT
	       ((SETQ TM (LOOKUP-VAR FORM VARS)) ; found in table of local variables
		(IF (AND (NOT P1VALUE) (NOT DONT-OPTIMIZE))
		    ;; The value is not being used, so the reference is
		    ;; expected to be deleted by later optimizations.
		    ;; Don't increment the variable's use count and just
		    ;; return a dummy placeholder.
		    (PROGN (WHEN (NULL (VAR-USE-COUNT TM))
			     (SETF (VAR-USE-COUNT TM) 0))
			   '(QUOTE |<unused_var>|))
		  (PROGN ; a genuine variable reference
		    (SETQ NEW-FORM (VAR-LAP-ADDRESS TM))
		    (IF (AND (CONSP NEW-FORM)
			     (EQ (CAR NEW-FORM) 'LOCAL-REF))
			(IF (AND (LOGTEST (CDDR NEW-FORM) PROPAGATE-VAR-SET)
				 PROPAGATE-ENABLE )
			    (PROGN (SETQ NEW-FORM (VAR-INIT-FORM TM))
				   (COND ((NULL NEW-FORM)
					  (SETQ NEW-FORM '(QUOTE NIL)))
					 ((ATOM NEW-FORM))
					 ((EQ (CAR NEW-FORM) 'LOCAL-REF)
					  (VAR-INCREMENT-USE-COUNT (SECOND NEW-FORM))
					  (SETQ USED-VAR-SET
						(LOGIOR USED-VAR-SET (CDDR NEW-FORM))))
					 ((EQ (CAR NEW-FORM) 'BREAKOFF-FUNCTION)
					  (INCF (COMPILAND-USE-COUNT (SECOND NEW-FORM))))
					 (T (DEBUG-ASSERT (NO-SIDE-EFFECTS-P NEW-FORM))))
				   (WHEN (NULL (VAR-USE-COUNT TM))
				     (SETF (VAR-USE-COUNT TM) 0))
				   (RETURN-FROM P1 NEW-FORM))
			  (PROGN
			    (UNLESS (OR (NULL *VAR-LEVEL-COUNTS*)
					(ZEROP 1-IF-LIVE-CODE))
			      (LET (( VC (VAR-COMPILAND TM) ))
				(UNLESS (EQ VC *CURRENT-COMPILAND*)
				  (INCF (NTH (COMPILAND-NESTING-LEVEL VC)
					     *VAR-LEVEL-COUNTS*)
					(LOOP-WEIGHTED-INCREMENT *LOOP-LEVEL*)
				    ))))
			    (SETQ USED-VAR-SET (LOGIOR USED-VAR-SET (CDDR NEW-FORM)))
			    ))
		      (WHEN (SYMBOLP NEW-FORM)
			(WHEN (OR (EQ (VAR-KIND TM) 'FEF-ARG-FREE)
				  (NEQ (VAR-COMPILAND TM) *CURRENT-COMPILAND*))
			  (UNLESS (ZEROP 1-IF-LIVE-CODE)
			    (PUSHNEW NEW-FORM FREEVARS :TEST 'EQ) ) )
			(UNLESS (LOGTEST SPECIAL-VAR-BIT USED-VAR-SET)
			  (SETF USED-VAR-SET (LOGIOR USED-VAR-SET SPECIAL-VAR-BIT)))))
		    (VAR-INCREMENT-USE-COUNT TM)
		    NEW-FORM) ))
	       ((AND SELF-FLAVOR-DECLARATION
		     (TRY-REF-SELF FORM)))
	       ((AND (COMPILING-SCHEME-P)
		     (OR (FBOUNDP FORM)
			 (UNLESS (EQ (SETQ TM (GET FORM 'INTEGRABLE '|<Undefined>|))
				     '|<Undefined>|)
			   (PUSHNEW FORM MACROS-EXPANDED :TEST #'EQ)
			   (RETURN-FROM P1 (P1 TM DONT-OPTIMIZE)))
			 (WHEN (EQ (SYMBOL-PACKAGE FORM) *KEYWORD-PACKAGE*)
			   (RETURN-FROM P1 (LIST 'QUOTE FORM)))
			 (NOT (SPECIALP FORM T))))
		(LOCALLY ;; The values of the these are assigned when the Scheme system is loaded.
		  (declare (special PCS-INTEGRATE-T-AND-NIL SCHEME-T SCHEME-NIL))
		  (COND ((AND (EQ FORM SCHEME-T) PCS-INTEGRATE-T-AND-NIL)
			 '(QUOTE T))
			((AND (EQ FORM SCHEME-NIL) PCS-INTEGRATE-T-AND-NIL)
			 '(QUOTE NIL))
			(T (UNLESS (LOGTEST SPECIAL-VAR-BIT USED-VAR-SET)
			     (SETF USED-VAR-SET (LOGIOR USED-VAR-SET SPECIAL-VAR-BIT)))
			   `(FUNCTION ,FORM)))))
	       ((BLOCK CONSTANT?
		  (AND (< (OPT-SAFETY OPTIMIZE-SWITCH) 2)
		       (NOT DONT-OPTIMIZE)
		       (LET ( CONST )
			 (COND ((SETQ CONST (ASSOC FORM FILE-CONSTANTS-LIST :TEST #'EQ))
				(SETQ TM (CDR CONST)) )
			       ((AND (SETQ CONST (GET-FOR-TARGET FORM 'SYSTEM-CONSTANT))
				     (NOT (EQ CONST 'COMPILER:QC-PROCESS-INITIALIZE))
				     ;; DEFCONSTANT, not a machine-dependent constant
				     (BOUNDP-FOR-TARGET FORM))
				(SETQ TM (SYMEVAL-FOR-TARGET FORM)) )
			       (T (RETURN-FROM CONSTANT? NIL)) )
			 (OR (NUMBERP TM)
			     (SYMBOLP TM)
			     (CHARACTERP TM) ) ) ) )
		(SETF (GETF (COMPILAND-CONSTANTS-EXPANDED *CURRENT-COMPILAND*) FORM) TM)
		(LIST 'QUOTE TM))
	       (T (IF P1VALUE
		      (PROGN (MAKESPECIAL FORM)
			     (UNLESS (LOGTEST SPECIAL-VAR-BIT USED-VAR-SET)
			       (SETF USED-VAR-SET (LOGIOR USED-VAR-SET SPECIAL-VAR-BIT))))
		    (LET ((FREEVARS FREEVARS)) 
		      (MAKESPECIAL FORM)))
		  FORM))))
      ((EQ (CAR FORM) 'QUOTE)
       (SETQ EXPRESSION-SIZE NEW-SIZE)
       (RETURN-FROM P1 (IF (AND QC-FILE-IN-PROGRESS
				(NOT QC-FILE-LOAD-FLAG)
				(CONSP (SECOND FORM))
				(LOAD-TIME-EVAL-P (SECOND FORM) 0) )
			   `(QUOTE-LOAD-TIME-EVAL ,FORM) ; hide the value from optimization
			 FORM)))
      ;; Certain constructs must be checked for here
      ;; so we can call P1 recursively without setting TLEVEL to NIL.
      ((NOT (ATOM (CAR FORM)))
       (LET ((FCTN (CAR FORM)))
	 (UNLESS (SYMBOLP (CAR FCTN))
	   (WARN 'BAD-FUNCTION-CALLED ':IMPOSSIBLE
		 "There appears to be a call to a function whose CAR is ~S."
		 (CAR FCTN)))
	 (COND ((MEMBER (CAR FCTN)
			'(GLOBAL:LAMBDA GLOBAL:NAMED-LAMBDA CLI:LAMBDA NAMED-LAMBDA)
			:TEST #'EQ)
		;;added extra arg to expand lambda to indicate that args not processed
		(RETURN-FROM P1
		  (P1 (EXPAND-LAMBDA FCTN (CDR FORM) NIL nil)) ))
	       (T ;; Old Maclisp evaluated functions.
		(WARN 'EXPRESSION-AS-FUNCTION ':VERY-OBSOLETE
		      "The expression ~S is used as a function; use FUNCALL."
		      (CAR FORM))
		(RETURN-FROM P1 (P1 `(FUNCALL . ,FORM)))))))
      ((NOT (SYMBOLP (CAR FORM)))
       (WARN 'BAD-FUNCTION-CALLED ':IMPOSSIBLE
	     "~S is used as a function to be called." (CAR FORM))
       (RETURN-FROM P1 (P1 (CONS 'PROGN (CDR FORM)))))
      )
    (SETQ NEW-FORM
	  (COND
	    ((SETQ TM (ASSOC (CAR FORM) LOCAL-FUNCTIONS :TEST #'EQ))
	     ;; local function defined by FLET or LABELS
	     (SETQ NEW-FORM (P1EVARGS FORM))
	     (SETQ EXPRESSION-SIZE NEW-SIZE)
	     (OR (AND (EQ (COMPILAND-DEFINITION *CURRENT-COMPILAND*)
			  (THIRD TM)) ; function is calling itself
		      (CONSP P1VALUE)
		      (LET ((X (ASSOC (COMPILAND-FUNCTION-SPEC *CURRENT-COMPILAND*)
				      P1VALUE :TEST #'EQ)))
			(AND X ; this is a tail recursive call
			     (MEMBER X TRE-OK :TEST #'EQ) ; no special bindings in effect
			     (<= (OPT-SAFETY OPTIMIZE-SWITCH) (OPT-SPEED-OR-SPACE OPTIMIZE-SWITCH))
			     (SECOND X) ; loop-back tag provided
			     (NOT DONT-OPTIMIZE)
			     (TAIL-RECURSION-ELIMINATION
			       NEW-FORM (SECOND X) (THIRD X) (FIFTH X)) )))
		 `(FUNCALL ,(REF-LOCAL-FUNCTION-VAR (SECOND TM))
			   . ,(CDR NEW-FORM)) ))
	    ((MEMBER (CAR FORM) '(LET LET*) :TEST #'EQ)
	     (P1-WITH-ANNOTATION FORM #'P1LET 'UNKNOWN DONT-OPTIMIZE))
	    ((EQ (CAR FORM) 'BLOCK)
	     (P1-WITH-ANNOTATION FORM #'P1BLOCK 'UNKNOWN DONT-OPTIMIZE))
	    ((EQ (CAR FORM) 'TAGBODY)
	     (P1-WITH-ANNOTATION FORM #'P1TAGBODY 'NULL DONT-OPTIMIZE))
	    ((EQ (CAR FORM) '%POP) FORM )	;P2 specially checks for this
	    ((EQ (CAR FORM) 'COMPILER-LET)
	     ;; handled specially here so that the result will not be re-optimized
	     ;; after the bindings are un-done.
	     (RETURN-FROM P1
	       (SI:EVAL1 `(COMPILER-LET ,(SECOND FORM)
			    (P1 '(PROGN . ,(CDDR FORM))) ))))
	    ((EQ (CAR FORM) 'COMPILER-LETF) ;;; JPR.
	     (RETURN-FROM p1
		(do-compiler-letf
		  (second form) `(p1 '(progn ,@(rest (rest form)))))))
	    ((SETQ TLEVEL NIL))
	    ((EQ (CAR FORM) 'COND)
	     (P1-WITH-ANNOTATION FORM #'P1COND 'UNKNOWN DONT-OPTIMIZE))
	    ;; Check for functions with special P1 handlers.
	    ((AND (SETQ HANDLER (GET (CAR FORM) 'P1))
		  (OR (NEQ INDECL 'NOTINLINE)
		      (NOT (MEMBER HANDLER '(P1SIMPLE P1-DOWNWARD-FUNARG
					     P1-DOWNWARD-FUNARG-DESTRUCTIVE) :TEST #'EQ))) )
	     (UNLESS (MEMBER (CAR FORM)
			     '( PROGN IGNORE P1-HAS-BEEN-DONE RETURN-FROM %BLOCK-BODY
			        #+compiler:debug P1-ALREADY-DONE ; this one is obsolete 9/19/86
				COMPILER-LET BLOCK-FOR-PROG
				)
			     :TEST #'EQ)
	       (SETQ EXPRESSION-SIZE NEW-SIZE) )
	     (FUNCALL HANDLER FORM))
	    ((AND ALLOW-VARIABLES-IN-FUNCTION-POSITION-SWITCH
		  (LOOKUP-VAR (CAR FORM) VARS)
		  (NULL (FUNCTION-P (CAR FORM))))
	     (WARN 'EXPRESSION-AS-FUNCTION ':VERY-OBSOLETE
		   "The variable ~S is used in function position; use FUNCALL."
		   (CAR FORM))
	     (RETURN-FROM P1 (P1 (CONS 'FUNCALL FORM))))
	    ((EQ (CAR FORM) 'FUNCALL)
	     (SETQ TM (COMPILAND-CHILDREN *CURRENT-COMPILAND*))
	     (LET (( F (LET (( P1VALUE 'DOWNWARD-ONLY ))
			 (P1 (SECOND FORM)) )))
	       (COND ((AND (CONSP F)
			   (MEMBER (FIRST F) '(QUOTE FUNCTION) :TEST #'EQ)
			   (NOT DONT-OPTIMIZE)
			   (OR (SYMBOLP (SECOND F))
			       (CONSP (SECOND F)))
			   (NOT (ASSOC (SECOND F) LOCAL-FUNCTIONS :TEST #'EQUAL)) ; 12/16/88
			   (FUNCTIONP (SECOND F)) )
		      ;; (FUNCALL #'f a b) ==> (f a b)
		      ;; (FUNCALL #'(LAMBDA ...) a b) ==> ((LAMBDA ...) a b)
		      (RETURN-FROM P1 (P1 (CONS (SECOND F) (CDDR FORM)))))
		     ((AND (QUOTEP F)
			   (FUNCTIONP (SECOND F) NIL)
			   (SYMBOLP (SETQ TM (FUNCTION-NAME (SECOND F))))
			   (FBOUNDP TM)
			   (EQ (SYMBOL-FUNCTION TM) (SECOND F))
			   (NOT DONT-OPTIMIZE)
			   (EXTERNAL-SYMBOL-P TM))
		      ;; ('#<DTP-FUNCTION fn ...> a b)  ==> (fn a b)
		      ;; This idiom is used by some Scheme macros to ensure access to the 
		      ;; global definition.
		      (SETQ EXPRESSION-SIZE NEW-SIZE)
		      (SETQ FORM (PRE-OPTIMIZE (CONS TM (CDDR FORM))
					       T (EQ (SETQ INDECL (GET TM 'INLINE)) 'NOTINLINE)))
		      (FUNCALL (GET (CAR FORM) 'P1 #'P1EVARGS) FORM)
		      )
		     (T (SETQ EXPRESSION-SIZE NEW-SIZE)
			(WHEN (AND (MEMBER (CAR-SAFE F) '(BREAKOFF-FUNCTION LEXICAL-CLOSURE))
				   (EQ (SECOND F) (FIRST (COMPILAND-CHILDREN *CURRENT-COMPILAND*)))
				   (EQ TM (REST (COMPILAND-CHILDREN *CURRENT-COMPILAND*))))
			  ;; Encourage PROCEDURE-INTEGRATION.
			  (SETF (GETF (COMPILAND-PLIST (SECOND F)) 'USED-ONLY-ONCE) T))
			(PROG1 (LET ((SAVE-ALLVARS ALLVARS))
				 (FIX-FUNCALL-EVALUATION-ORDER
				   (CONS 'FUNCALL (P1EVARGS (CONS F (CDDR FORM))))
				   SAVE-ALLVARS))
			       (ARBITRARY-SIDE-EFFECTS))) )) )
	    ( T	  ; general function
	     (SETQ EXPRESSION-SIZE NEW-SIZE)
	     (UNLESS (NULL (CDR FORM))
	       (SETQ FORM (P1ARGC FORM ) ))
	     (COND
	       ((AND (CONSP P1VALUE)  ; still has initial value from QCOMPILE1
		     (SETQ TM (ASSOC (CAR FORM) P1VALUE :TEST #'EQ))
						; this is a tail recursive call
		     (OR (EQL (OPT-SAFETY OPTIMIZE-SWITCH) 0) ; user permits optimizing
			 (COMPILING-SCHEME-P))	; Scheme users expect this to happen.
		     (MEMBER TM TRE-OK :TEST #'EQ)	 ; no special bindings in effect
		     TRE-ENABLE 
		     (NOT DONT-OPTIMIZE)
		     (NOT (GETL (CAR FORM)
				'(P2 OPCODE))) ; not expanded by pass 2
		     (TAIL-RECURSION-ELIMINATION
		       FORM (SECOND TM) (THIRD TM) (FIFTH TM) ) ))
	       ((AND (SETQ TM (ASSOC (CAR FORM) INLINE-EXPANSIONS :TEST #'EQ))
		     (NEQ (FIRST TM) (COMPILAND-FUNCTION-SPEC *CURRENT-COMPILAND*)) )
		;; This is a recursive call to a function which we are
		;;   currently in the process of expanding inline.
		;; Abort the inline expansion.
		(THROW (SECOND TM) 'RECURSIVE) ); the CATCH is in function PROCEDURE-INTEGRATION
	       ((AND (EQ INDECL 'NOTINLINE)
		     (EQ (CAR ORIGINAL-FORM) (CAR FORM)) )
		(SETQ DONT-OPTIMIZE INDECL)
		(ARBITRARY-SIDE-EFFECTS)
		(IF (AND (GET (CAR FORM) 'P2)
			 (FUNCTIONP (CAR FORM)) )
		    `(FUNCALL (FUNCTION ,(CAR FORM)) . ,(CDR FORM))
		  FORM) )
	       (T (SETQ HANDLER 'P1ARGC)
		  FORM) )
	    )))
    ;; Apply post-optimizations
    (UNLESS (OR DONT-OPTIMIZE
		;; Don't optimize dead code -- not only to avoid
		;; wasting time, but because constant folding could
		;; get an argument type error which would be irrelevant.
		(ZEROP 1-IF-LIVE-CODE))
      (SETQ TM (POST-OPTIMIZE NEW-FORM))
      (WHEN (AND (MEMBER HANDLER '(P1ARGC P1-DOWNWARD-FUNARG P1-DOWNWARD-FUNARG-DESTRUCTIVE) :TEST #'EQ)
		 (OR (EQ TM NEW-FORM)
		     (NOT (TRIVIAL-FORM-P TM))))
	;; possibility of inline expansion of the called function
	(SETQ FORM (IF (OR (EQ (CAR ORIGINAL-FORM) (CAR TM))
			   (EQ INDECL 'INLINE))
		       (MAYBE-INTEGRATE (CAR TM) (CDR TM) NIL INDECL)
		     (MAYBE-INTEGRATE (CAR TM) (CDR TM)) ))
	(UNLESS (NULL FORM)
	  (SETQ TM (POST-OPTIMIZE FORM))
	  (SETQ HANDLER NIL)))
      (WHEN (NEQ NEW-FORM TM)
	(SETQ HANDLER NIL) ; don't update var sets below
	(SETQ NEW-FORM TM)
	(WHEN (TRIVIAL-FORM-P NEW-FORM)
	  ;; optimized down to just a constant or variable --
	  ;; count its size as only 1
	  (SETQ EXPRESSION-SIZE NEW-SIZE)
      ) ) )
    (WHEN (AND INLINE-EXPANSIONS
	       (> EXPRESSION-SIZE EXPRESSION-SIZE-LIMIT) )
      ;; inline expansion of function call has become too big 
      ;;  to be desirable -- abort back to CATCH in
      ;;  function PROCEDURE-INTEGRATION
      (THROW (SECOND (FIRST INLINE-EXPANSIONS)) 'SIZE) )
    (WHEN (EQ HANDLER 'P1ARGC)
      (BLOCK USE-SPECIAL
	(UNLESS (LOGTEST SPECIAL-VAR-BIT USED-VAR-SET)
	  (WHEN (FUNCTION-WITHOUT-SIDE-EFFECTS-P (FIRST NEW-FORM))
	    (RETURN-FROM USE-SPECIAL))
	  (SETF USED-VAR-SET (LOGIOR USED-VAR-SET GLOBAL-SIDE-EFFECTS)))
	(UNLESS (OR (LOGTEST DATA-ALTERATION-BIT ALTERED-VAR-SET)
		    (FUNCTION-WITHOUT-SIDE-EFFECTS-P (FIRST NEW-FORM)))
	  (SETF ALTERED-VAR-SET (LOGIOR ALTERED-VAR-SET GLOBAL-SIDE-EFFECTS)))))
    (WHEN (AND SI:FILE-IN-COLD-LOAD ; Current file has attribute COLD-LOAD:T
	       (CONSP NEW-FORM)
	       (NOT (ZEROP 1-IF-LIVE-CODE))
	       (NOT (AND (SYMBOLP (FIRST NEW-FORM))
			 (GETL (FIRST NEW-FORM) '(P2 OPCODE)))) )
      (CHECK-COLD (FIRST NEW-FORM)) )
    (RETURN-FROM P1 NEW-FORM)
    ))