;;; -*- Mode:Common-Lisp; Package:ZWEI; Base:10 -*-

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;	Zmacs Auto-save Hack
;
;	Author:		John Nguyen
;
;	Address:	545 Technology Square, Room 626
;			Cambridge, MA  02139
;			(617) 253-6028
;
;	E-mail:		johnn@hx.lcs.mit.edu
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; This relies on the timer-queue facility created by Dan Cerys at TI.
; If that is not available, a background process can be created which calls
; check-autosave1 periodically.

; Usage: After this file is loaded, Zmacs buffers are periodically scanned.
; When any modified ones are found that has not been saved in a while, they
; are saved in a the subdirectory "zmacs-auto-save" of the user's home directory.
; If a crash does occur before files are saved, invoke the command "M-x recover-files"
; which pulls the files from the "zmacs-auto-save" directory and places them as
; buffers with the original names.

; The commands, enable-zmacs-autosave, disable-zmacs-autosave, and zmacs-autosave-p
; allow enabling, disabling, and checking of the autosave feature.

(defvar *auto-save-directory* nil)
(defvar *auto-save-filename* "auto-save-file.text")
(defvar *auto-save-logname* "log.text")

(defvar *auto-save-tick-threshold* 3000)

(defvar *auto-save-list* nil)


(defun initialize-auto-save-directory ()
  (unless *auto-save-directory*
    (let ((homedir (user-homedir-pathname)))
      (setq *auto-save-directory*
	    (send homedir :new-suggested-directory
		  (append (send homedir :directory) '("zmacs-auto-save")))))
    (ignore-errors (send (pathname *auto-save-directory*) :create-directory))))


(defun auto-save (buffer)
  (when (and (buffer-modified-p buffer) (not (typep buffer 'mail-file-buffer)))
    (let ((filename (merge-pathnames *auto-save-filename* *auto-save-directory*)))
      (with-open-file (s filename :direction :output :error :retry)
	(let (*window*
	      (*check-unbalanced-parentheses-when-saving* nil)
	      (*major-mode* :text-mode))
	  (w:map-over-sheets
	    #'(lambda (window)
		(when (typep window 'zwei:zmacs-frame)
		  (setq *window* (send window :editor-window)))))
	  (when *window*
	    (stream-out-interval s (interval-first-bp buffer) (interval-last-bp buffer) nil t)))
	(when (position (string (send buffer :name)) *auto-save-list* :key #'second :test #'equal)
	  (setq *auto-save-list*
		(delete (string (send buffer :name)) *auto-save-list*
			:test #'equal :key #'second)))
	(push (list (if (send buffer :pathname) (string (send buffer :pathname)))
		    (string (send buffer :name))
		    (string (send s :truename))
		    buffer
		    (or (parse-integer (send buffer :version-string) :start 2 :junk-allowed t) 0))
	      *auto-save-list*))))
  (let ((new-save-list nil))
    (dolist (entry *auto-save-list*)
      (let ((filename (merge-pathnames (make-pathname :version (1+ (fifth entry)))
				       (car entry))))
	(when (and (or (null (car entry)) (not (probe-file filename)))
		   (find (fourth entry) zwei:*zmacs-buffer-list*)
		   (buffer-modified-p (fourth entry)))
	  (push entry new-save-list))))
    (setq *auto-save-list* new-save-list))
  (when *auto-save-list*
    (with-open-file (s (merge-pathnames *auto-save-logname* *auto-save-directory*)
		       :direction :output :error :retry)
      (prin1 (mapcar #'(lambda (x) (list (car x) (second x) (third x))) *auto-save-list*) s))))


(defcom com-recover-files "Recover files" ()
  (let ((save-list nil)
	(count 0))
    (initialize-auto-save-directory)
    (with-open-file (s (merge-pathnames *auto-save-logname* *auto-save-directory*)
		       :direction :input :error nil)
      (when s
	(setq save-list (read s))))
    (when save-list
      (dolist (entry save-list)
	(ignore-errors (undelete-file (third entry)))
	(when (probe-file (third entry))
	  (incf count)
	  (let ((buffer (make-instance 'zmacs-buffer)))
	    (revert-buffer buffer (third entry))
	    (if (car entry)
		(set-buffer-pathname (pathname (car entry)) buffer)
		(send buffer :set-name (second entry)))
	    (send buffer :set-tick (+ 1 *auto-save-tick-threshold* (send buffer :tick)))
	    (send buffer :activate)))))
    (format *query-io* "~%~d files recovered" count)
    dis-none))


(defun check-autosave ()
  (check-autosave1)
  (si:add-timer-queue-entry
    '(:relative 180.)
    :once
    "Zmacs Auto Save"
    'check-autosave))

(defun check-autosave1 ()
  (initialize-auto-save-directory)
  (dolist (interval zwei:*zmacs-buffer-list*)
    (when (and (typep interval 'zmacs-buffer)
	       (send interval :pathname)
	       (buffer-modified-p interval)
	       (not (buffer-read-only-p interval))
	       (> (- (send interval :tick) (send interval :file-tick))
		  *auto-save-tick-threshold*))
      (send interval :set-file-tick (1- (send interval :tick)))
      (auto-save interval))))

; The following allows autosaving without relying on the timer-queue facility
; One can also make a background process which periodically calls check-autosave
;
;(defcom com-self-insert "Modified self insert" (NM)
;  (when (and (typep *interval* 'zmacs-buffer)
;	     (buffer-modified-p *interval*)
;	     (> (- (send *interval* :tick) (send *interval* :file-tick))
;		*auto-save-tick-threshold*))
;    (send *interval* :set-file-tick (1- (send *interval* :tick)))
;    (format *query-io* "Autosaving...")
;    (process-run-function "Zmacs Auto Save" 'auto-save *interval*))
;  (old-self-insert))

(set-comtab *standard-comtab* () '(("Recover Files" . com-recover-files)))

(DEFUN disable-zmacs-autosave ()
  (SETQ *auto-save-directory* NIL)
  (SETQ *auto-save-list*      NIL)
  (si:remove-timer-queue-entry "Zmacs Auto Save"))

(deff enable-zmacs-autosave 'check-autosave)

(defun zmacs-autosave-p ()
  *auto-save-directory*)

(add-initialization 'start-autosave
		    '(enable-zmacs-autosave)
		    '(:once))

