;;; -*- Mode:Common-Lisp; Package:TV; Base:10; Patch-file:T -*-

;;; First, some code for deexposed window documentation:

;; This mouse handler serves for windows which want to do things the simple way.
;; A second argument of T says that the window should have a scroll bar.
;; This function is also used to track the mouse when it isn't inside any window,
;; by calling it with an argument of NIL.
;; An arg of T is used when the mouse has been seized by a process not
;; for any specific window.
(DEFUN MOUSE-DEFAULT-HANDLER (WINDOW &OPTIONAL SCROLL-BAR-FLAG
				  &AUX
				  (WINDOW-X-OFFSET 0) (WINDOW-Y-OFFSET 0)
				  WINDOW-X WINDOW-Y)
      (unless (SYMBOLP WINDOW)
	(MULTIPLE-VALUE-SETQ (WINDOW-X-OFFSET WINDOW-Y-OFFSET)
			     (SHEET-CALCULATE-OFFSETS SELF MOUSE-SHEET)))
    
      ;; Be careful not to do the :update method when the who line documentation window
      ;; isn't there (which is the case during a window system build).
      (WHEN (AND (BOUNDP 'WHO-LINE-DOCUMENTATION-WINDOW) WHO-LINE-DOCUMENTATION-WINDOW)
	;; Update who-line when entering new handlers.
	(send who-line-documentation-window :update))
      
      (DO ((DX) (DY) (BU) (BD) (HAND) (X) (Y)
	   (OLD-OWNER WINDOW-OWNING-MOUSE WINDOW-OWNING-MOUSE)
	   (LEFT-OFFSET 0)
	   (right-offset 0)
	   (WAIT-FLAG NIL T))
	  (MOUSE-RECONSIDER)
	;; Wait until the mouse moves
	(MULTIPLE-VALUE-SETQ (DX DY BD BU X Y) (MOUSE-INPUT WAIT-FLAG))

    (if (and (logtest 2 bd) (key-state :hyper))
	(progn
	  (uwm-handler 'dynamic-change-window)
	  (return nil)))

    (if (and (logtest 1 bd) (key-state :hyper))
	(progn
	  (uwm-handler 'dynamic-shrink-window)
	  (return nil)))

    (if (and (logtest 4 bd) (key-state :hyper))
	(progn
	  (uwm-handler 'dynamic-bury-window)
	  (return nil)))

	;; If asked to reconsider, do so immediately.
	;; Don't bother updating blinker since it is likely to change soon, and
	;; in any case we are going to be called back shortly.
	(when MOUSE-RECONSIDER (RETURN NIL))
	;; Update console-idle time when buttons pushed
	(unless (ZEROP BD) (SETQ KBD-LAST-ACTIVITY-TIME (TIME)))
	;; Approximate speed of the mouse in inches per second
	(SETQ MOUSE-SPEED (/ (ISQRT (+ (* MOUSE-X-SPEED MOUSE-X-SPEED)
				       (* MOUSE-Y-SPEED MOUSE-Y-SPEED)))
			     100.0s0))
	;; If the mouse is moving incredibly fast, flash up something to
	;; help the user find it.  Thus if you can't find the mouse, you must whip it.
	(when (> MOUSE-SPEED MOUSE-FAST-MOTION-SPEED)
	  (if mouse-fast-track-bitmap-mouse-p
	      (draw-bitmap-mouse-cursor mouse-speed)
	      ;;ELSE
	      (draw-mouse-fast-motion-cursor)))
    
	(SETQ WINDOW-X (- X WINDOW-X-OFFSET)	; X offset of mouse within window
	      WINDOW-Y (- Y WINDOW-Y-OFFSET))	; Y offset of mouse within window
    
	;; Consider entering the scroll bar.  [Perhaps this should be changed so that
	;; it is in the move-handler rather than here.  Problem with that is LEFT-OFFSET.]
	;; If there is a scroll bar and we are entering it, activate it.
	;; However, the mouse must move at least a certain distance past the edge
	;; of the window in order to qualify for scrolling (this is set by
	;; the SCROLL-BAR-RELUCTANCE variable in the window).  Before entering
	;; scroll bar, send a :MOUSE-MOVES message in order to let the window know
	;; what's happening.
    
	;; LEFT-OFFSET is how far out the left side of the window the mouse has moved,
	;; or 0 if the mouse is inside the window.
	;; If the window is at the left edge of the screen, MOUSE-X will not itself
	;; move out the left edge of the window, but DX will.  When the mouse reaches
	;; the left edge of the window, accumulate leftward motion into LEFT-OFFSET.  
	;; RIGHT-OFFSET does the same thing for when the scroll-bar is on the right.
	(COND ((< WINDOW-X 0)	;; may 9-7-88 was <= see HACK below
	       (SETQ LEFT-OFFSET  (IF (PLUSP LEFT-OFFSET)
				      (MAX (- LEFT-OFFSET DX) 1)
				      1)))	   		;First time, don't use all of DX
	      ((and (typep window 'sheet)
		    (>= window-x (sheet-width window))
		    (send window :send-if-handles :scroll-bar-on-right))
	       (setq right-offset (if (plusp right-offset)
				      (max (+ right-offset dx) 1)
				      1)))
	      ;; [may] 9-7-88
	      ;; HACK : the left edge is a special case. Its possible to hang up the
	      ;;	    the system when scroll bar on right and mouse on left edge
	      ;;	    looping in :handle-mouse and here with wait-flag always nil
	      ;;	    which allows #'mouse-input to be process greedy! We COULD always set
	      ;;	    left-offset to 0 in/of 1 BUT left edge of scroll-bar would not make
	      ;;	    double-arrow for scrolling sometimes.
	      ;; cases : 1. left-offset=0 AND no scroll-bar ( fixed lying :enable-scrolling-p )
	      ;;	    2. left-offset=0 AND scroll-bar on right - force left-offset = 0 to check right-offset
	      ;;	    3. left-offset=0 AND left scroll-bar - this case only happens SOMETIMES as scroll-bar is being
	      ;;	      added. The other two cases happen EVERY time.
	      ((and (= window-x 0) (eq scroll-bar-flag t)) ;; may 01/30/89 ADDED (eq t scroll-bar-flag) check I left out.
	       ;; we are called with scroll-bar-flag indicating a scroll bar but if
	       ;; scroll bar is on RIGHT we really should check right-offset FIRST in COND below
	       (COND ((and (typep window 'sheet)
			   (send window :send-if-handles :scroll-bar-on-right)) ;; scroll bar is on RIGHT
		      (SETQ left-offset 0)) ;; force skipping of left-offset in COND below
		     (t
		      (SETQ LEFT-OFFSET  (IF (PLUSP LEFT-OFFSET)
				      (MAX (- LEFT-OFFSET DX) 1)
				      0))))) ;First time, don't use all of DX [may 9-13-88 was 1 - fix case 3]
	      (t
	       (SETQ LEFT-OFFSET 0
		     right-offset 0)))
	(COND ((or old-owner window-owning-mouse)) ; These disable scroll-bar.
	      ((EQ SCROLL-BAR-FLAG 'T)
	       (cond ((AND SCROLL-BAR-MAX-SPEED
			  (> MOUSE-SPEED SCROLL-BAR-MAX-SPEED)))
		     ((PLUSP LEFT-OFFSET)
		      (cond ((NOT (SEND WINDOW :send-if-handles :scroll-bar))
			     (return nil))
			    ((> LEFT-OFFSET SCROLL-BAR-RELUCTANCE)
			     (SEND (IF (SYMBOLP WINDOW)
				       'MOUSE-SET-BLINKER-CURSORPOS WINDOW)
				   :mouse-moves WINDOW-X WINDOW-Y)
			     (RETURN (SEND WINDOW :HANDLE-MOUSE-SCROLL)))
			    (T (SETQ WINDOW-X 0))))	;Don't escape the window yet
		     ((plusp right-offset)
		      (cond ((NOT (SEND WINDOW :send-if-handles :scroll-bar-on-right))
			     (return nil))
			    ((> right-OFFSET SCROLL-BAR-RELUCTANCE)
			     (SEND (IF (SYMBOLP WINDOW)
				       'MOUSE-SET-BLINKER-CURSORPOS WINDOW)
				   :mouse-moves WINDOW-X WINDOW-Y)
			     (RETURN (SEND WINDOW :HANDLE-MOUSE-SCROLL)))
			    (T (SETQ WINDOW-X (1- (sheet-width window))))))))	;Don't escape the window yet
	      
	      ((EQ SCROLL-BAR-FLAG :IN)
	       ;; We are in the scroll bar.  Moving the mouse faster than the exit
	       ;; speed, or moving it into the interior of the window more than 40. 
	       ;; will escape.  
	       (COND ((AND SCROLL-BAR-MAX-EXIT-SPEED
			   (> MOUSE-SPEED SCROLL-BAR-MAX-EXIT-SPEED))
		      ;; Moving like a bat, let the guy out of the scroll bar
		      (RETURN NIL))
		     ((MINUSP WINDOW-X)		;Trying to go out left, shove back in.
		      (WITHOUT-INTERRUPTS
			(%OPEN-MOUSE-CURSOR)
			(SETQ WINDOW-X 0)
			(SETQ MOUSE-LAST-X (SETQ MOUSE-X WINDOW-X-OFFSET))
			(SETQ MOUSE-CURSOR-STATE MOUSE-CURSOR-CLOSED-STATE
			      PREPARED-SHEET NIL)))
		     ((>= window-x (sheet-width window))	;Trying to go out right, shove back in.
		      (WITHOUT-INTERRUPTS
			(%OPEN-MOUSE-CURSOR)
			(SETQ WINDOW-X (- (sheet-width window) 8.))  ; hack
			(SETQ MOUSE-LAST-X (SETQ MOUSE-X (+ WINDOW-X-OFFSET window-x)))
			(SETQ MOUSE-CURSOR-STATE MOUSE-CURSOR-CLOSED-STATE
			      PREPARED-SHEET NIL)))
		     ((and (> WINDOW-X (+ 40. (sheet-inside-left window)))	; Escape scroll-bar
			   (< window-x (- (sheet-inside-right window) 40.)))
		      (RETURN NIL)))))
	;; Update the position of the mouse before checking for button clicks, so
	;; that button clicks get processed with knowledge of where the mouse
	;; was when the button was first clicked.  The arguments to the move handler
	;; may be where the mouse was when the button was clicked, whereas the
	;; mouse cursor follows MOUSE-X and MOUSE-Y, which may be different.   
	(SETQ MOUSE-WARP NIL)
	(SEND (IF (SYMBOLP WINDOW)
		     'MOUSE-SET-BLINKER-CURSORPOS WINDOW)
		 :mouse-moves WINDOW-X WINDOW-Y)
	;; Check for all the ways of losing control of the mouse.
	(IF (COND ;; The move handler may have decided to warp the mouse so that it will not
		  ;; move out of the window.  This test is a crock but should work.
		  (MOUSE-WARP NIL)
		  ;; Check for mouse ceasing to be grabbed.
		  ((EQ WINDOW T)
		   (NEQ WINDOW-OWNING-MOUSE T))
		  ;; Check for window becoming grabbed.
		  ((EQ WINDOW-OWNING-MOUSE T)
		   (NEQ WINDOW T))
		  ;; Check for some other window (not above this one) being greedy.
		  (WINDOW-OWNING-MOUSE
		   (NOT (SHEET-ME-OR-MY-KID-P WINDOW WINDOW-OWNING-MOUSE)))
		  ;; Check for moving into a window when not in any
		  ((NULL window)
		   (OR (window-owning-mouse x y)
		       ;; Also check for mouse escaping MOUSE-SHEET.  CJJ 05/11/88.
		       ;;; For Mouse-Transport support.  Added by KJF for CJJ on 08/16/88 for
		       ;;; Multiple Monitor (MMON) support.
		       (AND (mmon-p) (NOT (mouse-in-mouse-sheet-p)))))
		  ;; Check for leaving the boundaries of the current window
		  ;; HYSTERETIC-WINDOW-MIXIN requires that we wait at least once before returning
		  ((NOT (AND (SHEET-EXPOSED-P WINDOW)
			     (>= WINDOW-X 0)
			     (<  WINDOW-X (SHEET-WIDTH WINDOW))
			     (>= WINDOW-Y 0)
			     (<  WINDOW-Y (SHEET-HEIGHT WINDOW))
			     ;; Also check for mouse escaping MOUSE-SHEET.  CJJ 05/11/88.
			     ;;; For Mouse-Transport support.  Added by KJF for CJJ on 08/16/88 for
			     ;;; Multiple Monitor (MMON) support.
			     (IF (mmon-p)
				 (OR (EQ scroll-bar-flag :in) (mouse-in-mouse-sheet-p))
				 ;; else
				 t)))
		   WAIT-FLAG)
		  ;; Check for moving into an inferior of the current window
		  ((NEQ (LOWEST-SHEET-UNDER-POINT WINDOW WINDOW-X WINDOW-Y
						  :HANDLE-MOUSE :EXPOSED)
			WINDOW)
		   T))
	    ;; Return to overseer, saving any pending button click.
	    (RETURN (MOUSE-DEFER-BUTTONS BU BD)))
	;; Now process button pushes if mouse is not seized.
	(COND ((OR (ZEROP BD) (EQ WINDOW T) (EQ OLD-OWNER T))	;; may 9-7-88
	       ;;; The following added by JPR to mirror Acuff's deexposed mouse
	       ;;; buttons.
	       (setq hand (window-under-mouse :deexposed-mouse-moves :active x y))
	       (if hand
		   (multiple-value-bind (xoff yoff)
		       (sheet-calculate-offsets hand mouse-sheet)
		     (send hand :deexposed-mouse-moves (- x xoff) (- y yoff))))
	      )
	      ;; If over an exposed window, send it the button-push
	      (WINDOW (SEND WINDOW :mouse-buttons BD WINDOW-X WINDOW-Y))
	      ;;RDA:- call :DEEXPOSED-MOUSE-BUTTONS if we're over a
	      ;; deexposed window that has a handler for it
	      ((setq hand (window-under-mouse :deexposed-mouse-buttons :active x y))
	       (multiple-value-bind (xoff yoff)
		   (sheet-calculate-offsets hand mouse-sheet)
		 (send hand :deexposed-mouse-buttons bd (- x xoff) (- y yoff))))
	      ;; Default action for left button is to select what mouse is pointing at
	      ((LOGTEST 1 BD)
	       (AND (SETQ HAND (WINDOW-UNDER-MOUSE :MOUSE-SELECT :ACTIVE X Y))
		    ;; Next line temporarily papers over a bug with :MOUSE-SELECT
		    (GET-HANDLER-FOR HAND :SELECT)
		    (MOUSE-SELECT HAND)))
	      ;; Default action for middle button is to switch to the main screen
	      ((LOGTEST 2 BD)
	       (when (TYPEP MOUSE-SHEET 'SCREEN)
		 (if (mac-screen-p mouse-sheet)
		     (give-mouse-ownership-to-the-Explorer)
		     (PROCESS-RUN-FUNCTION "Set mouse sheet" #'MOUSE-SET-SHEET DEFAULT-SCREEN))))
	      ;; Default action for right button is to call the system menu
	      ((LOGTEST 4 BD)
	       (MOUSE-CHARACTER-BUTTON-ENCODE BD)	   ;Satisfy those who double-click out of habit
	       (MOUSE-CALL-SYSTEM-MENU)))))

;;; The following lines added by JPR to get around a problem on the uX.
(defparameter tv:*has-been-booted* t)

(Add-Initialization "Say that I've booted."     '(setq tv:*has-been-booted* t)   '(:User-Application :Normal))
(Add-Initialization "Say that I've booted."     '(setq tv:*has-been-booted* t)   '(:Login :Normal))
(Add-Initialization "Say that I've logged out." '(setq tv:*has-been-booted* nil) '(:Logout :Normal))

; To allow who-line to display docs when hyper key is pressed

(DEFUN WHO-LINE-DOCUMENTATION-FUNCTION (WHO-SHEET)
  "This displays the who line documentation for the currently selected
window.  If the selected window is locked an error message is blinked
in the who line area.  Who line documentation may be either a string
of text which is to be displayed or a list of keyword value pairs.  To
see what keywords are accepted see the documentation for the
tv:process-who-line-documentation-list function."
  (DECLARE (:SELF-FLAVOR WHO-LINE-SHEET))
  (LET* ((W MOUSE-WINDOW)
	 (MSG  "*** Error with window locked; try TERMINAL C-CLEAR-INPUT or TERMINAL C-C ***")
	 (MSG1 "    Error with window locked; try TERMINAL C-CLEAR-INPUT or TERMINAL C-C")
	 (NEW-STATE (COND (LOCKED-ERROR-WINDOWS
			    ;; To attract attention, make this message blink.
			    ;; The following EQ test should remain as EQ in spite of what the
			    ;; compiler says.  In this case EQ is both faster and correct.
			    (IF (compiler-let ((inhibit-style-warnings-switch t))
				  (EQ WHO-LINE-ITEM-STATE MSG))	;Compiler-let added by PMH 7/2/87
				MSG1 MSG))
                          ((and w (SYMBOLP W))
                           WHO-LINE-MOUSE-GRABBED-DOCUMENTATION)
			  ((w:key-state :hyper)
			   '(:mouse-l-1 "Iconify window" :mouse-m-1 "Resize/Move window"
					:mouse-r-1 "Bury window"))
			  ;;RDA:- ask about deexposed doc
			  ((and *has-been-booted* (null w))
			   (let ((handling-window
				   (window-under-mouse
				     :deexposed-who-line-documentation-string)))
			     (when handling-window
			       (send handling-window
				     :deexposed-who-line-documentation-string))
			     )
			   )
			  ((null w) w)
                          (T (MULTIPLE-VALUE-BIND (DOC ERROR)
                                 (CATCH-ERROR
                                   (FUNCALL W :WHO-LINE-DOCUMENTATION-STRING)
                                   NIL)
                               (IF ERROR
                                   "Error getting documentation string" DOC))))))
    (COND ((AND (NEQ WHO-LINE-ITEM-STATE NEW-STATE)
                (NOT (EQUAL WHO-LINE-ITEM-STATE NEW-STATE)))
           (SETQ WHO-LINE-ITEM-STATE NEW-STATE)
           (SHEET-CLEAR WHO-SHEET)
           (SHEET-SET-CURSORPOS WHO-SHEET 2 1)
           (CATCH-ALL
             (COND ((TYPEP NEW-STATE 'STRING)
                    (CATCH 'PAGE-OVERFLOW
                      (SHEET-STRING-OUT WHO-SHEET NEW-STATE)))
                   ((CONSP NEW-STATE)
                    (PROCESS-WHO-LINE-DOCUMENTATION-LIST WHO-SHEET NEW-STATE))
                   (T (CATCH 'PAGE-OVERFLOW
                        (PROCESS-WHO-LINE-DOCUMENTATION-LIST
                          WHO-SHEET
                          '(:MOUSE-R-1 "Bring up the System Menu."))))))))))


(defun uwm-handler (function)
  (flet ((new-function ()
	   (condition-case (condition)
	       (funcall function)
	     (error (tv:notify nil "Error from mouse-window-manager: ~a"
			       (send condition :report-string))))))
    (process-run-function "UWM Handler" #'new-function)))

;--------------------------------------------------------------------------------
; The main routine that actually moves the windows (called by the handler process)
;--------------------------------------------------------------------------------

(defun dynamic-bury-window ()
  (let ((window (dynamic-find-window (get-window-edge-alist mouse-sheet))))
    (send (car window) :bury)))

(defun dynamic-shrink-window ()
  (let ((window (dynamic-find-window (get-window-edge-alist mouse-sheet))))
    (w:shrink-window (car window))))

(defun dynamic-change-window ()
  (let ((window (dynamic-find-window (get-window-edge-alist mouse-sheet))))
    (dynamic-perform-window-change (car window))))

(defun dynamic-find-window (window-edge-alist)
  (let ((x mouse-x)
	(y mouse-y))
    (dolist (w window-edge-alist)
      (if (and (>= x (third w)) (>= y (fourth w)) (< x (fifth w)) (< y (sixth w)))
	  (return w)))))


;--------------------------------------------------------------------------------
; Decide whether to move or resize the window:
;--------------------------------------------------------------------------------

(defun dynamic-perform-window-change (window &aux sides)
  (if (not (sheet-me-or-my-kid-p window mouse-sheet))
      (ferror nil "Attempt to set position of ~S, which is not inferior to MOUSE-SHEET" window))
  
  (let ((superior (sheet-superior window))
	(x (sheet-x window))
	(y (sheet-y window)))
    
    (multiple-value-bind (xoff yoff)
	(sheet-calculate-offsets superior mouse-sheet)
      
      (multiple-value-bind (mouse-x mouse-y)
	  (send mouse-blinker :read-cursorpos)
	
	(let ((mouse-offset-x (- mouse-x xoff x))
	      (mouse-offset-y (- mouse-y yoff y))
	      
	      (one-third-x (truncate (sheet-width window) 3))
	      (two-thirds-x (* 2 (truncate (sheet-width window) 3)))
	      
	      (one-third-y (truncate (sheet-height window) 3))
	      (two-thirds-y (* 2 (truncate (sheet-height window) 3))))
	  
	  (cond
	    ((< mouse-offset-x one-third-x) (push 'left sides))
	    ((> mouse-offset-x two-thirds-x) (push 'right sides)))
	  
	  (cond
	    ((< mouse-offset-y one-third-y) (push 'top sides))
	    ((> mouse-offset-y two-thirds-y) (push 'bottom sides)))

	  (unless sides
	    (setq sides '(left right top bottom)))))))

  (dynamic-resize window sides))

;--------------------------------------------------------------------------------
; Dynamic-resize:
;--------------------------------------------------------------------------------

(defun dynamic-resize (window sides)
  (let ((superior (sheet-superior window))
	(x (sheet-x window))
	(y (sheet-y window)))
    
    (multiple-value-bind (xoff yoff)
	(sheet-calculate-offsets superior mouse-sheet)
      
      (let ((x-left   (+ x xoff))
	    (x-right  (+ x xoff (sheet-width window)))
	    (y-top    (+ y yoff))
	    (y-bottom (+ y yoff (sheet-height window)))
	    (superior-width (sheet-width superior))
	    (superior-height (sheet-height superior)))
	
	(with-mouse-usurped
	  
	  (without-interrupts
	    (mouse-set-blinker-definition :grid-stay-inside-blinker 0 0 nil
					  :set-size-and-cursorpos
					  (- x-right x-left)
					  (- y-bottom y-top)
					  x-left y-top))

	  (send mouse-blinker :set-follow-p nil)
	  (send mouse-blinker :set-visibility t)
	  
	  (loop until (or (zerop (w:mouse-buttons)) (not (w:key-state :hyper))) do
		
		(setq tv:who-line-mouse-grabbed-documentation
		      (format nil "Window size ~d x ~d at ~d x ~d"
			      (- x-right x-left) (- y-bottom y-top)
			      x-left y-top))

		(multiple-value-bind (delta-x delta-y)
		    (with-timeout (20 (values 0 0))
		      (w:mouse-input t))
		  
		  (let ((old-x-left x-left)
			(old-x-right x-right)
			(old-y-top y-top)
			(old-y-bottom y-bottom))
		    (when (memq 'left sides)
		      (incf x-left delta-x))
		    (when (memq 'right sides)
		      (incf x-right delta-x))
		    (when (memq 'top sides)
		      (incf y-top delta-y))
		    (when (memq 'bottom sides)
		      (incf y-bottom delta-y))
		    (when (or (< x-left 0)
			      (> x-right superior-width)
			      (< y-top 0)
			      (> y-bottom superior-height))
		      (setq x-left old-x-left
			    x-right old-x-right
			    y-top old-y-top
			    y-bottom old-y-bottom)))

		  (send mouse-blinker :set-size-and-cursorpos
			(- x-right x-left)
			(- y-bottom y-top)
			x-left
			y-top)

		  (setq tv:who-line-mouse-grabbed-documentation
			(format nil "Window size ~d x ~d at ~d x ~d"
				(- x-right x-left) (- y-bottom y-top)
				x-left y-top))

		  ))

	  (send mouse-blinker :set-visibility nil)

	  (when (w:key-state :hyper)
	    (when (send window :set-edges (- x-left xoff) (- y-top yoff)
			(- x-right xoff) (- y-bottom yoff) :verify)
	      (send window :set-edges (- x-left xoff) (- y-top yoff)
		    (- x-right xoff) (- y-bottom yoff))))

	  )))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defflavor mouse-grid-stay-inside-blinker
	   ()
	   (mouse-box-stay-inside-blinker))

(DEFMETHOD (MOUSE-GRID-STAY-INSIDE-BLINKER :BLINK) ()
  (IF (mac-window-p sheet)
      (LET ((*dont-clip-at-the-margins* t))
	(send-drawhollowrectangle x-pos y-pos
				  width height 2
				  w:black (mouse-alu phase) sheet))
      ;; else...
      (%DRAW-RECTANGLE-CLIPPED
	2 HEIGHT X-POS                          Y-POS (MOUSE-ALU PHASE) SHEET)
      (%DRAW-RECTANGLE-CLIPPED
	(- WIDTH 2) 2 (+ X-POS 2)               Y-POS (MOUSE-ALU PHASE) SHEET)
      (%DRAW-RECTANGLE-CLIPPED
	2 (- HEIGHT 2) (+ X-POS WIDTH -2) (+ Y-POS 2) (MOUSE-ALU PHASE) SHEET)
      (%DRAW-RECTANGLE-CLIPPED
	(- WIDTH 4) 2 (+ X-POS 2) (+ Y-POS HEIGHT -2) (MOUSE-ALU PHASE) SHEET)
      ; Add the following to make grid
      (%draw-rectangle-clipped
	1 (- height 4) (+ x-pos (round width 3)) (+ y-pos 2)
	(mouse-alu phase) sheet)
      (%draw-rectangle-clipped
	1 (- height 4) (+ x-pos (lsh (round width 3) 1)) (+ y-pos 2)
	(mouse-alu phase) sheet)
      (%draw-rectangle-clipped
	(- width 4) 1 (+ x-pos 2) (+ y-pos (round height 3))
	(mouse-alu phase) sheet)
      (%draw-rectangle-clipped
	(- width 4) 1 (+ x-pos 2) (+ y-pos (lsh (round height 3) 1))
	(mouse-alu phase) sheet)
      ))


(MOUSE-DEFINE-BLINKER-TYPE :grid-stay-inside-blinker
                           #'(LAMBDA (SCREEN)
                               (MAKE-BLINKER SCREEN 'mouse-grid-stay-inside-blinker
                                 :VISIBILITY NIL)))


