;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:USER -*-

;;; File "STRIP-EXPLORER-FONTS".
;;;
;;; The TI Explorer editor, ZMACS, has support for multiple fonts in source
;;; code.  There are minor-modes and commands for automatically putting 
;;; function names in one font, docstrings in another, etc.
;;;
;;; This is aesthetically pleasing, but renders the source code unportable.
;;; The code in this file (written in Common Lisp (and without fonts)) will
;;; read in a file using the TI font format, and will write out a file
;;; stripped of the font-change codes.
;;;
;;; To satisfy your curiosity, other things which might be found in a -*- line
;;; from an Explorer are:
;;;
;;; Mode:Lisp;		means ZetaLisp code.
;;; Mode:Common-Lisp;	means CommonLisp code.
;;; Syntax:Common-Lisp;	means CommonLisp code, regardless of Mode: field.
;;;			Usually used as "Mode:Lisp; Syntax:Common-Lisp;"
;;; Package:MYPACK;	The name of the default package.
;;; Package:(MYPACK);	If wrapped in parens, the editor and/or loader will 
;;;			create the package automatically if it doesn't exist.
;;;			Arguments can be passed as well, as in
;;;			Package:(MYPACK :NICKNAMES (FOO));
;;; Base:10;		value of *READ-BASE* and *PRINT-BASE*.
;;; VSP:2;		extra pixels of inter-line spacing.
;;; Patch-File:T;	This file redefines system functions, and warnings
;;;			about this should be suppressed.
;;; Backspace:T;	Backspace characters in the file should cause
;;;			overstriking.
;;;
;;; ChangeLog:
;;;
;;; 22 Nov 89	Jamie Zawinski	Created.
;;;  4 Apr 90	Jamie Zawinski	Changed :characters t to :element-type 'string-char.


(defvar *font-change-char* (int-char 6)
  "On the Explorer, this is an Epsilon character; on systems using the ASCII
 character set, it is probably Control-F.  You may need to tweak this.")


(defun strip-explorer-fonts-from-file (input-pathname output-pathname)
  "Make a copy of the given file, stripping Explorer font-change codes."
  (with-open-file (in input-pathname :direction :input
		      :element-type 'string-char)
    (with-open-file (out output-pathname :direction :output
			 :element-type 'string-char)
      ;;
      ;; Read the properties (the -*- line) to make sure there are fonts.
      ;; Allow blank lines before it.
      (loop
	(let* ((line (read-line in)))
	  (write-line line out)
	  (when (and (> (length line) 0)
		     (not (every #'(lambda (x)
				     (or (char= x #\Space) (char= x #\Tab)))
				 line)))
	    (let* ((start (search "-*-" line :test #'char=))
		   (end (and start
			     (search "-*-" line :test #'char=
				     :start2 (+ 3 start)))))
	      (cond ((and start (null end))
		     (cerror "Try parsing anyway."
			     "The properties line has unmatched -*-'s.")
		     (setq end (length line)))
		    ((null start)
		     (cerror "Try to strip fonts anyway."
			     "There is no properties line.")
		     (setq start 0 end (length line))))
	      (let* ((fonts (search "Fonts:" line :start1 start :end2 end
				    :test #'char-equal)))
		(unless fonts
		  (cerror "Strip fonts anyway."
			  "This file doesn't claim to have any fonts."))))
	    (return))))
      (loop
	(let* ((line (read-line in nil nil)))
	  (unless line (return))
	  (strip-explorer-fonts-from-line line out)))
      (truename out))))


(defun strip-explorer-fonts-from-line (line output-stream)
  (let* ((pos (position (the string-char *font-change-char*)
			(the simple-string line)
			:test #'char=))
	 (len (length line)))
    (cond ((null pos)
	   ;; No font changes on this line.  Do it the easy way.
	   (write-line line output-stream))
	  (t
	   ;; Write out the part before the first font change.
	   (unless (zerop pos)
	     (write-string line output-stream :end pos))
	   ;; Iterate over the rest character by character.
	   (do* ((i pos (1+ i)))
		((= i len))
	     (let* ((c (char line i)))
	       (if (char/= c *font-change-char*)
		   ;; If it's not a font-change, write it.
		   (write-char c output-stream)
		   ;; Otherwise, sanity-check and discard the next char.
		   (let* ((c2 (if (= i (1- len))
				  nil
				  (char line (1+ i)))))
		     (incf i)
		     (cond ((null c2)
			    ;; Font-change followed by newline.  Bad news.
			    (warn "Font-change found at end of line - ~
 				   file likely corrupt.")
			    (return))
			   ((char= c2 *font-change-char*)
			    ;; A literal control-F.
			    (write-char *font-change-char* output-stream))
			   ((or (char= c2 #\*) (digit-char-p c2 36))
			    ;; Font change followed by 0-9, A-Z, or *.  Ok.
			    nil)
			   ((char= c2 #\#)
			    (cerror "Try to continue anyway."
				    "This file contains a Diagram Line.  ~
				     You're hosed.")
			    (write-char c output-stream)
			    (write-char c2 output-stream)
			    )
			   (t (cerror "Try to continue anyway."
				      "Unknown font change code, ~C~C" c c2)
			      (write-char c output-stream)
			      (write-char c2 output-stream)))))))
	   (terpri output-stream))))
  nil)
