;; -*- Mode:Common-Lisp; Package:(SW); Fonts:(MEDFNT TR12BI TR12B); Base:10 -*-

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;	Network Version of Explorer Space War
;
;	Author:		John Nguyen
;
;	Address:	545 Technology Square, Room 626
;			Cambridge, MA  02139
;			(617) 253-6028
;
;	E-mail:		johnn@hx.lcs.mit.edu
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(proclaim '(optimize (safety 0) (speed 3)))

(defmacro game2screen (x)
  `(ash ,x -10))

(defmacro screen2game (x)
  `(ash ,x 10))

(defvar *option-list* nil)

(defmacro define-option (name description default-value option-values)
  "define a new compiler debugging option switch"
  `(progn
     (defparameter ,name ,default-value ,description)
     (setq *option-list*
	   (delete-old-entry ',name *option-list*))
     (push-end '(,name ,description ,@option-values) *option-list*)
     nil))

(defmacro define-blank ()
  `(push-end "" *option-list*))

(defun delete-old-entry (x l)
  (delete x l :test #'(lisp:lambda (x y) (when (consp y) (eq x (car y))))))

(defun control-panel ()
  (w:choose-variable-values *option-list* :label "Space War Options"))

(define-blank)
(define-option *min-floaters* "2Min number of floaters*" 0 (:non-negative-fixnum))
(define-option *max-floaters* "2Max number of floaters*" 1 (:non-negative-fixnum))
(define-blank)
(define-option *min-shooters* "2Min number of shooters*" 0 (:non-negative-fixnum))
(define-option *max-shooters* "2Max number of shooters*" 2 (:non-negative-fixnum))
(define-blank)
(define-option *min-healers*  "2Min number of healers*"  0 (:non-negative-fixnum))
(define-option *max-healers*  "2Max number of healers*"  1 (:non-negative-fixnum))
(define-blank)
(define-option *crystals* "2Bonus crystals*" :shields (:choose (:points :shields :none)))
(define-blank)
(define-option *min-warpers*  "2Min number of warpers*"  0 (:non-negative-fixnum))
(define-option *max-warpers*  "2Max number of warpers*"  0 (:non-negative-fixnum))
(define-option *warper-gravity* "2Warper has gravity*" nil (:boolean))
(define-blank)
(define-option *cloak-p* "2Player has cloak ability*" nil (:boolean))
(define-option *bullet-limit* "2Number of player bullets*" 5 (:non-negative-fixnum))
(define-option *player-bullet-life* "2Player bullet life*" 40 (:non-negative-fixnum))
(define-option *fragment-life* "2Explosion bullet life*" 100 (:non-negative-fixnum))
(define-blank)
(define-option *gravity-choice* "2Gravity at center*" :random (:choose (:attract :repel :none :random)))
(define-option *edge-action-choice* "2Edge action*" :random (:choose (:bounce :go-through :random)))
(define-blank)

(defvar *gravity*)
(defvar *edge-action*)

(defconstant *turn-speed* 1)
(defconstant *max-shields* 350)
(defconstant *base-x* (screen2game 100))

(defconstant *player-thrust* (screen2game 2))
(defconstant *player-speed* (screen2game 5))
(defconstant *player-bullet-speed* (screen2game 6))

(defconstant *floater-speed* (screen2game 10))
(defconstant *shooter-speed* (screen2game 9))
(defconstant *warper-speed* (screen2game 5))
(defconstant *shooter-thrust* (screen2game 3))
(defconstant *warper-thrust* (screen2game 2))
(defconstant *healer-thrust* (screen2game 2))

(defconstant *shooter-bullet-life* 30)
(defconstant *shooter-bullet-speed* (screen2game 6))
(defconstant *fragment-speed* (screen2game 15))

(defconstant *healer-speed* (screen2game 8))

(defconstant *rotation* 32)
(defconstant *rot-factor* (/ 6.28318 *rotation*))
(defconstant *bsize/2* 10)

(defconstant *end-life* 150)

(defconstant *collision-grain* -4)
(defconstant *gravity-grain* -4)


(defconstant *cmd-no-command* 48)
(defconstant *cmd-new-player* 49)
(defconstant *cmd-reset-score* 50)
(defconstant *cmd-acknowledge* 51)
(defconstant *cmd-end-game* 52)
(defconstant *cmd-send-random* 53)

(defvar *net-war-random*)
(defsubst nw-random (x)
  (random x *net-war-random*))

(defsubst limit (value bound)
  (min (max value (- bound)) bound))

(defsubst square (x)
  (* x x))

(defsubst cube (x)
  (* x x x))

(defvar *keys* '(#\q #\e #\t #\u #\y))
(defconstant *key-names* '("Turn left" "Turn right" "Thrust" "Fire" "Cloak"))

(defvar *extra-keys* '(:left-shift :right-shift
		       :left-symbol :right-symbol
		       :left-control :right-control
		       :left-meta :right-meta
		       :left-super :right-hyper))

(defun get-keystroke (stream)
  (loop with key
	until key do
	(setq key (read-char-no-hang stream))
	(unless key
	  (loop for k in *extra-keys* do
		(when (w:key-state k) (setq key k))))
	finally
	(return (if (numberp key) (int-char key) key))))

(setf (get :tyi 'w:choose-variable-values-keyword)
      (list #'princ '(get-keystroke)))

(defvar *key-alist*)
(loop initially (setq *key-alist* (list ""))
      for i from 0 below 5
      collect
      (list (locf (nth i *keys*))
	    (nth i *key-names*)
	    :tyi)
      into alist
      finally (setq *key-alist* alist))

(defun define-keys ()
  (let ((old-keys (copy-list *keys*)))
    (condition-case ()
	(w:choose-variable-values *key-alist* :label "Keyboard definitions")
      (sys:abort (setq *keys* old-keys)))))

(defvar *cos* (make-array 32 :type 'art-fix))
(defvar *sin* (make-array 32 :type 'art-fix))
(loop for i from 0 below 32
      for angle = (* i (/ 6.28318 32)) do
      (setf (aref *cos* i) (truncate (* (cos angle) 1024)))
      (setf (aref *sin* i) (truncate (* (sin angle) 1024))))

(unless (boundp 'fonts:space-duel)
  (load "sys:mit.hacks;space-duel-font"))

(defflavor moving-object
	   (char
	    x-pos
	    y-pos
	    (x-speed 0)
	    (y-speed 0)
	    (alive-p t)
	    (size 32)
	    gsize/2
	    (trace t)
	    width
	    height
	    window)
	   ()
  :gettable-instance-variables
  :settable-instance-variables
  :inittable-instance-variables)

(defmethod (moving-object :after :init) (&rest ignore)
  (setq width (send window :width))	   ;1reset*
  (setq height (send window :height))
  (setq gsize/2 (ash (screen2game size) -1)))

(defmethod (moving-object :reset) ()
  (when alive-p
    (send self :draw)))

(defmethod (moving-object :draw) ()
  (when char
    (w:prepare-sheet (window)
      (tv:%draw-char fonts:space-duel char
		     (- (game2screen x-pos) *bsize/2*)
		     (- (game2screen y-pos) *bsize/2*)
		     w:alu-ior window))))

(defmethod (moving-object :erase) ()
  (when char
    (w:prepare-sheet (window)
      (tv:%draw-char fonts:space-duel char
		     (- (game2screen x-pos) *bsize/2*)
		     (- (game2screen y-pos) *bsize/2*)
		     w:alu-andca window))))

(defmethod (moving-object :die) ()
  (when alive-p
    (send window :dealloc-object self)
    (send self :erase)
    (setq alive-p nil)))

(defmethod (moving-object :tick) ()
  (when (eq alive-p t)
    (send self :erase)
    (case *gravity*
      (:attract
       (decf x-speed (send window :x-gravity x-pos y-pos))
       (decf y-speed (send window :y-gravity x-pos y-pos)))
      (:repel
       (incf x-speed (send window :x-gravity x-pos y-pos))
       (incf y-speed (send window :y-gravity x-pos y-pos))))
    (incf x-pos x-speed)
    (incf y-pos y-speed)
    (case *edge-action*
      (:bounce
       (unless (< gsize/2 x-pos (- (screen2game width) gsize/2))
	 (setq x-speed (- x-speed))
	 (incf x-pos (* 2 x-speed)))
       (unless (< gsize/2 y-pos (- (screen2game height) gsize/2))
	 (setq y-speed (- y-speed))
	 (incf y-pos (* 2 y-speed))))
      (:go-through
       (setq x-pos (+ gsize/2 (mod (- x-pos gsize/2) (- (screen2game width) (ash gsize/2 1)))))
       (setq y-pos (+ gsize/2 (mod (- y-pos gsize/2) (- (screen2game height) (ash gsize/2 1)))))))
    (send self :draw)
    (when trace
      (send window :mark size size (game2screen (- x-pos gsize/2)) (game2screen (- y-pos gsize/2))))
    (when (and (eq *gravity* :attract)
	       (< (+ (square (- (game2screen x-pos) (lsh width -1)))
		     (square (- (game2screen y-pos) (lsh height -1))))
		  50))
      (send self :warp t))
    ))

(defmethod (moving-object :warp) (&rest ignore)
  (when alive-p
    (send self :erase)
    (let ((dist (- (ash (min width height) -1) size))
	  (angle (send window :warp-angle)))
      (setq x-pos (+ (ash (screen2game width) -1) (* (aref *cos* angle) dist))
	    y-pos (+ (ash (screen2game height) -1) (* (aref *sin* angle) dist))))
    (send self :draw)))

(defmethod (moving-object :collide-sound) (player &optional (pitch :noise))
  (tv:reset-sound t)
  (let* ((myself? (equal (send player :number) (send window :my-num)))
	 (volume (if myself? 0 3)))
    (if (eq pitch :noise)
	(progn
	  (tv:do-sound (tv:noise 0 (if myself? 0 1)))
	  (tv:do-sound (tv:volume 3 volume)))
	(progn
	  (tv:do-sound (tv:tone 0 pitch))
	  (tv:do-sound (tv:volume 0 volume))))))

(defflavor player
	   (number
	    (name "")
	    shields
	    character-offset
	    dir
	    bullet-count
	    cloak
	    (score 0))
	   (moving-object)
  (:default-init-plist
    :trace nil)
  :gettable-instance-variables
  :settable-instance-variables
  :inittable-instance-variables)

(defmethod (player :before :reset) ()
  (setq shields 0)
  (setq bullet-count 0)
  (setq x-speed 0)
  (setq y-speed 0)
  (setq cloak nil)
  (setq character-offset (if (evenp number) 64 96))
  (setq x-pos (if (evenp number) *base-x* (- (screen2game width) *base-x*)))
  (setq y-pos (+ (if (< number 2) 0 (ash (screen2game height) -1))
		 (ash (screen2game height) -2)))
  (send self :set-direction (if (evenp number) 0 16))
  (send self :draw-score)
  (send self :change-shields *max-shields*))

(defmethod (player :set-direction) (d)
  (setq dir (mod d *rotation*)))

(defmethod (player :draw) ()
  (setq char (+ character-offset dir))
  (unless cloak
    (w:prepare-sheet (window)
      (tv:%draw-char fonts:space-duel char
		     (- (game2screen x-pos) *bsize/2*)
		     (- (game2screen y-pos) *bsize/2*)
		     w:alu-ior window)
      (when (> number 1)
	(tv:%draw-char fonts:space-duel 10
		       (- (game2screen x-pos) *bsize/2*)
		       (- (game2screen y-pos) *bsize/2*)
		       w:alu-ior window)))))

(defmethod (player :erase) ()
  (w:prepare-sheet (window)
    (tv:%draw-char fonts:space-duel char
		   (- (game2screen x-pos) *bsize/2*)
		   (- (game2screen y-pos) *bsize/2*)
		   w:alu-andca window)
    (when (> number 1)
      (tv:%draw-char fonts:space-duel 10
		     (- (game2screen x-pos) *bsize/2*)
		     (- (game2screen y-pos) *bsize/2*)
		     w:alu-andca window))))

(defmethod (player :after :tick) ()
  (when alive-p
    (when cloak
      (if (<= shields 50)
	  (setq cloak nil)
	  (send self :change-shields -1)))
    (let ((sp (sqrt (+ (square x-speed) (square y-speed)))))
      (unless (zerop sp)
;	(let ((player-drag (* (expt (/ sp *player-speed*) 4)
;			      *player-thrust*)))
	(let ((player-drag (max (- sp *player-speed*) 0)))
	  (decf x-speed (truncate (/ (* player-drag x-speed) sp)))
	  (decf y-speed (truncate (/ (* player-drag y-speed) sp))))))))


(defmethod (player :thrust) ()
  (send self :add-vector
	(ash (* *player-thrust* (aref *cos* dir)) -10)
	(ash (* *player-thrust* (aref *sin* dir)) -10)))

(defmethod (player :add-vector) (dx dy)
  (incf x-speed dx)
  (incf y-speed dy))

(defmethod (player :before :warp) (&optional damage)
  (when damage
    (send self :change-shields -80)))

(defmethod (player :fire) ()
  (when (< bullet-count *bullet-limit*)
    (incf bullet-count)
    (send window
	  :make-bullet self
	  (+ x-pos (ash (* (+ gsize/2 (ash gsize/2 -1)) (aref *cos* dir)) -10))
	  (+ y-pos (ash (* (+ gsize/2 (ash gsize/2 -1)) (aref *sin* dir)) -10))
	  dir *player-bullet-speed*
	  x-speed y-speed *player-bullet-life* 2)))

(defmethod (player :before :set-cloak) (c)
  (setq cloak c)
  (when cloak
    (if (<= shields 50)
	(setq cloak nil)
	(send self :change-shields -1))))

(defmethod (player :kill-bullet) ()
  (decf bullet-count))

(defmethod (player :turn-left) ()
  (send self :set-direction (- dir *turn-speed*)))

(defmethod (player :turn-right) ()
  (send self :set-direction (+ dir *turn-speed*)))

(defsubst collidep (object d)
  (let ((dx (abs (- (send object :x-pos) x-pos)))
	(dy (abs (- (send object :y-pos) y-pos))))
    (and (< dx d) (< dy d))))

(defmethod (player :collidep) (object)
  (when (and alive-p (not (eq object self)))
    (collidep object (send object :gsize/2))))

(defmethod (player :change-shields) (x)
  (when alive-p
    (when (and (plusp x) (< shields *max-shields*))
      (tv:do-sound (tv:volume 1 0)))
    (send self :draw-shields
	  (if (plusp x)
	      (min (- *max-shields* shields) x)
	      (min shields x)))
    (setq shields (min (+ shields x) *max-shields*))
    (when (< shields 0)
      (send self :die)
      (send window :blow-up x-pos y-pos))))

(defmethod (player :collide) (player)
  (let* ((dxs (ash (- (send player :x-speed) x-speed) -1))
	 (dys (ash (- (send player :y-speed) y-speed) -1)))
    (send self :collide-sound player)
    (send player :set-x-speed (- dxs))
    (send player :set-y-speed (- dys))
    (setq x-speed dxs)
    (setq y-speed dys)
    (dotimes (i 2)
      (send player :tick)
      (send self :tick))
    (let ((damage (* (game2screen (+ (abs dxs) (abs dys))) -3)))
      (send player :change-shields damage)
      (send self :change-shields damage))))

(defmethod (player :draw-shields) (d)
  (let ((stats-window (send window :stats-window))
	(s shields)
	(y (+ 9 (* 20 (lsh number -1)))))
    (when (< width 900)
      (setq s (ash s -1)
	    d (ash d -1)))
    (if (plusp d)
	(if (evenp number)
	    (send stats-window :draw-line s y (+ s d) y 5)
	    (send stats-window :draw-line (- width s) y (- width s d) y 5))
	(if (evenp number)
	    (send stats-window :draw-line s y (+ s d) y 5 2)
	    (send stats-window :draw-line (- width s) y (- width (+ s d)) y 5 2)))))

(defmethod (player :after :set-score) (ignore)
  (send self :draw-score))

(defmethod (player :draw-score) ()
  (when alive-p
    (send (send window :stats-window) :string-out-explicit
	  (if (evenp number)
	      (format nil "~8@a ~6,48d" name score)
	      (format nil "~6,48d ~8a" score name))
	  (+ (ash width -1) (if (evenp number) -130 20))
	  (+ 4 (* 20 (lsh number -1)))
	  nil nil fonts:medfnt w:normal)))

(defflavor bullet
	   (owner
	    (rot nil)
	    char
	    time-left)
	   (moving-object)
  (:default-init-plist
    :size 22)
  :gettable-instance-variables
  :settable-instance-variables
  :inittable-instance-variables)

(defmethod (bullet :before :reset) ()
  (setq alive-p nil))

(defmethod (bullet :after :tick) ()
  (when (and alive-p (zerop (decf time-left)))
    (send self :die)))

(defmethod (bullet :before :die) ()
  (when owner
    (send owner :kill-bullet)))

(defmethod (bullet :collide) (player)
  (send player :change-shields -12)
  (send self :collide-sound player)
  (when (and (typep owner 'player) (not (equal (send owner :number) (send player :number))))
    (send owner :set-score (+ (send owner :score) 20)))
  (send self :die))

(defflavor bad-guy
	   (damage)
	   (moving-object)
  :inittable-instance-variables)

(defmethod (bad-guy :before :reset) ()
  (setq x-pos (screen2game (+ 10 (* -20 (nw-random 2)) (ash width -1))))
  (setq y-pos (screen2game (+ 50 (nw-random (- height 100))))))

(defmethod (bad-guy :collide) (player)
  (let* ((dxs (ash (- (send player :x-speed) x-speed) -1))
	 (dys (ash (- (send player :y-speed) y-speed) -1))
	 (tds (/ *player-speed* (coerce (max (sqrt (+ (* dxs dxs) (* dys dys))) 1) 'float)))
	 (x-power (truncate (* dxs tds)))
	 (y-power (truncate (* dys tds))))
    (send self :collide-sound player)
    (send player :set-x-speed (- x-power))
    (send player :set-y-speed (- y-power))
    (setq x-speed x-power)
    (setq y-speed y-power)
    (dotimes (i 3)
      (send player :tick)
      (send self :tick)))
  (send player :change-shields damage))

(defflavor floater
	   ()
	   (bad-guy)
  (:default-init-plist
    :damage -10))

(defmethod (floater :before :reset) ()
  (let ((dir (nw-random 32)))
    (setq x-speed (ash (* *floater-speed* (aref *cos* dir)) -10))
    (setq y-speed (ash (* *floater-speed* (aref *sin* dir)) -10))))

(defflavor shooter
	   (victim)
	   (bad-guy)
  :gettable-instance-variables
  :settable-instance-variables
  (:default-init-plist
    :damage -15))

(defmethod (shooter :before :reset) ()
  (setq victim nil)
  (setq x-speed 0)
  (setq y-speed 0))

(defmethod (shooter :before :tick) ()
  (when alive-p
    (when (or (not victim)
	      (not (send victim :alive-p))
	      (zerop (nw-random 100)))
      (setq victim (nth (nw-random (length (send window :players))) (send window :players))))
    (when (and (zerop (nw-random 8)) (send victim :alive-p))
      (let* ((dx (- (send victim :x-pos) x-pos))
	     (dy (- (send victim :y-pos) y-pos))
	     (dist (max (sqrt (+ (* dx dx) (* dy dy))) 1)))
	(let ((dtx (truncate (- (/ (* *shooter-speed* dx) dist) x-speed)))
	      (dty (truncate (- (/ (* *shooter-speed* dy) dist) y-speed))))
	  (incf x-speed (limit dtx *shooter-thrust*))
	  (incf y-speed (limit dty *shooter-thrust*)))
	(when (zerop (nw-random 4))
	  (send window :make-bullet
		self x-pos y-pos nil 0
		(+ (truncate (/ (* *shooter-bullet-speed* dx) dist)) x-speed)
		(+ (truncate (/ (* *shooter-bullet-speed* dy) dist)) y-speed)
		*shooter-bullet-life* 3))))))

(defmethod (shooter :kill-bullet) ()
  (setq victim nil))

(defmethod (shooter :after :collide) (ignore)
  (setq victim nil))

(defflavor warper
	   (damage)
	   (moving-object)
  :gettable-instance-variables
  :settable-instance-variables
  (:default-init-plist
    :damage -30))

(defmethod (warper :before :reset) ()
  (setq x-pos (screen2game (+ 10 (* -20 (nw-random 2)) (ash width -1))))
  (setq y-pos (screen2game (+ 50 (nw-random (- height 100)))))
  (setq x-speed 0)
  (setq y-speed 0))

(defmethod (warper :before :draw) ()
  (setq char (+ 60 (logand (ash (time) -3) 1))))

(defmethod (warper :before :tick) ()
  (when alive-p
    (when (zerop (nw-random 7))
      (setq y-speed (limit (+ y-speed (* *warper-thrust* (- (* (nw-random 2) 2) 1)))
			   *warper-speed*))
      (setq x-speed (limit (+ x-speed (* *warper-thrust* (- (* (nw-random 2) 2) 1)))
			   *warper-speed*)))
    (when *warper-gravity*
      (loop for player in (send window :players)
	    for dx = (- (send player :x-pos) x-pos)
	    for dy = (- (send player :y-pos) y-pos)
	    for dist = (max (sqrt (+ (* dx dx) (* dy dy))) 1)
	    for power = (* (expt (/ (- (screen2game 1500) dist)
				 (coerce (screen2game 1500) 'float)) 7)
			   .7 2048)
	    when (send player :alive-p) do
	    (send player :add-vector 
		  (- (truncate (/ (* power dx) dist)))
		  (- (truncate (/ (* power dy) dist))))))))

(defmethod (warper :collide) (player)
  (send self :collide-sound player)
  (send player :warp nil)
  (send player :change-shields damage))

(defflavor healer
	   ()
	   (moving-object)
  )

(defmethod (healer :before :reset) ()
  (setq x-speed 0)
  (setq y-speed 0)
  (setq x-pos (screen2game (+ 10 (* -20 (nw-random 2)) (ash width -1))))
  (setq y-pos (screen2game (+ 50 (nw-random (- height 100))))))

(defmethod (healer :before :tick) ()
  (when (and alive-p (zerop (nw-random 5)))
    (when (and (not (eq *crystals* :none)) (zerop (nw-random 50)))
      (send window :make-crystal x-pos y-pos))
    (setq y-speed (limit (+ y-speed (* *healer-thrust* (- (* (nw-random 2) 2) 1))) *healer-speed*))
    (setq x-speed (limit (+ x-speed (* *healer-thrust* (- (* (nw-random 2) 2) 1))) *healer-speed*))))

(defmethod (healer :before :draw) ()
  (setq char (+ 48 (logand (ash (time) -2) 3))))

(defmethod (healer :collide) (player)
  (send self :collide-sound player (+ 200 (* (send player :number) 200)))
  (send player :change-shields 5))


(defflavor crystal
	   ((bonus)
	    (bonus-string)
	    (time-count))
	   (moving-object)
  (:default-init-plist
    :size 30)
  :gettable-instance-variables
  :settable-instance-variables
  :inittable-instance-variables)

(defun random-crystal-bonus ()
  (case *crystals*
    (:points (+ (nw-random 40)
		(case (nw-random 20)
		  (0 (nw-random 100))
		  (1 (nw-random 50))
		  (t 10))))
    (:shields (1+ (nw-random 50)))))

(defmethod (crystal :before :reset) ()
  (setq alive-p nil)
  (setq char 18)
  (setq bonus (random-crystal-bonus))
  (setq bonus-string (format nil "~d" bonus))
  (setq x-speed (screen2game (- (nw-random 20) 10)))
  (setq y-speed (screen2game (- (nw-random 20) 10)))
  (setq time-count (+ 20 (nw-random 300))))

(defmethod (crystal :collide) (player)
  (when (eq alive-p t)
    (send self :collide-sound player 2000)
    (send self :erase)
    (setq char 0)
    (setq x-speed 0
	  y-speed 0)
    (setq alive-p 'touched)
    (case *crystals*
      (:points 
       (send player :set-score (+ bonus (send player :score))))
      (:shields
       (send player :change-shields bonus)))))

(defmethod (crystal :after :tick) ()
  (when alive-p
    (case alive-p
      (touched
       (setq alive-p 'waiting
	     time-count 5))
      (waiting
       (when (zerop (decf time-count))
	 (setq alive-p 'show-bonus time-count 20 )))
      (show-bonus
       (w:prepare-sheet (window)
	 (send window :string-out-explicit bonus-string
	       (- (game2screen x-pos) *bsize/2*)
	       (- (game2screen y-pos) *bsize/2*)
	       width height fonts:courier w:alu-ior))
       (when (zerop (decf time-count))
	 (setq alive-p 'dead)))
      (dead
       (w:prepare-sheet (window)
	 (send window :string-out-explicit bonus-string
	       (- (game2screen x-pos) *bsize/2*)
	       (- (game2screen y-pos) *bsize/2*)
	       width height fonts:courier w:alu-andca))
       (send self :die))
      (t
       (when (and (> bonus 100) (zerop (ldb (byte 1 1) time-count)))
	 (send self :erase))
       (when (zerop (decf time-count))
	 (setq alive-p 'dead)
	 (send self :die)))
       )))

(defflavor graviton
	   ()
	   (moving-object))

(defmethod (graviton :before :reset) ()
  (setq x-speed 0
	y-speed 0)
  (setq x-pos (screen2game (ash width -1))
	y-pos (screen2game (ash height -1))))

(defmethod (graviton :before :draw) ()
  (setq char (+ 44 (if (eq *gravity* :repel)
		       (logand (lsh (time) -2) 3)
		       (logand (- (lsh (time) -2)) 3)))))

(defmethod (graviton :warp) (ignore)
  ())

(defmethod (graviton :collide) (player)
  (ignore player))

(defflavor space-war-window
	   ((players nil)
	    (bullets nil)
	    (crystals nil)
	    (objects nil)
	    num-objects
	    (collision-array nil)
	    (gravity-array nil)
	    alive
	    stats-window
	    type-window
	    end-count
	    angle-count
	    backup-array
	    chaos-stream
	    server-p
	    (conn nil)
	    my-num
	    num-players
	    (player-keys (make-string 5))
	    )
	   (w:essential-window-with-typeout-mixin
	    w:process-mixin
	    w:select-mixin
	    w:graphics-mixin
	    w:stream-mixin
	    w:borders-mixin
	    w:delay-notification-mixin
	    w:minimum-window)
  (:default-init-plist
    :typeout-window '(w:typeout-window)
    :deexposed-typeout-action :permit
    :blinker-p nil
    :save-bits t
    :font-map '(fonts:medfnt))
  (:settable-instance-variables stats-window end-count)
  :gettable-instance-variables)

(defmethod (space-war-window :after :init) (&rest ignore)
  (setq w:process (w:make-process "Space War"))
  (send w:process :run-reason self)
  (setq type-window (send self :typeout-window))
  (process-preset w:process self :loop)
  (loop for i from 4 downto 1 do
	(push (make-instance 'player
			     :number (1- i)
			     :char 64
			     :window self)
	      players))
  (send self :setup))

(defmethod (space-war-window :before :kill) (&rest ignore)
  (tv:sib-sound-bit :off)
  (when conn
    (send conn :close)))

(defmethod (space-war-window :before :deselect) (&rest ignore)
  (tv:sib-sound-bit :off))

(defmethod (space-war-window :get-connection) ()
  (send self :clear-screen)
  (tv:sib-sound-bit :off)
  (send self :help nil)
  (let ((server
	  (fquery `(:type :readline :stream ,self :choices (:any) :list-choices nil)
		  "~%~%               Enter server name or return to be server: ")))
    (if (zerop (length server))
	(progn
	  (setq server-p t)
	  (setq chaos-stream nil)
	  (setq *net-war-random* (make-random-state *random-state*))
	  (send (car players) :set-name (user-name))
	  (setq my-num 0)
	  (setq num-players 1))
	(progn
	  (setq server-p nil)
	  (setq chaos-stream (chaos:open-stream server "net-war"))
	  (prin1 (user-name) chaos-stream)
	  (send chaos-stream :force-output)
	  (setq my-num (tyi chaos-stream))
	  (send (nth my-num players) :set-name (user-name))
	  (loop for player in players
		for i from 0 below my-num do
		(send player :set-name (read chaos-stream)))
	  (setq num-players (1+ my-num))))))

(defmethod (space-war-window :look-for-connection) ()
  (unless conn
    (setq conn (chaos:listen "net-war" chaos:default-window-size nil)))
  (when (eq (chaos:conn-state conn) 'chaos:rfc-received-state)
    (if (< num-players 4)
	(progn
	  (chaos:accept conn)
	  (let ((new-stream (chaos:make-stream conn :ascii-translation nil)))
	    (setq conn nil)
	    (let ((name (read new-stream)))
	      (send (nth num-players players) :set-name name)
	      (write-char num-players new-stream)
	      (loop for player in players
		    for i from 0 below num-players do
		    (prin1 (send player :name) new-stream))
	      (send new-stream :force-output)
	      (incf num-players)
	      (send self :broadcast *cmd-new-player*)
	      (loop for stream in chaos-stream do
		    (prin1 name stream)
		    (send stream :force-output))
	      (push-end new-stream chaos-stream))
	    (send self :reset-scores)
	    (send self :reset)))
	(chaos:reject conn "We already have 4 players"))))

(defmethod (space-war-window :check-for-cmds) ()
  (loop for cmd = (tyi chaos-stream)
	until (equal cmd *cmd-no-command*) do
	(selector cmd =
	  (*cmd-new-player*
	   (write-char *cmd-acknowledge* chaos-stream)
	   (send chaos-stream :force-output)
	   (send (nth num-players players) :set-alive-p t)
	   (send (nth num-players players) :set-name (read chaos-stream))
	   (incf num-players)
	   (send self :reset-scores)
	   (send self :reset))
	  (*cmd-reset-score*
	   (write-char *cmd-acknowledge* chaos-stream)
	   (send chaos-stream :force-output)
	   (send self :reset-scores)
	   (send self :reset))
	  (*cmd-end-game*
	   (write-char *cmd-acknowledge* chaos-stream)
	   (send chaos-stream :force-output)
	   (throw 'restart t))
	  )))

(defmethod (space-war-window :broadcast) (cmd)
  (when server-p
    (loop for stream in chaos-stream do
	  (write-char cmd stream)
	  (send stream :force-output))
    (loop for stream in chaos-stream do
	  (loop until (equal (tyi stream) *cmd-acknowledge*)))))

(defmethod (space-war-window :input-1) ()
  (loop for i from 0 below 5
	for key in *keys*
	sum (if (w:key-state key) (lsh 1 i) 0)
	into player-key
	finally (setf (aref player-keys my-num) (+ 64 player-key)))
  (unless server-p
;    (write-char angle-count chaos-stream)
    (write-char (aref player-keys my-num) chaos-stream)
    (send chaos-stream :force-output)))

(defmethod (space-war-window :input-2) ()
  (when server-p
    (loop for i from 1 below num-players
	  for stream in chaos-stream
	  for player in (cdr players)
	  do
	  (setf (aref player-keys i) (tyi stream))
	  (loop while (send stream :tyi-no-hang) do ()))
    (send self :look-for-connection)
    (loop for stream in chaos-stream
	  for player in (cdr players)
	  do
	  (write-char *cmd-no-command* stream)
	  (princ player-keys stream)
	  (send stream :force-output)))
  (send self :get-keys))

(defmethod (space-war-window :input-3) ()
  (unless server-p
    (send self :check-for-cmds)
    (loop for i from 0 below num-players
	  do
	  (setf (aref player-keys i) (tyi chaos-stream))))
  (loop for i from 0 below num-players
	for player in players
	for kb-state = (aref player-keys i)
	when (send player :alive-p) do
	(when (plusp (ldb (byte 1 0) kb-state))
	  (send player :turn-left))
	(when (plusp (ldb (byte 1 1) kb-state))
	  (send player :turn-right))
	(when (plusp (ldb (byte 1 2) kb-state))
	  (send player :thrust))
	(when (plusp (ldb (byte 1 3) kb-state))
	  (send player :fire))
	(when *cloak-p*
	  (if (plusp (ldb (byte 1 4) kb-state))
	      (send player :set-cloak t)
	      (send player :set-cloak nil)))))

(defmethod (space-war-window :setup) ()
  (setq bullets '())
  (setq crystals nil)
  (setq objects (copy-list players))
  (loop for i from 0 below *max-floaters* do
	(push (make-instance 'floater
			     :char 1
			     :window self)
	      objects))
  (loop for i from 0 below *max-shooters* do
	(push (make-instance 'shooter
			     :char 6
			     :window self)
	      objects))
  (loop for i from 0 below *max-warpers* do
	(push (make-instance 'warper
			     :char 5
			     :window self)
	      objects))
  (loop for i from 0 below *max-healers* do
	(push (make-instance 'healer
			     :char 48
			     :window self)
	      objects))
  (push (make-instance 'graviton :char 44 :window self) objects)
  )

(defsubst random-limit (min max)
  (if (= min max)
      min
      (+ min (nw-random (1+ (- max min))))))

(defmacro draw-star ()
  `(w:prepare-sheet (self)
     (let ((x (+ (w:sheet-inside-left)
		 (nw-random (w:sheet-inside-width))))
	   (y (+ (w:sheet-inside-top)
		 (nw-random (w:sheet-inside-height)))))
       (setf (aref w:screen-array y x) 1))))

(defmethod (space-war-window :make-arrays) ()
  (setq backup-array (make-array (list w:height (dpb 0 (byte 5 0) (+ w:width 31))) :type 'art-1b))
  (setq collision-array (make-array (list (+ 4 (ash w:height *collision-grain*))
					  (+ 4 (ash w:width  *collision-grain*)))
				    :type 'art-1b))
  (setq gravity-array (make-array (list (+ 4 (ash w:height *gravity-grain*))
					(+ 4 (ash w:width *gravity-grain*))
					2)
				  :type 'art-fix))
  (loop with x-mid = (ash (array-dimension gravity-array 1) -1)
	and y-mid = (ash (array-dimension gravity-array 0) -1)
	for y from 0 below (array-dimension gravity-array 0) do
	(loop for x from 0 below (array-dimension gravity-array 1)
	      for dx = (- x x-mid)
	      for dy = (- y y-mid)
	      for dist = (sqrt (+ (square dx) (square dy)))
	      for power = (max (* dist dist dist) 20)
	      do
	      (setf (aref gravity-array y x 0)
		    (truncate (/ (ash dy 15) power)))
	      (setf (aref gravity-array y x 1)
		    (truncate (/ (ash dx 15) power))))))

(defmethod (space-war-window :y-gravity) (x y)
  (aref gravity-array
	(+ 2 (ash (game2screen y) *gravity-grain*))
	(+ 2 (ash (game2screen x) *gravity-grain*))
	0))

(defmethod (space-war-window :x-gravity) (x y)
  (aref gravity-array
	(+ 2 (ash (game2screen y) *gravity-grain*))
	(+ 2 (ash (game2screen x) *gravity-grain*))
	1))

(defmethod (space-war-window :reset) (&optional (throw t))
  (if server-p
      (progn
	(send self :broadcast *cmd-send-random*)
	(loop for stream in chaos-stream do
	      (prin1 *net-war-random* stream)
	      (send stream :force-output)))
      (loop for c = (tyi chaos-stream)
	    while (/= c *cmd-send-random*)
	    finally
	    (write-char *cmd-acknowledge* chaos-stream)
	    (send chaos-stream :force-output)
	    (setq *net-war-random* (read chaos-stream))))
  (tv:sib-sound-bit :off)
  (unless (and collision-array gravity-array)
    (send self :make-arrays))
  (send self :clear-screen)
  (send stats-window :clear-screen)
  (setq end-count nil)
  (setq angle-count 0)
  (setq alive num-players)
  (setq num-objects 0)
  (setq *gravity*
	(if (eq *gravity-choice* :random)
	    (nth (nw-random 3) '(:attract :repel :none))
	    *gravity-choice*))
  (setq *edge-action*
	(if (eq *edge-action-choice* :random)
	    (nth (nw-random 2) '(:bounce :go-through))
	    *edge-action*))
  (let ((table `((floater ,(random-limit *min-floaters* *max-floaters*))
		 (shooter ,(random-limit *min-shooters* *max-shooters*))
		 (warper ,(random-limit *min-warpers* *max-warpers*))
		 (healer ,(random-limit *min-healers* *max-healers*))
		 (graviton ,(if (eq *gravity* :none) 0 1))
		 (bullet ,0))))
    (loop for object in objects do
	  (send object :set-alive-p nil))
    (loop for entry in table
	  for type = (car entry)
	  for count = (cadr entry) do
	  (loop for object in objects do
		(when (and (eq (type-of object) type) (plusp count))
		  (decf count)
		  (send self :alloc-object object)
		  (send object :reset)
		  (send object :set-alive-p t)))))
  (loop for player in players do
	(when (< (send player :number) num-players)
	  (send self :alloc-object player)
	  (send player :set-alive-p t))
	(send player :reset))
  (dotimes (i 1000)
    (draw-star))
  (when throw
    (throw 'restart nil)))

(defmethod (space-war-window :alloc-object) (object)
  (let ((i (position object objects)))
    (when (> i num-objects)
      (let ((x (nth i objects)))
	(setf (nth i objects) (nth num-objects objects))
	(setf (nth num-objects objects) x)))
    (incf num-objects)))

(defmethod (space-war-window :dealloc-object) (object)
  (let ((i (position object objects)))
    (decf num-objects)
    (when (< i num-objects)
      (let ((x (nth i objects)))
	(setf (nth i objects) (nth num-objects objects))
	(setf (nth num-objects objects) x)))))

(defmethod (space-war-window :mark) (w h x y)
  (let ((w/2 (ash w -1))
	(h/2 (ash h -1)))
    (unless (zerop alive)
      (loop for x1 from (ash (- x w/2) *collision-grain*) to (ash (+ x w/2) *collision-grain*) do
	    (loop for y1 from (ash (- y h/2) *collision-grain*) to (ash (+ y h/2) *collision-grain*) do
		  (setf (aref collision-array (+ y1 2) (+ x1 2)) 1))))))

(defmethod (space-war-window :loop) ()
  (let ((*debug-io* type-window))
    (do-forever
      (when (send type-window :bottom-reached)
	(send type-window :clear-screen)
	(send type-window :deactivate))
      (send self :get-connection)
      (send self :reset nil)
      (do-forever 
	(when
	  (catch 'restart
	    (when (send type-window :bottom-reached)
	      (send type-window :clear-screen)
	      (send type-window :deactivate))
	    (tv:sib-sound-bit :on)
	    (process-allow-schedule)
	    (tv:do-sound (tv:volume 0 15))
	    (tv:do-sound (tv:volume 1 15))
	    (unless end-count
	      (tv:do-sound (tv:volume 3 15)))
	    (send self :input-1)
	    (unless (zerop alive)
	      (tv:%draw-rectangle (array-dimension collision-array 1)
				  (array-dimension collision-array 0)
				  0 0 w:erase collision-array))
	    (loop for object in objects
		  for i from 0 below num-objects do
		  (send object :tick))
	    (unless (zerop alive)
	      (send self :detect-collisions))
	    (send self :input-2)
	    (draw-star)
	    (setq angle-count (logand (1+ angle-count) 255))
	    (loop for object in objects
		  for i from 0 below num-objects do
		  (send object :tick))
	    (unless (zerop alive)
	      (send self :detect-collisions))
	    (send self :input-3)
	    (when end-count
	      (tv:do-sound (tv:noise 1 2))
 	      (tv:do-sound (tv:volume 3 (- 5 (floor (* end-count 5) *end-life*))))
	      (when (<= (decf end-count) 0)
		(if (<= alive 1)
		    (send self :end-game)
		    (setq end-count nil)))))
	(return)
	)))))

(defmethod (space-war-window :warp-angle) ()
  (ash angle-count -3))

(defmethod (space-war-window :get-keys) ()
  (loop for char = (send self :tyi-no-hang)
	while char do
	(selector char =
	  (#\help
	   (send self :help))
	  (#\space
	   (tv:sib-sound-bit :off)
	   (send self :tyi))
;	  (#\f1
;	  (control-panel)
;	  (send self :setup)
;	  (send self :reset))
	  (#\f2
	   (define-keys))
	  (#\end
	   (when server-p
	     (send self :broadcast *cmd-end-game*)
	     (throw 'restart t)))
	  (#\clear-input
	   (when server-p
	     (send self :reset-scores)
	     (send self :broadcast *cmd-reset-score*)
	     (send self :reset)))
	)))


(defmethod (space-war-window :reset-scores) ()
  (loop for player in players do
	(send player :set-score 0)))
  

(defmethod (space-war-window :end-game) (&aux winner)
  (loop for player in players do
	(when (send player :alive-p)
	  (setq winner player)))
  (tv:do-sound (tv:volume 1 15))
  (tv:do-sound (tv:volume 2 15))
  (tv:do-sound (tv:volume 3 15))
  (when winner
    (tv:do-sound (tv:tone 0 (+ 200 (* (send winner :number) 200))))
    (let ((x1 (- (ash w:width -1) 100))
	  (x2 (+ (ash w:width -1) 70))
	  (y1 (- (ash w:height -1) 50))
	  (y2 (+ (ash w:height -1) 50)))
      (send self :string-out-explicit "Points for win:" x1 y1 nil nil fonts:medfnt w:normal)
      (loop for i from 20 to 800 by 20 do
	    (send self :get-keys)
	    (unless end-count
	      (throw 'restart nil))
	    (process-sleep 2)
	    (tv:do-sound (tv:volume 0 2))
	    (send self :set-cursorpos x2 y1)
	    (send self :clear-string "00000")
	    (send winner :set-score (+ (send winner :score) 20))
	    (format self "~4d" i)
	    (tv:do-sound (tv:volume 0 15)))
      (send self :string-out-explicit "  Shield bonus:" x1 y2 nil nil fonts:medfnt w:normal)
      (loop for i from 5 to (send winner :shields) by 5 do
	    (send self :get-keys)
	    (unless end-count
	      (throw 'restart nil))
	    (process-sleep 2)
	    (tv:do-sound (tv:volume 0 2))
	    (send self :set-cursorpos x2 y2)
	    (send self :clear-string "00000")
	    (send winner :change-shields -5)
	    (send winner :set-score (+ (send winner :score) 10))
	    (format self "~4d" (ash i 1))
	    (tv:do-sound (tv:volume 0 15)))
      (sleep 2)))
  (send self :reset))

(defmethod (space-war-window :detect-collisions) ()
  (loop for player in players do
	(when (and (not (send player :cloak))
		   (send self :collidep player))
	  (loop for object in objects
		for i from 0 below num-objects
		when (and (send object :alive-p) (not (typep object 'player)))
		do
		(when (send player :collidep object)
		  (send object :collide player)))))
  (loop for player1 in players
	for i from 0 below num-players
	do
	(loop for player2 in players
	      for j from 0 below num-players
	      do
	      (when (and (/= i j)
			 (send player1 :alive-p)
			 (send player2 :alive-p)
			 (send player1 :collidep player2))
		(send player1 :collide player2)))))

(defmethod (space-war-window :collidep) (player)
  (plusp (aref collision-array
	       (+ 2 (ash (game2screen (send player :y-pos)) *collision-grain*))
	       (+ 2 (ash (game2screen (send player :x-pos)) *collision-grain*)))))

(defmethod (space-war-window :make-bullet) (owner x y dir sp x-speed y-speed life char)
  (let ((bullet (or (find-if #'(lambda (x)
				 (not (send x :alive-p)))
			     bullets)
		    (let ((b (make-instance 'bullet
					    :char char
					    :window self)))
		      (push b objects)
		      (push b bullets)
		      b))))
    (send self :alloc-object bullet)
    (send bullet :set-char char)
    (send bullet :set-alive-p t)
    (send bullet :set-owner owner)
    (send bullet :set-x-speed (+ x-speed
				 (if dir (ash (* sp (aref *cos* dir)) -10)
				     0)))
    (send bullet :set-y-speed (+ y-speed
				 (if dir (ash (* sp (aref *sin* dir)) -10)
				     0)))
    (send bullet :set-x-pos x)
    (send bullet :set-y-pos y)
    (send bullet :set-time-left life)
    (send bullet :draw)))

(defmethod (space-war-window :make-crystal) (x y)
  (let ((crystal (or (find-if #'(lambda (x)
				 (not (send x :alive-p)))
			     crystals)
		     (let ((c (make-instance 'crystal
					     :window self)))
		       (push c objects)
		       (push c crystals)
		       c))))
    (send self :alloc-object crystal)
    (send crystal :reset)
    (send crystal :set-alive-p t)
    (send crystal :set-x-pos x)
    (send crystal :set-y-pos y)
    (send crystal :draw)))

(defmethod (space-war-window :blow-up) (x y)
  (decf alive)
  (tv:do-sound (tv:volume 0 15))
  (tv:do-sound (tv:volume 1 15))
  (tv:do-sound (tv:volume 2 15))
  (tv:do-sound (tv:noise 1 2))
  (tv:do-sound (tv:volume 3 0))
  (loop for i from 0 below 32 by 4 do
	(send self :make-bullet nil x y i *fragment-speed* 0 0 *fragment-life* 9))
  (loop for i from 2 below 32 by 4 do
	(send self :make-bullet nil x y i (ash *fragment-speed* -1) 0 0 *fragment-life* 9))
  (setq end-count *end-life*))

(defmethod (space-war-window :after :change-of-size-or-margins) (&rest ignore)
  (loop for object in objects do
	(send object :set-width w:width)
	(send object :set-height w:height)
	(send object :set-alive-p nil))
  (send self :make-arrays)
  (setq end-count 0))

(defflavor space-war-stats
	   ()
	   (w:graphics-mixin w:stream-mixin w:borders-mixin w:minimum-window)
  (:default-init-plist
    :font-map '(fonts:medfnt)
    :save-bits t
    :blinker-p nil))

(defmacro help-line (char desc)
  `(progn
    (setq w:cursor-x (- (lsh w:width -1) 150))
    (format self "~a" ,char)
    (setq w:cursor-x (lsh w:width -1))
    (send self :string-out ,desc)
    (incf w:cursor-y 17)))

(defmethod (space-war-window :print-intro) ()
  (send self :string-out-centered-explicit
	"Space War" (tv:sheet-inside-left) (floor w:height 10) (tv:sheet-inside-right)
	(tv:sheet-inside-right) fonts:43vxms)
  (send self :string-out-centered-explicit
	"Commands" (tv:sheet-inside-left) (+ (floor w:height 10) 50) (tv:sheet-inside-right)
	(tv:sheet-inside-right) fonts:43vxms)
  (send self :set-cursorpos 0 (floor w:height 3)))

(defmethod (space-war-window :help) (&optional (wait t))
  (when wait
    (w:bitblt w:normal w:width w:height w:screen-array 0 0 backup-array 0 0)
    (loop for object in objects
	  for i from 0 below num-objects
	  when (send object :alive-p) do
	  (send object :erase)))
  (send self :print-intro)
  (help-line #\clear-input "Reset game")
  (help-line #\end "End game")
  (help-line #\space "Pause game")
  (help-line #\f1 "Change game parameters")
  (help-line #\f2 "Edit command keys")
  (format self "~%")
  (loop for key in *keys*
	for key-name in *key-names* do
	(help-line key key-name))
  (when wait
    (send self :tyi)
    (w:bitblt w:normal w:width w:height backup-array 0 0 w:screen-array 0 0)))

(defflavor space-war-frame
	   ()
	   (w:select-mixin
	    w:alias-for-inferiors-mixin
	    w:inferiors-not-in-select-menu-mixin
	    w:bordered-constraint-frame-with-shared-io-buffer)
  (:default-init-plist
    :panes
    '((stats-pane space-war-stats)
      (play-pane space-war-window))
    :constraints
    '((main . ((stats-pane play-pane)
	       ((stats-pane 40))
	       ((play-pane :even)))))))

(defmethod (space-war-frame :after :init) (&rest ignore)
  (send self :set-selection-substitute (send self :get-pane 'play-pane))
  (send self :set-configuration 'main)
  (send (send self :get-pane 'play-pane)
	:set-stats-window
	(send self :get-pane 'stats-pane)))

(when (assoc #\w w:*system-keys*)
  (w:remove-system-key #\w))
(w:add-system-key #\w 'space-war-frame "2Space war*" t)

(compile-flavor-methods space-war-frame space-war-stats space-war-window
			floater shooter healer warper crystal
			bullet player moving-object)

