;;; -*- Mode:Common-Lisp; Package:SYSTEM; Base:10 -*-

;;; 22 Nov 87  Jamie Zawinski   Created.


;;; This file redefines QUERY-ABOUT-REDEFINITION so that it will timeout to a default answer after N seconds.
;;; Variables of importance: *REDEFINITION-QUERY-TIMEOUT* and *REDEFINITION-DEFAULT-ANSWER*.


(defparameter *redefinition-query-timeout* 60
  " The number of seconds that function redefinition queries will wait before defaulting to *REDEFINITION-DEFAULT-ANSWER*.
 Note that *REDEFINITION-DEFAULT-ANSWER* must have a valid value for any defaulting to happen.")


(defparameter *redefinition-default-answer* :no-default
  " This is the default action that will be taken if the user doesn't respond to a function redefinition query in
 *REDEFINITION-QUERY-TIMEOUT* seconds.  It must be one of the following values:
     :REDEFINE-ONCE        meaning redefine this time only
     :INHIBIT-DEFINITION   meaning do not redefine
     :PROCEED              meaning redefine and never query about these two files again
     <anything else>       do not default to any answer; sit there and wait for the user.")



(defun query-about-redefinition (function-spec new-pathname type old-pathname)
  ;; Detect any cross-file redefinition worth complaining about.
  (cond ((or (eq (if (stringp old-pathname) old-pathname
		     (and old-pathname (funcall old-pathname :translated-pathname)))
		 (if (stringp new-pathname) new-pathname
		     (and new-pathname (funcall new-pathname :translated-pathname))))
	     (member old-pathname
		     (if new-pathname
			 (funcall new-pathname :get :redefines-files)
			 non-pathname-redefined-files)
		     :test #'eq))
	 t)
	(t
	 ;; This redefinition deserves a warning or query.
	 ;; If it is within a file operation with warnings,
	 ;; record a warning.
	 (when (and (variable-boundp file-warnings-datum) file-warnings-datum)
	   (record-warning 'redefinition :probable-error nil
			   (if new-pathname
			       "~A ~S being redefined by file ~A.
 It was previously defined by file ~A."
			       "~A ~S being redefined;~* it was previously defined by file ~A.")
			   (or (get type 'definition-type-name) type) function-spec
			   new-pathname old-pathname))
	 (let (condition choice)
	   (setq condition
		 (make-condition 'sys:redefinition
				 (if new-pathname
				     "~A ~S being redefined by file ~A.
It was previously defined by file ~A."
				     "~A ~S being redefined;~* it was previously defined by file ~A.")
				 (or (get type 'definition-type-name) type)
				 function-spec
				 new-pathname old-pathname))
	   (setq choice (signal condition))
	   (unless choice
	     (unless (and inhibit-fdefine-warnings
			  (neq inhibit-fdefine-warnings :just-warn))
	       (format *query-io* "~&~A" condition))
	     (if inhibit-fdefine-warnings
		 (setq choice t)
		 (if (and (numberp *redefinition-query-timeout*)
			  (member *redefinition-default-answer*
				  '(:redefine-once :inhibit-definition :proceed)))
		     (with-timeout ((* 60 *redefinition-query-timeout*)
				    (progn (setq choice *redefinition-default-answer*)
					   (format t "timeout to ~A~%" choice)))
		       (format *query-io* "~&(Automatic ~A after ~D second~:P) "
			       *redefinition-default-answer* *redefinition-query-timeout*)
		       (setq choice (query-user-about-redefinition)))
		     (setq choice (query-user-about-redefinition)))
		 ))
	   (case choice
	     ((t :no-action :redefine-once) t)
	     ((nil :inhibit-definition) nil)
	     (error (error condition) t)
	     ((proceed :proceed)
	      (if new-pathname
		  (push old-pathname (get new-pathname :redefines-files))
		  (push old-pathname non-pathname-redefined-files))
	      t))))))


;;; This function was broken out of QUERY-ABOUT-REDEFINITION so's not to duplicate code...
(defun query-user-about-redefinition ()
  (fquery `(:choices ,query-about-redefinition-choices
		     :help-function (,(if (fboundp 'common-lisp-on-p)
					  (if (common-lisp-on-p)
					      'lambda
					      'global:lambda)  
					  'lambda) 
				     (stream &rest ignore)
				     (princ "

  Type Y - Yes, allow the redefinition of this one function,
       N - No, don't allow this function to be redefined,
       P - Proceed and not ask in the future (for this pair of files),
    or E - Enter the error handler:
 "
								 stream))
					  :clear-input t
					  :fresh-line nil
					  :select t
					  :beep t)
			       " OK? "))
