;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:PRINTER -*-

;;; File "PRINTER-REQUESTS-PATCH"
;;; Various patches to the TI print server code.
;;; Written and maintained by Jamie Zawinski.
;;;
;;; ChangeLog:
;;;
;;;  9 Nov 88  Jamie Zawinski    Created.
;;;  2 Dec 88  Jamie Zawinski    Fixed a bogosity in INSERT-[FILE,ARRAY]-IN-QUEUE, gee I'm silly.
;;;


;;; This file makes the following changes:
;;;
;;; o  When a file is queued that is not destined for a printer attached to the local machine,
;;;    it is shipped to its destination machine immediately.
;;;
;;;    It used to be that PRINT-FILE did this:
;;;      Make a print request and put it in the queue.
;;;      Wake up the print daemon.
;;;      The print daemon would then process the queue one element at a time.
;;;      If an element was bound for the local printer, it would process it.
;;;      If it was bound for a remote printer, it would ship it.
;;;
;;;    This caused the following scenario to be possible:
;;;      You are using machine A, which has printer 1 attached to it.
;;;      There are several large jobs queued to be printed on printer 1.
;;;      You want to print a file, so you tell PRINT-FILE to send it to printer 2 on machine B.
;;;      Your print request goes into the queue on the local machine (machine 1), and will not be
;;;      processed until everything ahead of it in the queue is processed.  Even if printer 2
;;;      is free, your request will not be shipped to it until printer 1 is free as well.
;;;
;;;    With the changes in this file, print requests to foreign printers do not go into the
;;;    print-queue on the local machine at all, but rather are shipped to the remote server 
;;;    immediately.
;;;
;;;
;;; o  The :FASD-FORM methods of FILE-PRINT-REQUEST and ARRAY-PRINT-REQUEST were incorrect.
;;;    They were returning lists of the form ( ... :slot-name ,slot-value ... ) when the contents
;;;    of some slots were not self-evaluating, requiring instead a form like
;;;    ( ... :slot-name ',slot-value ... ).  These methods are redefined to add a quote before
;;;    each comma.
;;;
;;;
;;; o  Also, the :FASD-FORM methods have been redefined to use a common macro which looks at
;;;    the SYS:FLAVOR object at macroexpand-time to determine what instance variables exist,
;;;    and composes an appropriate call to MAKE-INSTANCE.
;;;    This means that if you add a slot to one of these flavors, you don't need to also edit
;;;    the FASD methods, just recompile them.
;;;
;;;
;;; o  The base request flavor, BASIC-PRINT-REQUEST, has been modified to include 
;;;    the PROPERTY-LIST-MIXIN.  When additional keyword pairs are passed to PRINT-FILE, they
;;;    will be placed on the PLIST slot of the request.  This can be used to communicate with
;;;    exotic printer drivers.
;;;
;;;
;;; o  The base printer flavor, BASIC-PRINTER, has also been modified to include a plist.
;;;    The :MAKE-PRINTER-DEVICE methods of FILE-PRINT-REQUEST and ARRAY-PRINT-REQUEST have 
;;;    been changed so that the printer created has the same plist as the print request
;;;    from which it is being created; so the printer-specific methods can look at the plist
;;;    passed in, and modify their behavior accordingly.  A printer drive could, for example,
;;;    determine from the plist that it should print in two column landscape mode, something
;;;    which is not provided for in the general print mechanism, but which specific drivers
;;;    may know how to do.
;;;
;;;
;;; o  The :LISPM-REMOTE-PRINT-FILE-REQUEST method has been redefined to send more information
;;;    to the remote system - such as what is on the above-mentioned property-list.
;;;    It has been changed to use the :FASD-FORM method to build the list to transmit, again
;;;    making adding a slot easier.  It has a list of uninteresting slots which it does not
;;;    send, like the request number, and the print-status.  Therefore the default is for new
;;;    slots to be sent, not to be dropped, as used to be the case.


;;; Redefinition of the base request flavor, with a PList added.

(DEFFLAVOR BASIC-PRINT-REQUEST
	   ((PRINT-DEVICE NIL)
	    (USER-NAME NIL)
	    (SENDER-HOST NIL)
	    (PRINT-STATUS NIL)			; T if currently printing
	    (REQUEST-NO NIL)
	    (COPIES 1)
	    (HEADER T)				; allow a header page
	    (FILE-OF-SELF NIL)
	    )
	   (sys:property-list-mixin)  ; <-- JWZ
  :SETTABLE-INSTANCE-VARIABLES)



;;; Queuing remote prints at print-time instead of putting them in a local queue.

(defun request-foreign-host-p (print-request)
  "T if the print request is destined for a printer not attached to the local machine."
  (let* ((print-device (send print-request :print-device))
	 (printer-host (parse-host-name (get print-device :host) t t)))
    (or (null printer-host)
	(not (eq si:local-host printer-host)))))


(defun insert-request-in-queue (request)
  "If the print request is being sent to a printer on the local machine, insert it in the end of the queue.
  If it is being sent to a foreign printer, ship it there immediately."
  (cond ((request-foreign-host-p request)
	 (let ((printer-host (parse-host-name (get (send request :print-device) :host) t t)))
	   (send request :handle-remote-request printer-host))
	 (values request nil))
	(t
	 (start-print-daemon)
	 (let* (l)
	   (without-interrupts
	     (setq *print-queue* (nconc *print-queue* (cons request nil)))
	     (setq l (length *print-queue*)))
	   (send request :save-self-in-file)
	   (values request l)))))




;;; For building the PList of extra parameters passed to PRINT-FILE.
;;;
(defmacro bind-vars-and-plist ((plist-var-name plist) var-names &body body)
  "This is sort of like a destructuring-bind.
  PLIST should evaluate to a property list.
  VAR-NAMES is a list of LET bindings.
  The PLIST is parsed like &KEY arguments - if :FOO was a key in the plist, and
  there was a variable in VAR-NAMES called FOO, then that variable would be set to
  the corresponding value from the plist.
  The special thing here is that if there are keys in the plist that don't have
  corresponding variables, then the keys go into the variable PLIST-VAR-NAME.
  This will hold a plist of all unhandled keywords passed in.

  This is like saying (&REST ,plist-var-name &KEY ,@var-names &ALLOW-OTHER-KEYS)
  except that only the allow-other keys go into the rest arg."
  
  (let* ((rest (gensym))
	 (key (gensym))
	 (val (gensym)))
    `(let (,plist-var-name
	   ,@var-names)
       (do* ((,rest ,plist (cddr ,rest)))
	    ((null ,rest))
	 (let* ((,key (car ,rest))
		(,val (cadr ,rest)))
	   (case ,key
	     ,@(mapcar #'(lambda (var-name)
			   (when (consp var-name) (setq var-name (car var-name)))
			   `(,(intern (string var-name) "KEYWORD") (setq ,var-name ,val)))
		       var-names)
	     (t (setq ,plist-var-name (list* ,key ,val ,plist-var-name))))))
       ,@body)))


;;; Just a little more abstraction...
;;;
(defun fix-up-request-page-heading (page-heading header-name file-name)
  (cond ((null page-heading) nil)
	((consp page-heading) page-heading)
	((stringp page-heading)
	 (list (or header-name (and file-name (send file-name :short-string-for-printing)))
	       (string-subst-char #\Space #\Newline page-heading)))
	(t (list (or header-name (and file-name (send file-name :short-string-for-printing)))))))


;;; The next two functions are redefined to:
;;;    o  gather and store a plist;
;;;    o  call the abstraction INSERT-REQUEST-IN-QUEUE, which does the ship-immediately stuff.
;;;
(defun insert-file-in-queue (file-name real-printer-name &rest keywords)
  "Insert filename into printer queue and startup print daemon, if not already running."
  
  (declare (arglist &key copies lines font font-list "PRINTER" host delete-after cpi lpi header header-name
		    page-heading print-wide ps-font-map property-list &allow-other-keys))

  (bind-vars-and-plist (plist keywords)
		       ((copies 1)
			(lines *default-lines*)
			font font-list
			(user "PRINTER")
			(host (send si:local-host :name))
			sender-host
			user-name
			delete-after
			(cpi *default-cpi*) (lpi *default-lpi*)
			(header *default-header*) header-name
			(page-heading *default-page-heading*)
			(print-wide *default-print-wide*)
			(ps-font-map *default-ps-font-map*)
			property-list
			chaos-stream printer-name   ; bind these just so they don't go into the plist.
			)
    (setq printer-name real-printer-name)
    (macrolet ((fix (x n)
		 `(unless (and (numberp ,x) (plusp ,x)) (setq ,x ,n))))
      (fix copies 1)
      (fix lines 60)
      (fix lpi 6)
      (fix cpi 12))
    (unless (stringp printer-name) (setq printer-name (string printer-name)))
    (when sender-host (setq host sender-host))
    (when user-name (setq user user-name))
    
    (setq page-heading (fix-up-request-page-heading page-heading header-name file-name))
    
    (let* ((request-number (without-interrupts (incf *next-print-request-number*)))
	   (file-entry (make-instance 'FILE-PRINT-REQUEST
				      :print-device (get-printer-device printer-name)
				      :request-no request-number
				      :file-name file-name
				      :header-name header-name
				      :user-name user
				      :sender-host host
				      :copies copies
				      :lines lines
				      :cpi cpi
				      :lpi lpi
				      :header header
				      :page-heading page-heading
				      :font-list (or font font-list)
				      :delete-after delete-after
				      :print-wide print-wide
				      :ps-font-map ps-font-map
				      :property-list (append plist property-list)
				      )))
      (insert-request-in-queue file-entry))))



(defun insert-array-in-queue (screen-name real-printer-name screen-array width height start-x start-y &rest keywords)
  "Insert array entry into printer queue and startup print daemon, if not already running"

  (declare (arglist screen-name printer-name screen-array width height start-x start-y
		    &key dots-per-inch (orientation *default-orientation*) copies spool-file (delete-after t)
		    header page-heading user host property-list &allow-other-keys))

  (bind-vars-and-plist (plist keywords)
		       ((dots-per-inch nil)
			(orientation *default-orientation*)
			(copies 1)
			spool-file
			(delete-after t)
			(header *default-header*)
			(page-heading *default-page-heading*)
			(user "PRINTER")
			(host (send si:local-host :name))
			property-list
			chaos-stream printer-name   ; bind these just so they don't go into the plist.
			)
    (setq printer-name real-printer-name)
    (unless (stringp printer-name) (setf printer-name (string printer-name)))
    (setq page-heading (fix-up-request-page-heading page-heading screen-name nil))
    (start-print-daemon)
    (let* ((request-number (without-interrupts (incf *next-print-request-number*)))
	   (file-entry (make-instance 'ARRAY-PRINT-REQUEST
				      :print-device (get-printer-device printer-name)
				      :request-no request-number
				      :screen-name screen-name
				      :bitmap-array screen-array
				      :width width :height height :start-x start-x :start-y start-y
				      :orientation orientation :dots-per-inch dots-per-inch
				      :user-name user :sender-host host
				      :copies copies :header header :page-heading page-heading
				      :spool-file spool-file :delete-after delete-after
				      :property-list (append plist property-list)
				      )))
      (insert-request-in-queue file-entry))))


;;;
;;; These two :FASD-FORM methods were incorrect.  They were not quoting some slots that are not necessarily constants,
;;; so the forms returned were not EVALable.
;;;
;;; The following macro looks at the SYS:FLAVOR object at macroexpand-time to determine what instance variables exist,
;;; and composes an appropriate call to MAKE-INSTANCE.
;;;
;;; This means that if you add a slot to one of these flavors, you don't need to also edit the FASD methods, just 
;;; recompile them.
;;;

(defmacro instance-fasd-form (type)
  "Expands into a form which will return a list which, when evaluated, will make an instance of TYPE that has the
  same instance variable values that SELF has."
  (let* ((result '())
	 (list (sys:flavor-all-instance-variables (get type 'sys::flavor))))
    (dolist (symbol list)
      (push (intern (string symbol) "KEYWORD") result)
      (push `(list 'QUOTE ,symbol) result))
    (list* 'LIST ''MAKE-INSTANCE (list 'QUOTE (list 'QUOTE type))
	   (nreverse result))))


(defmethod (file-print-request :fasd-form) ()
  "Returns a form to recreate this file print request.  Used to copy itself in compiled form to a file."
  (instance-fasd-form FILE-PRINT-REQUEST))

(defmethod (array-print-request :fasd-form) ()
  "Returns a form to recreate this array print request.  Used to copy itself in compiled form to a file."
  (instance-fasd-form ARRAY-PRINT-REQUEST))



;;; Redefining :LISPM-REMOTE-PRINT-FILE-REQUEST to communicate more information!
;;; This code makes no mention of the plist slot, but rather is structured such that if it is
;;; there, it will get sent.

(defvar *useless-print-request-slots*
	'(:file-name :print-device :print-status :request-no :file-of-self)
  "These are the slots of PRINT-REQUEST structures that need not be sent to the remote printer.")


(defun strip-useless-slots (plist)
  "Destructively modify the plist to not have the *USELESS-PRINT-REQUEST-SLOTS* keyed in."
  (dolist (key *useless-print-request-slots*)
    (remf plist key))
  plist)


(DEFMETHOD (FILE-PRINT-REQUEST :LISPM-REMOTE-PRINT-FILE-REQUEST)
	   (PRINTER-HOST &AUX KEYWORD-RESPONSE CHAOS-STREAM)
  "Send to a print server on a remote Lisp Machine the print request's filename,
printer name, and printer options"

  (UNWIND-PROTECT (LET
		    ((*PRINT-PRETTY* NIL)	;set up default parameters before sending
		     (*PRINT-BASE* 10)		;request string to remote print server.
		     (*PRINT-RADIX* T) (*PRINT-LEVEL* NIL) (*PRINT-LENGTH* NIL))
		    (SETQ CHAOS-STREAM (CHAOS:OPEN-STREAM PRINTER-HOST "PRINT" :ERROR ()))
		    (COND
		      ((ERRORP CHAOS-STREAM)
		       (TV:NOTIFY () "Error opening ~A Print Server stream: ~A" PRINTER-HOST
				  CHAOS-STREAM)
		       (IF (AND DELETE-AFTER
				(EQUALP (SEND FILE-NAME :DIRECTORY) *PRINTER-DIRECTORY-NAME*))
			   (DELETE-FILE FILE-NAME)))
		      (T
			;; stream exists
			;; send keyword FOR queueing file
			(SEND CHAOS-STREAM :LINE-OUT "QUEUE-FILE")
			(SEND CHAOS-STREAM :FORCE-OUTPUT)
			;; listen for keyword validation.
			(SETQ KEYWORD-RESPONSE (SEND CHAOS-STREAM :LINE-IN t))
			(COND
			  ((Not (EQl #\+ (AREF KEYWORD-RESPONSE 0)))
			   (TV:NOTIFY () "Error in sending QUEUE-FILE keyword to ~A: ~A"
				      PRINTER-HOST KEYWORD-RESPONSE))
			  (T
			    (SEND CHAOS-STREAM :LINE-OUT
				  (SEND (SEND FILE-NAME :TRUENAME) :short-STRING-FOR-PRINTING))
			    (SEND CHAOS-STREAM :LINE-OUT (FORMAT () "~A" (CAR PRINT-DEVICE)))

			    ;; Change!  JWZ
			    (let* ((plist (cddr (send self :fasd-form))))
			      (setq plist (strip-useless-slots plist))
			      (setq plist (mapcar #'eval plist)) ; to ditch the quoting.
			      (SEND CHAOS-STREAM :LINE-OUT (FORMAT () "~S" plist)))

			    (SEND CHAOS-STREAM :FORCE-OUTPUT)
			    (SEND SELF :CHECK-REMOTE-PRINT-SERVER-RESPONSE CHAOS-STREAM
				  PRINTER-HOST))))))
    (UNLESS (ERRORP CHAOS-STREAM)
	    (SEND CHAOS-STREAM :FINISH)
	    (SEND CHAOS-STREAM :close))))


;;; Adding a plist to BASIC-PRINTER.

(DEFFLAVOR BASIC-PRINTER
	   ((PRINTER-STREAM NIL)
	    (CURRENT-TIME NIL)
	    (ESC 27)
	    (BACKSPACE 8)
	    (HTAB 9)
	    (LF 10)
	    (VTAB 11)
	    (FF 12)
	    (CR 13)
	    (NUL 0)
	    (LINE-COUNT 0)
	    (LINES-PER-PAGE 60)
	    (LINE-SPACING 6)
	    (CHAR-PITCH 12)
	    (CHAR-COUNT 0)
	    (CHARS-PER-LINE 0)
	    (LFPAD 20)
	    (FFPAD 100)
	    (PAGE-COUNT 0)
	    (PAGE-HEADING NIL)
	    (PAGE-WAIT NIL)
	    (PRINT-WIDE NIL)
	    (PS-FONT-MAP NIL))
	   (sys:property-list-mixin)
  :SETTABLE-INSTANCE-VARIABLES)


;;; When making an instance of some subclass of BASIC-PRINTER, give it the same PList as SELF (a PRINT-REQUEST).
;;;
(DEFMETHOD (FILE-PRINT-REQUEST :MAKE-PRINTER-DEVICE) (OUTPUT-STREAM)
  "Make an instance of the printer device using output-stream"
  (MAKE-PRINTER-DEVICE (GET PRINT-DEVICE :TYPE)
		       OUTPUT-STREAM :LINE-SPACING LPI :CHAR-PITCH CPI
		       :LINES-PER-PAGE LINES :PAGE-HEADING PAGE-HEADING :PRINT-WIDE PRINT-WIDE
		       :property-list sys:property-list))


(DEFMETHOD (ARRAY-PRINT-REQUEST :MAKE-PRINTER-DEVICE) (OUTPUT-STREAM)
  "Make an instance of the printer device using output-stream"
  (MAKE-PRINTER-DEVICE (GET PRINT-DEVICE :TYPE) OUTPUT-STREAM :PAGE-HEADING PAGE-HEADING
		       :property-list sys:property-list))
