;;;  -*- Mode:Common-Lisp; Package:ETHERNET; Base:10 -*-
;;;
;;;  Netwatch hacks.
;;;
;;;  Things to do:  filtering by type and hosts.
;;;		    tracking "conversations" -- sequences of packets between hosts.
;;; 
;;; Date: Sun, 18 Mar 90  13:50:28 CST
;;; From: Paul Fuqua <pf@islington-terrace.csc.ti.com>
;;; Subject: Network Monitoring Code (740+ lines)
;;; To:   jwz@teak.Berkeley.EDU, acuff@sumex-aim.stanford.edu
;;; Moon: Waning Gibbous (56% of full)
;;; 
;;;      Okay, here it is.  Feel free to put it with any other Explorer code
;;; you have available;  it's based on the Monitor Link code in the Ethernet
;;; system, and Austin won't be including it in any future product that I
;;; know about.
;;;      This stuff is nasty and ugly and uncommented and parts don't work.
;;; You've been warned (so don't critique it).  The main function is
;;; ETHERNET:NETWATCH, which takes keyword args for STREAM, PRINT-MODE
;;; (:brief, :verbose, and :detailed), PROMISCUOUS-MODE (T to listen to
;;; everybody's packets, NIL to listen only to your own), ONLY-TYPES (names
;;; of packet types to listen to, default unrestricted), and ONLY-ADDRS
;;; (which doesn't work).
;;;      When executed, it takes over the Ethernet board and prints packet
;;; info on the stream.  Normal network traffic will be blocked while it's
;;; running.  It understands Chaos, IP, and ARP packets.  I don't run
;;; Decnet, so it doesn't do them;  anyway, Decnet broadcast traffic is
;;; really multicast, so non-Decnet machines won't see it.
;;;      Some example calls:
;;; 
;;;    (ethernet:netwatch :promiscuous-mode nil)
;;;    (ethernet:netwatch :promiscuous-mode nil :print-mode :verbose)
;;;    (ethernet:netwatch :print-mode :detailed :only-types "DOD IP")
;;;    (ethernet:netwatch :only-types '("DOD IP" "ARP"))
;;; 
;;; The ONLY-TYPES arguments have to match items on the
;;; *official-protocol-alist* variable at the bottom of the file.
;;;      Enjoy it and good luck.
;;; 
;;;                               pf


(proclaim '(optimize speed))

(defmethod (nubus-enc :netwatch) (stream print-mode promiscuous-mode only-types only-addrs)
  "Print Ethernet frames as received on this link.  Disables normal reception on Ethernet while running."
  (unwind-protect
      (progn					; Gracefully halt receive process:
	(setf net:enabled nil)			; Put controller in disabled state
	(process-reset-and-enable
	  (send self :receiver-process))	; Throw into wait state by resetting process and giving it
	(process-allow-schedule)		; a chance to stop in a known place.
	(when promiscuous-mode
	  (send self :promiscuous-on))
	(loop
	  (multiple-value-bind (dest src type data nbytes)
	      (send self :receive)		; Call bottom level receive function.
	    (unwind-protect
		(progn				; Print out the packet
		  (let ((official-type (dpb (ldb (byte 8 0) type) (byte 8 8) (ldb (byte 8 8) type))))
		    (when (and (or (null only-types)
				   (and (eq (first only-types) :not)
					(not (member official-type (the list (cdr only-types)))))
				   (and (numberp (first only-types))
					(member official-type only-types)))
			       (or (null only-addrs)
				   (and (eq (first only-addrs) :not)
					(or (not (member src (the list (cdr only-addrs))))
					    (not (member dest (the list (cdr only-addrs))))))
				   (and (numberp (first only-addrs))
					(or (member src (the list only-addrs))
					    (member dest (the list only-addrs))))))
		      (case official-type
			(#x804			; Chaos
			 (print-chaos-pkt src dest stream data nbytes print-mode))
			(#x800			; IP
			 (print-ip-pkt src dest stream data nbytes print-mode))
			(#x806			; ARP
			 (print-arp-pkt src dest stream data nbytes print-mode))
			(otherwise
			 (print-random-pkt official-type src dest stream data nbytes print-mode))))))
	      (when data		   ; Return the packet
		(deallocate-net-packet type data))))))
    (when promiscuous-mode
      (setq enet-prm nil)
      (send self :initialize))
    (send self :enable)))

(defmethod (nubus-enc :promiscuous-on) ()
   (setq enet-prm t)
   (send self :configure))

(defun netwatch (&key (stream *standard-output*) (print-mode :brief) (promiscuous-mode t)
		      only-types only-addrs)
;   (net:all-hosts)				; Fetch the hosts first.
   (unless (listp only-types)
     (setq only-types (list only-types)))
   (unless (listp only-addrs)
     (setq only-addrs (list only-addrs)))
   (let ((type-codes (loop for type in only-types
			   when (numberp type)
			   collect type
			   else collect (or (car (rassoc type *official-protocol-alist* :test #'equalp))
					    (error "No such protocol name as ~A." type)))))
     (send (net:select-controller) :netwatch stream print-mode promiscuous-mode type-codes only-addrs)))

(defvar *resolve-hosts-local-only* t
   "When T, only look locally to resolve host addresses into hosts.
When NIL, look in the namespace (should not be used in netwatch).")

(defun host-from-ethernet-address (addr &optional (local-only *resolve-hosts-local-only*))
   (if (= addr *ethernet-broadcast-address*)
       "Ethernet Broadcast"
       (loop for (protocol list) in address-translations-lists
	     as translations = (symbol-value list)
	     as match = (first (find addr translations :key #'second))
	     when match
	     return (or (sys:get-host-from-address match protocol local-only)
			(case protocol
			  (:chaos (format nil "CHAOS|~O" match))
			  (:ip    (format nil "IP|~A" (dotted-format-ip-address match)))
			  (otherwise (format nil "~A|~A" protocol match))))
	     finally (return nil))))

(defparameter *bytes-per-line* 16.)

(defun print-random-pkt (official-type src dest stream data nbytes print-mode)
   (format stream "~&~A packet from ~A to ~A"
	   (or (cdr (assoc official-type *official-protocol-alist*))
	       (format nil "Type ~X" official-type))
	   (or (host-from-ethernet-address src)
	       (format nil "Unknown ~X." src))
	   (or (host-from-ethernet-address dest)
	       (format nil "Unknown ~X." dest)))
   (when (eq print-mode :detailed)
     (loop for line from 0 below nbytes by *bytes-per-line*
	   doing (format stream "~& ~3,'0X:" line)
	   doing (loop for index from line below (+ line (floor *bytes-per-line* 2))
		       as byte from line below nbytes by 2
		       as chunk = (aref data index)
		       doing (format stream " ~4,'0X" chunk))
	   doing (format stream "~VT  " (+ 5 (ceiling (* 5 *bytes-per-line*) 2) 5))
	   doing (loop for index from line below (+ line (floor *bytes-per-line* 2))
		       as byte from line below nbytes by 2
		       as chunk = (aref data index)
		       doing (format stream "~C~C" (ldb (byte 7 0) chunk) (ldb (byte 7 8) chunk))))
     (terpri stream)
     (terpri stream)))

(defun print-chaos-pkt (src dest stream pkt nbytes print-mode)
   (declare (ignore nbytes))
   (let* ((src-addr (chaos:pkt-source-address pkt))
	  (dest-addr (chaos:pkt-dest-address pkt))
	  (opcode (chaos:pkt-opcode pkt))
	  (op (or (nth opcode chaos:opcode-list)
		  (and (>= opcode chaos:dat-op)
		       'dat))))
     (if (eq print-mode :detailed)
	 (format stream "~&Chaosnet ~A (~4,'0X) packet "
		 (or op "<unknown opcode>")
		 opcode)
	 (format stream "~&Chaosnet ~A packet "
		 (or op (format nil "<unknown op ~X>" opcode))))
     (if (eq print-mode :detailed)
	 (format stream "from ~A (Chaos ~O, Ethernet ~X) "
		 (if (zerop src-addr)
		     "Chaos Broadcast"
		     (or (get-host-from-address src-addr :chaos *resolve-hosts-local-only*)
			 (host-from-ethernet-address src)
			 "<Unknown>"))
		 src-addr
		 src)
	 (format stream "from ~A "
		 (if (zerop src-addr)
		     "Chaos Broadcast"
		     (or (get-host-from-address src-addr :chaos *resolve-hosts-local-only*)
			 (host-from-ethernet-address src)
			 (format nil "<Unknown> (Chaos ~O)" src-addr)))))
     (if (eq print-mode :detailed)
	 (format stream "to ~A (Chaos ~O, Ethernet ~X) "
		 (if (zerop dest-addr)
		     "Chaos Broadcast"
		     (or (get-host-from-address dest-addr :chaos *resolve-hosts-local-only*)
			 (host-from-ethernet-address dest)
			 "<Unknown>"))
		 dest-addr
		 dest)
	 (format stream "to ~A "
		 (if (zerop dest-addr)
		     "Chaos Broadcast"
		     (or (get-host-from-address dest-addr :chaos *resolve-hosts-local-only*)
			 (host-from-ethernet-address dest)
			 (format nil "<Unknown> (Chaos ~O)" dest-addr)))))
     (unless (eq print-mode :brief)
       (when (eq print-mode :detailed)
	 (format stream "~& Number ~4,'0X, from-index ~D, to-index ~D, ~D bytes."
		 (chaos:pkt-num pkt)
		 (chaos:pkt-source-index-num pkt)
		 (chaos:pkt-dest-index-num pkt)
		 (chaos:pkt-nbytes pkt))
	 (format stream "~& Ack number = ~D, forwarded ~D time~:P, retransmitted ~D time~:P~:[ (last at ~\\time\\)~;~]."
		 (chaos:pkt-ack-num pkt)
		 (chaos:pkt-fwd-count pkt)
		 (chaos:pkt-times-transmitted pkt)
		 (zerop (chaos:pkt-times-transmitted pkt))
		 (chaos:pkt-time-transmitted pkt)))
       (unless (or (= opcode chaos:opn-op)
		   (= opcode chaos:fwd-op)
		   (= opcode chaos:sns-op)
		   (= opcode chaos:sts-op)
		   (= opcode chaos:mnt-op)
		   (= opcode chaos:unc-op)
		   (= opcode chaos:rut-op))
	 (let ((pkt-string (make-array chaos:max-data-bytes-per-pkt
				       :element-type 'string-char
				       :fill-pointer (chaos:pkt-nbytes pkt)
				       :displaced-to pkt
				       :displaced-index-offset 16)))
	   (compiler:dont-optimize
	     (format stream "~& Contents:  ~~A~" pkt-string)))))))

(defun print-ip-pkt (src dest stream pkt nbytes print-mode)
   (case (first (rassoc (ip:ip-protocol pkt) ip:*ip-protocol-mapping* :test #'equal))
;      (SEND handler :receive-data complete-pkt
;	    (SETF header-length (* 2 (ip-header-length complete-pkt)))	; start
;	    (- (ip-total-length complete-pkt) (* 2 header-length))	; length
;	    (ip-src-addr complete-pkt)					; src
;	    (ip-dst-addr complete-pkt))					; dst
     (:tcp
      (print-ip-tcp-pkt src dest stream pkt nbytes print-mode))
     (:udp
      (print-ip-udp-pkt src dest stream pkt nbytes print-mode))
     (:icmp
      (print-ip-icmp-pkt src dest stream pkt nbytes print-mode))))

(defun show-ip-options (pkt stream)
   (format stream "~& IP options: ~D bytes, version ~D"
	   (ip:ip-total-length pkt)
	   (ip:ip-version pkt))
   (format stream "~@[~*, don't fragment~]~@[~*, more fragments~]"
	   (ip:ip-dont-fragment-p pkt)
	   (ip:ip-more-fragments-p pkt))
   (format stream ", fragment offset ~4,'0X, time-to-live ~2,'0X, checksum ~4,'0X."
	   (ip:ip-fragment-offset pkt)
	   (ip:ip-time-to-live pkt)
	   (ip:ip-header-checksum pkt)))

(defun print-ip-icmp-pkt (src dest stream pkt nbytes print-mode)
   (declare (ignore nbytes))
   (let* ((start (* 2 (ip:ip-header-length pkt)))
	  (length (- (ip:ip-total-length pkt) (* 2 start)))
	  (ip-src-addr (ip:ip-src-addr pkt))
	  (src-host (or (get-host-from-address ip-src-addr :ip *resolve-hosts-local-only*)
			(host-from-ethernet-address src)))
	  (src-addr (dotted-format-ip-address ip-src-addr))
	  (ip-dst-addr (ip:ip-dst-addr pkt))
	  (dst-addr (dotted-format-ip-address ip-dst-addr))
	  (dst-host (or (and (ip:local-broadcast-address-p ip-dst-addr)
			     (format nil "IP Broadcast (~A)" dst-addr))
			(get-host-from-address ip-dst-addr :ip *resolve-hosts-local-only*)
			(host-from-ethernet-address dest)))
	  (packet nil))
     (unwind-protect
	 (progn
	   (setq packet (ip:allocate-icmp-packet))
	   (ip:copy-pkt pkt packet length start)
	   (format stream "~&IP ICMP ")
	   (case (ip:icmp-type packet)
	     (#.ip:address-mask-request
	      (format stream "address mask request"))
	     (#.ip:address-mask-reply
	      (format stream "address mask reply~:[~; (~X)~]"
		      (neq print-mode :brief)
		      (ip:icmp-address-mask packet)))
	     (#.ip:redirect
	      (format stream "~[network~;host~;tos-net~;tos-host~:;unknown~] redirect"
		      (ip:icmp-code packet))
	      (unless (eq print-mode :brief)
		(format nil " (addr ~X through gateway ~X)"
			(+ (ash (ip:icmp-option-1 packet) 16) (ip:icmp-option-2 packet))
			(dpb (aref packet 20) (byte 8 24)
			     (dpb (aref packet 21) (byte 8 16)
				  (dpb (aref packet 22) (byte 8 8)
				       (aref packet 23)))))))
	     (#.ip:destination-unreachable
	      (format stream "destination unreachable ~
			     (~[net~;host~;protocol~;port~;fragmentation-needed~;source-route-failed~:;unknown ~:*~D~])"
		      (ip:icmp-code packet)))
	     (#.ip:*icmp-echo*
	      (format stream "echo request"))
	     (#.ip:*icmp-echo-reply*
	      (format stream "echo reply"))
	     (#.ip:*icmp-timestamp*
	      (format stream "timestamp request"))
	     (#.ip:*icmp-timestamp-reply*
	      (format stream "timestamp reply"))
	     (#.ip:parameter-problem
	      (format stream "parameter problem"))
	     (#.ip:time-exceeded
	      (format stream "time exceeded (~[time-to-live~;fragment-reassembly~:;unknown ~:*~D~])"
		      (ip:icmp-code packet)))
	     (#.ip:source-quench
	      (format stream "source quench"))
	     (otherwise
	      (format stream "unimplemented message ~D" (ip:icmp-type packet))))
	   (if src-host
	       (format stream " from ~A" src-host)
	       (format stream " from <Unknown> (IP ~A~:[~*~;, Ethernet ~X~])"
		       src-addr
		       (eq print-mode :detailed)
		       src))
	   (if dst-host
	       (format stream " to ~A" dst-host)
	       (format stream " to <Unknown> (IP ~A~:[~*~;, Ethernet ~X~])"
		       dst-addr
		       (eq print-mode :detailed)
		       dest))
	   (when (eq print-mode :detailed)
	     (show-ip-options pkt stream)))
       (ip:free-icmp-packet packet))))

(defun print-ip-udp-pkt (src dest stream pkt nbytes print-mode)
   (declare (ignore nbytes))
   (let* ((start (* 2 (ip:ip-header-length pkt)))
	  (length (- (ip:ip-total-length pkt) (* 2 start)))
	  (ip-src-addr (ip:ip-src-addr pkt))
	  (src-host (or (get-host-from-address ip-src-addr :ip *resolve-hosts-local-only*)
			(host-from-ethernet-address src)))
	  (src-addr (dotted-format-ip-address ip-src-addr))
	  (ip-dst-addr (ip:ip-dst-addr pkt))
	  (dst-addr (dotted-format-ip-address ip-dst-addr))
	  (dst-host (or (and (ip:local-broadcast-address-p ip-dst-addr)
			     (format nil "IP Broadcast (~A)" dst-addr))
			(get-host-from-address ip-dst-addr :ip *resolve-hosts-local-only*)
			(host-from-ethernet-address dest)))
	  (packet nil))
     (unwind-protect
	 (progn
	   (setq packet (allocate-resource 'ip:udp-packet length
					   :source-address src
					   :destination-address dest))
	   (ip:copy-pkt pkt packet length start)
	   (let* ((src-port (ip:udp-header-source-port packet))
		  (src-port-name (or (cdr (assoc src-port (the list *official-port-alist*) :test #'eq))
				     (cdr (assoc src-port (the list *unix-udp-port-alist*) :test #'eq))))
		  (dst-port (ip:udp-header-destination-port packet))
		  (dst-port-name (or (cdr (assoc dst-port (the list *official-port-alist*) :test #'eq))
				     (cdr (assoc dst-port (the list *unix-udp-port-alist*) :test #'eq)))))
	     (format stream "~&IP UDP~@[ ~A~] packet"
		     (or dst-port-name src-port-name))
	     (if src-host
		 (format stream " from ~A, port ~D~@[ (~A)~]"
			 src-host
			 src-port
			 src-port-name)
		 (format stream " from <Unknown> (IP ~A~:[~*~;, Ethernet ~X~]), port ~D~@[ (~A)~]"
			 src-addr
			 (eq print-mode :detailed)
			 src
			 src-port
			 src-port-name))
	     (if dst-host
		 (format stream " to ~A, port ~D~@[ (~A)~]"
			 dst-host
			 dst-port
			 dst-port-name)
		 (format stream " to <Unknown> (IP ~A~:[~*~;, Ethernet ~X~]), port ~D~@[ (~A)~]"
			 dst-addr
			 (eq print-mode :detailed)
			 dest
			 dst-port
			 dst-port-name))
	     (unless (eq print-mode :brief)
;	       (format stream "~& [Would show data here, but don't know how just yet.]")
	       (show-raw-data packet ip:udp-header-size length stream)))
	   (when (eq print-mode :detailed)
	     (show-ip-options pkt stream)))
       (when packet
	 (deallocate-resource 'ip:udp-packet packet)))))

(defun show-raw-data (packet start length stream)
   (loop for line from start below length by *bytes-per-line*
	 doing (format stream "~& ~3,'0X:" line)
	 doing (loop for index from line below (+ line *bytes-per-line*)
		     as byte from line below length
		     as chunk = (aref packet index)
		     doing (format stream " ~2,'0X" chunk))
	 doing (format stream "~V@T  " (- (+ 5 (* 3 *bytes-per-line*) 5)
					  (+ 5 (* 3 (min (- length line) *bytes-per-line*)))))
	 doing (loop for index from line below (+ line *bytes-per-line*)
		     as byte from line below length
		     as chunk = (aref packet index)
		     doing (if (graphic-char-p chunk)
			       (write-char chunk stream)
			       (write-char #\Space stream)))))

(defun print-ip-tcp-pkt (src dest stream pkt nbytes print-mode)
   (declare (ignore nbytes))
   (let* ((start (* 2 (ip:ip-header-length pkt)))
	  (length (- (ip:ip-total-length pkt) (* 2 start)))
	  (ip-src-addr (ip:ip-src-addr pkt))
	  (ip-dst-addr (ip:ip-dst-addr pkt))
	  (src-host (or (get-host-from-address ip-src-addr :ip *resolve-hosts-local-only*)
			(host-from-ethernet-address src)))
	  (src-addr (dotted-format-ip-address ip-src-addr))
	  (dst-addr (dotted-format-ip-address ip-dst-addr))
	  (dst-host (or (and (ip:local-broadcast-address-p ip-dst-addr)
			     (format nil "IP Broadcast (~A)" dst-addr))
			(get-host-from-address ip-dst-addr :ip *resolve-hosts-local-only*)
			(host-from-ethernet-address dest)))
	  (segment nil))
     (unwind-protect
	 (progn
	   (setf segment (allocate-resource 'ip:tcp-segment length))
	   (ip:tcp-copy-packet-to-segment pkt segment length start)
	   (let* ((src-port (ip:tcp-header-source-port segment))
		  (src-port-name (or (cdr (assoc src-port (the list *official-port-alist*) :test #'eq))
				     (cdr (assoc src-port (the list *unix-tcp-port-alist*) :test #'eq))))
		  (dst-port (ip:tcp-header-destination-port segment))
		  (dst-port-name (or (cdr (assoc dst-port (the list *official-port-alist*) :test #'eq))
				     (cdr (assoc dst-port (the list *unix-tcp-port-alist*) :test #'eq)))))
	     (format stream "~&IP TCP~@[ ~A~] packet"
		     (or dst-port-name src-port-name))
	     (if src-host
		 (format stream " from ~A, port ~D~@[ (~A)~]"
			 src-host
			 src-port
			 src-port-name)
		 (format stream " from <Unknown> (IP ~A~:[~*~;, Ethernet ~X~]), port ~D~@[ (~A)~]"
			 src-addr
			 (eq print-mode :detailed)
			 src
			 src-port
			 src-port-name))
	     (if dst-host
		 (format stream " to ~A, port ~D~@[ (~A)~]"
			 dst-host
			 dst-port
			 dst-port-name)
		 (format stream " to <Unknown> (IP ~A~:[~*~;, Ethernet ~X~]), port ~D~@[ (~A)~]"
			 dst-addr
			 (eq print-mode :detailed)
			 dest
			 dst-port
			 dst-port-name)))
	   (when (eq print-mode :detailed)
	     (show-ip-options pkt stream))
	   (unless (eq print-mode :brief)
	     (when (eq print-mode :detailed)
	       (format stream "~& sequence ~D, ack ~D, window ~D"
		       (ip:tcp-header-sequence-# segment)
		       (ip:tcp-header-ack-# segment)
		       (ip:tcp-header-window segment))
	       (format stream "~& Flags: ~@[~* urgent~]~@[~* ack~]~@[~* push~]~@[~* reset~]~@[~* syn~]~@[~* fin~]"
		       (ip:tcp-header-urgent-p segment)
		       (ip:tcp-header-ack-p segment)
		       (ip:tcp-header-push-p segment)
		       (ip:tcp-header-reset-p segment)
		       (ip:tcp-header-syn-p segment)
		       (ip:tcp-header-fin-p segment))
	       (format stream "~& Urgent pointer ~4,'0X, data offset ~4,'0X, checksum ~4,'0X"
		       (ip:tcp-header-urgent-pointer segment)
		       (ip:tcp-header-data-offset segment)
		       (ip:tcp-header-checksum segment)))

;	     (format stream "~& [Would show data here, but don't know how just yet.]")
	     (show-raw-data segment (ip:tcp-header-data-offset segment) length stream)))
       (when segment
	 (deallocate-resource 'ip:tcp-segment segment)))))

(defun print-arp-pkt (src dest stream data nbytes print-mode)
   (declare (ignore dest nbytes))
   (format stream "~&ARP ~A"
	   (select (ar-opcode data)
	     (*ar-request* "request")
	     (*ar-reply*   "reply")
	     (otherwise    "unknown opcode")))
   (when (neq print-mode :brief)
     (format stream " (hardware type ~A)" (ar-hw-type data)))
   (let ((protocol (dpb (ldb (byte 8 0) (ar-protocol data)) (byte 8 8) (ldb (byte 8 8) (ar-protocol data)))))
     (case protocol
       (#x800					; IP
	(let* ((sender-ip (ar-ip-sender-ip data))
	       (recver-ip (ar-ip-recver-ip data))
	       (sender (or (get-host-from-address sender-ip :ip *resolve-hosts-local-only*)
			   (host-from-ethernet-address src)))
	       (for (get-host-from-address recver-ip :ip *resolve-hosts-local-only*)))
	  (if sender
	      (format stream " from ~A" sender)
	      (format stream " from <Unknown> (IP ~A~:[~;, Ether ~X~])"
		      (dotted-format-ip-address sender-ip)
		      (neq print-mode :brief)
		      (ar-ip-sender-ether data)))
	  (cond (for
		 (format stream " for ~A~:[ (IP ~A)~]"
			 for
			 (neq print-mode :brief)
			 (dotted-format-ip-address recver-ip)))
		((ip:local-broadcast-address-p recver-ip)
		 (format stream " for IP Broadcast (IP|~A)"
			 (dotted-format-ip-address recver-ip)))
		(:else
		 (format stream " for IP|~A"
			 (dotted-format-ip-address recver-ip))))))
       (#x804					; Chaos
	(let* ((sender-chaos (ar-chaos-sender-chaos data))
	       (recver-chaos (ar-chaos-recver-chaos data))
	       (sender (or (get-host-from-address sender-chaos :chaos *resolve-hosts-local-only*)
			   (host-from-ethernet-address src)))
	       (for (get-host-from-address recver-chaos :chaos *resolve-hosts-local-only*)))
	  (if sender
	      (format stream " from ~A" sender)
	      (format stream " from <Unknown> (Chaos ~O~:[~;, Ether ~X~])"
		      sender-chaos
		      (neq print-mode :brief)
		      (ar-chaos-sender-ether data)))
	  (cond (for
		 (format stream " for ~A~:[ (Chaos ~O)~]"
			 for
			 (neq print-mode :brief)
			 recver-chaos))
		((zerop recver-chaos)
		 (format stream " for Chaos Broadcast (Chaos|0)"))
		(:else
		 (format stream " for Chaos|~O" recver-chaos)))))
       (otherwise
	(format stream " for unhandled protocol ~A: "
		(or (cdr (assoc protocol *official-protocol-alist*))
		    (format nil "Type ~X" protocol)))
	(loop for i from 4 below (length data)
	      doing (format t " ~4,'0X" (aref data i)))))))


(defparameter *official-port-alist*		; From RFC 1010, Assigned Numbers.
  '(
    (5 . rje)					; Remote Job Entry                       [9,JBP]
    (7 . echo)					; Echo                                  [70,JBP]
    (9 . discard)				; Discard                               [69,JBP]
    (11 . users)				; Active Users                          [65,JBP]
    (13 . daytime)				; Daytime                               [68,JBP]
    (17 . quote)				; Quote of the Day                      [75,JBP]
    (19 . chargen)				; Character Generator                   [67,JBP]
    (20 . ftp-data)				; File Transfer [Default Data]          [71,JBP]
    (21 . ftp)					; File Transfer [Control]               [71,JBP]
    (23 . telnet)				; Telnet                                [87,JBP]
    (25 . smtp)					; Simple Mail Transfer                  [77,JBP]
    (27 . nsw-fe)				; NSW User System FE                    [17,RHT]
    (29 . msg-icp)				; MSG ICP                               [63,RHT]
    (31 . msg-auth)				; MSG Authentication                    [63,RHT]
    (33 . dsp)					; Display Support Protocol                 [MLC]
    (35 . "private printer server")		; [JBP]
    (37 . time)					; Time                                  [83,JBP]
    (39 . rlp)					; Resource Location Protocol                [MA]
    (41 . graphics)				; Graphics                              [98,JBP]
    (42 . nameserver)				; Host Name Server                      [74,JBP]
    (43 . nicname)				; Who Is                               [46,JAKE]
    (44 . mpm-flags)				; MPM FLAGS Protocol                       [JBP]
    (45 . mpm)					; Message Processing Module [recv]      [73,JBP]
    (46 . mpm-snd)				; MPM [default send]                    [73,JBP]
    (47 . ni-ftp)				; NI FTP                               [103,SK8]
    (49 . login)				; Login Host Protocol                     [PHD1]
    (51 . la-maint)				; IMP Logical Address Maintenance       [58,AGM]
    (53 . domain)				; Domain Name Server                 [61,70,PM1]
    (55 . isi-gl)				; ISI Graphics Language                  [6,RB9]
    (57 . "private terminal access")		; [JBP]
    (59 . "private file service")		; [JBP]
    (61 . ni-mail)				; NI MAIL                                [4,SK8]
    (63 . via-ftp)				; VIA Systems - FTP                        [DXD]
    (65 . tacacs-ds)				; TACACS-Database Service                [3,RHT]
    (67 . bootps)				; Bootstrap Protocol Server            [29,WJC2]
    (68 . bootpc)				; Bootstrap Protocol Client            [29,WJC2]
    (69 . tftp)					; Trivial File Transfer                [95,DDC1]
    (71 . netrjs-1)				; Remote Job Service                    [8,RTB3]
    (72 . netrjs-2)				; Remote Job Service                    [8,RTB3]
    (73 . netrjs-3)				; Remote Job Service                    [8,RTB3]
    (74 . netrjs-4)				; Remote Job Service                    [8,RTB3]
    (75 . "private dial out service")		; [JBP]
    (77 . "private RJE service")		; [JBP]
    (79 . finger)				; Finger                                [44,KLH]
    (81 . hosts2-ns)				; HOSTS2 Name Server                      [EAK1]
    (83 . mit-ml-dev)				; MIT ML Device                            [DPR]
    (85 . mit-ml-dev)				; MIT ML Device                            [DPR]
    (87 . "private terminal link")		; [JBP]
    (89 . su-mit-tg)				; SU/MIT Telnet Gateway                    [MRC]
    (91 . mit-dov)				; MIT Dover Spooler                        [EBM]
    (93 . dcp)					; Device Control Protocol                 [DT15]
    (95 . supdup)				; SUPDUP                                [20,MRC]
    (97 . swift-rvf)				; Swift Remote Vitural File Protocol       [MXR]
    (98 . tacnews)				; TAC News                                [FRAN]
    (99 . metagram)				; Metagram Relay                          [GEOF]
    (101 . hostname)				; NIC Host Name Server                 [45,JAKE]
    (102 . iso-tsap)				; ISO-TSAP                              [12,MTR]
    (103 . x400)				; X400                                    [HCF2]
    (104 . x400-snd)				; X400-SND                                [HCF2]
    (105 . csnet-ns)				; Mailbox Name Nameserver              [96,MAS3]
    (107 . rtelnet)				; Remote Telnet Service                 [76,JBP]
    (109 . pop-2)				; Post Office Protocol - Version 2     [11,JKR1]
    (111 . sunrpc)				; SUN Remote Procedure Call                [DXG]
    (113 . auth)				; Authentication Service               [99,MCSJ]
    (115 . sftp)				; Simple File Transfer Protocol        [56,MKL1]
    (117 . uucp-path)				; UUCP Path Service                     [35,MAE]
    (119 . nntp)				; Network News Transfer Protocol        [53,PL4]
    (121 . erpc)				; HYDRA Expedited Remote Procedure Call[101,JXO]
    (123 . ntp)					; Network Time Protocol                [60,DLM1]
;;  Local usurpation of a port.
    (125 . mdb)					; # Mark Meyer - RSS
;    (125 . locus-map)				; Locus PC-Interface Net Map Server    [105,BXG]
    (127 . locus-con)				; Locus PC-Interface Conn Server       [105,BXG]
    (129 . pwdgen)				; Password Generator Protocol          [107,FJW]
    (130 . cisco-fna)				; CISCO FNATIVE                            [WXB]
    (131 . cisco-tna)				; CISCO TNATIVE                            [WXB]
    (132 . cisco-sys)				; CISCO SYSMAINT                           [WXB]
    (133 . statsrv)				; Statistics Service                      [DLM1]
    (134 . ingres-net)				; INGRES-NET Service                       [MXB]
    (135 . loc-srv)				; Location Service                         [JXP]
    (136 . profile)				; PROFILE Naming System                    [LLP]
    (137 . netbios-ns)				; NETBIOS Name Service                     [JBP]
    (138 . netbios-dgm)				; NETBIOS Datagram Service                [JBP]
    (139 . netbios-ssn)				; NETBIOS Session Service                 [JBP]
    (140 . emfis-data)				; EMFIS Data Service                       [GB7]
    (141 . emfis-cntl)				; EMFIS Control Service                    [GB7]
    (142 . bl-idm)				; Britton-Lee IDM                         [SXS1]
    (243 . sur-meas)				; Survey Measurement                      [5,AV]
    (245 . link)				; LINK                                 [10,RDB2]
    ))

(defparameter *unix-udp-port-alist*		; Unix-specific ports, all above the official allocation.
	      '(
		(512 . biff)			; comsat
		(513 . who)			; rwho
		(514 . syslog)			; syslog daemon
		(518 . talk)
		(520 . route)			; routed
		(525 . timed)			; time server
		(533 . netwall)			; emergency broadcasts
		(2049 . nfs)			; NFS?
		(2201 . bfs)			; boot file server
		))

(defparameter *unix-tcp-port-alist*
	      '(
		(512 . exec)
		(513 . login)			; rlogin
		(514 . shell)			; rsh
		(515 . printer)			; printer spooler
		(520 . efs)			; Something for Lucasfilm.
		(526 . tempo)			; newdate
		(530 . courier)			; rpc
		(531 . conference)		; chat
		(532 . netnews)			; readnews
		(540 . uucp)			; uucpd		# uucp daemon
		(556 . remotefs)		; Brunhoff remote filesystem
		(1524 . ingreslock)
		))

(defparameter *official-protocol-alist*		; From RFC 1010, Assigned Numbers
	      '(

		(#x0200 . "Xerox PUP")		;[7,XEROX]
		(#x0201 . "PUP Addr. Trans.")	;[XEROX]
		(#x0600 . "Xerox NS IDP")	;[102,XEROX]
		(#x0800 . "DOD IP")		;[80,JBP]
		(#x0801 . "X.75 Internet")	;[XEROX]
		(#x0802 . "NBS Internet")	;[XEROX]
		(#x0803 . "ECMA Internet")	;[XEROX]
		(#x0804 . "Chaosnet")		;[XEROX]
		(#x0805 . "X.25 Level 3")	;[XEROX]
		(#x0806 . "ARP")		;[64,JBP]
		(#x0807 . "XNS Compatability")	;[XEROX]
		(#x081c . "Symbolics Private")	;[DCP1]
		(#x0888 . "Xyplex")		; mail
		(#x0900 . "Ungermann-Bass network debugger")	; mail
		(#x0a00 . "Xerox IEEE 802.3 PUP")	; mail
		(#x0a01 . "Xerox IEEE 802.3 PUP Addr. Trans.")	; mail
		(#x0bad . "Banyan Systems")	; mail
		(#x1000 . "Berkeley Trailer negotiation")	;[XEROX]
		(#x1001 . "Berkeley Trailer encapsulation")
		(#x1002 . "Berkeley Trailer encapsulation")
		(#x1003 . "Berkeley Trailer encapsulation")
		(#x1004 . "Berkeley Trailer encapsulation")
		(#x1005 . "Berkeley Trailer encapsulation")
		(#x1006 . "Berkeley Trailer encapsulation")
		(#x1007 . "Berkeley Trailer encapsulation")
		(#x1008 . "Berkeley Trailer encapsulation")
		(#x1009 . "Berkeley Trailer encapsulation")
		(#x100a . "Berkeley Trailer encapsulation")
		(#x100b . "Berkeley Trailer encapsulation")
		(#x100c . "Berkeley Trailer encapsulation")
		(#x100d . "Berkeley Trailer encapsulation")
		(#x100e . "Berkeley Trailer encapsulation")
		(#x100f . "Berkeley Trailer encapsulation")
		(#x1600 . "Valid")		;[XEROX]
		(#x5208 . "BBN Simnet")		;[XEROX]
		(#x6000 . "DEC unassigned")	; mail
		(#x6001 . "DEC MOP Dump/Load")	;[XEROX]
		(#x6002 . "DEC MOP Remote Console")	;[XEROX]
		(#x6003 . "DEC DECNET Phase IV")	;[XEROX]
		(#x6004 . "DEC LAT")		;[XEROX]
		(#x6005 . "DEC diagnostic")	;[XEROX]
		(#x6006 . "DEC customer")	;[XEROX]
		(#x6007 . "DEC LAVC")		; mail
		(#x6008 . "DEC unassigned")	; mail
		(#x6009 . "DEC unassigned")	; mail
		(#x7000 . "Ungermann-Bass download")	; mail
		(#x7002 . "Ungermann-Bass diagnostic")	; mail
		(#x8003 . "Cronus VLN")		;[100,DT15]
		(#x8004 . "Cronus Direct")	;[100,DT15]
		(#x8005 . "HP Probe")		;[XEROX]
		(#x8006 . "Nestar")		;[XEROX]
		(#x8010 . "Excelan")		;[XEROX]
		(#x8013 . "Silicon Graphics diagnostic")	; mail
		(#x8013 . "Silicon Graphics network games")	; mail
		(#x8013 . "Silicon Graphics reserved")	; mail
		(#x8013 . "Silicon Graphics XNS NameServer")	; mail
		(#x8019 . "Apollo DOMAIN")	; mail
		(#x8035 . "RARP")		;[40,JXM]
		(#x8038 . "DEC LANBridge")	;[XEROX]
		(#x8039 . "DEC unassigned")	; mail
		(#x803a . "DEC unassigned")	; mail
		(#x803b . "DEC unassigned")	; mail
		(#x803c . "DEC unassigned")	; mail
		(#x803d . "DEC Ethernet Encryption")	; mail
		(#x803e . "DEC unassigned")	; mail
		(#x803f . "DEC LAN Traffic Monitor")	; mail
		(#x8040 . "DEC unassigned")	; mail
		(#x8041 . "DEC unassigned")	; mail
		(#x8042 . "DEC unassigned")	; mail
		(#x805b . "Stanford V Kernel experimental")	;[XEROX]
		(#x805c . "Stanford V Kernel production")	;[XEROX]
		(#x807c . "Merit Internodal")	;[HWB]
		(#x8080	. "VitaLink TransLAN III")	;[Nancy@ftp.com]
		(#x809b . "EtherTalk")		;[XEROX]
		(#x80de . "TRFS")		; mail
		(#x80f3 . "AARP")		; mail
		(#x8107 . "Symbolics Private")	; mail
		(#x8108 . "Symbolics Private")	; mail
		(#x8109 . "Symbolics Private")	; mail
		(#x8137 . "Novell")		; mail
		(#x9000 . "Loopback")		;[Nancy@ftp.com]
		(#x9001 . "Bridge XNS")		;[Nancy@ftp.com]
		(#x9002 . "Bridge TCP/IP")	;[Nancy@ftp.com]
		(#xff00 . "BBN VITAL-LANBridge")	;[Nancy@ftp.com]
		))
