;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:(MANDELBROT); Vsp:0; Fonts:(CPTFONT HL12 TR12I COURIER CPTFONT HL12B HL12BI HL12I) -*-

;1;; File "3MANDELBROT*"*
;1;; A window-based utility for computing and examining Mandelbrot and Julia fractals.*
;1;; Load it, and select 5Mandelbrot* from the 5Programs* column of the System Menu.*
;1;;*
;1;; 5ChangeLog:**
;1;;*
;1;;  28 Nov 88*	1Jamie Zawinski*	1Created.*
;1;;  16 May 89*	1Jamie Zawinski*	1Added quad-tree descent, and a window interface.*
;1;;  19 May 89*	1Jamie Zawinski *	1Made the menu much wizzier.*
;1;;  20 May 89*	1Jamie Zawinski *	1Added the PostScript interface.  Added two new display modes, 5Contrast* and 5Contrast/No-Detail.**
;1;;*				1Also added ability to do a conventional every-pixel iteration.*
;1;;  22 May 89*	1Jamie Zawinski *	1Made the float-format used in computation be user-settable.  Added online documentation.*
;1;;  24 May 89*	1Jamie Zawinski *	1Made it possible to compute a non-square part of the set.*
;1;;*  12 Nov 89*	1Jamie Zawinski *	1Made the menu use smaller fonts if the normal fonts won't fit.*
;1;;*				1Conditionally compiled in the KSL window accelerators code if it is around.*
;1;;*  17 Nov 89*	1Jamie Zawinski *	1Added a menu option for shuffling the Mandelbrot process's priority.*
;1;;*				1Removed the call to 5adjust-array* since it didn't do what I expected, and took a long time as well.*
;1;;* 129 Apr 90*	1Jamie Zawinski *	1Added ability to do Julia fractals as well.  Added an icon.*
;1;;* 130 Apr 90*	1Jamie Zawinski *	1Made it possible to select coordinate pairs by clicking as well as by typeing them in.*
;1;;*  11 May 90*	1Jamie Zawinski *	1Use 5ticl:zunderflow* instead of 5condition-bind* to trap floating-point underflow conditions, which is*
;1;;*				1 many orders of magnitude faster.  Thanks to Paul Fuqua.*


;1;; 5Wish List:**
;1;;*
;1;;      Add a fixed-point mode that does the calculations using integers.  Might win, might not...*
;1;;      Short-floats are immediate objects, so they might be just as fast as fixnums.*
;1;;      I don't actually know that this works in color since they took my color monitor away.*
;1;;      I should include my run-length-encoded-PostScript code here so the images print faster.*
;1;;      Distributed processing.*
;1;;      Dithering to display multi-plane images in monochrome.*
;1;;      Save images to disk.*
;1;;      Online help needs more work.*
;1;;      A ``batch mode'' might be neat...  Tell it to go compute a set of images overnight.*
;1;;*


(eval-when (load eval compile)
  (defpackage "3MANDELBROT*"
    (:nicknames "3MAND*")
    (:export mandelbrot mandelbrot-screenhack mandelbrot-one-point
	     4*mandelbrot-frame-default-configuration**
	     4*mandelbrot-pan-ratio** 4*mandelbrot-scroll-ratio**)))


(defsubst 4mandelbrot-one-point* (x y w h range depth real-origin imaginary-origin drop-out-distance
				&optional seed-x seed-y)
  "2Computes the Mandelbrot mapping of a pixel XY.
  Returns a fixnum, the number of iterations to stability, or NIL if *DROP-OUT-DISTANCE2 is reached.

     X,Y:*		2The pixel we are computing (fixnums).
     W, H:*	2The pixel size of the target area (fixnums).
     RANGE:*	2The size of the source area (float).
     DEPTH:*	2Maximum number of iterations (fixnum).
     REAL, IMAG:*	2The origin of the source area (floats).
     DROP-OUT:*	2float.
     SEED-X,Y:*	2The image seed; if these are NIL, then the seed is the initial point.

  All floating-point computations take place in the precision of the numbers passed in.  So, if* 2you 
  want to use only short-float math,* 2then all floating point values supplied should be short-floats.

  X,Y are really the complex number (REAL+X) + (IMAG+Y)i.*"

  (declare (values . iterations-or-nil))
  (declare (optimize (speed 3) (safety 0))
	   (fixnum x y w h depth drop-out-distance)
	   (float range real-origin imaginary-origin))
  ;1;*
  ;1; z0 = x + yi*
  ;1; z1 = z0^2 + mu*
  ;1; For Julia, mu is a constant; for Mandelbrot, mu is z0.*
  ;1;*
  (let* ((s (max w h))

	 (real-z       (+ real-origin      (float (/ (* x range) s) range)))  ;5 real-origin + xr/s1 with same precision as *r1.**
	 (imaginary-z  (- imaginary-origin (float (/ (* y range) s) range)))  ;5 imag-origin + yr/s1 with same precision as *r1.**

	 (real-mu      (or seed-x real-z))
	 (imaginary-mu (or seed-y imaginary-z))
	 (value nil))
    (declare (float real-mu imaginary-mu real-z imaginary-z)
	     (fixnum s)
	     (type (or null fixnum) value))

    (dotimes (iteration depth)
      (declare (fixnum iteration))
      
      (if (>= (+ (* real-z real-z) (* imaginary-z imaginary-z))
	      drop-out-distance)
	  (return)
	  (setq value iteration))
      
      (let ((new-real-z (- (- (* real-z real-z) (* imaginary-z imaginary-z)) real-mu)))     ;1 (real-z^2 - imag-z^2) - real-mu*
	(declare (float new-real-z))
	(setq imaginary-z (- (* (* real-z 2) imaginary-z) imaginary-mu))     ;1 (2 * real-z * imag-z) - imag-mu*
	(setq real-z new-real-z)
	))
    value))


(defsubst 4iteration-to-color* (iteration top-color bottom-color contrast-distance outline-only color-p)
  "2  Given an iteration (fixnum or *NIL2) returns the color to which it should be mapped.  If *COLOR-P2 is* NIL2, this will be 1 or 0.
  If *CONTRAST-DISTANCE2 is non-*NIL2, and ITERATION is less than it, then the color will be zero.
  If both* CONTRAST-DISTANCE2 and* OUTLINE-ONLY2 are non-*NIL2, and* ITERATION2 is greater than *CONTRAST-DISTANCE2, then the color* 2will be 1.
  If *COLOR-P2 is non-*NIL2, then the resultant color will never fall out of the range from *BOTTOM-COLOR2 to *TOP-COLOR2.*"
  (declare (values . color-fixnum))
  (declare (fixnum top-color bottom-color)
	   (type (or null fixnum) iteration contrast-distance))
  (let* ((color (cond ((null iteration) 0)
		      (color-p (mod (+ bottom-color iteration) top-color))
		      (t (if (oddp iteration) 1 0)))))
    (when (and iteration contrast-distance)
      (setq color (if (< iteration contrast-distance)
		      0
		      (if outline-only color 1))))
    color))


(defsubst 4cache-mandelbrot-1* (cache-bitmap data-bitmap x y w h range depth real-origin imaginary-origin drop-out-distance
			      bottom-color top-color contrast-distance outline-only color-p &optional seed-x seed-y)
  "2Invokes *MANDELBROT-ONE-POINT2 if the desired value is not cached.  Updates the cache arrays.  Remeber - returns a 5COLOR*, not an 5ITERATION*!*"
  (declare (values . color-fixnum))
;  (declare (notinline iteration-to-color mandelbrot-one-point)) ;1 ##*
  (if t ;1 *(zerop (aref cache-bitmap y x))
      ;1;*
      ;1; Compute the value and save it.*
      ;1;*
      (let* ((val (iteration-to-color
		    (mandelbrot-one-point x y w h range depth real-origin imaginary-origin drop-out-distance seed-x seed-y)
		    bottom-color top-color contrast-distance outline-only color-p)))
	(declare (type fixnum val))
	(setf (aref cache-bitmap y x) 1)	;1 Say that this point has indeed been computed.*
	(setf (aref data-bitmap y x) val)	;1 Say what its value was.*
	val)
      ;1;*
      ;1; Return the saved value.*
      ;1;*
      (aref data-bitmap y x)))


(defsubst 4mandrect* (window x y w h color)
  "2Draw a rectangle of the appropriate color on the window.*"
  (unless color (setq color 0))
  (let ((alu (if (w:color-system-p window)
		 w:alu-transp
		 (if (zerop color) w:alu-setz w:alu-seta)))
	(ww (w:sheet-inside-width window))
	(wh (w:sheet-inside-height window))
	(x (+ x (w:sheet-left-margin-size window)))
	(y (+ y (w:sheet-top-margin-size window))))
    (tv:prepare-color (window color)   ;1 Why isn't this in the W package??*
      (w:sheet-force-access (window)
	(w:prepare-sheet (window)
	  (sys:%draw-rectangle (min (- ww x) w) (min (- wh y) h) x y alu window))))))


(defsubst 4mandpoint* (window x y color)
  "2Draw a point of the appropriate color on the window.*"
  (let* ((x (+ x (w:sheet-left-margin-size window)))
	 (y (+ y (w:sheet-top-margin-size window))))
    (when (and (< y (w:sheet-inside-height window))
	       (< x (w:sheet-inside-width window)))
      (setf (aref (w:sheet-screen-array window) y x) color))))

(defmacro 4prepare-sheet-if *((sheet condition) &body body)
  "2Like W:PREPARE-SHEET and W:SHEET-FORCE-ACCESS, but does body in a PROGN if CONDITION evaluates to NIL (at runtime).*"
  `(if ,condition
       (w:sheet-force-access (,sheet)
	 (w:prepare-sheet (,sheet)
	   ,@body))
       (progn ,@body)))

(defmacro 4maybe-prepare-sheet *((sheet) &body body)
  "2Like W:PREPARE-SHEET and W:SHEET-FORCE-ACCESS, but SHEET may be NIL.*"
  `(prepare-sheet-if (,sheet ,sheet) ,@body))


(defsubst 4iterative-compute-rectangle *(data-array cache-map window
				       x y w h initial-w initial-h depth real-origin imaginary-origin range
				       contrast-distance outline-only drop-out-distance bottom-color top-color color-p
				       &optional seed-x seed-y)
  "2Mandelbrot in the given rectangle.  WARNING - this function will freeze the window system until it is done.  Do NOT call it on large areas.*"
  (declare (optimize (speed 3) (safety 0))
	   (fixnum x y w h initial-w initial-h depth drop-out-distance bottom-color top-color)
	   (float real-origin imaginary-origin range))
;  (declare (notinline iteration-to-color cache-mandelbrot-1)) ;1 ##*
  (maybe-prepare-sheet (window)
    (dotimes (i h)
      (declare (fixnum i))
      (dotimes (j w)
	(declare (fixnum j))
	(let* ((xx (+ x j))
	       (yy (+ y i))
	       (color (cache-mandelbrot-1 cache-map data-array xx yy
					  initial-w initial-h range depth real-origin imaginary-origin drop-out-distance
					  top-color bottom-color contrast-distance outline-only color-p seed-x seed-y)))
	  (declare (fixnum xx yy color))
	  (when window (mandpoint window xx yy color))
	  )))))


(defsubst 4recursive-compute-rectangle *(data-array cache-map window
				       x y w h initial-w initial-h depth real-origin imaginary-origin range
				       contrast-distance outline-only drop-out-distance bottom-color top-color color-p
				       &optional seed-x seed-y)
  (declare (optimize (speed 3) (safety 0))
	   (fixnum x y w h initial-w initial-h depth drop-out-distance bottom-color top-color)
	   (float real-origin imaginary-origin range))
;  (declare (notinline iteration-to-color cache-mandelbrot-1)) ;1 ##*
  (let* ((first-time-p t)
	 (first 0)
	 (color 0))
    (declare (type fixnum first color)
	     (type (member t nil) first-time-p))
    (let* ((differ
	     (block DIFF
	       (macrolet ((check (xx yy)
			    `(progn
			       (setq color (cache-mandelbrot-1 cache-map data-array ,xx ,yy
							       initial-w initial-h range depth real-origin
							       imaginary-origin drop-out-distance
							       top-color bottom-color contrast-distance
							       outline-only color-p seed-x seed-y))
			       (when window (mandpoint window ,xx ,yy color))
			       (if first-time-p
				   (setq first color first-time-p nil)
				   (unless (= color first) (return-from DIFF t))))))
		 (let* ((x2 (+ x w))
			(y2 (+ y h)))
		   (declare (fixnum x2 y2))
		   (dotimes (i w)
		     (declare (fixnum i))
		     (check (+ x i) y)
		     (check (+ x i) y2))
		   (dotimes (i h)
		     (declare (fixnum i))
		     (check x  (+ y i))
		     (check x2 (+ y i)))
		   nil)))))
      
      (cond (differ
	     (let* ((half-w (round w 2))
		    (half-h (round h 2)))
	       (declare (fixnum half-w half-h))
	       (macrolet ((recurse (x y w h)
			    `(quad-tree-compute-1 data-array cache-map window ,x ,y ,w ,h
						  initial-w initial-h depth real-origin imaginary-origin
						  range contrast-distance outline-only drop-out-distance
						  bottom-color top-color color-p seed-x seed-y)))
		 (recurse x            y            half-w       half-h)
		 (recurse (+ x half-w) y            (- w half-w) half-h)
		 (recurse x            (+ y half-h) half-w       (- h half-h))
		 (recurse (+ x half-w) (+ y half-h) (- w half-w) (- h half-h))
		 )))
	    (t
	     (sys:%draw-rectangle w h 0 0 W:ALU-SETA cache-map)		;1 Fill this part of the cache-map with 1's.*
	     (when window (mandrect window x y w h color))		;1 Draw on the window if necessary.*
	     ;1;*
	     ;1; Now update the data-array.  If we are drawing 1's or 0's, then we can just slam them down with 5sys:%draw-rectangle.**
	     ;1; But we cannot use 5%draw-rectangle* to draw into an array with other than 1's and 0's.  You can use this function to draw*
	     ;1; on a 7sheet* in a different color with the 5prepare-color* form, but for arrays, there is no way to specify the destination color.*
	     ;1; So... in this case we must iterate size x size times and set elements of the data array to 5color*.*
	     ;1;*
	     (cond ((= 1 color) (sys:%draw-rectangle w h x y W:ALU-SETA data-array))
		   ((= 0 color) (sys:%draw-rectangle w h x y W:ALU-SETZ data-array))
		   (t (dotimes (i w)
			(dotimes (j h)
			  (setf (aref data-array (+ y j) (+ x i)) color)))))
	     )))))

(defvar 4*division-threshold* *5 
  "2 At some point there is a tradeoff between the win of maybe not having to calculate a quadrant, and the expense of recursion.
 This variable reresents that.*")
;1;;*
;1;;* 1I suspect the breakeven point is pretty low, based on some simple trials.  So, if the rectangle we are going to compute is very small,*
;1;;* 1where 7very small* is 4 or less pixels per side, then we simply iterate over all of the pixels rather than invoking ourself recursively on *
;1;; the tiny quadrants.*
  

(defun 4quad-tree-compute-1* (data-array cache-map window
			    x y w h initial-w initial-h depth real-origin imaginary-origin range contrast-distance
			    outline-only drop-out-distance bottom-color top-color color-p &optional seed-x seed-y)
  (declare (optimize (speed 3) (safety 0))
	   (fixnum x y w h initial-w initial-h depth drop-out-distance bottom-color top-color)
	   (float real-origin imaginary-origin range))
;  (declare (notinline iterative-compute-rectangle recursive-compute-rectangle)) ;1 ##*
  (when (>= (+ y h) initial-h)
    (setq h (- initial-h y 1)))
  (when (>= (+ x w) initial-w)
    (setq w (- initial-w x 1)))
  (if (or (< w (the fixnum *division-threshold*))
	  (< h (the fixnum *division-threshold*)))
      (iterative-compute-rectangle data-array cache-map window
				   x y w h initial-w initial-h depth real-origin imaginary-origin range
				   contrast-distance outline-only drop-out-distance bottom-color top-color color-p
				   seed-x seed-y)
      (recursive-compute-rectangle data-array cache-map window
				   x y w h initial-w initial-h depth real-origin imaginary-origin range
				   contrast-distance outline-only drop-out-distance bottom-color top-color color-p
				   seed-x seed-y)))



(defun 4quad-tree-compute-image* (&optional data cache
				&key (window w:selected-window)
				     (color-p nil cp-specified)
				     (width 128) (height 128) (depth 20)
				     (real-origin -2.0s0)
				     (imaginary-origin 2.0s0)
				     (range 4.0s0)
				     (contrast-distance nil) (outline-only nil)
				     (drop-out-distance 4.0s0)
				     (bottom-color 32)
				     (top-color 255)
				     julia-seed)
2   *"2Draw the Mandelbrot set, computing it with quad-tree descent.*"
  (if window
      (unless cp-specified (setq color-p (w:color-system-p window)))
      (assert cp-specified (color-p) "3If WINDOW is NIL, then COLOR-P must be specified.*"))
  (let* ((half-w (ceiling width 2))
	 (half-h (ceiling height 2)))
;    (condition-bind ((EH:FLOATING-POINT-UNDERFLOW #'(lambda (ignore) :USE-ZERO)))	;1 ## This is way too slow...*
    (let ((TICL:ZUNDERFLOW t))	       ;1 An A-Memory ucode switch that does the above.*
      (macrolet ((doit (x y)
		       `(quad-tree-compute-1 data cache window ,x ,y
					     half-w half-h width height depth real-origin imaginary-origin range
					     contrast-distance outline-only drop-out-distance bottom-color top-color color-p
					     (car julia-seed) (cdr julia-seed))))
	(doit 0      0)
	(doit 0      half-h)
	(doit half-w 0)
	(doit half-w half-h)
	))))


;1;;; Doing it the hard way.*


(defun 4iterative-compute-image* (&optional data cache
				&key (window w:selected-window)
				     (color-p nil cp-specified)
				     (width 128) (height 128) (depth 20)
				     (real-origin -2.0s0)
				     (imaginary-origin 2.0s0)
				     (range 4.0s0)
				     (contrast-distance nil) (outline-only nil)
				     (drop-out-distance 4.0s0)
				     (bottom-color 32)
				     (top-color 255)
				     julia-seed)
  "2Draw the Mandelbrot set, computing each and every pixel.*"
  (declare (optimize (speed 3) (safety 0))
	   (fixnum width height depth bottom-color top-color)
	   (float real-origin imaginary-origin range drop-out-distance)
;	   (notinline mandelbrot-one-point iteration-to-color) ;1 ##*
	   )

  (if window
      (unless cp-specified (setq color-p (w:color-system-p window)))
      (assert cp-specified (color-p) "3If WINDOW is NIL, then COLOR-P must be specified.*"))

  (unwind-protect
      (let* ((seed-x (car julia-seed))
	     (seed-y (cdr julia-seed))
	     (eltype (if color-p
			 (if window
			     (array-element-type (w:sheet-screen-array window))
			     'FIXNUM)
			 'BIT))
	     ;1; We write into a temporary array W x 1 x Depth and blit that onto the window instead of drawing a point at a time.*
	     ;1; This is much faster, since we only have to prepare the sheet once per scanline instead of once per point.*
	     (temp-array (make-array (list 1 (+ 32 (* 32 (floor (1- width) 32)))) :element-type eltype :initial-element 0)))
;	(condition-bind ((EH:FLOATING-POINT-UNDERFLOW #'(lambda (ignore) :USE-ZERO)))	;1 ## This is way too slow...*
	(let ((TICL:ZUNDERFLOW t))	       ;1 An A-Memory ucode switch that does the above.*
	  (dotimes (y height)
	    (declare (fixnum y))
	    (dotimes (x width)
	      (declare (fixnum x))
	      (let* ((color
		       (iteration-to-color
			 (mandelbrot-one-point x y width height range depth real-origin imaginary-origin drop-out-distance
					       seed-x seed-y)
			 bottom-color top-color contrast-distance outline-only color-p)))
		(declare (fixnum color))
		(setf (aref temp-array 0 x) color)))
	    (when window
	      (maybe-prepare-sheet (window)
		(send window :bitblt w:alu-seta width 1 temp-array 0 0 0 y)))
	    (bitblt w:alu-seta width 1 temp-array 0 0 data 0 y))))
    (fill cache 1)	 ;1 Mark it as computed.*
    ))



;1;;; Displayer Window *


(defstruct 4mandelbrot-descriptor*
  (width	128	:type fixnum)
  (height	128	:type fixnum)
  (depth	20	:type fixnum)
  (real-origin -2.0s0	:type float)
  (imag-origin	2.0s0	:type float)
  (range	4.0s0	:type float)
  (drop-out	4.0s0	:type float)
  (bpp          1	:type fixnum)
  (contrast	nil	:type (or null fixnum))
  (detail	t	:type (member t nil))
  (float-type 'short-float :type (member short-float single-float long-float double-float))
  (julia-seed	nil	:type (or null cons))  ;1 *NIL1 or *(x . y)
  )


;1;;; This is some conditional compilation noise - ignore it.*
(eval-when (eval compile)
  (when (get 'w::label-accelerators-for-superior 'sys:flavor)
    (pushnew :accelerators *features*))) ;1 removed later*


(defflavor 4basic-mandelbrot-displayer*
	   ((data-array	 nil)
	    (cache-array nil)
	    (iterative-p nil)	;1 Whether to use a straight iteration or a quad-tree approach.*
	    (displayed-descriptor (make-4mandelbrot-descriptor*))	;1 The parameters of the image currently being displayed.*
	    (working-descriptor   (make-4mandelbrot-descriptor*))	;1 The parameters which will be used next - the user edits these.*
	    
	    (bottom-color 32)
	    (top-color 255)	;1 These are only meaningful if *BPP1 is >1.*

	    (image-x-offset 0)	;1 For displaying images larger than the window.*
	    (image-y-offset 0)
	    )
	   (w:window)
  (:default-init-plist
    :save-bits t
    )
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


(defmethod 4(basic-mandelbrot-displayer :reset-needed-p)* ()
  "2Whether the user-edited parameters are such that we must recompute the image.*"
  (or (null displayed-descriptor)
      (null data-array)
      (not (equalp displayed-descriptor working-descriptor))))


(macrolet ((define-methods-for-working-descriptor (slot-name)
	     (let* ((getter  (intern (string slot-name) "3KEYWORD*"))
		    (setter  (intern (string-append "3SET-*" (string slot-name)) "3KEYWORD*"))
		    (rgetter (intern (string-append "3REAL-*" (string slot-name)) "3KEYWORD*"))
		    (accessor (intern (string-append "3MANDELBROT-DESCRIPTOR-*" slot-name)))
		    (doc1 (format nil "3Return the ~A slot of SELF's working-mandelbrot-descriptor.*" slot-name))
		    (doc2 (format nil "3Return the ~A slot of SELF's displayed-mandelbrot-descriptor.*" slot-name))
		    (doc3 (format nil "3Set the ~A slot of SELF's working-mandelbrot-descriptor.*" slot-name)))
	       `(progn
		  (defmethod (basic-mandelbrot-displayer ,getter) ()
		    ,doc1
		    (,accessor working-descriptor))
		  (defmethod (basic-mandelbrot-displayer ,rgetter) ()
		    ,doc2
		    (,accessor displayed-descriptor))
		  (defmethod (basic-mandelbrot-displayer ,setter) (,slot-name)
		    ,doc3
		    (setf (,accessor working-descriptor) ,slot-name))
		  ))))
  (define-methods-for-working-descriptor width)
  (define-methods-for-working-descriptor height)
  (define-methods-for-working-descriptor depth)
  (define-methods-for-working-descriptor real-origin)
  (define-methods-for-working-descriptor imag-origin)
  (define-methods-for-working-descriptor range)
  (define-methods-for-working-descriptor drop-out)
  (define-methods-for-working-descriptor bpp)
  (define-methods-for-working-descriptor contrast)
  (define-methods-for-working-descriptor detail)
  (define-methods-for-working-descriptor float-type)
  (define-methods-for-working-descriptor julia-seed)
  )



(defmethod 4(basic-mandelbrot-displayer :adjust-arrays*) ()
  "2Make the cache-arrays be the appropriate size for the current settings.*"

  (unless displayed-descriptor (setq displayed-descriptor (make-mandelbrot-descriptor)))
  (unless working-descriptor   (setq working-descriptor   (make-mandelbrot-descriptor)))
  
  (let* ((height (mandelbrot-descriptor-height working-descriptor))
	 (width (mandelbrot-descriptor-width working-descriptor))
	 (width32 (+ 32 (* 32 (floor (1- width) 32)))))   ;1 If we want to use BITBLT, second dimension must be mod 32.*
    
    (flet ((tweak-array (array bits-per-pixel initial-element)
	     "2Make the arrays be of the right shape and depth.*"
	     ;1; This function used to call ADJUST-ARRAY if the array was of the right depth but wrong size, but*
	     ;1; that took 2forever*, because a new array was being made, and the old array was being filled with (unnecessary)*
	     ;1; forwarding pointers.  So blow that off; any time the size is changed, we garbage the old array and make*
	     ;1; a new one.*
	     (cond ;1;*
	1       *     1 *;1; The array doesn't exist, or it is of the wrong element type, or is too small.*
	           ((or (null array)
			(not (typep array `(array (unsigned-byte ,bits-per-pixel) 2)))
			(> height  (array-dimension array 0))
			(> width32 (array-dimension array 1)))
		    (make-array (list height width32)
				:element-type `(unsigned-byte ,bits-per-pixel)
				:initial-element initial-element
				:adjustable t))
		   ;1;*
		   ;1; The array is too small.  Adjust it.*
;		   ((or (> height  (array-dimension array 0))
;			(> width32 (array-dimension array 1)))
;		    (adjust-array array (list height width32)))
		   
		   (t array))))
      
      (setq data-array  (tweak-array data-array (mandelbrot-descriptor-bpp working-descriptor) 0))
      (setq cache-array (tweak-array cache-array 1 0))
      ))
  (when (send self :reset-needed-p) (send self :reset))
  )


(defun 4bitmaps-have-same-depth-p *(bm1 bm2)
  "2T if the two arrays have the same number of bits per pixel.*"
  (and (subtypep (array-element-type bm1) (array-element-type bm2))
       (subtypep (array-element-type bm2) (array-element-type bm1))))


(defmethod 4(basic-mandelbrot-displayer :image-displayable-p)* ()
  (or (bitmaps-have-same-depth-p data-array w:screen-array)	;1 This bitmap is displayable.*
      (typep data-array '(array bit 2))))

(defmethod 4(basic-mandelbrot-displayer :after :refresh*) (&optional (type :complete-redisplay))
  (when (and data-array (eq type :complete-redisplay))
    (send self :clear-screen)
    (setq image-x-offset (round image-x-offset))
    (setq image-y-offset (round image-y-offset))
    (cond ((send self :image-displayable-p)
	   (let* ((x2 (+ (w:sheet-inside-width) image-x-offset))
		  (y2 (+ (w:sheet-inside-height) image-y-offset))
		  (w (max 0 (min (array-dimension data-array 1) (- x2 image-x-offset) (w:sheet-inside-width))))
		  (h (max 0 (min (array-dimension data-array 0) (- y2 image-y-offset) (w:sheet-inside-height)))))
	     (send self :bitblt W:ALU-SETA w h data-array image-x-offset image-y-offset 0 0)))
	  (t
	   (send self :set-current-font fonts:mets t)
	   (setq w:line-height (w:font-char-height fonts:mets))
	   (send self :string-out-x-y-centered-explicit
		 (format nil "3Cannot display this array.~%Array is ~A bits deep, but screen is of type ~A.~%~*
				3To see it, you must make a hardcopy.*"
			 (send self :real-bpp) (array-element-type w:screen-array)))))))


(defmethod 4(basic-mandelbrot-displayer :reset*) ()
  "2Clear the cached data.*"
  (and cache-array (fill cache-array 0))
  (and data-array (fill data-array 0))
  (setq displayed-descriptor (copy-mandelbrot-descriptor working-descriptor))
  )


(defmethod 4(basic-mandelbrot-displayer :display*) (&optional (draw-while-computing t) (color-p (w:color-system-p self)))
  (send self :adjust-arrays)
  (when (car w:blinker-list)
    (send (car w:blinker-list) :set-visibility nil))
  (let* ((function (if iterative-p 'ITERATIVE-COMPUTE-IMAGE 'QUAD-TREE-COMPUTE-IMAGE))
	 (float-type (mandelbrot-descriptor-float-type displayed-descriptor)))
    (funcall function
	     data-array cache-array
	     :window            (and draw-while-computing self)
	     :color-p           color-p
	     :width             (mandelbrot-descriptor-width  displayed-descriptor)
	     :height            (mandelbrot-descriptor-height displayed-descriptor)
	     :depth             (mandelbrot-descriptor-depth displayed-descriptor)
	     :range             (coerce (mandelbrot-descriptor-range displayed-descriptor) float-type)
	     :contrast-distance (mandelbrot-descriptor-contrast    displayed-descriptor)
	     :outline-only      (mandelbrot-descriptor-detail      displayed-descriptor)
	     :real-origin       (coerce (mandelbrot-descriptor-real-origin displayed-descriptor) float-type)
	     :imaginary-origin  (coerce (mandelbrot-descriptor-imag-origin displayed-descriptor) float-type)
	     :drop-out-distance (coerce (mandelbrot-descriptor-drop-out    displayed-descriptor) float-type)
	     :top-color         top-color
	     :bottom-color      bottom-color
	     :julia-seed	(let ((seed (mandelbrot-descriptor-julia-seed displayed-descriptor)))
				  (and seed
				       (cons (coerce (car seed) float-type)
					     (coerce (cdr seed) float-type))))
	     )))


;1;;; Frames*


(defvar 4*mandelbrot-pane-list** '((displayer mandelbrot-displayer-pane)
				 (typein    mandelbrot-typein-pane)
				 (menu      mandelbrot-menu-pane)))

(defvar 4*mandelbrot-frame-default-configuration* *'nonsquare-display
  "2Which configuration to use by default.  If you like your displayer window to always be square, set this to *'SQUARE-DISPLAY2.*")

(defvar 4*mandelbrot-constraints-list**
	'((square-display . ((whole)
			     ((whole :horizontal (:even)
				(left-side right-side)
				
				((left-side :vertical (0.7)
				   (displayer-block bottom-filler typein)
				   ((typein :limit (5 6 :lines) 0.2))
				   
				   ((displayer-block :horizontal (:ask-window displayer :displayer-inside-size)
				      (left-filler displayer right-filler)
				      
				      ((displayer :ask :displayer-inside-size))
				      
				      ((left-filler  :blank #.W:50%-GRAY :even)
				       (right-filler :blank #.W:50%-GRAY :even))))
				   
				   ((bottom-filler :blank #.W:50%-GRAY :even))
				   ))
				
				((right-side :vertical (:even)
				   (menu)
				   ((menu :even))))
				))))
	  (nonsquare-display . ((whole)
				((whole :horizontal (:even)
				   (left-side right-side)
				   
				   ((left-side :vertical (0.7)
				      (displayer typein)
				      ((typein :limit (5 6 :lines) 0.2))
				      ((displayer :even))))
				   
				   ((right-side :vertical (:even)
				      (menu)
				      ((menu :even))))
				   ))))
	  ))


(defvar 4*mandelbrot-command-table*)*


(defflavor 4mandelbrot-frame*
	   ((displayer nil)
	    (typein nil)
	    (menu nil))
	   (#+ACCELERATORS w:label-accelerator-mixin
	    ucl:command-loop-mixin
	    ucl:selective-features-mixin
	    w:inferiors-not-in-select-menu-mixin
	    w:list-mouse-buttons-mixin
	    w:bordered-constraint-frame-with-shared-io-buffer
	    w:window)
  (:default-init-plist
    :remove-features '(:LISP-TYPEIN :ALL-UNIVERSAL-COMMANDS)
    :active-command-tables '(*mandelbrot-command-table*)
    :all-command-tables    '(*mandelbrot-command-table*)
    :typein-handler nil
    :panes	   *mandelbrot-pane-list*
    :constraints   *mandelbrot-constraints-list*
    :configuration 4*mandelbrot-frame-default-configuration**
    :more-p nil :label nil
    :blinker-deselected-visibility nil
    :border-margin-width 2)
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


(defflavor 4mandelbrot-typein-pane*
	   ()
	   (ucl:command-and-lisp-typein-window)
  (:default-init-plist :label nil :more-p nil)
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


(defflavor 4mandelbrot-menu-pane* ()
	   (w:basic-mouse-sensitive-items
	    w:top-box-label-mixin
	    w:centered-label-mixin
	    w:window)
  (:default-init-plist
    :blinker-deselected-visibility nil
    :border-margin-width 2
    :label '(:string "3Mandelbrot*" :font fonts:mets)
    :more-p nil :save-bits t
    )
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


(defflavor 4mandelbrot-displayer-pane*
	   ((draw-while-computing t))
	   (#+ACCELERATORS w:label-accelerators-for-superior
	    w:basic-mouse-sensitive-items
	    basic-mandelbrot-displayer)
  (:default-init-plist :label nil :more-p nil :blinker-p nil)
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)


(defmethod 4(mandelbrot-frame :after :init*) (ignore)
  (setq displayer (send self :get-pane 'displayer))
  (setq typein    (send self :get-pane 'typein))
  (setq menu      (send self :get-pane 'menu))
  ;1;*
  ;1; Enable mouse-sensitive-text input.*
  (push '(:TYPEOUT-EXECUTE :HANDLE-TYPEOUT-EXECUTE) ucl:blip-alist)
  ;1;*
  ;1; Set the default size to be the size of the displayer window.*
  (send displayer :set-width (w:sheet-inside-width displayer))
  (send displayer :set-height (w:sheet-inside-height displayer))
  (w:sheet-force-access (menu) (send menu :refresh :complete-redisplay))
  )


(defmethod 4(mandelbrot-frame :print-notification)* (time string ignore)
  (beep 'W:NOTIFY)
  (format typein "3~&[~A ~A]~%*" (time:print-brief-universal-time time nil) string))


(defmethod 4(mandelbrot-displayer-pane :displayer-inside-size*) (rw rh tw th ignore)
  "2This is used by the constraint-list in an* :ASK-WINDOW2 clause - it keeps the* MANDELBROT-DISPLAYER-PANE2 square.*"
  (min rw rh tw th))

(defmethod 4(mandelbrot-frame :designate-io-streams*) ()
  (setq *terminal-io* (send self :get-pane 'typein)))

(defmethod 4(mandelbrot-frame :name-for-selection*) () (send self :name))

(defmethod 4(mandelbrot-frame :handle-typeout-execute*) ()
  (declare (special UCL:KBD-INPUT))
  (let* ((value (second UCL:KBD-INPUT)))
    (if (typep value 'UCL:COMMAND)
	(send value :execute self)
	(let* ((fspec `(:METHOD MANDELBROT-FRAME ,value))
	       (command (ucl:get-command fspec)))
	3  *(assert command () "3There is no ~S command.*" fspec)
	   (send command :execute self)))))


(defmethod 4(mandelbrot-frame :after :set-more-p*) (x)	;1 So we can toggle more mode in the typein window.*
  (send typein :set-more-p x))

(defmethod 4(mandelbrot-typein-pane :after :refresh)* (&optional (type :complete-redisplay))
  (when (eq type :complete-redisplay) (send self :clear-screen)))

(defwhopper 4(mandelbrot-typein-pane :mouse-buttons*) (bd x y)
  (if (send w:superior :operation-handled-p :dispatch-accelerated-operation)
      (send w:superior :dispatch-accelerated-operation x y (tv:mouse-character-button-encode bd))
      (continue-whopper bd x y)))

(defwhopper 4(mandelbrot-typein-pane :who-line-documentation-string*) ()
  (if (send w:superior :operation-handled-p :accelerator-who-line-doc)
      (send w:superior :accelerator-who-line-doc)
      (continue-whopper)))



(defmethod (4mandelbrot-frame* :user-activity-string) ()
  "2Returns a string for FINGER to show.*"
  (let* ((activity (send (send self :process) :whostate)))
    (if (string-equal "3Keyboard*" activity)
	"3Mandelbrot Viewer - idle*"
	(string-append "3Mandelbrot Viewer - *" activity))))


;1;; The Menu display*


(defun 4float-to-string *(float n-decimals)
  "2Convert a float to a string with at most N digits after the decimal point; trailing zeros and precision specs are suppressed.*"
  (let* ((string (string-right-trim "30*" (format nil "3~,vF*" n-decimals float)))
	 (dot (position #\. string :test #'char=)))
    (if (= (1+ dot) (length string))
	(string-append string "30*")
	string)))

(defun 4write-two-floats *(stream float1 float2 n-decimals)
  (format stream "3~A, ~A*" (float-to-string float1 n-decimals) (float-to-string float2 n-decimals)))


(defvar 4*mandelbrot-menu-descriptor**
	'(""
	  :help
	  :quit
	  ""
	  :print-image
	  :save-image
	  :show-saved-image
	  ""
	  :redisplay
	  :revert-settings
	  (:set :window-configuration)
	  (:set :process-priority)
	  ""
	  (:toggle :iterative-p)
	  (:toggle :draw-while-computing)
	  ""
	  ("3Seed:*" (let ((seed (send (send (send self :superior) :displayer) :julia-seed)))
		      (if seed
			  (write-two-floats nil (car seed) (cdr seed) 3)
			  "3dynamic*"))
		    :set-julia-seed)
	  ""
	  (:set :contrast)
	  (:toggle :detail)
	  ("3Precision:*" (format nil "3~:(~A~)*" (send (send (send self :superior) :displayer) :float-type))
			  :set-float-type)
	  ""
	  ("3Size:*" (format nil "3~D, ~D*" (send (send (send self :superior) :displayer) :width)
				         (send (send (send self :superior) :displayer) :height))
		    :set-size)
	  (:set :depth)
	  ("3Origin:*" (write-two-floats nil
		       (send (send (send self :superior) :displayer) :real-origin)
		       (send (send (send self :superior) :displayer) :imag-origin)
		       3)
		      :set-image-origin)
          (:set :range)
	  (:set :drop-out)
	  (:set :bpp)
	  ""
	  :pan-up
	  (:2col :pan-left :pan-right)
	  :pan-down
	  ""
	  :select-rectangle
	  :zoom-out
	  )
  "2The commands to put on the Mandelbrot menu.  Elements of this list are interpreted thusly:*

	2o  Strings are printed centered on the menu window.*

	2o  A Keyword means grind a menu-item representing the command named by that keyword on the *MANDELBROT-FRAME2.*

	2o  A list of the form 3(:set* <keyword> 3)* means grind a pair of strings.  The left string is the string-name of the command*
	2    whose def-name is 3:SET-*<keyword>, and the right string is the value returned by sending the <keyword> message to the*
	2    displayer pane.  Clicking on the right string issues the 3:SET-*<keyword> to the *MANDELBROT-FRAME2.*

	2o  A list of the form 3(:toggle* <keyword> 3)* means grind a pair of strings.  The left string is the string-name of the command*
	2    whose def-name is 3:TOGGLE-*<keyword>, and the right string is the value returned by sending the <keyword> message to the*
	2    displayer pane.  Clicking on the right string issues the 3:TOGGLE-*<keyword> to the *MANDELBROT-FRAME2.*

	2o  A list of the form 3(*<string> <form-to-eval> <keyword>3)* means grind a pair of strings.  The left string is the <string> given,*
	2    and the right string is the result of evaluating <form-to-eval>.  The <keyword> is the type of mouse-item that the right*
	2    string will be.  It is guarenteed that, during the evaluation of <form-to-eval>, *SELF2 will be the *MANDELBROT-MENU-PANE2.*

	2o  A list of the form 3(:2col* <keyword-1> <keyword-2> 3)* means grind two menu-items on the same line.  The menu items*
	2    invoke the commands named by the keywords on the *MANDELBROT-FRAME2.  This is a special-case hack so that we can*
	2    implement the Up, Left, Right, and Down commands.*")


(defmethod 4(mandelbrot-menu-pane :centered-mousable) (string type)*
  "2Write the string centered on SELF at the current Y position.  The string will be a mousable item of type TYPE.*"
  (let* ((size (w:sheet-string-length self string 0 nil nil (aref w:font-map 0))) ;1 *fonts:tr12
	 (x (round (- (w:sheet-inside-width) size) 2)))
    (setq w:cursor-x x)
    (send self :set-current-font 0)  ;1 *fonts:tr12
    (format self "3~VM~%*" type string)))

(defmethod 4(mandelbrot-menu-pane :*balanced-mousable4)* (left right type)
  "2Write two strings on SELF at the current Y position.  LEFT will be printed in italic, flush-right against the center line of SELF.
  RIGHT will be printed flush-left against the center line of SELF, and will be a mousable item of type TYPE.*"
  (unless (stringp right) (setq right (princ-to-string right)))
  (let* ((lsize (w:sheet-string-length self left 0 nil nil (aref w:font-map 1))) ;1 *fonts:tr12i
	 (lx (- (round (w:sheet-inside-width) 2) lsize 5))
	 (rx (+ (round (w:sheet-inside-width) 2) 5)))
    (send self :set-current-font 1) ;1 *fonts:tr12i
    (setq w:cursor-x lx)
    (princ left self)
    (setq w:cursor-x rx)
    (send self :set-current-font 0) ;1 *fonts:tr12
    (format self "3~VM~%*" type right)))

(defmethod 4(mandelbrot-menu-pane :keyword-to-command)* (keyword) (ucl:get-command `(:method mandelbrot-frame ,keyword)))
(defmethod 4(mandelbrot-menu-pane :command-to-keyword)* (command) (third (send command :defname)))

(defmethod 4(mandelbrot-menu-pane :assure-command-present)* (keyword)
  "2Make sure there is a mousable-item type defined on SELF for the command named by KEYWORD.*"
  (let* ((command (send self :keyword-to-command keyword)))
    (unless (member command w:item-type-alist :test #'eq :key #'second)
      (push (list keyword command (or (send command :documentation) "")) w:item-type-alist))))

(defmethod 4(mandelbrot-menu-pane :*command-mousable4)* (keyword)
  "2Just like the :CENTERED-MOUSABLE method, but extracts the appropriate name and documentation out of the command named by KEYWORD.*"
  (let* ((command (send self :keyword-to-command keyword)))
    (send self :assure-command-present keyword)
    (send self :centered-mousable (send command :name) keyword)))

(defmethod 4(mandelbrot-menu-pane :*command-balanced-mousable4)* (keyword value-string)
  "2Like :COMMAND-MOUSABLE, but the name of the command and the VALUE-STRING are printed as by the :BALANCED-MOUSABLE method.*"
  (let* ((command (send self :keyword-to-command keyword)))
    (send self :assure-command-present keyword)
    (let* ((name (send command :name)))
      (cond ((string-equal name "3Set *" :end1 4)    (setq name (subseq name 4)))
	    ((string-equal name "3Toggle *" :end1 7) (setq name (subseq name 7))))
      (send self :balanced-mousable (string-append name "3:*") value-string keyword))))


(defmethod (mandelbrot-menu-pane :after :refresh) (&optional (type :complete-redisplay))
  (if (< (tv:sheet-inside-height) 540)
      (send self :set-font-map '(fonts:tr10 fonts:tr10i))
      (send self :set-font-map '(fonts:tr12 fonts:tr12i)))
  (when (and (eq type :complete-redisplay) (send w:superior :displayer))
    (send self :clear-screen)
    (let ((julia-p (mandelbrot-descriptor-julia-seed (send (send w:superior :displayer) :working-descriptor))))
      (send self :set-label
	    `(:string ,(if julia-p "3Julia*" "3Mandelbrot*") :font fonts:mets)))
    
    (let* ((displayer (send w:superior :displayer)))
      (dolist (item *mandelbrot-menu-descriptor*)
	(cond ((stringp item)
	       (send self :string-out-centered-explicit item)
	       (terpri self))
	      
	      ((keywordp item) (send self :command-mousable item))
	      ((eq (car item) :set)
	       (let* ((get-name (second item))
		      (set-name (intern (string-append "3SET-*" get-name) "3KEYWORD*"))
		      (value (send displayer get-name))
		      (value-string (cond ((null value) "3none*")
					  ((typep value 'float)
					   (float-to-string (coerce value (send displayer :float-type)) 3))
					  (t (princ-to-string value)))))
		 (send self :command-balanced-mousable set-name value-string)))

	      ((eq (car item) :toggle)
	       (let* ((get-name (second item))
		      (tog-name (intern (string-append "3TOGGLE-*" get-name) "3KEYWORD*"))
		      (value-string (if (send displayer get-name) "3Yes*" "3No*")))
		 (send self :command-balanced-mousable tog-name value-string)))
	      
	      ((eq (car item) :2COL)
	       (let* ((left-name (second item))
		      (right-name (third item))
		      (left-command (ucl:get-command `(:method mandelbrot-frame ,left-name)))
		      (right-command (ucl:get-command `(:method mandelbrot-frame ,right-name)))
		      (w (max (w:sheet-string-length self (send left-command :name)  0 nil nil (aref w:font-map 0)) ;1 *tr12
			      (w:sheet-string-length self (send right-command :name) 0 nil nil (aref w:font-map 0)))))
		 (send self :assure-command-present left-name)
		 (send self :assure-command-present right-name)
		 (setq w:cursor-x (- (round (w:sheet-inside-width) 2) w 10))
		3 *(format self "3~VM*" (third (send left-command :defname)) (send left-command :name))
		  (setq w:cursor-x (+ (round (w:sheet-inside-width) 2) 10))
		3  *(format self "3~VM~%*" (third (send right-command :defname)) (send right-command :name))))
	      (t
	       (let* ((left (first item))
		      (right (eval (second item)))
		      (keyword (third item))
		      (command (ucl:get-command `(:method mandelbrot-frame ,keyword)))
		      (doc (or (send command :documentation) "")))
		 (unless (member command w:item-type-alist :test #'eq :key #'second)
		   (push (list keyword command doc) w:item-type-alist))
		 (send self :balanced-mousable left right keyword))))))))


;1;; The Commands*

(defcommand 4(mandelbrot-frame :redisplay*) ()
  '(:names "3Recompute Image*" :description "2Begin generating a Mandelbrot set image from the current parameters.*")
  (send typein :clear-screen)
  (format t "3~&Adjusting arrays...*")
  (send displayer :adjust-arrays)
  (format t "3 done.~%Computing...*")
  (let* ((draw-p (and (send displayer :image-displayable-p)
		      (send displayer :draw-while-computing)))
	 (small-image-p (or (< (send displayer :width)  (tv:sheet-inside-width displayer))
			    (< (send displayer :height) (tv:sheet-inside-height displayer)))))
    (send displayer :clear-screen)
    (cond ((not draw-p)
	   (send displayer :set-current-font fonts:cmr18 t)
	   (send displayer :string-out-x-y-centered-explicit "3Working...*"))
	  (t
	   (send displayer :bitblt w:alu-seta
		 (- (w:sheet-inside-width displayer) 2)
		 (- (w:sheet-inside-height displayer) 2)
		 w:12%-gray 0 0 0 0)))
    (send displayer :set-image-x-offset 0)
    (send displayer :set-image-y-offset 0)
    (let* ((completed-normally nil))
      (unwind-protect
	  (progn
	    (send displayer :display draw-p)
	    (setq completed-normally t))
	(when (or (not completed-normally) (not draw-p) small-image-p)
	  (send displayer :refresh)))))
  (format t "3 done.~%*"))


(defun 4read-char-or-click *()
  "2Reads a character or mouse click, whichever comes first.  If it's a click, returns a cons of X,Y on the mouse-sheet.
  This is a HACK for when you're in a state where READ-ANY doesn't work.*"
  (tv:with-mouse-grabbed
    (process-wait "3Keyboard/Click*" #'(lambda (stream)
					 (or (not (zerop (tv:mouse-buttons))) (listen stream)))
		  (sys:follow-syn-stream *standard-input*))
    (multiple-value-bind (buttons nil x y) (tv:mouse-buttons)
      (if (plusp buttons)
	  (progn (process-wait "3Mouse Up*" #'(lambda () (zerop (tv:mouse-buttons))))
		 (cons x y))
	  (read-char)))))


(defun 4pixel-to-range *(displayer x y)
  "2Convert a pixel-position in the mandelbrot-displayer window to a coordinate in image space.
  X and Y are relative to the inside-left of the displayer window.*"
  (let* ((w (send displayer :real-width))
	 (h (send displayer :real-height))
	 (range (send displayer :real-range))
	 (r (send displayer :real-real-origin))
	 (i (send displayer :real-imag-origin))
	 (x-offset (send displayer :image-x-offset))
	 (y-offset (send displayer :image-y-offset)))
    (incf x x-offset)
    (incf y y-offset)
    (let* ((type (coerce 1.0 (send displayer :float-type)))
	   (x-ratio-selected (float (/ x w) type))
	   (y-ratio-selected (float (/ y h) type))
	   (converted-x (+ r (float (* x-ratio-selected range) type)))
	   (converted-y (- i (float (* y-ratio-selected range) type))))
      (values converted-x converted-y))))


(defun 4parse-float* (string &key (start 0) end)
  "2Just like* (PARSE-INTEGER <string> :JUNK-ALLOWED T)2 except that it works on floats instead of integers.
Returns the Float parsed, and the position of the first character not used.*"
  (declare (simple-string string))
  (unless end (setq end (length string)))
  (let* ((period (position #\. string :start start :end end :test #'char=)))
    (cond ((null period)  ;1 If the string contains no decimal point, there's no need to try to parse a decimal-part.*
	   (multiple-value-bind (number stop) (parse-integer string :junk-allowed t :start start :end end)
	     (values (when number (float number))
		     stop)))
	  (t
	   (let* ((int-part (or (parse-integer string :start start :end period :junk-allowed t)
				0)))
	     (multiple-value-bind (number stop)
				  (parse-integer string :start (1+ period) :end end :junk-allowed t)
	       (let* ((end-of-number (or (position-if-not #'digit-char-p string :start (1+ period) :end stop)
					 stop))
		      (real-part (float (/ (or number 0)
					   (expt 10 (- end-of-number period 1))))))
		 ;1; In the above, we can't just use STOP to find the weight of the decimal part - suppose there was*
		 ;1; whitespace after the number in the string.  STOP would be at the end of the whitespace instead*
		 ;1; of at the end of the number.  So we calculate the END-OF-NUMBER.*
		 (values (* (if (zerop int-part) 1 (signum int-part))
			    (+ (abs int-part) real-part))
			 (if number stop (1- stop))))))))))


(defun 4prompt-and-read-two-numbers *(first second &optional displayer default-1 default-2)
  "2Prompts the user for two numbers in the form X,Y.  If they type only one number, the second is prompted for on its own line.
  FIRST is the string to use when prompting for one or both, and SECOND is the string to use when prompting for the second
  number (if they have typed only one).  If they just hit return at the two-number prompt, then the defaults will be returned.

 DISPLAYER, if non-NIL, is a Mandelbrot displayer pane; if provided, the user may click in that window to select the two numbers.*"

  (declare (values n1 n2))
  (let ((one nil) (two nil))
    (loop
      (if (and default-1 default-2 (not one))
	  (if displayer
	      (format t "3~?(default [~A, ~A], or click mouse) *" first () default-1 default-2)
	      (format t "3~?(default ~A, ~A) *" first () default-1 default-2))
	  (format t "3~?~:[~;(or click mouse) ~]*" (if one second first) () displayer))
      
      (let* ((char-or-cons (if displayer (read-char-or-click) (read-char)))
	     (line (unless (consp char-or-cons)
		     (unread-char char-or-cons)
		     (read-line)))
	     (comma (or (position #\, line :test #'char-equal)
			(position #\x line :test #'char-equal))))
	;1;*
	;1; Converting a click...*
	;1;*
	(when (and displayer (consp char-or-cons))
	  (multiple-value-bind (xoff yoff) (w:sheet-calculate-offsets displayer nil)
	    (let* ((x (- (car char-or-cons) xoff))
		   (y (- (cdr char-or-cons) yoff)))
	      (multiple-value-bind (tx ty) (pixel-to-range displayer x y)
		(write-two-floats t tx ty 5)
		(terpri)
		(return (values tx ty))))))
	;1;*
	;1; Reading characters...*
	;1;*
	(when (and (not one) (string= line ""))
	  (return (values default-1 default-2)))
	(cond (comma
	       (unless one
		 (setq one (parse-float line :end comma)
		       two (parse-float line :start (1+ comma)))))
	      (t
	       (cond (one
		      (setq two (parse-float line)))
		     (t
		      (setq one (parse-float line))
		      (when one
			(format t second)
			(setq two (parse-float (read-line)))))))))
      (if (and one two)
	  (return (values one two))
	  (beep))
      )))


(defcommand 4(mandelbrot-frame :set-size*) ()
  '(:names "3Set Size*"
    :documentation #.(string-append "2The size of the image in pixels.  If this is smaller than the default, the image will not fill the display.*"
				    #\Newline "2If it is larger, you can scroll to the hidden parts with the cursor keys.*"))
  (let* ((ww (w:sheet-inside-width displayer))
	 (hh (w:sheet-inside-height displayer)))
    (multiple-value-bind (w h) (prompt-and-read-two-numbers "3~&New Display Size: *" "3~&New Display Height: *"
							    nil ww hh)
      (send displayer :set-width  (round w))
      (send displayer :set-height (round h)))
    (send menu :refresh)))


(defcommand 4(mandelbrot-frame :set-julia-seed*) ()
  '(:names "3Set Julia*"
    :documentation
    #.(string-append "2The seed is the constant which drives a Julia-set image. *" #\Return
		     "2If this is ``dynamic'' then the seed for a point x,y is x+yi, in which case we are generating a Mandelbrot-set image.*"))
  (multiple-value-bind (julia-x julia-y)
		       (if (send displayer :real-julia-seed)
			   (prompt-and-read-two-numbers "3~&New Julia Seed: (default dynamic) *" "3~&New Julia Seed Y: *")
			   (prompt-and-read-two-numbers "3~&New Julia Seed, default dynamic: *" "3~&New Julia Seed Y: *"
							displayer))
    (send displayer :set-julia-seed (if julia-x
					(cons julia-x julia-y)
					nil)))
  (send menu :refresh))


(defcommand 4(mandelbrot-frame :set-contrast*) ()
  '(:names "3Set Contrast*"
    :documentation
    #.(string-append "2The Contrast Distance is a number in the range of 0 to Depth.* " #\Return
		     "2Any points whose value is* 2greater than or equal to Contrast Distance will be mapped out.*"))
  (let* ((c (prompt-and-read '(:number :or-nil t) "3~&New Contrast: (default none) *")))
    (send displayer :set-contrast (if c (round c) nil)))
  (send menu :refresh))


(defcommand 4(mandelbrot-frame :set-depth*) ()
  '(:names "3Set Depth*"
    :documentation #.(string-append "2The Depth of the image is the maximum number of iterations that will be made for each pixel.*"
				    #\Newline "2Higher numbers generally result in more complex (and computationally expensive) images.*"))
  (send displayer :set-depth (round (prompt-and-read :number "3~&New Image Depth: *")))
  (send menu :refresh))


;1;; ## unused?*
(defcommand 4(mandelbrot-frame :set-real-origin*) ()
  '(:names "3Set Real Origin*")
  (send displayer :set-real-origin
	(coerce (prompt-and-read :number "3~&New Real Origin: *") (send displayer :float-type)))
  (send menu :refresh))

;1;; ## unused?*
(defcommand 4(mandelbrot-frame :set-imag-origin*) ()
  '(:names "3Set Imaginary Origin*")
  (send displayer :set-imag-origin
	(coerce (prompt-and-read :number "3~&New Imaginary Origin: *") (send displayer :float-type)))
  (send menu :refresh))


(defcommand 4(mandelbrot-frame :set-image-origin*) ()
  '(:names "3Set Origin*"
    :documentation #.(string-append "2The origin of the image is the position of the upper left corner of the display on the *"
				    "2Real/Imaginary plane3.**" #\Newline
				    "2This, combined with Range, defines the part of the set you are looking at.*"))
  (multiple-value-bind (real imag)
		       (prompt-and-read-two-numbers "3~&New Origin: *" "3~&New Origin (Imaginary): *"
						    displayer (send displayer :real-origin) (send displayer :imag-origin))
    (send displayer :set-real-origin (coerce real (send displayer :float-type)))
    (send displayer :set-imag-origin (coerce imag (send displayer :float-type))))
  (send menu :refresh))



(defcommand 4(mandelbrot-frame :set-range*) ()
  '(:names "3Set Range*"
    :documentation #.(string-append "2The range of the image is how large a rectangle on the Real/Imaginary plane is being mapped into *"
				    "2the Size by Size rectangle on the screen.*"))
  (send displayer :set-range (coerce (prompt-and-read :number "3~&New Range: *") (send displayer :float-type)))
  (send menu :refresh))


(defcommand 4(mandelbrot-frame :set-drop-out*) ()
  '(:names "3Set Drop-Out*"
    :documentation "2If ever a point's mapping extends beyond this distance from the origin, it is dropped.  Should probably always be 4.0.*")
  (send displayer :set-drop-out (coerce (prompt-and-read :number "3~&New Drop-Out Distance: *")
					(send displayer :float-type)))
  (send menu :refresh))


(defcommand 4(mandelbrot-frame :set-bpp*) ()
  '(:names "3Set Bits per Pixel*"
    :documentation #.(string-append "2The number of bits per pixel of the image.*" #\Newline
				    "2If this is not the same as the screen's bpp, you must print out the image to see it.*"))
  (send displayer :set-bpp (round (prompt-and-read :number "3~&New Bits per Pixel: *")))
  (send menu :refresh))


(defcommand 4(mandelbrot-frame :revert-settings*) ()
  '(:names "3Revert Settings*" :documentation "2Make the displayed parameters match the current image.*")
  (send displayer :set-working-descriptor (copy-mandelbrot-descriptor (send displayer :displayed-descriptor)))
  (send menu :refresh))


(defcommand 4(mandelbrot-frame :toggle-draw-while-computing*) ()
  '(:names "3Toggle Dynamic*"
    :documentation "2If Yes, draw the image as we compute it.  Otherwise, do not display it until we are done (this is slightly faster).*")
  (send displayer :set-draw-while-computing (not (send displayer :draw-while-computing)))
  (send menu :refresh))


(defcommand 4(mandelbrot-frame :toggle-iterative-p*) ()
  '(:names "3Toggle Iterative*"
    :documentation #.(string-append "2If Yes, compute the image pixel by pixel.  If No, compute the image using a quad-tree descent.*"
				    #\Newline "2For some Julia-set images, this must be ``No'' to get a correct image.*"))
  (send displayer :set-iterative-p (not (send displayer :iterative-p)))
  (send menu :refresh))


(defcommand 4(mandelbrot-frame :toggle-detail*) ()
  '(:names "3Toggle Detail*"
    :documentation #.(string-append "2Whether to draw what is within the Contrast Distance perimeter.  ``No'' is faster.*" #\Newline
				    "2This only applies when there is a Contrast Distance.*"))
  (send displayer :set-detail (not (send displayer :detail)))
  (send menu :refresh))

(defcommand 4(mandelbrot-frame :set-float-type*) ()
  '(:names "3Set Precision*"
    :documentation "2What form of floating-point math to use in the computation.*")
  (let* ((result (w:menu-choose '#.(mapcar #'(lambda (x) (list (string-subst-char #\Space #\- (string-capitalize x))
							       :value x :font fonts:hl12
							       :documentation "2Select the floating-point precision to use.*"))
					   '(short-float single-float long-float double-float))
				:label '(:string "3Float Precision:*" :font fonts:hl12b))))
    (when result
      (send displayer :set-float-type result)
      (send menu :refresh))))


(setf (get 'square-display 'pretty-name) "2Square*")
(setf (get 'nonsquare-display 'pretty-name) "2Non-Square*")

(defmethod 4(mandelbrot-frame :window-configuration*) ()
  (or (get (send self :configuration) 'pretty-name)
      (string-capitalize (string (send self :configuration)))))

(defmethod 4(mandelbrot-displayer-pane :window-configuration*) ()
  (send tv:superior :window-configuration))

(defcommand 4(mandelbrot-frame :set-window-configuration*) ()
  '(:names "3Set Configuration*" :documentation "2Chose a different constraint-frame configuration.*")
  (let* ((choice (w:menu-choose (mapcar #'(lambda (x)
					    (cons (or (get (car x) 'pretty-name)
						      (string-capitalize (string (car x))))
						  (car x)))
					(send self :constraints))
				:label '(:string "2Configurations*" :font fonts:hl12b)
				:columns 1)))
    (when choice
      (send self :set-configuration choice)
      (send menu :refresh))))


(defmethod 4(mandelbrot-frame :process-priority*) () (send (send self :process) :priority))
(defmethod 4(mandelbrot-displayer-pane :process-priority*) () (send tv:superior :process-priority))

(defcommand 4(mandelbrot-frame :set-process-priority*) ()
  '(:names "3Set Priority*"
    :documentation
    "2Change the priority of the Mandelbrot process; making the priority negative gives preference to other activities.  -1 is a good choice.*")
  (send (send self :process) :set-priority (round (or (prompt-and-read '(:number :or-nil t)
								       "2~&New process priority (default 0): *")
						      0)))
  (send menu :refresh))


(defcommand 4(mandelbrot-frame :select-rectangle*) ()
  '(:names "3Select Rectangle*" :documentation "2Select a rectangle to view.  This computes a new image.*")

  (let* ((w:*default-rubber-band-size-type* nil))	;1 Code which uses this may or may not be loaded...*
    (declare (special w:*default-rubber-band-size-type*))
    (multiple-value-bind (x y x2 y2) (w:mouse-specify-rectangle nil nil nil nil displayer 5 5 nil)
      (multiple-value-bind (r1 i1) (pixel-to-range displayer x y)
	(multiple-value-bind (r2 i2) (pixel-to-range displayer x2 y2)
	  (let* ((r-range (- r2 r1))
		 (i-range (- i2 i1))
		 (new-range (max r-range i-range)))
	    (send displayer :set-range	 new-range)
	    (send displayer :set-real-origin r1)
	    (send displayer :set-imag-origin i1)
	    )))))
  (send menu :refresh)
  (send self :redisplay))


(defvar 4*mandelbrot-pan-ratio** 0.5  "2The amount by which the Pan commands move.*")
(defvar 4*mandelbrot-scroll-ratio** 0.1  "2The amount by which the Scroll commands move.*")


(defcommand 4(mandelbrot-frame :pan-up*) ()
  '(:names "3Pan Up*" :documentation "2Look at what's above here.  This computes a new image.*")
  (pan-mandelbrot 0 *mandelbrot-pan-ratio*))

(defcommand 4(mandelbrot-frame :pan-down*) ()
  '(:names "3Pan Down*" :documentation "2Look at what's below here.  This computes a new image.*")
  (pan-mandelbrot 0 (- *mandelbrot-pan-ratio*)))

(defcommand 4(mandelbrot-frame :pan-left*) ()
  '(:names "3Pan Left*" :documentation "2Look at what's to the left of here.  This computes a new image.*")
  (pan-mandelbrot (- *mandelbrot-pan-ratio*) 0))

(defcommand 4(mandelbrot-frame :pan-right*) ()
  '(:names "3Pan Right*" :documentation "2Look at what's to the right of here.  This computes a new image.*")
  (pan-mandelbrot *mandelbrot-pan-ratio* 0))


(defcommand 4(mandelbrot-frame :scroll-up*) ()
  '(:names "3Scroll Up*" :keys (#\Down-Arrow)
	   :documentation "2Look at what's below here.  This does not compute a new image, it just scrolls.*")
  (scroll-mandelbrot 0 *mandelbrot-scroll-ratio*))

(defcommand 4(mandelbrot-frame :scroll-down*) ()
  '(:names "3Scroll Down*" :keys (#\Up-Arrow)
	   :documentation "2Look at what's above here.  This does not compute a new image, it just scrolls.*")
  (scroll-mandelbrot 0 (- *mandelbrot-scroll-ratio*)))

(defcommand 4(mandelbrot-frame :scroll-left*) ()
  '(:names "3Scroll Left*" :keys (#\Left-Arrow)
	   :documentation "2Look at what's to the left of here.  This does not compute a new image, it just scrolls.*")
  (scroll-mandelbrot (- *mandelbrot-scroll-ratio*) 0))

(defcommand 4(mandelbrot-frame :scroll-right*) ()
  '(:names "3Scroll Right*" :keys (#\Right-Arrow)
	   :documentation "2Look at what's to the right of here.  This does not compute a new image, it just scrolls.*")
  (scroll-mandelbrot *mandelbrot-scroll-ratio* 0))


(defun 4pan-mandelbrot* (dx dy)
  (declare (:self-flavor mandelbrot-frame))
  (let* ((old-real (send displayer :real-real-origin))		;1 ## It would be nice if this function arranged so that*
	 (old-imag (send displayer :real-imag-origin))		;1 ## the overlapping region was not recomputed.*
	 (range (send displayer :real-range))
	 (type (send displayer :float-type)))
    (setf (send displayer :real-origin) (+ old-real (* (coerce dx type) range)))
    (setf (send displayer :imag-origin) (+ old-imag (* (coerce dy type) range)))
    (send menu :refresh)
    (send self :redisplay)))

(defun 4scroll-mandelbrot* (dx dy)
  (declare (:self-flavor mandelbrot-frame))
  (let* ((win-width (w:sheet-inside-width displayer))
	 (win-height (w:sheet-inside-height displayer))
	 (image-width (send displayer :real-width))
	 (image-height (send displayer :real-height))
	 (old-x (send displayer :image-x-offset))
	 (old-y (send displayer :image-y-offset)))
    (setf (send displayer :image-x-offset) (max 0 (min (- image-width win-width)   (+ old-x (round (* dx win-width))))))
    (setf (send displayer :image-y-offset) (max 0 (min (- image-height win-height) (+ old-y (round (* dy win-height))))))
    (when (or (/= old-x (send displayer :image-x-offset))
	      (/= old-y (send displayer :image-y-offset)))
      (send displayer :refresh))))
						

(defcommand 4(mandelbrot-frame :zoom-out*) ()
  '(:names "3Zoom Out*" :documentation "2Display twice as much.  This computes a new image.*")
  (let* ((range (send displayer :real-range))
	 (type (coerce 1.0 (send displayer :float-type))))
    (setf (send displayer :real-origin) (float (- (send displayer :real-real-origin) (float (/ range 2) type)) type))
    (setf (send displayer :imag-origin) (float (+ (send displayer :real-imag-origin) (float (/ range 2) type)) type))
    (setf (send displayer :range) (float (* 2 range) type)))
  (send menu :refresh)
  (send self :redisplay))


(defcommand 4(mandelbrot-frame :window-refresh*) () '(:names "3Refresh Windows*" :keys (#\Control-L #\Clear-Screen))
  (send self :refresh))



;1;; Documentation.*


(defcommand 4(mandelbrot-frame :help)* () '(:names "3Help*" :keys (#\Help) :documentation "2Display some documentation.*")
  (send displayer :help))

(defcommand 4(mandelbrot-frame :quit)* () '(:names "3Quit*" :keys (#\End) :documentation "2Exit the Mandelbrot viewer.*")
  (w:deselect-and-maybe-bury-window self))


(defun heading (string &optional font)
  (when font (send self :set-current-font font t))
  (terpri self)
  (send self :string-out-centered string)
  (terpri self))

(defun text-no-cr (&rest strings)
  (declare (special margin))
  (declare (:self-flavor w:graphics-mixin))
  (send self :set-current-font fonts:hl12 t)
  (when (< (- w:cursor-x w:left-margin-size) margin) (setq w:cursor-x (+ w:left-margin-size margin)))
  (dolist (string strings)
    (cond ((typep string 'w:font) (send self :set-current-font string t))	;1 Fonts mean font-changes.*
	  
	  ((eql string #\Clear-Screen)		;1 Clear-Screen char means force ``more'' and page.*
	   (fresh-line self) (terpri self)
	   (w:sheet-more-handler)
	   (send self :clear-screen)
	   (send self :set-cursorpos margin w:cursor-y))
	  
	  ((characterp string)			;1 Other characters are just written.  We check the margin afterwards.*
	   (write-char string self)
	   (when (< (- w:cursor-x w:left-margin-size) margin)
	     (setq w:cursor-x (+ w:left-margin-size margin))))
	  (t
	   (let* ((start 0))
	     (loop
	       (let* ((end (position #\Space string :start (1+ start) :test #'char=))
		      (length (w:sheet-string-length self string start (or end (length string))))
		      (break-p (> (+ w:cursor-x length) (- (w:sheet-inside-width) 10))))
		 (when break-p
		   (terpri self)
		   (incf w:cursor-x margin)
		   (when (char= #\Space (char string start)) (incf start)))
		 (write-string string self :start start :end (or end (length string)))
		 (if (null end)
		     (return)
		     (setq start end)))))))))

(defun text (&rest strings)
  (apply #'text-no-cr strings) (terpri self))

(defun para (&rest strings)
  (apply #'text strings) (terpri self))

(defun 4subheading *(string)
  (let* ((margin 10))
    (declare (special margin))
    (para #\Newline fonts:hl12bi string)))

(defun enum (title &rest strings)
  (declare (special margin))
  (setq margin 40)
  (text-no-cr fonts:hl12b title)
  (setq margin 120)
  (when (> (send self :cursor-x) (- margin 3)) (text-no-cr #\Newline))
  (apply #'para strings))


(defmethod 4(mandelbrot-displayer-pane :help)* ()
  (send self :clear-screen)
  (send self :set-more-p t)
  (unwind-protect
      (let* ((margin 10))
	(declare (special margin))
	(heading "3Mandelbrot Viewer*" fonts:mets)
	(heading "3by Jamie Zawinski*" fonts:hl12b)
	(format self "3~2%*")
	(para "3This program generates and displays fractal images from the Mandelbrot and Julia fractal sets.  *"
	      "3But you already guessed that, right?*")
	(send self :display-buttons))
    (send (send w:superior :typein) :clear-screen)
    (send self :set-more-p nil)
    (send self :refresh)))


(defmethod 4(mandelbrot-displayer-pane :display-buttons)* ()
  (send self :set-item-type-alist '((:commands	 :command-help "3Explain what the commands do.*")
				    (:mandelbrot :mandelbrot-help "3Explain what the Mandelbrot Set is.*")
				    (:algorithm  :algorithm-help "3Explain the algorithm this program uses.*")
				    (:done	 :abort "3Done reading help.*")))
  (send self :set-current-font fonts:hl12b t)
  (format self "3~4&*")
  (send self :clear-eof)
  (let* ((buttons '(("3Commands*" . :commands)
		    ("3Mandelbrot*" . :mandelbrot)
		    ("3Algorithm*" . :algorithm)
		    ("3Done*" . :done)))
	 (max-size (apply #'max (mapcar #'(lambda (s) (w:sheet-string-length self (car s))) buttons)))
	 (n (+ 2 (length buttons)))
	 (m (floor (- (w:sheet-inside-width) (* n max-size)) n))
	 (lm (round m 2))
	 (i 0))
    (dolist (b buttons)
      (send self :set-cursorpos (+ lm (* (+ m max-size) (incf i))) (- w:cursor-y w:top-margin-size))
      (format self "3~VM*" (cdr b) (car b)))
    (terpri self)
    (send self :string-out-centered-explicit "3Make a Selection:*"
	  w:left-margin-size (- w:cursor-y (* 4 w:line-height)) (w:sheet-inside-right) nil fonts:hl12i)
    (send self :draw-rectangle (+ lm m)
	  (round (- w:cursor-y (* 2.7 w:line-height))) (* (1+ n) max-size) (* 4 w:line-height)))
  
  (loop (let* ((blip (w:read-any self)))
	  (cond ((and (consp blip) (eq (car blip) :typeout-execute))
		 (send self (second blip))
		 (return (send self :display-buttons)))
		(t (beep))))))

(defmethod 4(mandelbrot-displayer-pane :abort)* () (signal-condition eh:abort-object))


(defmethod (4mandelbrot-displayer-pane :command-help)* ()
  (send self :clear-screen)
  (subheading "3The Commands:*")
  (enum "3Print this Image:*"
	"1This command produces a PostScript file of the image currently displayed.  You will be prompted* 1for a number of parameters before "*
	1"the file is printed; they all default to the answers you gave* 1the last time you executed this command.  You also have the option of "*
	1"just producing the PS file,* 1but not actually printing it.*")
  (para "1The code for translating bitmaps to PostScript is fairly powerful; it can print images with* 1multiple bits* 1per pixel even if such an image "*
	1"cannot be displayed on the screen.  (But since this* 1uses PostScript's automatic halftoning capabilities, the values for Bits per Pixel "*
	1"must be 1, 2,* 14, or 8).*")
  (para "1We don't just use the *" fonts:cptfontb "1print-bitmap*" fonts:hl12 "1 function, because that only works* 1on 1-bitplane images.*")
  (enum "3Save this Image:*"
	"1This command makes a copy of the currently displayed bitmap and parameters, and stores it on a* 1global variable under a "*
	1"prompted-for name.  You can later display this image again with:*")
  (enum "3Show Saved Image:*"
	"1This command restores the settings and image to those of a previously saved image.  There is not* 1currently any way to save an "*
	1"image to disk; when you reboot, your saved images will be lost.*")
  (enum "3Recompute Image:*"
	"1This recalculates the current image with the current parameters.  You should change the parameters* 1as you like, and then issue this "*
	1"command.  Some commands (like *" fonts:hl12b "1Select Rectangle"* fonts:hl12 "1) do this automatically.*")
  (enum "3Revert Settings:*"
	"1This command makes the current settings be the settings that the currently-displayed image was* 1calculated with.  This is useful if "*
	1"you have changed the settings, but want to back out.*")
  (enum "3Select Rectangle:*"
	"1This command lets you select a rectangle with the mouse, and recompute with parameters such that* 1the specified rectangle will fill "*
	1"the entire display window.  This automatically changes the"* fonts:hl12b "1Origin*" fonts:hl12 "1 and *" fonts:hl12b "1Range*"
	fonts:hl12 "1 parameters.  All other parameters* 1are the same.*")
  (enum "3Zoom Out:*"
	"1This command is roughly the opposite of *" fonts:hl12b "1Select Rectangle.*" fonts:hl12 "1  It will change* 1the *" fonts:hl12b
	"1Origin*" fonts:hl12 "1 and *" fonts:hl12b "1Range*" fonts:hl12 "1 parameters such that they* 1specify a rectangle twice as large "*
	1"as the current one, centered around the same center-point.* 1This automatically recomputes the image.*")
  (enum "3Pan Left, Right, Up, Down:*"
	"1These commands are for ``scrolling'' around in the Mandelbrot set.  They will automatically change* 1the *" fonts:hl12b "1Origin*"
	fonts:hl12 "1 parameter, and recompute.  The amount by which they scroll is controlled by the global variable *" fonts:cptfontb
	"1*mandelbrot-pan-ratio**" fonts:hl12 "1, which should* 1be a float between 0 and 1.  Its current value is *"
	(princ-to-string *mandelbrot-pan-ratio*) "1.*")

  (subheading "3The Display Modes:*")
  (enum "3Iterative:*"
	"1If this is ``yes,'' then the image will be computed by calculating the Mandelbrot mapping for* 1every pixel in the destination rectangle.  "*
	1"This takes quite a long time and is not fun to watch.* 1But if this is ``no,'' then we are allowed to make some assumptions (see the *"
	fonts:hl12i "1Algorithm*" fonts:hl12 "1 section)2, *and the image is computed using a quad-tree descent (which is much *"
	"1faster, but in certain cases is not as accurate).*")
  (enum "3Dynamic:*"
	"1If this is ``yes,'' then the image is drawn as it is computed.  The computation might be slightly* 1faster if this is ``no'' (but then, you *"
	"1don't get to watch).*")

1   *(subheading "3The Parameters:*")
1   *(enum "3Contrast:*"
	"1If there is no Contrast Distance specified, then the image displayed represents the iteration values* 1for a rectangle of points on the *"
	fonts:hl12i "1real x imaginary*" fonts:hl12 "1 plane.  With a Contrast Distance, any point whose iteration value is less than the *"
	"1Contrast Distance is not displayed.  If* 1the Contrast Distance is very close to the Depth (it cannot be larger) then you will clearly *"
	"1see* 1the ``fractal edge'' of the Mandelbrot image.*")
  (enum "3Detail:*"
	"1If there is a Contrast Distance, and Detail is ``No'', then points whose iteration values are less* 1than the Contrast Distance will all be *"
	"1the same color.  This will produce a two-color image, with* 1the boundary between the two colors being a fractal edge.  If this is *"
	"1``Yes,'' then the area within* 1the fractal edge will be colored by the usual method.  This is meaningful only if there is a* "
	"1Contrast Distance.*")
  (enum "3Precision:*"
	"1Calculating the Mandelbrot mapping uses floating-point math.  This parameter controls the* 1floating-point precision to be used; *"
	"1usually *" fonts:hl12b "1short-float*" fonts:hl12 "1 is good* 1enough, but if you are zoomed in far enough that your image looks *"
	"1grainy, crank this up.  Long-Float doesn't start to fail until *" fonts:hl12b "1range*" fonts:hl121 *"3 1is less than about **"
	fonts:hl12b "17.53 *e-14*" fonts:hl12 "1, so it shouldn't be a problem...*")

  (enum "3Size:*"
	"1This is the size in pixels of the square which will be generated.* 1It defaults to the size of the ``display'' window, but may be larger or *"
	"1smaller.  If it is* 1larger, then more data will be computed than the display window can show at once, but the cursor* 1keys may be *"
	"1used to scroll around.*")
  (enum "3Depth:*"
	"1This is the maximum number of iterations which will be made on each pixel.  Larger values for* 1this will produce more complex *"
	"1images, but the images will take longer to compute.*")
  (enum "3Origin, Range:*"
	"1The origin and range define the rectangle in the *" fonts:hl12i "1real x imaginary*" fonts:hl12 "1 plane which is being mapped into *"
	"1the *" fonts:hl12i "1size x size*" fonts:hl12 "1 rectangle on the display.*")
  (enum "3Bits per Pixel:*"
	"1The bitmap which is generated can be arbitrarily deep.  However, it can only be displayed on the* 1screen if it is of the same depth *"
	"1as the screen; this means that, on a monochrome monitor, you* 1can only display the image if it is one bit deep, and on a color *"
	"1monitor, you can only display* 1it if it is one or eight bits deep.*")
  )


(defmethod (4mandelbrot-displayer-pane :algorithm-help)* ()
  (declare (special margin))
  (send self :clear-screen)
  (setq margin 10)
  (heading "3The Methods Used*" fonts:tr18)
  (terpri self)
  (para "1This program can generate the set in two ways.*")
  (para1 *"1One is a brute force method, which merely calculates* 1the mapping for every point in the destination* 1image.*")
  (para "1The other takes advantage of the fact that it is extremely rare for there to be a rectangle in the* 1destination image whose perimeter "*
	1"is all of the same color, but which has different colors on the interior.  That is, we assume that if all of the pixels on the perimeter "*
	1"of a rectangle are of the same color, then all of the pixels* 1inside that rectangle are of that color as well.*")
  (setq margin 30)
  (para "1We implement this like so:*")
  (setq margin 50)
  (para "1  Divide the destination image into four quadrants.*")
  (para "1  For each quadrant,* 1begin calculating the mapping of the pixels on its perimiter.*")
  (text-no-cr "1  *") (incf margin 10)
  (para "1If ever we compute the color for a* 1pixel which is different from the color of the other pixels we* 1have computed so far "*
	1"on this quadrant,*")
  (text-no-cr "1  *") (incf margin 10)
  (para "1give up on the perimiter-walk, subdivide the quadrant into four sub-quadrants, and recurse* 1on them in the same manner.*")
  (decf margin 20)
  (text-no-cr "1  *") (incf margin 10)
  (para "1Otherwise, we assume that all of the pixels within the quadrant are the color of the pixels on the* 1perimiter, and we fill it.*")
  (setq margin 10)
  (para "1With a brute-force iteration, most of the time is spent computing the values of pixels with very high iteration counts, such as those "*
	1"near 0,0. * 1Since these pixels will likely exceed the Depth limit,* 1they* 1will all be of the same color.* 1With the quad-tree approach, "*
	1"these uninteresting parts of the* 1image* 1are actually *" fonts:hl12i "1faster*" fonts:hl12 "1 than the less deep (but more interesting) "*
	1"parts* 1of the* 1image.*")
  (para "1This means that, not only is less time spent on the hard-to-compute parts of the image, but* 1also that most of the time is spent on "*
	1"the *" fonts:hl12i "1interesting*" fonts:hl12 "1 parts.* 1If a section of the* 1image is mostly the same color, not much time is spent "*
	1"there.  If a part* 1of the image is very* 1crinkly, more time will be spent there, but, since it is interesting to look* 1at, there is a big* 1"*
	1"cognitive win -- you don't get bored as easilly."*
;	#\Clear-Screen
	)
  
  (subheading "1Why Would I Ever Use the Iterative Method?*")
  (para "1The assumption that the interior of a rectangle with a solid-colored perimiter is also solid is not* 1always a valid one.  One "*
	1"interesting feature of the Julia set is that it contains ``islands,''* 1that is, areas of color that are not ``attached'' to the main "*
	1"body (really what this means is that* 1the number of iterations occasionally increases as distance from 0,0 increases).2  **")
  (para "1You will only get islands if you are using a *" fonts:hl12b "1Contrast Distance*" fonts:hl12 "1 and it* 1is set to a value close to "*
	1"the value of *" fonts:hl12b "1Depth*" fonts:hl12 "1.  In this situtation,* 1it is possible for islands to be lost, and you should "*
	1"probably use the Iterative method.*")
  (para "1If you are not using a contrast ratio, islands do not occur, and the quad-tree approach always works.*")
  )


(defmethod (4mandelbrot-displayer-pane :mandelbrot-help)* ()
  (send self :clear-screen)
  (heading "1What the Mandelbrot and Julia Sets are*" fonts:tr18)
  (terpri self)
  (let ((margin 3))
    (declare (special margin))
    (para "1Let *" fonts:cptfontb "1z0*" fonts:hl121 *"1 be a number in the complex plane (x + yi), and *" fonts:cptfontb "1C*" fonts:hl12
	  "1 be a constant (also a complex3 *number).* 1Calculate *" fonts:cptfontb "1z1 = z0^2 + C*" fonts:hl12 "1.  Repeat this* 1recursively, *"
	  "1so that *" fonts:cptfontb "1z2 = z1^2 + C*" fonts:hl12 "1, *" fonts:cptfontb "1z3 = z2 ** 2 + C*" fonts:hl12 "1 and* 1so on.  *"
	  fonts:cptfontb "3z*" fonts:hl12 "1 will tend towards either infinity or zero, depending on its initial value and the constant.  *")
    (para "1A Julia set is computed by taking each point in a section of the complex plane and running it* 1through the above formula, coloring *"
	  "1the point based on how many iterations it takes it to diverge.* 3 1The ``Depth'' parameter is the maximum number of iterations to **"
	  "1allow a point to progress before we* 1assume it will go to infinity.  *")
    (para "1These points may be colored any way you like; the usual way (on a color system) is to simply use* 1them as indices into the *"
	  "1hardware color map.  On a monocrhome system, we color them white or black* 1based on whether they are odd or even.  *")
    (para "1A Mandelbrot set is a Julia set where the constant *" fonts:cptfontb "1C*" fonts:hl12 "1, for a pixel *" fonts:cptfontb "1Z*"
	  fonts:hl12 "1, is *" fonts:cptfontb "1Z*" fonts:hl12 "1.  That is, where in a Julia image, the same value of *" fonts:cptfontb
	  "1C*" fonts:hl12 "1 is used for every pixel, in a Mandelbrot image the value of *" fonts:cptfontb "1C*" fonts:hl12 "1 changes for *"
	  "1every pixel, and is the coordinate of that pixel (*" fonts:cptfontb "1C = z0*" fonts:hl12 "1).  *")
    (para "1Mandelbrot and Julia sets are obviously very closely related.  *If the constant " fonts:cptfontb "1C*" fonts:hl12
	  " 1for a Julia image is chosen from the interior of the Mandelbrot set, then the Julia set calculated from that seed will be* "
	  "1connected;* 1however, if the seed is chosen from outside of the Mandelbrot set, the Julia set will* 1be a discontinuous set of* "
	  "1``islands.''  For some* 1renderings of these images, an iterative method* 1must be used, as the quad-tree method can map out some* "
	  "1features.  If the seed is* 1chosen from the* 1perimiter of the Mandelbrot set, then the Julia set will be more convoluted.  To take* "
	  "1advantage of this close* 1relationship between the two image sets, you can choose a new Julia seed by clicking in a* 1rendered *"
	  "1Mandelbrot image.  *")
    ))


;1;; Saving and Printing commands.*

(defmethod 4(mandelbrot-frame :descriptive-string*) (&optional depth real imag range drop seed)
  (format nil "3Depth = ~D, Origin = ~F + ~F i, Range = ~F, Drop-Out = ~F~:[.~;, Seed = ~F + ~F i.~]*"
	  (or depth (send displayer :real-depth))
	  (or real  (send displayer :real-real-origin))
	  (or imag  (send displayer :real-imag-origin))
	  (or range (send displayer :real-range))
	  (or drop  (send displayer :real-drop-out))
	  (or seed  (setq seed (send displayer :real-julia-seed)))
	  (car seed)
	  (cdr seed)
	  ))

(defmethod 4(mandelbrot-frame :descriptor-descriptive-string*) (mandelbrot-descriptor)
  (send self :descriptive-string 
	     (mandelbrot-descriptor-depth mandelbrot-descriptor)
	     (mandelbrot-descriptor-real-origin mandelbrot-descriptor)
	     (mandelbrot-descriptor-imag-origin mandelbrot-descriptor)
	     (mandelbrot-descriptor-range mandelbrot-descriptor)
	     (mandelbrot-descriptor-drop-out mandelbrot-descriptor)
	     (mandelbrot-descriptor-julia-seed mandelbrot-descriptor)))



;1;; This is an 6ACTUAL REAL USE* of a top-level dynamic closure.*
;1;;*
;1;; I want the 5choose-variable-values* menu to default its answers to the answers given the last time it was invoked.*
;1;; We could do this by 5defvar*'ing all of the choices, but that's kind of gross - global variables used only one place?*
;1;; Instead, we have a set of special variables that are private to the 5mandelbrot-print-image-prompt* function - they*
;1;; retain their values across subsequent calls to that function.*
;1;;*
;1;; You can create lexical closures just by wrapping a 5let* around a 5defun*, but that doesn't work for dynamic closures.*
;1;; To make a dynamic closure, you must call the 5closure* function (or use 5let-closed*, which does some stuff for you).*
;1;; So, we define a function 5mandelbrot-print-image-prompt-internal* which expects a set of special bindings to be*
;1;; around when it is called.  then we define a function 5mandelbrot-print-image-prompt*, which has in its value cell a*
;1;; 5closure* object that dynamically closes a set of variables over 5mandelbrot-print-image-prompt-internal*.*
;1;;*

(defun 4mandelbrot-print-image-prompt-internal* ()
  "2Expects special bindings of title-string-p, orientation, dots-per-inch, spool-file, print-after, delete-after, header-page-p, and invert-p.*"
  (declare (special title-string-p orientation dots-per-inch spool-file print-after delete-after header-page-p invert-p))
  (unless spool-file
	  (setq spool-file (make-pathname :defaults (user-homedir-pathname) :version :newest :name "3MAND*" :type "3PS*")))
  (let* ((w:*choose-variable-values-name-standard-font* fonts:hl12))
    (w:choose-variable-values
      '((title-string-p "3Print a Title String*"
			:documentation "2Whether to print out a line describing the settings that generated this image.*"
			:boolean)
	(orientation "3Orientation*"
		     :documentation "2Whether to rotate the image on the page.*"
		     :assoc (("3Portrait*" . :portrait) ("3Landscape*" . :landscape)))
	(dots-per-inch "3Dots per Inch*"
		       :documentation "2How large the resultant image should be.  NIL means the image will be scaled to fit the page.*"
		       :number-or-nil)
	(spool-file "3Spool File*"
		    :documentation "2The file in which the PostScript code will be written.*"
		    :pathname)
	(print-after "3Print after Spooling*"
		     :documentation "2Whether the file should be shipped to the printer after the file is written.*"
		     :boolean)
	(delete-after "3Delete after Printing*"
		      :documentation
		      "2Whether the file should be deleted after it is printed.  This is only meaningful if ``Print after Spooling'' is ``Yes.''*"
		      :boolean)
	(header-page-p "3Print a Header Page*"
		       :documentation
		       2"Whether to print a header page with this print request.  This is only meaningful if ``Print after Spooling'' is ``Yes.''*"
		       :boolean)
	(invert-p "3Image Color*" :documentation "2Whether to invert the image in the output PostScript code.*"
		  :assoc (("3Black on White*" . t) ("3White on Black*" . nil)))
	)
      :label '(:string "3Mandelbrot Printer Preferences*" :font fonts:hl12b)
      :margin-choices '(("3Abort*" (signal-condition eh:abort-object)) "3Do It*")
      :value-tab 180))
  (values title-string-p orientation dots-per-inch spool-file print-after delete-after header-page-p invert-p))


;1;; This form puts in the function cell of 5mandelbrot-print-image-prompt* a dynamic closure object which closes a bunch of variables over*
;1;; the 5mandelbrot-print-image-prompt-internal* function.*
;1;;*
(fdefine '4mandelbrot-print-image-prompt*
  (let-closed ((title-string-p t) (orientation :portrait) (dots-per-inch nil) (spool-file nil)
	       (print-after t) (delete-after t) (header-page-p nil) (invert-p t))
    '4mandelbrot-print-image-prompt*-internal))

(setf (documentation 'mandelbrot-print-image-prompt 'function)
      "2Prompt the user for some image-printing preferences and return them.  This remembers what they answered last time.*")



(defcommand 4(mandelbrot-frame :print-image*) ()
  '(:names "3Print This Image*" :description "2Print the current image on the default bitmap printer (PostScript printers only).*")
  (multiple-value-bind (title-string-p orientation dots-per-inch spool-file print-after delete-after header-page-p invert-p)
		       (mandelbrot-print-image-prompt)
    (let* ((bitmap (send displayer :data-array))
	   (name (when title-string-p
		   (string-append (if (send displayer :real-julia-seed) "3Julia Set: *"  "3Mandelbrot Set: *")
				  (send self :descriptive-string)
				  )))
	   truename)
      (check-type bitmap array)
      (with-open-file (file-stream spool-file :direction :output)
	(multi-plane-bitmap-to-postscript-code file-stream bitmap
					       :title-string name :orientation orientation
					       :dots-per-inch dots-per-inch :invert-p invert-p)
	(setq truename (send file-stream :truename)))
      (when print-after
	(print-file truename :header header-page-p :printer-name (get-default-image-printer)
		    :header header-page-p :page-heading nil :delete-after delete-after)))))



(defvar 4*saved-images** '() "2An Alist of saved Mandelbrot bitmaps.  Is this a core-leak or what?*")


(defcommand 4(mandelbrot-frame :save-image*) ()
  '(:names "3Save This Image*" :documentation "2Store the current image away for future reference.*")
  (let* ((bitmap (send displayer :data-array))
	 name)
    (check-type bitmap array)
    (loop
      ;1; ## We should do something about replacing existing saved-images of the same name.*
      (format t "3~&Type a symbol to name this image: *")
      (setq name (let ((*package* *keyword-package*)) (read)))
      (if (and name (symbolp name))
	  (return)
	  (beep)))
    (setq bitmap (copy bitmap))
    (let* ((cons (assoc name *saved-images*))
	   (data (cons bitmap (copy-mandelbrot-descriptor (send displayer :displayed-descriptor)))))
      (if cons
	  (setf (cdr cons) data)
	  (push (cons name data) *saved-images*)))))


(defcommand 4(mandelbrot-frame :show-saved-image*) ()
  '(:names "3Show Saved Image*" :description "2Display an image previously saved with the ``Save This Image'' command.*")
  (unless *saved-images*
    (beep)
    (format t "3~&There are no saved images.~%*")
    (signal-condition eh:abort-object))
  (let* ((cons (w:menu-choose (mapcar #'(lambda (x)
					  `(,(car x)
					    :value ,x
					    :documentation ,(string-append (string (car x)) "3:  *"
							     (send self :descriptor-descriptive-string (cddr x)))))
				      *saved-images*)))
	 (array (car (cdr cons)))
	 (desc  (cdr (cdr cons))))
    (when cons
      (assert (and (mandelbrot-descriptor-p desc)
		   (if array (arrayp array) t))
	      (desc array)
	      "2Something bad has gotten onto *SAVED-IMAGES*.*")
      (setf (send displayer :data-array)           (copy array)
	    (send displayer :displayed-descriptor) (copy-mandelbrot-descriptor desc)
	    (send displayer :working-descriptor)   (copy-mandelbrot-descriptor desc))
      (send displayer :set-image-x-offset 0)
      (send displayer :set-image-y-offset 0)
      (cond (array
	     (send displayer :refresh))
	    (t
	     (let ((font fonts:cmr10))
	       (send displayer :clear-screen)
	       (send displayer :string-out-x-y-centered-explicit
		     #.(format nil "3Image not saved; select~%``Recompute Image.''*")
		     (tv:sheet-inside-left displayer) (tv:sheet-inside-top displayer) (tv:sheet-inside-right displayer)
		     (tv:sheet-inside-bottom displayer) font (tv:sheet-char-aluf displayer) 0 nil
		     (tv:font-char-height font)))
	     (send displayer :adjust-arrays)))
      (when (send displayer :cache-array) (fill (send displayer :cache-array) 0))
      (send menu :refresh)
      )))


;1;; This command-table is used only for the sake of commands with keybindings;*
;1;; The menu-pane would work even if there was no command table.*
;1;;*
(build-command-table '*mandelbrot-command-table* 'mandelbrot-frame
  '(:redisplay
    :window-refresh
    :set-size
    :set-contrast
    :set-julia-seed
    :set-depth
    :set-range
    :set-drop-out
    :set-bpp
    :select-rectangle :zoom-out
    :pan-up :pan-down :pan-left :pan-right
    :scroll-up :scroll-down :scroll-left :scroll-right
    :help
    :quit
    :print-image
    :save-image
    :show-saved-image
    :toggle-draw-while-computing
    :toggle-detail
    :revert-settings
    :toggle-iterative-p
    :set-float-type
    :set-image-origin
    :set-window-configuration
    :set-process-priority
    ))



;1;; If we're running the KSL desktop, give this program an icon.*

#+KSL
(defvar 4*icon**
 '#.(make-array '(64 64) :element-type 'bit :displaced-to
     (make-array 128 :element-type '(unsigned-byte 32) :initial-contents
      '#(0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 65536 0 229376 0 507904 0 1032192 0 507904 0 229376 0 6288384 0 8387776
	 0 234880960 0 268435328 0 134217696 0 268435448 0 268435448 0 1073741820 0 1073741820 67108864 536870908 255852544
	 536870910 1069547520 536870910 2146435072 536870910 4293918720 536870910 4293918720 268435454 4294311936 268435454
	 4294967294 67108863 4294311936 268435454 4293918720 268435454 4293918720 536870910 2146435072 536870910 1069547520
	 536870910 255852544 536870910 67108864 536870908 0 1073741820 0 1073741820 0 268435448 0 268435448 0 134217696 0
	 268435328 0 234880960 0 8387776 0 6288384 0 229376 0 507904 0 1032192 0 507904 0 229376 0 65536 0 0 0 0 0 0 0 0 0 0
	 0 0 0 0 0 0 0 0 807473154 139409077 167772672 810061841 842017843 876096567 842083385 926231603 55 772800515
	 167772184 167772161 813862962 3932195 5374017 2949204 4325425 3538989 2949172 3407926 3276832 3538995 3342387
	 3538994 4063285 0 0 0 0 0 0 0 0 0 0 0 0 0 842581257 2591303821 772800515 167772160 167772161 813862962 0 0 0 0 0 0
	 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 810061841 842017843 876096567 858860601 892480561 48 809013319 167772200
	 167772275 167772261 167772276 167772273 167772192 167772258 167772192 167772200 167772279 167772218 167772274
	 167772261 167772257 167772260 167772205 167772258 167772265 167772276 167772205 167772257 167772274 167772274
	 167772257 167772281 167772205 167772262 167772265 167772268 167772261 167772192 167772194 167772234 167772247
	 167772250 167772218 167772192 167772234 167772247 167772250 167772206 167772238 167772229 167772247 167772219
	 167772192 167772237 167772225 167772238 167772228 167772229 167772236 167772226 167772242 167772239 167772244
	 167772205 167772233 167772227 167772239 167772238 167772206 167772226 167772233 167772244 167772237 167772225
	 167772240 167772194 167772201 167772201 810061888 40 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 810061842 1652122955 1685217647
	 1682251834 1919906921 13088 774897666 702899401 1 774897666 702852352 1 810061841 842017843 876096567 858860601
	 842607672 51 810061888 115 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 810061842 1652122955 1685217647 1682251834 1919906921
	 13088 810061888 101 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 810061888 116 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 810061888 113 0 0 0 0
	 0 0 0 0 0 0 0 0 0 0 0 810061828 1903453555 810061888 32 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 810061888 99 0 0 0 0 0 0 0 0
	 0 0 0 0 0 0 0 810061888 40 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 774897665 40823404 774897666 2130699 1 774897666 2130688 1
	 774897666 2147483647 131071 774897666 721665238 4096 774897666 2147483647 131071 774897666 721665238 4096 774897665
	 40823916 774897666 2130699 1 774897666 2130688 1 774897666 2130688 1 774897665 40824428 774897666 2130699 1
	 774897666 2130688 1 774897666 2147483647 131071 774897666 721665238 4096 774897665 40824920 774897666 2130699 1
	 774897666 2130688 1 774897666 2147483647 131071 774897666 721665238 4096 810061841 842017843 876096567 875637817
	 959786038 54 810061888 109 0 0 0 0 0 0 0 0)
      )))

#+KSL
(defmethod 4(mandelbrot-frame :make-icon*) ()
  (make-instance 'w:graphics-icon :window-being-represented self :borders 1
		 :picture *icon* :label (list :string w:name :font FONTS:HL12B)))



;1;; Other noise.*

(defun 4mandelbrot* ()
  "2Invoke a Mandelbrot viewer.*"
  (w:select-or-create-window-of-flavor 'MANDELBROT-FRAME)
  nil)

(w:delete-from-system-menu-column :programs "2Mandelbrot*")
(w:add-to-system-menu-column :PROGRAMS "2Mandelbrot*" '(mandelbrot) '(:documentation "2Select a mandelbrot viewer window.*") nil)

(tv:add-window-type "2Mandelbrot*" 'MANDELBROT-FRAME "2A mandelbrot viewer window.*")


;1;;; This is some conditional compilation noise - ignore it.*
(eval-when (eval compile)
  (setq *features* (delete :accelerators *features*)))

(compile-flavor-methods
  BASIC-MANDELBROT-DISPLAYER
  4MANDELBROT-FRAME*
  4MANDELBROT-TYPEIN-PANE*
  MANDELBROT-MENU-PANE
  4MANDELBROT-DISPLAYER-PANE*)



;1;; Hardcopy.*

(defun 4flip-bits* (n)
  "2N is an 8-bit number.  Returns a number which is N with its bits in the opposite order.*"
  (let* ((m 0))
    (setq m (dpb (ldb (byte 1 0) n) (byte 1 7) m))
    (setq m (dpb (ldb (byte 1 1) n) (byte 1 6) m))
    (setq m (dpb (ldb (byte 1 2) n) (byte 1 5) m))
    (setq m (dpb (ldb (byte 1 3) n) (byte 1 4) m))
    (setq m (dpb (ldb (byte 1 4) n) (byte 1 3) m))
    (setq m (dpb (ldb (byte 1 5) n) (byte 1 2) m))
    (setq m (dpb (ldb (byte 1 6) n) (byte 1 1) m))
    (setq m (dpb (ldb (byte 1 7) n) (byte 1 0) m))
    m))

(defun 4make-bit-flippage-array* ()
  "2Returns a vector (256 long) where an index N has in it an integer which is N with its bits in reverse order.*"
  (let* ((a (make-array 256 :element-type '(integer 0 255))))
    (dotimes (i 256)
      (setf (svref a i) (flip-bits i)))
    a))

(defvar 4*bit-flippage** (make-bit-flippage-array)
  "2A vector where an index N has in it an integer which is N with its bits in reverse order.
  This is used to do this silly calculation quickly.*")

(defun 4multi-plane-bitmap-to-postscript-code *(output-stream bitmap
					      &key w h
						   title-string
						   (orientation :best)
						   dots-per-inch
						   invert-p
						   )
  (unless w (setq w (array-dimension bitmap 1)))
  (unless h (setq h (array-dimension bitmap 0)))
  (let* ((eltype (array-element-type bitmap))
	 (bpp (or (and (eq eltype 'BIT) 1)
		  (and (consp eltype)
		       (or (and (eq (car eltype) 'MOD) (1- (integer-length (1+ (second eltype)))))
			   (and (eq (car eltype) 'UNSIGNED-BYTE) (second eltype))
			   ))
		  (error "3Don't know how to get bits-per-pixel out of typespec ~S.*" eltype))))
    (assert (member bpp '(1 2 4 8)) ()
	    "3PostScript can't cope with ~D-bit images.  Must be 1, 2, 4, or 8.*" bpp)
    (write-bitmap-prologue output-stream w h bpp title-string orientation dots-per-inch "3Helvetica*" 10 t)
    
    (write-line "3picimage*" output-stream)
    (let* ((displaced (make-array (ceiling (* w h bpp) 8)
				  :element-type '(unsigned-byte 8)
				  :displaced-to bitmap)))
1        *(let* ((tick 0) ;1 for inserting a newline every 78 characters.*
	     (chars "30123456789ABCDEF*"))
	(dotimes (i (length displaced))
	  (let* ((n (svref *bit-flippage* (aref displaced i))))
	    (declare (type (unsigned-byte 8) n))
	    (when invert-p (setq n (lognot n)))
	    (write-char (schar chars (ldb (byte 4 4) n)) output-stream)
	    (write-char (schar chars (ldb (byte 4 0) n)) output-stream))
	   (when (= 78 (incf tick 2))
	     (setq tick 0)
	     (terpri output-stream))))))
  (format output-stream "3~2%showpage~%*")
  )



(defun write-bitmap-prologue4 *(output-stream width height bits-per-pixel title-string orientation dots-per-inch
			      font size center-p)
  (write-line "3%!PS-Adobe-1.0*" output-stream)
  (write-line "3%%Pages:1*" output-stream)
   (format output-stream "3%%DocumentFonts:~A~%*" (or font ""))
  (when title-string (format output-stream "3%%Title:~A~%*" title-string))
  (write-line "3%%EndComments*" output-stream)
  (terpri output-stream)
  (let* ((byte-width (ceiling (* bits-per-pixel width) 8)))
    (format output-stream "3/picstr ~D string def~%*" byte-width)	;1 Make a buffer for each scanline.*
    (format output-stream
	    #.(string-append "3/picimage {*"
			     "3~D ~D ~D *"					;1 Size and depth of image.*
			     "3[~D 0 0 ~D 0 ~D] *"				;1 Transformation matrix.*
			     "3{currentfile picstr readhexstring pop} *"	;1 Proc to pass to 5image.**
			     "3image} bind def*~%")
	    width height bits-per-pixel
	    width (- height) height
	    ))
  (write-line "3%%EndProlog*" output-stream)
  (terpri output-stream)
  (write-line "3%%Page:1 1*" output-stream)
  (let* ((page-width-in-points  (* 72.3 8.5))
	 (page-height-in-points (* 72.3 11.0))
	 (portrait-p (ecase orientation
		       (:portrait t)
                       (:landscape nil)
		       ((:best nil) (<= width height))))
	 (scale (if dots-per-inch
		    (let* ((dots-per-point (float (/ 72.3 dots-per-inch))))
		      dots-per-point)
		    (min (float (/ page-width-in-points width))
			 (float (/ page-height-in-points height)))))
	 (image-width-in-points (* width scale))
	 (image-height-in-points (* height scale)))
    
    (when (and font title-string)
      (format output-stream "3gsave~% 30 ~,3F moveto~% /~A findfont ~D scalefont setfont~%*"
	      (- page-height-in-points (+ size 30)) font size)
      (format output-stream "3 (~A)show~%*" title-string)
      (format output-stream "330 30 moveto~% /~A findfont ~D scalefont setfont~%(~A)show grestore~%*"
	      font (float (round size 2)) "3software by Jamie Zawinski <jwz@teak.berkeley.edu>*"))
    
    (if center-p
	(format output-stream "3~,3F ~,3F translate~%*"
		(float (/ (- page-width-in-points image-width-in-points) 2))
		(float (/ (- page-height-in-points image-height-in-points) 2)))
	(format output-stream "30 ~,3F translate~%*" image-height-in-points))
    
    (unless portrait-p
      (format output-stream "30 ~,3F translate -90 rotate~%*" image-height-in-points page-height-in-points))
    (format output-stream "3~,3F ~,3F scale~%*" image-width-in-points image-height-in-points)
    ))



;1;; Some default saved-images.*


(defmacro 4defimage *(name &rest init-args)
  `(pushnew (cons ',name (cons nil (make-mandelbrot-descriptor ,@init-args)))
	    *saved-images* :test #'eq :key #'car))

(defimage :many-faces
	  :width 540 :height 540 :depth 20 :float-type 'short-float
	  :real-origin -0.43846s0 :imag-origin 0.415024s0 :range 0.135899s0 :contrast 18 :detail nil)
(defimage :clouds
	  :width 540 :height 540 :depth 21 :detail t :float-type 'short-float
	  :real-origin -0.0307693s0 :imag-origin -0.81711s0 :range 0.271797s0 :contrast 18)
(defimage :seuss-limit
	  :width 540 :height 540 :depth 101 :float-type 'short-float
	  :real-origin -0.47244s0 :imag-origin 0.129639s0 :range 0.262737s0 :detail nil)
(defimage :hooks
	  :width 540 :height 540 :depth 20 :float-type 'short-float :real-origin -0.43846s0 :imag-origin 0.55545s0
	  :range 0.98753s0 :contrast 8 :julia-seed '(1.8401s0 . -0.010788s0))
(defimage :blot
	  :width 540 :height 540 :depth 20 :float-type 'short-float :real-origin -0.067005s0 :imag-origin 1.30153s0
	  :range 1.70776s0 :contrast 5 :julia-seed '(0.89787s0 . 0.600746s0))
(defimage :flow
	  :width 540 :height 540 :depth 15 :detail nil :float-type 'short-float
	  :real-origin 1.68018s0 :imag-origin 0.115486s0 :range 0.212921s0)
(defimage :ugly-snowflake
	  :width 540 :height 540 :depth 50 :float-type 'short-float :depth 50
	  :real-origin 0.64472s0 :imag-origin -0.437355s0 :range 0.049324s0 :contrast 48 :detail nil)
(defimage :fjords
	  :width 540 :height 540 :depth 50 :float-type 'short-float
	  :real-origin 0.48564s0 :imag-origin -0.40036s0 :range 0.22197s0)
(defimage :nested
	  :width 540 :height 540 :depth 14 :float-type 'short-float :real-origin -0.0068054s0 :imag-origin 0.255127s0
	  :range 2.1645s0 :julia-seed '(0.94162 . -1.9918699))
(defimage :geode
	  :width 540 :height 540 :depth 21 :float-type 'short-float :julia-seed '(0.77556s0 . 0.138695s0))
(defimage :twisty
	  :width 540 :height 540 :float-type 'short-float :contrast 18 :detail nil :julia-seed '(0.87692s0 . 0.173111s0))
(defimage :the-big-picture
	  :width 540 :height 540 :float-type 'short-float :real-origin -0.43846s0 :imag-origin 1.13075s0 :range 2.44617s0
	  :contrast 18 :detail nil)


;1;; An interface to the ScreenSaver.*
;1;; This is still kind of buggy - or maybe it's the ScreenSaver...*
;1;; Sometimes when the ScreenSaver tries to turn itself off, the Blackout window is still locked by the screenhack process, and the*
;1;; whole window system is wedged.  7Usually* 5Term* 5C-Clear-Input* wakes it up, but not always...*
;1;;*

(defvar 4*mand-screenhack-data* *nil "2The bitmap in which the Mandelbrot-Screenhack is working.*")
(defvar 4*mand-screenhack-cache* *nil "2The other bitmap in which the Mandelbrot-Screenhack is working.*")

(defun 4mandelbrot-screenhack *(&optional (window tv:selected-window))
  "2Pick some random numbers and generate an endless succession of Mandelbrot images on the given window.
  You can put this on the **SCREEN-SAVER-HACKS-LIST*2, but it's interaction with the ScreenSaver is still kind of flaky.
  Use at your own risk...*"
  (unless (and *mand-screenhack-data* *mand-screenhack-cache*)
    (let* ((width  (array-dimension (w:sheet-screen-array tv:default-screen) 1))
	   (height (array-dimension (w:sheet-screen-array tv:default-screen) 0))
	   (dd (list height width)))
      (unless *mand-screenhack-data*
	(setq *mand-screenhack-data* (make-array dd :element-type (array-element-type (tv:sheet-screen-array window)))))
      (unless *mand-screenhack-cache*
	(setq *mand-screenhack-cache* (make-array dd :element-type 'bit)))))
  (let* ((win-w (tv:sheet-inside-width window))
	 (win-h (tv:sheet-inside-height window)))
    (loop
      (send window :clear-screen)
      (let* ((r (- (random 4.0s0) 2.0s0))
	     (i (- (random 4.0s0) 2.0s0))
	     (max-w (- 4.0s0 (+ 2.0s0 (min r i))))
	     (w (random max-w))
	     (depth (+ 5 (random 25)))
	     (cd (if (zerop (random 2))
		     (+ (round depth 2) (random (round depth 2)))
		     nil))
	     (outline-only (and cd (zerop (random 5))))
	     (seed-p (zerop (random 5)))
	     (seed (and seed-p (cons (- (random 4.0s0) 2.0s0)
				     (- (random 4.0s0) 2.0s0))))
	     )
	(when (and cd (or (< cd (1- depth)) (< cd 3)))
	  (setq cd nil outline-only nil))
	(unwind-protect
	    (quad-tree-compute-image *mand-screenhack-data* *mand-screenhack-cache*
				     :window window :width win-w :height win-h
				     :real-origin r :imaginary-origin i :range w :depth depth
				     :contrast-distance cd :outline-only outline-only
				     :julia-seed seed)
	  (sys:without-interrupts
	    (fill *mand-screenhack-cache* 0)
	    (fill *mand-screenhack-data* 0)))))))

;(when (boundp 'tv:*screen-saver-hacks-list*)
;  (pushnew 'mandelbrot-screenhack tv:*screen-saver-hacks-list*))
