;;; -*- Mode:Lisp; Package:TV; Syntax:Common-Lisp -*-

;;; File "FIREWORKS"
;;; A fireworks display for the screensaver.
;;; Written and maintained by Jamie Zawinski.
;;;
;;; ChangeLog:
;;;
;;; 30 Dec 88  Jamie Zawinski    Created.
;;;


(defun draw-circle-points (window radius num-points x-center y-center alu color)
  "Draws NUM-POINTS pixels along the perimeter of a circle, but does not connect them."
  (let* ((delta-angle (quotient W:TWO-PI num-points))
	 (cos-delta   (cos delta-angle))
	 (sin-delta   (sin delta-angle))
	 (x radius)
	 (y 0.0))
    (do ((i 0 (1+ i)))
	((>= i num-points))
      (psetq x (- (* x cos-delta) (* y sin-delta))
	     y (+ (* x sin-delta) (* y cos-delta)))
      (send window :draw-point (round (+ x x-center)) (round (+ y y-center)) alu color))))



(defun boomboom (window x y total-radius num-points &optional (do-sound-p t))
  "Explodes a firework at XY with maximum radius of TOTAL-RADIUS.  If DO-SOUND-P is T, makes a fizzing noise as well."
  (let* ((delta-y (float (/ total-radius 100)))
	 (delta-delta-y (float (/ delta-y 50)))
	 (delta-radius (float (/ total-radius 50)))
	 (delta-delta-radius 0)
	 (lag (round total-radius 3))
	 (iters (+ total-radius lag 1))
	 (vol 15.0)
	 (sound-inc (float (/ 30 iters)))
	 (going-up t)
	 (color (if (tv:color-system-p window)
		    (nth (random 4) '#,(list w:white w:green w:yellow w:magenta ))
		    -1))
	 )
    (when do-sound-p
      (tv:reset-sound t)
      (tv:do-sound (tv:volume 0 :off))
      (tv:do-sound (tv:volume 1 :off))
      (tv:do-sound (tv:volume 2 :off))
      (tv:do-sound (tv:volume 3 0)))
    (unwind-protect
	(dotimes (i iters)
	  (when do-sound-p
	    (tv:do-sound (tv:volume 3 (max 0 (min 15 (round vol)))))
	    (when (<= vol 0) (setq going-up nil))
	    (if going-up
		(decf vol (* 1.5 sound-inc))
		(incf vol sound-inc)))
	  
	  (when (< i total-radius)
	    (let* ((radius (floor (* i (+ delta-radius (* delta-delta-radius i)))))
		   (cy (floor (+ y (* i (+ delta-y (* delta-delta-y i)))))))
	      (draw-circle-points window radius num-points x cy TV:ALU-TRANSP color)))
	  (when (>= i lag)
	    (let* ((i (- i lag))
		   (radius (floor (* i (+ delta-radius (* delta-delta-radius i)))))
		   (cy (floor (+ y (* i (+ delta-y (* delta-delta-y i)))))))
	      (draw-circle-points window radius num-points x cy TV:ALU-BACK color)))
	  )
      (when do-sound-p (tv:reset-sound nil))
      )))



(defun boom-for-lunch (window &optional (do-sound-p t))
  "Launches a pixel from the bottom of the screen and explodes a firework somewhere along the flight path.
  If DO-SOUND-P is T, makes a fizzing noise as well."
  
  (unless (zerop (tv:sib-sound-bit :query)) (setq do-sound-p nil))   ; don't do sound if someone else already is.
  (let* ((x (+ (round (tv:sheet-inside-width window) 4)
	       (random (round (tv:sheet-inside-width window) 2))))
	 (y (tv:sheet-inside-height window))
	 (x-neg-p (zerop (random 2)))
	 (delta-x (* (random 10) (if x-neg-p -1 1)))
	 (scale (+ 20 (random (round (tv:sheet-inside-height window) 20))))
	 (delta-y (- scale))
	 (color (if (tv:color-system-p window) W:RED -1))
	 (tone 128)
	 (tone-inc 2)
	 (erase (send window :erase-aluf))
	 )
    (when do-sound-p
      (tv:reset-sound t)
      (tv:do-sound (tv:volume 0 :off))
      (tv:do-sound (tv:volume 1 :off))
      (tv:do-sound (tv:volume 2 0))
      (tv:do-sound (tv:volume 3 :off)))
    (unwind-protect
	(loop
	  (when do-sound-p (tv:do-sound (tv:tone 2 (max 1 (decf tone tone-inc)))))
	  (sleep 0.05)
	  (send window :draw-point (round x)      (round y)      erase color)
	  (send window :draw-point (1+ (round x)) (round y)      erase color)
	  (send window :draw-point (round x)      (1+ (round y)) erase color)
	  (send window :draw-point (1+ (round x)) (1+ (round y)) erase color)
	  (when (> delta-y (round scale 2)) (return))
	  (incf x delta-x)
	  (incf y delta-y)
	  (incf delta-y (round scale 20))
	  (send window :draw-point (round x)      (round y)      TV:ALU-SETA color)
	  (send window :draw-point (1+ (round x)) (round y)      TV:ALU-SETA color)
	  (send window :draw-point (round x)      (1+ (round y)) TV:ALU-SETA color)
	  (send window :draw-point (1+ (round x)) (1+ (round y)) TV:ALU-SETA color)
	  )
      (when do-sound-p (tv:reset-sound nil))
      )
    (when do-sound-p (tv:do-sound (tv:tone 2 512)))
    (boomboom window x y (+ 20 (random 50)) 20 do-sound-p)))


(defvar *boomboom-sound-p* nil "Whether the fireworks screenhack should make noise.")

(defun boom-screenhack (window)
  (send window :clear-screen)
  (loop
    (boom-for-lunch window *boomboom-sound-p*)
    (sleep 1)))

(when (boundp 'tv:*screen-saver-hacks-list*)
  (pushnew 'BOOM-SCREENHACK tv:*screen-saver-hacks-list*))
