;;; -*- Mode:Common-Lisp; Package:NAME; Base:10 -*-
;;; 10-Nov-88 22:17:54, Bjrn Victor
;;; Copyright (c) 1988, Bjrn Victor, Department of Computer Systems, Uppsala University

;;; This file implements a flavor of basic-namespace that looks up DNS objects at read time.
;;; No TTL info is honored, so it should be rewritten.

;;; Some modifications by Jamie Zawinski, UC Berkeley. [jwz]

(defvar *dns-stub-resolver-enabled* t
  "Whether the stub resolver is enabled or not")

;;;**** Check this out
(defvar *use-original-resolver-search-algorithm* t
  "See comment before (:method name:dns-stub-namespace :find-object-internal)
Normally, sites should leave this non-NIL (I think)")

(defvar *dns-stub-negative-cache-timeout* (* 60 2)
  "Number of seconds no cache negative answers.
Should use DNS TTL instead.")

(define-site-variable *dns-stub-domain-search-list* :domain-search-list
  "Domain search list for stub resolver")

(define-site-variable *dns-stub-default-safety-belt-servers* :domain-safety-belt-servers
  "Default safety belt servers for stub resolver")

;;;
;;; [jwz] Initialize the necessary site variables at load-time.
;;;
(eval-when (load eval)
  (setq *dns-stub-domain-search-list* (or (get-site-option :domain-search-list) '("DoCS.UU.SE" "CSD.UU.SE")))
  (setq *dns-stub-default-safety-belt-servers* (or (get-site-option :domain-safety-belt-servers)
						   '("130.238.64.4" "130.238.64.49"))))

(defconstant *dns-stub-namespace-type* :domain
  "The namespace type of a DNS stub namespace")


;;; jwz: Everybody should have these services.  Even if they don't advertise them, try them anyway.
;;;
(setf (get :IP :default-network-services)
      '((:show-users :tcp-stream :ascii-name)
	(:login :tcp-stream :telnet)))

(defflavor dns-stub-namespace
	   ((safety-belt-servers nil)
	    (local-only-operation nil))
	   (basic-namespace)
  (:special-instance-variables local-only-operation)	;**** this doesn't work as I thought it would.
  :gettable-instance-variables
  (:inittable-instance-variables safety-belt-servers)
  (:settable-instance-variables safety-belt-servers)
  )

(add-namespace-flavor *dns-stub-namespace-type* 'dns-stub-namespace)

(defmethod (dns-stub-namespace :print-self) (stream &rest ignore)
  ;; JIC there are more than one - give me the pointer too
  (sys:printing-random-object (self stream :typep)
    (prin1 domain-name)))

(defmethod (dns-stub-namespace :after :init) (plist)
  (declare (ignore plist))
  (unless safety-belt-servers
    (setq safety-belt-servers *dns-stub-default-safety-belt-servers*)))

(defmethod (dns-stub-namespace :type) ()
  *dns-stub-namespace-type*)


;;; jwz: trivialized this method - the work now happens in (:method basic-namespace :wrapper :view-object)
(defmethod (dns-stub-namespace :find-object-internal) (name class)
  (with-stack-list* (key-space name class)
    (gethash key-space namespace)))

;;; jwz: added this method for more general handling of domain search lists.
(defwhopper (basic-namespace :view-object) (name class &rest args)
  "When looking up an object from the :HOST class, try appending/removing the
  local domain suffixes if the first try doesn't match."
  (or (lexpr-continue-whopper name class args)
      (and (eq class :HOST)
	   *dns-stub-resolver-enabled*
	   (not (boundp 'WITHIN-VIEW-OBJECT))
	   (let* ((dot (position #\. (the string name) :test #'char=))
		  (domain-supplied-p dot)
		  (local-domain-p (and domain-supplied-p
				       (member name *dns-stub-domain-search-list*
					       :test #'(lambda (x y)
							 (string-equal (the string x) (the string y) :start1 (1+ dot))))))
		  (WITHIN-VIEW-OBJECT t))
	     (declare (special WITHIN-VIEW-OBJECT))  ; So we can prune out some unnecessary recursion.
	     (cond (local-domain-p
		    ;; A domain was supplied, and it is a local domain.
		    ;; Try looking up the short-name of the host (the host without the local domain attached).
		    (let* ((short-name (subseq name 0 dot)))
		      (when #+ICU-DoCS (dbg:debug-p) #-ICU-DoCS dns:*debug-dns*
			(format *debug-io* "~&Trying ~S as ~S in namespace ~A..." name short-name self))
		      (lexpr-continue-whopper short-name class args)))
		   
		   (domain-supplied-p
		    ;; A domain was supplied, but it is not a local domain.
		    ;; There's nothing we can do.
		    nil)
		   (t
		    ;; No domain was supplied.
		    ;; Try appending each of the local domains in turn, and see if any of those match.
		    (dolist (search *dns-stub-domain-search-list*)
		      (let* ((new-name (string-append name "." search)))
			(when #+ICU-DoCS (dbg:debug-p) #-ICU-DoCS dns:*debug-dns*
			  (format *debug-io* "~&Trying ~S as ~S in namespace ~A..." name new-name self))
			(lexpr-continue-whopper new-name class args))))
		   )))))



;;; jwz: changed this to look up the safety-belt-servers if there aren't any stored on SELF.
;;;      When this is disk-saved, DNS-STUB-NAMESPACE-INITIALIZATION-FORM is apparently getting
;;;      called too early in the bootup process, and it's not getting set.
;;; jwz: made this not try to contact a nameserver except for :HOST objects.
;;;
(defmethod (dns-stub-namespace :find-object)
	   (name class)
  "Finds the actual object named NAME,CLASS in the namespace.  Returns 
   the actual database object or nil if the object does not exist.
   FOR INTERNAL USE BY THE NAMESPACE ONLY.  Ignores delete status.
In the DNS-STUB-NAMESPACE case, do a DNS lookup if the object isn't found, and save the result."
  (declare (special local-only-operation))
  (if (not (eq class :HOST))
      (send self :find-object-internal name class)
      (when *dns-stub-resolver-enabled*
	(unless safety-belt-servers			; [jwz] None?  Come on, there can't be none!  Try again.
	  (setf safety-belt-servers (get-site-option :domain-safety-belt-servers)))
	(let* ((name (string name))
	       (obj (send self :find-object-internal name class)))
	  (cond (local-only-operation		;If it's local-only, don't bother with network.
		 nil)
		((null obj)			;Never heard of it, try to find it
		 (setq obj (send self :contact-nameserver-find-object name class))
		 (when (null obj)		;Not found, cache a miss
		   (setq obj (send self :put-object (build-object name class nil)))
		   (send self :mark-object-updated obj)
		   (object-put-property obj :cache-miss t)
		   (setf (object-deleted obj) t)))
		((and (object-deleted obj)	;Cached a miss for this before?
		      (object-get-property obj :cache-miss))
		 (if (< (- (get-universal-time) (object-get-property obj :last-update))
			*dns-stub-negative-cache-timeout*)
		     obj			;Hasn't timed out yet.
		     ;;Cache miss timed out, try again to find it.
		     (when (null (setq obj (send self :contact-nameserver-find-object name class)))
		       ;;Cache a miss again.
		       (setq obj (send self :put-object (build-object name class nil)))
		       (send self :mark-object-updated obj)
		       (object-put-property obj :cache-miss t)
		       (setf (object-deleted obj) t)))))
	  
	  (when (and obj (null (object-get-property obj :last-update)))
	    (object-put-property obj :last-update 0))
	  obj))))



;;; jwz: Modified this to better deal with improperly-specified aliases in the nameserver.
;;;      Made it not bother talking to the nameserver about names without dots in them.
;;;
(defmethod (dns-stub-namespace :contact-nameserver-find-object) (name class)
  (declare (values internal-object))
  (declare (ignore class))
;  (when (and (eq class :host)
;	     (find #\. (string name) :test #'char=))	;[jwz] Don't talk to nameservers about unspecified-domain hosts.
  (progn						;[jwz] Wrong!!  Makes (parse-host "foo") not get "foo.local.domain".
    (let ((object (dns-stub-make-host-object name safety-belt-servers)))
      (when object
	(send self :mark-object-updated object)
	(send self :put-object object)
	;; Make alias objects
	(let ((aliases (get-attribute-value object :aliases)))
	  ;; jwz: If what DNS-STUB-MAKE-HOST-OBJECT returns has a different name than what we were trying to parse,
	  ;;      then what we were trying to parse is an alias (whether they have listed it as such or not).
	  ;;      For example, parsing "sgi.com" gives you "cypress.sgi.cs.net", but "cypress" doesn't list it as an alias!
	  ;;      The same is true of "csc.ti.com" and "ti.com".
	  (unless (or (string-equal name (object-name object))
		      ;; [jwz] for the "foo" -> "foo.local.domain" case.
		      (string-equal name (get-attribute-value object :short-name)))
	    (when #+ICU-DoCS (dbg:debug-p) #-ICU-DoCS dns:*debug-dns*
	      (cerror "Proceed, making ~s an alias of ~s"
		      "The domains ~s and ~s have fouled-up entries in the DNS.~
		       One of them should be an alias of the other, but they are entered~
		       as separate objects.  If you know what I'm talking about, you should~
		       write a letter to the domain administrators and ask them to fix it.~
		       If you don't want to or don't understand, just press <RESUME>"
		      name (object-name object)))
	    (pushnew name aliases :test #'string-equal))
	  (when aliases
	    (dolist (alias aliases)
	      ;; Build the alias object.
	      (let ((obj (build-object alias :host (list *alias-of* (object-name object)))))
		(send self :mark-object-updated obj)
		(send self :put-object obj))
	      ;;[jwz] Add the alias to the primary object's :ALIASES list.
	      (unless (member alias aliases :test #'string-equal)
		(push alias aliases))
	      (add-attribute (object-name object) :host :aliases aliases :namespace self)
	      ))))
      object)))

(defmethod (dns-stub-namespace :around :extract-objects-from-properties) (continuation mapping-table op-arglist
									  &optional name-pattern class first-only
									  property-list test format)
  "Return a subset of the objects in the namespace.  NAME-PATTERN can be the actual name of
   an object or a string containing wildcard characters.  CLASS restricts to objects of that class.
   FIRST-ONLY says return only the first object matching the criteria.  
   PROPERTY-LIST is a list of attributes and values required. 
   FORMAT is :BRIEF :LIST :READ-ONLY or :COPY.
   TEST can be a function which takes a candiate object, NAME-PATTERN, CLASS, and PROPERTY-LIST and
   returns T or NIL. " 
  (declare (ignore format first-only)
	   (special local-only-operation))

  ;;If this is a typical call from NET:GET-HOST-FROM-ADDRESS
  (if (and (eq test 'net:ns-get-host-from-address) (eq class :host)
	   (null name-pattern)
	   (= 4 (length property-list))
	   (and (eq (getf property-list :network) :ip)
		(not (null (getf property-list :address)))))
      (or
	;; first try the basic method
	(lexpr-funcall-with-mapping-table continuation mapping-table op-arglist)
	;; else look up the host from the address, and add it
	(unless local-only-operation		;...unless it's a local operation
	  (let ((object (dns-stub-lookup-host-object-from-address (getf property-list :address) safety-belt-servers)))
	    (when object
	      (send self :mark-object-updated object)
	      (send self :put-object object)
	      ;; Make alias objects
	      (let ((aliases (get-attribute-value object :aliases)))
		(when aliases
		  (dolist (alias aliases)
		    (let ((obj (build-object alias :host (list *alias-of* (object-name object)))))
		      (send self :mark-object-updated obj)
		      (send self :put-object obj)))))
	      ;; and use the basic method to find it again...
	      (lexpr-funcall-with-mapping-table continuation mapping-table op-arglist)))))
      ;; not our special case, call basic method.
      (lexpr-funcall-with-mapping-table continuation mapping-table op-arglist)))

;;; This is kind of a strange way of getting the local argument to these methods
;;; since the base methods have additional optional arguments used internally,
;;; it is hard to just add a local argument usable by a client who doesn't know
;;; what kind of namespace this is.  The default version of this method just passes
;;; on the message.
;;;
;;; In the DNS case, bind a special instance variable to make :FIND-OBJECT note
;;; that it shouldn't look up the object on the network.
(defmethod (dns-stub-namespace :local) (msg &rest args)
  (let ((local-only-operation t))
    (declare (special local-only-operation))
    (lexpr-send self msg args)))

(defmethod (dns-stub-namespace :around :find-or-synthesize) (continuation mapping-table op-arglist
							     NAME CLASS
							     &OPTIONAL (SYNTHESIZE-OK T)
						             A-NAME A-TYPE
							     MEMBER-NAME KEY
						             UNDELETE CHECK-TIME
							     LOOKUP-ATTR LOOKUP-MEMBER)
  (declare (ignore A-NAME A-TYPE MEMBER-NAME KEY UNDELETE CHECK-TIME LOOKUP-ATTR LOOKUP-MEMBER))
  (if synthesize-ok
      ;; Then call basic method, which calls our :find-object, which creates the object if necessary
      (lexpr-funcall-with-mapping-table continuation mapping-table op-arglist)
      (let ((obj
	      (if (find #\. (the string name) :test #'char=)
		  (with-stack-list* (key-space name class)
		    (gethash key-space namespace))
		  ;; Apply search-list when looking in hash table
		  (dolist (search *dns-stub-domain-search-list*)
		    (with-stack-list* (key-space (string-append name "." search) class)
		      (let ((x (gethash key-space namespace)))
			(when x (return x))))))))
	(if obj
	    ;; Then call basic method, which does the job, really.
	    (lexpr-funcall-with-mapping-table continuation mapping-table op-arglist)
	    ;; Else don't "synthesize" (lookup) the object.
	    nil))))


(defun dns-stub-lookup-host-object-from-address (address servers)
  (let ((names (condition-case-if (not #+ICU-DoCS (dbg:debug-p) #-ICU-DoCS dns:*debug-dns*) (cond)
		   (dns:stub-get-name-of-address address servers)
		 (error nil))))
    (when (and names (not (null (first names))))
      (let ((namestring (format nil "~{~a.~}" (first names))))
	(dns-stub-make-host-object namestring servers)))))

(defun dns-stub-make-host-object (host servers)
  (flet ((find-cname-and-addresses (spec)
	   (when #+ICU-DoCS (dbg:debug-p) #-ICU-DoCS dns:*debug-dns*
	      (format *debug-io* "~&Trying ~s..." spec))
	   (condition-case-if (not #+ICU-DoCS (dbg:debug-p) #-ICU-DoCS dns:*debug-dns*) (cond)
	       (values-list (append (multiple-value-list (dns:stub-canonicalize-case spec servers)) (list spec)))
	     (error nil))))

    (multiple-value-bind (cname addresses spec)
	(cond ((char= #\. (char host (1- (length host))))
	       (find-cname-and-addresses host))
	      ((find #\. host :test #'char=)
	       (find-cname-and-addresses (string-append host ".")))
	      (t
	       (loop with cname and addresses
		     for search in *dns-stub-domain-search-list*
		     as spec = (string-append host "." search ".")
		     do
		     (multiple-value-setq (cname addresses spec)
		       (find-cname-and-addresses spec))
		     when (and (not (null cname)) (not (null addresses)))
		     do (return (values cname addresses spec))
		     ;;Don't try top level - there are no top level hosts.
;		     finally
;		     (return
;		       (find-cname-and-addresses (string-append host ".")))
		     )))
      (when (or (null cname) (null addresses))
	(return-from dns-stub-make-host-object nil))
      (let* ((any (dns:stub-get-any-info-of-name cname servers))
	     (alias (and spec (nth-value 1 (dns:stub-get-canonical-name-of-name spec servers))))
	     (alias-name (and alias (format nil "~{~a~^.~}" alias)))
	     (hinfo (if (assoc :hinfo any)
			(first (cdr (assoc :hinfo any)))
			(dns:stub-get-host-info-of-name cname servers)))
	     (wks (if (assoc :wks any)
		      (cdr (assoc :wks any))
		      (dns:stub-get-wks-of-name cname servers)))
	     (mx (if (assoc :mx any)
		     (cdr (assoc :mx any))
		     (dns:stub-get-mail-exchange-of-name cname servers))))
	(stub-build-host-object-from-info
	  cname addresses :short-name (first cname) :alias-name alias-name :host-info hinfo :wks wks :mx mx)))))

;;; jwz: fixed this to deal with (:property :ip :default-network-services) better -
;;;      Now, even if the host has some services, the default services will be appended.
;;;
(defun stub-build-host-object-from-info (cname addresses &key short-name alias-name host-info wks mx)
  "CNAME is domain name (list of label strings).
ADDRESSES are numbers.
Short-name, alias-name are strings.
Host-info is a host-info object
WKS is a wks object.
mx is a mail-exchange object."

  (labels ((get-subnet-bits (addr)
	     (let* ((route (condition-case (cond)	; (it's silly to have #bits instead of a real mask!)
			       (ip:get-routing-entry addr)
			     (ip:incomplete-routing-table nil)))
		    (sub (if route
			     (logxor (ip:ip-routing-mask route) (ip:get-default-mask addr))
			     0)))
	       (if (zerop sub)
		   0
		   (- (integer-length sub)
		      (integer-length (logxor sub (1- (expt 2 (ceiling (log sub 2))))))))))
	   (dname-to-string (dn)
	     (format nil "~{~a~^.~}" dn))
	   ;; This doesn't implement what MXs are ment to implement, but it's something... and the Explorer uses it.
	   ;; Choose the best MX record which doesn't point at the host itself, since that is tried anyway.
	   (best-mx (mx-list)
	     (flet ((mx-sorter (a b)
		      (and (< (dns:mail-exchange-preference a)
			      (dns:mail-exchange-preference b))
			   ;; don't sort the host itself as a good one.
			   (not (equalp (dns:mail-exchange-exchange a)
					cname)))))
	       (let ((best-mx (first (sort mx-list #'mx-sorter))))
		 (unless (equalp (dns:mail-exchange-exchange best-mx) cname)
		   (dname-to-string (dns:mail-exchange-exchange best-mx)))))))

    (let ((name (dname-to-string cname))	;Get name
	  (address-strings (mapcar #'ethernet:dotted-format-ip-address addresses))
	  (mail-gateway (when mx (best-mx mx)))		;Get best MX as :mail-gateway
	  (subnet-bits nil)
	  (hinfo (canonicalize-hinfo host-info))	;Canonicalize
	  (services (make-services-from-ip-wks-records wks))	; and transform
	  )

      (when (and #+ICU-DoCS (dbg:debug-p) #-ICU-DoCS dns:*debug-dns*
		 (null wks))
	(format *debug-io* "~&DNS:  The name ~s hasn't got any WKS records~%" name))

      (when (> (length addresses) 1)		;Is this a gateway?
	(push '(:gateway :ip :ip-gateway) services)	; add gateway service
	(mapc #'(lambda (addr addr-string)	; and try to get subnet bits
		  (push (list addr-string (get-subnet-bits addr))
			subnet-bits))
	      addresses address-strings))
	    
      
      (let ((attrs `(:short-name ,short-name
		     ,@(when alias-name `(:aliases (,alias-name)))
		     :system-type ,(second hinfo)
		     :machine-type ,(first hinfo)
		     (:services :group)
;		     ((:status :tcp :ip-status) ,@(or services
;						      (get :ip :default-network-services)))
		     ;;[jwz] All objects have default services.
		     ((:status :tcp :ip-status) ,@(delete-duplicates (append services
									     (get :ip :default-network-services))))
		     (:addresses :group) ,(loop for addr in address-strings
						collect (list :ip addr))
		     ;; This should be plural and a group!
		     ,@(if mail-gateway `(:mail-gateway-host ,mail-gateway))
		     ;; This should be handled altogether different
		     ,@(if subnet-bits `((:ip-addr-subnet-bits :group) ,subnet-bits)))))
	(build-object name :host attrs)))))


;;; Tremendous!
;;; jwz: modified this to return :unknown for machine and system instead of NIL;
;;;      machine was defaulting to :LISPM in that case...
;;;
(defun canonicalize-hinfo (hinfo)
  (flet ((pseudo-regexp-search (x y)
	   (if (char= (char y 0) #\^)		;Yuck!
	       (string= y (string x) :start1 1 :end2 (- (length y) 1))
	       (search y (string x) :test #'char=))))
    (let* ((mach (ticl:selector (first hinfo)	;machine-type
				pseudo-regexp-search
		   ("^SYM" :symbolics-36xx)
		   ("VAX" :vax)
		   ("^SUN" :sun)
		   ("^MICROEXPLORER" :microexplorer)
		   ("EXPLORER" :explorer)
		   ("^DEC-10" :dec10)
		   ("^DEC-20" :dec20)
		   ("^XEROX" :xerox)
		   ("DN3000" :apollo)
		   ("DN4000" :apollo)
		   ("^MAC" :macintosh)
		   (t (or (first hinfo) :unknown))))
	   (sys
	     (ticl:selector (second hinfo) pseudo-regexp-search	;OS
	       ("UNIX" (case mach
			 ((:vax :sun :apollo) :unix-ucb)
			 (t :unix)))
	       ("BSD" :unix-ucb)
	       ("^TOPS-20" :tops20)
	       (t
		 (case mach			;hardwired system types
		   (:symbolics-36xx :symbolics)
		   (:explorer :explorer)
		   (:microexplorer :microexplorer)
		   ((:cadr :lambda) :lispm)
		   (:macintosh :mac-os)
		   (t (or (second hinfo) :unknown)))))))
      (list mach sys))))

(defun make-services-from-ip-wks-records (list-of-wkses)
  (let ((services ()))
    (dolist (wks list-of-wkses (delete-duplicates services :test #'equal))
      (let ((protocol (case (dns:wks-ip-protocol wks)
			(6 :tcp)
			(17 :udp))))
	(when protocol
	  (dolist (port (dns:wks-ip-ports wks))
	    (let ((implementation
		    (case port
		      (:users :lispm-finger)
		      ((:ftp :telnet :smtp :supdup) port)
		      (:time :time-simple-msb)
		      (:finger :ascii-name))))
	      (when implementation
		(let ((medium (case implementation
				((:lispm-finger :ftp) protocol)
				((:time-simple-msb :time-simple) (case protocol
								   (:tcp :tcp-simple)
								   (:udp :udp)))
				((:telnet :smtp :supdup :ascii-name) (case protocol
								       (:tcp :tcp-stream)
								       (:udp :udp-stream)))))
		      (service (case implementation
				 (:ascii-name :show-users)
				 (:lispm-finger :lispm-finger)
				 ((:telnet :supdup) :login)
				 (:smtp :mail-to-user)
				 (:ftp :file)
				 ((:time-simple-msb :time-simple) :time))))
		  (when (and medium service)
		    (push (list service medium implementation) services)))))))))))


;;;****************
(compile-flavor-methods dns-stub-namespace)
