;;; -*- Mode:Common-Lisp; Package:NAME; Base:10; Patch-file:T -*-
;;; 10-Nov-88 22:44:40, Bjrn Victor
;;; Reason: Patch to make ordinary namespace hosts possibly equivalent with DNS hosts.
;;; Copyright (c) 1988, Bjrn Victor, Department of Computer Systems, Uppsala University
;;;
;;; Additional patches made by Jamie Zawinski, UC Berkeley. [jwz]

(defvar *dns-stub-namespace-name* "DNS")
(defvar *dns-stub-namespace* nil)

(DEFUN dns-stub-namespace-initialization-form ()
  (if (null *dns-stub-namespace*)
      (setf *dns-stub-namespace* (name:new-namespace *dns-stub-namespace-name* *dns-stub-namespace-type*))
      (delete-namespace *dns-stub-namespace-name* nil))
  (setf (send *dns-stub-namespace* :safety-belt-servers) (get-site-option :domain-safety-belt-servers))
  (name:register-namespace *dns-stub-namespace* :end)
  (build-ns-search-list nil nil)
  *dns-stub-namespace*)

;;; jwz: we really have to work at it to get the DNS namespace to be last on the list, since
;;; some of the system code really works at it to keep the BOOT namespace last.
;;;
(compiler-let ((sys:compile-encapsulations-flag t))  ; Compile this advice.
  
  (sys:advise REGISTER-NAMESPACE :after 'DNS-on-end nil
    "Force the DNS namespace to be the absolute last on the namespace search list.
 This circumvents code which forces the Distribution namespace alwways be last."
    (when (and *dns-stub-namespace* *namespace-search-list*)
      (let* ((cons (member *dns-stub-namespace* *namespace-search-list* :test #'eq)))
	(unless (null (cdr cons))
	  (without-interrupts
	    (setq *namespace-search-list*
		  (delete *dns-stub-namespace* *namespace-search-list* :test #'eq :count 1))	; take it off
	    (nconc cons (cons *dns-stub-namespace* nil))					; put it on the end
	    )))))
  )

(add-initialization "Setup DNS Stub Namespace" '(dns-stub-namespace-initialization-form) '(:namespace))

(defun clear-dns-stub-namespace ()
  (when *dns-stub-namespace*
    (ticl:send *dns-stub-namespace* :flush)))

;;; Patch required for mailer, if you have ICU: Enhancements; MAIL-ADDRESS-CANONICAL-DOMAIN loaded.
#!C
; From file ADDRESS.LISP#> RELEASE-4.MAIL-DAEMON; JJ:
#10R MAIL#:
(COMPILER-LET ((*PACKAGE* (FIND-PACKAGE "MAIL"))
                          (SI:*LISP-MODE* :COMMON-LISP)
                          (*READTABLE* SYS:COMMON-LISP-READTABLE)
                          (SI:*READER-SYMBOL-SUBSTITUTIONS* SYS::*COMMON-LISP-SYMBOL-SUBSTITUTIONS*))
  (COMPILER#:PATCH-SOURCE-FILE "SYS: MAIL-DAEMON; ADDRESS.#"



(defun CANONICAL-DOMAIN (domain)

  (let* ((host (si:parse-host domain t))
	 (name (and host (send host :name))))
    (when host
      ;;[ICU,Vic] Find this host's mail domain
      (let ((mail-domain (and (not (find #\. name :test #'char=))	;Already has domain
			      ;;**************** this isn't very good 
			      ;;if you have namespace hosts in another domain, that don't have a :mail-domain attr.
			      (or #-(and Explorer RELEASE-6) (host-mail-domain host)
				  (first (local-mail-domains))))))
	(if mail-domain
	    (let ((full-name (string-append name #\. mail-domain)))
	      (when (not (equalp full-name domain))
		full-name))
	    (when (not (equalp name domain))
	      name))))))

))

;;; Patch to put DNS namespace second last on search list when rebuilding the search list.
;;; JWZ: Modified this to put it on the search-list LAST, *after* the BOOT namespace, so that
;;;      parsing a logical-host doesn't contact an Arpa nameserver, fail, and *then* get to BOOT.
#!C
; From file BOOT.LISP#> RELEASE-4.NAMESPACE; JJ:
#10R NAME#:
(COMPILER-LET ((*PACKAGE* (FIND-PACKAGE "NAME"))
                          (SI:*LISP-MODE* :COMMON-LISP)
                          (*READTABLE* SYS:COMMON-LISP-READTABLE)
                          (SI:*READER-SYMBOL-SUBSTITUTIONS* SYS::*COMMON-LISP-SYMBOL-SUBSTITUTIONS*))
  (COMPILER#:PATCH-SOURCE-FILE "SYS: NAMESPACE; BOOT.#"


(DEFUN BUILD-NS-SEARCH-LIST (&OPTIONAL SEARCH-LIST (BOOT-NEW-SERVERS T))
  
  (UNLESS SEARCH-LIST
    (let ((boot-ns (send net:local-host :domain)))
      (SETF SEARCH-LIST (if (typep (find-namespace boot-ns) 'symbolics-remote-namespace)
			    (or (GET-SITE-OPTION :NAMESPACE-SEARCH-LIST)
				(name:lookup-attribute-value boot-ns :namespace :search-rules)) ;this may return a string
			    (GET-SITE-OPTION :NAMESPACE-SEARCH-LIST)))
      (unless (consp search-list) (setf  search-list (list  search-list))) ;results must be a list 12-11-87 DAB
      ))
    
  (WHEN SEARCH-LIST
    (LET (NEW-SEARCH-LIST)
      (LOOP
	FOR NAMESPACE-NAME IN SEARCH-LIST
	WITH NS
	WITH ERROR-STRING 
	DO
	(SETQ NS NIL)				;on each loop iteration
	
        ;; Assume any of the namespaces on the new search list may already be known
        (CONDITION-CASE (COND)
	    (SETF NS (FIND-NAMESPACE NAMESPACE-NAME))
	  (ERROR (SETQ ERROR-STRING (SEND COND :REPORT-STRING))
		 (SETF NS NIL)))
	
        (IF NS 
	    (PUSH NS NEW-SEARCH-LIST)
	    ;; ELSE
	    (NOTIFY "Unable to place namespace ~A on new search list because: ~A" NAMESPACE-NAME ERROR-STRING)))
      
      ;; Make sure the network and distribution namespaces are there
      (LET ((NETWORK-NAMESPACE (ON-SEARCH-LIST *NET-DOMAIN*)))
	(WHEN (AND NETWORK-NAMESPACE (NOT (ON-SEARCH-LIST *NET-DOMAIN* NEW-SEARCH-LIST)))
	  (PUSH NETWORK-NAMESPACE NEW-SEARCH-LIST))
	(WHEN (AND *DISTRIBUTION-NAMESPACE* (NOT (ON-SEARCH-LIST *DISTRIBUTION-NAME* NEW-SEARCH-LIST)))
	  (PUSH *DISTRIBUTION-NAMESPACE* NEW-SEARCH-LIST))
	;;[Victor]**** Make sure DNS namespace is there too!
	;;[jwz] **** put it on LAST, not second-to-last - logical-hosts live in the BOOT namespace, and 
	;;           they should have higher priority than DNS hosts.
	(when (and *dns-stub-namespace* (not (on-search-list *dns-stub-namespace-name* new-search-list)))
	  (push *dns-stub-namespace* new-search-list))
	)
      
      (SETQ *NAMESPACE-SEARCH-LIST* (NREVERSE NEW-SEARCH-LIST))

      ;; If we are supposed to be a local server and don't know it, bootstrap us
      (LOOP
	FOR NS IN *NAMESPACE-SEARCH-LIST* 
   	WITH SERVERS 
	WITH NAMESPACE-NAME DO
         (CONDITION-CASE (COND)
	    (WHEN (AND  BOOT-NEW-SERVERS
		       *NAME-SUPERVISOR*
		       (NOT (FIND-SERVER (SETF NAMESPACE-NAME (SEND NS :DOMAIN-NAME))))
		       (SEND NS :SEND-IF-HANDLES :SERVER-QUERY (RELATIVE-MACHINE-NAME))
		       (SETF SERVERS (LOOKUP-ATTRIBUTE-VALUE NAMESPACE-NAME :NAMESPACE :SERVERS :NAMESPACE NS)))
	      (SEND *NAME-SUPERVISOR* :BACKGROUND :ADD-SERVER NAMESPACE-NAME SERVERS :BEGINNING :CLIENT-EXISTS))
	  (ERROR
	   (NOTIFY "Placing namespace ~A on new search list, but can't tell if local machine is a server because ~A"
		   NAMESPACE-NAME (SEND COND :REPORT-STRING)))))
      
      *NAMESPACE-SEARCH-LIST*)))

))

;;; Patch to make ordinary namespace hosts possibly equivalent with DNS hosts.
#!C
; From file HOSTS.LISP#> RELEASE-4.NETWORK-SUPPORT; JJ:
#10R NET#:
(COMPILER-LET ((*PACKAGE* (FIND-PACKAGE "NET"))
                          (SI:*LISP-MODE* :COMMON-LISP)
                          (*READTABLE* COMMON-LISP-READTABLE)
                          (SI:*READER-SYMBOL-SUBSTITUTIONS* SYS::*COMMON-LISP-SYMBOL-SUBSTITUTIONS*))
  (COMPILER#:PATCH-SOURCE-FILE "SYS: NETWORK-SUPPORT; HOSTS.#"


(defun host-equivalent-p (host1 host2)
  (setf host1 (net:parse-host host1 :no-error))
  (setf host2 (net:parse-host host2 :no-error))
  (cond ((and host1 host2 (eq host1 host2))
	 (return-from host-equivalent-p t))
	(t
	 (let* ((host-object1 (when host1 (name:lookup-object (send host1 :name) :host)))
		(host-object2 (when host2 (name:lookup-object (send host2 :name) :host)))
		(host1-alias  (when host-object1
				(net:parse-host
				  (name:get-attribute-value host-object1 name:*alias-of*)
				  :no-error)))
		(host2-alias  (when host-object2
				(net:parse-host
				  (name:get-attribute-value host-object2 name:*alias-of*)
				  :no-error))))
	   
	   (cond ((and host1-alias host2-alias)
		  (or (and (neq host2 host2-alias  )
			   (host-equivalent-p   host1        host2-alias))
		      (and (neq host1 host1-alias)
			   (host-equivalent-p   host1-alias  host2))
		      (and (neq host1 host1-alias  )
			   (neq host2 host2-alias  )
			   (host-equivalent-p   host1-alias  host2-alias))))
		 
		 ((and host1-alias (neq host1 host1-alias  ))
		  (host-equivalent-p host1-alias host2))
		 
		 ((and host2-alias (neq host2 host2-alias  ))
		  (host-equivalent-p host2-alias host1))
		 ;;[Vic 881207]  This is to test DNS hosts equivalence with NS hosts.
		 ;; If some class of addresses are equal, the hosts are deemed to be equivalent.
		 ((and host1 host2)
		  (let ((a1 (send host1 :addresses))
			(a2 (send host2 :addresses)))
		    (and a1 a2
			 (not (null (intersection a1 a2
				     :test #'(lambda (a b)
					       (and (eq (first a) (first b))
						    ;; Addresses are not ordered
						    (null (set-difference (rest a) (rest b)))))))))))
		 )))))

))



;;; Patch so that hosts whose system-type is :UNKNOWN default to using Unix-pathname format.
;;; If a host's system-type is :UNKNOWN, then that means there was an incomplete entry in the
;;; name-server's database; it's more likely that such a machine will be a Unix box than a Lispm...
;;;

(defmethod (net:host :case :pathname-flavor-internal :UNKNOWN) ()
  'fs:unix-ucb-pathname)

;;; Patch so that hosts whose system-type is specified, but is an unknown type default to using
;;; Unix-pathname format.  Given the ratio of Lispms/Unixes out there, this seems reasonable...

(defmethod (net:host :pathname-flavor-internal) (ignore)
  "Return the pathname flavor for a BSD Unix machine (default)"
  'fs:unix-ucb-pathname)

(defmethod (net:host :case :pathname-flavor-internal :LISPM) ()
  'fs:lispm-pathname)

(defmethod (net:host :case :pathname-flavor-internal :EXPLORER) ()
  'fs:lispm-pathname)

;;; Make FTP realize that if the system-type is, say, ULTRIX, the FTP parsing should happen
;;; in exactly the same way as if the system-type was UNIX-UCB.  System names were taken from
;;; RFC 1060.
;;;

(defvar fs:*unix-ucb-system-types*
	'(:UNKNOWN :AEGIS :APOLLO :IRIX :MULTICS :PERQ/OS :SCO-XENIX/386 :SUN :SUN-OS-3.5 :SUN-OS-4.0 :ULTRIX :UNIX
	  :UNIX-BSD :UNIX-PC :UNIX-UCB :XENIX)
  "Those system types which are, for FTP purposes, the same as UNIX-UCB.")

;;; ######  I don't know what these are.  Some of them should probably be on the above list.
;;;  
;;;  :BS-2000 :CEDAR :SCS :SIMP :CGW :CHORUS :CHRYSALIS :CMOS :CMS :COS :CPIX :CTOS :CTSS :DCN :DDNOS :DOMAIN :EDX :ELF
;;;  :EMBOS :EMMOS :EPOS :FOONEX :FUZZ :GCOS :GPOS :HDOS :IOS :ISI-68020 :LOCUS :ACOS :MINOS :MOS :MPE5 :MVS :MVS/SP :NEXUS
;;;  :NMS :NONSTOP :NOS-2 :OS/DDP :OS4 :OS86 :OSX :PLI :PRIMOS :PSDOS/MIT :RMX/RDOS :ROS :RSX11M :SATOPS :SWIFT :TAC :TANDEM
;;;  :TENEX :TOPS10 :TOPS20 :TOS :TP3010 :TRSDOS :WAITS :WANG :XDE


(defun fs:ftp-canonicalize-system-type (system-type)
  (if (member (the keyword system-type) (the list fs:*unix-ucb-system-types*) :test #'eq)
      :unix
      system-type))
