;;; -*- Mode:Common-Lisp; Package:DOMAIN-NAME-SYSTEM; Base:10 -*-
;;; 20-Mar-88 12:31:15, Bjrn Victor
;;; Copyright (c) 1988, Bjrn Victor, Department of Computer Systems, Uppsala University

;;; This could use PARSE-BODY and do all kinds of tricks.  This should be enough, though.
(defmacro defsubst (name args &body body)
  "Among other things, this automagically tries to define a SETF method, and proclaims the accessor and updater INLINE."
  (unless (= 1 (length body))
    (error "This simple DEFSUBST only handles simple bodies"))
  (let ((set-fun (intern (concatenate 'string "%SET-" (string name)) (symbol-package name))))
    (let* ((val-var (gensym))
	   (set-args (append args (list val-var)))
	   (set-body (if (member (caar body) '(let let*) :test #'eq)
			 `(,(first (car body)) ,(second (car body))
			   (setf ,(third (car body)) ,val-var))
			 `(setf ,@body ,val-var))))
      `(progn
	 (defun ,name ,args
	   ,@body)
	 (defun ,set-fun ,set-args
	   ,set-body)
	 (defsetf ,name ,set-fun)
	 (proclaim '(inline ,name ,set-fun))
	 ',name))))

;;; Accessors for octet packets
(defsubst 8b-ref (pkt n)
  (aref pkt n))

(defsubst 1b-ref (pkt n bit)
  (ldb (byte 1 bit) (8b-ref pkt n)))

(defun boole-ref (pkt n bit)
  (= 1 (1b-ref pkt n bit)))
(defun %set-boole-ref (pkt n bit value)
  (unless (integerp value)
    (setq value (case value ((t) 1) (nil 0))))
  (%set-1b-ref pkt n bit value))
(defsetf boole-ref %set-boole-ref)

(defun 16b-ref (pkt n)
  (dpb (aref pkt n) (byte 8 8) (aref pkt (1+ n))))
(defun %set-16b-ref (pkt n val)
  (setf (aref pkt n) (ldb (byte 8 8) val)
	(aref pkt (1+ n)) (ldb (byte 8 0) val))
  val)
(defsetf 16b-ref %set-16b-ref)

(defun 32b-ref (pkt n)
  (dpb (16b-ref pkt n) (byte 16 16) (16b-ref pkt (+ 2 n))))
(defun %set-32b-ref (pkt n val)
  (setf (16b-ref pkt n) (ldb (byte 16 16) val)
	(16b-ref pkt (+ 2 n)) (ldb (byte 16 0) val))
  val)
(defsetf 32b-ref %set-32b-ref)

;;; Constructors
(defun make-32b (data)
  (list (ldb (byte 8 24) data) (ldb (byte 8 16) data) (ldb (byte 8 8) data) (ldb (byte 8 0) data)))

(defun make-16b (data)
  (list (ldb (byte 8 8) data) (ldb (byte 8 0) data)))

(defun make-8b (data)
  (list (ldb (byte 8 0) data)))


;;; Accessors for raw header fields

;; Can't use this for obvious reasons
;(defstruct raw-header
;  ((id (byte 16 16) 0)
;   (response-p (byte 1 15) 0)
;   (opcode (byte 4 11) 0)
;   (authoritative-p (byte 1 10) 0)
;   (truncation-p (byte 1 9) 0)
;   (recursion-desired (byte 1 8) 0)
;   (recursion-avaliable (byte 1 7) 0)
;   (zero (byte 3 4) 0)
;   (response-code (byte 4 0) 0))
;  ((question-count (byte 16 16) 0)
;   (answer-count (byte 16 0) 0))
;  ((authority-count (byte 16 16) 0)
;   (additional-count (byte 16 0) 0)))

(defsubst raw-header-id (m)
  (16b-ref m 0))
(defsubst raw-header-response-p (m)
  (boole-ref m 2 7))
(defsubst raw-header-opcode (m)
  (ldb (byte 4 3) (8b-ref m 2)))
(defsubst raw-header-authoritative-p (m)
  (boole-ref m 2 2))
(defsubst raw-header-truncated-p (m)
  (boole-ref m 2 1))
(defsubst raw-header-recursion-desired (m)
  (boole-ref m 2 0))
(defsubst raw-header-recursion-available (m)
  (boole-ref m 3 7))
(defsubst raw-header-zero (m)
  (ldb (byte 3 4) (8b-ref m 3)))
(defsubst raw-header-response-code (m)
  (ldb (byte 4 0) (8b-ref m 3)))
(defsubst raw-header-question-count (m)
  (16b-ref m 4))
(defsubst raw-header-answer-count (m)
  (16b-ref m 6))
(defsubst raw-header-authority-count (m)
  (16b-ref m 8))
(defsubst raw-header-additional-count (m)
  (16b-ref m 10))


;;;; "Data types"

;;; <character-string>
(defun get-character-string (pkt start-index)
  (let ((len (aref pkt start-index)))
    (coerce (subseq pkt (1+ start-index) (+ 1 start-index len)) 'string)))

(defun parse-character-string (string)
  (concatenate '(vector (unsigned-byte 8)) (list (length string)) string))

;;; <domain-name>
;;; These  handle parsed domains as lists, to avoid confusion
;;; with domain names like "XX\.LCS.MIT.EDU"  "XX.LCS.MIT.EDU" when parsing and unparsing.

(defun get-domain-name (pkt start-index)
  (do* ((i start-index)
	(slen (aref pkt i) (aref pkt i))
	(domain nil))
       ((zerop slen)
	(nreverse domain))
    (cond ((< slen 64)
	   (push (coerce (subseq pkt (+ i 1) (+ i 1 slen)) 'string) domain)
	   (incf i (1+ slen)))
	  ((= (ldb (byte 2 6) slen) #b11)	;Pointer
	   (let ((offset (dpb 0 (byte 2 14) (16b-ref pkt i))))
	     (let ((domain-name (get-domain-name pkt offset)))
	       (return (append (nreverse domain) domain-name))
	       )))
	  (t (error "Undefined length field in label: ~b" slen)))))

(defun %set-domain-name (pkt start-index new-value-as-string-or-list)
  (multiple-value-bind (domain length)
      (parse-domain-name new-value-as-string-or-list)
    (setf (subseq pkt start-index (+ start-index length)) domain)
    new-value-as-string-or-list))

(defsetf get-domain-name %set-domain-name)

(defun parse-domain-name (string-or-list)
  "Parse a domain name to \"binary\" form, suitable for a low-level packet."
  (etypecase string-or-list
    (string (parse-domain string-or-list))
    (list (parse-domain-list string-or-list))
    (vector (values string-or-list (fill-pointer string-or-list)))))

(defun parse-domain (string)
  (declare (values vector length))
  (let ((domain (make-array 64 :element-type '(unsigned-byte 8) :fill-pointer 0)))
    (do* ((s 0 (1+ e))
	  ;; Handle escaped dots.  This must be possible in a simpler way.
	  (e (let ((i 0))
	       (do ((dot (position #\. string :test #'char= :start i)))
		   ((if (and dot (plusp dot))
			(not (char= #\\ (char string (1- dot))))
			t)
		    dot)
		 (setq i (1+ dot))))
	     (let ((i s))
	       (do ((dot (position #\. string :test #'char= :start i)))
		   ((if (and dot (plusp dot))
			(not (char= #\\ (char string (1- dot))))
			t)
		    dot)
		 (setq i (1+ dot))))))
	((null e)
	 (let* ((end-domain (subseq string s e))
		(l (length end-domain))
		(f (fill-pointer domain)))
	   (unless (zerop l)
	     (warn "~&>>Warning: Domain name doesn't end with a \".\""))
	   (when (>= l 64)
	     (error "Label length exceeds 63: ~s" end-domain))
	   (vector-push-extend l domain)
	   (dotimes (i l)
	     (vector-push-extend (char string (+ i f)) domain))
	   (values domain (fill-pointer domain))))
      (let* ((subdomain (subseq string s e))
	     (l (length subdomain))
	     (f (fill-pointer domain)))
	(when (>= l 64)
	  (error "Label length exceeds 63: ~s" subdomain))
	(unless (zerop l)
	  (vector-push-extend l domain))
	(dotimes (i l)
	  (vector-push-extend (char string (+ i f)) domain))
	))))

(defun parse-domain-list (list)
  (declare (values vector length))
  (let ((domain (make-array 64 :element-type '(unsigned-byte 8) :fill-pointer 0)))
    (dolist (dom list)
      (let ((l (length dom)))
	(vector-push-extend l domain)
	(dotimes (i l)
	  (vector-push-extend (elt dom i) domain))))
    (vector-push-extend 0 domain)
    (values domain (fill-pointer domain))))

;;; Question section accessors
;; No error checking
(defun domain-name-field-length (pkt index)
  (let ((byte (8b-ref pkt index)))
    (cond ((zerop byte)
	   1)
	  ((< byte 64)
	   (+ 1 byte (domain-name-field-length pkt (+ index 1 byte))))
	  ((= #b11 (ldb (byte 2 6) byte))
	   2)
	  (t (error "Undefined label length field: ~b" byte)))))

(defun compute-query-start-index (pkt query-index)
  (if (zerop query-index)
      12
      (+ 12 (let ((total 0)
		  (n query-index))
	      (dotimes (i n total)
		(let ((name-len (domain-name-field-length pkt (+ 12 total))))
		  (incf total (+ name-len 4))))))))

(defsubst query-name (m query-index)
  (let ((start-index (compute-query-start-index m query-index)))
    (get-domain-name m start-index)))
(defsubst query-type (m query-index)
  (let* ((start-index (compute-query-start-index m query-index))
	 (name-len (domain-name-field-length m start-index)))
    (16b-ref m (+ start-index name-len))))
(defsubst query-class (m query-index)
  (let* ((start-index (compute-query-start-index m query-index))
	 (name-len (domain-name-field-length m start-index)))
    (16b-ref m (+ start-index name-len 2))))


;;; Resource records
(defun compute-rr-start-index (pkt rr-type type-index)
  "RR-TYPE is a keyword, TYPE-INDEX is the NTH block of that type"
  (let* ((nqueries (raw-header-question-count pkt))
	 (nanswers (raw-header-answer-count pkt))
	 (nauthority (raw-header-authority-count pkt))
	 (naddinfo (raw-header-additional-count pkt))
	 (base (ecase rr-type
		 (:answer (compute-query-start-index pkt nqueries))
		 (:authority (compute-rr-start-index pkt :answer nanswers))
		 (:additional (compute-rr-start-index pkt :authority nauthority)))))
;    (let ((max (1- (ecase rr-type (:answer nanswers) (:authority nauthority) (:additional naddinfo)))))
;      (when (> type-index max)
;	(error "RR index too large: ~d > ~d" type-index max)))
    (if (zerop type-index)
	base
	(+ base (let ((total 0))
		  (dotimes (n type-index total)
		    (let ((name-len (domain-name-field-length pkt (+ base total))))
		      ;; <Domain name length>+<type len(2)>+<class len(2)>+<ttl len(4)>+<datalen len(2)>+<data len>
		      (incf total (+ name-len 10 (resource-data-length pkt (+ base total)))))))))))

(defsubst resource-name (m rr-start-index)
  (get-domain-name m rr-start-index))
(defsubst resource-type (m rr-start-index)
  (let ((index (+ rr-start-index (domain-name-field-length m rr-start-index))))
    (16b-ref m index)))
(defsubst resource-class (m rr-start-index)
  (let ((index (+ rr-start-index (domain-name-field-length m rr-start-index) 2)))
    (16b-ref m index)))
(defsubst resource-time-to-live (m rr-start-index)
  (let ((index (+ rr-start-index (domain-name-field-length m rr-start-index) 2 2)))
    (32b-ref m index)))
(defsubst resource-data-length (m rr-start-index)
  (let ((index (+ rr-start-index (domain-name-field-length m rr-start-index) 2 2 4)))
    (16b-ref m index)))
(defsubst resource-data-index (m rr-start-index)
  (let ((index (+ rr-start-index (domain-name-field-length m rr-start-index) 2 2 4 2)))
    index))

;;; Answer section

(defsubst answer-name (m answer-index)
  (let ((start-index (compute-rr-start-index m :answer answer-index)))
    (get-domain-name m start-index)))
(defsubst answer-type (m answer-index)
  (let ((start-index (compute-rr-start-index m :answer answer-index)))
    (resource-type m start-index)))
(defsubst answer-class (m answer-index)
  (let ((start-index (compute-rr-start-index m :answer answer-index)))
    (resource-class m start-index)))
(defsubst answer-time-to-live (m answer-index)
  (let ((start-index (compute-rr-start-index m :answer answer-index)))
    (resource-time-to-live m start-index)))
(defsubst answer-data-length (m answer-index)
  (let ((start-index (compute-rr-start-index m :answer answer-index)))
    (resource-data-length m start-index)))
(defsubst answer-data-index (m answer-index)
  (let ((start-index (compute-rr-start-index m :answer answer-index)))
    (resource-data-index m start-index)))

;;; Authority section

(defsubst authority-name (m authority-index)
  (let ((start-index (compute-rr-start-index m :authority authority-index)))
    (get-domain-name m start-index)))
(defsubst authority-type (m authority-index)
  (let ((start-index (compute-rr-start-index m :authority authority-index)))
    (resource-type m start-index)))
(defsubst authority-class (m authority-index)
  (let ((start-index (compute-rr-start-index m :authority authority-index)))
    (resource-class m start-index)))
(defsubst authority-time-to-live (m authority-index)
  (let ((start-index (compute-rr-start-index m :authority authority-index)))
    (resource-time-to-live m start-index)))
(defsubst authority-data-length (m authority-index)
  (let ((start-index (compute-rr-start-index m :authority authority-index)))
    (resource-data-length m start-index)))
(defsubst authority-data-index (m authority-index)
  (let ((start-index (compute-rr-start-index m :authority authority-index)))
    (resource-data-index m start-index)))

;;; Additional section

(defsubst additional-name (m additional-index)
  (let ((start-index (compute-rr-start-index m :additional additional-index)))
    (get-domain-name m start-index)))
(defsubst additional-type (m additional-index)
  (let ((start-index (compute-rr-start-index m :additional additional-index)))
    (resource-type m start-index)))
(defsubst additional-class (m additional-index)
  (let ((start-index (compute-rr-start-index m :additional additional-index)))
    (resource-class m start-index)))
(defsubst additional-time-to-live (m additional-index)
  (let ((start-index (compute-rr-start-index m :additional additional-index)))
    (resource-time-to-live m start-index)))
(defsubst additional-data-length (m additional-index)
  (let ((start-index (compute-rr-start-index m :additional additional-index)))
    (resource-data-length m start-index)))
(defsubst additional-data-index (m additional-index)
  (let ((start-index (compute-rr-start-index m :additional additional-index)))
    (resource-data-index m start-index)))

;;;; Higher-level definitions, parsed/created in HIGH

(defstruct message
  header questions answers authorities additionals)

(defstruct message-header
  (id 0 :type (unsigned-byte 16))
  (response-p nil :type symbol)
  (opcode -1 :type (unsigned-byte 4))
  (authoritative-p nil :type symbol)
  (truncated-p nil :type symbol)
  (recursion-desired nil :type symbol)
  (recursion-available nil :type symbol)
  (zero 0)
  (response-code *response-no-error*)
  (question-count 0) (answer-count 0) (authority-count 0) (additional-count 0))

(defstruct message-question
  name type class)

(defstruct message-resource-record
  name type class time-to-live data
  authoritative-p				;**** Addition.  Should I really have it here?
  )

;;; Resource record data representation

;; Domains are represented as lists of labels (strings).
;; NOTE that this is the same format as *type-text* records, so you'd better check the type!

;; Chaos addresses

(defstruct chaos-address
  (domain "" :type list)
  (address 0 :type (unsigned-byte 16)))

;; Host info
(defstruct (host-info-resource-data (:conc-name host-info-))
  (cpu "" :type string)
  (os "" :type string))

;; Mail info
(defstruct (mail-info-resource-data (:conc-name mail-info-))
  (responsible nil :type list)
  (errors-to nil :type list))

;; Mail exchange
(defstruct (mail-exchange-resource-data (:conc-name mail-exchange-))
  (preference 0 :type (unsigned-byte 16))
  (exchange nil :type list))

;; Start of Authority
(defstruct (start-of-authority-resource-data (:conc-name soa-))
  (master nil :type list)
  (responsible nil :type list)
  (serial-number 0 :type (unsigned-byte 32))
  (refresh-interval 0 :type (unsigned-byte 32))
  (retry-interval 0 :type (unsigned-byte 32))
  (expire-interval 0 :type (unsigned-byte 32))
  (minimum-ttl 0 :type (unsigned-byte 32)))

;; WKS for IP
(defstruct (well-known-services-ip-resource-data (:conc-name wks-ip-))
  (address 0 :type (unsigned-byte 32))
  (protocol 0 :type (unsigned-byte 8))
  (ports nil))				;Suggestions for representation?

(defstruct (well-known-services-chaos-resource-data (:conc-name wks-chaos-))
  (contacts nil))


;;;; Debug

(defun describe-dns-packet (pkt &optional (stream *standard-output*))
  (format stream "~&Header:~%~@3T~~a~"
	  (describe-dns-packet-header pkt nil))
  (unless (zerop (raw-header-question-count pkt))
    (format stream "~&Questions:~%~@3T~~a~"
	    (describe-dns-question-section pkt nil)))
  (unless (zerop (raw-header-answer-count pkt))
    (format stream "~&Answers:~%~@3T~~a~"
	    (describe-dns-answer-section pkt nil)))
  (unless (zerop (raw-header-authority-count pkt))
    (format stream "~&Authority:~%~@3T~~a~"
	    (describe-dns-authority-section pkt nil)))
  (unless (zerop (raw-header-additional-count pkt))
    (format stream "~&Additional information:~%~@3T~~a~"
	    (describe-dns-additional-section pkt nil)))
  pkt)


(defun describe-dns-packet-header (pkt &optional (stream *standard-output*))
  (or 
    (format stream "~&ID: #x~x (~:*~d.)~@
		      Response: ~:[No~;Yes~]~@
		      Opcode: ~a~@
		      ~@[Authoritative;~* ~]~@[Truncated;~* ~]~@[Recursion Desired;~* ~]~@[Recursion Avaliable~*~]~&~
		      Resonse code: ~a~@
		      Queries: ~d.  Answers: ~d.  Authoritative: ~d.  Additional: ~d."
	    (raw-header-id pkt)
	    (raw-header-response-p pkt)
	    (opcode-string (raw-header-opcode pkt))
	    (raw-header-authoritative-p pkt) (raw-header-truncated-p pkt)
	    (raw-header-recursion-desired pkt) (raw-header-recursion-available pkt)
	    (response-code-string (raw-header-response-code pkt))
	    (raw-header-question-count pkt)
	    (raw-header-answer-count pkt)
	    (raw-header-authority-count pkt)
	    (raw-header-additional-count pkt)
	    )
    pkt))

(defun describe-dns-question-section (pkt &optional (stream *standard-output*))
  (let ((n (raw-header-question-count pkt)))
    (if (null stream)
	(with-output-to-string (s)
	  (dotimes (i n)
	    (format s "~&Name: ~s" (query-name pkt i))
	    (format s "~%Type: ~a" (type-string (query-type pkt i)))
	    (format s "~%Class: ~a" (class-string (query-class pkt i)))))
	(dotimes (i n pkt)
	  (format stream "~&Name: ~s" (query-name pkt i))
	  (format stream "~%Type: ~a" (type-string (query-type pkt i)))
	  (format stream "~%Class: ~a" (class-string (query-class pkt i)))))))

(defun describe-dns-answer-section (pkt &optional (stream *standard-output*))
  (let ((count (raw-header-answer-count pkt)))
    (if (null stream)
	(with-output-to-string (stream)
	  (dotimes (i count)
	    (format stream "~&Name: ~s" (answer-name pkt i))
	    (format stream "~%Type: ~a" (type-string (answer-type pkt i)))
	    (format stream "~%Class: ~a" (class-string (answer-class pkt i)))
	    (format stream "~%TTL: ~d" (answer-time-to-live pkt i))
	    (let ((len (answer-data-length pkt i))
		  (idx (answer-data-index pkt i)))
	      (format stream "~%Data length: ~d" len)
	      (format stream "~%Data: ~s"
		      (decode-data-for-describe pkt idx len (answer-type pkt i) (answer-class pkt i))))))
	(dotimes (i count pkt)
	  (format stream "~&Name: ~s" (answer-name pkt i))
	  (format stream "~%Type: ~a" (type-string (answer-type pkt i)))
	  (format stream "~%Class: ~a" (class-string (answer-class pkt i)))
	  (format stream "~%TTL: ~d" (answer-time-to-live pkt i))
	  (let ((len (answer-data-length pkt i))
		(idx (answer-data-index pkt i)))
	    (format stream "~%Data length: ~d" len)
	    (format stream "~%Data: ~s"
		    (decode-data-for-describe pkt idx len (answer-type pkt i) (answer-class pkt i))))))))

(defun describe-dns-authority-section (pkt &optional (stream *standard-output*))
  (let ((count (raw-header-authority-count pkt)))
    (if (null stream)
	(with-output-to-string (stream)
	  (dotimes (i count)
	    (format stream "~&Name: ~s" (authority-name pkt i))
	    (format stream "~%Type: ~a" (type-string (authority-type pkt i)))
	    (format stream "~%Class: ~a" (class-string (authority-class pkt i)))
	    (format stream "~%TTL: ~d" (authority-time-to-live pkt i))
	    (let ((len (authority-data-length pkt i))
		  (idx (authority-data-index pkt i)))
	      (format stream "~%Data length: ~d" len)
	      (format stream "~%Data: ~s"
		      (decode-data-for-describe pkt idx len (authority-type pkt i) (authority-class pkt i))))))
	(dotimes (i count pkt)
	  (format stream "~&Name: ~s" (authority-name pkt i))
	  (format stream "~%Type: ~a" (type-string (authority-type pkt i)))
	  (format stream "~%Class: ~a" (class-string (authority-class pkt i)))
	  (format stream "~%TTL: ~d" (authority-time-to-live pkt i))
	  (let ((len (authority-data-length pkt i))
		(idx (authority-data-index pkt i)))
	    (format stream "~%Data length: ~d" len)
	    (format stream "~%Data: ~s"
		    (decode-data-for-describe pkt idx len (authority-type pkt i) (authority-class pkt i))))))))

(defun describe-dns-additional-section (pkt &optional (stream *standard-output*))
  (let ((count (raw-header-additional-count pkt)))
    (if (null stream)
	(with-output-to-string (stream)
	  (dotimes (i count)
	    (format stream "~&Name: ~s" (additional-name pkt i))
	    (format stream "~%Type: ~a" (type-string (additional-type pkt i)))
	    (format stream "~%Class: ~a" (class-string (additional-class pkt i)))
	    (format stream "~%TTL: ~d" (additional-time-to-live pkt i))
	    (let ((len (additional-data-length pkt i))
		  (idx (additional-data-index pkt i)))
	      (format stream "~%Data length: ~d" len)
	      (format stream "~%Data: ~s"
		      (decode-data-for-describe pkt idx len (additional-type pkt i) (additional-class pkt i))))))
	(dotimes (i count pkt)
	  (format stream "~&Name: ~s" (additional-name pkt i))
	  (format stream "~%Type: ~a" (type-string (additional-type pkt i)))
	  (format stream "~%Class: ~a" (class-string (additional-class pkt i)))
	  (format stream "~%TTL: ~d" (additional-time-to-live pkt i))
	  (let ((len (additional-data-length pkt i))
		(idx (additional-data-index pkt i)))
	    (format stream "~%Data length: ~d" len)
	    (format stream "~%Data: ~s"
		    (decode-data-for-describe pkt idx len (additional-type pkt i) (additional-class pkt i))))))))

(defun decode-data-for-describe (pkt start len type class)
  (cond ((= type *type-canonical-name*)
	 (get-domain-name pkt start))
	((= type *type-host-info*)
	 (cons (get-character-string pkt start)
	       (get-character-string pkt (+ start (8b-ref pkt start) 1))))
	((= type *type-mailbox*)
	 (get-domain-name pkt start))
	((= type *type-mail-destination*)
	 (get-domain-name pkt start))
	((= type *type-mail-forwarder*)
	 (get-domain-name pkt start))
	((= type *type-mail-group*)
	 (get-domain-name pkt start))
	((= type *type-mail-info*)
	 (let ((resp-dom-len (domain-name-field-length pkt start)))
	   (cons (get-domain-name pkt start)
		 (get-domain-name pkt (+ start resp-dom-len)))))
	((= type *type-mail-rename*)
	 (get-domain-name pkt start))
	((= type *type-mail-exchange*)
	 (cons (16b-ref pkt start)
	       (get-domain-name pkt (+ start 2))))
	((= type *type-null*)
	 (subseq pkt start (+ start len)))
	((= type *type-name-server*)
	 (get-domain-name pkt start))
	((= type *type-pointer*)
	 (get-domain-name pkt start))
	((= type *type-start-of-authority*)
	 (let* ((mname-len (domain-name-field-length pkt start))
		(rname-len (domain-name-field-length pkt (+ start mname-len))))
	   (list (get-domain-name pkt start)	;MNAME
		 (get-domain-name pkt (+ start mname-len))	;RNAME
		 (32b-ref pkt (+ start mname-len rname-len))	;SERIAL
		 (32b-ref pkt (+ start mname-len rname-len 4))	;REFRESH
		 (32b-ref pkt (+ start mname-len rname-len 4 4))	;RETRY
		 (32b-ref pkt (+ start mname-len rname-len 4 4 4))	;EXPIRE
		 (32b-ref pkt (+ start mname-len rname-len 4 4 4 4))	;MINIMUM
		 )))
	((= type *type-text*)
	 (do ((data nil)
	      (s start (+ s 1 (8b-ref pkt s)))
	      (e (+ start len)))
	     ((>= s e) (nreverse data))
	   (push (get-character-string pkt s) data)))
	((and (= type *type-address*) (or (= class *class-internet*) (= class *class-chaos*)))
	 (cond ((= class *class-internet*)
		(format nil "~d.~d.~d.~d"
			(8b-ref pkt start) (8b-ref pkt (+ start 1)) (8b-ref pkt (+ start 2)) (8b-ref pkt (+ start 3))))
	       ((= class *class-chaos*)
		(let ((dname-len (domain-name-field-length pkt start)))
		  (format nil "~a|~o"
			  (get-domain-name pkt start)
			  (16b-ref pkt (+ start dname-len)))))))
	((and (= type *type-well-known-service*)
	      (or (= class *class-internet*) (= class *class-chaos*)))
	 (cond ((= class *class-internet*)
		(list (format nil "~d.~d.~d.~d"
			      (8b-ref pkt start) (8b-ref pkt (+ start 1)) (8b-ref pkt (+ start 2)) (8b-ref pkt (+ start 3)))
		      (8b-ref pkt (+ start 4))
		      (let ((bytes nil))
			(dotimes (i (- len 4) (nreverse bytes))
			  (push (format nil "~8,48b" (8b-ref pkt (+ start 4 i 1))) bytes)))))
	       ((= class *class-chaos*)
		(let ((n-contacts (16b-ref pkt start))
		      (contacts nil)
		      (idx 2))
		  (dotimes (i n-contacts)
		    (push (get-domain-name pkt (+ start idx)) contacts)
		    (incf idx (domain-name-field-length pkt (+ start idx))))
		  (format nil "~{~a~^, ~}" (nreverse contacts))))))
	(t (warn "~&>>Warning:  Unknown type/class combination in ~s: ~d/~d" 'decode-data-for-describe type class)
	   (subseq pkt start (+ start len)))))