;;; -*- Mode:Common-Lisp; Package:WEB2; Base:10. -*-

;;; Lisp Machine Dictionary using MIT LCS Webster Server

;;; Author: John Nguyen (johnn@hx.lcs.mit.edu)
;;;         MIT NE43-626
;;;         617-253-6028
;;; Inspired by webster hack done at Berkeley by Jamie Zawinski


(net:define-logical-contact-name "webster" '((:tcp 103)))

(defvar *webster-host* "mintaka.lcs.mit.edu"
  "The name of the host which runs the Webster's Dictionary server.")

(defvar *webster-lock* nil
  "To prevent interfering access to the server.")

(defvar *webster-stream* nil
  "Bidirectional stream to the webster server.")


(defmacro with-webster-lock (&body body)
  #+LISPM `(sys:with-lock (*webster-lock* :whostate "DictServer Lock") ,@body)
  #-LISPM `(progn ,@body))
  


;;; DICTIONARY server protocol:

;;; Contact name is "DICTIONARY".  A full connectional is established
;;; (additional data in the RFC is ignored, there's no simple mode)

;;; Command lines to the server are of the form

;;; 	COMMAND[<space>ARGUMENT]<NL>

;;; where the part in brackets, [], is optional.  <space> is ASCII space,
;;; octal 40, and <NL> for Chaosnet is the LispMachine NewLine character,
;;; octal 215, and for Internet it's CRLF, octal 15 then octal 12.

;;; The server responds with a single line of the same format, and then if
;;; there's additional data it comes next, followed by an EOF packet.

;;; The actual response will be either

;;; 	ERROR<space>RECOVERABLE<error message><NL>
;;; or	ERROR<space>FATAL<error message><NL>

;;; or a command-depenedent response.  FATAL-type errors are just that,
;;; fatal, and the server will go away after sending the ERROR message.
;;; 
;;; Command:	HELP<NL>

;;; This command will send back the text of this document, the dictionary
;;; protocol, followed by <EOF>.
;;; 
;;; Command:	DEFINE<space>word<NL>

;;; This is the command that asks for the defintion of a word from the
;;; dictionary.  The possible response are:

;;; 	WILD<space>0<NL>

;;; or
;;; 	WILD<NL>
;;; 	<word#><space><word1><NL>
;;; 	<word#><space><word2><NL>
;;; 		. . .
;;; 	<word#><space><wordN><NL>
;;; 	<EOF>

;;; A WILD response is given when the word to be defined contained
;;; wildcard characters ('%' which matches exactly one character, or '*'
;;; which matches 0 or more characters).  If the wild string had no
;;; matches, a WILD response with argument 0 is returned.  If there are
;;; one or more matches, a WILD with no arg is returned, and then the
;;; matching words are sent, one per line, followed by an EOF packet.  For
;;; each returned word there is a word#, a string of ASCII digits
;;; representing a decimal number.  For user convenience, that word#
;;; may be specified in place of the word itself, in a DEFINE request.

;;; 	SPELLING<space>0<NL>

;;; or
;;; 	SPELLING<NL>
;;; 	{ same response as WILD }

;;; When a word is specified that couldn't be found verbatim, Webster
;;; attempts to Do What You Mean, and try to fix common typos (transposed
;;; letter, one missing or one additional letter, or one letter wrong).
;;; If any such matches are found, a SPELL response is returned, listing
;;; all the "possible" words.  If no such words were found, (e.g. it
;;; couldn't make ANY sense out of the input word), a SPELL with argument
;;; 0 is returned.

;;;	DEFINITION<space>n<NL>
;;; then n	{ WILD-response-like lines }
;;; then	<any amount of ASCII text>
;;;	<EOF>

;;; A DEFINITION response means the word matched an entry, and the definition
;;; follows.  The argument (always present), n,  is the # of cross-references
;;; in the definition that might prove interesting.  If n > 0, then follows
;;; one line per cross-reference, in the same fork as the WILD responses.  Then
;;; comes the body of the definition, followed by and <EOF>


(defun find-webster-stream ()
  (unless (and *webster-stream* (eq (send *webster-stream* :status) :established))
    (setq *webster-stream*
	  (net:open-connection-on-medium *webster-host* :tcp-stream "webster"
					 :stream-type :ascii-translating-character-stream)))
  (do ()
      ((not (with-timeout (30 nil) (send *webster-stream* :tyi)))))
  *webster-stream*)

(defun read-until-eof (stream)
  (let (lines)
    (do-forever
      (if (= (peek-char t stream) #\null)
	  (progn
	    (send stream :tyi)
	    (return lines))
	  (push-end (read-line stream) lines)))))

(defun read-related-words (stream)
  (let (words)
    (do-forever
      (if (= (peek-char t stream) #\null)
	  (progn
	    (send stream :tyi)
	    (return words))
	  (progn
	    (read stream) ; read word #
	    (push-end (string-upcase (read-line stream)) words))))))

(defun do-define (word)
  (with-webster-lock
    (let ((stream (find-webster-stream)))
      (format stream "DEFINE ~a~%" word)
      (send stream :force-output)
      (let* ((line (read-line stream))
	     reply string)
	(multiple-value-bind (word pos) (read-from-string line)
	  (setq reply word)
	  (setq string (subseq line pos)))
	(case reply
	  (DEFINITION
	   (let (related-words)
	     (dotimes (i (parse-integer string))
	       (read stream) ; read word #
	       (push-end (string-upcase (read-line stream)) related-words))
	     (values (read-until-eof stream) related-words)))
	  (SPELLING
	   (let ((definition (format nil "~a misspelled" (string-upcase word))))
	     (if (zerop (length string))
		 (values definition (read-related-words stream))
		 (values definition nil))))
	  (WILD
	   (if (zerop (length string))
	       (values (format nil "~a pattern expansion" (string-upcase word))
		       (read-related-words stream))
	       (values (format nil "~a pattern expansion failed" (string-upcase word))
		       nil)))
	  (t
	   (values (format nil "~a not found" (string-upcase word))) nil))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun define (word &optional (stream *standard-output*))
  (multiple-value-bind (definition related-words)
      (do-define word)
    (if (consp definition)
	(dolist (line definition)
	  (format stream "~%~a" line))
	(format stream "~%~a" definition))
    (when related-words
      (format stream "~%~%See also: ~a" (car related-words))
      (format stream "~{, ~a~}" (cdr related-words)))
    (values)))

(export 'define)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(zwei:defcom zwei:com-define "Look up a word in the online dictionary." (KM)
  (multiple-value-bind (ignore ignore interval)
      (zwei:completing-read-from-mini-buffer "Word to look up:" nil t nil)
    (define-zwei (zwei:string-interval interval)))
  ZWEI:DIS-NONE)

(defvar *char-table* (make-array 256))
(dotimes (i 256)
  (setf (aref *char-table* i)
	(or (<= #\A i #\Z)
	    (<= #\a i #\z)
	    (= i #\-))))

(defun define-zwei (word &optional (stream *standard-output*))
  (multiple-value-bind (definition related-words)
      (do-define word)
    (if (consp definition)
	(dolist (line definition)
	  (format stream "~%")
	  (let ((old-pos nil)
		(length (length line)))
	    (dotimes (i length)
	      (if (aref *char-table* (aref line i))
		  (unless old-pos (setq old-pos i))
		  (progn
		    (when old-pos
		      (format stream "~vM" 'webster-word (subseq line old-pos i))
		      (setq old-pos nil))
		    (format stream "~c" (aref line i)))))
	    (when old-pos
	      (format stream "~vM" 'webster-word (subseq line old-pos length)))))
	(format stream "~%~a" definition))
    (when related-words
      (format stream "~%~%See also: ~vM" 'webster-word (car related-words))
      (dolist (word (cdr related-words))
	(princ ", " stream)
	(when (> (send stream :read-cursorpos :character) 80)
	  (format stream "~%          "))
	(format stream "~vM" 'webster-word word))
      )
    (format stream "~%")
    ))

(w:add-typeout-item-type zwei:*typeout-command-alist*
			 WEBSTER-WORD "View Definition" define-zwei t
			 "Print the definitions of this word.")

(zwei:set-comtab ZWEI:*STANDARD-COMTAB*
		 '(#\hyper-d zwei:com-define)
		 (zwei:make-command-alist '(zwei:com-define)))
