;;; -*- Mode:Lisp; Syntax:Common-Lisp; Package:PRINTER -*-

;;; File "COMPRESS-POSTSCRIPT"
;;; Converting LISPM bitmaps to compressed postscript.
;;; Written and maintained by Jamie Zawinski.
;;;
;;; ChangeLog:
;;;
;;; 12 Sep 88  Jamie Zawinski    Created.
;;;

;;; The code in this file takes a bitmap and produced PostScript code to draw it.
;;; The bitmap data is shipped to the PS interpreter using run-length encoding.
;;; This compression relies on the fact that most bitmaps have large repeated areas.
;;; Depending on the image, it can reduce the number of bits shipped by 70% or more.
;;;
;;; The Lisp part of this code will not port easilly away from Lisp machines, for it relies 
;;; on one crucial feature: the ability to indirect arrays of different element-sizes.  On a 
;;; Lispm, if one indirects an 8-bit vector to a 1-bit bitmap, one slot of the vector 
;;; corresponds to 8 slots of the bitmap, making it very easy to extract one ``byte'' of 
;;; the bitmap.  This is not part of the CommonLisp standard.
;;;
;;; Also, this code makes the assumption that bitmaps are referenced in column-major order, 
;;; which is in violation of CommonLisp, but on Lispms, is true of 2d, 1-bit arrays.  Because 
;;; of this, the vectors displaced on to the bitmaps appear to be LSBit first insead of MSbit 
;;; first.  So we reverse the bit order when writing out the file.  Also we invert the bits,
;;; so that the output pages will be black-on-white instead of white-on-black.
;;;
;;; However, the PostScript which this code produces should run on any PostScript printer.
;;;
;;; BUT: this might not work for you if you don't have an extremeley reliable connection to 
;;; your printer; when an uncompressed bitmap is shipped to the printer, a bit flipped by
;;; random line noise will cause a speck on the page.  When this happens to a compressed 
;;; image, the rest of the image after the glitch will turn into static.

(defmacro round-up-to (limit number)
  "Returns the next multiple of LIMIT greater than NUMBER."
  `(+ ,limit (* ,limit (floor (1- ,number) ,limit))))


(defvar *bit-flippage* (let* ((a (make-array 256 :element-type '(unsigned-byte 8))))
			 (dotimes (i 256)
			   (let* ((flip-byte 0))
			     (dotimes (j 8)
			       (setq flip-byte (dpb (ldb (byte 1 (- 7 j)) i) (byte 1 j) flip-byte)))
			     (setf (aref a i) flip-byte)))
			 a)
  "A table for quickly reversing the order of bits in a byte.")

(proclaim '(type (array (unsigned-byte 8) (256)) *bit-flippage*))


(defmacro flipped-aref (array index)
  "Just like AREF, but filters the result through the *BIT-FLIPPAGE* vector first.  Don't SETF this."
  `(svref (the vector *bit-flippage*)
	  (aref ,array ,index)))

(defsetf flipped-aref (array index) (newval)
  (declare (ignore array index newval))
  (error "Don't Setf FLIPPED-AREF, you bogon!"))

(defun bitmap-to-bytemap (bm)
  "Returns a new 1-dimensional array displaced to the two-dimensional array BM.
  The returned array will be of element-type (INTEGER 0 255)."
  (let* ((w (array-dimension bm 1))
	 (h (array-dimension bm 0))
	 (l (/ (* w h) 8)))
    (make-array l :element-type '(integer 0 255) :displaced-to bm)))


(eval-when (load eval compile)

(proclaim '(inline write-byte-as-hex write-word-as-hex
		   read-byte-as-hex read-word-as-hex))

(defun write-byte-as-hex (n stream)
  "Write N as a 2-character hex number (8 bits)."
  (let* ((ms-nibble (ldb (byte 4 4) n))
	 (ls-nibble (ldb (byte 4 0) n))
	 (conversion "0123456789ABCDEF"))
    (write-char (schar conversion ms-nibble) stream)
    (write-char (schar conversion ls-nibble) stream))
  n)

(defun write-word-as-hex (n stream)
  "Write N as a 4-character hex number (16 bits)."
  (let* ((ms-byte (ldb (byte 8 8) n))
	 (ls-byte (ldb (byte 8 0) n)))
    (write-byte-as-hex ms-byte stream)
    (write-byte-as-hex ls-byte stream))
  n)


(defun read-byte-as-hex (stream)
  "Read two hex digits and return the 8 bit unsigned integer representing them.
  All preceeding and interspersed whitespace is ignored.  Non-hex characters cause an error."
  (let* ((conversion "0123456789ABCDEF")
	 ms-nibble-char
	 ls-nibble-char)
    (peek-char t stream)  ; discard whitespace.
    (setq ms-nibble-char (read-char stream))
    (peek-char t stream)
    (setq ls-nibble-char (read-char stream))
    (+ (* 16 (position ms-nibble-char conversion :test #'char-equal))
       (position ls-nibble-char conversion :test #'char-equal))))


(defun read-word-as-hex (stream)
  "Read four hex digits and return the 16 bit unsigned integer representing them.
  All preceeding and interspersed whitespace is ignored.  Non-hex characters cause an error."
  (+ (* 256 (read-byte-as-hex stream))
     (read-byte-as-hex stream)))

 ) ; Closes EVAL-WHEN




(defvar *scratch-bitmap* (make-array '(0 0) :element-type 'bit :adjustable t)
  "Before dumping a section of a bitmap, we must copy that section into its own bitmap, so we can properly displace to it.
  This bitmap is adjusted to be the size ofthe bitmap being copied.")


(defun dump-runlength-bitmap (bitmap x y w h
			      &optional (stream *standard-output*) (invert-image-p t)
					(compress-across-scanlines nil) (insert-newline 78))
  "Dump a hexadecimal represtation of the indiciated portion of BITMAP to STREAM.
  The bitmap is dumped using Byte Run 1 compression.
  Each logical ``byte'' written to the stream is represented as a two-digit hexadecimal number, from 00 to FF.
  The first 8 characters written represent the width and height of the bitmap, 16 bits each, MSB first.
  If COMPRESS-ACROSS-SCANLINES is non-NIL, then multiple scanlines may be compressed together.
  This won't work if the bitmap portion is wider than 1023 bits.
  If INSERT-NEWLINE is a number, then extra whitespace will be inserted every that-many characters in the output.
  This is helpful when dealing with things which line-buffer things.  If NIL, no newlines will be inserted.

  Returns the number of logical bytes output - this is half the number of characters written if INSERT-NEWLINE is NIL."

  (let* ((w32 (round-up-to 32 w))  ; next multiple of 32 above width, to placate BITBLT.
	 (bw (/ w32 8))     ; width in bytes.
	 displaced
	 len
	 (bytes-written 0)
	 (output-column 0))
    
    (adjust-array *scratch-bitmap* (list h w32))  ; Lispm bitmaps are column-major.
    (bitblt tv:alu-seta w h bitmap x y *scratch-bitmap* 0 0)
    (setq displaced (bitmap-to-bytemap *scratch-bitmap*))
    (setq len (length displaced))

    (when insert-newline (fresh-line stream))
    (write-word-as-hex w32 stream)  ; Write out the width and the height of the bitmap
    (write-word-as-hex h   stream)  ; as two four-character hex strings.
    (incf bytes-written 4)

    (when insert-newline (terpri stream))
    (do* ((pos 0))
	 ((>= pos len))
      (let* ((value (flipped-aref displaced pos))
	     (next-scan-end (if compress-across-scanlines
				len
				(min len (round-up-to bw (1+ pos)))))
	     (rep (do* ((i 0 (1+ i)))
		       ((or (>= (1+ i) (- next-scan-end pos))
			    (/= (flipped-aref displaced (+ pos i 1))
				value))
			i))))
	(when invert-image-p (setq value (lognot value)))
	(cond ((>= rep 2)
	       ;;
	       ;; We can do a replication run now.
	       ;;
	       (write-byte-as-hex (- (1- rep)) stream)          ; control byte
	       (write-byte-as-hex (flipped-aref displaced pos) stream)  ; data byte
	       (incf pos rep)
	       (incf bytes-written 2)
	       (when insert-newline
		 (incf output-column 4)
		 (when (>= output-column insert-newline)
		   (terpri stream)
		   (setq output-column 0)))
	       )
	      (t
	       ;; Otherwise we must do a literal run now.
	       ;; That means we need to know when the literal run should stop - which is when there is replication of
	       ;; two or more bytes.
	       ;;
	       (let* ((stop (do* ((i 1 (1+ i)))
				 ((or (>= (1+ i) (- next-scan-end pos))
				      (= (flipped-aref displaced (+ pos i))
					 (flipped-aref displaced (+ pos i 1))))
				  i))))
		 (when (zerop stop) (error "Impossible case.  Dumper broken."))
		 (incf output-column 2)
		 (write-byte-as-hex (1- stop) stream)   ; control-byte
		 (dotimes (i stop)                      ; data-bytes
		   (write-byte-as-hex (flipped-aref displaced (+ pos i)) stream)
		   (when insert-newline
		     (incf output-column 2)
		     (when (>= output-column insert-newline)
		       (terpri stream)
		       (setq output-column 0)))
		   )
		 (incf pos stop)
		 (incf bytes-written (1+ stop))
		 )))))
    (when insert-newline (terpri stream))
    bytes-written))


;;; Testing hack.
(defun read-runlength-bitmap (stream)
  "Read a runlength compressed bitmap written in hexdump format from STREAM, and return a new bitmap of it."
  (let* ((w (read-word-as-hex stream))
	 (h (read-word-as-hex stream))
	 (bitmap (make-array (list h w) :element-type 'bit))
	 (displaced (bitmap-to-bytemap bitmap))
	 (len (length displaced))
	 )
    (do* ((pos 0))
	 ((>= pos len))
      (let* ((control (read-byte-as-hex stream)))
	(when (> control 127) (decf control 256))
	(cond ((= control -128)
	       (format t "~&NOOP~%")
	       nil)
	      
	      ((>= control 0)  ; literal
;	       (format t "~&lit ~S~%" control)   ; ####################################
	       (dotimes (i (1+ control))
		 (setf (aref displaced (+ pos i)) (aref (the vector *bit-flippage*) (read-byte-as-hex stream))))
	       (incf pos (1+ control)))

	      (t    ; replicate
	       (let* ((times (1+ (- control)))
		      (repl (read-byte-as-hex stream)))
;		 (format t "~&rep ~S  ~s~%" control repl)   ; ####################################
		 (dotimes (i times)
		   (setf (aref displaced (+ pos i)) (aref (the vector *bit-flippage*) repl))
		   )
		 (incf pos times))))))
    bitmap))


(defun read-ps-bitmap (file)
  "This is a testing hack.  Discard all lines from file up to the line ending with the string ``do-image'', and then
   read the compressed bitmap data from it.  This is so we can snarf the bitmap data out of the PS files we produced
   without having to hand-bash the file.
  Returns a new bitmap."
  (with-open-file (stream file :direction :input)
    (do* ((line (read-line stream) (read-line stream))
	  (target "do-image"))
	 ((and (>= (length line) (length target))
	       (string= line target :start1 (- (length line) (length target))))))
    (read-runlength-bitmap stream)))


;;;; Here begins PostScript Land.
;;;;
(defvar *runlength-prolog* "%!ps
% PostScript code for printing a run-length encoded bitmap.
% Copyright 1988 by Jamie Zawinski.

/controlbuf 1 string def
/repbuf 1 string def
/WHbuf 4 string def
/chunk-count 0 def

/read-w-and-h {
  currentfile WHbuf readhexstring pop   % read the width and height into /WHbuf as a 4 byte quantity.
  /w WHbuf 0 get 256 mul  WHbuf 1 get  add def   % set /w and /h to the values in this string, MSB first.
  /h WHbuf 2 get 256 mul  WHbuf 3 get  add def
  } def

/read-runlength-chunk {
  currentfile controlbuf readhexstring pop pop  % read one hex character from the stream.
  /code controlbuf 0 get def                    % set /code to the character read.
  code 127 gt {/code code 256 sub def} if       % If /code is >127, subtract 256, converting to be in the range [-128,127].
  /chunk-count chunk-count 1 add def

  code -128 eq  % -128 is illegal.  Print an error message and stop.
   { (\\n%%[ Error: illegal run-length control byte at chunk ) print
     chunk-count 1 sub (     ) cvs print
     ( - probably lost some bits.  Try re-printing this file. ]%%\\n) print
     stop }
   { code 0 ge
     { lit }    % >=0 means copy next CODE+1 bytes literally.
     { rep }    % < 0 means replicate next byte -CODE+1 times.
     ifelse }
   ifelse
  } def

/lit {    % uses dynamic binding of /code - leaves a new string on the stack.
  currentfile
  code 1 add string
  readhexstring pop
  } def

/rep {    % uses dynamic binding of /code - leaves a new string on the stack.

  currentfile repbuf readhexstring pop pop
  /repchar repbuf 0 get def               % get next code.
  /newstring  code neg 1 add string  def  % build a string.
  
  0 1 newstring length 1 sub              % set every element of the new string to be the new char.
  { newstring exch repchar put }
  for
  newstring     % leave the new string on the stack.
  } def

/do-image {     % takes dest-w and dest-h on stack.  There must be image data textually after this.
  /dest-h exch def
  /dest-w exch def
  0 0 moveto
  20  20 translate
  dest-w dest-h scale
  read-w-and-h
  w h 1 [w 0 0 h neg 0 h] {read-runlength-chunk} image
  showpage
  } def
"
 "This is all of the PostScript code necessary to decode run-length compressed data shipped after the file.")


(defun dump-ps-bitmap (bitmap x y w h file &optional (width-in-points (* 8 72)) (height-in-points width-in-points)
		       (verbose t))
  "Produce a PostScript file which will print the indiciated portion of BITMAP.
   The bitmap drawn on the page will be WIDTH-IN-POINTS wide - this is the width of the entire image, not of one pixel."
  (with-open-file (stream file :direction :output)
    (princ *runlength-prolog* stream)
    (format stream "~&~D ~D do-image~%~%" width-in-points height-in-points)
    (let* ((pos (file-position stream)))
      (dump-runlength-bitmap bitmap x y w h stream)
      (when verbose
	(let* ((total-chars (file-position stream))
	       (chars (- total-chars pos))
	       (uncompressed-chars (ceiling (* w h 2) 8)))
	  (format t "~&~D% bitmap compression; ~D% file-size savings (including prolog size)."
		  (floor (* 100 (float (/ chars uncompressed-chars))))
		  (floor (* 100 (float (/ total-chars (+ uncompressed-chars 20)))))))))
    (send stream :truename)))
