;;; -*- Mode:Lisp; Package:PROFILE; Syntax:Common-Lisp -*-

;;; File "CHOOSE-COLOR"
;;;
;;; Redefines PROFILE:CHOOSE-COLOR to show the colors in question on the menu.
;;; Makes Profile variables of type :COLOR print as their color names if they are a named color,
;;; instead of always printing as their color number.
;;;
;;; Written and maintained by Jamie Zawinski.
;;;
;;; ChangeLog:
;;;
;;; 25 Nov 88  Jamie Zawinski    Created.
;;; 13 Dec 88  Jamie Zawinski    Added a call to COMPILE-FLAVOR-METHODS, because when it tries to compile the two methods
;;;                               defined at instantiation-time, it always gets the "stack frame larger than 256" error.
;;; 22 Dec 88  Jamie Zawinski    Made the :after :init and :after :refresh methods be functions instead, because I was 
;;;                               STILL getting that stack-frame error.  The compiler is being really dumb.
;;;



(defflavor color-request-menu
	   ()
	   (w:menu)
  (:default-init-plist
    :item-list w:color-alist
    :pop-up t
    )
  :gettable-instance-variables
  :settable-instance-variables
  :initable-instance-variables)

(defun color-request-menu-after-init (&optional ignore ignore)
  (declare (:self-flavor color-request-menu))
  (send self :set-size (+ (tv:sheet-width self) 50) (tv:sheet-height self))
  (send self :set-label-color tv:*default-menu-label-foreground*)
  (send self :set-label-background tv:*default-menu-label-background*)
  (send self :set-foreground-color tv:*default-menu-foreground*)
  (send self :set-background-color tv:*default-menu-background*)
  )

(defmethod (color-request-menu :after :init) color-request-menu-after-init)


(defun color-request-menu-after-refresh (&optional ignore (type :complete-redisplay))
  "Draw a column of colored boxes on SELF, of the colors and order in the ITEM-LIST.
  If the CDR of an element of the item list is not an integer, that item is skipped.
  The left position of the rectangles is determined by the length of the strings in W:COLOR-ALIST."
  (declare (:self-flavor color-request-menu))
  (when (eq type :complete-redisplay)
    (let* ((max-color-name-length 0))
      (dolist (cons w:color-alist)
	(setq max-color-name-length (max max-color-name-length (tv:sheet-string-length self (car cons)))))
      (let* ((w 50)
	     (h (tv:sheet-line-height self))
	     (x (+ max-color-name-length 10))
	     (y 0)
	     (cy y)
	     )
	(dolist (cons (send self :item-list))
	  (when (and (consp cons)
		     (integerp (cdr cons)))
	    (let* ((color (cdr cons)))
	      (send self :draw-rectangle w h x cy TV:ALU-SETA color)))
	  (incf cy h))))))


(defmethod (color-request-menu :after :refresh) color-request-menu-after-refresh)


(compile-flavor-methods COLOR-REQUEST-MENU)


(defun choose-color-1 (label default-value &optional other-options)
  "Pop a menu to select colors colorfully.  OTHER-OPTIONS is appended to the end of the ITEM-ALIST."
  ;; This function creates and then kills a menu each time.
  ;; This is maybe not as efficient as it could be, but sizing the menu easier this way...
  (let* (menu)
    (unwind-protect
	(progn
	  (let* ((item-list (append '(("color names:" :no-select t :font fonts:hl12b))
				      W:COLOR-ALIST
				      (when other-options
					(list* "" '("other:" :no-select t :font fonts:hl12b)
					       other-options)))))
	    (setq menu (make-instance 'color-request-menu
				      :label label
				      :item-list item-list
				      ))
	    (or (send menu :choose) default-value)))
      (when menu (send menu :kill)))))


(DEFUN choose-color (default-value &aux color-map)
  "Prompts user to select color value via menus.  If no color is selected, the DEFAULT-VALUE is returned."
  (DECLARE (SPECIAL tv:*color-system* tv:*default-color-map*))
  (declare (values color-value))
  (flet ((choose-system-color-or-something (color-map-p)
	   (choose-color-1 "Select a Color"
			   nil
			   (if color-map-p
			       '(("Show Color Map" :value :color-map)
				 ("Enter Color Number" :value :color-number))
			       '(("Enter Color Number" :value :color-number))))))
    (cond ((AND (VARIABLE-BOUNDP tv:*color-system*) tv:*color-system*)
	   (SETQ color-map (OR (AND tv:selected-window
				    (SEND tv:selected-window :send-if-handles :color-map))
			       tv:*default-color-map*))
	   (MULTIPLE-VALUE-BIND (choice selected?)
	       (choose-system-color-or-something t)
	     (IF (AND (NULL choice) (NULL selected?))
		 (values default-value)
		 (CASE choice
		   (:color-map (values (w:select-color-with-mouse color-map)))
		   (:color-number (values (w:pop-up-read-number)))
		   (t  (values choice))))))
	  (t 
	   (MULTIPLE-VALUE-BIND (choice selected?)
	       (choose-system-color-or-something nil)
	     (IF (AND (NULL choice) (NULL selected?))
		 (values default-value)
		 (CASE choice
		   (:color-number (w:pop-up-read-number))
		   (t choice))))))
    ))




(defun cvv-print-color (color-number &optional stream)
  (princ (or (car (rassoc color-number w:color-alist)) color-number)
	 stream))

(setf (get :color 'w:choose-variable-values-keyword)
      '(cvv-print-color nil nil nil choose-color))


;;;
;;; The blinker-offset isn't really a color, so always print it as its number.
;;;
(define-profile-variable tv:*default-blinker-offset* (:color)
  :cvv-type (:FIXNUM-IN-INTERVAL 0 255)
  :form-for-init-file
  (lambda (var)
    `(WHEN (VARIABLE-BOUNDP tv:*default-blinker-offset*)
       (PROFILE-SETQ ,var ,tv:*default-blinker-offset*))))
