
;; This tutorial makes some synthetic stereograms, and then describes
;; some algorithms for measuring disparity in a stereo pair.  

;; First load in some required code, and define a few simple functions
;; (using C-c e):
(progn
  (setf (obv::background (obv::current-screen)) :black)
  (set-default 'filter :display-type nil)
  (set-default 'overlay :zoom 2)
  ;; Load some software to run the full-blown multi-scale algorithm:
  (compile-load (merge-pathnames "tutorials/stereo/anaglyphs"
				 obv::*obvius-directory-path*))
  (compile-load (merge-pathnames "tutorials/stereo/synth-stereo"
				 obv::*obvius-directory-path*))
  (compile-load (merge-pathnames "tutorials/stereo/block-matching"
				 obv::*obvius-directory-path*)))

;; The first example here creates a random-dot stereogram with a
;; raised square in the center.

;; MAKE-RECTANGLES creates the disparity map.  The first argument is
;; the dimensions of the disparity map image.  The second argument is
;; a list of lists that specify the:
;;
;;        (width height center-x center-y disparity)
;;
;; for a rectangular patch of constant disparity
(setq dmap (make-rectangles '(128 128) '((128 128 64 64 1)
					 (64 64 64 64 5))))

;; The next step is to convert the disparity map to a random-dot
;; stereogram.  DOT-PAIR-FROM-DISPARITY-MAP returns a pair of
;; bit-images.  Use C-M-right (hold down the Control and Meta keys,
;; and then click the right mouse button) to flip display them in
;; sequence.
(setq stereopair (dot-pair-from-disparity-map dmap))

;;; DISPLAY-ANAGLYPH creates a red-green anaglyph for viewing the
;;; stereogram through red-green glasses.  With the red filter in
;;; front of your right eye, you should see the central square
;;; floating in front of the background.  If you flip the glasses
;;; around the other way, then the central square is farther away than
;;; the background.
(display-anaglyph stereopair)

;; Create a 10% density random-dot stereogram (using the optional
;; :dot-density keyword argument):
(display-anaglyph (dot-pair-from-disparity-map dmap :dot-density 1/10))

;; Create a stereogram with a pyramid disparity map (i.e. a set
;; of concentric squares)
(setq pyramid-map
      (make-rectangles '(128 128) '((128 128 64 64 1)
				    (96 96 64 64 3)
				    (64 64 64 64 5)
				    (32 32 64 64 7))))
(display-anaglyph (dot-pair-from-disparity-map pyramid-map))

;; Make a ramp disparity map.  Even thought the ramp is continuous,
;; the random dot stereogram has steps of disparity because it is
;; generated by shifting the random dot pattern an integer number of
;; pixels.
(setq ramp (make-ramp '(128 128) :origin '(0 0) :slope 1/16))
(display-anaglyph (dot-pair-from-disparity-map ramp))

;; MAKE-RECTANGLES can also be used to create multiple transparent
;; depth planes (by specifying an optional second depth value in the
;; list):
(setq transparent-square-map
      (make-rectangles '(128 128)
		       '((128 128 64 64 1)
			 (64 64 64 64 -3 3))))
(display-anaglyph (dot-pair-from-disparity-map transparent-square-map))

;; You can also use OBVIUS synthetic image generation functions for
;; creating a greater variety of depth maps. Some examples are:
;; make-ramp, make-r-squared, make-sin-grating,
;; make-saw-tooth-grating, make-square-grating, make-disc,
;; make-fractal.

(display-anaglyph (dot-pair-from-disparity-map (make-sin-grating '(128 128)
								 :period 64
								 :phase (* -1/2 pi)
								 :dc 0.0
								 :amplitude 6.0)))

(display-anaglyph (dot-pair-from-disparity-map
		   (mul (make-r-squared '(128 128)) (/ 36 (sqr 128)))))

;; Clean up:
(purge!)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; We also have code for synthesizing stereo pairs of texture-mapped
;; planar surfaces.  This will be useful below for testing stereo
;; algorithms.  Here's an example.  First, make a texture image:
(setq plaid-texture (add (make-sin-grating '(128 128) :orientation 0 
					   :x-freq 0.5 :y-freq 0.5)
			 (make-sin-grating '(128 128) :orientation 0 
					   :x-freq 0.5 :y-freq -0.5)))

;; Then, compute the stereo-pair and the disparity-map:
(multiple-value-setq (stereo-pair1 disparity-map1)
  (make-stereo-pair plaid-texture))
(display disparity-map1 'vector-field)

;; The MAKE-STEREO-PAIR function takes a number of keywords to specify
;; the orientation and distance of the planar surface and the vergence
;; angle of the "eyes".  For example, to tilt the plane further back:
(multiple-value-setq (stereo-pair2 disparity-map2)
  (make-stereo-pair plaid-texture :x-rot-angle -50))

;; To move the plane closer:
(multiple-value-setq (stereo-pair3 disparity-map3)
  (make-stereo-pair plaid-texture :x-rot-angle -50 :dist 5))

;; To verge the eyes more:
(multiple-value-setq (stereo-pair4 disparity-map4)
  (make-stereo-pair plaid-texture :x-rot-angle -50 :dist 5 :vergence 12))

;; And to tilt the plane in x and in y:
(multiple-value-setq (stereo-pair5 disparity-map5)
  (make-stereo-pair plaid-texture :x-rot-angle -50 :y-rot-angle 20 :dist 5))

;; You can also do this, using a real image (instead of a synthetic
;; image) for the texture map:
(load-image (merge-pathnames "images/einstein" obv::*obvius-directory-path*))
(multiple-value-setq (stereo-pair-al disparity-map-al)
  (make-stereo-pair (gauss-out einstein) :x-rot-angle -50 :dist 5))

;; One important thing to notice in all of the above examples is that
;; when the eyes are verged, there is vertical (as well as horizontal)
;; disparity, particularly in the corners of the image.  When the eyes
;; are parallel, on the other hand, there is horizontal positional
;; disparity between the left and right eyes' images.
(display disparity-map-al 'vector-field)

;; *** Need to fix up the synth-stereo code so we can show an example
;; in which the eyes are parallel.  Currently, the code is restricted
;; to the cases in which the eyes are always verged to fixate on the
;; "center" of the planar surface.

;; Clean up:
(purge!)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Block matching stereo algorithms: One of the simplest algorithms
;; for estimating positional disparity is the block matching
;; algorithms.  Take a patch from one image (say the left image),
;; overlay in various possible positions on top of the right image,
;; and pick the overlayed position that gives the "best match".  The
;; best match is defined to minimize the Sum of the Squared
;; Differences (SSD) between the two images.

;; Here's simple stereogram to use as a test input:
(progn
  (setq dots (dot-pair-from-disparity-map
	      (make-rectangles '(128 128) '((128 128 64 64 1)
					    (64 64 64 64 5)))))
  (setq stereopair (make-image-pair (list (coerce-to-float (frame 0 dots))
					  (coerce-to-float (frame 1 dots))))))

;; Here's the squared difference between the two images:
(square (sub (left-image stereopair) (right-image stereopair)))

;; Here's the squared difference after shifting one of the images to
;; compensate for the positional disparity in the center part of the
;; images:
(square (sub (circular-shift (left-image stereopair) :x 5) (right-image stereopair)))

;; Here's the squared difference after shifting one of the images to
;; compensate for the positional disparity in the surrounding part of
;; the images:
(square (sub (circular-shift (left-image stereopair) :x 1) (right-image stereopair)))

;; Here's the squared error for a whole sequence of shifts (use
;; C-M-left to step through the sequence):
(setq sqrd-diffs
      (make-image-sequence
       (loop for i from 0 below 6
	     collect
	     (square (sub (circular-shift (left-image stereopair) :x i) (right-image stereopair))))))

;; The sum of squared difference (SSD) images can be computed by
;; blurring (summing over spatial patches) each of the squared
;; difference images:
(setq ssd-images (blur sqrd-diffs :level 2))

;; Again, use C-M-left to step through the sequence.  Now the only
;; zero (black) pixels are those for which we have the correct shift.
;; That is, the positional disparity in the center is 5 and the
;; positional disparity in the surround is 1.

;; The function block-matching-ssd bundles these steps together and
;; returns a disparity map (pictured here as a vector field):
(display (setq disparity-map
	       (block-matching-ssd (left-image stereopair) (right-image stereopair) :blur-level 2))
	 'vector-field)
(setp :scale 6.0)

;; A more interesting example (the pyramid disparity map that we looked at above):
(progn
  (setq dots (dot-pair-from-disparity-map
	      (make-rectangles '(128 128) '((128 128 64 64 1)
				    (96 96 64 64 3)
				    (64 64 64 64 5)
				    (32 32 64 64 7)))))
  (setq stereo-pyramid (make-image-pair (list (coerce-to-float (frame 0 dots))
					      (coerce-to-float (frame 1 dots))))))

;; The estimated disparity-map:
(display (setq disparity-map
	       (block-matching-ssd (left-image stereo-pyramid) (right-image stereo-pyramid) :blur-level 2))
	 'vector-field)
(setp :scale 8)

;; Displaying the magnitude of the disparity sometimes makes it easier
;; to see the result:
(magnitude disparity-map)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Nishihara's bit correlation scheme:

;; ***

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Phase-based stereo algorithm:

;; ***

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Comparison on some real stereo pairs:

;; ***
