;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File:  mcl-window.lisp
;;;  Author: Simoncelli/Heeger
;;;  Description: 
;;;  Creation Date: 12/93 modified from lv-screen.lisp
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)

;;; These are defined in optional file x-control-panel.  Put them here
;;; to avoid export errors.
(export '(make-control-panel destroy-control-panel))

;;; Make sure this is Mac Common Lisp
#-MCL
(eval-when (load compile eval)
  (error "This file is meant to be run in Mac Common Lisp only"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; mcl-pane object

;;; This method must be provided for each type of system-dependent screen.
(defmethod make-pane ((screen mcl-screen) &rest keys 
                      &key left bottom right top width height
		      border-width &allow-other-keys)
  (declare (ignore left bottom right top width height border-width))
  (apply 'make-instance 'mcl-pane :screen-of screen keys))

;;; The underlying window object, which inherits from the generic
;;; obvius:pane object and the ccl::window object.
;;; Add a :confirm-quit function to ask the user if they're sure!
(def-simple-class mcl-pane (pane ccl::window)
  ((status :initform t))
  (:default-initargs
    :width 300
    :height 256
    :window-type :document-with-grow
    :window-title "obvius pane"
    ))

(defmethod initialize-instance :around ((pane mcl-pane) &rest initargs
                                        &key width height)
  (remf initargs :width)
  (remf initargs :height)
  (let ((screen (getf initargs :screen-of)))
    (unless (and screen (typep screen 'mcl-screen))
      (error "Must provide an mcl-screen as a :screen-of argument"))
    (setf (getf initargs :view-size) (ccl::make-point width height))
    (apply #'call-next-method pane initargs)
    ;; set fore color
    (ccl::set-fore-color pane (convert (foreground screen) :encoded-color))
    (ccl::set-back-color pane (convert (background screen) :encoded-color))))

;;; Modify to set up picture-specific mouse bindings of pane.
(defmethod draw-pane ((pane mcl-pane) &rest keys)
  (unless (eq (status pane) :destroyed)
    (call-next-method)))

;;; *** Could be smarter about this: only need to draw a portion of
;;; the window!
#|
;;; Would like to do something like this to guarantee redrawing the entire pane.
;;; In order to do this, need to load traps and records.
;;; This gets stuck in an infininte loop. Perhaps need without-interrupts even
;;; though docs say that isn't necessary (see p. 390).
(defmethod ccl::view-draw-contents ((pane mcl-pane))
  (ccl::with-port (ccl::wptr pane)
    (#_invalrect :ptr (ccl::rref (ccl::wptr pane) :windowRecord.portrect)))
  (draw-pane pane :clear nil))
|#
(defmethod ccl::view-draw-contents ((pane mcl-pane))
  (draw-pane pane :clear nil))

(defmethod x-dim ((pane mcl-pane))
  (ccl::point-h (ccl::view-size pane)))

(defmethod y-dim ((pane mcl-pane))
  (ccl::point-v (ccl::view-size pane)))

(defmethod dimensions ((pane mcl-pane))
  (list (ccl::point-v (ccl::view-size pane))
        (ccl::point-h (ccl::view-size pane))))

(defmethod clear ((pane mcl-pane)
		  &key (y0 0) (x0 0) 
		  (y1 (y-dim pane))
		  (x1 (x-dim pane))
		  (color (background (screen-of pane))))
  (ccl::with-focused-view pane
    (ccl::with-back-color (convert color :encoded-color)
      (ccl::erase-rect pane x0 y0 x1 y1))))

(defmethod set-selected-pane ((pane mcl-pane))
  (when (not (eq pane *current-pane*))
    (ccl::set-part-color pane :frame (convert (selected-color (screen-of pane)) 
                                              :encoded-color))
    (when *current-pane* (ccl::set-part-color *current-pane* :frame nil))
    (call-next-method)))

(defmethod set-pane-title-bar ((pane mcl-pane) title)
  (ccl::set-window-title pane (string-right-trim "." title)))

;;; Destroy the picture stack and clean up.
(defmethod (setf status) ((val (eql :destroyed)) (pane mcl-pane))
  (destroy pane))

(defmethod destroy ((pane mcl-pane) &key &allow-other-keys)
  (call-next-method)
  (setf (slot-value pane 'status) :destroyed)
  (ccl::window-close pane))

(defmethod ccl::window-close ((pane mcl-pane))
  (unless (equal (status pane) :destroyed)
    (destroy pane))
  (call-next-method))

(defmethod depth ((pane mcl-pane))
  (depth (screen-of pane)))


;;; Local Variables:
;;; buffer-read-only: t 
;;; End:
