;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: lucid-hacks.lisp
;;;  Author: Heeger
;;;  Description: Lucid dependent extensions to common-lisp
;;;  Creation Date: 9/92
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)
(export '(status-message))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Setf iref defined in image.lisp and bit-image.lisp is
;;; implementation-dependent.  These methods must be defined in those files,
;;; after the image and bit-image classes are defined.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Used by memory.lisp

(defun make-static-array (size type)
  (make-array size :element-type type))

(defun static-array-p (arr)
  nil)

(defun displaced-array-p (arr)
  (ccl:displaced-array-p arr))

;;; *** what to do here
(defmacro with-scheduling-inhibited (&body body)
  `(ccl::without-interrupts ,@body))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; multiprocessing stuff

;;; Only allow one process at a time to access the pane.  Nested calls
;;; to with-locked-pane to this by the same process are ok.
;;; *** Currently in mcl, this does nothing!
(defmacro with-locked-pane (pane . body)
  `(progn ,@body))

(defun push-onto-eval-queue (form &key silent)
  (unless silent
    (let ((*print-length* 2) (*print-level* 2))
      (status-message "Enqueuing form: ~A" form)))
  (ccl::eval-enqueue form))
	
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Used by conversion.lisp (could use obvius functions obv::array-read and
;;; obv::array-write instead).

(defun write-array (&rest args)
  (error "Not implemented"))

(defun read-array (&rest args)
  (error "Not implemented"))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; stepit.lisp.
#|
;;; *** Lucid hair. Needs to be made into MCL hair.
;;; Need to acquire a function pointer to the foreign callable.
(defun foreign-function-pointer (symbol)
  (let* ((name (string-downcase (symbol-name symbol)))
	 (address (LCL:foreign-variable-address name)))
    (LCL:make-foreign-pointer :address address)))
|#

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; fileio

(defun directory-p (path)
  (let ((path-string (namestring path)))
    (probe-file (pathname (concatenate 'string path-string ":")))))

(defun directory-path (path)
  (let ((path-string (namestring path)))
    (pathname (concatenate 'string path-string ":"))))

(defun trim-right-delimiter (path)
  (let ((path-string (namestring path)))
    (pathname (string-right-trim ":" path-string))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Status message stuff

;;; Global switch to turn off messages.
(defvar *status-messages* t
  "Switch to turn off the status-message line.  Use the macro without-status-messages
to temporarily turn it off.")
(eval-when (load eval) (setf (get '*status-messages* :type) '(member nil t)))

(defmacro without-status-messages (&rest body)
  `(let ((*status-messages* nil))
    (declare (special *status-messages*))
    ,@body))

;;; Print "<msg> ..." at start, and "<msg> ... done." at end
;;; msg can be a string, or a list or a string and format-args
;;; *** Conses more than it should.
(defmacro with-status-message (msg &rest body)
  (let ((str (gensym)))
    `(let* ((,str (concatenate 'string ,msg "... ")))
       (prog1
	   (progn (status-message ,str) ,@body)
	 (status-message (concatenate 'string ,str "done") )))))

;;; This will be reset to t by emacs if Lisp is running in a
;;; cl-shell-mode buffer.
(defvar user::*emacs-cl-shell* nil)

;;; These control I/O of messages, passive mouse documentation and
;;; user selection of viewables with the mouse. They can be lists
;;; (stacks), or just single objects.
(defvar *status-reporter* nil)		;typically a status message line
(defvar *mouse-doc-reporter* nil)	;also typically a status message line
(defvar *selection-receiver* nil)	;typically a dialog box

;;; If *obvius-status-window* is non-nil, send the string to the
;;; status line.  else if *emacs-cl-shell* is non-nil, send the string
;;; to emacs to be put in the minibuffer.  Otherwise, print it on the
;;; *trace-output* stream.
(defun status-message (string &rest format-args)
  (cond ((null *status-messages*) nil)
	(*status-reporter*
	 (report (if (consp *status-reporter*)
		     (car *status-reporter*)
		     *status-reporter*)
		 (apply #'format nil string format-args)))
	((ccl::view-mini-buffer ccl::*top-listener*)
         (apply #'ccl::set-mini-buffer ccl::*top-listener* string format-args))
	(t
	 (format *trace-output* ";;; ")    ;Print as comment.
	 (apply #'format *trace-output* string format-args)
	 (fresh-line *trace-output*)
	 (force-output *trace-output*)))) ;*** is this needed?

;;; Thingy for outputing an expression which will be evaluated.  If
;;; *emacs-cl-shell* is non-nil, assumes that the lisp process is
;;; running inside of emacs, and that the hacks in the file
;;; cl-shell.el have been loaded to allow the process to insert text
;;; at the current location of the point (cursor) for evaluation.
;;; Otherwise, prints the sexpr in a comment.
(defun insert-sexpr-for-evaluation (sexpr)
  (cond (*selection-receiver*
	 (insert-selection (if (consp *selection-receiver*)
			       (car *selection-receiver*)
			       *selection-receiver*)
			   sexpr))
	(user::*emacs-cl-shell*
	 (format t "[[INPUT-STREAM>>")
	 (force-output)
	 (format t " ~S" sexpr)
	 (force-output)
	 (format t "<<INPUT-STREAM]]")		;closing delimiter.
	 (force-output))
	(t (format t ";;; ~A~%" sexpr))))

(defun display-mouse-documentation (pane left-doc middle-doc right-doc)
  (declare (ignore pane))
  (cond (*mouse-doc-reporter*
	 (mouse-report (if (consp *mouse-doc-reporter*)
			   (car *mouse-doc-reporter*)
			   *mouse-doc-reporter*)
		       left-doc middle-doc right-doc))
	(user::*emacs-cl-shell*
	 (format t "[[MESSAGE-STREAM>>")
	 (force-output)
	 (format t "MOUSE:  ~66,1,4,<~A~;~A~;~A~>" left-doc middle-doc right-doc)
	 (force-output)
	 (format t "<<MESSAGE-STREAM]]")
	 (force-output))
	(t nil)))
