;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File:  mcl-draw.lisp
;;;  Author: Heeger/Simoncelli
;;;  Description: 
;;;  Creation Date: 12/93 modified from lv-draw.lisp
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)
(export '())

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; system dependent stuff to draw to mcl panes

(defmethod font ((pane mcl-pane))
  (ccl::view-font pane))

(defmethod font-height ((font-spec list))
  (multiple-value-bind (ascent descent maxwidth leading)
     (ccl::font-info font-spec)
    (+ ascent descent)))

(defmethod string-width ((font-spec list) string)
  (ccl::string-width string font-spec))

(defmethod draw-text ((pane mcl-pane) y x string &rest keys)
  (ccl::move-to pane x y)
  (format pane string))

(defmethod draw-line ((pane mcl-pane) from-y from-x to-y to-x 
                      &rest keys 
                      &key foreground)
  (ccl::move-to pane from-x from-y)
  (ccl::with-focused-view pane
    (if foreground
      (ccl::with-fore-color (convert foreground :encoded-color)
        (ccl::line-to pane to-x to-y))
      (ccl::line-to pane to-x to-y))))

;;; *** fill-p should be a keyword
(defmethod draw-rect ((pane mcl-pane) y0 x0 y1 x1 
                      &rest keys 
                      &key foreground)
  (ccl::with-focused-view pane
    (if foreground
      (ccl::with-fore-color (convert foreground :encoded-color)
        (ccl::paint-rect pane x0 y0 x1 y1))
      (ccl::paint-rect pane x0 y0 x1 y1))))

;;; *** fill-p should be a keyword
(defmethod draw-circle ((pane mcl-pane) y-center x-center radius 
                        &rest keys 
                        &key foreground)
  (let ((left (- x-center radius))
        (top (- y-center radius))
        (right (+ x-center radius))
        (bottom (+ y-center radius)))
    (ccl::with-focused-view pane
      (if foreground
        (ccl::with-fore-color (convert foreground :encoded-color)
          (ccl::paint-oval pane left top right bottom))
        (ccl::paint-oval pane left top right bottom)))))

(defmethod draw-lines ((pane mcl-pane) y0 x0 y1 x1
		       &key
		       line-width line-style
		       x-offset y-offset
                       foreground)
  (declare (type (array fixnum (*)) x0 x1 y0 y1)
	   (fixnum x-offset y-offset))
  (let ((original-pen-state (ccl::pen-state pane)))
    (when line-width (ccl::set-pen-size pane line-width line-width))
    (ccl::with-focused-view pane
      (if foreground
        (ccl::with-fore-color (convert foreground :encoded-color)
          (draw-lines-internal pane y0 x0 y1 x1 x-offset y-offset))
        (draw-lines-internal pane y0 x0 y1 x1 x-offset y-offset)))
    (ccl::set-pen-state pane original-pen-state))
  pane)

(defun draw-lines-internal (pane y0 x0 y1 x1 x-offset y-offset)
  (cond ((or x-offset y-offset)
         (setq x-offset (or x-offset 0))
         (setq y-offset (or y-offset 0))
         (dotimes (i (length y0))
           (ccl::move-to pane 
                         (coerce (+ (aref x0 i) x-offset) '(signed-byte 16))
                         (coerce (+ (aref y0 i) y-offset) '(signed-byte 16)))
           (ccl::line-to pane 
                         (coerce (+ (aref x1 i) x-offset) '(signed-byte 16))
                         (coerce (+ (aref y1 i) y-offset) '(signed-byte 16)))))
        (t (dotimes (i (length y0))
             (ccl::move-to pane 
                           (coerce (aref x0 i) '(signed-byte 16))
                           (coerce (aref y0 i) '(signed-byte 16)))
             (ccl::line-to pane 
                           (coerce (aref x1 i) '(signed-byte 16))
                           (coerce (aref y1 i) '(signed-byte 16)))))))
                 
(defmethod draw-circles ((pane mcl-pane) yarr xarr
			 &key 
                         line-width line-style
			 fill-p radius
                         foreground)
  (declare (type (array fixnum (*)) xarr yarr))
  (let ((original-pen-state (ccl::pen-state pane)))
    (when line-width (ccl::set-pen-size pane line-width line-width))
    (ccl::with-focused-view pane
      (if foreground
        (ccl::with-fore-color (convert foreground :encoded-color)
          (draw-circles-internal pane yarr xarr radius fill-p))
        (draw-circles-internal pane yarr xarr radius fill-p)))
    (ccl::set-pen-state pane original-pen-state))
  pane)

(defun draw-circles-internal (pane yarr xarr radius fill-p)
  (dotimes (i (length yarr))
    (if fill-p
      (ccl::paint-oval pane
                       (coerce (- (aref xarr i) radius) '(signed-byte 16))
                       (coerce (- (aref yarr i) radius) '(signed-byte 16))
                       (coerce (+ (aref xarr i) radius)  '(signed-byte 16))
                       (coerce (+ (aref yarr i) radius) '(signed-byte 16)))
      (ccl::frame-oval pane
                       (coerce (- (aref xarr i) radius) '(signed-byte 16))
                       (coerce (- (aref yarr i) radius) '(signed-byte 16))
                       (coerce (+ (aref xarr i) radius) '(signed-byte 16))
                       (coerce (+ (aref yarr i) radius) '(signed-byte 16))))))
                 
(defmethod draw-squares ((pane mcl-pane) yarr xarr
			 &key 
                         line-width line-style
			 fill-p size
                         foreground)
  (declare (type (array fixnum (*)) xarr yarr))
  (let ((original-pen-state (ccl::pen-state pane)))
    (when line-width (ccl::set-pen-size pane line-width line-width))
    (ccl::with-focused-view pane
      (if foreground
        (ccl::with-fore-color (convert foreground :encoded-color)
          (draw-squares-internal pane yarr xarr size fill-p))
        (draw-squares-internal pane yarr xarr size fill-p)))
    (ccl::set-pen-state pane original-pen-state))
  pane)

(defun draw-squares-internal (pane yarr xarr size fill-p)
  (dotimes (i (length yarr))
    (if fill-p
      (ccl::paint-rect pane
                       (coerce (- (aref xarr i) size) '(signed-byte 16))
                       (coerce (- (aref yarr i) size) '(signed-byte 16))
                       (coerce (+ (aref xarr i) size)  '(signed-byte 16))
                       (coerce (+ (aref yarr i) size) '(signed-byte 16)))
      (ccl::frame-rect pane
                       (coerce (- (aref xarr i) size) '(signed-byte 16))
                       (coerce (- (aref yarr i) size) '(signed-byte 16))
                       (coerce (+ (aref xarr i) size) '(signed-byte 16))
                       (coerce (+ (aref yarr i) size) '(signed-byte 16))))))

(defmethod draw-rects ((pane mcl-pane) y0 x0 y1 x1
			 &key line-width line-style fill-p
                         foreground)
  (declare (type (array fixnum (*)) xarr yarr))
  (let ((original-pen-state (ccl::pen-state pane)))
    (when line-width (ccl::set-pen-size pane line-width line-width))
    (ccl::with-focused-view pane
      (if foreground
        (ccl::with-fore-color (convert foreground :encoded-color)
          (draw-rects-internal pane y0 x0 y1 x1 fill-p))
        (draw-rects-internal pane y0 x0 y1 x1 fill-p)))
    (ccl::set-pen-state pane original-pen-state))
  pane)

(defun draw-rects-internal (pane y0 x0 y1 x1 fill-p)
  (dotimes (i (length y0))
    (if fill-p
      (ccl::paint-rect pane
                       (coerce (min (aref x0 i) (aref x1 i)) '(signed-byte 16))
                       (coerce (min (aref y0 i) (aref y1 i)) '(signed-byte 16))
                       (coerce (max (aref x0 i) (aref x1 i)) '(signed-byte 16))
                       (coerce (max (aref y0 i) (aref y1 i)) '(signed-byte 16)))
      (ccl::frame-rect pane
                       (coerce (min (aref x0 i) (aref x1 i)) '(signed-byte 16))
                       (coerce (min (aref y0 i) (aref y1 i)) '(signed-byte 16))
                       (coerce (max (aref x0 i) (aref x1 i)) '(signed-byte 16))
                       (coerce (max (aref y0 i) (aref y1 i)) '(signed-byte 16))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; draw-graph for mcl-pane

(defmethod draw-graph ((pane mcl-pane)
		       data graph-type
		       graph->frob-y graph->frob-x data->frob-y data->frob-x
		       y-range x-range y-axis x-axis
		       y-tick-step x-tick-step y-tick-length x-tick-length
		       y-tick-gap x-tick-gap y-tick-format-string x-tick-format-string
		       y-label x-label
		       &key
		       (font (ccl::view-font pane))
		       color axis-color
                       line-width plot-symbol fill-symbol-p symbol-size
		       x-offset y-offset)
  (declare (ignore y-offset x-offset y-label x-label))
  (let ((original-pen-state (ccl::pen-state pane))
        (original-clip-region (ccl::clip-region pane)))
    (when line-width (ccl::set-pen-size pane line-width line-width))
    (ccl::with-focused-view pane
      (if axis-color
        (ccl::with-fore-color (convert axis-color :encoded-color)
          (plot-axes-internal pane x-axis y-axis
                              graph->frob-x graph->frob-y
                              x-range x-tick-step x-tick-length x-tick-gap
                              x-tick-format-string 
                              y-range y-tick-step y-tick-length y-tick-gap
                              y-tick-format-string
                              font))
        (plot-axes-internal pane x-axis y-axis
                            graph->frob-x graph->frob-y
                            x-range x-tick-step x-tick-length x-tick-gap
                            x-tick-format-string 
                            y-range y-tick-step y-tick-length y-tick-gap
                            y-tick-format-string
                            font))
      (let ((region (ccl::new-region))
            (x-axis-pos
             (transform-point graph->frob-y 
                              (or x-axis (apply 'clip 0 y-range)))))
        (ccl::set-rect-region region
                              (transform-point graph->frob-x (car x-range))
                              (transform-point graph->frob-y (cadr y-range))
                              (1+ (transform-point graph->frob-x (cadr x-range)))
                              (transform-point graph->frob-y (car y-range)))
        (ccl::set-clip-region pane region)
        (if color
          (ccl::with-fore-color (convert color :encoded-color)
            (plot-data graph-type data pane
                       data->frob-y data->frob-x x-axis-pos
                       plot-symbol fill-symbol-p symbol-size))
          (plot-data graph-type data pane
                       data->frob-y data->frob-x x-axis-pos
                       plot-symbol fill-symbol-p symbol-size))
        (ccl::dispose-region region)))
    (ccl::set-clip-region pane original-clip-region)
    (ccl::set-pen-state pane original-pen-state))
  pane)
  
(defun plot-axes-internal (pane x-axis y-axis
                           graph->frob-x graph->frob-y
                           x-range x-tick-step x-tick-length x-tick-gap
                           x-tick-format-string 
                           y-range y-tick-step y-tick-length y-tick-gap
                           y-tick-format-string
                           font)
  (when x-axis
    (plot-x-axis pane graph->frob-x 
                 (transform-point graph->frob-y x-axis)
                 x-range x-tick-step x-tick-length x-tick-gap
                 x-tick-format-string font))
  (when y-axis
    (plot-y-axis pane graph->frob-y 
                 (transform-point graph->frob-x y-axis)
                 y-range y-tick-step y-tick-length y-tick-gap
                 y-tick-format-string font)))
  


;;; Local Variables:
;;; buffer-read-only: t 
;;; End:
