;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: x-screen.lisp
;;;  Author: Heeger
;;;  Description: 
;;;  Creation Date: 11/91
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)
(export '())

;;; This code requires the LispView interface to X windows (written by
;;; Rmori@Eng.Sun.Com and hmuller@Eng.Sun.Com).
#-:LispView
(error "You must load the LispVIew interface to X windows to run ~
          this version of OBVIUS-~a.  LispView is available from Sun ~
          Microsystems." obv::*obvius-version*)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(eval-when (eval load)
  (pushnew 'initialize-X-window-system *initialization-functions*))

;;; *** Generalize to deal with multiple screens when LispView does the same.
(defun initialize-X-window-system ()
  (make-screen))

;;; NOTE: Must ensure that the lispview:*default-display* has been
;;; created when we make the screen.  Currently, LispView only
;;; supports single displays: so we always use the
;;; lv:*default-display*.

;;; Called by initialize-X-window-system.  Also called by (current-screen)
;;; if there is no *current-pane* or *screen-list*.
(defun make-screen (&rest initargs)
  (let* ((X-display (or (and (typep (getf initargs :X-display) 'lispview:display)
			     (getf initargs :X-display))
			lispview:*default-display*
			(make-instance 'lispview:display)))
	 (depth (lispview:depth (lispview:root-canvas X-display)))
	 ;; *** should we use supported-depths instead of depth?
	 ;;(depth (apply 'max (mapcar 'car (lispview:supported-depths (X-display screen)))))
	 )
    (setf (getf initargs :X-display) X-display)
    (cond ((= depth 1) (apply 'make-instance '1bit-X-screen initargs))
	  ((= depth 8) (apply 'make-instance '8bit-X-screen initargs))
	  ((= depth 24) (apply 'make-instance '24bit-X-screen initargs)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; X-screen object

;;; The values in the gray-lut correspond to sequential indexes into
;;; the gray colors of the screen's actual colormap.  The values in
;;; color-lut are also a meta cmap for color dithering color pictures.
;;; lispview::*default-colormap-allocation* controls the maximum
;;; number of colors that can be allocated from LispView (set in
;;; obvius-window-init.lisp).  We leave a few for graph colors.  The
;;; foreground and background colors will be used for all panes on
;;; this screen, and all bltables.

;;; IMPORTANT NOTE: use find-color to find the colors in a lut to
;;; destroy them.  Use make-color otherwise.  Make-color makes a new
;;; entry in the registered colors table.

;;; *** Would be nice to have X-screen inherit from lispview::display,
;;; but when I tried it, it insisted on allocating another
;;; colormap.  -ES

;;; *** gray-shades is redundant: it is the size of gray-lut (but it
;;; is used by the dialog box that is automatically generated from the
;;; slots).  -DH
(def-simple-class X-screen (screen)
  (X-display
   (foreground :type lispview:color
	       :documentation
	       "Foreground color of screen.  This is used as a foreground \
color for drawing and blt'ing 1bit images to all panes on this screen.  Value \
can be a color name keyword (see the list of X colors in rgb.txt) or a list of \
length three containing RGB values between 0.0 and 1.0")
   (background :type lispview:color
	       :documentation "Background color of screen.  This is used as a background
color for all panes on this screen.  Value can be a color name keyword (see the list
of X colors in rgb.txt) or a list of length three containing RGB values between
0.0 and 1.0"))
  (:default-initargs
     :X-display lispview:*default-display*
     :background :black
     :foreground :white))

(def-simple-class 1bit-X-screen (X-screen)
  ())

(def-simple-class 24bit-X-screen (X-screen)
  ())

(def-simple-class 8bit-X-screen (X-screen)
  (gray-lut
   color-lut
   pseudo-color-lut
   registered-colors			;array of all registered lv:color objects
   (gray-shades :type (integer 2 255)
		:documentation
		"Determines the number of gray levels allocated in the colormap.")
   (gray-gamma :type (float 0.1 10.0)
	       :documentation
	       "Determines the gamma correction factor of the colormap.")
   (gray-dither :initform 64 :type (or (member nil t) number)
		:documentation
		"Determines whether pictures are dithered into the gray shades in \
the colormap.  If t, dither.  If nil, just quantize.  If a number, the pictures \
will be dithered if the value of gray-shades is less than this number.")
   (rgb-bits :type cons
	     :documentation
	     "List of three integers that determine the number of bits for \
dithering color pictures.")
   (pseudo-colors :type (integer 0 255)
		  :documentation
		  "Determines the number of pseudo colors allocated in the colormap.")
   (pseudo-color-function :documentation
			  "Function used to allocate pseudo-colors, argument is an integer, returns a list of rgb values.")
   )
  (:default-initargs
     :gray-shades 128
     :gray-gamma 0.9
     :rgb-bits '(2 2 2)
     :pseudo-colors 7
     :pseudo-color-function #'pseudo-color-ramp
     ))

;;; Add to screen list.
(defmethod initialize-instance ((screen X-screen) &rest initargs)
  (setq *screen-list* (nconc *screen-list* (list screen)))
  (apply #'call-next-method screen initargs)
  screen)

;;; Register the reserved-colors, and then call shared-initialize to
;;; allocate the gray-lut and color-lut.
(defmethod initialize-instance ((screen 8bit-X-screen) &rest initargs)
  (setf (getf initargs :registered-colors)
	(or (getf initargs :registered-colors)
	    (make-array 256)))
  (apply #'call-next-method screen initargs)
  (let* ((cmap (lispview:default-colormap (X-display screen)))
	 (reserved-colors (lispview:reserved-colormap-colors (X-display screen))))
    ;; *** problem here, not all colors get registered ***
    ;; Register reserved colormap cells into the table of registered colors.
    (loop for i from 0 below (length reserved-colors)
	  for col = (aref reserved-colors i)
	  do
	  (make-color :colormap cmap :pixel (lispview:pixel col)
		      :if-not-found :warn))
    screen))

;;; *** If set both gray-shades and gray-gamma, then set-gray-lut gets
;;; called twice.

;;; This is called by initialize-instance and reinitialize-instance.
(defmethod shared-initialize ((screen X-screen) slot-names &rest initargs)
  (declare (ignore slot-names))
  ;; Fill slots according to initargs (some reset below).
  (call-next-method)			
  ;; Register the foreground and background colors (weird to use the
  ;; setf methods to reset the slot, but it is correct).
  (when (getf initargs :foreground)
    (setf (foreground screen) (foreground screen)))
  (when (getf initargs :background)
    (setf (background screen) (background screen)))
  screen)

;;; This is called by initialize-instance and reinitialize-instance.
(defmethod shared-initialize ((screen 8bit-X-screen) slot-names &rest initargs)
  (declare (ignore slot-names))
  ;; Fill slots according to initargs (some reset below).
  (call-next-method)			
  ;; Allocate grays and register them.  Weird to use the
  ;; setf method to reset the slot, but it is correct (invokes set-gray-lut).
  (when (or (getf initargs :gray-gamma) (getf initargs :gray-shades))
    (setf (gray-shades screen) (gray-shades screen)))
  ;; Allocate colors and register them.
  (when (getf initargs :rgb-bits)
    (set-color-lut screen))
  (when (getf initargs :pseudo-colors)
    (allocate-pseudo-colors screen))
  screen)

(defmethod settable-parameters ((class-name (eql 'X-screen)))
  (append '(foreground background)
	  (call-next-method)))

(defmethod settable-parameters ((class-name (eql '8bit-X-screen)))
  (append '(gray-shades gray-gamma gray-dither
	    rgb-bits pseudo-colors pseudo-color-function)
	  (call-next-method)))

;;; *** Should probably get rid of all the setf methods (below) and just force use of
;;; reinitialize-instance (i.e., make the accessor a reader only, not a writer).

(defmethod (setf gray-shades) (num (screen 8bit-X-screen))
  (let ((X-display (X-display screen)))
    (setq num (clip num 2 (- (lispview:colormap-length (lispview:default-colormap X-display))
			     (length (lispview:reserved-colormap-colors X-display))
			     (* (expt 2 (first (rgb-bits screen)))
				(expt 2 (second (rgb-bits screen)))
				(expt 2 (third (rgb-bits screen)))))))
    (setf (slot-value screen 'gray-shades) num)
    (set-gray-lut screen :old-lut (gray-lut screen))))

(defmethod (setf gray-gamma) (num (screen 8bit-X-screen))
  (setf (slot-value screen 'gray-gamma) num)
  (set-gray-lut screen :old-lut (gray-lut screen)))

(defmethod (setf rgb-bits) (list (screen 8bit-X-screen))
  (unless (and (plusp (first list)) (plusp (second list)) (plusp (third list)))
    (error "Must be at least 1 bit for each r, g, and b."))
  (when (> (apply '+ list) 7)
    (error "Can not allocate more than 7 bits for color."))
  (setf (slot-value screen 'rgb-bits) list)
  (set-color-lut screen))

(defmethod (setf pseudo-colors) (num (screen 8bit-X-screen))
  (setf (slot-value screen 'pseudo-colors) num)
  (allocate-pseudo-colors screen :old-lut (pseudo-color-lut screen)))

(defmethod (setf pseudo-color-function) (function (screen 8bit-X-screen))
  (unless (fboundp function)
    (error "~a must be a symbol with a function binding" function))
  (setf function (symbol-function function))
  (setf (slot-value screen 'pseudo-color-function) function)
  (allocate-pseudo-colors screen :old-lut (pseudo-color-lut screen)))

;;; Allow foreground/background to be set to a symbol (name of a color).
(defmethod (setf background) ((color-name symbol) (screen X-screen))
  (let ((clr (make-color :name color-name :if-not-exact :error :display (X-display screen))))
    (when clr (call-next-method clr screen))))

(defmethod (setf foreground) ((color-name symbol) (screen X-screen))
  (let ((clr (make-color :name color-name :if-not-exact :error :display (X-display screen))))
    (when clr (call-next-method clr screen))))

;;; Enforce situation that all panes get their foreground and
;;; background from the screen they live on. 
(defmethod (setf foreground) (color (screen X-screen))
  (when (typep (slot-value screen 'foreground) 'lv:color)
    (setf (lv:status (slot-value screen 'foreground)) :destroyed))
  (setf (lispview:foreground (lispview:graphics-context (X-display screen))) color)
  (dolist (p (pane-list screen))
    (setf (foreground p) color)) ;must use setf here to achieve proper side-effects.
  (setf (slot-value screen 'foreground) color)
  color)

(defmethod (setf background) (color (screen X-screen))
  (declare (special *status-reporter* *control-panel*))
  (when (typep (slot-value screen 'background) 'lv:color)
    (setf (lv:status (slot-value screen 'background)) :destroyed))
  (setf (lispview:background (lispview:graphics-context (X-display screen))) color)
  (dolist (p (pane-list screen))
    (setf (background p) color) ;must use setf here to achieve proper side-effects.
    (draw-pane p :clear t))
  (setf (slot-value screen 'background) color)
  (when (and *status-reporter* *control-panel*)
    (setf (lispview:background *control-panel*) color))
  color)

#|
;;; modified 2/93 to get rid of reference to retain-bitmap. -DH

(defmethod (setf background) (color (screen X-screen))
  (declare (special *status-reporter* *control-panel*))
  (when (typep (slot-value screen 'background) 'lv:color)
    (setf (lv:status (slot-value screen 'background)) :destroyed))
  (setf (lispview:background (lispview:graphics-context (X-display screen))) color)
  (dolist (p (pane-list screen))
    (setf (background p) color) ;must use setf here to achieve proper side-effects.
    (draw-pane p :clear nil)
    (loop for pic in (picture-stack p) do
	  (when (and (typep pic 'drawing) (retain-bitmap pic))
	    (setf (current pic) nil)
	    (when (eq pic (car (picture-stack p)))
	      (refresh p)))))
  (setf (slot-value screen 'background) color)
  (when (and *status-reporter* *control-panel*)
    (setf (lispview:background *control-panel*) color))
  color)
|#

(defmethod destroy ((screen X-screen) &key &allow-other-keys)
  (when (find *current-pane* (pane-list screen))
    (setq *current-pane* nil))
  (loop for pane in (pane-list screen) do
	(destroy pane))
  (setq *screen-list* (remove screen *screen-list*)))

(defmethod destroy ((screen 8bit-X-screen) &key &allow-other-keys)
  (declare (special *screen-list*))
  (let* ((gray-lut (gray-lut screen))
	 (color-lut (color-lut screen))
	 (cmap (lispview:default-colormap (X-display screen))))
    ;; If gray-lut exists, de-allocate colors.
    (when gray-lut	      
      (loop for i from 0 below (length gray-lut)
	    for color = (lispview:find-color
			 :colormap cmap :pixel (aref gray-lut i) :if-not-found nil)
	    do
	    (setf (lispview:status color) :destroyed)))
    ;; If color-lut exists, de-allocate colors.
    (when color-lut	      
      (loop for i from 0 below (length color-lut)
	    for color = (lispview:find-color
			 :colormap cmap :pixel (aref color-lut i) :if-not-found nil)
	    do
	    (setf (lispview:status color) :destroyed))))
  (call-next-method screen))

(defmethod print-object ((screen x-screen) stream)
  (with-slots (X-display) screen
    (format stream "#<X-screen ~A:~A.~A>"
	    (lispview:host X-display)
	    (lispview:server X-display)
	    (lispview:screen X-display))))

(defmethod depth ((screen X-screen))
  (lispview:depth (lispview:root-canvas (X-display screen))))

(defmethod depth ((screen 8bit-X-screen))
  8)

(defmethod depth ((screen 1bit-X-screen))
  1)

(defmethod depth ((screen 24bit-X-screen))
  24)

(defmethod x-dim ((screen X-screen))
  (lispview:region-width
   (lispview:bounding-region (lispview:root-canvas (X-display screen)))))

(defmethod y-dim ((screen X-screen))
  (lispview:region-height
   (lispview:bounding-region (lispview:root-canvas (X-display screen)))))

(defmethod dimensions ((screen X-screen))
  (let ((br (lispview:bounding-region (lispview:root-canvas (X-display screen)))))
    (list (lispview:region-height br)
	  (lispview:region-width br))))

;;; *** This really only needs to be done on color-pictures, grays, and
;;; flipbooks of grays.
(defmethod set-not-current ((screen 8bit-X-screen))
  (loop for pane in (pane-list screen) do
	(loop for pic in (picture-stack pane) do
	      (set-not-current pic))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Allocates grays in the X colormap.  Makes internal lookup table
;;; to find allocated grays.  Destroys old colors.
(defmethod set-gray-lut ((screen 8bit-X-screen)
			 &key
			 (gray-size (gray-shades screen))
			 (gray-gamma (gray-gamma screen))
			 (old-lut (gray-lut screen)))
  (let* ((cmap (lispview:default-colormap (X-display screen)))
	 (lut (make-array gray-size :element-type '(unsigned-byte 8)))
	 (cmap-overflow nil))		;flag for full cmap.
    ;;If old-lut exists, de-allocate non-reserved colors
    (when old-lut			
      (loop for i from 0 below (length old-lut)
	    for color = (lispview:find-color
			 :colormap cmap :pixel (aref old-lut i) :if-not-found nil)
	    do (when color (setf (lispview:status color) :destroyed))))
    (format t ";;; (Re-)allocating ~A grayscales...~%" gray-size)
    (loop for i from 0 below gray-size
	  for intensity = (float (expt (/ i (1- gray-size)) gray-gamma))
	  for color = (make-color
		       :colormap cmap :hue 0.0 :saturation 0.0 :intensity intensity
		       :if-not-exact nil
		       :display (X-display screen))
	  until (setq cmap-overflow (and (not color) (> i 2) i))
	  do
	  (unless (eq (lispview:status color) :realized)
	    (setf (lispview:status color) :realized)) ;make sure it is :realized
	  (setf (aref lut i) (lispview:pixel color)))
    (if cmap-overflow
	(progn
	  (warn "Failed to allocate ~A gray shades!" gray-size)
	  (set-gray-lut screen
			:gray-size cmap-overflow
			:gray-gamma gray-gamma
			:old-lut (make-array cmap-overflow
					     :element-type '(unsigned-byte 8)
					     :displaced-to lut)))
	(setf (slot-value screen 'gray-shades) gray-size
	      (slot-value screen 'gray-lut) lut)))
  (set-not-current screen))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Allocates colors in the X colormap for dithering color pictures.  Makes
;;; internal lookup table to find allocated colors.  Destroys old colors.
(defmethod set-color-lut ((screen 8bit-X-screen))
  (let* ((red-levels (expt 2 (first (rgb-bits screen))))
	 (green-levels (expt 2 (second (rgb-bits screen))))
	 (blue-levels (expt 2 (third (rgb-bits screen))))
	 (old-lut (color-lut screen))
	 (lut (make-array (* red-levels green-levels blue-levels)
			  :element-type '(unsigned-byte 8)))
	 (cmap-overflow nil)
	 (cmap (lispview:default-colormap (X-display screen))))
    ;;If old-lut exists, de-allocate non-reserved colors
    (when old-lut
      (loop for i from 0 below (length old-lut)
	    for color = (lispview:find-color
			 :colormap cmap :pixel (aref old-lut i) :if-not-found nil)
	    do (when color (setf (lispview:status color) :destroyed))))
    (format t ";;; (Re-)allocating colors, ~A by ~A by ~A...~%"
	    red-levels green-levels blue-levels)
    (loop for i from 0 below (total-size lut)
	  for rval = (float (/ (mod i red-levels) (1- red-levels)))
	  for gval = (float (/ (mod (floor i red-levels)
				    green-levels)
			       (1- green-levels)))
	  for bval = (float (/ (mod (floor i (* red-levels green-levels))
				    blue-levels)
			       (1- blue-levels)))
	  for color = (make-color :colormap cmap :red rval :green gval :blue bval
				  :if-not-exact nil
				  :display (X-display screen))
	  until (setq cmap-overflow (and (not color) i))
	  do
	  ;;(print-db rval gval bval)
	  (unless (eq (lispview:status color) :realized)
	    (setf (lispview:status color) :realized)) ;make sure it is :realized
	  (setf (aref lut i) (lispview:pixel color)))
    (if cmap-overflow
	(warn "Failed to allocate colors, only got ~a" cmap-overflow)
	(setf (color-lut screen) lut)))
  (set-not-current screen))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Allocates colors in the X colormap for pseudo-color pictures.

(defmethod allocate-pseudo-colors
    ((screen 8bit-X-screen)
     &key old-lut)
  (let* ((num-pseudo-colors (pseudo-colors screen))
	 (lut (make-array num-pseudo-colors :element-type '(unsigned-byte 8)))
	 (pseudo-color-function (pseudo-color-function screen))
	 (cmap-overflow nil)
	 (cmap (lispview:default-colormap (X-display screen))))
    ;;If old-lut exists, de-allocate non-reserved colors
    (when old-lut
      (loop for i from 0 below (length old-lut)
	    for color = (lispview:find-color
			 :colormap cmap :pixel (aref old-lut i) :if-not-found nil)
	    do (when color (setf (lispview:status color) :destroyed))))
    (format t ";;; (Re-)allocating ~a pseudo colors...~%" num-pseudo-colors)

    (loop for i from 0 below (total-size lut)
	  for rgb-vals = (funcall pseudo-color-function i num-pseudo-colors)
	  for rval = (first rgb-vals)
	  for gval = (second rgb-vals)
	  for bval = (third rgb-vals)
	  for color = (make-color :colormap cmap
				  :red (float rval)
				  :green (float gval)
				  :blue (float bval)
				  :if-not-exact nil
				  :display (X-display screen))
	  until (setq cmap-overflow (and (not color) i))
	  do
	  (unless (eq (lispview:status color) :realized)
	    (setf (lispview:status color) :realized)) ;make sure it is :realized
	  (setf (aref lut i) (lispview:pixel color)))
    (when cmap-overflow
      (warn "Failed to allocate colors, only got ~a" cmap-overflow))
    (setf (slot-value screen 'pseudo-color-lut) lut))
  (set-not-current screen))

;;; Example pseudo-color-function:
(defun pseudo-color-ramp (i size)
  (let* ((rval (/ i (- size 1)))
	 (gval (- 1 rval))
	 (bval (* 1/2 (- 1 (abs (- (* 2 rval) 1)))))
	 )
    ;;(print-db rval gval bval)
    (list rval gval bval)))



;;; Local Variables:
;;; buffer-read-only: t 
;;; fill-column: 79
;;; End:
