;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: x-control-panel.lisp
;;;  Author: Eero Simoncelli/Celine Fung
;;;  Description: Code to create the OBVIUS control panel.
;;;  Creation Date: Fall, 1990
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)
(export '(make-control-panel destroy-control-panel))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; NOTES & COMMENTS:

;;; We create a control panel for OBVIUS containing a status line, a
;;; menu button bar, and a mouse documentation bar.

;;; The menu bar contains menu buttons for the following:
;;; 1) parameters: current picture, picture defaults, vbl defaults, global, 
;;;    current pane, pane defaults
;;; 2) statistics
;;; 3) unary point operations
;;; 4) binary point operations (combine with #5?)
;;; 5) non-point ops: gauss-out, gauss-in, crop, paste, filtering
;;; 6) synthetic images.

;;; The function menus are updated dynamically from a set of global
;;; variables: *obvius-stats-functions* *obvius-arith-functions*
;;; *obvius-filter-functions* *obvius-compare-functions*
;;; *obvius-geom-functions* *obvius-synth-functions*
;;; *obvius-matrix-functions* *obvius-misc-functions*.

;;; *** The function dialogs need a rewrite.  Optional/key args should
;;; be able to depend on previous arg values, and optional args should
;;; be filled in if there are other optional or keywords filled in
;;; after them.  Would also be nice to allow user to adjust default
;;; values of optional/key args (use defadvice?).

;;; *** statistics menu is worthless!  Should pop up a statistics info
;;; window with a "compute" button that computes all of the stats and
;;; displays them.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Need to keep a reference to the window so we can destroy it if necessary!
(defvar *control-panel* nil)

(def-simple-class control-panel (lispview:base-window) 
  ()
  (:default-initargs
    :keyboard-focus-mode :passive
    :show-resize-corners nil
    :icon (make-standard-pane-icon)))

;;; Ugly function that makes the control panel and all of its widgets.
(defun make-control-panel (&rest keys &key left bottom right top)
  (declare (ignore  TOP RIGHT BOTTOM LEFT))
  (cond ((and (typep *control-panel* 'control-panel)
	      (not (eq (lispview:status *control-panel*) :destroyed)))
	 (setf (lispview:mapped *control-panel*) t)
	 (lispview:expose *control-panel*)
	 *control-panel*)
	(t
	 (format t ";;; Creating OBVIUS control panel ...~%")
	 (let* ((width 500)		;empirically derived!!
		(menu-width 420)
		(screen (current-screen))
		(control-win (apply 'make-instance
				    'control-panel
				    :height 110 :width width
				    :border-width 1 :mapped nil
				    :label "obvius control panel"
				    :foreground (foreground screen)
				    :background (background screen)
				    :display (X-display screen)
				    keys)))
	   (unwind-protect
		(let* ((STATUS-PANEL (make-instance 'lispview:panel
						    :parent control-win :top 3 :left 3
						    :height 22 :width 1024))
		       (status-title (make-instance 'lispview:message
						    :parent status-panel :width 60
						    :label "Status:" :label-bold t))
		       (status-message (make-instance 'lispview:message
						      :parent status-panel :left 60
						      :label "" :label-bold nil))
		       (BUTTON-PANEL1 (make-instance 'lispview:panel
						     :parent control-win :top 27 :left 3 
						     :height 26 :width (- width 7)))
		       (parameters-button (make-instance 'lispview:menu-button
							 :parent button-panel1
							 :left (floor (- width menu-width) 2)
							 :label "Parameters"
							 :menu (make-parameters-menu control-win)))
		       (stats-button
			(make-instance 'lispview:menu-button
				       :parent button-panel1
				       :label "Statistics"
				       :menu (make-function-menu
					      "Statistics" '*obvius-stats-functions*
					      control-win)))
		       (arith-button
			(make-instance 'lispview:menu-button
				       :parent button-panel1
				       :label "Arith Ops"
				       :menu (make-function-menu
					      "Arith Ops" '*obvius-arith-functions*
					      control-win)))
		       (filter-button
			(make-instance 'lispview:menu-button
				       :parent button-panel1
				       :label "Filter Ops"
				       :menu (make-function-menu
					      "Filter Ops" '*obvius-filter-functions*
					      control-win)))
		       (BUTTON-PANEL2 (make-instance 'lispview:panel
						     :parent control-win :top 55 :left 3 
						     :height 26 :width (- width 7)))
		       (geom-button
			(make-instance 'lispview:menu-button
				       :parent button-panel2
				       :left (floor (- width menu-width) 2)
				       :label "Geom Ops"
				       :menu (make-function-menu
					      "Geom Ops" '*obvius-geom-functions*
					      control-win)))
		       (compare-button
			(make-instance 'lispview:menu-button
				       :parent button-panel2
				       :label "Compare"
				       :menu (make-function-menu
					      "Comparison Ops" '*obvius-compare-functions*
					      control-win)))
		       (synth-button
			(make-instance 'lispview:menu-button
				       :parent button-panel2
				       :label "Synth"
				       :menu (make-function-menu
					      "Synthetic Images" '*obvius-synth-functions*
					      control-win)))
		       (matrix-button
			(make-instance 'lispview:menu-button
				       :parent button-panel2
				       :label "Matrix"
				       :menu (make-function-menu
					      "Matrix Ops" '*obvius-matrix-functions* control-win)))
		       (misc-button (make-instance 'lispview:menu-button
						   :parent button-panel2
						   :label "Misc"
						   :menu (make-misc-menu control-win)))
		       (MOUSE-DOC-PANEL (make-instance 'lispview:panel
						       :parent control-win :top 83 :left 3
						       :height 22 :width 1024))
		       (mouse-doc-title (make-instance 'lispview:message
						       :parent mouse-doc-panel :width 60
						       :label "Mouse:" :label-bold t))
		       (mouse-doc-message (make-instance 'mouse-doc-message
							 :parent mouse-doc-panel :left 60
							 :width (- width 60)
							 :label "" :label-bold nil)))
		  (declare (ignore MOUSE-DOC-TITLE MISC-BUTTON MATRIX-BUTTON
				   SYNTH-BUTTON COMPARE-BUTTON GEOM-BUTTON FILTER-BUTTON
				   ARITH-BUTTON STATS-BUTTON PARAMETERS-BUTTON
				   STATUS-TITLE))

		  (setq *status-reporter* status-message)
		  (setq *mouse-doc-reporter* mouse-doc-message))
	     (setf (lispview::mapped control-win) t)
	     (setq *control-panel* control-win))
	   control-win))))		;ensure that we have a pointer to this!

(defun destroy-control-panel (&optional (cp *control-panel*))
  (setf (lv::status cp) :destroyed))

(defmethod (setf lv::status) ((val (eql :destroyed)) (cp control-panel))
  (setq *control-panel* nil
	*status-reporter* nil
	*mouse-doc-reporter* nil)
  (call-next-method))

;;;; Functions used by the rest of obvius (see emacs.lisp)

;;; This is called by status-message to display messages to the user.
(defun report (message-object message-string)
  (setf (lispview::label message-object) message-string))

;;; *** Message should know how big it is and position the doc appropriately.
(defun mouse-report (message-object left-doc middle-doc right-doc)
  (declare (ignore message-object left-doc middle-doc right-doc))
  (let ((message-string (format nil (doc-format-string message-object)
				left-doc middle-doc right-doc)))
    (setf (lispview::label message-object) message-string)))

(def-simple-class mouse-doc-message (lispview:message)
  (doc-format-string))			;justification format string

(defmethod initialize-instance ((m mouse-doc-message) &rest initargs)
  (call-next-method)
  (let* ((font (lv:font (lv:graphics-context (lv:display m))))
	 (char-width (lispview:char-width (lv:max-char-metrics font)))
	 (mincol (round (getf initargs :width 400) char-width)))
    (setf (doc-format-string m)
	  (format nil "~~~A,1,4<~~A~~;~~A~~;~~A~~>" mincol))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Automatic generation of standard menus

;;; *** The Current picture dialog currently doesn't do a refresh.  It
;;; should also have an "apply" button.
(defun make-parameters-menu (&optional owner)
  (let ((cpic-item (make-instance
		    'lispview:command-menu-item
		    :label "Current picture ..."
		    :command #'(lambda ()
				 (let* ((pane *current-pane*)
					(pic (when pane (car (picture-stack pane)))))
				   (when pic
				     (make-slot-value-dialog
				      pic :slot-names (settable-parameters pic)
				      :label "Picture slot dialog"
				      :update-function 'picture-slot-update-function))))))
	(cscreen-item (make-instance
		       'lispview:command-menu-item
		       :label "Current screen ..."
		       :command #'(lambda ()
				    (make-slot-value-dialog
				     (current-screen)
				     :slot-names (settable-parameters (current-screen))
				     :label "Screen slot dialog"
				     :update-function 'screen-slot-update-function))))
	(global-item (make-instance
		      'lispview:command-menu-item
		      :label "Globals ..."
		      :command #'(lambda ()
				   (make-global-parameter-dialog
				    (set-difference (obvius-parameter-list)
						    '(*obvius-version* *current-pane*))
				    :label "obvius global parameters")))))
    (make-instance
     'lispview:menu
     :label "Parameters"
     :owner owner :default nil :pushpin t
     :choices
     #'(lambda ()
	 ;; these are made on-the-fly:
	 (let ((pic-menu (make-instance
			  'lispview:submenu-item
			  :label "Picture defaults:"
			  :menu (make-instance
				 'lispview:menu
				 :default nil
				 :choices (make-class-defaults-menu-items
					   (find-class 'picture)))))
	       (vbl-menu (make-instance
			  'lispview:submenu-item
			  :label "Viewable defaults:"
			  :menu (make-instance
				 'lispview:menu
				 :default nil
				 :choices (make-class-defaults-menu-items
					   (find-class 'viewable))))))
	   (list cpic-item cscreen-item pic-menu vbl-menu global-item))))))

(defun make-class-defaults-menu-items (root-class &optional include-root)
  (loop with classes = (find-all-subclasses root-class)
	for class in (if include-root classes (cdr classes))
	for class-name = (clos:class-name class)
	collect
	(make-instance
	 'lispview:command-menu-item
	 :label (mk-dlg-name class-name)
	 :command
	 (let ((class-nm class-name))	;rebind for lexical closure!
	   #'(lambda ()
	       (make-slot-default-dialog
		class-nm
		:slot-names (settable-parameters class-nm)
		:exit-function 'unmapping-exit-function
		:label"obvius slot default dialog"))))))

;;; *** A little broken, since e.g. it will recompute the gray-lut
;;; twice if the user has altered both the gray-shades and gray-gamma
;;; parameters.  Should call reinitialize-instance!
(defun screen-slot-update-function (widgets)
  (loop with val
	with errorp
	with any-errors
	for widget in widgets
	for screen = (object widget)
	do
	(multiple-value-setq (val errorp) (get-widget-value widget))
	(when errorp (set-widget-value widget val))
	(unless (equal val (get-object-value widget))
	  (eval `(setf (,(slot-name widget) ,screen) ,val)))
	(setq any-errors (or errorp any-errors))
	finally (return any-errors)))

;;; Lists of functions or label-function pairs
(defvar *obvius-stats-functions*
  '(MINIMUM MAXIMUM RANGE
    MINIMUM-LOCATION MAXIMUM-LOCATION
    MEAN VARIANCE SKEW KURTOSIS
    ENTROPY make-histogram))

(defvar *obvius-arith-functions*
  '(ADD SUB MUL DIV
    SQUARE SQUARE-ROOT
    LINEAR-XFORM 
    ABS-VALUE NEGATE
    POWER EXP. NATURAL-LOGARITHM
    round. truncate. floor. 
    QUANTIZE CLIP
    POINT-OPERATION PERIODIC-POINT-OPERATION
    MAGNITUDE SQUARE-MAGNITUDE
    ZERO-CROSSINGS))
    
(defvar *obvius-filter-functions*
  '(APPLY-FILTER EXPAND-FILTER
    BLUR GAUSS-IN GAUSS-OUT
    FFT HILBERT-TRANSFORM))

(defvar *obvius-compare-functions*
  '(LESS-THAN GREATER-THAN EQUAL-TO
    POINT-MINIMUM POINT-MAXIMUM
    SQUARE-ERROR ABS-ERROR 
    MEAN-SQUARE-ERROR MEAN-ABS-ERROR
    MAX-ABS-ERROR
    CORRELATE))

(defvar *obvius-geom-functions*
  '(PASTE SIDE-BY-SIDE CROP MAKE-SLICE
    FLIP-X FLIP-Y CIRCULAR-SHIFT TRANSPOSE
    rotate resample
    UPSAMPLE DOWNSAMPLE))

;;; *** All these currently break because of keyword defaults.  -DH and EPS
;;; Make-Synthetic-Image Make-Impulse Make-Ramp Make-1-Over-R
;;; Make-R-Squared Make-Atan Make-Disc
(defvar *obvius-synth-functions*
  '(Make-Grating Make-Sin-Grating Make-Square-Grating Make-Saw-Tooth-Grating
    Make-Zone-Plate Make-Pinwheel 
    Make-Uniform-Noise Make-Gaussian-Noise Make-Random-Dots Make-Fractal))

(defvar *obvius-matrix-functions*
  '(matrix-mul matrix-transpose-mul matrix-mul-transpose
    cross-product outer-product
    matrix-transpose  matrix-inverse
    normalize identity-matrix diagonal-matrix symmetric-p
    svd principal-components condition-number
    qr-decompose
    basis-for-left-nullspace
    determinant))

;;; *** Break becauses of keyword defaults.  -DH and EPS
;;; purge!
(defvar *obvius-misc-functions*
  '(print-values 
    load-image save-image hardcopy
    new-pane
    ogc))

(defun make-misc-menu (&optional owner)
  (make-instance
   'lispview:menu
   :label "Misc" :owner owner :pushpin t :default nil
   :choices
   #'(lambda ()
       (let* ((modules (loop for plist = *obvius-module-plist* then (cddr plist)
			     until (null plist)
			     collect (car plist))))
	 (setq modules (remove-if #'(lambda (m) (member m *obvius-features*))
				  modules))
	 (setq modules (sort modules #'string<))
	 (setq modules (mapcar #'(lambda (m)
				   (make-instance
				    'lv::command-menu-item
				    :label (mk-dlg-name m nil)
				    :command 
				    (let ((expr `(obv-require ,m)))
				      #'(lambda ()
					  (push-onto-eval-queue
					   `(progn 
					     (format t "~S~%" ',expr)
					     (print-top-level-values
					      (multiple-value-list ,expr))))))))
			       modules))
	 (cons (make-instance
		'lv:submenu-item
		:label "Load module:"
		:menu (make-instance 'lv:menu :choices modules))
	       (loop for op in *obvius-misc-functions*
		     collect
		     (make-instance 'lv::command-menu-item
				    :label (mk-dlg-name op)
				    :command
				    (let ((fn op)) ;rebind for lexical closure
				      #'(lambda ()
					  (make-function-dialog fn))))))))))

(defun mk-dlg-name (sym &optional (suffix " ..."))
  (let ((string (if (keywordp sym) (format nil "~S" sym) (format nil "~A" sym))))
    (if suffix
	(concatenate 'string (string-downcase string) suffix)
	(string-downcase string))))
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; FUNCTION ARGUMENT DIALOG BOXES

;;; Func-list-symbol is a globally bound symbol containing a list of
;;; functions or label-function pairs.
(defun make-function-menu (label func-list-symbol &optional owner)
  (make-instance
   'lispview:menu
   :label label :owner owner :default nil :pushpin t 
   :choices
   #'(lambda ()
       (loop for op in (symbol-value func-list-symbol)
	     for name = (if (consp op) (car op) (mk-dlg-name op))
	     collect
	     (make-instance
	      'lispview:command-menu-item
	      :label name
	      :command
	      (let ((fn (if (consp op) (cadr op) op))) ;rebind for lexical closure
		#'(lambda () (make-function-dialog fn))))))))

;;; Only allow one at a time:
(defvar *current-function-dialog* nil)

(def-simple-class function-dialog (lispview:popup-window)
  ((func :type symbol)
   arg-fields)
  (:default-initargs :show-resize-corners nil))

;;; Put up a dialog allowing the user to enter the args to the
;;; function (including all optional and keyword args.
;;; *** should save these in a hash table (see
;;; *slot-default-dialogs*), and use the unmapping-exit-function.
(defun make-function-dialog
    (func &key
	  (update-function 'function-dialog-update-function)
	  (update-args (list func))
	  (exit-function 'destroying-exit-function))
  (let ((width 300)
	(all-button-width 180)
	(arglist (arglist func))
	arg-fields panel dialog)
    (multiple-value-bind (required-arg optional-arg key-arg)
	(make-parsed-arglists arglist)
      (setq dialog (make-instance 'function-dialog
				  :func func
				  :label "obvius function dialog"
				  :pushpin nil
				  ;;*** recompute height!  don't know doc length!
				  :height (* (+ (1+ (length required-arg))
						(1+ (length optional-arg))
						(1+ (length key-arg)) 4) 27)
				  :width width :top 140 :left 0))
      (setq panel (make-instance 'lispview:panel :parent dialog))
      (let* ((func-message (make-instance 'lispview:message :parent panel
					  :label (format nil "~S:  " func)))
	     (br (lispview::bounding-region func-message)))
	(make-instance 'lispview:message :parent panel :label ""
		       :top (+ 10 (lispview::region-bottom br))))
      ;;(make-instance 'lispview:message :parent panel
      ;; :label (or (documentation func 'function) "(no documentation)"))
      (setq arg-fields
	    (loop for arglist in (list required-arg optional-arg key-arg)
		  with label = '("Required Arguments:" "Optional Arguments:"
				 "Keyword Arguments:")
		  with field-type = '(required-arg-field
				      optional-arg-field
				      key-arg-field)
		  if (null arglist)
		  do (setq label (cdr label))
		  (setq field-type (cdr field-type))
		  else append
		  (loop initially
		    (make-instance 'lispview:message :parent panel
				   :label (car label) :label-bold t)
		    (setq label (cdr label))
		    for item in arglist
		    collect
		    (make-instance (car field-type) :parent panel
				   :arg-name (get-arg-name item)
				   :arg-default (get-arg-default item))
		    finally (setq field-type (cdr field-type)))))
      (setf (arg-fields dialog) arg-fields)
      (when arg-fields
	(setf (lispview:keyboard-focus panel) (first arg-fields)))
      (make-instance 'lispview:command-button :parent panel
		     :left (round (- width all-button-width) 2)
		     :label "Execute"
		     :command
		     #'(lambda () (apply update-function arg-fields update-args)))
      (make-instance 'lispview:command-button :parent panel
		     :label "Cancel"
		     :command #'(lambda ()
				  ;;(remove-interest-from-all-panes mouse-interest)
				  (when (eq dialog *selection-receiver*)
				    (setf *selection-receiver* nil))
				  (funcall exit-function dialog))))
    ;; Only allow one function dialog to exist.
    (when *current-function-dialog*
      (setf (lispview::status *current-function-dialog*) :destroyed))
    (setq *current-function-dialog*
	  (setq *selection-receiver* dialog))))

(defmethod (setf lv::status) ((val (eql :destroyed)) (fd function-dialog))
  (setf *current-function-dialog* nil)
  (when (eq *selection-receiver* fd)
    (setq *selection-receiver* nil))
  (call-next-method))

(defun function-dialog-update-function (arg-fields func)
  (loop with errorp
	with val
	for widget in arg-fields
	do 
	(multiple-value-setq (val errorp) (make-value-list widget))
	(when errorp (return t))	;indicate error
	append val into arg-list
	finally
	(let ((expr (append (list func) arg-list)))
	  (push-onto-eval-queue
	   `(progn 
             (format t "~S~%" ',expr)
	     (print-top-level-values
	      (multiple-value-list ,expr))))
	  (return nil))))		;no error

(defun get-arg-name (item)
  (if (listp item) (car item) item))

(defun get-arg-default (item)
  (if (listp item) (cadr item) :no-default))

(defun make-parsed-arglists (arglist)
  (let ((arg-type 'required-arg))
    (loop for arg in arglist
	  when (equal arg '&optional)
	    do (setq arg-type 'optional-arg)
	  else when (equal arg '&key)
	         do (setq arg-type 'key-arg)
	  else when (or (equal arg '&rest) (equal arg '&allow-other-keys)
			(equal arg '&aux))
	         do (setq arg-type nil)
	  else when (equal arg-type 'required-arg)
	         collect arg into required-arg
	  else when (equal arg-type 'optional-arg)
	         collect arg into optional-arg
	  else when (equal arg-type 'key-arg)
	         when (and (listp arg) (listp (first arg)))
		   collect (list (first (first arg)) (second arg)) into key-arg
		 else collect arg into key-arg
	  finally (return (values required-arg optional-arg key-arg)))))

;;; Things are a little different for the function argument widgets.
;;; Since we can't check their types, they must all be text-field
;;; objects.  Since required args (and sometimes optional and key args
;;; as well) have no default values, calling get-object-value on a
;;; required-arg-field signals an error.  Get-object-value is called
;;; by get-widget-value (which is called by make-value-list).
(def-simple-class arg-field (lispview:text-field)
  ())

(def-simple-class required-arg-field (arg-field)
  ((arg-name :type 'symbol)))

(def-simple-class optional-arg-field (arg-field)
  ((arg-name :type 'symbol)
   (arg-default)))

(def-simple-class key-arg-field (arg-field)
  ((arg-name :type 'symbol)
   (arg-default)))

;;; If user leaves field blank for required arg, signal an error.
(defmethod get-widget-value
    ((widget required-arg-field)  &optional (lispview-value (lispview:value widget)))
  (multiple-value-bind (val errorp)
      (call-next-method)
    (when (eq errorp :empty-string)
      (widget-notice
       widget
       (format nil "Must provide a value for required argument ~A." (arg-name widget))))
    (values val errorp)))

(defmethod set-widget-value ((widget arg-field) val)
  (if (eq val :no-default)
      (setf (lispview:value widget) "")
      (call-next-method)))

;;; These are used in place of get-widget-value when we are consing up
;;; the arglist to call the function.
(defmethod make-value-list ((widget required-arg-field))
  (multiple-value-bind (val errorp)
      (get-widget-value widget)
    (values (list val) errorp)))

(defmethod make-value-list ((widget optional-arg-field))
  (multiple-value-bind (val errorp)
      (get-widget-value widget)
    (cond ((eq errorp :empty-string)
	   (if (eq val :no-default)
	       (values nil nil)
	       (values (list val) nil)))
	  (errorp (values (list val) t))
	  (t (values (list val) nil)))))

(defmethod make-value-list ((widget key-arg-field))
  (multiple-value-bind (val errorp)
      (get-widget-value widget)
    (cond ((eq errorp :empty-string)
	   (if (eq val :no-default)
	       (values nil nil)
	       (values (list (arg-name widget) val) nil)))
	  (errorp (values (list (arg-name widget) val) t))
	  (t (values (list (find-symbol (arg-name widget) 'keyword)
			   val) nil)))))

;;; Retrieve the current value of the object associated with the
;;; widget.  This will be called to get an initial value.
(defmethod get-object-value ((widget required-arg-field))
  :no-default)

(defmethod get-object-value ((widget arg-field))
  (arg-default widget))

;;; Initialization:
(defmethod initialize-instance :around ((widget arg-field) &rest initargs)
  (let ((arg-name (getf initargs :arg-name)))
    (unless (and arg-name (symbolp arg-name)) (error "Bad arg-name ~S." arg-name))
    (unless (getf initargs :label)
      (setf (getf initargs :label) (symbol-name arg-name)))
    (unless (getf initargs :update-value)
      (setf (getf initargs :update-value)
	    #'(lambda (string)
		(set-widget-value widget (get-widget-value widget string)))))
    (apply #'call-next-method widget initargs)
    widget))

(defmethod initialize-instance :around ((widget optional-arg-field) &rest initargs)
  (let ((arg-default (getf initargs :arg-default)))
    (apply #'call-next-method widget initargs)
    (unless (getf initargs :value) (set-widget-value widget arg-default))
    widget))

(defmethod initialize-instance :around ((widget key-arg-field) &rest initargs)
  (let ((arg-name (getf initargs :arg-name))
	(arg-default (getf initargs :arg-default)))
    (unless (getf initargs :label)
      (setf (getf initargs :label) (format nil "  ~A" (symbol-name arg-name))))
    (apply #'call-next-method widget initargs)
    (unless (getf initargs :value) (set-widget-value widget arg-default))
    widget))

;;; This will be called when user selects a viewable
(defmethod insert-selection ((dlg function-dialog) sexpr)
  (let* ((panel (find-if #'(lambda (c) (typep c 'lv:panel))
			 (lv:children dlg)))
	 (item (lispview:keyboard-focus panel))
	 (arg-fields (arg-fields dlg)))
    (cond (item
	   (setf (lispview:value item) (format nil "~S" sexpr))
	   ;; Set new focus to be next text/numeric field....
	   (setf (lispview:keyboard-focus panel)
		 (nth (mod (1+ (position item arg-fields)) (length arg-fields))
		      arg-fields)))
	  (t (warn "No item selected in dialog: Cannot insert selection.")))))

#|
;;;; Stuff for getting viewable parameters with the mouse

(defun push-interest-onto-all-panes (interest)
  (dolist (s *screen-list*)
    (dolist (p (pane-list s))
      (pushnew interest (lispview:interests p)))))

;;; Argument can be an instance of an interest or an interest class-name.
(defun remove-interest-from-all-panes (interest)
  (dolist (s *screen-list*)
    (dolist (p (pane-list s))
      (setf (lispview:interests p)
	    (remove interest (lispview:interests p)
		    :test (if (typep interest 'lispview:interest)
			      #'eq
			      #'(lambda (class int) (typep int class))))))))

;;; Catch all mouse clicks (with no bucky keys)
(def-simple-class get-viewable-for-dialog-interest (lispview:mouse-interest)
  (the-panel arg-fields)
  (:default-initargs :event-spec '(() (:left :down))))

    
(defmethod lispview:receive-event
    ((pane X-pane) (interest get-viewable-for-dialog-interest) event)
  (let* ((pic (car (picture-stack pane)))
	 (panel (the-panel interest))
	 (item (lispview:keyboard-focus panel))
	 vbl)
    (when (and pic item)
      (setq vbl (viewable pic))
      (setf (lispview:value item) (format nil "~S" (get-viewable pane)))
)))
|#

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; An experimental function to create a slot with type t, nil or number.
(defun make-expt-dialog ()
  (let (dialog panel excl num-f
	       (t-nil-state :active)
	       (number-state :inactive))
    (setq dialog (make-instance 'lispview:popup-window
				:label "expt"
				:height 200
				:width 400 :top 0 :left 0
				:foreground (foreground (current-screen))
				:background (background (current-screen)))) 
    (setq panel (make-instance 'lispview:panel :parent dialog))
    (make-instance 'lispview:command-button :parent panel
		   :label "Number/t-or-nil"
		   :command #'(lambda ()
				(psetq t-nil-state number-state
				       number-state t-nil-state)
				(setf (lispview:state excl) t-nil-state)
				(setf (lispview:state num-f) number-state)))
    (setq excl (make-instance 'lispview:exclusive-setting :parent panel
			      :choices (list t nil) 
			      :layout :horizontal
			      :state t-nil-state))
    (setq num-f (make-instance 'lispview:numeric-field :parent panel
			       :min-value 0 :max-value 100
			       :layout :horizontal
			       :state number-state))
    (make-instance 'lispview:command-button :parent panel
		   :left 300 :top 150
		   :label "OK"
		   :command #'(lambda ()
				(if (eq t-nil-state :active)
				    (print (lispview:value excl))
				    (print (lispview:value num-f)))
				(setf (lispview:status dialog) :destroyed)))
    (setf (lispview:value excl) t)
    dialog))
				    


;;; Local Variables:
;;; buffer-read-only: t 
;;; End:
