;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: conversion.lisp
;;;  Author: Chichilnisky
;;;  Description:
;;;  Creation Date: summer 1992
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)

(export '(load-hips-image write-cap-matrix read-cap-matrix save-sun-raster-file
	  load-sun-raster-file))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Cap:

(defun write-cap-matrix (arr path &key (comment nil) (header t))
  (with-open-file (file path :direction :output :if-does-not-exist :create
			:if-exists :rename-and-delete)
    (when comment (format file "; ~S~%" comment))
    (when header
      (format file "~D ; rows~%~D ; cols~%" (row-dim arr) (col-dim arr)))
    (dolist (row (rows arr))
      (dotimes (i (length row))
	(format file "~F " (aref row i)))
      (format file "~%")))
  arr)

;; Read an ascii cap matrix file
(defun read-cap-matrix (file-name &key static)
  (with-open-file (stream (pathname file-name) :direction :input :if-does-not-exist :error)
    (let* ((rows (round (read stream nil)))
	   (cols (round (read stream nil)))
	   (data (if static
		     (allocate-array (list rows cols) :element-type 'single-float)
		     (make-array (list rows cols) :element-type 'single-float))))
      (dotimes (row rows)
	(dotimes (col cols)
	  (setf (aref data row col) (float (read stream)))))
      data)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; HIPS:

;; Read in a HIPS 2 image file.  This is only a beginning, only works
;; on 8bit grays.

(defun load-hips-image (path &key ->)
  (let (version original-name sequence-name frames
		original-date orows ocols rows cols frows fcols
		pixel-format number-of-colors history description skip
		number-of-parameters binary-size)

    (declare (ignore original-name sequence-name frames
		     original-date orows ocols frows fcols pixel-format number-of-colors))
    (with-open-file (stream path :direction :input)

      ;; Version depends on the contents of the first line
      ;; We only support Version 2
      (setq version (if (string= (read-line stream nil nil) "HIPS") 2 1))
      (unless (equal version 2)
	(error "HIPS images other than version 2 are not supported"))

      ;; Read the required stuff in the header
      (setq original-name (read-line stream nil nil))
      (setq sequence-name (read-line stream nil nil))
      (setq frames (read stream nil nil))
      ;; *** Seems like this shuold be in there. Why not?
      ;; (setq original-date (read stream nil nil))
      (setq orows (read stream nil nil))
      (setq ocols (read stream nil nil))
      (setq rows (read stream nil nil))
      (setq cols (read stream nil nil))
      (setq frows (read stream nil nil))
      (setq fcols (read stream nil nil))
      (setq pixel-format (read stream nil nil))
      (setq number-of-colors (read stream nil nil))

      ;; Some basic error checking
      (unless 'every 'numberp (list frames orows ocols rows cols frows
				    fcols pixel-format number-of-colors)
	      (error "Failed to read HIPS header"))
      (unless (and (zerop pixel-format) (= 1 number-of-colors))
	(error "Only 8-bit HIPS images supported now"))

      ;; Read in the history size and then string
      (setq history (make-string (read stream nil nil)))
      (read-array stream history)

      ;; Read in the description size and then string
      (setq description (make-string (read stream nil nil)))
      (read-array stream description)

      ;; A few more parameters that are dealt with poorly (we require them to be zero)
      (setq number-of-parameters (read stream nil nil))
      (unless (zerop number-of-parameters)
	(error "Nonzero HIPS extra parameter count not supported yet"))

      (setq binary-size (read stream nil nil))
      (unless (zerop binary-size)
	(error "Nonzero HIPS extra binary size not supported yet"))

      ;; Figure out where we are for reading the data
      (setq skip (file-position stream))

      ;; Handle images and sequences
      (if (= frames 1)
	  (image-from-file path :xsize cols :ysize rows :skip-bytes skip
			   :-> (or original-name ->))
	  (make-image-sequence
	   (loop for frame from 0 below frames
		 for offset from skip by (* cols rows)
		 collect (image-from-file path :xsize cols :ysize rows :skip-bytes offset))
	   :-> (or sequence-name ->))))))


#|

(load-hips-image "/usr/csh/images/hips/nick.hips2")
(load-hips-image "/usr/csh/images/hips/hands_movie.hips2")

|#

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Sun Raster:

;; Crufty. Assumes that data are in RGB,RGB,... order.  Wrote with
;; possible color map keywords in mind.

(defun save-sun-raster-file (arr path &key (depth 3)
				 (dimensions (guess-dimensions (/ (total-size arr) 3))))

  (unless (equal (array-element-type arr) '(unsigned-byte 8))
    (error "Can only write 8bit arrays to sun raster file"))
  (unless (equal depth 3)
    (error "Only 24bit format data currently supported"))
  
  (let* ((header (make-array 8 :element-type '(signed-byte 32)))
	 (sun-raster-magic-number #x59a66a95)
	 (rows (second dimensions))
	 (cols (first dimensions))
	 (padding-bytes (if (oddp (* depth cols)) 1 0));; Must round to nearest word
	 (data-bytes (* rows (+ (* depth cols (sizeof (array-element-type arr))) padding-bytes)))
	 (type 1)
	 (color-map-type 0)
	 (color-map-size 0))

    ;; Fill the header and write it
    (fill! header (list sun-raster-magic-number cols rows (* depth 8)
			data-bytes type color-map-type color-map-size))
    (with-open-file (stream path :direction :output :if-exists :rename-and-delete
			    :if-does-not-exist :create
			    :element-type (array-element-type header))
      (write-array stream header))

    ;; Write the image data
    (with-open-file (stream path :direction :output :if-exists :append
			    :element-type (array-element-type arr))
      (loop with image-vec = (vectorize arr)
	    with line-bytes = (* depth cols)
	    with bgr-line = (similar image-vec :dimensions line-bytes)
	    for row from 0 below rows
	    for offset from 0 by (* depth cols)
	    for line = (vectorize image-vec :x-offset offset :size line-bytes)
	    do
	    (flip-rgb line :-> bgr-line)
	    (write-array stream bgr-line)
	    (dotimes (i padding-bytes)
	      (write-byte #\0 stream))))
    arr))

(defun load-sun-raster-file (path)
  (let ((header (make-array 8 :element-type '(signed-byte 32)))
	(sun-raster-magic-number #x59a66a95)
	rows cols padding-bytes data-bytes depth
	type color-map-type color-map-size data)

    (declare (ignore data-bytes color-map-type))
    
    ;; Read in the header array
    (with-open-file (stream path :direction :input :element-type (array-element-type header))
      (read-array stream header))
    
    (unless (= (aref header 0) sun-raster-magic-number)
      (error "~a is ot a sun raster file" path))
    (setf cols (aref header 1))
    (setf rows (aref header 2))
    (setf depth (/ (aref header 3) 8))
    (setf data-bytes (aref header 4))
    (setf type (aref header 5))
    (unless (= type 1)
      (error "Bad raster file type ~a" type))
    (setf color-map-type (aref header 6))
    (setf color-map-size (aref header 7))
    (when (plusp color-map-size)
      (warn "Color maps not supported"))
    
    (setf data (allocate-array (list (* rows cols) depth) :element-type '(unsigned-byte 8)))
    (setf padding-bytes (if (oddp (* depth cols)) 1 0));; Must round to nearest word
    (unless (equal depth 3) (error "Only 24bit format data currently supported"))

    ;; Read the image data
    (with-open-file (stream path :direction :input :element-type (array-element-type data))
      (dotimes (i 32) (read-byte stream))
      (loop with vector = (vectorize data)
	    for row from 0 below rows
	    for offset from 0 by (* depth cols)
	    for line = (vectorize vector :x-offset offset :size (* depth cols))
	    do
	    (read-array stream line)
	    (dotimes (i padding-bytes)
	      (read-byte stream))))
    
    (flip-rgb data :-> data)
    (values data cols rows)))

;; reverse order of RGB triplets
(defun flip-rgb (arr &key ((:-> result) (similar arr)))
  (unless (equal arr result)
    (copy arr :-> result))
  (with-displaced-vectors ((vec (vectorize result)))
    (loop for i from 0 below (length vec) by 3
	  for i+2 =  (+ i 2)
	  for val = (aref vec i)
	  do
	  (setf (aref vec i) (aref vec i+2))
	  (setf (aref vec i+2) val)))
  result)

(defun guess-dimensions (x)
  (let ((factor (ceiling (sqrt x))))
    (loop until (zerop (mod (/ x factor) 1))
	  do (decf factor 1))
    (list (/ x factor) factor)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Local Variables:
;;; buffer-read-only: t 
;;; fill-column: 79
;;; End:
