;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: amoeba.lisp
;;;  Author: Chichilnisky. Adapted from "Numerical Recipes in C",  pp. 307-309.
;;;  Description: 
;;;  Creation Date: 10.8.92 Rewritten by EJC. Interface changed,
;;;  internal calls put in, various speed enhancements and checks.
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(in-package 'obvius)
(export '(amoeba-fit))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Find minimum of the error function using downhill simplex method.
;;; Start with a given n-dim simplex,  which is a vector of float vectors.
;;; There are a total of (n+1) of these vectors in a simplex.
;;; The initial simplex is "centered" at the starting point specified.
;;; Each vector is a vector in the domain of the function.
;;; A vector with minimal function value is returned.
;;; The error function should return values >= 0.0.
;;; Routine stops when difference of values at vertices
;;; divided by absolute value reaches criterion,
;;; or, optionally, when the absolute error gets small.
;;; User can specify how often a count of function calls and error is reported.
;;; After finding a minimum, amoeba restarts itself, according to the restart keyword..

(defun amoeba-fit (function initial-parameters function-args
			    &key randomize
			    (alpha -1.0) (beta 0.5) (gamma 2.0)
			    (tolerance *machine-tolerance*) (criterion *tolerance*)
			    (restart 0)
			    (max-function-calls most-positive-fixnum)
			    (count max-function-calls))

	      
  (declare (fixnum max-function-calls count)
	   (single-float tolerance criterion alpha beta gamma))
  
  (let* ((simplex (make-simplex initial-parameters :randomize randomize))
	 (first (aref simplex 0))	; first point in simplex
	 (dim (total-size first))	; dimension of function domain space
	 (vertex-sum (similar first))	; sum of vertices in simplex
	 (vertex (similar first))	; scratch space for calculation
	 (function-values (make-array (+ dim 1) :element-type 'single-float))
					; values of error function at simplex vertices
 	 value last-value		; stored error function values
	 (function-calls 0)		; How many times the error function has been called
	 (next-count count)		; When is the next time info should be printed
	 min-index max2-index max-index	; indices of vertices with extreme error function values
	 )
    (declare (type (vector single-float) function-values) 
	     (single-float value last-value)
	     (fixnum min-index  max-index max2-index function-calls next-count))

    (simplex-evaluate simplex function function-args function-values)
    (incf function-calls (+ dim 1))
    (simplex-vertex-sum simplex vertex-sum)

    ;; Main minimization loop
    (do* () ((progn (multiple-value-setq (min-index max2-index max-index)
		      (amoeba-min-max2-max function-values))
		    (or (< (amoeba-criterion (aref function-values min-index)
					     (aref function-values max-index))
			   criterion)
			(< (abs (aref function-values min-index)) tolerance)
			(> function-calls max-function-calls)))
	     ())
	     
      (setq value (simplex-crawl (aref simplex max-index) vertex function-values
				 vertex-sum function function-args max-index alpha))
      (incf function-calls)

      ;; Inform user about status of search
      (when (> function-calls next-count)
	(incf next-count count)
	(status-message "~a: ~a" function-calls (aref function-values min-index)))
      
      (cond ((<= value (aref function-values min-index))
	     (setq value (simplex-crawl (aref simplex max-index) vertex
					function-values vertex-sum function function-args
					max-index gamma))
	     (incf function-calls))
	    ((>= value (aref function-values max2-index))
	     (setq last-value (aref function-values max-index))
	     (setq value (simplex-crawl (aref simplex max-index) vertex function-values
					vertex-sum function function-args max-index beta))
	     (incf function-calls)
	     (when (>= value last-value)
	       (simplex-contract simplex min-index)
	       (simplex-evaluate simplex function function-args function-values)
	       (incf function-calls (+ dim 1))
	       (simplex-vertex-sum simplex vertex-sum)))
	    (t nil)))

    ;; If restarts are demanded, begin again at the minimum that we found
    ;; If we got wedged and had too many function calls, start at random place.
    (cond
      ((and restart (plusp restart))
       (amoeba-fit function (aref simplex min-index) function-args :restart (- restart 1)
		   :alpha alpha :beta beta :gamma gamma
		   :tolerance tolerance :criterion criterion :count count
		   :randomize randomize :max-function-calls max-function-calls))
      ((> function-calls max-function-calls)
       (warn "~a function calls in amoeba exceeded limit of ~a. Random restart..."
	     function-calls max-function-calls)
       (amoeba-fit function initial-parameters function-args :randomize (or randomize 1.0)
		   :restart restart :alpha alpha :beta beta :gamma gamma
		   :tolerance tolerance :criterion criterion :count count
		   :max-function-calls max-function-calls))
      (t (values (aref simplex min-index) (aref function-values min-index) function-calls)))))
#|
;; test code
(time (amoeba-fit #'vector-distance (make-matrix 2 2) (list (make-matrix 1 1))
		  :tolerance *tolerance* :count 10 :restart 0
		  :max-function-calls 500))

(time (amoeba-fit #'vector-distance (make-matrix '((1) (1))) (list (make-matrix '((1) (1))))
		  :randomize 0.1 :tolerance *tolerance*))

;; compare to stepit??
(time (stepit-fit #'vector-distance (make-matrix 2.1 2)
		  (list (make-matrix 1 1))
		  :count 10 :initial-steps (make-matrix 10 10)))
|#

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Make an n-dim simplex "centered" at a desired point.
;; Can randomize the vertices if desired.
(defun make-simplex (midpoint &key (offset midpoint) randomize)
  (let* ((dim (total-size midpoint))
	 (result (make-array (+ dim 1)))
	 (d-offset (vectorize offset)))
    (dotimes (i (length result))
      (declare (fixnum i))
      (let ((vertex (copy midpoint)))
	(when (< i dim)
	  (incf (aref (vectorize vertex) i)
		(aref d-offset i)))
	(when randomize
	  (randomize vertex randomize :-> vertex))
	(setf (aref result i) vertex)))
    result))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Contract simplex toward the minimal vertex
(defun simplex-contract (simplex min-index) 
  (declare (fixnum min-index))
  (let* ((min-vertex (aref simplex min-index))
	 (dim (total-size min-vertex))
	 vertex)
    (declare (type (array single-float) vertex min-vertex))
    (dotimes (index (length simplex))
      (declare (fixnum index))
      (setq vertex (aref simplex index))
      ;; (add vertex min-vertex :-> vertex)
      (obv::internal-add vertex min-vertex vertex dim)
      ;; (mul vertex 0.5 :-> vertex)
      (obv::internal-sc-mul vertex vertex dim 0.5)
      ))
  simplex)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Reflect max-vertex of the simplex through the other vertices 
(defun simplex-crawl (max-vertex vertex function-values vertex-sum
				 function function-args max-index factor)

  (declare (type (array single-float) max-vertex vertex vertex-sum function-values)
	   (fixnum max-index)
	   (single-float factor))
  (let* ((dim (total-size max-vertex))
	 (factor1 (/ (- 1.0 factor) dim))
	 (factor2 (- factor1 factor))
	 (function-val 0.0))
    (declare (single-float function-val))

    ;; (mul vertex-sum factor1 :-> vertex)
    (obv::internal-sc-mul vertex-sum vertex dim (float factor1))
    
    ;; *** Don't forget to invert this operation!!
    ;; (mul max-vertex factor2 :-> max-vertex)
    (obv::internal-sc-mul max-vertex max-vertex dim (float factor2))
    ;; (sub vertex max-vertex :-> vertex)
    (obv::internal-sub vertex max-vertex vertex dim)
    ;; (div max-vertex factor2 :-> max-vertex)
    (obv::internal-sc-mul max-vertex max-vertex dim (/ 1.0 (float factor2)))
    
    (setq function-val (apply function vertex function-args))
    
    (when (< function-val (aref function-values max-index))
      (setf (aref function-values max-index) function-val)
      ;; (add vertex-sum vertex :-> vertex-sum)
      (obv::internal-add vertex-sum vertex vertex-sum dim)
      ;; (sub vertex-sum max-vertex :-> vertex-sum)
      (obv::internal-sub vertex-sum max-vertex vertex-sum dim)
      ;; (copy vertex :-> max-vertex)
      (obv::internal-copy-array vertex max-vertex dim)
      )
    function-val))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Compute function at each of the vertices of simplex, return a vector
(defun simplex-evaluate (simplex function function-args result)
  (declare (type (vector single-float) result))
  (dotimes (vertex (length simplex))
    (declare (fixnum vertex))
    (setf (aref result vertex)
	  (apply function (aref simplex vertex) function-args)))
  result)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Compute sum of simplex vertices and put into sum-vertex
(defun simplex-vertex-sum (simplex result)
  (let ((dim (total-size result)))
    ;; (zero! result)
    (obv::internal-const result 0.0 dim)
    (dotimes (vertex (length simplex))
      (declare (fixnum vertex))
      ;;(add (aref simplex vertex) result :-> result)
      (obv::internal-add (aref simplex vertex) result result dim))
    )
  result)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Normalized difference between min and max values.
(defun amoeba-criterion (min max)
  (declare (single-float min max))
  (/ (* 2.0 (abs (- max min))) (+ (abs max) (abs min))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Return positions of min, 2nd highest, and max values in vector
(defun amoeba-min-max2-max (vector)
  (declare (type (vector single-float) vector))
  (let* ((min-pos 0)
	 (max-pos 0)
	 (max2-pos 0)
	 (value 0.0))
    (declare (fixnum min-pos max-pos max2-pos)
	     (single-float value))
    (dotimes (i (length vector))
      (declare (fixnum i))
      (setq value (aref vector i))
      (cond ((< value (aref vector min-pos)) (setq min-pos i))
	    ((> value (aref vector max-pos)) (setq max2-pos max-pos
						   max-pos i))
	    ((> value (aref vector max2-pos)) (setq max2-pos i))))
    (values min-pos max2-pos max-pos)))


;;; Local Variables:
;;; buffer-read-only: t 
;;; fill-column: 79
;;; End:
