
#|
This file is used to make the index file for the OBVIUS documentation.
It reads as input the file "main.idx", produced by the \makeindex
command in LaTeX.  It creates an index file called "index.tex" that
can be included in the final document.

You run latex (with the \makeindex command in the main.tex file
uncommented).  This creates a main.idx file.  Then run this piece of
lisp code (make-index), which generates the index.tex file.  Then
you'll have to run latex on main.tex again to include the new
index.tex file.
|#

#|
(setq *doc-dir* "/v/dev/obvius-2.2/doc/") ;MIT
(setq *doc-dir* "/usr/src/obvius-2.2/doc/") ;STANFORD
(load (merge-pathnames *doc-dir* "index.lisp"))
(make-index)
|#

(in-package 'user)

(defvar *doc-dir* (pwd))

;;; Parse the .idx file created by Latex, turning it into an index.tex
;;; file.  It would probably be easier to write this in Emacs lisp.
(defun make-index (&optional (doc-directory *doc-dir*))
  (let* ((idx-file (merge-pathnames doc-directory "main.idx")) ;generated by LaTeX
	 (index-file (merge-pathnames doc-directory "index.tex"))
	 index-tree)
    (with-open-file (stream idx-file)	
      (loop with names with page with end with rest
	    for line = (read-line stream nil nil)
	    until (null line)
	    when (string/= (string-trim '(#\Space #\Tab #\Newline) line) "")
	    do
	    ;(format t "~A " line)
	    (if (string= (subseq line 0 (min (length line) 12)) "\\indexentry{")
		(setq rest (subseq line 12))
		(error "Unrecognized line in .idx file: ~A~%" line))
	    (if (setq end (match-brace rest))
		(setq names (subseq rest 0 end)  rest (subseq rest (1+ end)))
		(error "Missing closing brace in line: ~A~%" line))
	    (if (and (char= (char rest 0) #\{) ;opening brace
		     (setq end (position #\} rest))) ;closing brace
		(setq page (subseq rest 1 end))
		(error "Bad page number field in .idx file: ~A~%" line))
	    (setq names
		  (loop for string = names then (subseq string (1+ pos))
			for pos = (position #\, string)
			until (null pos)
			collect (subseq string 0 pos) into res
			finally (return (append res (list string)))))
	    (setq index-tree (insert-index-entry names page index-tree))))
    (when (probe-file index-file) (delete-file index-file))
    (with-open-file (stream index-file :if-does-not-exist :create :direction :output)
      (format stream "%%% Index file: automatically generated from main.idx~%")
      (format stream "%%% using the make-index function defined in index.lisp.~%~%")
      (format stream "\\begin{theindex}~%~%")
      (mapcar #'(lambda (tree) (write-index-entry stream tree))
	      index-tree)
      (format stream "~%\\end{theindex}~%"))))

;;; Return position of the next close brace that doesn't match any open brace.
(defun match-brace (str &optional (start 0))
  (loop for next-close = (position #\} str :start start)
	for next-open = (position #\{ str :start start)
	until (or (null next-close) (null next-open) (> next-open next-close))
	do (setq start (1+ (match-brace str (+ next-open 1))))
	finally (return (if (null next-close) nil next-close))))

;;; ** fix multiple definition warning!
(defun insert-index-entry (name-list page index-tree &optional (level 0))
  (when (> level 3) (error "Index nested more than 3 levels"))
  (let ((entry (find (car name-list) index-tree :test #'string= :key #'car))
	temp)
    (when (string= (car (last name-list)) "DEF") ;definition
      (when (setq temp (find-if #'(lambda (x) (and (listp x) (eq (cadr x) :definition)))
				(cadr entry)))
	(warn "Multiple definitions for index entry ~A on pages ~A and ~A"
	      (car name-list) page (car temp)))
      (setq page (list page :definition))
      (setq name-list (butlast name-list)))
    (cond (entry
	   (if (cdr name-list)	;more sub-entries
	       (setf (cddr entry)	;add another sub-entry
		     (insert-index-entry (cdr name-list) page (cddr entry)
					 (1+ level)))
	       (setf (cadr entry)	;add another page
		     (insert page (cadr entry)
			     :predicate
			     #'(lambda (s1 s2)
				 (string< (if (listp s1) (car s1) s1)
					  (if (listp s2) (car s2) s2)))
			     :duplicate-check
			     #'(lambda (s1 s2)
				 (string= (if (listp s1) (car s1) s1)
					  (if (listp s2) (car s2) s2))))))
	   index-tree)
	  (t (insert (make-new-entry name-list page) index-tree
		     :predicate #'string<
		     :key #'car)))))

(defun make-new-entry (name-list page &optional (level 0))
  (when (> level 3) (error "Index nested more than 3 levels"))
  (when name-list
    (let ((rest-names (cdr name-list)))
      (if rest-names
	  (list (car name-list) nil (make-new-entry rest-names page (1+ level)))
	  (list (car name-list) (list page))))))

(defun write-index-entry (stream tree &optional (level 0))
  (when tree
    (let ((entry (car tree))
	  (pages (cadr tree))
	  (sub-entries (cddr tree)))
      (format stream (cond ((= level 0) "\\item ")
			   ((= level 1) "\\subitem ")
			   ((= level 2) "\\subsubitem ")
			   (t (error "Index entry is too many levels deep: ~A" tree))))
      (format stream entry)
      (loop for page in pages
	    for first-p = t then nil
	    do
	    (if first-p (format stream " ") (format stream ", "))
	    (if (listp page)
		(if (eq (cadr page) :definition)
		    (format stream "{\\bf ~A}" (car page))
		    (error "Bad page number entry: ~A" page))
		(format stream page)))
      (terpri stream)
      (incf level)
      (mapcar #'(lambda (tree) (write-index-entry stream tree level))
	      sub-entries))))

;;; Insert entry into ordered list, ordered by predicate.  If duplicate-check is
;;; non-nil, it should be a function to test if entries are equal, in which case
;;; th eitem will not be added.
(defun insert (item list &key (predicate #'<) duplicate-check (key #'identity))
  (loop with item-key = (funcall key item)
	for pos from 0
	for last-list-item = nil then list-item
	for list-item in list
	until (funcall predicate item-key (funcall key list-item))
	finally
	(return (if (and last-list-item duplicate-check
			 (funcall duplicate-check (funcall key last-list-item) item-key))
		    list
		    (append (subseq list 0 pos) (cons item (nthcdr pos list)))))))

