;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: stepit.lisp
;;;  Author: Chichilnisky
;;;  Description: Stepit interfce. Basic code is C code, ported from Fortran.
;;;  Creation Date: 12.23.91 brought in from working homebrew version.
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(in-package 'obvius)
(export '(stepit-fit))

#|
;; BUGS/Comments.
;; *** Probably should eliminate the extra C routine that sets pointers, and do it here,
;; If I could figure out how to do that (need to get the address of an array).

;; *** More orthodox ffi usage, should probably be incorporated in routine.
;; I do not know why the existing compiled code works.
;; Something like this should be used.
(setf stepit-data-struct (make-foreign-pointer
			  :address
			  (LCL:foreign-variable-address *stepit-global-data-structure-name*)
			  :type
			  '(:pointer lsteptype)))
(lsteptype-XMIN stepit-data-struct)

|#

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; General Fitting Function Interface
;;; Here is the setup for fitting with stepit. After some experimentation,
;;; it seems to be the right general model for a fitting procedure. EJC 12.14.91.

;;; Stepit is called with the following arguments:
;;; 1. error-function - function to be called with parameters and other arguments (see below)
;;;    It is the value of this function that stepit attempts to minimize. 
;;; 2. initial-parameters - starting point of parameters (a single-float vector or array,
;;;    or a list).
;;; 3. error-function-args - list of arguments that are passed (after the parameters)
;;;    to the error function at each iteration. These parameters are passed
;;;    to error-function as parameters, not as a list.
;;; keywords - things specific to stepit, as opposed to another fitting procedure.
;;; The main routines declares and accesses special global variables.
;;; Note: when the parameters are an array, stepit will avoid copying things
;;; and move faster. On the other hand, you can use lists if the error function
;;; wants to use an (apply) call with the parameters.

;;; Example:
;;; (stepit-fit 'my-error-function (make-matrix 1.0 2.0) (list 100 200))
;;; then, at each iteration, my-error-function will be called these arguments:
;;; #(xx.xx xx.xx) 100 200
;;; where the first array contains the current parameters selected by stepit.
;;; See the test code for a simple example.

;;; Special notes:
;;; Lucid's foreign-function interface is used heavily.
;;; Note the magic number 128, which is hardcoded in stepit.c, but in this file
;;; must remain as a number. Also the number 16512.
;;; Optimization: Could try to use typed-foreign-aref (p 4-45 of Lucid C interface)
;;; Note that I use the function steptp_, which sets the pointer in the global structure
;;; to be equal to the array passed in. Hence, the array must be static, else
;;; it may get garbage collected on the fly!

(defconstant *stepit-global-data-structure-name* "_stepit_global_data_structure")

;;; The stepit data structure is defined in the C code.
;;; This one must be in precise correpondence.
(LCL:def-foreign-struct lsteptype
  (X :type (:pointer :single-float)) ; Actually, a single-float pointer!
  (XMAX :type (:array :single-float (128)))
  (XMIN :type (:array :single-float (128)))
  (DELTX :type (:array :single-float (128)))
  (DELMN :type (:array :single-float (128)))
  (ERR :type (:array :single-float (16512))) ; (* 128 129)
  (FOBJ :type :single-float)
  (NV :type :signed-32bit)
  (NTRAC :type :signed-32bit)
  (MATRX :type :signed-32bit)
  (MASK :type (:array :signed-32bit (128)))
  (NFMAX :type :signed-32bit)
  (NFLAT :type :signed-32bit)
  (JVARY :type :signed-32bit)
  (NXTRA :type :signed-32bit)
  (KFLAG :type :signed-32bit)
  (NOREP :type :signed-32bit)
  (KERFL :type :signed-32bit)
  (KW :type :signed-32bit))

;;; Main interface function.
(defun stepit-fit (error-function
		   initial-parameters
		   error-function-args
		   &key
		   (lower-bounds (fill! (copy initial-parameters) most-negative-single-float))
		   (upper-bounds (fill! (copy initial-parameters) most-positive-single-float))
		   (smallest-steps (mul initial-parameters 1e-3))
		   (initial-steps (mul initial-parameters 0.1))
		   (masks (make-array (total-size initial-parameters))) ; t means ignore this var
		   (max-function-calls 10000)
		   (count nil)
		   (iterations 1)
		   (random-factor 0.1)
		   )

  (let* ((foreign-error-function 'stepit-foreign-error-function)
	 (stepit-data-struct (LCL:foreign-variable-pointer *stepit-global-data-structure-name*))
	 (mask-array (lsteptype-MASK stepit-data-struct))
	 (smallest-step-size-array (lsteptype-DELMN stepit-data-struct))
	 (initial-step-size-array (lsteptype-DELTX stepit-data-struct))
	 (lower-bound-array (lsteptype-XMIN stepit-data-struct))
	 (upper-bound-array (lsteptype-XMAX stepit-data-struct))
	 (stepit-error-function-args error-function-args)
	 (stepit-internal-parameters (fill! (allocate-array (dimensions initial-parameters)
							:element-type 'single-float)
					    initial-parameters))
	 (stepit-parameters (cond ((listp initial-parameters) (copy initial-parameters))
				  ((arrayp initial-parameters) stepit-internal-parameters)
				  (t (error "Invalid parameters for stepit"))))
	 (stepit-error-function error-function)
	 (stepit-counter 0)
	 (stepit-interval count)
	 (foreign-error-function-name
	  (concatenate 'string "_"
		       (substitute #\_ #\- (string-downcase (symbol-name foreign-error-function)))))
	 (foreign-error-function-address
	  (LCL:make-foreign-pointer :address
				    (LCL:foreign-variable-address foreign-error-function-name))))

    ;; Declare special variables that will be globally available during the course of this function
    (declare (special stepit-internal-parameters stepit-error-function stepit-parameters
		      stepit-data-struct stepit-error-function-args stepit-counter stepit-interval))

    ;; Initialize magic numbers in the stepit structure
    (setf (lsteptype-FOBJ stepit-data-struct) 0.0 ; return value of error function goes here
	  (lsteptype-NFMAX stepit-data-struct) max-function-calls
	  (lsteptype-MATRX stepit-data-struct) 0
	  (lsteptype-NTRAC stepit-data-struct) -1 ; Normal output, -1, no output 1 extra
	  (lsteptype-KW stepit-data-struct) 0 ;logical unit no of print = stderr
	  (lsteptype-NV stepit-data-struct) (total-size initial-parameters))

    ;; Start with all parameters disabled.
    (dotimes (i 128) (setf (LCL:foreign-aref mask-array i) 1))

    ;; Fill all arrays, including parameters and mask
    ;; Arrays need to be made into vectors to fill the foreign arrays.
    (when (arrayp stepit-parameters)
      (setq smallest-steps (vectorize smallest-steps))
      (setq initial-steps (vectorize initial-steps))
      (setq lower-bounds (vectorize lower-bounds))
      (setq upper-bounds (vectorize upper-bounds)))

    (dotimes (i (total-size initial-parameters))
      (setf (LCL:foreign-aref mask-array i) (if (and masks (elt masks i)) 1 0)
	    (LCL:foreign-aref smallest-step-size-array i) (elt smallest-steps i)
	    (LCL:foreign-aref initial-step-size-array i) (elt initial-steps i)
	    (LCL:foreign-aref lower-bound-array i) (elt lower-bounds i)
	    (LCL:foreign-aref upper-bound-array i) (elt upper-bounds i)))

    (let (error-list parameter-list position)
      (dotimes (iteration iterations)
	;; Generate starting parameters from the initial parameters
	(when (plusp iteration)
	  (randomize stepit-internal-parameters random-factor :-> stepit-internal-parameters))
	
	;; Call the C routine, passing it the address of the error function
	;; and the location of the internal parameters array to access.
	(steptp_ foreign-error-function-address stepit-internal-parameters)

	;; Hang on to the last error value, and the corresponding parameters
	(push (lsteptype-FOBJ stepit-data-struct) error-list)
	(push (fill! (similar initial-parameters) stepit-internal-parameters)  parameter-list)
	)
      (free-array stepit-internal-parameters)

      ;; Return the parameter list that corresponds to the minimum error
      (setq position (position (minimum error-list) error-list))
      (values (elt parameter-list position)
	      (elt error-list position)))))

(LCL:def-foreign-callable stepit-foreign-error-function ()
  (declare (special stepit-error-function-args stepit-parameters stepit-internal-parameters
		    stepit-error-function stepit-data-struct stepit-counter stepit-interval))
  ;; When the parameters are an array, just pass the internal array to the error function (fast).
  ;; Otherwise, the parameters must first be filled from the internal array (slow).
  (when stepit-interval
    (incf stepit-counter)
    (when (zerop (mod stepit-counter stepit-interval))
      (status-message "~a" stepit-counter)))
  (unless (eq stepit-parameters stepit-internal-parameters)
    (fill! stepit-parameters stepit-internal-parameters))
  (setf (lsteptype-FOBJ stepit-data-struct)
	(apply stepit-error-function stepit-parameters stepit-error-function-args)))
	      


#|

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Sample fitting code
(setf vector (make-matrix 1 3))

(stepit-fit #'vector-distance (make-matrix 10 10) (list vector))
(stepit-fit #'vector-distance (make-matrix 10 10) (list vector) :count 2)

(stepit-fit #'vector-distance (make-matrix 10 10) (list vector) :iterations 5)
(stepit-fit #'vector-distance (make-matrix 10 10) (list vector))
(stepit-fit #'vector-distance (make-matrix 10 10) (list vector) :lower-bounds (make-matrix 4 4))
(stepit-fit #'vector-distance (make-matrix 10 10) (list vector) :upper-bounds (make-matrix 4 4))

;; Using lists
(setf list (list 1 4))
(stepit-fit #'vector-distance (list 10 10) (list list))
(stepit-fit #'vector-distance (list 10 10) (list list) :iterations 5)

;;; Test the error handling with a non-deterministic error-function,
;;; to see that the error-handling works OK.
(defun my-error-function (parameters vector)
  (random 1.0))
(stepit-fit 'my-error-function (make-matrix 10 10) (list vector))


|#


;;; Local Variables:
;;; buffer-read-only: t 
;;; fill-column: 79
;;; End:
